package tv.athena.thirdparty.impl

import android.content.Intent
import tv.athena.core.interceptor.ActivityResultCallback
import tv.athena.klog.api.KLog
import tv.athena.platform.components.AeFragmentActivity
import tv.athena.thirdparty.api.IThirdParty
import tv.athena.thirdparty.api.IThirdPartyListener
import tv.athena.thirdparty.api.IThridPartyTakeANumberListener
import tv.athena.thirdparty.api.ThirdParty
import tv.athena.thirdparty.api.ThirdPartyFailResult
import tv.athena.thirdparty.api.ThirdPartyProduct
import tv.athena.thirdparty.api.ThirdPartyUserInfo
import tv.athena.util.hiido.HiidoUtils
import tv.athena.thirdparty.impl.unbind.UnbindLogin

object ThirdPartyImpl : IThirdParty, IThirdPartyListener {

    private val TAG = "ThirdPartyImpl"

    private var mCurrentLogin: IThirdPartyLogin? = null
    private var mListenerList: IThirdPartyListener? = null

    private const val DEFAULT_CODE: Int = 50228

    private var sCode = DEFAULT_CODE

    private val initProducts: MutableList<ThirdPartyProduct> = mutableListOf()

    fun start(products: MutableList<ThirdPartyProduct>) {
        initProducts.addAll(products)
        ThirdParty.mThirdPartyImpl = this
        ThirdPartyBase.mThirdPartyImpl = this
        for (product in initProducts) {
            try {
                getLoginInstance(product, false)?.initSdk()
            } catch (error: Throwable) {
                KLog.e(TAG, "init $product product error.", error)
            }
        }
    }

    fun setSCode(sCode: Int) {
        this.sCode = sCode
    }

    private val mActivityResultCallback: ActivityResultCallback = object : ActivityResultCallback {
        override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent): Boolean {
            KLog.i(TAG, "mActivityResultCallback")
            return mCurrentLogin?.handleActivityResult(requestCode, resultCode, data) ?: false
        }

        override fun onDestroy() {
            mCurrentLogin?.recycle()
        }
    }

    override fun thirdParty(activity: AeFragmentActivity, product: ThirdPartyProduct) {
        KLog.i(TAG, "start thirdParty $product")
        activity.registerActivityResultInterceptor(mActivityResultCallback)
        getLoginInstance(product)?.doLogin(activity, this)
        HiidoUtils.reportMatrixCount(sCode, "thirdParty${product.key}", "click", 1)
    }

    override fun <T> thirdPartyBySdkApi(activity: AeFragmentActivity, product: ThirdPartyProduct, sdkApi: T) {
        activity.registerActivityResultInterceptor(mActivityResultCallback)
        getLoginInstance(product)?.thirdPartyBySdkApi(activity, this, sdkApi)
        HiidoUtils.reportMatrixCount(sCode, "thirdPartyBySdkApi${product.key}", "click", 1)
    }

    override fun takeANumber(
        activity: AeFragmentActivity, product: ThirdPartyProduct, listener: IThridPartyTakeANumberListener
    ) {
        getLoginInstance(product)?.takeANumber(activity, listener)
    }

    override fun setEventListener(listener: IThirdPartyListener) {
        mListenerList = listener
    }

    override fun removeEventListener() {
        mListenerList = null
    }

    override fun onTPLSuccess(product: ThirdPartyProduct, userInfo: ThirdPartyUserInfo) {
        KLog.i(TAG, "onTPLSuccess product $product, userInfo $userInfo")
        mListenerList?.onTPLSuccess(product, userInfo)
        getLoginInstance(product)?.recycle()
        HiidoUtils.reportMatrixCount(sCode, "thirdParty${product.key}", "suc", 1)
    }

    override fun onTPLFailed(product: ThirdPartyProduct, result: ThirdPartyFailResult, throwable: Throwable) {
        KLog.i(TAG, "onTPLFailed product $product, error:, $throwable")
        mListenerList?.onTPLFailed(product, result, throwable)
        getLoginInstance(product)?.logout()
        getLoginInstance(product)?.recycle()
        HiidoUtils.reportMatrixCount(sCode, "thirdParty${product.key}", "fail", 1)
    }

    override fun onCancel(product: ThirdPartyProduct) {
        KLog.i(TAG, "onCancel product $product")
        mListenerList?.onCancel(product)
        getLoginInstance(product)?.recycle()
        HiidoUtils.reportMatrixCount(sCode, "thirdParty${product.key}", "cancel", 1)
    }

    override fun logout(product: ThirdPartyProduct) {
        KLog.i(TAG, "logout, $product")
        if (product != ThirdPartyProduct.NONE) {
            getLoginInstance(product)?.logout()
        } else {
            mCurrentLogin?.logout()
            mCurrentLogin = null
        }
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent): Boolean {
        return mActivityResultCallback.onActivityResult(requestCode, resultCode, data)
    }

    private fun getLoginInstance(product: ThirdPartyProduct): IThirdPartyLogin? {
        return getLoginInstance(product, true)
    }

    private fun getLoginInstance(product: ThirdPartyProduct, setCurrent: Boolean): IThirdPartyLogin? {
        if (!setCurrent || mCurrentLogin == null || mCurrentLogin?.product != product) {
            val thirdPartyLogin: IThirdPartyLogin? = when (product) {
                ThirdPartyProduct.FACEBOOK -> {
                    tv.athena.thirdparty.impl.facebook.FacebookLogin(product)
                }
                ThirdPartyProduct.GOOGLE -> {
                    tv.athena.thirdparty.impl.google.GoogleLogin(product)
                }
                ThirdPartyProduct.VK -> {
                    tv.athena.thirdparty.impl.vk.VKLogin(product)
                }
                ThirdPartyProduct.TWITTER -> {
                    tv.athena.thirdparty.impl.twitter.TwitterLogin(product)
                }
                ThirdPartyProduct.INSTAGRAM -> {
                    tv.athena.thirdparty.impl.instagram.InstagramLogin(product)
                }
                ThirdPartyProduct.TWITCH -> {
                    tv.athena.thirdparty.impl.twitch.TwitchLogin(product)
                }
                ThirdPartyProduct.LINE -> {
                    tv.athena.thirdparty.impl.line.LineLogin(product)
                }
                ThirdPartyProduct.WEIBO -> {
                    tv.athena.thirdparty.impl.weibo.WeiBoLogin(product)
                }
                ThirdPartyProduct.QQ -> {
                    tv.athena.thirdparty.impl.qq.QQLogin(product)
                }
                ThirdPartyProduct.WECHAT -> {
                    tv.athena.thirdparty.impl.wechat.WeChatLogin(product)
                }
                ThirdPartyProduct.MOBILE_CMCC -> {
                    tv.athena.thirdparty.impl.cmcc.CMCCLogin(product)
                }
                ThirdPartyProduct.MOBILE_CTCC -> {
                    tv.athena.thirdparty.impl.ctcc.CTCCLogin(product)
                }
                ThirdPartyProduct.SNAPCHAT -> {
                    tv.athena.thirdparty.impl.snapchat.SnapChatLogin(product)
                }
//                ThirdPartyProduct.MOBILE -> {
//                    tv.athena.thirdparty.impl.mobile.MobileLogin(product)
//                }
                else -> {
                    UnknownLogin(product)
                }
            }
            if (setCurrent) {
                mCurrentLogin = thirdPartyLogin
            } else {
                return thirdPartyLogin
            }
        }
        return mCurrentLogin
    }

    override fun unBindThirdToken(
        activity: AeFragmentActivity,
        appId: String, deviceId: String, otpTicket: String, region: String, clientSys: String,
        product: ThirdPartyProduct
    ) {
        UnbindLogin(product, appId, deviceId, otpTicket, region, clientSys).unBindProduct(activity)
    }
}