package tv.athena.live.component.business.broadcasting

import com.thunder.livesdk.ThunderEventHandler
import com.thunder.livesdk.ThunderNotification
import com.yy.liveplatform.proto.nano.LpfHeartbeat
import com.yy.liveplatform.proto.nano.LpfMedia
import com.yy.platform.baseservice.task.TaskOptions
import tv.athena.annotation.MessageBinding
import tv.athena.live.api.IDataCallback
import tv.athena.live.api.broadcast.IAudioFilePlayer
import tv.athena.live.api.broadcast.IBroadcastComponentApi
import tv.athena.live.api.broadcast.ILivePublishQualityListener
import tv.athena.live.api.broadcast.bean.AthCatonPromptInfo
import tv.athena.live.api.broadcast.bean.LivePublishQuality
import tv.athena.live.api.broadcast.bean.StartLiveParams
import tv.athena.live.basesdk.config.BaseDataConfig
import tv.athena.live.basesdk.thunderblotwrapper.AbscThunderEventListener
import tv.athena.live.basesdk.thunderblotwrapper.IThunderBlotApi
import tv.athena.live.basesdk.thunderblotwrapper.ThunderHandle
import tv.athena.live.component.business.broadcasting.accessibility.audioFilePlayer.AudioFilePlayerImpl
import tv.athena.live.component.business.broadcasting.repository.BroadcastRepository
import tv.athena.live.utils.ALog
import tv.athena.live.utils.ServiceUtils
import tv.athena.service.api.MessageResponse
import tv.athena.service.api.ServiceFailResult
import tv.athena.service.api.event.ServiceBroadcastStrGroupEvent

/**
 *    create by chenhaofeng 2019-08-13
 *
 *
 */
class BroadcastComponentApiImpl(private val mThunderHandle: ThunderHandle) :
    IBroadcastComponentApi,
    AbscThunderEventListener(),
    IThunderBlotApi by mThunderHandle {

    private var TAG = "BroadcastComponentApiImpl"

    override fun getApiKey(): Class<out IBroadcastComponentApi> {
        return IBroadcastComponentApi::class.java
    }

    private var mBroadcastViewModel: BroadcastViewModel? = null
    private var mBroadcastView: BroadcastView? = null
    private var mBroadcastComponent: BroadcastComponent? = null
    private var iAudioFilePlayer: IAudioFilePlayer? = null
    private var mILivePublishQualityListener: ILivePublishQualityListener? = null
    private var mStartReportLivePublishParams = false

    /**
     * 本地发送视频流编码等信息存储，用于告诉后端当前的状态，判断视频是否卡顿
     * 目前收集到 15个点 即可上报
     */
    //private var mCollectSize = BaseDataConfig.getDefaultPublishMediaParamCount()
    // 收集到第一个点立刻上报
    private var mFirstCollectPoint = 1
    private var mLocalVideoStats = ArrayList<LocalVideoStats>()
    private var hasReqPushCdn = false

    fun setComponent(broadcastComponent: BroadcastComponent) {
        ALog.i(TAG, "setComponent ($broadcastComponent)")
        this.mBroadcastComponent = broadcastComponent
        this.mBroadcastView = broadcastComponent.view
        this.mBroadcastViewModel = broadcastComponent.viewModel
        this.iAudioFilePlayer = AudioFilePlayerImpl(mThunderHandle)
        mThunderHandle.registerThunderEventListener(this@BroadcastComponentApiImpl)
    }

    init {
        /*Sly.subscribe(this)
        mThunderHandle.registerThunderEventListener(this@BroadcastComponentApiImpl)*/
    }

    fun onCreate(broadcastComponent: BroadcastComponent) {
        ALog.i(TAG, "onCreate ($broadcastComponent)")
        setComponent(broadcastComponent)
    }

    override fun onDestroy() {
        mThunderHandle.unRegisterRtcEventListener(this)
        hasReqPushCdn = false
    }

    override fun onLeave() {
        mThunderHandle.unRegisterRtcEventListener(this)
    }

    override fun stopLive(extend: String?, callback: IDataCallback<LpfMedia.EndLiveResp>?) {
        mBroadcastViewModel?.iBroadcast?.stopLive(extend = extend, callback = callback)
    }

    override fun stopLive(callback: IDataCallback<LpfMedia.EndLiveResp>?) {
        iAudioFilePlayer?.destroyAudioFilePlayer()
        mBroadcastViewModel?.iBroadcast?.stopLive(extend = null, callback = callback)
        mBroadcastView?.removeAllView()
    }

    override fun stopLive() {
        stopLive(extend = null, callback = null)
    }

    private var mAthCatonPromptInfo: AthCatonPromptInfo? = null

    override fun onNetworkQuality(uid: String?, txQuality: Int, rxQuality: Int) {
        super.onNetworkQuality(uid, txQuality, rxQuality)
    }

    override fun onRoomStats(stats: ThunderNotification.RoomStats?) {
        super.onRoomStats(stats)
        stats?.let {
        }
    }

    override fun onFirstLocalVideoFrameSent(elapsed: Int) {
        super.onFirstLocalVideoFrameSent(elapsed)
    }

    override fun startLive(
        title: String,
        coverUrl: String,
        extend: String?,
        liveBzType: Int,
        callBack: IDataCallback<Int>?,
        audioSourceType: Int?
    ) {
        mBroadcastViewModel?.iBroadcast?.startLive(title, coverUrl, extend, liveBzType,
            audioSourceType!!, callBack)
    }

    override fun startLive(startLiveParams: StartLiveParams) {
        mBroadcastViewModel?.iBroadcast?.startLive(startLiveParams)
    }

    override fun uploadCoverUrl(imagePath: String, callback: IDataCallback<String>?) {
        mBroadcastViewModel?.iBroadcast?.uploadCoverUrl(imagePath, callback)
    }

    override fun registerAbsThunderEventListener(listener: AbscThunderEventListener) {
        mThunderHandle.registerThunderEventListener(listener)
    }

    override fun unRegisterAbsThunderEventListener(listener: AbscThunderEventListener) {
        mThunderHandle.unRegisterRtcEventListener(listener)
    }

    override fun unRegisterAllAbsThunderEventListener() {
        mThunderHandle.unRegisterAllRtcEventListener()
    }

    override fun takeMic(isEnable: Boolean) {
        mBroadcastViewModel?.iBroadcast?.takeMic(isEnable)
    }

    override fun getAudioFilePlayer(): IAudioFilePlayer? {
        return iAudioFilePlayer
    }

    override fun checkLivePermission(
        liveBzType: Int,
        callBack: IDataCallback<LpfMedia.CheckLivePermissionResp>
    ) {
        mBroadcastViewModel?.iBroadcast?.checkLivePermission(liveBzType, callBack)
    }

    private fun addLocalVideoStats(
        localVideoStats: ThunderEventHandler.LocalVideoStats, streamRoomId: String
    ) {
        mLocalVideoStats.add(localVideoStats)
        when (mLocalVideoStats.size) {
            mFirstCollectPoint -> {
                //如果没有注册到即使开了视频也不上报给后端了
                handleReportLivePublishParams(streamRoomId)
            }
            BaseDataConfig.getDefaultPublishMediaParamCount() -> {
                handleReportLivePublishParams(streamRoomId)
                mLocalVideoStats.clear()
            }
        }
    }

    /**
     * 上报数据到服务器
     */
    private fun handleReportLivePublishParams(streamRoomId: String) {
        if (mStartReportLivePublishParams) {
            reportLivePublishParams(streamRoomId)
        } else {
            ALog.i(TAG,
                "addLocalVideoStats [mStartReportLivePublishParams : $mStartReportLivePublishParams]")
        }
    }

    private fun reportLivePublishParams(streamRoomId: String) {
        val configMediaParam = LpfHeartbeat.ConfigMediaParam()
        val encodeMediaParams = ArrayList<LpfHeartbeat.EncodeMediaParam>()
        var videoCodecType = -1
        mLocalVideoStats.first().let {
            configMediaParam.videoBitrate = it.configBitRate
            configMediaParam.videoFrameRate = it.configFrameRate
            configMediaParam.videoHeight = it.configHeight
            configMediaParam.videoWidth = it.configWidth
            videoCodecType = it.codecType
        }
        mLocalVideoStats.forEach {
            val encodeMediaParam = LpfHeartbeat.EncodeMediaParam()
            encodeMediaParam.transVideoBitrate = it.targetBitRate
            encodeMediaParam.videoBitrate = it.encodedBitrate
            encodeMediaParam.videoHeight = it.encodedFrameHeight
            encodeMediaParam.videoWidth = it.encodedFrameWidth
            encodeMediaParams.add(encodeMediaParam)
        }
        val liveParamMediaParam = LpfHeartbeat.LivePublishMediaParam()
        liveParamMediaParam.configParam = configMediaParam
        liveParamMediaParam.encodeParams = encodeMediaParams.toTypedArray()
        liveParamMediaParam.videoCodecType = videoCodecType
        val reportLivePublishMediaParamReq = LpfHeartbeat.ReportLivePublishMediaParamReq()
        reportLivePublishMediaParamReq.param = liveParamMediaParam
        reportLivePublishMediaParamReq.streamRoomId = streamRoomId
        BroadcastRepository.reportLivePublishMediaParam(reportLivePublishMediaParamReq,
            object :
                ServiceUtils.SvcResultCusRetryCallBack<LpfHeartbeat.ReportLivePublishMediaParamResp>() {
                override fun customBundle() {
                    super.customBundle()
                    val time = BaseDataConfig.getPublishMediaParamTimeout().toLong() * 1000L
                    ALog.i(TAG, "customBundle [time : $time]")
                    mBundle.putLong(TaskOptions.OPT_TIMOUTTS, time)
                }

                override fun get(): LpfHeartbeat.ReportLivePublishMediaParamResp {
                    return LpfHeartbeat.ReportLivePublishMediaParamResp()
                }

                override fun onMessageFail(errorCode: ServiceFailResult, ex: Exception?) {
                    ALog.i(TAG,
                        "reportLivePublishMediaParam onMessageFail $errorCode, ${ex?.message}")
                    safeStrategy(reportLivePublishMediaParamReq)
                }

                override fun onMessageSuccess(
                    response:
                    MessageResponse<LpfHeartbeat.ReportLivePublishMediaParamResp>
                ) {
                    ALog.i(TAG,
                        "reportLivePublishMediaParam onMessageSuccess ${response.message.code}")
                    if (response.message.code == 0) {
                        if (response.message.qualityResult == null) {
                            ALog.i(TAG, "reportLivePublishMediaParam qualityResult == null")
                            return
                        }
                        notifyLivePublishQuality(streamRoomId, response.message.qualityResult)
                    } else {
                        safeStrategy(reportLivePublishMediaParamReq)
                    }
                }
            })
    }

    fun safeStrategy(reportLivePublishMediaParamReq: LpfHeartbeat.ReportLivePublishMediaParamReq) {
        if (reportLivePublishMediaParamReq.param.encodeParams.size == 1) {
            return
        }
        val currentData = reportLivePublishMediaParamReq.param.encodeParams
            .toList()
            .sumBy { it.videoBitrate }
            .toDouble()
        val size = reportLivePublishMediaParamReq.param.encodeParams.toList().size
        val standardData =
            reportLivePublishMediaParamReq.param.configParam.videoBitrate * size.toDouble()
        try {
            val currentCalculation = currentData / standardData
            val standardCalculation = BaseDataConfig.getLocalLivePublishMediaParamCalculaterate()
            val streamRoomId = reportLivePublishMediaParamReq.streamRoomId
            ALog.i(TAG, "safeStrategy [currentData : $currentData]" +
                " [size : $size] [standardData : $standardData] " +
                "[percentage : $currentCalculation] " +
                "[standardCalculaterate : $standardCalculation] " +
                "[streamRoomId : $streamRoomId]")
            when (currentCalculation) {
                in 0.0..standardCalculation -> {
                    val livePublishQualityResult = LpfHeartbeat.LivePublishQualityResult()
                    livePublishQualityResult.catonLevel = LpfHeartbeat.LivePublishQualityResult.HIGH
                    livePublishQualityResult.liveUid = -1L
                    notifyLivePublishQuality(streamRoomId, livePublishQualityResult)
                }
                else -> {
                    val livePublishQualityResult = LpfHeartbeat.LivePublishQualityResult()
                    livePublishQualityResult.catonLevel =
                        LpfHeartbeat.LivePublishQualityResult.NORMAL
                    livePublishQualityResult.liveUid = -1L
                    notifyLivePublishQuality(streamRoomId, livePublishQualityResult)
                }
            }
        } catch (e: Exception) {
            ALog.e(TAG, "safeStrategy [e : ${e.message}]")
        }
    }

    override fun observeLivePublishQuality(
        startReport: Boolean,
        iLivePublishQualityListener: ILivePublishQualityListener?
    ) {
        ALog.i(TAG, "observeLivePublishQuality ($startReport, " +
            "$iLivePublishQualityListener)")
        this.mStartReportLivePublishParams = startReport
        mBroadcastComponent?.componentContext?.commonViewModel?.streamRoomId?.let {
            val groupId = "StreamRoomId_$it"
            val groupIds = HashSet<String>()
            groupIds.add(groupId)
            if (startReport) {
                this.mILivePublishQualityListener = iLivePublishQualityListener
                ServiceUtils.subscribeStrBroadcast(groupIds)
            } else {
                this.mILivePublishQualityListener = null
                ServiceUtils.unSubscribeStrBroadcast(groupIds)
            }
        } ?: kotlin.run {
            ALog.i(TAG, "observeLivePublishQuality failed streamRoomId == null")
        }
    }

    @MessageBinding
    fun onLivePublishQualityResult(event: ServiceBroadcastStrGroupEvent?) {
        event?.let {
            if (it.serverName == BroadcastRepository.FUNC_LPF_HEARTBEAT) {
                val livePublishQualityResultCast = LpfHeartbeat.LivePublishQualityResultCast
                    .parseFrom(event.message)
                val cacheSid = mBroadcastComponent?.componentContext?.commonViewModel?.streamRoomId
                    ?: "-1"
                if (livePublishQualityResultCast.streamRoomId != cacheSid) {
                    return@let
                }
                ALog.i(TAG, "onLivePublishQualityResult $livePublishQualityResultCast")
                notifyLivePublishQuality(streamRoomId = cacheSid,
                    livePublishQualityResult = livePublishQualityResultCast.qualityResult)
            }
        }
    }

    private fun notifyLivePublishQuality(
        streamRoomId: String,
        livePublishQualityResult: LpfHeartbeat.LivePublishQualityResult
    ) {
        var livePublishQuality: LivePublishQuality? = null
        when (livePublishQualityResult.catonLevel) {
            1 -> {
                livePublishQuality =
                    LivePublishQuality(streamRoomId,
                        livePublishQualityResult.liveUid.toString(),
                        LivePublishQuality.CatTonLeval.NORMAL)
            }
            2 -> {
                livePublishQuality =
                    LivePublishQuality(streamRoomId,
                        livePublishQualityResult.liveUid.toString(),
                        LivePublishQuality.CatTonLeval.WEAK)
            }
            3 -> {
                livePublishQuality =
                    LivePublishQuality(streamRoomId,
                        livePublishQualityResult.liveUid.toString(),
                        LivePublishQuality.CatTonLeval.MEDIUM)
            }
            4 -> {
                livePublishQuality =
                    LivePublishQuality(streamRoomId,
                        livePublishQualityResult.liveUid.toString(),
                        LivePublishQuality.CatTonLeval.HIGH)
            }
            5 -> {
                livePublishQuality =
                    LivePublishQuality(streamRoomId,
                        livePublishQualityResult.liveUid.toString(),
                        LivePublishQuality.CatTonLeval.DUMP)
            }
        }
        livePublishQuality?.let {
            if (mStartReportLivePublishParams && mILivePublishQualityListener != null) {
                ALog.i(TAG, "notifyLivePublishQuality $it")
                mILivePublishQualityListener?.notifyLivePublishQuality(it)
            }
        }
    }

    /**
     * ==============sdk 回调 ============
     */

    override fun onLocalVideoStats(stats: ThunderEventHandler.LocalVideoStats) {
        super.onLocalVideoStats(stats)
        mBroadcastComponent?.componentContext?.commonViewModel?.streamRoomId?.let {
            addLocalVideoStats(stats, it)
        }
    }

    override fun onVideoSizeChanged(uid: String?, width: Int, height: Int, rotation: Int) {
        super.onVideoSizeChanged(uid, width, height, rotation)
        //第一帧数据过来需要还原存储视频流信息的集合
        mLocalVideoStats.clear()
        hasReqPushCdn = false
    }

    /**
     * 主播更改开播类型
     *
     * @param liveBzType 开播类型
     */
    override fun changeLiveRoomType(
        liveBzType: Int, callBack: IDataCallback<LpfMedia.ChangeLiveRoomTypeResp>
    ) {
        mBroadcastViewModel?.iBroadcast?.changeLiveRoomType(liveBzType, callBack)
    }

    /**
     * 主播或嘉宾更改自己的媒体类型，接口内部会控制视频流的开启或中断
     *
     * @param mediaType 媒体类型
     *
     * @see LpfMedia.MT_AUD 结音频
     * @see LpfMedia.MT_AV  音视频
     */
    override fun changeLiveMediaType(
        mediaType: Int, callBack: IDataCallback<LpfMedia.UpdateMediaTypeResp>
    ) {
        mBroadcastViewModel?.iBroadcast?.changeLiveMediaType(mediaType, callBack)
    }

    private var mVideoFrameListener: VideoFrameListener? = null

    open interface VideoFrameListener {
        fun onFrameHandle(frameTime: Double, beautyFrameTime: Long, encodeTime: Long, maxTime: Long)

        /**
         * 是否开启帧率计算和收集
         */
        fun isOpen(): Boolean
    }
}