/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate;

import com.hyphenate.chat.EMGroupReadAck;
import com.hyphenate.chat.EMMessage;
import com.hyphenate.chat.EMMessagePinInfo;
import com.hyphenate.chat.EMMessageReactionChange;
import com.hyphenate.chat.EMRecallMessageInfo;

import java.util.List;

/**
 * \~chinese
 * 消息事件监听器。
 * 用于监听消息接收情况，消息成功发送到对方手机后会有回执（需开启送达回执，详见 {@link com.hyphenate.chat.EMOptions#setRequireDeliveryAck(boolean)}）。
 * 对方阅读了这条消息也会收到回执（需开启允许已读回执，详见 {@link com.hyphenate.chat.EMOptions#setRequireAck(boolean)}）。
 * 发送消息过程中，消息 ID 会从最初本地生成的 uuid 变更为服务器端生成的全局唯一 ID，该 ID 在使用 SDK 的所有设备上均唯一。
 * 应用需实现此接口监听消息变更状态。
 * 
 *     添加消息监听：
 *     ```java
 *     EMClient.getInstance().chatManager().addMessageListener(mMessageListener);
 *     ```
 *
 *     移除消息监听：
 *     ```java
 *     EMClient.getInstance().chatManager().removeMessageListener(mMessageListener);
 *     ```
 * 
 *
 * \~english
 * The message event listener.
 * This listener is used to check whether messages are received. If messages are sent successfully, a delivery receipt will be returned (delivery receipt needs to be enabled: {@link com.hyphenate.chat.EMOptions#setRequireDeliveryAck(boolean)}.
 * If the peer reads the received message, a read receipt will be returned (read receipt needs to be enabled: {@link com.hyphenate.chat.EMOptions#setRequireAck(boolean)})
 * During message delivery, the message ID will be changed from a local uuid to a global unique ID that is generated by the server to uniquely identify a message on all devices using the SDK.
 * This API should be implemented in the app to listen for message status changes.
 * 
 *     Adds the message listener:
 *     ```java
 *     EMClient.getInstance().chatManager().addMessageListener(mMessageListener);
 *     ```
 *
 *     Removes the message listener:
 *     ```java
 *     EMClient.getInstance().chatManager().removeMessageListener(mMessageListener);
 *     ```
 * 
 */
public interface EMMessageListener
{
	/**
     * \~chinese
	 * 收到消息。
	 * 在收到文本、图片、视频、语音、地理位置和文件等消息时，通过此回调通知用户。
     *
     * \~english
	 * Occurs when a message is received.
	 * This callback is triggered to notify the user when a message such as texts or an image, video, voice, location, or file is received.
	 */
	void onMessageReceived(List<EMMessage> messages);

	/**
     * \~chinese
	 * 收到命令消息。
	 * 与 {@link #onMessageReceived(List)} 不同, 这个回调只包含命令的消息，命令消息通常不对用户展示。
	 *
     * \~english
	 * Occurs when a command message is received.
	 * Unlike {@link EMMessageListener#onMessageReceived(List)}, this callback only contains a command message body that is usually invisible to users.
	 *
	 */
	default void onCmdMessageReceived(List<EMMessage> messages) {
	}

    /**
     * \~chinese
     * 收到消息的已读回执。
     *
     * \~english
     * Occurs when a read receipt is received for a message. 
     */
	default void onMessageRead(List<EMMessage> messages) {}

	/**
	 * \~chinese
	 * 收到群组消息的已读回执。
	 *
	 * \~english
	 * Occurs when a read receipt is received for a group message.
	 */
	default void onGroupMessageRead(List<EMGroupReadAck> groupReadAcks) {}

	/**
	 * \~chinese
	 * 收到群组消息的读取状态更新。
	 *
	 * \~english
	 * Occurs when the update for the group message read status is received.
	 */
	default void onReadAckForGroupMessageUpdated() {}

    /**
     * \~chinese
     * 收到消息的送达回执。
     *
     * \~english
     * Occurs when a delivery receipt is received.
     */
	default void onMessageDelivered(List<EMMessage> messages){}

	/**
 	 * \~chinese
 	 * 撤回收到的消息。
 	 *
 	 * \~english
 	 * Occurs when a received message is recalled.
 	 */
	@Deprecated
	default void onMessageRecalled(List<EMMessage> messages){}

	/**
	 * \~chinese
	 * 撤回收到的消息。
	 * 
	 * 支持使用 `onMessageRecalledWithExt`，因为该回调接收在离线期间被撤回的消息。
	 * 
	 * @param recallMessageInfo 被撤回消息的详情 注: 接收方离线期间被撤回消息 `recallMessage` 参数的值为 null。
	 *
	 * \~english
	 * Occurs when a received message is recalled. 
	 * 
	 * `onMessageRecalledWithExt` is recommended as it can return the message that is recalled when the recipient is offline.
	 * 
	 * This event is triggered regardless of whether the recipient is online or offline during message recall.
	 * 
	 * @param recallMessageInfo The details of the recalled message. If the message recipient is offline,
	 * the value of the `recallMessage` parameter is null.
	 *
	 */
	default void onMessageRecalledWithExt(List<EMRecallMessageInfo> recallMessageInfo){}
    
	/**
     * \~chinese
     * 收到消息变化，包括消息 ID 的变化。
	 * 
     * @param message	修改后的消息。
     * @param change    消息的变化。
     *
     * \~english
     * Occurs when a message modification is received, including the changed message ID.
     *
     * @param message   The modified message.
     * @param change    The message modification.
     */
	default void onMessageChanged(EMMessage message, Object change) {}

	/**
	 * \~chinese
	 * 收到 Reaction 变更消息。
	 *
	 * \~english
	 * Occurs when a message Reaction is changed.
	 * 
	 */
	default void onReactionChanged(List<EMMessageReactionChange> messageReactionChangeList) {}

	/**
	 * \~chinese
	 * 收到消息内容变化。
	 * 
	 * @param messageModified 修改的消息对象，其中的 message body 包含消息修改次数、最后一次修改的操作者、最后一次修改时间等信息。
	 *               你也可通过 `onMessageContentChanged` 回调获得最后一次修改的操作者和最后一次修改时间等信息。
	 * @param operatorId 最后一次修改消息的用户。
	 * @param operationTime 消息的最后一次修改时间戳，单位为毫秒。
	 *
	 * \~english
	 * Occurs when the message content is modified.
	 * 
	 * @param messageModified  The modified message object, where the message body contains the information such as the number of message modifications, the operator of the last modification, and the last modification time.
	 * 	Also, you can get the operator of the last message modification and the last modification time via the `onMessageContentChanged` method.
	 * @param operatorId The user ID of the operator that modified the message last time.
	 * @param operationTime The last message modification time. It is a UNIX timestamp in milliseconds.
	 */
	default void onMessageContentChanged(EMMessage messageModified, String operatorId, long operationTime){
	}
	/**
	 * \~chinese
	 * 收到消息的置顶状态变化。
	 *
	 * @param messageId	消息 ID。
	 * @param conversationId 会话 ID。
	 * @param pinOperation 	消息置顶操作的类型：包含置顶和取消置顶。详见 {@link EMMessagePinInfo.PinOperation}。
	 * @param pinInfo 消息置顶详情，包含置顶操作者的用户 ID 和置顶时间等。
	 *
	 * \~english
	 * Occurs when the pinning status of a message is changed.
	 *
	 * @param messageId	 The ID of the message to be pinned or unpinned.
	 * @param conversationId  The ID of the conversation.
	 * @param pinOperation   The message pinning or unpinning operation. For more information, see {@link EMMessagePinInfo.PinOperation}.
	 * @param pinInfo   The message pinning information, including the ID of the operator who pinned the message and time when the message is pinned.
	 *
	 */
	default void onMessagePinChanged(String messageId, String conversationId, EMMessagePinInfo.PinOperation pinOperation,EMMessagePinInfo pinInfo) {
	}
}
