package com.hyphenate.push;

import com.hyphenate.push.common.PushUtil;

/**
 * \~chinese
 * 用于设置自定义的推送设置。
 *
 * \~english
 * The push settings class for customized push.
 */
public abstract class PushListener {

    public abstract void onBindTokenSuccess(EMPushType pushType,String pushToken);
    public abstract void onError(EMPushType pushType, long errorCode);

    /**
     * \~chinese
     * 判断是否支持某个类型的推送。
     *
     * 该函数会回调多次，每次向用户传递一个 pushType，旨在让开发者根据 SDK 提供的 pushType 返回一个布尔值，用于 SDK 判断客户端环境是否支持该类推送。
     * 例如:
     * 1.如果想使用FCM推送，需要调用EMPushHelper.getInstance().setPushListener()设置监听，并重写isSupportPush函数，在参数pushType == ChatPushType.FCM时返回true。
     * ```java
     * EMPushHelper.getInstance().setPushListener(new PushListener() {
     *                  ……
     *                 @Override
     *                 public boolean isSupportPush(EMPushType pushType, EMPushConfig pushConfig) {
     *                     // 由用户实现代码判断设备是否支持FCM推送
     *                     if(pushType == EMPushType.FCM){
     *                         EMLog.d("FCM", "GooglePlayServiceCode:"+GoogleApiAvailabilityLight.getInstance().isGooglePlayServicesAvailable(context));
     *                         return  GoogleApiAvailabilityLight.getInstance().isGooglePlayServicesAvailable(context) == ConnectionResult.SUCCESS;
     *                     }else if (pushType == EMPushType.HONORPUSH){
     *                        ……
     *                     }
     *                     return super.isSupportPush(pushType, pushConfig);
     *                 }
     *             });
     * ```
     * 同理，如果对其他某个特定的 pushType (例如 HONORPUSH) 返回 true，SDK 会采用该类型的推送，该函数后续将不再回调询问。
     *
     * 注意：最终是否能注册成功还得取决于客户环境是否集成了相关推送套件。
     *
     * 2.若在isSupportPush回调函数里对所有类型的pushType都返回 false，最终走环信通道。
     *
     * 3.若不实现该函数，则 SDK 默认根据设备机型来判断支持哪个厂商的推送实现。
     *
     * @param pushType 推送类型。
     * @param pushConfig 推送配置。
     * @return 返回是否支持该类型的推送。
     *
     * \~english
     * Determines whether a push type is supported.
     * This function will be called multiple times, each time passing a pushType to the user to return a boolean value based on the pushType provided by the SDK, allowing the SDK to determine whether the client environment supports the push type.
     * For example:
     * 1. If you want to use FCM push, you need to call EMPushHelper.getInstance().setPushListener() to set the listener and override the isSupportPush function, returning true when the parameter pushType == ChatPushType.FCM.
     * ```java
     * EMPushHelper.getInstance().setPushListener(new PushListener() {
     *                  ……
     *                 @Override
     *                 public boolean isSupportPush(EMPushType pushType, EMPushConfig pushConfig) {
     *                     // 由用户实现代码判断设备是否支持FCM推送
     *                     if(pushType == EMPushType.FCM){
     *                         EMLog.d("FCM", "GooglePlayServiceCode:"+GoogleApiAvailabilityLight.getInstance().isGooglePlayServicesAvailable(context));
     *                         return  GoogleApiAvailabilityLight.getInstance().isGooglePlayServicesAvailable(context) == ConnectionResult.SUCCESS;
     *                     }else if (pushType == EMPushType.HONORPUSH){
     *                        ……
     *                     }
     *                     return super.isSupportPush(pushType, pushConfig);
     *                 }
     *             });
     * ```
     * Similarly, if you return true for a specific pushType (such as HONORPUSH), the SDK will use that type of push, and the function will not be called again.
     * Note: The final success of registration depends on whether the client environment has integrated the relevant push suite.
     * 2. If you return false for all pushTypes in the isSupportPush callback function, the final push will be through the Huanxin channel.
     * 3. If this function is not implemented, the SDK will default to determining which push implementation is supported based on the device model.
     *
     * @param pushType The push type.
     * @param pushConfig The push configurations.
     * @return Returns whether the push type is supported.
     */
    public boolean isSupportPush(EMPushType pushType, EMPushConfig pushConfig) {
        return PushUtil.isSupportPush(pushType, pushConfig);
    }

    /**
     * \~chinese
     * 获取 FCM 设备 token。
     * @param pushType      推送类型。
     * @param pushConfig    推送配置。
     * @return              返回要注册的设备 token。
     *
     * \~english
     * Gets the FCM device token.
     * @param pushType      The push type.
     * @param pushConfig    The push configs.
     * @return              Returns the device token to register.
     */
    public String getPushToken(EMPushType pushType,EMPushConfig pushConfig) {
        return "";
    }
}
