/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import static com.hyphenate.util.EasyUtils.convertToCerts;
import static com.hyphenate.util.EasyUtils.getSystemDefaultTrustManager;

import android.annotation.TargetApi;
import android.app.Activity;
import android.app.Application;
import android.app.Application.ActivityLifecycleCallbacks;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.http.X509TrustManagerExtensions;
import android.os.Build;
import android.os.Bundle;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMConnectionListener;
import com.hyphenate.EMError;
import com.hyphenate.EMLogListener;
import com.hyphenate.EMMultiDeviceListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatClient;
import com.hyphenate.chat.adapter.EMAChatClient.EMANetwork;
import com.hyphenate.chat.adapter.EMAConnectionListener;
import com.hyphenate.chat.adapter.EMADeviceInfo;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMALogCallbackListener;
import com.hyphenate.chat.adapter.EMALogoutInfo;
import com.hyphenate.chat.adapter.EMAMultiDeviceListener;
import com.hyphenate.chat.adapter.EMANetCallback;
import com.hyphenate.chat.core.EMChatConfigPrivate;
import com.hyphenate.chat.core.EMPreferenceUtils;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.monitor.EMNetworkMonitor;
import com.hyphenate.notification.core.EMNotificationHelper;
import com.hyphenate.push.EMPushConfig;
import com.hyphenate.push.EMPushHelper;
import com.hyphenate.push.EMPushType;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.NetUtils;
import com.hyphenate.util.PathUtil;
import com.hyphenate.util.Utils;

import org.json.JSONException;
import org.json.JSONObject;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Pattern;

import javax.net.ssl.X509TrustManager;

import internal.com.getkeepsafe.relinker.ReLinker;

/**
  * \~chinese
  * Chat 客户端类。该类是 Chat SDK 的入口，负责登录、登出及管理 SDK 与 chat 服务器之间的连接。
  * 
  * ```java
  * EMChatManager chatManager = EMClient.getInstance().chatManager();
  * ```
  *
  * \~english
  * The chat client class, which is the entry of the chat SDK. It defines how to log in to and log out of the chat app and how to manage the connection between the SDK and the chat server.
  *  
  * ```java
  * EMChatManager chatManager = EMClient.getInstance().chatManager();
  * ```
  */
public class EMClient {
    public final static String TAG = "EMClient";
    private volatile static EMClient instance = null;
    static boolean libraryLoaded = false;

    private EMGroupManager groupManager;
    private EMChatRoomManager chatroomManager;
    private EMChatManager chatManager;
    private EMContactManager contactManager;
    private EMUserInfoManager userInfoManager;
    private EMPushManager pushManager;
    private volatile EMPresenceManager presenceManager;
    private EMChatThreadManager threadManager;
    private EMStatisticsManager statisticsManager;

    private EMAChatClient emaObject;
    private Context mContext;
    private ExecutorService executor = null;
    private ExecutorService logQueue = Executors.newSingleThreadExecutor();
    private ExecutorService mainQueue = Executors.newSingleThreadExecutor();
    private ExecutorService sendQueue = Executors.newSingleThreadExecutor();
    private EMEncryptProvider encryptProvider = null;
    private boolean sdkInited = false;
    private EMChatConfigPrivate mChatConfigPrivate;
    private List<EMConnectionListener> connectionListeners = Collections.synchronizedList(new ArrayList<EMConnectionListener>());
    private List<EMLogListener> logListeners = Collections.synchronizedList(new ArrayList<EMLogListener>());
    private ClientLogListener clientLogListener;

	private MyConnectionListener connectionListener;
	private EMSmartHeartBeat smartHeartbeat = null;
    private List<EMMultiDeviceListener> multiDeviceListeners = Collections.synchronizedList(new ArrayList<EMMultiDeviceListener>());
    private MyMultiDeviceListener multiDeviceListenerImpl;
	private WakeLock wakeLock;

	private ConnectivityManager connManager;
	private EMANetwork currentNetworkType = EMANetwork.NETWORK_NONE;
    private boolean mIsLoginWithToken = false;
    private boolean mIsDBOpened = false;

	public final static String VERSION = "4.15.3";

	/**
     * \~chinese
     * 请确保 EMClient 由 IM SDK 初始化。
     * 
     * \~english
	 * Make sure that this EMClient is initialized by the Chat SDK.
	 *
	 */
	private EMClient() {
	}

	public static EMClient getInstance() {
		if(instance == null){
		    synchronized (EMClient.class) {
		        if(instance == null) {
                    instance = new EMClient();
		        }
            }
		}
		return instance;
	}


    /**
    * \~chinese
    * 初始化 SDK。
    * 
    * 请确保在主进程中进行初始化。
    * 
    * @param context 上下文，必填。
    * @param options 配置项，必填。详见 {@link EMOptions}。
    *
    * \~english
    * Initializes the SDK.
    * 
    * Make sure to initialize the SDK in the main thread.
    *
    * @param context The context. Make sure to set the parameter.
    * @param options The configuration options. Make sure to set the parameter. See {@link EMOptions}.
    */
    public synchronized void init(Context context, EMOptions options) {
        if(sdkInited){
            return;
        }

        executor = Executors.newCachedThreadPool();

        final EMTimeTag tag = new EMTimeTag();
        tag.start();

        mContext = context.getApplicationContext();
        connManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        //if android sdk >= 14,register callback to know app enter the background or foreground
        registerActivityLifecycleCallbacks();

        loadLibrary(options);

        mChatConfigPrivate = new EMChatConfigPrivate();
        mChatConfigPrivate.load(mContext, options);
        options.setConfig(mChatConfigPrivate);
        // multi-device is enabled by setDeviceUuid
        // config setDeviceUuid before EMAChatClient.create being called
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(mContext);
        mChatConfigPrivate.setDeviceUuid(deviceFactory.getDeviceUuid().toString());
        String deviceName= Build.MANUFACTURER + Build.MODEL;
        //The custom device name may override the system device name, and record the actual device with an error level to facilitate bug tracing
        EMLog.e(TAG,"system device name :"+deviceName);
        if(TextUtils.isEmpty(options.getCustomDeviceName())) {
            mChatConfigPrivate.setDeviceName(deviceName);
        }
        mChatConfigPrivate.setDid(getDidInfo());
        mChatConfigPrivate.setServiceId(UUID.randomUUID().toString());

        // Init push relevant.
        EMPushConfig pushConfig = options.getPushConfig();
        if (pushConfig == null) {
            pushConfig = new EMPushConfig.Builder(mContext).build();
        }

        EMPushConfig.Builder builder = new EMPushConfig.Builder(mContext, pushConfig);
        EMPushHelper.getInstance().init(mContext, builder.build());

        emaObject = EMAChatClient.create(mChatConfigPrivate.emaObject);
        // sandbox
        // rest 103.241.230.122:31080 im-msync 103.241.230.122:31097
//        mChatConfigPrivate.setChatServer("118.193.28.212");
//        mChatConfigPrivate.setChatPort(31097);
//        mChatConfigPrivate.setRestServer("https://118.193.28.212:31443");
//        mChatConfigPrivate.enableDnsConfig(false);

        connectionListener = new MyConnectionListener();
        emaObject.addConnectionListener(connectionListener);

        multiDeviceListenerImpl = new MyMultiDeviceListener();
        emaObject.addMultiDeviceListener(multiDeviceListenerImpl);

        // init all the managers
        initManagers();

        mIsLoginWithToken =EMSessionManager.getInstance().isLastLoginWithToken();

        final String lastLoginUser = EMSessionManager.getInstance().getLastLoginUser();

        EMLog.e(TAG, "is autoLogin : " + options.getAutoLogin());
        EMLog.e(TAG, "lastLoginUser : " + lastLoginUser);
        EMLog.e(TAG, "hyphenate SDK is initialized with version : " + getChatConfigPrivate().getVersion());

        PowerManager pm = (PowerManager) mContext.getSystemService(Context.POWER_SERVICE);
        wakeLock = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "emclient");
        EMNotificationHelper.getInstance().onInit(mContext);
        sdkInited = true;

        mContext.registerReceiver(connectivityBroadcastReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
        onNetworkChanged();//主动更新一次，上面注册由于异步原因不会及时更新.数据上报需要

        if (options.getAutoLogin() && isLoggedInBefore()) {
                final String lastLoginToken = EMSessionManager.getInstance().getLastLoginToken();
                final String lastLoginPwd = EMSessionManager.getInstance().getLastLoginPwd();

                //sessionManager.login(lastLoginUser, lastLoginPwd,false, null);
                EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);

                final EMCallBack callback = new EMCallBack() {

                    @Override
                    public void onSuccess() {
                        EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);
                        EMLog.d(TAG, "hyphenate login onSuccess");
                        tag.stop();
                        EMLog.d(TAG, "[Collector][sdk init]login time is : " + tag.timeStr());
                    }

                    @Override
                    public void onError(int code, String error) {
                        EMLog.d(TAG, "hyphenate login onError:" + code + " error:" + error);
                        tag.stop();
                        EMLog.d(TAG, "[Collector][sdk init]login failed time is : " + tag.timeStr());
                    }

                    @Override
                    public void onProgress(int progress, String status) {
                    }
                };

                _login(lastLoginUser,
                        mIsLoginWithToken ? lastLoginToken : lastLoginPwd,
                        callback,
                        /*autoLogin*/ true,
                        /*loginWithToken*/ mIsLoginWithToken ? EMLoginType.LOGIN_TOKEN : EMLoginType.LOGIN_PASSWORD);

        } else {
            tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
        }
    }

    private String getDidInfo() {
        String manufacturer =Build.MANUFACTURER;
        String model = Build.MODEL;
        String hardware = Build.HARDWARE;
        int sdkInt = Build.VERSION.SDK_INT;
        String osVersion = Build.VERSION.RELEASE;

        String did=manufacturer+"/"
                +model+"/"
                +hardware+"/"
                +sdkInt+"/"
                +osVersion;
        return did;
    }

    /**
     * \~chinese
     * 创建账号。
     *
     * 该方法不推荐使用，建议调用相应的 RESTful 方法。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username 用户 ID。
     *                 该参数必填。用户 ID 不能超过 64 个字符，支持以下类型的字符：
     *                 - 26 个小写英文字母 a-z；
     *                 - 26 个大写英文字母 A-Z；
     *                 - 10 个数字 0-9；
     *                 - "_", "-", "."
     *                 - 如果使用正则表达式，则可以将表达式写为：^[a-zA-Z0-9_-]+$。
     *
     *                 用户 ID 不区分大小写，大写字母会自动转换为小写字母。
     *
     *                 用户的电子邮件地址和 UUID 不能作为用户 ID。
     *
     *                 可通过以下格式的正则表达式设置用户 ID：^[a-zA-Z0-9_-]+$。
     *
     * @param password 密码，长度不超过 64 个字符，必填。
     *
     * @throws HyphenateException 如果该方法调用失败，会包含调用失败的原因。例如 username（账号），（password）密码为空，或者账号不符合要求。
     *                           
     *
     * \~english
     * Adds a new user account.
     *
     * This method is not recommended and you are advised to call the RESTful API.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID. The maximum length is 64 characters. Ensure that you set this parameter. The user ID can contain characters of the following types:
   *                 - 26 English letters (a-z).
   *                 - 10 numbers (0-9).
   *                 - "_", "-", "."
   *                 The user ID is case-insensitive, so Aa and aa are the same user ID.
   *                 The email address or the UUID of the user cannot be used as the user ID.
   *                 You can also set this parameter with the regular expression ^[a-zA-Z0-9_-]+$.
     * @param password The password. The maximum length is 64 characters. Ensure that you set this parameter.
     *                 
     *
     * @throws HyphenateException A description of the issue that caused this call to fail. For example, the user account or password is null, or the account is illegal.
     *                 
     */
    public void createAccount(String username, String password) throws HyphenateException {
        username = username.toLowerCase();

        Pattern pattern = Pattern.compile("^[a-zA-Z0-9_.-]+$");
        boolean find = pattern.matcher(username).find();
        if (!find) {
            throw new HyphenateException(EMError.USER_ILLEGAL_ARGUMENT, "illegal user name");
        }
        EMAError error = emaObject.createAccount(username, password);
        handleError(error);
    }

    /**
     * \~chinese
     * 使用密码登录服务器。
     *
     * 异步方法。
     *
     * @param id          用户 ID，必填。
     * @param password    用户密码，必填。
     * @param callback    登录回调，必填。登录的结果将通过回调返回。
     * @deprecated 该方法已废弃，请使用 {@link #loginWithToken(String, String, EMCallBack)} 方法通过 token 登录服务器。
     *
     * \~english
     * Logs in to the chat server with a password.
     *
     * This is an asynchronous method.
     *
     * @param id 		The unique chat user ID. Make sure to set the parameter.
     * @param password 	The password. Make sure to set the parameter.
     * @param callback 	The login callback. Make sure to set the parameter. The login result is returned via the callback.
     * @deprecated Deprecated. Please use {@link #loginWithToken(String, String, EMCallBack)} instead.
     */
    @Deprecated
    public void login(String id, String password, @NonNull final EMCallBack callback) {

        if (id == null || password == null || id.equals("") || password.equals("")) {
            callback.onError(EMError.INVALID_PARAM, "username or password is null or empty!");
            return;
        }

        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey()) && TextUtils.isEmpty(getChatConfigPrivate().getAppId())) {
            callback.onError(EMError.INVALID_PARAM, "please setup your App Key or AppId via Options");
            return;
        }

        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        id = id.toLowerCase();
        initLoginWithToken(false,"",0);
        _login(id, password, callback, false, EMLoginType.LOGIN_PASSWORD);
    }

    /**
     * \~chinese
     * 通过 token 登录服务器。
     * 
     * 该方法支持自动登录。
     *
     * 通过密码登录服务器，见 {@link #login(String, String, EMCallBack)}。
     *
     * 异步方法。
     *
     * @param username 用户 ID，必填。
     * @param token    验证 token，必填。
     * @param callback 登录回调，返回登录结果。必填且不能为 `null`。登录的结果将通过 callback 返回。
     *
     * \~english
     * Logs in to the chat server with the user ID and token.
     * 
     * This method supports automatic login.
     *
     * You can also call {@link #login(String, String, EMCallBack)} to log in to the chat server with the user ID and password.
     *
     * This is an asynchronous method.
     *
     * @param username The user ID. Make sure to set the parameter.
     * @param token    The user token. Make sure to set this parameter.
     * @param callback The login callback. Make sure to set the parameter. Also, this parameter cannot be `null`. The result of login is returned via the callback.
     */
    public void loginWithToken(String username, String token, @NonNull final EMCallBack callback) {
        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey()) && TextUtils.isEmpty(getChatConfigPrivate().getAppId())) {
            callback.onError(EMError.INVALID_PARAM, "please setup your App Key or AppId via Options");
            return;
        }

        if (username == null || token == null || username.equals("") || token.equals("")) {
            callback.onError(EMError.INVALID_PARAM, "username or token is null or empty!");
            return;
        }

        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        username = username.toLowerCase();
        initLoginWithToken(true,"",0);
        _login(username, token, callback, false, EMLoginType.LOGIN_TOKEN);
    }

    /**
     * \~chinese
     * 通过用户 ID 和声网 Agora token 登录 chat 服务器。
     * 
     * 该方法支持自动登录。
     *
     * 通过用户 ID 和密码登录 chat 服务器，见{@link #login(String, String, EMCallBack)}。
     *
     * 异步方法。
     *
     * @param username      用户 ID。不能为空。
     * @param agoraToken    验证 Agora token。不能为空。
     * @param callback      EMCallback 回调函数。不能为空。登录的结果将通过 callback 返回。
     * @deprecated 已废弃，请使用 {@link #loginWithToken(String, String, EMCallBack)} 替代.
     *
     * \~english
     * Logs in to the chat server with the user ID and Agora token.
     * 
     * This method supports automatic login.
     *
     * An app user can also log in to the chat server with the user ID and token. See {@link #login(String, String, EMCallBack)}.
     *
     * This an asynchronous method.
     *
     * @param username      The user ID. Make sure to set the parameter.
     * @param agoraToken    The Agora token. Make sure to set the parameter.
     * @param callback      The login callback. Make sure to set the parameter. The result of login is returned via the callback.
     * @deprecated Deprecated. Please use {@link #loginWithToken(String, String, EMCallBack)} instead.
     */
    @Deprecated
    public void loginWithAgoraToken(String username, String agoraToken, @NonNull final EMCallBack callback){
        loginWithToken(username, agoraToken,callback);
    }

    private synchronized void initLoginWithToken(boolean isLoginWithToken, String expireTimestamp, long tokenAvailablePeriod) {
        mIsLoginWithToken =isLoginWithToken;
        if(isLoginWithToken) {
            //存储
            EMSessionManager.getInstance().setLoginWithTokenData(isLoginWithToken,expireTimestamp,tokenAvailablePeriod);
            EMSessionManager.getInstance().startCountDownTokenAvailableTime(connectionListener);
        }else{
            EMSessionManager.getInstance().clearLoginWithTokenData();
        }
    }

    /**
     * \~chinese
     *
     * 通知 token 过期。(内部使用)
     *
     * SDK 将token 过期事件回调给 connectionListener。如果用户注册了 connectionListener，则会收到 token 过期事件。

     *
     * @param response 请求返回的结果。
     *
     * \~english
     *
     * Notifies that the token expires.
     *
     * The SDK triggers the token expiry notification callback via connectionListener.
     *
     * @param response The request result, which includes the description of the issue that cause the method fails.
     */
    public void notifyTokenExpired(String response){
        //只在 Token 方式登录下才通知。
        if(connectionListener!=null&& mIsLoginWithToken) {
            //解析
            try {
                JSONObject object = new JSONObject(response);
                if(object!=null) {
                    String errorDescription = object.optString("error_description");
                    EMLog.e(TAG,"notifyTokenExpired--errorDescription:"+errorDescription);
                    if(errorDescription.contains("milliseconds ago")||errorDescription.contains("has expired")||
                            errorDescription.contains("Unable to authenticate due to expired access Token")) {
                        connectionListener.onTokenNotification(401);
                        EMLog.e(TAG,"notifyTokenExpired--onTokenNotification(401) ");
                    }
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }
    /**
     * \~chinese
     *
     * 更新 Agora token。
     *
     * 当用户利用声网 token 登录的情况下在 {@link EMConnectionListener} 类中收到 token 即将过期事件的回调通知时，可调用该方法更新 token，避免因 token 失效产生的未知问题。
     *
     * @param newAgoraToken 声网 token。
     * @deprecated 已废弃，请使用 {@link #renewToken(String, EMCallBack)} 替代.
     *
     * \~english
     *
     * Renews the Agora token.
     *
     * If you log in with an Agora token and are notified by a callback method {@link EMConnectionListener} that the token is to be expired, you can call this method to update the token to avoid unknown issues caused by an invalid token.
     *
     * @param newAgoraToken The new token.
     * @deprecated Deprecated. Please use {@link #renewToken(String, EMCallBack)} instead.
     */
    @Deprecated
    public void renewToken(String newAgoraToken){
        renewToken(newAgoraToken,null);
    }

    /**
     * \~chinese
     *
     * 更新  token。
     *
     * 当用户利用 token 登录的情况下在 {@link EMConnectionListener} 类中收到 token 即将过期事件的回调通知时，可调用该方法更新 token，避免因 token 失效产生的未知问题。
     *
     * @param newToken  新token。
     * @param callback  EMCallback 回调函数,不能为空，更新token的结果将通过callback返回
     *
     * \~english
     *
     * Renews the Agora token.
     *
     * If you log in with a token and are notified by a callback method {@link EMConnectionListener} that the token is to be expired, you can call this method to update the token to avoid unknown issues caused by an invalid token.
     *
     * @param newToken  The new token.
     * @param callback  renew token callback. Make sure to set the parameter. The result of renew token is returned via the callback.
     */
    public void renewToken(String newToken, @NonNull EMCallBack callback){
        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "SDK not initialized");
            return;
        }
        if(!emaObject.isLoggedIn()) {
            callback.onError(EMError.USER_NOT_LOGIN, "User has not loggin.");
            return;
        }
        if(TextUtils.isEmpty(newToken)) {
            callback.onError(EMError.INVALID_TOKEN, "New token is invalid");
            return;
        }
        execute(() -> {
            EMAError error = new EMAError();
            long expireTimestamp = emaObject.getTokenExpiredTs(newToken,error);
            if (error.errCode() == EMError.EM_NO_ERROR) {
                EMLog.d(TAG, "getTokenExpiredTs success");
                //更新 Token
                emaObject.renewToken(newToken);
                long tokenAvailablePeriod=expireTimestamp-System.currentTimeMillis();
                initLoginWithToken(true,String.valueOf(expireTimestamp),tokenAvailablePeriod);
                saveToken();
                //打日志
                callback.onSuccess();
                EMLog.d(TAG, "renewToken(): expireTimestamp = " + expireTimestamp + " tokenAvailablePeriod = " + tokenAvailablePeriod);
            } else {
                callback.onError(error.errCode(), error.errMsg());
                EMLog.e(TAG, "getChatTokenbyAgoraToken failed error:" + error.errCode() + "  errorMessage:" + error.errMsg());
            }
        });
    }

    /**
     * \~chinese
     * 退出登录。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param unbindToken 登出时是否解绑 token。
     *                    - `true`：是；
     *                    - `false`：否
     * @return 登出结果：
     *         - 返回值为 0，即 {@link EMError#EM_NO_ERROR}，表示退出登录成功；
     *         - 其他值为退出失败，详见 {@link EMError}。
     *
     * \~english
     * Logs out of the chat app.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param unbindToken Whether to unbind the token upon logout.
     *                    - `true`: Yes.
         *                - `false`: No.
     * @return The logout result.
     *         - If success, the SDK returns {@link EMError#EM_NO_ERROR} if the user successfully logs out;
     *         - If a failure occurs, the SDK returns the description of the cause to the failure. See {@link EMError}.
     */
    public int logout(boolean unbindToken) {
        if (!emaObject.isLogout()) {
            // 如果用户（跨平台）调用了bindDeviceToken，解绑时执行下面的逻辑
            String pushToken = EMPreferenceUtils.getInstance().getPushToken();
            String pushNotifierName = EMPreferenceUtils.getInstance().getPushNotifierName();
            if(!TextUtils.isEmpty(pushToken) && !TextUtils.isEmpty(pushNotifierName)) {
                try {
                    if (unbindToken) {
                        pushManager().unBindDeviceToken();
                    } else {
                        EMLog.d(TAG, "logout:"+ " don't unbind token");
                    }
                } catch (Exception e) {
                    return EMError.USER_UNBIND_DEVICETOKEN_FAILED;
                }
            }else {
                // 原生android默认执行下面的逻辑来解绑
                boolean result = EMPushHelper.getInstance().unregister(unbindToken);
                if (!result) {
                    return EMError.USER_UNBIND_DEVICETOKEN_FAILED;
                }
            }
        } else {
            EMPushHelper.getInstance().unregister(false);
            EMLog.e(TAG, "already logout, skip unbind token");
        }

        logout();

        return EMError.EM_NO_ERROR;
    }

    /**
     * \~chinese
     * 登出聊天服务器。
     * 
     * 同步方法，会阻塞当前线程。
     * 
     * 如果需要异步请参考 {@link EMClient#logout(EMCallBack callback)}。
     *
     * \~english
     * Logs out of the chat server.
     * 
     * This is a synchronous method and blocks the current thread.
     * 
     * For the asynchronous method, see {@link EMClient#logout(EMCallBack callback)}.
     */
    void logout() {
        EMLog.d(TAG, " SDK Logout");

//        try {
//            if (connectivityBroadcastReceiver != null) {
//                mContext.unregisterReceiver(connectivityBroadcastReceiver);
//            }
//        } catch (Exception e) {
//        }

        EMSessionManager.getInstance().clearLastLoginUser();
        EMSessionManager.getInstance().clearLastLoginToken();
        EMSessionManager.getInstance().clearLoginWithTokenData();
        mIsDBOpened = false;

        // ============ gaode code start ===============
        // EMGDLocation.getInstance().onDestroy();
        // ============ gaode code end ===============

        if (smartHeartbeat != null) {
            smartHeartbeat.stop();
        }

        releaseWakelock();

        // make sure we clear the username and pwd in a sync way
        if (emaObject != null) {
            emaObject.logout();
        }
        if (chatManager != null) {
            chatManager.onLogout();
        }
        if (groupManager != null) {
            groupManager.onLogout();
        }
        if (contactManager != null) {
            contactManager.onLogout();
        }
        if (chatroomManager != null) {
            chatroomManager.onLogout();
        }

        // clear all memory cache

        if (EMChatConfigPrivate.isDebugTrafficMode()) {
            EMNetworkMonitor.stopTrafficStat();
        }
    }

    /**
     * \~chinese
     * 登出聊天服务器。
     *
     * 异步方法。
     *
     * @param unbindToken 是否解绑 token。   
     *                    - `true`：是；
     *                    - `false`：否。
     * @param callback    方法完成的回调，包含调用失败的原因。   
     *
     * \~english
     * Logs out of the chat server.
     *
     * This is an asynchronous method.
     *
     * @param unbindToken  Whether to unbind the token upon logout.
     *                     - `true`: Yes;
     *                     - `false`: No.
     * @param callback     The completion callback, which contains the error message if the method call fails.
     */
    public void logout(final boolean unbindToken, final EMCallBack callback) {
        new Thread() {
            @Override
            public void run() {
                int error = logout(unbindToken);

                if (error != EMError.EM_NO_ERROR) {
                    if (callback != null) {
                        callback.onError(error, "faild to unbind device token");
                    }
                } else {
                    if (callback != null) {
                        callback.onSuccess();
                    }
                }
            }
        }.start();
    }

    /**
     * \~chinese
     * 登出聊天服务器。
     * 
     * 异步方法。
     *
     * @param callback    方法完成的回调，包含调用失败的原因。 
     *
     * \~english
     * Logs out of the chat server.
     * 
     * This is an asynchronous method.
     *
     * @param callback     The completion callback, which contains the error message if the method fails.
     */
    void logout(final EMCallBack callback) {
        Thread logoutThread = new Thread() {
            @Override
            public void run() {
                if (callback != null) {
                    callback.onProgress(0, null);
                }

                logout();

                if (callback != null) {
                    callback.onSuccess();
                }
            }
        };
        logoutThread.setPriority(Thread.MAX_PRIORITY - 1);
        logoutThread.start();
    }

    /**
     * \~chinese
     * 更新 App Key。
     *
     * @note
     *
     * - App Key 用于控制对你的 app 的 Chat 服务的访问，只有在未登录状态才能修改 App Key。
     *
     * - 修改 App Key 是为了方便你切换到其他 App Key。
     * 
     * - 你可以在 Console 上获取 App Key。
     *
     * - 也通过 {@link EMOptions#setAppKey(String)} 设置 App Key，同样只有在未登录状态才能修改。
     *
     * @param appkey  App Key，应用的唯一标识，不可为空。
     *
     * \~english
     * Update the App Key.
     *  
     * @note
     *
     * - As this key controls access to the chat service for your app, you can only update the key when the current user logs out.
     *
     * - Updating the App Key means to switch to a new App Key.
     *
     * - You can retrieve the new App Key from the Console.
     * 
     * - You can set App Key by calling {@link EMOptions#setAppKey(String)} when you log out of the chat service.
     *
     * @param appkey  The App Key, make sure to set the param.
     */
    public void changeAppkey(String appkey) throws HyphenateException {
        EMAError error = emaObject.changeAppkey(appkey);
        if (error.errCode() == EMAError.EM_NO_ERROR) {
            this.getOptions().updatePath(appkey, "");
        }
        handleError(error);
    }

    /**
     * \~chinese
     * 更新 AppId。
     *
     * @note
     *
     * - AppId 用于控制对你的 app 的 Chat 服务的访问，只有在未登录状态才能修改 AppId。
     *
     * - 修改 AppId 是为了方便你切换到其他 AppId。
     *
     * - 你可以在 Console 上获取 AppId。
     *
     * - 也通过 {@link EMOptions#setAppId(String)} 设置 AppId，同样只有在未登录状态才能修改。
     *
     * @param appId ，应用的唯一标识，不可为空。
     *
     * \~english
     * Update the AppId.
     *
     * @note
     *
     * - As this appId controls access to the chat service for your app, you can only update the key when the current user logs out.
     *
     * - Updating the AppId means to switch to a new AppId.
     *
     * - You can retrieve the new AppId from the Console.
     *
     * - You can set AppId by calling {@link EMOptions#setAppId(String)} when you log out of the chat service.
     *
     * @param appId  The AppId, make sure to set the param.
     */
    public void changeAppId(String appId) throws HyphenateException {
        EMAError error = emaObject.changeAppId(appId);
        if (error.errCode() == EMAError.EM_NO_ERROR) {
            this.getOptions().updatePath("", appId);
        }
        handleError(error);
    }
    /**
     * \~chinese
     * 设置连接状态监听器。
     *
     * 该监听器用于监听 Chat app 与 Chat 服务器之间的连接。
     *
     * @param listener 要添加的连接状态监听器。
     *                 - {@link EMConnectionListener#onConnected()} 表示与 chat 服务器连接成功；
     *                 - {@link EMConnectionListener#onDisconnected(int)} 表示与 chat 服务器断开连接。参数表示错误码，详见 {@link EMError}。
     *
     * \~english
     * Adds a connection status listener.
     *
     * The listener listens for the connection between the chat app and the chat server.
     *
     * @param listener The connection status listener to add.
     *                 - {@link EMConnectionListener#onConnected()} indicates a successful connection to the chat server.
     *                 - {@link EMConnectionListener#onDisconnected(int)} indicates the connection to the chat server fails. It contains the error code. See {@link EMError}.
     */
    public void addConnectionListener(final EMConnectionListener listener) {
        if (listener == null) {
            return;
        }

        synchronized (connectionListeners) {
            if (!connectionListeners.contains(listener)) {
                connectionListeners.add(listener);
            }
        }

        execute(new Runnable() {

            @Override
            public void run() {
                if (isConnected()) {
                    listener.onConnected();
                } else {
                    if (isLoggedIn()) {
                        listener.onDisconnected(EMError.NETWORK_ERROR);
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 移除 Chat 服务器连接监听。
     *
     * @param listener  要移除的 Chat 服务器连接监听。
     *
     * \~english
     * Removes the connection status listener.
     *
     * @param listener  The connection status listener to remove.
     */
    public void removeConnectionListener(final EMConnectionListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (connectionListeners) {
            connectionListeners.remove(listener);
        }
    }

    /**
     * \~chinese
     * 设置日志回调监听器。
     *
     * @param listener 日志回调监听，{@link EMLogListener#onLog(String)}。
     *
     * \~english
     * Adds the log callback listener of SDK.
     *
     * @param listener The log callback listener, {@link EMLogListener#onLog(String)}.
     */
    public void addLogListener(final EMLogListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (logListeners) {
            if (!logListeners.contains(listener)) {
                logListeners.add(listener);
            }
        }

        if(clientLogListener==null) {
            clientLogListener = new ClientLogListener();
            emaObject.addLogCallbackListener(clientLogListener);
        }
    }

    /**
     * \~chinese
     * 移除日志回调监听。
     *
     * @param listener  日志回调监听。
     *
     * \~english
     * Removes the log callback listener.
     *
     * @param listener  The log callback listener.
     */
    public void removeLogListener(final EMLogListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (logListeners) {
            logListeners.remove(listener);
        }
    }

    /**
     * \~chinese
     * 获取群组管理类。
     *
     * 该方法需在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 群组管理类。
     *
     * \~english
     * Gets the `GroupManager` class.
     *
     * This method can be called only after the EMClient is initialized. See {@link #init(Context, EMOptions)}.
     *
     * @return The `GroupManager` class.
     */
    public EMGroupManager groupManager() {
        if (groupManager == null) {
            synchronized (EMClient.class) {
                if(groupManager == null) {
                    groupManager = new EMGroupManager(this, emaObject.getGroupManager());
                }
            }
        }
        return groupManager;
    }

    /**
     * \~chinese
     * 获取推送管理类。
     *
     * 该方法需在 EMClient 初始化完成后调用。详见 {@link #init(Context, EMOptions)}。
     *
     * @return 推送管理类。
     *
     * \~english
     * Gets the `PushManager` class.
     *
     * This method can be called only after the EMClient is initialized. See {@link #init(Context, EMOptions)}.
     *
     * @return The `PushManager` class.
     */
    public EMPushManager pushManager() {
        if (pushManager == null) {
            synchronized (EMClient.class) {
                if(pushManager == null) {
                    pushManager = new EMPushManager(this, emaObject.getPushMnager());
                }
            }
        }
        return pushManager;
    }

    /**
     * \~chinese
     * 获取聊天室管理类。
     *
     * 该方法需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 聊天室管理类。
     *
     * \~english
     * Gets the `ChatRoomManager` class.
     *
     * This method can be called only after the EMClient is initialized. See {@link #init(Context, EMOptions)}.
     *
     * @return The `ChatRoomManager` class.
     */
    public EMChatRoomManager chatroomManager() {
        if (chatroomManager == null) {
            synchronized (EMClient.class) {
                if(chatroomManager == null) {
                    chatroomManager = new EMChatRoomManager(this, emaObject.getChatRoomManager());
                }
            }
        }
        return chatroomManager;
    }

    /**
     * \~chinese
     * 获取聊天管理类。
     *
     * 该方法需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 聊天管理类。
     *
     * \~english
     * Gets the `ChatManager` class.
     *
     * This method can be called only after the EMClient is initialized. See {@link #init(Context, EMOptions)}.
     *
     * @return The `ChatManager` class.
     */
    public EMChatManager chatManager() {
        if (chatManager == null) {
            synchronized (EMClient.class) {
                if(chatManager == null) {
                    chatManager = new EMChatManager(this, emaObject.getChatManager(), emaObject.getReactionManager());
                }
            }
        }
        return chatManager;
    }

    /**
     * \~chinese
     * 获取用户信息管理类。
     *
     * 需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 用户信息管理类。
     *
     * \~english
     * Gets the `EMUserInfoManager` class.
     *
     * This method can be called only after the EMClient is initialized. See {@link #init(Context, EMOptions)}.
     *
     * @return The `EMUserInfoManager` class.
     */
    public EMUserInfoManager userInfoManager() {
        if (userInfoManager == null) {
            synchronized (EMClient.class) {
                if(userInfoManager == null) {
                    userInfoManager = new EMUserInfoManager(emaObject.getUserInfoManager());
                }
            }
        }
        return userInfoManager;
    }



    /**
     * \~chinese
     * 获取联系人管理类。
     *
     * 需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 联系人管理类。
     *
     * \~english
     * Gets the `ContactManager` class.
     *
     * This method can be called only after the EMClient is initialized. See {@link #init(Context, EMOptions)}.
     *
     * @return The `ContactManager` class.
     */
    public EMContactManager contactManager() {
        if (contactManager == null) {
            synchronized (EMClient.class) {
                if(contactManager == null) {
                    contactManager = new EMContactManager(this, emaObject.getContactManager());
                }
            }
        }
        return contactManager;
    }

    public EMPresenceManager presenceManager(){
        if(presenceManager == null){
            synchronized (EMClient.class) {
                if(presenceManager == null){
                    presenceManager = new EMPresenceManager(emaObject.getPresenceManager());
                }
            }
        }
        return presenceManager;
    }

    /**
     * \~chinese
     * 获取子区管理类。
     *
     * 需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return Thread 管理类。
     *
     * \~english
     * Gets the `EMChatThreadManager` class.
     *
     * This method can be called only after the EMClient is initialized. See {@link #init(Context, EMOptions)}.
     *
     * @return The `EMChatThreadManager` class.
     */
    public EMChatThreadManager chatThreadManager(){
        if(threadManager == null){
            synchronized (EMClient.class) {
                if(threadManager == null){
                    threadManager = new EMChatThreadManager(this, emaObject.getThreadManager());
                }
            }
        }
        return threadManager;
    }

    public EMStatisticsManager statisticsManager(){
        if(statisticsManager == null){
            synchronized (EMClient.class) {
                if(statisticsManager == null){
                    statisticsManager = new EMStatisticsManager(this, emaObject.getStatisticsManager());
                }
            }
        }
        return statisticsManager;
    }

    public Context getContext() {
        return mContext;
    }

    /**
     * \~chinese
     * 获取当前登录用户的用户名。
     *
     * @return 当前登录的用户。
     *
     * \~english
     * Gets the user ID of the current logged-in user.
     *
     * @return The user ID of the current logged-in user.
     */
    public synchronized String getCurrentUser() {
        if (EMSessionManager.getInstance().currentUser == null ||
                EMSessionManager.getInstance().currentUser.username == null ||
                EMSessionManager.getInstance().currentUser.username.equals("")) {
            return EMSessionManager.getInstance().getLastLoginUser();
        }
        return EMSessionManager.getInstance().currentUser.username;
    }

    /**
     * \~chinese
     * 根据用户 ID 和密码获取 token。
     *
     * @param username 用户 ID。
     * @param password 密码。
     * @param callBack 结果回调。   
     *                 - {@link EMValueCallBack#onSuccess(Object)} 表示请求成功的回调，onSuccess 中的参数即为返回的 token；
     *                 - {@link EMValueCallBack#onError(int, String)} 表示请求失败的回调，第一个参数为错误码，第二个为错误描述。
     *
     * \~english
     * Get a token by using the user ID and password.
     *
     * @param username The user ID.
     * @param password The password.
     * 
     * @param callBack The result callback：  
     *                 - If success, the SDK calls {@link EMValueCallBack#onSuccess(Object)}, where the parameter is the token;
     *                 - If a failure occurs, the SDK calls {@link EMValueCallBack#onError(int, String)}, in which the first parameter is the error code, and the second is the error description.
     */
    public void getUserTokenFromServer(final String username, final String password, final EMValueCallBack<String> callBack) {
        execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                final String token = emaObject.getUserTokenFromServer(username, password, error);

                if (callBack == null) {
                    return;
                }

                if (error.errCode() == EMError.EM_NO_ERROR) {
                    callBack.onSuccess(token);
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~chinese
     * 查询是否登录过。
     * 
     * 如果登录成功后未调用 `logout` 方法，这个方法的返回值一直是 `true`。
     *
     * 如果需要判断当前是否连接到服务器，请使用 {@link #isConnected()} 方法。
     *
     * ```java
     * if(EMClient.getInstance().isLoggedInBefore()){
     *     // enter main activity
     * }else{
     *     // enter login activity
     * }
     * ```
     *
     * @return 返回是否登录过的结果。   
     *         - `true` 表示登录过。
     *         - `false` 表示未登录过或者已经调用过 {@link #logout()} 方法。
     *
     * \~english
     * Checks whether the user has logged in before.
     * 
     * This method always returns `true` if you log in successfully and have not called the {@link #logout()} method yet.
     *
     * If you need to check whether the SDK is connected to the server, call {@link #isConnected()}.
     *
     * ```java
     * if(EMClient.getInstance().isLoggedInBefore()){
     *     // Enter the main activity.
     * }else{
     *     // Enter the login activity.
     * }
     * ```
     *
     * @return Whether the user has logged in before.
     *         - `true`: The user has logged in before;
     *         - `false`: The user has not logged in before or has called the {@link #logout()} method.
     */
    public boolean isLoggedInBefore() {
        EMSessionManager sessionMgr = EMSessionManager.getInstance();
        String user = sessionMgr.getLastLoginUser();
        // no need decrypt password or token when user empty
        if (TextUtils.isEmpty(user)) return false;

        String pwd = sessionMgr.getLastLoginPwd();
        String token = sessionMgr.getLastLoginToken();

        if (user != null  && !user.isEmpty() &&
                ((pwd != null   && !pwd.isEmpty()) ||
                 (token != null && !token.isEmpty())) ) {
            return true;
        }

        return false;
    }


    /**
     * \~chinese
     * 检查 SDK 是否连接到 Chat 服务器。
     *
     * @return 是否连接到 Chat 服务器的结果。
     *         - `true` 表示已经连接到 Chat 服务器。
     *         - `false` 表示没有连接到 Chat 服务器。
     *
     * \~english
     * Checks whether the SDK is connected to the chat server.
     *
     * @return Whether the SDK is connected to the chat server.
     *         - `true`: Yes;
     *         - `false`: No.
     */
    public boolean isConnected() {
        return emaObject.isConnected();
    }

    /**
     *\~chinese
     * 检查用户是否登录 Chat 服务。
     *
     * @return 是否登录 Chat 服务。
     *         - `true`：是。
     *         - `false`：否。
     * \~english
     * Checks whether the user has logged in to the Chat app.
     *
     * @return Whether the user has logged in to the Chat app.
     *         - `true`: Yes;
     *         - `false`: No.
     */
    public boolean isLoggedIn() {
        return emaObject.isLoggedIn();
    }

    /**
     * \~chinese
     * 设置是否输出调试信息。
     *
     * 在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @param debugMode 是否输出调试信息。
     *                  - `true`：SDK 会在 log 里输出调试信息；
     *                  -  `false`：不会输出调试信息。
     *
     * \~english
     * Sets whether to output the debug information.
     * 
     * This method can be called only after the EMClient is initialized. See {@link #init(Context, EMOptions)}.
     *
     * @param debugMode  Whether to output the debug information:
     *                   - `true`: Yes;
     *                   - `false`: No.
     */
    public void setDebugMode(boolean debugMode) {
        EMLog.debugMode = debugMode;
        getChatConfigPrivate().setDebugMode(debugMode);
    }

    /**
     * \~chinese
     * 上传本地日志。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param callback 预留参数。
     *
     * \~english
     * Uploads local logs.
     *
     * The debug logs provide reference for our engineers to fix errors and improve system performance.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param callback Reserved parameter.
     */
    public void uploadLog(EMCallBack callback) {
        chatManager().emaObject.uploadLog();
    }


    /**
     * \~chinese
     * 获取 SDK 的各种配置选项。
     *
     * \~english
     * Gets configuration options of the SDK.
     */
    public EMOptions getOptions() {
        return mChatConfigPrivate.getOptions();
    }


    /**
     * \~chinese
     * 压缩调试日志文件。
     *
     * 强烈建议方法完成之后删除该压缩文件。
     *
     * @return 压缩后的 log 文件路径。
     * @throws HyphenateException 压缩文件失败报的错。
     *
     * \~english
     * Compresses the debug log file into a gzip archive.
     *
     * It is strongly recommended that you delete this debug archive as soon as this method is called.

     * @return The path of the compressed gz file.
     * @throws HyphenateException  A description of the cause of the exception if the method fails.
     */
    public String compressLogs() throws HyphenateException {
        EMAError error = new EMAError();
        String path = emaObject.compressLogs(error);
        handleError(error);
        return path;
    }

    /**
     * \~chinese
     * 添加多设备监听器。
     *
     * @param listener 要添加的多设备监听器。
     *                 详见 {@link EMMultiDeviceListener}, {@link EMMultiDeviceListener#onContactEvent(int, String, String)} 联系人事件回调，
     *                 {@link EMMultiDeviceListener#onGroupEvent(int, String, List)} 群组事件回调。
     *
     * \~english
     * Adds the multi-device listener.
     * 
     * @param listener The multi-device listener to add.
     *                See {@link EMMultiDeviceListener}. {@link EMMultiDeviceListener#onContactEvent(int, String, String)} is the contact event callback and
     *                 {@link EMMultiDeviceListener#onGroupEvent(int, String, List)} is the group event callback.
     */
    public void addMultiDeviceListener(EMMultiDeviceListener listener) {
        multiDeviceListeners.add(listener);
    }

    /**
     * \~chinese
     * 移除多设备监听器。
     * 
     * @param listener 要移除的多设备监听器。见 {@link EMMultiDeviceListener}。
     *
     * \~english
     * Removes the multi-device listener.
     *
     * @param listener The multi-device listener to remove. See {@link EMMultiDeviceListener}.
     */
    public void removeMultiDeviceListener(EMMultiDeviceListener listener) {
        multiDeviceListeners.remove(listener);
    }

    /**
     * \~chinese
     * 获取指定账号下登录的在线设备列表。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username  用户 ID。
     * @param password  密码。
     * @return 在线登录设备列表。
     * @throws HyphenateException 具体错误详见 {@link EMError}。
     *
     * \~english
     * Gets the list of online devices where a specified user is logged in.
     *
     * This is a synchronous method and blocks the current thread. 
     *
     * @param username The user ID.
     * @param password The password.
     * @return The list of online devices where the specified user is logged in.
     * @throws HyphenateException A description of the exception. See {@link EMError}.
     */
    public List<EMDeviceInfo> getLoggedInDevicesFromServer(String username, String password) throws HyphenateException {
        EMAError error = new EMAError();
        List<EMADeviceInfo> devices = emaObject.getLoggedInDevicesFromServer(username, password, error);
        handleError(error);
        List<EMDeviceInfo> result = new ArrayList<>();
        for (EMADeviceInfo info : devices) {
            result.add(new EMDeviceInfo(info));
        }
        return result;
    }

    /**
     * \~chinese
     * 获取指定账号下登录的在线设备列表。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username  用户 ID。
     * @param token  用户 token。
     * @return 在线登录设备列表。
     * @throws HyphenateException 具体错误详见 {@link EMError}。
     *
     * \~english
     * Gets the list of online devices where a specified user is logged in.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param token The user token.
     * @return The list of online devices where the specified user is logged in.
     * @throws HyphenateException A description of the exception. See {@link EMError}.
     */
    public List<EMDeviceInfo> getLoggedInDevicesFromServerWithToken(@NonNull String username, @NonNull String token) throws HyphenateException {
        EMAError error = new EMAError();
        List<EMADeviceInfo> devices = emaObject.getLoggedInDevicesFromServerWithToken(username, token, error);
        handleError(error);
        List<EMDeviceInfo> result = new ArrayList<>();
        for (EMADeviceInfo info : devices) {
            result.add(new EMDeviceInfo(info));
        }
        return result;
    }

    /**
     * \~chinese
     * 将指定账号从登录的指定设备踢下线。
     *
     * 可通过 {@link EMDeviceInfo#getResource()} 方法获取设备 ID。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username  用户 ID。
     * @param password  用户的密码。
     * @param resource  设备 ID, 见 {@link EMDeviceInfo#getResource()}。
     * @throws HyphenateException 如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
     *
     * \~english
     * Kicks a user out of the app on a device. 
     *
     * You can call {@link EMDeviceInfo#getResource()} to get the device ID.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param password The password.
     * @param resource The device ID. See {@link EMDeviceInfo#getResource()}.
     * @throws HyphenateException A description of the exception if the method fails. See {@link EMError}.
     */
    public void kickDevice(String username, String password, String resource) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickDevice(username, password, resource, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 将指定账号从登录的指定设备踢下线。
     *
     * 可通过 {@link EMDeviceInfo#getResource()} 方法获取设备 ID。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username  用户 ID。
     * @param token  用户 token。
     * @param resource  设备 ID, 见 {@link EMDeviceInfo#getResource()}。
     * @throws HyphenateException 如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
     *
     * \~english
     * Kicks a user out of the app on a device. 
     *
     * You can call {@link EMDeviceInfo#getResource()} to get the device ID.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param token The user token.
     * @param resource The device ID. See {@link EMDeviceInfo#getResource()}.
     * @throws HyphenateException A description of the exception if the method fails. See {@link EMError}.
     */
    public void kickDeviceWithToken(@NonNull String username,@NonNull String token, String resource) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickDeviceWithToken(username, token, resource, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 将指定账号登录的所有设备都踢下线。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username 用户 ID。
     * @param password 密码。
     * @throws HyphenateException 如果有异常会抛出，包括错误码和错误描述，错误码详见 {@link EMError}。
     *
     * \~english
     * Kicks a user out of the app on all device. 
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param password The password.
     * @throws HyphenateException A description of the exception. See {@link EMError}.
     */
    public void kickAllDevices(String username, String password) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickAllDevices(username, password, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 将指定账号登录的所有设备都踢下线。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username 用户 ID。
     * @param token 用户 token。
     * @throws HyphenateException 如果有异常会抛出，包括错误码和错误描述，错误码详见 {@link EMError}。
     *
     * \~english
     * Kicks a user out of the app on all device.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param token The user token.
     * @throws HyphenateException A description of the exception. See {@link EMError}.
     */
    public void kickAllDevicesWithToken(@NonNull String username, @NonNull String token) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickAllDevicesWithToken(username, token, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 上传 FCM token 至 Chat 服务器。
     * 
     * 上传 token 需满足以下条件：
     * - 被上传的 token 不为空；
     * - 当前已有用户登录；
     * - 当前设备支持 Google 推送；
     * - Google 推送类型为 FCM {@link EMPushType#FCM}。
     * 
     * @param fcmToken 要上传的 token。
     *
     * \~english
     * Uploads the FCM token to the chat server.
     * 
     * The token can be uploaded when the following conditions are met:
     * - The token is not empty;
     * - The user has logged in to the Chat app;
     * - The current device supports Google PUSH service;
     * - The push type is FCM {@link EMPushType#FCM}.
     * 
     * @param fcmToken The token to upload.
     */
    public void sendFCMTokenToServer(String fcmToken) {
        EMLog.d(TAG, "sendFCMTokenToServer: " + fcmToken);
        if (TextUtils.isEmpty(fcmToken)) {
            return;
        }

        // If no user logged in, stop upload the fcm Token.
        String userName = getCurrentUser();
        if (TextUtils.isEmpty(userName)) {
            EMLog.i(TAG, "No user login currently, stop upload the token.");
            return;
        }
        // let's check if the push service available
        EMPushType pushType = EMPushHelper.getInstance().getPushType();
        EMLog.i(TAG, "pushType: " + pushType);
        if (pushType == EMPushType.FCM) {
            EMPushHelper.getInstance().onReceiveToken(pushType, fcmToken);
        }
    }

    /**
     * \~chinese
     * 发送华为推送 token 到服务器。
     *
     * @param token    华为推送 token。
     *
     * \~english
     * Sends the HUAWEI Push token to the server.
     * @param token    The Huawei Push token.
     */
    public void sendHMSPushTokenToServer(String token){
        if (EMPushHelper.getInstance().getPushType() == EMPushType.HMSPUSH) {
            EMPushHelper.getInstance().onReceiveToken(EMPushType.HMSPUSH, token);
        }
    }

    /**
     * \~chinese
     * 发送荣耀推送 token 到服务器。
     *
     * @param token    荣耀推送 token。
     *
     * \~english
     * Sends the Honor Push token to the server.
     * @param token    The Honor Push token.
     */
    public void sendHonorPushTokenToServer(String token){
        if (EMPushHelper.getInstance().getPushType() == EMPushType.HONORPUSH) {
            EMPushHelper.getInstance().onReceiveToken(EMPushType.HONORPUSH, token);
        }
    }

	//---------------------------private methods ---------------------------------

	private void initManagers(){
		// invoke constructor before login, to listener PB event
		EMHttpClient.getInstance().onInit(mChatConfigPrivate);
		chatManager();
		contactManager();
		groupManager();
		chatroomManager();
        presenceManager();
        chatThreadManager();

		setNatvieNetworkCallback();
		EMSessionManager.getInstance().init(this, emaObject.getSessionManager());
	}

	void _login(final String username, final String code, final EMCallBack callback, final boolean autoLogin, final EMLoginType loginType) {
        if (getChatConfigPrivate() == null || sdkInited == false) {
            callback.onError(EMError.GENERAL_ERROR, "");
            return;
        }

        EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid());

        // convert to lowercase
        execute(new Runnable() {

            @Override
            public void run() {
                EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid() + " threadName:" + Thread.currentThread().getName() + " ID:" + Thread.currentThread().getId());

                if (username == null) {
                    callback.onError(EMError.INVALID_USER_NAME, "Invalid user name");
                    return;
                }

                EMAError error = new EMAError();
                emaObject.login(username, code, autoLogin, loginType.ordinal(), error);

                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    EMSessionManager.getInstance().setLastLoginUser(username);
                    if (loginType == EMLoginType.LOGIN_PASSWORD ) {
                        EMSessionManager.getInstance().setLastLoginPwd(code);
                        EMSessionManager.getInstance().setLastLoginWithToken(false);
                        EMSessionManager.getInstance().clearLastLoginToken();
                    }else{
                        saveToken();
                    }
                    // Move from before login, do not open db first
                    if(autoLogin) {
                        loadDataFromDb();
                    }

                    onNewLogin();
                    EMPushHelper.getInstance().register();
                    // set EMConferenceManager parameters, lite jar build need to remove this line
//                    conferenceManager().set(getAccessToken(), getOptions().getAppKey(), getCurrentUser());
                    callback.onSuccess();
                } else {
                    callback.onError(error.errCode(), error.errMsg());
                }

                if (error.errCode() == EMAError.USER_AUTHENTICATION_FAILED) {
                    EMSessionManager.getInstance().clearLastLoginPwd();
                    EMSessionManager.getInstance().clearLastLoginToken();
                    EMSessionManager.getInstance().clearLoginWithTokenData();
                }
            }
        });
    }

    private void loadDataFromDb() {
        execute(()-> {
            groupManager().loadAllGroups();
            if(getOptions().isAutoLoadAllConversations()) {
                chatManager().loadAllConversationsFromDB();
            }
        });
    }

    /**
     * \~chinese
     * 当登录方式为 Agora token 时，登录成功后，有可能还没有返回置换为 Chat token，为了自动登录的逻辑，
     * 需要在 {@link MyConnectionListener#onReceiveToken(String, long)} 方法中重新调用一次本方法。
     * 目前 {@link MyConnectionListener#onReceiveToken(String, long)} 只会在使用 Agora token 登录时会进行回调。
     *
     * \~english
     * When the login method is Agora token, after the login is successful, the replacement to Chat token may not be returned.
     * For the logic of automatic login, this method needs to be called again in the {@link MyConnectionListener#onReceiveToken(String, long)} method.
     * Currently {@link MyConnectionListener#onReceiveToken(String, long)} will only call back when logging in with Agora token.
     */
    private void saveToken() {
        if(emaObject == null) {
            return;
        }
        EMAError emaError = new EMAError();
        String token = emaObject.getUserToken(false, emaError);
        if (emaError.errCode() == EMAError.EM_NO_ERROR) {
            EMSessionManager.getInstance().setLastLoginToken(token);
            EMSessionManager.getInstance().setLastLoginWithToken(true);
            EMSessionManager.getInstance().clearLastLoginPwd();
        }
    }

    /**
     * \~chinese
     * 检查 FCM 推送是否可用。
     *
     * @return  FCM 推送是否可用：
     *          - `true`：可用；
     *          - `false`：不可用。
     *
     * \~english
     * Checks whether the FCM push is available.
     *
     * @return  Whether the FCM push is available:
     *          - `true`: Yes;
     *          - `false`: No.
     */
    public boolean isFCMAvailable() {
        return EMPushHelper.getInstance().getPushType() == EMPushType.FCM;
    }

    void onNewLogin() {
        EMLog.d(TAG, "on new login created");

        String username = EMSessionManager.getInstance().getLastLoginUser();

        PathUtil.getInstance().initDirs(getChatConfigPrivate().getAppKey(), username, mContext);

        if (smartHeartbeat == null) {
            smartHeartbeat = EMSmartHeartBeat.create(mContext);
        }

        if (getChatConfigPrivate().emaObject.hasHeartBeatCustomizedParams()) {
            smartHeartbeat.setCustomizedParams(getChatConfigPrivate().emaObject.getWifiHeartBeatCustomizedParams(),
                    getChatConfigPrivate().emaObject.getMobileHeartBeatCustomizedParams());
        }
        smartHeartbeat.onInit();
        if (getOptions().getFixedInterval() != -1) {
            smartHeartbeat.setFixedInterval(getOptions().getFixedInterval());
        }
    }

    /**
     * \~chinese
     * 从内存中获取身份认证 token。
     * 
     * 在上传下载附件（语音，图片，文件等）时必须将 token 添加到请求 header 中，当出现任何异常时将返回 `null`。
     *
     * 如果为 `null`，在打开 EMLog 日志时，可以看到异常原因。
     *
     * 也可以通过 {@link EMOptions#getAccessToken(boolean)} 方法传入 `true`从服务端获取 token。
     *
     * @return 身份认证 token。
     *
     * \~english
     * Gets the access token from the memory.
     * 
     * When uploading or downloading an attachment (a voice, image, or file), you must add the token to the request header. The SDK returns `null` when any exception occurs.
     * 
     * If the token is `null`, you can check the EMLog file for the possible reason.
     *
     * You can also get the token from the server by calling {@link EMOptions#getAccessToken(boolean)} and passing `true`.
     *
     * @return  The access token.
     */
    public String getAccessToken() {
        return getChatConfigPrivate().getAccessToken();
    }

    /**
     * \~chinese
     * 检查 SDK 是否已初始化完毕。
     *
     * @return  SDK 是否已初始化完毕：
     *          - `true`：是。
     *          - `false`：否。
     *
     * \~english
     * Checks whether the SDK is initialized.
     *
     * @return   Whether the SDK is initialized:
     *           - `true`: Yes.
     *           - `false`: No.
     */
    public boolean isSdkInited() {
        return sdkInited;
    }

    private boolean _loadLibrary(final String library, boolean trace) {
        try {
            ReLinker.loadLibrary(mContext, library);
            EMLog.e(TAG, "_loadLibrary: " + library + " success");
            return true;
        } catch (Throwable e) {
            if (trace) {
                e.printStackTrace();
            }
        }
        return false;
    }

    private boolean _loadLibrary(final String library) {
        return _loadLibrary(library, true);
    }

    private void loadLibrary(EMOptions options) {
        if (libraryLoaded == false) {
            String nativeLibBasePath = options.getNativeLibBasePath();
            EMLog.e(TAG, "nativeLibBasePath: " + nativeLibBasePath);
            if(TextUtils.isEmpty(nativeLibBasePath)) {
                _loadLibrary("cipherdb", false);
                _loadLibrary("hyphenate",true);
                _loadLibrary("aosl", false);
            }else{
                //自定义nativelib路径加载
                _loadCustomPathLibrary(nativeLibBasePath+"/libcipherdb.so");
                _loadCustomPathLibrary(nativeLibBasePath+"/libhyphenate.so");
                _loadCustomPathLibrary(nativeLibBasePath+"/libaosl.so");
            }
            libraryLoaded = true;
        }
    }

    private void _loadCustomPathLibrary(String nativeLibFullPath) {
        try {
            System.load(nativeLibFullPath);
            EMLog.e(TAG, "_loadCustomPathLibrary: " + nativeLibFullPath + " success");
        } catch (Throwable e) {
            e.printStackTrace();
        }
    }

    class MyConnectionListener extends EMAConnectionListener {

        @Override
        public void onConnected() {
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        try {
                            for (EMConnectionListener listener : connectionListeners) {
                                listener.onConnected();
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onDisconnected(final int errCode, EMALogoutInfo info) {
            EMLog.e(TAG,"onDisconnected errcode = "+errCode+",getDeviceInfo = "+info.getDeviceInfo() +",ext = " + info.getDeviceExt());
            EMLoginExtensionInfo logoutInfo = new EMLoginExtensionInfo(
                info.getDeviceInfo(), info.getDeviceExt()
            );
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        switch (errCode) {
                            case EMError.USER_REMOVED:
                                EMSessionManager.getInstance().clearLastLoginUser();
                                EMSessionManager.getInstance().clearLastLoginToken();
                                EMSessionManager.getInstance().clearLastLoginPwd();
                                break;
                            case EMError.USER_LOGIN_ANOTHER_DEVICE:
                            case EMError.USER_BIND_ANOTHER_DEVICE:
                            case EMError.USER_DEVICE_CHANGED:
                            case EMError.SERVER_SERVICE_RESTRICTED:
                            case EMError.USER_LOGIN_TOO_MANY_DEVICES:
                            case EMError.USER_KICKED_BY_CHANGE_PASSWORD:
                            case EMError.USER_KICKED_BY_OTHER_DEVICE:
                            case EMError.APP_ACTIVE_NUMBER_REACH_LIMITATION:
                                EMSessionManager.getInstance().clearLastLoginToken();
                                EMSessionManager.getInstance().clearLastLoginPwd();
                                if(isSdkInited()) {
                                    // clear push token info
                                    EMPreferenceUtils.getInstance().setPushNotifierName("");
                                    EMPreferenceUtils.getInstance().setPushToken("");
                                }
                                break;
                        }

                        try {
                            for (EMConnectionListener listener : connectionListeners) {
                                listener.onDisconnected(errCode);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }

                        if ((errCode == EMError.USER_LOGIN_ANOTHER_DEVICE) ||
                                (errCode == EMError.USER_REMOVED) ||
                                (errCode == EMError.USER_BIND_ANOTHER_DEVICE) ||
                                (errCode == EMError.USER_DEVICE_CHANGED) ||
                                (errCode == EMError.SERVER_SERVICE_RESTRICTED) ||
                                (errCode == EMError.USER_LOGIN_TOO_MANY_DEVICES) ||
                                (errCode == EMError.USER_KICKED_BY_CHANGE_PASSWORD) ||
                                (errCode == EMError.USER_KICKED_BY_OTHER_DEVICE)||
                                (errCode == EMError.APP_ACTIVE_NUMBER_REACH_LIMITATION)){
                            for (EMConnectionListener listener : connectionListeners) {
                                try {
                                    listener.onLogout(errCode,logoutInfo);
                                } catch (Exception e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    }
				}
			});
		}

        @Override
        public boolean verifyServerCert(List<String> certschain,String domain) {
            if (certschain == null) {
                EMLog.d(TAG, "List<String> certschain : null ");
                return false;
            }
            if(TextUtils.isEmpty(domain)) {
                EMLog.d(TAG, "domain is empty or null ");
                return false;
            }
            EMLog.d(TAG, "domain = " + domain);
            X509Certificate[] certsArray = convertToCerts(certschain);
            try {
                X509TrustManager x509TrustManager = getSystemDefaultTrustManager();
                X509TrustManagerExtensions managerExtensions = new X509TrustManagerExtensions(x509TrustManager);
                managerExtensions.checkServerTrusted(certsArray, certsArray[0].getType(),domain);
            } catch (Exception e) {
                e.printStackTrace();
                EMLog.e(TAG, e.getMessage());
                EMLog.d(TAG, "List<String> certschain :" + certschain.toString());

                return false;
            }
            return true;
        }
        public void onTokenNotification(int code) {
            //从EMSessionManager::onDisconnect、com.hyphenate.chat.EMSessionManager.checkTokenAvailability、EMClient#notifyTokenExpired中通知过来
            execute(new Runnable() {
                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        try {
                            if(code==EMError.TOKEN_EXPIRED||code==401) {
                                logout();
                                for (EMConnectionListener listener : connectionListeners) {
                                    EMLog.d(TAG,"MyConnectionListener onToken has expired, code: "+code);
                                    listener.onTokenExpired();
                                }
                            }else{
                                for (EMConnectionListener listener : connectionListeners) {
                                    EMLog.d(TAG,"MyConnectionListener onToken will expire, code: "+code);
                                    listener.onTokenWillExpire();
                                }
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                            EMLog.e(TAG,"MyConnectionListener onTokenNotification Exception: "+e.getMessage());
                        }
                    }
                }
            });
        }

        /**
         * Come from Linux layer after exchanging with server
         * @param token
         * @param expireTimestamp
         */
        @Override
        public void onReceiveToken(String token, long expireTimestamp) {
            if(TextUtils.isEmpty(token) || expireTimestamp <= 0) {
                EMLog.e(TAG, "onReceiveToken: params received is invalid token: " + token + " expireTimestamp: " + expireTimestamp);
                return;
            }
            long tokenAvailablePeriod = expireTimestamp - System.currentTimeMillis();
            initLoginWithToken(mIsLoginWithToken, String.valueOf(expireTimestamp), tokenAvailablePeriod);
            //打日志
            EMLog.d(TAG, "onReceiveToken: token: " + token + " expireTimestamp: " + expireTimestamp + " tokenAvailablePeriod: " + tokenAvailablePeriod);
        }

        @Override
        public void onDatabaseOpened(int errCode) {
            EMLog.e(TAG, "onDatabaseOpened: errCode: "+errCode);
            mIsDBOpened = true;
        }

        @Override
        public void onOfflineMessageSyncStart() {
            EMLog.e(TAG,"MyConnectionListener onOfflineMessageSyncStart()");
            execute(new Runnable() {
                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        for (EMConnectionListener listener : connectionListeners) {
                            try {
                                listener.onOfflineMessageSyncStart();
                            } catch (Exception e) {
                                e.printStackTrace();
                                EMLog.e(TAG,"MyConnectionListener onOfflineMessageSyncStart Exception: "+e.getMessage());
                            }
                        }
                    }
                }
            });
        }

        @Override
        public void onOfflineMessageSyncFinish() {
            EMLog.e(TAG,"MyConnectionListener onOfflineMessageSyncFinish()");
            execute(new Runnable() {
                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        for (EMConnectionListener listener : connectionListeners) {
                            try {
                                listener.onOfflineMessageSyncFinish();
                            } catch (Exception e) {
                                e.printStackTrace();
                                EMLog.e(TAG,"MyConnectionListener onOfflineMessageSyncFinish Exception: "+e.getMessage());
                            }
                        }
                    }
                }
            });
        }
    }

    class MyMultiDeviceListener extends EMAMultiDeviceListener {
        @Override
        public void onContactEvent(final int event, final String target, final String ext) {
            EMLog.d(TAG, "onContactEvent:" + event + " target:" + target + " ext:" + ext);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (EMMultiDeviceListener listener : multiDeviceListeners) {
                                listener.onContactEvent(event, target, ext);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onGroupEvent(final int event, final String target, final List<String> usernames) {
            EMLog.d(TAG, "onGroupEvent:" + event + " target:" + target + " usernames:" + usernames);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (EMMultiDeviceListener listener : multiDeviceListeners) {
                                listener.onGroupEvent(event, target, usernames);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onRoamDeleteEvent(String conversation,List<String> msgIdList,String deviceId,long beforeTimeStamp) {
            EMLog.d(TAG, "onRoamDeleteEvent:" + conversation + " " + deviceId);
            if (TextUtils.isEmpty(conversation)) return;
            execute(new Runnable() {
                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            if (msgIdList != null && msgIdList.size() > 0){
                                chatManager.clearCaches(conversation,msgIdList);
                            }else {
                                chatManager.clearCaches(conversation,beforeTimeStamp);
                            }
                            for (EMMultiDeviceListener listener : multiDeviceListeners) {
                                listener.onMessageRemoved(conversation,deviceId);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onThreadEvent(int event, String target, List<String> username) {
            EMLog.d(TAG, "onThreadEvent:" + event + " target:" + target + " usernames:" + username);
            execute(()-> {
                synchronized (multiDeviceListeners) {
                    for (EMMultiDeviceListener listener : multiDeviceListeners) {
                        try {
                            listener.onChatThreadEvent(event, target, username);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onConversationEvent(int event, String conversationId, int type) {
            EMLog.d(TAG, "onConversationEvent: "+event+" conversationId: "+conversationId+" type: "+type);
            final EMConversation.EMConversationType cType;
            if(type == EMConversation.EMConversationType.GroupChat.ordinal()) {
                cType = EMConversation.EMConversationType.GroupChat;
            }else {
                cType = EMConversation.EMConversationType.Chat;
            }
            execute(()-> {
                synchronized (multiDeviceListeners) {
                    for (EMMultiDeviceListener listener : multiDeviceListeners) {
                        try {
                            listener.onConversationEvent(event, conversationId, cType);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }
    }

    class ClientLogListener extends EMALogCallbackListener {
        @Override
        public void onLogCallback(String log) {
            logQueue.submit(()-> {
                synchronized (logListeners) {
                    for (EMLogListener listener : logListeners) {
                        try {
                            listener.onLog(log);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }
    }

    void execute(Runnable runnable) {
        executor.execute(runnable);
    }

    void executeOnMainQueue(Runnable runnable) {
        mainQueue.submit(runnable);
    }

    void executeOnSendQueue(Runnable runnable) {
        sendQueue.submit(runnable);
    }

    public EMChatConfigPrivate getChatConfigPrivate() {
        return mChatConfigPrivate;
    }

    void setNatvieNetworkCallback() {
        EMANetCallback callback = new EMANetCallback() {

            @Override
            public int getNetState() {
                if (!NetUtils.hasDataConnection(mContext)) {
                    return EMANetwork.NETWORK_NONE.ordinal();
                } else {
                    if (NetUtils.isWifiConnected(mContext)
                            || NetUtils.isOthersConnected(mContext)) {
                        return EMANetwork.NETWORK_WIFI.ordinal();
                    } else if (NetUtils.isMobileConnected(mContext)) {
                        return EMANetwork.NETWORK_MOBILE.ordinal();
                    } else if (NetUtils.isEthernetConnected(mContext)) {
                        return EMANetwork.NETWORK_CABLE.ordinal();
                    } else {
                        return EMANetwork.NETWORK_NONE.ordinal();
                    }
                }
            }
        };

        mChatConfigPrivate.emaObject.setNetCallback(callback);
    }

    /**
     * \~chinese
     * 设置应用的加密实现。
     *
     * 如果未设置加密实现，SDK 会使用内置的加密算法。
     * 
     * @param provider 加密实现。
     * 
     * \~english
     * Sets the encryption provider.
     *
     * If the encryption provider is not set, the SDK uses the default encryption algorithm.
     *
     * @param provider The encryption provider.
     */
    void setEncryptProvider(EMEncryptProvider provider) {
        this.encryptProvider = provider;
    }

    /**
     * \~chinese
     * 获取加密实现。
     *
     * 如果你未设置，将返回 SDK 内置的加密算法。
     * 
     *  \~english
     * Gets the encryption provider.
     * 
     * If the encryption provider is not set, the SDK returns the default encryption algorithm.
     *
     * @return The encryption provider.
     */
    EMEncryptProvider getEncryptProvider() {
        if (encryptProvider == null) {
            EMLog.d(TAG, "encrypt provider is not set, create default");
            encryptProvider = new EMEncryptProvider() {

                public byte[] encrypt(byte[] input, String username) {
                    try {
                        String in = new String(input);
                        return emaObject.getSessionManager().encrypt(in).getBytes();
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

                public byte[] decrypt(byte[] input, String username) {
                    try {
                        String in = new String(input);
                        return emaObject.getSessionManager().decrypt(in).getBytes();
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

            };
        }
        return encryptProvider;
    }

    boolean sendPing(boolean waitPong, long timeout) {
        return emaObject.sendPing(waitPong, timeout);
    }

    void checkTokenAvailability(){
        if(mIsLoginWithToken) {
            EMSessionManager.getInstance().checkTokenAvailability(connectionListener);
        }
    }

    void forceReconnect() {
        EMLog.d(TAG, "forceReconnect");
        disconnect();
        reconnect();
    }

    void reconnect() {
        EMLog.d(TAG, "reconnect");
        try {
            wakeLock.acquire();
        } catch (Exception e) {
            EMLog.e(TAG, e.getMessage());
        }
        emaObject.reconnect();
        releaseWakelock();
    }

    void disconnect() {
        EMLog.d(TAG, "disconnect");
        emaObject.disconnect();
    }

    /**
     * release wake lock
     */
    private void releaseWakelock(){
        // Fix the bug: WakeLock under-locked emclient
        synchronized (this) {
            if (wakeLock != null && wakeLock.isHeld()) {
                try {
                    wakeLock.release();
                } catch (Exception e) {
                    EMLog.e(TAG, e.getMessage());
                }
                EMLog.d(TAG, "released the wake lock");
            }
        }
    }

    private void handleError(EMAError error) throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

    
    /**
     *\~chinese
     * 连接变化监听器。
     * 
     * 该方法只在重连时运行，会更新重连申请次数。
     * 
     *  \~english 
     * The connectivity change listener.
     * 
     * Only occurs in the reconnection thread and resets the attempts times.
     */
    private BroadcastReceiver connectivityBroadcastReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(final Context context, Intent intent) {
            String action = intent.getAction();
            if (!action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
                EMLog.d(TAG, "skip no connectivity action");
                return;
            }

            EMLog.d(TAG, "connectivity receiver onReceiver");

            EMANetwork networkType;
            NetUtils.Types types = NetUtils.getNetworkTypes(getContext());
            switch (types) {
                case WIFI:
                case OTHERS:
                    networkType = EMANetwork.NETWORK_WIFI;
                    break;
                case MOBILE:
                    networkType = EMANetwork.NETWORK_MOBILE;
                    break;
                case ETHERNET:
                    networkType = EMANetwork.NETWORK_CABLE;
                    break;
                case NONE:
                default:
                    networkType = EMANetwork.NETWORK_NONE;
                    break;
            }

            boolean prevNetworkAvailable = currentNetworkType != EMANetwork.NETWORK_NONE;
            boolean currentNetworkAvailable = networkType != EMANetwork.NETWORK_NONE;
            currentNetworkType = networkType;
            if (prevNetworkAvailable == currentNetworkAvailable && currentNetworkAvailable) {
               execute(new Runnable() {
                    @Override
                    public void run() {
                        if (smartHeartbeat != null) {
                            // Network availability no change, return.
                            EMLog.i(TAG, "Network availability no change, just return. " + currentNetworkType + ", but check ping");
                            smartHeartbeat.sendPingCheckConnection();
                        }
                    }
                });
                return;
            }

            EMLog.i(TAG, "Network availability changed, notify... " + currentNetworkType);

            execute(new Runnable() {

                @Override
                public void run() {
                    emaObject.onNetworkChanged(currentNetworkType);
                }
            });
        }
    };

    void onNetworkChanged() {
        try {
            if (NetUtils.isWifiConnected(mContext)
                    || NetUtils.isOthersConnected(mContext)) {
                EMLog.d(TAG, "has wifi connection");
                currentNetworkType = EMANetwork.NETWORK_WIFI;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_WIFI);
                return;
            }

            if (NetUtils.isMobileConnected(mContext)) {
                EMLog.d(TAG, "has mobile connection");
                currentNetworkType = EMANetwork.NETWORK_MOBILE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_MOBILE);
                return;
            }

            if (NetUtils.isEthernetConnected(mContext)) {
                EMLog.d(TAG, "has ethernet connection");
                currentNetworkType = EMANetwork.NETWORK_CABLE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_CABLE);
                return;
            }
            currentNetworkType = EMANetwork.NETWORK_NONE;
            EMLog.d(TAG, "no data connection");
            emaObject.onNetworkChanged(EMANetwork.NETWORK_NONE);
            return;
        } catch (Exception e) {
            e.printStackTrace();
            return;
        }
    }

    void onNetworkChanged(EMANetwork network) {
        emaObject.onNetworkChanged(network);
    }

    private AppStateListener appStateListener;
    private List<Activity> resumeActivityList = new ArrayList<>();

    void setAppStateListener(AppStateListener appStateListener) {
        this.appStateListener = appStateListener;
    }

    @TargetApi(14)
    private void registerActivityLifecycleCallbacks() {
        if (Utils.isSdk14()) {
            Object lifecycleCallbacks = new ActivityLifecycleCallbacks() {

                @Override
                public void onActivityStopped(Activity activity) {
                    resumeActivityList.remove(activity);
                    if (resumeActivityList.isEmpty()) {
                        if (appStateListener != null) appStateListener.onBackground();
                    }
                }

                @Override
                public void onActivityResumed(Activity activity) {
                }

                @Override
                public void onActivityCreated(Activity activity, Bundle savedInstanceState) {

                }

                @Override
                public void onActivityStarted(Activity activity) {
                    if(!resumeActivityList.contains(activity)) {
                        resumeActivityList.add(activity);
                        if(resumeActivityList.size() == 1) {
                            if (appStateListener != null) appStateListener.onForeground();
                        }
                    }
                }

                @Override
                public void onActivitySaveInstanceState(Activity activity, Bundle outState) {

                }

                @Override
                public void onActivityPaused(Activity activity) {

                }

                @Override
                public void onActivityDestroyed(Activity activity) {

                }

            };
            ((Application) mContext).registerActivityLifecycleCallbacks((ActivityLifecycleCallbacks) lifecycleCallbacks);
        }
    }

    interface AppStateListener {
        void onForeground();

        void onBackground();
    }

    // For service check.
    @Deprecated
    private boolean duringChecking = false;

    /**
     * \~chinese
     * 诊断 Chat 服务。
     * 
     * 服务诊断流程如下:
     *
     * 1.校验用户 ID 和密码；
     * 2.从服务端获取 DNS 列表；
     * 3.从服务端获取 token；
     * 4.连接 chat 服务器；
     * 5.断开连接(如果检查前已经有账户登录，则不执行该步骤)。
     * 
     * 如果在诊断过程中产生了错误，该流程将会被打断。
     *
     * @param username 用于服务诊断的用户 ID，如果已有账户登录，该用户 ID 会被替换为已登录账户的用户 ID，以防止更改当前已登录账户
     *                 的信息，比如 token 等。
     * @param password 密码，如果已有账户登录，该密码会被替换为已登录账户的密码。
     * @param listener 诊断结果的回调。
     * @deprecated 已废弃
     *
     * \~english
     * Checks the Chat service.
     * 
     * The service check process is as follows:
     *
     * 1. Validates the user ID and password.
     * 2. Gets the DNS list from the server.
     * 3. Gets the token from the server.
     * 4. Connects to the IM server.
     * 5. Logs out of the Chat app. (If a user logs in before the check, this step will be ignored.)
     * 
     * If an error occurs during the check, the check process will be interrupted.
     *
     * @param username The user ID for the service check. If a user account has logged in before, this user ID
     *                 will be changed to the logged-in user ID to avoid changes to the information of the current logged-in
     *                 user account.
     * @param password The password. If a user account has logged in before, the password will be changed to the password of the logged-in user.
     * @param listener The service check result callback.
     * @deprecated Deprecated.
     */
    @Deprecated
    public void check(String username, String password, final CheckResultListener listener) {
        // Already during checking, just return.
        if (duringChecking) {
            EMLog.i("EMServiceChecker", "During service checking, please hold on...");
            return;
        }

        duringChecking = true;

        // If is logged in before, the username must be current user.
        if (isLoggedInBefore()) {
            username = getCurrentUser();
            EMSessionManager sessionMgr = EMSessionManager.getInstance();
            password = sessionMgr.getLastLoginPwd();
        }

        final String finalUser = username;
        final String finalPwd = password;

        new Thread(new Runnable() {
            @Override
            public void run() {
                /**
                 * Contains account-validation check, get-dns check, get-Token check, login check.
                 * So the {@link EMAChatClient.CheckResultListener#onResult(int, int, String)} callback
                 * will be called four times.
                 */
                emaObject.check(finalUser, finalPwd, new EMAChatClient.CheckResultListener() {
                    @Override
                    public void onResult(final int type, final int result, final String desc) {
                        EMLog.i("EMServiceChecker", "type: " + type + ", result: " + result + ", desc: " + desc);
                        // Notify user once per callback.
                        notifyCheckResult(listener, type, result, desc);

                        // If occur a error during four checks, return.
                        if (result != EMError.EM_NO_ERROR) {
                            duringChecking = false;
                            return;
                        }

                        // If login successfully, logout
                        if (type == EMCheckType.DO_LOGIN) {
                            checkLogout(listener);
                        }
                    }
                });
            }
        }).start();
    }

    @Deprecated
    private void checkLogout(CheckResultListener listener) {
        // If is not a logged in service check, logout and notify the user.
        if (!isLoggedInBefore()) {
            logout();
            notifyCheckResult(listener, EMCheckType.DO_LOGOUT, EMError.EM_NO_ERROR, "");
        }

        duringChecking = false;
    }

    @Deprecated
    private void notifyCheckResult(CheckResultListener listener, @EMCheckType.CheckType int type, int result, String desc) {
        if (listener == null) return;
        listener.onResult(type, result, desc);
    }

    @Deprecated
    public interface CheckResultListener {
        void onResult(@EMCheckType.CheckType int type, int result, String desc);
    }

    boolean isDatabaseOpened() {
        return mIsDBOpened;
    }

}
