/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.util;

import android.content.Context;
import android.content.SharedPreferences;
import android.text.TextUtils;

import java.util.Map;
import java.util.UUID;

/**
 * Tries to get the device ID as a UUID and fallsback to a generated UUID value
 * if it doesn't work.
 * 
 * @see http 
 *      ://stackoverflow.com/questions/2785485/is-there-a-unique-android-device
 *      -id
 * 
 */
public class DeviceUuidFactory {
	protected static final String PREFS_FILE = "device_id.xml";
	protected static final String PREFS_DEVICE_ID = "device_id";
	protected static final String PREFS_DEVICE_ID_BAK_FILE = "prefs_device_id_bak";
	protected static final String PREFS_DEVICE_ID_BAK = "device_id_bak";

	protected static UUID uuid;

	public DeviceUuidFactory(Context context) {
		if (uuid == null) {
			final SharedPreferences prefs = context
					.getSharedPreferences(PREFS_FILE, 0);
			final String id = prefs.getString(PREFS_DEVICE_ID, null);

			if (id != null) {
				// Use the ids previously computed and stored in the
				// prefs file
				uuid = UUID.fromString(id);

			} else {
				EMLog.e("DeviceUuidFactory", "Need to generate device uuid");
				// Use random uuid to replace of android_id
				uuid = generateDeviceUuid(context);

				// Write the value out to the prefs file
				prefs.edit()
						.putString(PREFS_DEVICE_ID, uuid.toString())
						.commit();

			}
		}
	}

	private UUID generateDeviceUuid(Context context) {
		return UUID.randomUUID();
	}

	/**
	 * Returns a unique UUID for the current android device. As with all UUIDs,
	 * this unique ID is "very highly likely" to be unique across all Android
	 * devices. Much more so than ANDROID_ID is.
	 * 
	 * The UUID is generated by using ANDROID_ID as the base key if appropriate,
	 * falling back on TelephonyManager.getDeviceID() if ANDROID_ID is known to
	 * be incorrect, and finally falling back on a random UUID that's persisted
	 * to SharedPreferences if getDeviceID() does not return a usable value.
	 * 
	 * In some rare circumstances, this ID may change. In particular, if the
	 * device is factory reset a new device ID may be generated. In addition, if
	 * a user upgrades their phone from certain buggy implementations of Android
	 * 2.2 to a newer, non-buggy version of Android, the device ID may change.
	 * Or, if a user uninstalls your app on a device that has neither a proper
	 * Android ID nor a Device ID, this ID may change on reinstallation.
	 * 
	 * Note that if the code falls back on using TelephonyManager.getDeviceId(),
	 * the resulting ID will NOT change after a factory reset. Something to be
	 * aware of.
	 * 
	 * Works around a bug in Android 2.2 for many devices when using ANDROID_ID
	 * directly.
	 * 
	 * @see http://code.google.com/p/android/issues/detail?id=10603
	 * 
	 * @return a UUID that may be used to uniquely identify your device for most
	 *         purposes.
	 */
	public UUID getDeviceUuid() {
		return uuid;
	}
	
	private static boolean isEmpty(Object s) {
        if (s == null) {
            return true;
        }
        if ((s instanceof String) && (((String) s).trim().length() == 0)) {
            return true;
        }
        if (s instanceof Map) {
            return ((Map<?, ?>) s).isEmpty();
        }
        return false;
    }

	/**
	 * Save device id to bak file
	 * @param context
	 * @param deviceId
	 */
	public static void saveBakDeviceId(Context context, String deviceId) {
		if(context == null || TextUtils.isEmpty(deviceId)) {
		    return;
		}
		EMLog.e("DeviceUuidFactory", "saveBakDeviceId");
		SharedPreferences sp = context.getSharedPreferences(PREFS_DEVICE_ID_BAK_FILE, Context.MODE_PRIVATE);
		sp.edit().putString(PREFS_DEVICE_ID_BAK, deviceId).commit();
	}

	/**
	 * Get device id from bak file
	 * @param context
	 * @return
	 */
	public static String getBakDeviceId(Context context) {
		if(context == null) {
			return "";
		}
		SharedPreferences sp = context.getSharedPreferences(PREFS_DEVICE_ID_BAK_FILE, Context.MODE_PRIVATE);
		return sp.getString(PREFS_DEVICE_ID_BAK, "");
	}
}