/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *  
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.content.Context;
import android.text.TextUtils;

import android.support.annotation.NonNull;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMMessageListener;
import com.hyphenate.chat.core.EMChatConfigPrivate;
import com.hyphenate.push.EMPushConfig;
import com.hyphenate.util.EMLog;

/**
 * \~chinese
 * 提供 SDK 聊天相关的设置。
 * 
 * 用户可以用来配置 SDK 的各种参数、选项，
 * 
 * 比如，发送消息加密，是否自动接受加好友邀请。
 * 
 * \~english
 * The settings of the chat SDK.
 * 
 * You can set parameters and options of the SDK.
 * 
 * For example, whether to encrypt the messages before sending, whether to automatically accept the friend invitations.
 */
public class EMOptions {

	private final String TAG=getClass().getSimpleName();

	public static class AreaCode {
		public static final int AREA_CODE_CN = 1;
		public static final int AREA_CODE_NA = 2;
		public static final int AREA_CODE_EU = 4;
		public static final int AREA_CODE_AS = 8;
		public static final int AREA_CODE_JP = 16;
		public static final int AREA_CODE_IN = 32;
		public static final int AREA_CODE_GLOB = -1;

		public AreaCode() {
		}
	}



	private int areaCode = AreaCode.AREA_CODE_GLOB;

	/**
	 * \~chinese
	 * 是否自动接受加好友邀请。
	 * - （默认）`true`：自动接受好友申请；
	 * - `false`：不会自动接受好友申请。
	 *
	 * \~english
	 * Whether to accept friend invitations from other users automatically. 
	 * - (Default)`true`: Accepts friend invitations automatically;
	 * - `false`: Do not accept friend invitations automatically.
	 */
	private boolean acceptInvitationAlways = true;
	
	/**
	 * \~chinese
	 * 是否自动接受群组邀请。
	 * - （默认）`true`：自动接受群组申请；
	 * - `false`：不会自动接受群组申请。
	 *
	 * \~english
	 * Whether to accept group invitations automatically. 
	 * - (Default)`true`: Accepts group invitations automatically;
	 * - `false`: Do not accept group invitations automatically.
	 */
	private boolean autoAcceptGroupInvitation = true;
	
	/**
	 * \~chinese
	 * 是否对发送消息进行加密。
	 * - （默认）`false`：不加密；
	 * - `true`：加密。
	 *
	 * \~english
	 * Whether to encrypt the messages. 
	 * - (Default)`false`: Do not encrypt the messages;
	 * - `true`: Encrypt the messages.
	 */
	private boolean useEncryption = false;

	/**
	 * \~chinese
	 * 是否发送消息已读回执.
	 * - （默认）`true`：发送已读回执；
	 * - `false`：不发送已读回执。
	 *
	 * \~english
	 * Whether the message read receipt is required. 
	 * - (Default)`true`: The message read receipt is required. 
	 * - `false`: The message read receipt is NOT required.
	 */
	private boolean requireReadAck = true;
	
	/**
	 * \~chinese
	 * 是否发送消息已送达回执。
	 * - （默认）`false`：不发送已送达回执，SDK 收到单聊消息时不会自动发送送达回执。
	 * - `true`：发送已送达回执。
	 *
	 * \~english
	 * Whether to send the message delivery receipt. 
	 * - (Default)`false`: The SDK does not automatically send the delivery receipt when you receive a chat message;
	 * - `true`: The SDK automatically sends the delivery receipt when you receive a chat message.
	 */
	private boolean requireDeliveryAck = false;

    /**
	 * \~chinese
	 * 离开群组时是否删除消息。
	 * - （默认）`true`：离开群组时删除群组消息；
	 * - `false`：离开群组时不删除群组消息。
	 *
	 * \~english
	 * Whether to delete all of the group messages when leaving the group.
	 * - (Default)`true`: Deletes all of the group messages when leaving the group. 
	 * - `false`: Do not delete all of the group messages when leaving the group.
	 */
	private boolean deleteMessagesAsExitGroup = true;
	
    /**
	 * \~chinese
	 * 是否允许聊天室所有者离开并删除会话记录。
	 * - （默认）`true`：允许聊天室所有者离开；
	 * - `false`：不允许聊天室所有者离开。
	 * 
	 * \~english
	 * Whether to allow the chat room owner to leave the chat room.
	 * - (Default)`true`: Allow the chat room owner to leave the chat room. 
	 * - `false`: Do not allow the chat room owner to leave the chat room.
	 */

	private boolean isChatroomOwnerLeaveAllowed = true;

	/**
	 * \~chinese
	 * 离开聊天室是否删除消息。
	 * -（默认）`true`：离开聊天室删除聊天室消息；
	 * - `false`：离开聊天室不删除聊天室消息。
	 *
	 * \~english
	 * Whether to delete all of the chat room messages when leaving the chat room.
	 * - (Default)`true`: Deletes all of the chat room messages when leaving the chat room. 
	 * - `false`: Do not delete all of the chat room messages when leaving the chat room.
	 */
	private boolean deleteMessagesAsExitChatRoom = true;

	/**
	 * \~chinese
	 * 从本地数据库加载会话时是否包括空会话。
	 * -（默认）`false`：不包括；
	 * - `true`：包括。
	 *
	 * \~english
	 * Whether to include empty conversations when the SDK loads conversations from the local database.
	 * - (Default)`false`: Empty conversations are excluded.
	 * - `true`: Empty conversations are included.
	 */
	private boolean isLoadEmptyConversations = false;

	private boolean useReplacedMessageContents=false;

	private boolean includeSendMessageInMessageListener =false;
	private boolean regardImportedMsgAsRead = false;

	private String appkey = "";
	
	private EMChatConfigPrivate config = null;
	
	private boolean enableAutoLogin = true;

	private String fcmNumber = null;
	private boolean useFCM = true;

	private boolean enableDNSConfig = true;
	
	private boolean sortMessageByServerTime = true;

	private boolean useHttps = false;
	private String dnsUrl = "";

	private String restServer;
	private String reportServer;
	private String imServer;
	private int imPort;

	private boolean usingHttpsOnly = true;
	private boolean serverTransfer = true;
	private boolean isAutodownload = true;
	private boolean useStereoInput = false;
	private boolean enableStatistics = false;
	private boolean enableUseRtcConfig = false;
	private String  rtcConfigUrl;
	private EMPushConfig pushConfig;

	private int osCustomPlatform = -1;
	private String customDeviceName;

	private String customExt;

	private String nativeLibBasePath="";

	private boolean enableTLSConnection=false;

	private boolean autoLoadAllConversations = true;

	/**
	 * \~chinese
	 * 获取是否需要消息接收方发送已读回执的设置。
	 *
	 * @return 是否要求消息的接收方发送已读回执。
	 * - （默认）`true`：需要已读回执；
	 * - `false：不需要已读回执。
	 * 
	 * \~english
	 * Gets whether the read receipt is required by the message receiver. 
	 * 
	 * @return Whether the read receipt is required.
	 * - (Default) `true`: The read receipt is required;
	 * - `false`: The read receipt is not required.
	 */
	public boolean getRequireAck() {
		
		if(config == null){
			return requireReadAck;
		}
		return  config.getRequireReadAck();
	}

	/**
	 * \~chinese
	 * 设置是否需要接受方发送已读回执。
	 *
	 * @param requireAck 是否需要接收方发送已读回执。
	 * - （默认）`true`：需要发送已读回执；
	 * - `false`：不需要发送已读回执。
	 * 
	 * \~english
	 * Sets whether to require the read receipt. 
	 *
	 * @param requireAck Whether the read receipt is required.
	 * - (Default) `true`: The read receipt is required; 
	 * - `false`: The read receipt is not required.
	 */
	public void setRequireAck(boolean requireAck) {
		this.requireReadAck = requireAck;

		if (config == null) {
			return;
		}
		config.setRequireReadAck(requireAck);
	}
	
	/**
	 * \~chinese
	 * 获取送达回执设置。
	 * 
	 * @return 是否需要送达回执。
	 * - （默认）`true`：要求消息的接受方发送送达回执；
	 * - `false`： 不要求消息的接受方发送送达回执。
	 * 
	 * \~english
	 * Gets whether the delivery receipt is required.
	 *
	 * @return Whether the delivery receipt is required.
	 * - (Default)`true`: The read receipt is required; 
	 * - `false`: The read receipt is not required.
	 */
	public boolean getRequireDeliveryAck() {
		if (config == null) {
			return requireDeliveryAck;
		}
		return config.getRequireDeliveryAck();
	}

	/**
	 * \~chinese
	 * 设置是否需要接受方发送送达回执。
	 *
	 * @param requireDeliveryAck 是否需要送达回执。
	 * - （默认）`true`：需要送达回执。
	 * - `false`：不需要发送送达回执。
	 * 
	 * \~english
	 * Sets whether the delivery receipt is required.
	 * 
	 * @param requireDeliveryAck Whether the delivery receipt is required.
	 * - (Default)`true`: The delivery receipt is required; 
	 * - `false`: The delivery receipt is not required.
	 */
	public void setRequireDeliveryAck(boolean requireDeliveryAck) {
		this.requireDeliveryAck = requireDeliveryAck;
		
		if (config == null) {
			return;
		}
		config.setRequireDeliveryAck(requireDeliveryAck);
	}

	/**
	 * \~chinese
	 * 获取是否自动接受加好友邀请。
	 * 
	 * @return 是否自动接受加好友邀请。
	 * - （默认）`true`：自动接受好友邀请。
	 * - `false`：不自动接收好友邀请。
	 * 
	 * \~english
	 * Gets whether to accept other user's friend invitations automatically.
	 *
	 * @return Whether to accept friend invitation automatically.
	 * - (Default)`true`: Accepting friend invitation automatically.
	 * - `false`: Do not accepting friend invitation automatically.
	 */
	public boolean getAcceptInvitationAlways() {
		if (config == null) {
			return acceptInvitationAlways;
		}
		return config.getAutoAccept();
	}

	/**
	 * \~chinese
	 * 设置是否自动接受加好友邀请。
	 * 
	 * @param value 是否自动接受加好友邀请。
	 * - （默认）`true`：自动接受好友邀请。
	 * - `false`：不自动接收好友邀请。
	 * 
	 * \~english
	 * Sets whether to accept the friend invitation automatically. 
	 * 
	 * @param value Whether to accept the friend invitation automatically.
	 * - (Default)`true`: Accepting friend invitation automatically.
	 * - `false`: Do not accepting friend invitation automatically.
	 */
	public void setAcceptInvitationAlways(boolean value) {
		if (config == null) {
			acceptInvitationAlways = value;
			return;
		}
		config.setAutoAccept(value);
	}

	/**
	 * \~chinese
     * 设置退出(主动和被动退出)群组时是否删除聊天消息。
	 *
     * @param delete 
	 * - （默认）`true`: 退出群组时删除群组消息。
     * - `false`: 退出群组时不删除群组消息。 
     * 
     * \~english
	 * Sets whether to delete the group messages when leaving the group.
	 *
     * @param delete  
	 * - (Default)`true`: Delete the messages when leaving the group.
	 * - `false`: Do not delete the messages when leaving a group.
     */
    public void setDeleteMessagesAsExitGroup(boolean delete){
		if (config == null) {
			deleteMessagesAsExitGroup = delete;
			return;
		}
		
		config.setDeleteMessageAsExitGroup(delete);
    }
    
    /**
	 * \~chinese
     * 获取退出(主动和被动退出)群组时是否删除聊天消息。
	 * 
     * @return  - （默认）`true`: 退出群组时删除群组消息。
     * - `false`: 退出群组时不删除群组消息。
	 * 
	 * @deprecated 使用 {@link EMOptions#deleteMessagesOnLeaveGroup()} 替代。
     * 
     * \~english
	 * Gets whether to delete the group message when leaving a group.
	 * 
     * @return Whether to delete the group message when leaving a group:
	 * 
	 * - (Default)`true`: Delete the messages when leaving a group.
     * - `false`: Do not delete the messages when leaving a group.
	 * 
	 * @deprecated Use {@link EMOptions#deleteMessagesOnLeaveGroup()} instead.
     */
    @Deprecated
    public boolean isDeleteMessagesAsExitGroup() {
		return deleteMessagesOnLeaveGroup();
    }

	/**
	 * \~chinese
	 * 获取退出(主动和被动退出)群组时是否删除聊天消息。
	 * 
     * @return  - （默认）`true`: 退出群组时删除群组消息。
     * - `false`: 退出群组时不删除群组消息。
	 *
	 * \~english
	 * Gets whether to delete the group message when leaving a group.
	 * 
     * @return - (Default)`true`: Delete the messages when leaving a group.
     * - `false`: Do not delete the messages when leaving a group.
	 */
	public boolean deleteMessagesOnLeaveGroup() {
		if (config == null) {
			return deleteMessagesAsExitGroup;
		}

		return config.getDeleteMessageAsExitGroup();

	}
    
    /**
     * \~chinese
     * 设置是否自动接受加群邀请。
	 * 
     * @param value 是否自动接受加群邀请。
	 * - （默认）`true`：自动接受加群申请；
     * - `false`: 不自动接受加群申请。
     * 
     * \~english
     * Sets whether to accept a group invitation automatically. 
	 * 
     * @param value Whether to accept group invitation automatically.
	 * - (Default)`true`: Accept group invitations automatically；
	 * - `false`: Do not accept group invitations automatically.
     */
    public void setAutoAcceptGroupInvitation(boolean value) {
        if (config == null) {
            autoAcceptGroupInvitation = value;
            return;
        }
        config.setAutoAcceptGroupInvitation(value);
    }
    
    /**
     * \~chinese
     * 获取是否自动接受加群邀请。
	 * 
     * @return 返回是否自动接受加群邀请。
	 * - （默认） `true`：自动接受加群邀请；
	 * - `false`：不自动接受加群邀请。
	 * 
	 * @deprecated 使用 {@link EMOptions#autoAcceptGroupInvitations()} 替代。
     * 
     * \~english
     * Gets whether to accept a group invitation automatically.
	 * 
     * @return	Returns the result of whether to accept group invitation automatically.
	 * - (Default)`true`: Accept group invitations automatically；
	 * - `false`: Do not accept group invitations automatically.
	 * 
	 * @deprecated Use {@link EMOptions#autoAcceptGroupInvitations()} instead.
     */
    @Deprecated
    public boolean isAutoAcceptGroupInvitation() {
        return autoAcceptGroupInvitations();
    }

	/**
	 * \~chinese
	 * 获取是否自动接受加群邀请。
	 * 
	 * @return 返回是否自动接受加群邀请。
	 * - （默认） `true`：自动接受加群邀请；
	 * - `false`：不自动接受加群邀请。
	 *
	 * \~english
	 * Gets whether to accept group invitations automatically.
	 * 
	 * @return	Returns the result of whether to accept group invitation automatically. 
	 * - (Default)`true`: Accept group invitations automatically.
	 * - `false`: Do not accept group invitations automatically.
	 */
	public boolean autoAcceptGroupInvitations() {
		if (config == null) {
			return autoAcceptGroupInvitation;
		}
		return config.isAutoAcceptGroupInvitation();
	}

	/**
	 * \~chinese
	 * 设置是否允许聊天室所有者离开并删除会话记录。
	 * 
	 * @param allowed 是否允许聊天室所有者离开。
	 * - （默认） `true`：允许。即使聊天室所有者离开，该所有者仍具有聊天室的所有权限，只不过不再接收任何消息。
	 * - `false`：不允许。
	 * 
	 * \~english
	 * Sets whether to allow the chat room owner to leave the chat room.
	 * 
	 * @param allowed Whether to allow the chat room owner to leave the chat room.
	 * - (Default)`true`: Yes. Even if the chat room owner leaves the chat room, the owner still has all privileges, except for receiving messages in the chat room.
	 * - `false`: No. The chat room owner cannot leave the chat room.
	 */
    public void allowChatroomOwnerLeave(boolean allowed){
		if (config == null) {
			this.isChatroomOwnerLeaveAllowed = allowed;
			return;
		}
		
		config.setIsChatroomOwnerLeaveAllowed(allowed);
	}
	
	/**
	 * \~chinese
	 * 获取是否允许聊天室所有者离开。
	 * 
	 * @return 否允许聊天室所有者离开：
	 * - （默认）`true`：允许。即使聊天室所有者离开，该所有者仍具有聊天室的所有权限，只不过不再接收任何消息。
	 * - `false`：不允许。
	 * @deprecated 使用 {@link EMOptions#canChatroomOwnerLeave()} 代替。
	 * 
	 * \~english
	 * Gets whether to allow the owner of chat room to leave.
	 * 
	 * @return Whether to allow the owner of chat room to leave:
	 * - (Default)`true`: Yes. Even if the chat room owner leaves the chat room, the owner still has all privileges, except for receiving messages in the chat room. 
	 * - `false`: When the owner leave do keep the conversation.
	 *  @deprecated Use {@link EMOptions#canChatroomOwnerLeave()} instead.
	 */
	@Deprecated
	public boolean isChatroomOwnerLeaveAllowed(){

		return canChatroomOwnerLeave();
	}

	/**
	 * \~chinese
	 * 获取是否允许聊天室所有者离开。
	 * @return 是否允许聊天室所有者离开：
	 * - （默认）`true`：允许聊天室所有者离开；
	 * - `false`：不允许聊天室所有者离开。
	 *
	 * \~english
	 * Gets whether to allow owner of chat room to leave.
	 * @return Whether to allow the chat room owner to leave:
	 *         - (Default)`true`: Yes.
	 *         - `false`: No. The chat room owner cannot leave the chat room.
	 */
	public boolean canChatroomOwnerLeave(){
		if (config == null) {
			return isChatroomOwnerLeaveAllowed;
		}
		return config.getIsChatroomOwnerLeaveAllowed();
	}

	/**
	 * \~chinese
	 * 设置退出(主动和被动退出)聊天室时是否删除聊天消息。
	 * 
	 * @param delete 退出(主动和被动退出)聊天室时是否删除聊天消息：
	 *               - （默认）`true`：删除。
	 *               - `false`：保留。
	 *
	 * \~english
	 * Sets whether to delete the chat room message when leaving the chat room.
	 * 
	 * @param delete Whether to delete the chat room message when leaving the chat room:
	 *              - (Default)`true`: Delete the chat room related message record when leaving the chat room.
	 *              - `false`: Do not delete the chat room related message record when leaving the chat room.
	 */
	public void setDeleteMessagesAsExitChatRoom(boolean delete){
		if (config == null) {
			deleteMessagesAsExitChatRoom = delete;
			return;
		}

		config.setDeleteMessageAsExitChatRoom(delete);
	}

	/**
	 * \~chinese
	 * 获取退出(主动和被动退出)聊天室时是否删除聊天消息。
	 * 
	 * @return  退出(主动和被动退出)聊天室时是否删除聊天消息：
	 *         - （默认）`true`：删除。
	 *         - `false`：保留。
	 * 
	 * @deprecated 使用 {@link EMOptions#deleteMessagesOnLeaveChatroom()} 替代。
	 *
	 * \~english
	 * Gets whether to delete the chat room message when leaving the chat room.
	 * 
	 * @return  Whether to delete the chat room message when leaving the chat room:
	 *          - (Default)`true`: Delete the chat room related message record when leaving the chat room.
	 *          - `false`: Do not delete the chat room related message record when leaving the chat room.
	 * 
	 * @deprecated Use {@link EMOptions#deleteMessagesOnLeaveChatroom()} instead.
	 */
	@Deprecated
	public boolean isDeleteMessagesAsExitChatRoom() {

		return deleteMessagesOnLeaveChatroom();

	}
	/**
	 * \~chinese
	 * 获取退出(主动和被动退出)聊天室时是否删除聊天消息。
	 * 
	 * @return 退出(主动和被动退出)聊天室时是否删除聊天消息：
	 *         -（默认）`true`：退出聊天室时删除聊天室相关消息记录。
	 *         - `false`：离开聊天室时保留会话记录。
	 *
	 * \~english
	 * Gets whether to delete the chat room message when leaving the chat room.
	 * @return Whether to delete the chat room message when leaving the chat room:
	 *          - (Default)`true`: Delete the chat room related message record when leaving the chat room.
	 *          - `false`: Do not delete the chat room related message record when leaving the chat room.
	 */
	public boolean deleteMessagesOnLeaveChatroom() {
		if (config == null) {
			return deleteMessagesAsExitChatRoom;
		}

		return config.getDeleteMessageAsExitChatRoom();

	}
	 
	/**
	 * \~chinese
	 * 获取是否按照服务器收到时间进行排序。
	 * 
	 * @return 返回是否按照服务器收到的时间排序。
	 * - （默认）`true`：按服务器收到消息时间进行排序；
	 * - `false`：不按服务器收到消息时间进行排序。
	 * 
	 * \~english
	 * Gets whether to sort messages by the time when the server received them.
	 * 
	 * @return Returns whether to sort messages by the time when the server received them.
	 * - (Default)`true`: Yes. Messages are sorted by the time when the server received them;
	 * - `false`: No. Messages are not sorted by the time when the server received them.
	 */
    public boolean isSortMessageByServerTime() {
        if (config == null) {
            return sortMessageByServerTime;
        }
        return config.getSortMessageByServerTime();
    }

	/**
	 * \~chinese
	 * 设置是否按照服务器收到的时间对消息进行排序。
	 * 
	 * @param sortByServerTime 是否按照服务器收到的时间对消息进行排序。
	 * - （默认）`true`：按服务器收到消息时间进行排序；
	 * - `false`：不按服务器收到消息时间进行排序。
	 *
	 * \~english
	 * Sets whether to sort messages by the time when the server received them.
	 * 
	 * @param sortByServerTime Whether to sort messages by the time when the server received them:
	 * - (Default)`true`: Yes;
	 * - `false`: No.
	 */
	public void setSortMessageByServerTime(boolean sortByServerTime) {
        if (config == null) {
            this.sortMessageByServerTime = sortByServerTime;
            return;
        }
        
        config.setSortMessageByServerTime(sortByServerTime);
    }
	
	/**
	 * \~chinese
	 * 设置 app key。
	 * 
	 * @param appkey 创建 app 时在 console 后台上注册的 app 唯一识别符。
	 * 
	 * \~english
	 * Sets the app key.
	 * 
	 * @param appkey The app key you got from the console when creating a chat app.
	 */
	void updatePath(String appkey){
		if(config != null) {
            config.updatePath(appkey);
		}
	}

	/**
	 * \~chinese
	 * 设置 app key。
	 * 
	 * @param appkey 创建 app 时在 console 后台上注册的 app 唯一识别符。
	 *
	 * \~english
	 * Sets the app key.
	 * 
	 * @param appkey The app key you got from the console when creating an app.
	 */
	public void setAppKey(String appkey) {
		this.appkey = appkey;
		updatePath(appkey);
	}
	
	/**
	 * \~chinese
	 * 获取 app key。
	 *
	 * \~english
	 * Gets the app key.
	 */
	public String getAppKey(){
		if(config == null){
			return appkey;
		}
		
		return config.getAppKey();
	}

	/**
	 * \~chinese
	 * 设置 im 消息服务器地址，一般为私有部署服务，在开发者想实现数据隔离、特别注重数据安全时使用，如有需求请联系商务。
	 * 
	 * @param imServer im 消息服务器地址。
	 *
	 * \~english
	 * Sets the custom IM message server URL.
	 * 
	 * @param imServer The custom IM message server URL.
     */
	public void setIMServer(String imServer){
		if(imServer == null){
			return;
		}

		this.imServer = imServer;
		if(config == null){
			return;
		}
		config.setChatServer(imServer);
	}

	/**
	 * \~chinese
	 * 获取设置的 im 消息服务器地址，一般为私有部署服务，在开发者想实现数据隔离、特别注重数据安全时使用，如有需求请联系商务。
	 * 
	 * @return im 消息服务器地址。
	 *
	 * \~english
	 * Gets the custom IM message server URL.
	 * @return The custom IM message server URL.
	 */
	public String getImServer(){
		return imServer;
	}

	/**
	 * \~chinese
	 * 设置 im 消息服务器端口号，一般为私有部署服务，在开发者想实现数据隔离、特别注重数据安全时使用，如有需求请联系商务。
	 * 
	 * @param imPort 端口号。
	 *
	 * \~english
	 * Sets the custom IM server port.
	 * 
	 * @param imPort The custom IM server port.
	 */
	public void setImPort(int imPort){
		this.imPort = imPort;
		if(config == null){
			return;
		}
		config.setChatPort(imPort);
	}

	/**
	 * \~chinese
	 * 获取设置的 im 消息服务器端口号，一般为私有部署服务，在开发者想实现数据隔离、特别注重数据安全时使用，如有需求请联系商务。
	 * 
	 * @return 端口号。
	 *
	 * \~english
	 * Gets the custom IM server port.
	 * @return The custom IM server port.
	 */
	public int getImPort(){
		return imPort;
	}

	/**
	 * \~chinese
	 * 设置 REST 服务器地址，一般在开发者想实现数据隔离、特别注重数据安全时使用，如有需求请联系商务获取指定的服务器地址。
	 * 
	 * @param restServer REST 服务器地址。
	 *
	 * \~english
	 * Sets the custom REST server URL.
	 * 
	 * @param restServer The custom REST server URL.
	 */
	public void setRestServer(String restServer){
		if(restServer == null)
			return;

		this.restServer = restServer;
		if(config == null){
			return;
		}
		config.setRestServer(restServer);
	}

	/**
	 * \~chinese
	 * 获取设置的 REST 服务器地址，一般为私有部署服务，在开发者想实现数据隔离、特别注重数据安全时使用，如有需求请联系商务。
	 * 
	 * @return REST 服务器地址。
	 *
	 * \~english
	 * Gets the custom REST server URL.
	 * @return The custom REST server URL.
	 */
	public String getRestServer(){
		return restServer;
	}


	/**
	 * \~chinese
	 * 获取设置的数据上报服务器地址，一般为私有部署服务，在开发者想实现数据隔离、特别注重数据安全时使用，如有需求请联系商务。
	 * 
	 * @return 数据上报服务器地址，有可能为空。
	 *
	 * \~english
	 * Gets the URL of the custom server for data reporting.
	 * @return The URL of the server for data reporting, which may be null.
     */
	public String getReportServer(){
		return reportServer;
	}


	/**
	 * \~chinese
	 * 设置数据上报服务器地址，一般为私有部署服务，在开发者想实现数据隔离、特别注重数据安全时使用，如有需求请联系商务。
	 * 
	 * @param reportServer 数据上报服务器地址。
	 *
	 * \~english
	 * Sets the URL of the custom server for data reporting.
	 * 
	 * @param reportServer The URL of the custom server for data reporting.
	 */
	public void setReportServer(String reportServer){
		if(reportServer == null)
			return;
		this.reportServer = reportServer;
	}

	/**
	 * \~chinese
	 * 开启/关闭自动登录。
	 * 
	 * @param autoLogin 是否开启自动登录：
	 * - （默认）`true`：自动登录；
	 * - `false`：不自动登录。
	 * 
	 * \~english
	 * Enables/Disables automatic login.
	 * 
	 * @param autoLogin Whether to enable automatic login:
	 * - (Default)`true`: Enables automatic login;
	 * - `false`: Disables automatic login.
	 */
	public void setAutoLogin(boolean autoLogin){
		enableAutoLogin = autoLogin;
	}
	
	/**
	 * \~chinese
	 * 获取是否开启了自动登录。
	 * 
	 * @return 是否开启了自动登录：
	 * - （默认）`true`：已设置为自动登录；
	 * - `false`：未设置为自动登录。
	 * 
	 * \~english
	 * Checks whether the automatic login is enabled.
	 * 
	 * @return Whether the automatic login is enabled:
	 * - (Default)`true`: Automatic login is enabled;
	 * - `false`: Automatic login is disabled.
	 */
	public boolean getAutoLogin(){
		return enableAutoLogin;
	}

	void setConfig(EMChatConfigPrivate config){
		this.config = config;
	}

	/**
	 * \~chinese
	 * 设置是否关闭 DNS。
	 * 
	 * @param enable 是否关闭 DNS。
	 * - （默认）`true`：关闭 DNS，私有云部署需要关闭；
	 * - `false`：不关闭 DNS。
	 *
	 * \~english
	 * Sets whether to disable DNS.
	 * 
	 * @param enable Whether to disable DNS:
	 * - (Default)`true`: Yes;
	 * - `false`: No.
	 */
	public void enableDNSConfig(boolean enable) {
		enableDNSConfig = enable;
        if (config == null) {
            return;
        }
	    config.enableDnsConfig(enable);
	}

	/**
	 * \~chinese
	 * 获取是否关闭 DNS。
	 *
	 * @return 返回是否关闭 DNS。
	 * - （默认）`true`：关闭 DNS。
	 * - `false`：不关闭 DNS。
	 *
	 * \~english
	 * Gets whether DNS is disabled. 
	 * - (Default) `true`: Yes;
	 * - `false`: No.
	 *
	 * @return Returns whether to disable DNS.
	 */
	public boolean getEnableDNSConfig() {
	    return enableDNSConfig;
	}

	/**
	 * \~chinese
	 * 设置只使用 HTTPS 进行 REST 操作。
	 * 
	 * @param _usingHttpsOnly 是否只使用 HTTPS。
	 * - （默认）`true`：只使用 HTTPS 进行 REST 操作；
	 * - `false`：可以使用 HTTPS 和 HTTP 进行 REST 操作。
	 *
	 * \~english
	 * Sets whether only HTTPS is used for REST operation.
	 * 
	 * @param _usingHttpsOnly Whether only HTTPS is used.
	 * - (Default)`true`: Only HTTPS is used;
	 * - `false`: Both HTTP and HTTPS can be used. 
	 */
	public void setUsingHttpsOnly(boolean _usingHttpsOnly) {
        usingHttpsOnly = _usingHttpsOnly;
		if (config == null) {
			return;
		}
		config.setUsingHttpsOnly(_usingHttpsOnly);
	}

	/**
	 * \~chinese
	 * 获取是否只使用 HTTPS 进行 REST 操作。
	 *
	 * @return 是否只使用 HTTPS。
	 * - （默认）`true` 只使用 HTTPS 进行 REST 操作；
	 * - `false` 可以使用 HTTPS 和 HTTP 进行 REST 操作。
	 * 
	 * \~english
	 * Gets whether only HTTPS is used for REST operations. 
	 *
	 * @return Whether only HTTPS is used for REST operations. 
	 * - (Default) `true`: Only HTTPS is used;
	 * - `false`: Both HTTP and HTTPS can be used. 
	 */
	public boolean getUsingHttpsOnly() {
		if (config != null) {
			return config.getUsingHttpsOnly();
		}
		return usingHttpsOnly;
	}
	/**
	 * \~chinese
	 * 设置是否自动将消息附件上传到 Chat 服务器。
	 * 
	 * @param transfer 是否自动上传到 Chat 服务器。
	 * - （默认）`true`：自动使用 Chat 服务器上传下载；
	 * - `false`：不自动使用 Chat 服务器上传下载，自定义上传下载路径。
	 *
	 * \~english
	 * Sets whether to upload the message attachments automatically to the chat server.
	 * 
	 * @param transfer Whether to automatically upload the message attachments to the chat server.
	 * - (Default) `true`: Yes.
	 * - `false`: No. A custom path is used to for uploading.
	 */
	public void setAutoTransferMessageAttachments(boolean transfer) {
		if (config == null) {
			serverTransfer = transfer;
			return;
		}
		config.setAutoTransferMessageAttachments(transfer);
	}
	/**
	 * \~chinese
	 * 获取是否使用 Chat 服务器进行上传下载。
	 *
	 * @return 是否自动使用 Chat 服务器进行上传下载。
	 * - （默认）`true`：自动使用 Chat 服务器进行上传下载；
	 * - `false`：不自动使用 Chat 服务器上传下载，自定义上传下载路径。
	 *
	 * \~english
	 * Gets whether to upload the message attachments automatically to the chat server.
	 *
	 * @return Whether to upload the message attachments automatically to the chat server.
	 * - (Default)`true`: Yes.
	 * - `false`: No. A custom path is used to for uploading.
	 */
	public boolean getAutoTransferMessageAttachments() {
		if (config == null) {
			return serverTransfer;
		}
		return config.getAutoTransferMessageAttachments();
	}
	/**
	 * \~chinese
	 * 是否自动下载缩略图。
	 * 
	 * @param autodownload 是否自动下载缩略图。
	 * - （默认）`true`：自动下载；
	 * - `false`：不自动下载。
	 *
	 * \~english
	 * Sets whether to automatically download thumbnails.
	 *
	 * @param autodownload Whether to download the thumbnails automatically.
	 * - (Default)`true`: Yes;
	 * - `false`: No.
	 * 
	 */
	public void setAutoDownloadThumbnail(boolean autodownload) {
		if (config == null) {
			isAutodownload = autodownload;
			return;
		}
		config.setAutodownloadThumbnail(autodownload);
	}
	/**
	 * \~chinese
	 * 获取是否自动下载缩略图。
	 * 
	 * @return 是否自动下载缩略图。
	 * - （默认）`true`：自动下载；
	 * - `false`：不自动下载。
	 *
	 * \~english
	 * Gets whether to download the thumbnails automatically. 
	 * 
	 * @return Whether to download thumbnails automatically.
	 * - (Default)`true`: Yes;
	 * - `false`: No.
	 */
	public boolean getAutodownloadThumbnail() {
		if (config == null) {
			return isAutodownload;
		}
		return config.getAutodownloadThumbnail();
	}


	private int fixedInterval = -1;

	/**
	 * \~chinese
	 * 设置固定的心跳间隔，不设置时会自动探测最佳心跳间隔。
	 * 
	 * @param interval 心跳时间间隔，单位为秒，建议范围为 30s~300s 之间。
	 *
	 * \~english
	 * Sets a fixed heartbeat interval. 
	 * 
	 * If not set, the optimal heartbeat interval is automatically detected.
	 * 
	 * @param interval  The interval of heartbeat in seconds. The recommended value range is 30-300.
	 */
	public void setFixedHBInterval(int interval) {
		if (interval < 10) this.fixedInterval = 10;
		if (interval > 300) this.fixedInterval = 300;
		this.fixedInterval = interval;
	}

	/**
	 * \~chinese
	 * 获取固定的心跳间隔。
	 * 
	 * @return 心跳时间间隔，单位为秒。
	 *
	 * \~english
	 * Gets the fixed heartbeat interval.
	 * 
	 * @return The heartbeat interval in seconds. 
	 */
	public int getFixedInterval() {
		return this.fixedInterval;
	}


	/**
	 * \~chinese
	 * 获取 SDK 版本号。
	 * 
	 * @return SDK 版本号。
	 *
	 * \~english
	 * Gets the SDK version.
	 * 
	 * @return The SDK version.
	 */
	public String getVersion() {
		return config.getVersion();
	}

	/**
	 * \~chinese
	 * 获取本地保存的访问 token。
	 *
	 * @return 访问 token。
	 *
	 * \~english
	 * Gets the access token from the local database.
	 *
	 * @return The access token.
	 */
	public String getAccessToken() {
		return config.getAccessToken();
	}

	/**
	 * \~chinese
	 * 是否从服务器获取访问 token。
	 * 
	 * @param b 是否从服务器获取。
	 * - `true`：从服务器获取访问 token；
	 * - `false`：不从服务器获取访问 token。
	 * @return 访问 token。
	 *
	 * \~english
	 * Whether to get the access token from the server.
	 * 
	 * @param b Whether to get the access token from server.
	 * - (Default) `true`: Get the access token from server.
	 * - `false`: Do not get the access token from server.
	 * @return The access token.
	 */
	public String getAccessToken(boolean b) {
		return config.getAccessToken(b);
	}

	/**
	 * \~chinese
	 * 设置 DNS 地址。
	 * 
	 * @param dnsUrl DNS 地址。
	 *
	 * \~english
	 * Sets the DNS url.
	 * @param dnsUrl The DNS url.
	 */
	public void setDnsUrl(String dnsUrl) {
		this.dnsUrl = dnsUrl;
		if (config == null) {
			return;
		}
		config.setDnsUrl(dnsUrl);
	}

	/**
	 * \~chinese
	 * 获取 DNS 地址。
	 * 
	 * @return  DNS 地址。
	 *
	 * \~english
	 * Gets the DNS url.
	 * @return  The DNS url.
	 */
	public String getDnsUrl() {
		if (config != null) {
			return config.getDnsUrl();
		}
		return dnsUrl;
	}

	/**
	 * \~chinese
	 * 获取用户设置的推送相关配置。
	 *
	 * @return 推送配置。
	 *
	 * \~english
	 * Gets the push configurations.
	 */
	public EMPushConfig getPushConfig() {
		return pushConfig;
	}

	/**
	 * \~chinese
	 * 设置推送相关配置。
	 *
	 * @param pushConfig 推送相关配置。
	 *
	 * \~english
	 * Sets the push options.
	 *
	 * @param pushConfig  The push configuration.
	 */
	public void setPushConfig(EMPushConfig pushConfig) {
		this.pushConfig = pushConfig;
	}

	/**
	 * \~chinese
	 * 获取是否开启了本地消息的流量统计功能。
	 *
	 * @return 是否开启了消息流量统计功能：
	 * 			- `true`：开启；
	 * 			- （默认）`false`：关闭。
	 *
	 * \~english
	 * Checks whether the traffic statistics collection is enabled for local messages.
	 *
	 * @return Whether the message traffic statistics collection is enabled:
	 * 			- `true`: Enabled;
	 * 			- (Default)`false`: Disabled.
	 */
	public boolean isEnableStatistics() {
		if (config == null) {
			return enableStatistics;
		}
		return config.isEnableStatistics();
	}

	/**
	 * \~chinese
	 * 设置是否开启本地消息的流量统计功能。
	 * 
	 * 需要在调用 {@link EMClient#init(Context, EMOptions)} 之前进行设置。
	 *
	 * @param enableStatistics 是否开启消息流量统计功能：
	 * - `true`：开启；
	 * - （默认）`false`：关闭。
	 *
	 * \~english
	 * Sets whether to enable traffic statistics collection for local messages.
	 * 
	 * You need to enable traffic statistics collection before calling {@link EMClient#init(Context, EMOptions)}.
	 * 
	 * @param enableStatistics Whether to enable the message traffic statistics collection:
	 * - `true`: Enable;
	 * - (Default)`false`: Disable.
	 */
	public void setEnableStatistics(boolean enableStatistics) {
		this.enableStatistics = enableStatistics;
		if (config == null) {
			return;
		}
		config.setEnableStatistics(enableStatistics);
	}
  
    /**
	 * \~chinese
	 * 设置区域代号。
	 * 
	 * @param code 区域代号。该参数用于限制边缘节点的访问范围，默认值为 `AREA_CODE_GLOB`，表示不限制区域。应用运行时不能对该参数进行修改。
	 *
	 * \~english
     *
     * Sets the area code.
     *
	 * @param code The area code. This parameter is used to restrict the scope of accessible edge nodes. The default value is `AREA_CODE_GLOB`, indicating that no restriction is placed on the scope of accessible edge nodes. The parameter setting cannot be changed during the app runtime. 
	 * 
	 */
	public void setAreaCode(int code) {
		this.areaCode = code;
		if (config == null) {
			return;
		}
		config.setAreaCode(code);
	}

	/**
	 * \~chinese
	 * 获取区域代号。
     * 
     * 区域代码参数用于限制边缘节点的访问范围。
     *
	 * @return 区域代号。
	 *
	 * \~english
	 * Gets the area code. 
     *
     * This parameter is used to restrict the scope of accessible edge nodes. 
     *
	 * @return The area code.
	 */
	public int getAreaCode() {
		return areaCode;
	}


	/**
	 * \~chinese
	 * 获取登录设备的自定义平台代号。
     * 
	 * @return  登录设备的自定义平台代号。
	 * \~english
	 * Gets the custom platform code of a login device.
     * 
	 * @return  The custom platform code of the login device.
	 */
	public int getCustomOSPlatform() {
		if(config==null) {
			return osCustomPlatform;
		}
		return config.getCustomOSPlatform();
	}

	/**
	 * \~chinese
	 * 设置登录设备的自定义平台代号。
     * 
	 * 需要在调用 {@link EMClient#init(Context, EMOptions)} 之前进行设置。
     * 
	 * @param platform 自定义平台代号，取值范围为 1-100。
	 *
	 * \~english
	 * Sets a custom platform code for the login device.
     * 
	 * You need to set the custom platform code before calling {@link EMClient#init(Context, EMOptions)}.
     * 
	 * @param platform The custom platform code. The value range is 1-100.
	 */ 
	public void setCustomOSPlatform(int platform) {
		if(platform<1||platform>100) {
			EMLog.e(TAG,"EMOptions setCustomOSPlatform() error : platform ="+platform);
		    return;
		}
		this.osCustomPlatform = platform;
		if (config == null) {
			return;
		}
		config.setCustomOSPlatform(platform);
	}

	/**
	 *\~chinese
	 * 获取当前设备的自定义名称。
     * 
	 * @return 当前设备的自定义名称。
	 *
	 * \~english
	 * Gets the custom name of the current device.
     * 
	 * @return The custom name of the current device.
	 */
	public String getCustomDeviceName() {
		if(config==null) {
			return customDeviceName;
		}
		return config.getDeviceName();

	}

	/**
	 * \~chinese
	 * 设置当前设备自定义名称。
     * 
     * 设置当前设备自定义名称旨在多设备登录时区分设备。
     * 
	 * 你可以通过 {@link EMClient#getLoggedInDevicesFromServer(String, String)} 获取指定账号的多个登录设备的信息。
     * 
	 * 需要在调用 {@link EMClient#init(Context, EMOptions)} 之前设置当前设备自定义名称。
     * 
	 * @param customDeviceName 自定义设备名称。该参数不支持空字符串和 null 值，默认使用设备型号，例如 HUAWEICOL-AL10。
	 *
	 *\~english
	 * Sets a custom name for the current device. 
     * 
     * Setting a custom name for the current device is intended to differentiate devices during multi-device login scenarios.
     *  
     * You can call {@link EMClient#getLoggedInDevicesFromServer(String, String)} to get the information of login devices of a specified account.
     *
	 * You can set the custom name for the current device before calling {@link EMClient#init(Context, EMOptions)}.
	 * 
	 * @param customDeviceName  The custom name for the current device. The parameter value cannot be `null` or an empty string. The default value is the device model such as HUAWEICOL-AL10.
	 */
	public void setCustomDeviceName(@NonNull String customDeviceName) {
		this.customDeviceName = customDeviceName;
		if (config == null) {
			return;
		}
		config.setDeviceName(customDeviceName);
	}

	/**
	 * \~chinese
	 * 设置当前设备自定义扩展信息。
	 * 
	 * 未初始化 SDK 或使用自动登录时，需在调用 {@link EMClient#init(Context, EMOptions)} 方法之前设置当前设备扩展信息。
	 * 
	 * 若应用已完成初始化且使用手动登录，若要修改自定义信息，在下次调用 {@link EMClient#login(String,String,EMCallBack)} 或
	 * {@link EMClient#loginWithToken(String,String,EMCallBack)} 之前设置自定义信息。
	 *
	 * @param ext 自定义设备扩展信息。该参数不支持 null 值。长度限制为 1024 个字符。
	 * 
	 * 若用户因登录了当前设备而将另一台设备踢下线，向被踢下线设备发通知时会携带当前设备的自定义扩展信息。
	 *
	 *\~english
	 * Sets custom extension information for the current device.
	 *
	 * If the SDK is not initialized or automatic login is used, you need to set the extension information of the current device
	 * before you call the {@link EMClient#init(Context, EMOptions) method.
	 * If the SDK is initialized and manual login is used, you can set the extension information
	 * before your next call to the {@link EMClient#login(String,String,EMCallBack)} or {@link EMClient#loginWithToken(String,String,EMCallBack)} method.
	 *
	 * @param ext  The custom extension information for the current device. The extension information can contain up to 1024 characters and cannot be `null`.
	 * 
	 *  The extension information is contained in the notification sent to the device that is kicked offline due to the user's login to the current device. 
	 */
	public void setLoginCustomExt(@NonNull String ext) {
		this.customExt = ext;
		if (config == null) {
			return;
		}
		config.setLoginInfoCustomExt(customExt);
	}

	/**
	 *\~chinese
	 * 获取自定义设备扩展信息。
	 *
	 * @return 自定义设备扩展信息。
	 *
	 * \~english
	 * Gets the custom device extension information.
	 *
	 * @return The custom device extension information.
	 */
	public String getLoginCustomExt() {
		if(config==null) {
			return customExt;
		}
		return config.getLoginInfoCustomExt();

	}

	/**
	 * \~chinese
	 * 检查从数据库加载会话时是否包含空会话。
	 *
	 * @return 从数据库加载会话时是否包含空会话：
	 * - （默认）`false`：不包括；
	 * - `true`：包括。
	 *
	 * \~english
	 * Checks whether to include empty conversations when the SDK loads conversations from the local database.
	 *
	 * @return Whether to include empty conversations when the SDK loads conversations from the local database.
	 * - (Default)`false`: Empty conversations are excluded.
	 * - `true`: Empty conversations are included.
	 */
	public boolean isLoadEmptyConversations() {
		if (config == null) {
			return isLoadEmptyConversations;
		}
		return config.isLoadEmptyConversations();
	}

	/**
	 * \~chinese
	 * 设置从数据库加载会话时是否包含空会话。
	 * 
	 * 需要在调用 {@link EMClient#init(Context, EMOptions)} 之前进行设置。
	 *
	 * @param isLoadEmptyConversations 从数据库加载会话时是否包含空会话：
	 * - `true`：包括；
	 * - （默认）`false`：不包括。
	 *
	 * \~english
	 * Sets whether to include empty conversations when the SDK loads conversations from the local database.
	 * 
	 * As for the call sequence, this method comes before {@link EMClient#init(Context, EMOptions)}.
	 *
	 * @param isLoadEmptyConversations Whether to include empty conversations when the SDK loads conversations from the local database:
	 * - `true`: Empty conversations are included.
	 * - (Default)`false`: Empty conversations are excluded.
	 */
	public void setLoadEmptyConversations(boolean isLoadEmptyConversations) {
		this.isLoadEmptyConversations = isLoadEmptyConversations;
		if (config == null) {
			return;
		}
		config.setLoadEmptyConversations(isLoadEmptyConversations);
	}

	/**
	 * \~chinese
	 * 获取当发送的文本消息的内容被文本审核（Moderation）服务替换时，是否需要返回给发送方。
	 *
	 * @return 是否将内容替换后的消息返回给发送方。
	 *  - true：是。将内容替换后的消息返回给发送方。
     *  - （默认）false：否。将原消息返回给发送方。
	 *
	 * \~english
	 * Gets whether the server returns the sender the text message with the content replaced during text moderation.
	 *
	 * @return Whether to return the adjusted message to the sender:
	 * - `true`: Yes.
	 * - (Default) `false`: No. The server returns the original message to the sender.
	 */
	public boolean isUseReplacedMessageContents() {
		if (config == null) {
			return useReplacedMessageContents;
		}
		return config.isUseReplacedMessageContents();
	}

	/**
	 * \~chinese
	 * 设置当发送的文本消息的内容被文本审核（Moderation）服务替换时，是否需要返回给发送方。
     *  - true：将内容替换后的消息返回给发送方。
     *  - （默认）false：将原消息返回给发送方。
	 *
	 * 该方法只能在调用 {@link EMClient#init(Context, EMOptions)} 前才能调用。
	 *
	 * @param useReplacedMessageContents 是否将内容替换后的消息返回给发送方。
	 *  - true：是。将内容替换后的消息返回给发送方。
     *  - （默认）false：否。将原消息返回给发送方。
	 *
	 * \~english
	 * Sets whether the server returns the sender the text message with the content replaced during text moderation:
	 *
	 * As for the call sequence, this method comes before {@link EMClient#init(Context, EMOptions)}.
	 *
	 * @param useReplacedMessageContents Whether to return the adjusted message to the sender:
	 * - `true`: Yes.
	 * - (Default) `false`: No. The server returns the original message to the sender.
	 */
	public void setUseReplacedMessageContents(boolean useReplacedMessageContents) {
		this.useReplacedMessageContents = useReplacedMessageContents;
		if (config == null) {
			return;
		}
		config.setUseReplacedMessageContents(useReplacedMessageContents);
	}
	/**
	 * \~chinese
	 * 发送成功的消息是否在 {@link EMMessageListener#onMessageReceived(java.util.List)} 中回调给用户。
	 *
	 * @return 发送成功的消息是否回调给用户。
	 * - `true`：是。发送成功的消息包含在{@link EMMessageListener#onMessageReceived(java.util.List)} 中。
	 * - （默认）`false`：否。{@link EMMessageListener#onMessageReceived(java.util.List)} 中只包含接收的消息。
	 *
	 * \~english
	 * Whether the sent message is included in {@link EMMessageListener#onMessageReceived(java.util.List)}.
	 *
	 * @return Whether the sent message is included in {@link EMMessageListener#onMessageReceived(java.util.List)}.
	 * - `true`: Yes. Besides the received message, the sent message is also included in {@link EMMessageListener#onMessageReceived(java.util.List)}.
	 * - (Default)`false`: No. Only the received message is included in {@link EMMessageListener#onMessageReceived(java.util.List)}.
	 
	 */
	public boolean isIncludeSendMessageInMessageListener() {
		if (config == null) {
			return includeSendMessageInMessageListener;
		}
		return config.isCallbackSendMessageInMessageListener();
	}

	/**
	 * \~chinese
	 * 设置发送成功的消息是否在 {@link EMMessageListener#onMessageReceived(java.util.List)} 中回调给用户。
	 *
	 * @param include 发送成功的消息是否回调给用户。
	 * - `true`：是。发送成功的消息包含在{@link EMMessageListener#onMessageReceived(java.util.List)} 中。
	 * - （默认）`false`：否。{@link EMMessageListener#onMessageReceived(java.util.List)} 中只包含接收的消息。
	 *
	 * \~english
	 * Sets whether the sent message is included in {@link EMMessageListener#onMessageReceived(java.util.List)}.
	 *
	 * @param include Whether the sent message is included in {@link EMMessageListener#onMessageReceived(java.util.List)}.
	 * - `true`: Yes. Besides the received message, the sent message is also included in {@link EMMessageListener#onMessageReceived(java.util.List)}.
	 * - (Default) `false`: No. Only the received message is included in {@link EMMessageListener#onMessageReceived(java.util.List)}.
	 */
	public void setIncludeSendMessageInMessageListener(boolean include) {
		this.includeSendMessageInMessageListener = include;
		if (config == null) {
			return;
		}
		config.setCallbackSendMessageInMessageListener(includeSendMessageInMessageListener);
	}
	/**
	 * \~chinese
	 * 是否将通过服务器导入的消息设置为已读：
	 * 
	 * @return 是否将通过服务器导入的消息设置为已读：
	 * - `true`：将导入消息置为已读。
	 *  - （默认）`false`：不将导入消息置为已读。
	 * \~english
	 * Whether to set messages imported from the server side as read.
	 * 
	 * @return Whether to set messages imported from the server side as read:
	 * - `true`: Read. After the messages are imported, the number of unread messages returned by `EMConversation#getUnreadMsgCount() remains unchanged.
` remains unchanged.
	 * - (Default)`false`: Unread. After the messages are imported, the number of unread messages returned by `EMConversation#getUnreadMsgCount() is increased.
                 *
	 */
	public boolean regardImportedMsgAsRead() {
		if (config == null) {
			return regardImportedMsgAsRead;
		}
		return config.regardImportedMsgAsRead();
	}
	/**
	 * \~chinese
	 * 设置是否将通过服务器导入的消息设置为已读：
	 * 
	 * @return 是否将通过服务器导入的消息设置为已读：
	 *              - `true`：将导入消息置为已读。
	 *              - （默认）`false`：不将导入消息置为已读。
	 * \~english
	 * Sets whether to set messages imported from the server side as read.
	 * 
	 * @return  Whether to set messages imported from the server side as read:
	 *              - `true`: Read. After the messages are imported, the number of unread messages returned by `EMConversation#getUnreadMsgCount() remains unchanged.
` remains unchanged.
	 *             - (Default)`false`: Unread. After the messages are imported, the number of unread messages returned by `EMConversation#getUnreadMsgCount() is increased.
	 *             
	 */
	public void setRegardImportedMsgAsRead(boolean regardImportedMsgAsRead) {
		this.regardImportedMsgAsRead = regardImportedMsgAsRead;
		if (config == null) {
			return;
		}
		config.setRegardImportedMsgAsRead(regardImportedMsgAsRead);
	}

	/**
	 * \~chinese
	 * 设置 native 库的路径。
	 * 
	 * @param path  native 库的路径。
	 *              - 该路径为空或者不调用该方法时，SDK 内部会使用 system.loadLibrary 从系统默认路径中搜索并加载 so 库。
	 *              - 该路径不为空时，SDK内部会使用 System.load 从设置的路径下搜索和加载 so 库。该路径必须为有效的 App 的私有目录路径。
	 * \~english
	 * Sets the path of the native library.
	 * 
	 * @param path  The path of the native library.
	 *              - If the path is empty or the method is not called, the SDK will use system.loadLibrary to search for and load the so library from the default library path.
	 *              - If the path is not empty, the SDK will use System.load to search for and load the so library from the specified path. The path must be a valid private directory path of the app.
	 *              - This method needs to be called before init.
	 */
	public void setNativeLibBasePath(@NonNull String path){
		if(TextUtils.isEmpty(path)) {
			return;
		}
		//去掉路径最后的斜杠
		String result = path.replaceAll("/+$", "");
		this.nativeLibBasePath=result;
	}

	/**
	 * \~chinese
	 * 获取设置的 native 库的路径。
	 * 
	 * @return  开发者设置过的 native 库的路径，未设置过则为空。
	 *
	 * \~english
	 * Gets the path of native library.
	 * 
	 * @return The specified path of native library. If the path is not set, it is empty.
	 */
	public String getNativeLibBasePath(){
		return this.nativeLibBasePath;
	}

	/**
	 * \~chinese
	 * 设置是否开启 TLS 连接。
	 * 
	 * 该设置只在私有部署时使用。
	 *
	 * 需要在调用 {@link EMClient#init(Context, EMOptions)} 之前进行设置。
	 *
	 * @param enableTLSConnection 是否开启 TLS 连接。
	 * - `true`：开启；
	 * - （默认）`false`：不开启。
	 *
	 * \~english
	 * Sets whether to enable the TLS connection. 
	 * 
	 * This setting is used only in private deployment.
	 * 
	 * As for the call sequence, this method comes before {@link EMClient#init(Context, EMOptions)}.
	 *
	 * @param enableTLSConnection Whether to enable the TLS connection.
	 * - `true`: Enables the TLS connection.
	 * - (Default)`false`: Disables the TLS connection.
	 */
	public void setEnableTLSConnection(boolean enableTLSConnection) {
		this.enableTLSConnection = enableTLSConnection;
		if (config == null) {
			return;
		}
		config.setEnableTLSConnection(enableTLSConnection);
	}
	/**
	 * \~chinese
	 * 获取是否开启 TLS 连接。
	 *
	 * @return 是否开启 TLS 连接。
	 * - `true`：开启 TLS 连接；
	 * - `false`：未开启 TLS 连接。
	 *
	 * \~english
	 * Gets whether the TLS connection is enabled.
	 *
	 * @return Whether the TLS connection is enabled.
	 * - `true`: The TLS connection is enabled.
	 * - `false`: The TLS connection is disabled.
	 */
	public boolean isEnableTLSConnection() {
		if (config == null) {
			return enableTLSConnection;
		}
		return config.isEnableTLSConnection();
	}
	/**
	 * \~chinese
	 * 设置是否在登录成功后自动加载所有会话到缓存里。
	 *
	 * 需要在调用 {@link EMClient#init(Context, EMOptions)} 之前进行设置。
	 *
	 * @param autoLoadAllConversations 是否自动加载所有会话。
	 * - （默认）`true`：自动加载；
	 * - `false`：不自动加载。
	 *
	 * \~english
	 * Sets whether to automatically load all conversations into the memory upon a successful login.
	 *
	 * As for the call sequence, this method comes before {@link EMClient#init(Context, EMOptions)}.
	 *
	 * @param autoLoadAllConversations Whether to automatically load all conversations.
	 * - (Default)`true`: Automatically load all conversations.
	 * - `false`: Do not automatically load all conversations.
	 */
	public void setAutoLoadAllConversations(boolean autoLoadAllConversations){
		this.autoLoadAllConversations = autoLoadAllConversations;
		if (config == null) {
			return;
		}
		config.setAutoLoadAllConversations(autoLoadAllConversations);
	}
	/**
	 * \~chinese
	 * 获取是否在登录成功后自动加载所有会话到缓存里。
	 *
	 * @return 是否自动加载所有会话。
	 * - （默认）`true`：自动加载；
	 * - `false`：不自动加载。
	 *
	 * \~english
	 * Gets whether to automatically load all conversations into the memory upon a successful login.
	 *
	 * @return Whether to automatically load all conversations.
	 * - (Default)`true`: Automatically load all conversations.
	 * - `false`: Do not automatically load all conversations.
	 */
	public boolean isAutoLoadAllConversations(){
		if (config == null) {
			return autoLoadAllConversations;
		}
		return config.isAutoLoadAllConversations();
	}
}
