package com.hyphenate.chat;

import com.hyphenate.chat.adapter.EMAMessageStatistics;
import com.hyphenate.chat.adapter.EMAStatisticsManager;

/**
 * \~chinese
 * 本地消息流量统计管理类。
 *
 * 该类中的方法可用于统计一定时间段内发送和/接收的指定类型的本地消息数量及其流量。
 *
 * 本地消息的流量统计功能默认关闭。若要使用该功能，需在 SDK 初始化前设置 {@link EMOptions#setEnableStatistics(boolean)} 开启。
 *
 * SDK 只支持统计该功能开启后最近 30 天内发送和接收的消息。各类消息的流量计算方法如下：
 * - 对于文本、透传、位置和自定义消息，消息流量为消息体的流量；
 * - 对于图片和视频消息，消息流量为消息体、图片或视频文件以及缩略图的流量之和；
 * - 对于文件和语音消息，消息流量为消息体和附件的流量。
 *
 * **注意**
 * 1. 对于携带附件的消息，下载成功后 SDK 才统计附件的流量。若附件下载多次，则会对下载的流量进行累加。
 * 2. 对于从服务器拉取的漫游消息，如果本地数据库中已经存在，则不进行统计。
 *
 * SDK 仅统计本地消息的流量，而非消息的实际流量。一般而言，该统计数据小于实际流量，原因如下：
 * - 未考虑发送消息时通用协议数据的流量；
 * - 对于接收到的消息，服务端会进行消息聚合，添加通用字段，而消息流量统计为各消息的流量，未考虑通用字段的流量消耗。
 *
 *
 * \~english
 * The class for traffic statistics management of local messages.
 *
 * This class contains methods that are used to calculate the number of local messages of certain types sent and/or received in a specified period, as well as their traffic.
 *
 * This traffic statistics function is disabled by default. To use this function, you need to enable it by setting {@link EMOptions#setEnableStatistics(boolean)} prior to the SDK initialization.
 *
 * The SDK only calculates the traffic of messages that are sent and received within the last 30 days after the traffic statistics function is enabled.
 *
 *  The message traffic is calculated as follows:
 *  - For a text, command, location or custom message, the message traffic is the traffic of the message body.
 *  - For an image or video message, the message traffic is the traffic sum of the message body, the image or video file, and the thumbnail.
 *  - For a file or voice message, the message traffic is the traffic sum of the message body and the attachment.
 *
 * **Note**
 * 1. For messages with attachments, the traffic will be calculated only if the download is successful.
 *    If an attachment is downloaded multiple times, its download traffic will be accumulated.
 * 2. For roaming messages pulled from the server, if they already exist in the local database, the SDK ignores them during traffic calculation.
 *
 * The SDK only measures the traffic of local messages, but not the actual message traffic. Generally, the calculated traffic volume is smaller than the actual traffic because of the following:
 *   - The traffic of the common protocol data generated during message sending is not considered;
 *   - For the received messages, the server aggregates them and adds common fields. During traffic statistics, the SDK only calculates the traffic of individual messages, but ignoring the traffic of common fields.
 *
 */
public class EMStatisticsManager extends EMBase<EMAStatisticsManager> {

    /**
     * \~chinese
     * 消息方向枚举类，用于查询。
     *
     * \~english
     * The message directions for querying.
     */
    public enum EMSearchMessageDirect {
        /**
         * \~chinese
         * 当前用户发送的消息。
         *
         * \~english
         * This message is sent from the local client.
         */
        SEND(0),
        /**
         * \~chinese
         * 当前用户接收到的消息。
         *
         * \~english
         * The message is received by the local client.
         */
        RECEIVE(1),
        /**
         * \~chinese
         * 当前用户发送或接收的消息。
         *
         * \~english
         * The message is sent from or received by the local client.
         */
        ALL(100);

        private final int index;
        private EMSearchMessageDirect(int index) {
            this.index = index;
        }

        public int getIndex() {
            return index;
        }
    }

    /**
     * \~chinese
     * 消息类型枚举类，用于查询。
     *
     * \~english
     * The message types for querying.
     */
    public enum EMSearchMessageType {
        /**
         * \~chinese
         * 文本消息。
         *
         * \~english
         * Text message.
         */
        TXT(0),
        /**
         * \~chinese
         * 图片消息。
         *
         * \~english
         * Image message.
         */
        IMAGE(1),
        /**
         * \~chinese
         * 视频消息。
         *
         * \~english
         * Video message.
         */
        VIDEO(2),
        /**
         * \~chinese
         * 位置消息。
         *
         * \~english
         * Location message.
         */
        LOCATION(3),
        /**
         * \~chinese
         * 语音消息。
         *
         * \~english
         * Voice message.
         */
        VOICE(4),
        /**
         * \~chinese
         * 文件消息。
         *
         * \~english
         * File message.
         */
        FILE(5),
        /**
         * \~chinese
         * 命令消息（透传消息）。
         *
         * \~english
         * Command message.
         */
        CMD(6),
        /**
         * \~chinese
         * 用户自定义消息。
         *
         * \~english
         * Custom message.
         */
        CUSTOM(7),
        /**
         * \~chinese
         * 所有类型。
         *
         * \~english
         * All message types.
         */
        ALL(100);

        private final int index;
        private EMSearchMessageType(int index) {
            this.index = index;
        }

        public int getIndex() {
            return index;
        }
    }

    EMClient mClient;

    EMStatisticsManager(EMClient client, EMAStatisticsManager manager) {
        emaObject = manager;
        mClient = client;
    }

    /**
     * \~chinese
     * 根据消息 ID 获取消息流量统计信息。
     *
     * @param messageId 消息 ID。
     * @return 返回消息的流量统计信息。统计数据详见 {@link EMMessageStatistics}。
     *
     * \~english
     * Gets message traffic statistics by message ID.
     *
     * @param messageId The message ID.
     * @return  The message traffic statistics. See {@link EMMessageStatistics}.
     */
    public EMMessageStatistics getMessageStatistics(String messageId) {
        EMAMessageStatistics statistics = emaObject.getMsgStatistics(messageId);
        if(statistics != null) {
            return new EMMessageStatistics(statistics);
        }
        return null;
    }

    /**
     * \~chinese
     * 获取一定时间段内发送和/或接收的指定类型的消息条数。
     *
     * @param startTimestamp    起始时间戳，单位为毫秒。
     * @param endTimestamp      结束时间戳，单位为毫秒。
     * @param direct            消息方向。
     * @param type              消息类型。
     * @return                  返回符合条件的消息条数。调用失败时返回 `0`。
     *
     * \~english
     * Gets the count of messages of certain types that are sent and/or received in a specified period.
     *
     * @param startTimestamp    The starting timestamp for statistics. The unit is millisecond.
     * @param endTimestamp      The ending timestamp for statistics. The unit is millisecond.
     * @param direct            The message direction.
     * @param type              The message type.
     * @return                  The count of messages that meet the statistical conditions. `0` is returned in the case of a call failure.
     */
    public int getMessageCount(long startTimestamp, long endTimestamp, EMSearchMessageDirect direct, EMSearchMessageType type) {
        return emaObject.searchMsgStatisticsNumber(startTimestamp, endTimestamp, direct.getIndex(), type.getIndex());
    }

    /**
     * \~chinese
     * 获取一定时间段内发送和/或接收的指定类型的消息的总流量。
     *
     * 流量单位为字节。
     *
     * 消息流量的计算方法如下：
     * - 对于文本、透传、位置和自定义消息消息，消息流量为消息体的流量；
     * - 对于图片和视频消息，消息流量为消息体、图片或视频文件以及缩略图的流量之和；
     * - 对于文件和语音消息，消息流量为消息体和附件的流量。
     *
     * @param startTimestamp    起始时间戳。
     * @param endTimestamp      结束时间戳。
     * @param direct            消息方向。
     * @param type              消息类型。
     * @return                  返回符合条件的消息流量大小。调用失败时返回 `0`。
     *
     * \~english
     * Gets the total traffic volume of messages of certain types that are sent and/or received in a specified period.
     *
     * The traffic is measured in bytes.
     *
     *  The message traffic is calculated as follows:
     *  - For a text, command, location, or custom message, the message traffic is the traffic of the message body.
     *  - For an image or video message, the message traffic is the traffic sum of the message body, the image or video file, and the thumbnail.
     *  - For a file or voice message, the message traffic is the traffic sum of the message body and the attachment.
     *
     * @param startTimestamp    The starting timestamp for statistics. The unit is millisecond.
     * @param endTimestamp      The ending timestamp for statistics. The unit is millisecond.
     * @param direct            The message direction.
     * @param type              The message type.
     * @return                  The total traffic volume of messages that meet the statistical conditions. `0` is returned in the case of a call failure.
     */
    public long getMessageSize(long startTimestamp, long endTimestamp, EMSearchMessageDirect direct, EMSearchMessageType type) {
        return emaObject.searchMsgStatisticsSize(startTimestamp, endTimestamp, direct.getIndex(), type.getIndex());
    }

    /**
     * \~chinese
     * 删除指定时间戳前的本地消息流量数据。
     *
     * @param timestamp 流量数据删除的时间戳，单位为毫秒。
     * @return 删除的结果：
     *         - `true`：删除成功；
     *         - `false`：删除失败。
     *
     * \~english
     * Deletes the traffic data of local messages that are received by the server before the specified timestamp.
     *
     * @param timestamp The Unix timestamp threshold for traffic data deletion, in milliseconds. The traffic data of local messages that are received by the server before the specified timestamp will be deleted.
     * @return The deletion result:
     *         - `true`: The deletion succeeds;
     *         - `false`: The deletion fails.
     */
    boolean removeMessagesBeforeTime(long timestamp) {
        return emaObject.removeMsgStatisticsBeforeTime(timestamp);
    }


}
