/************************************************************
  *  * EaseMob CONFIDENTIAL 
  * __________________ 
  * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
  *  
  * NOTICE: All information contained herein is, and remains 
  * the property of EaseMob Technologies.
  * Dissemination of emaObject information or reproduction of emaObject material 
  * is strictly forbidden unless prior written permission is obtained
  * from EaseMob Technologies.
  */
package com.hyphenate.chat;

import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.chat.adapter.EMACallback;
import com.hyphenate.chat.adapter.EMAMessageReaction;
import com.hyphenate.chat.adapter.EMAThreadInfo;
import com.hyphenate.chat.adapter.message.EMACmdMessageBody;
import com.hyphenate.chat.adapter.message.EMACombineMessageBody;
import com.hyphenate.chat.adapter.message.EMACustomMessageBody;
import com.hyphenate.chat.adapter.message.EMAFileMessageBody;
import com.hyphenate.chat.adapter.message.EMAImageMessageBody;
import com.hyphenate.chat.adapter.message.EMALocationMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.chat.adapter.message.EMAMessage.EMADirection;
import com.hyphenate.chat.adapter.message.EMAMessage.EMAMessageStatus;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.adapter.message.EMATextMessageBody;
import com.hyphenate.chat.adapter.message.EMAVideoMessageBody;
import com.hyphenate.chat.adapter.message.EMAVoiceMessageBody;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMFileHelper;
import com.hyphenate.util.EMLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

/**
 * \~chinese
 * 消息对象，表示一条发送或接收到的消息。
 * 
 * 例如：
 * 构造一条文本发送消息：
 * ```java
 *     EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
 *     msg.setTo("user1");
 *     TextMessageBody body = new TextMessageBody("hello from hyphenate sdk");
 *     msg.addBody(body);
 * ```
 * 
 * 构造一条图片消息：
 * ```java
 *      EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
 *      msg.setTo("user1");
 *      ImageMessageBody body = new ImageMessageBody(imageFileUrl);
 *      msg.addBody(body);
 * ```
 * 
 * \~english
 * The message instance, which indicates a sent or received message.

 * 
 * For example:
 * Constructs a text message to send:
 * ```java
 *     EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
 *     msg.setTo("user1");
 *     TextMessageBody body = new TextMessageBody("hello from hyphenate sdk");
 *     msg.addBody(body);
 * ```
 * 
 * Constructs an image message to send:
 * ```java
 *      EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
 *      msg.setTo("user1");
 *      ImageMessageBody body = new ImageMessageBody(imageFileUrl);
 *      msg.addBody(body);
 * ```
 */
public class EMMessage extends EMBase<EMAMessage> implements Parcelable, Cloneable{
    static final String ATTR_ENCRYPTED = "isencrypted";
    
    private final static String TAG = "msg";

    /**
     * \~chinese
     * 消息类型枚举类。
     * 
     * \~english
     * The message types.
     */
    public enum Type {
       /**
        * \~chinese
        * 文本消息。
        * 
        * \~english
        * Text message.
        */
        TXT, 
       /**
        * \~chinese
        * 图片消息。
        * 
        * \~english
        * Image message.
        */
        IMAGE,
       /**
        * \~chinese
        * 视频消息。
        * 
        * \~english
        * Video message.
        */ 
        VIDEO, 
       /**
        * \~chinese
        * 位置消息。
        * 
        * \~english
        * Location message.
        */ 
        LOCATION, 
        /**
        * \~chinese
        * 语音消息。
        * 
        * \~english
        * Voice message.
        */ 
        VOICE, 
        /**
        * \~chinese
        * 附件消息。
        * 
        * \~english
        * File message.
        */ 
        FILE, 
        /**
        * \~chinese
        * 命令消息（透传消息）。
        * 
        * \~english
        * Command message.
        */ 
        CMD, 
        /**
        * \~chinese
        * 用户自定义消息。
        * 
        * \~english
        * Custom message.
        */
        CUSTOM,
        /**
        * \~chinese
        * 合并消息。
        *
        * \~english
        * Combined message.
        */
        COMBINE
    }
    
    /**
     * \~chinese
     * 消息的方向类型枚举类。
     * 
     *  该类指定消息是发送消息还是接收的消息。
     * 
     * \~english
     * The message direction.
     * 
     * This class indicates whether the message is sent or received.
     */
    public enum Direct {
      /* \~chinese 该消息是当前用户发送出去的。\~english This message is sent from the local client.*/
        SEND, 
     /* \~chinese 该消息是当前用户接收到的。 \~english The message is received by the local client.*/
        RECEIVE
    }
    
    /**
     * \~chinese
     * 消息的发送/接收状态枚举类。
     * 
     * \~english
     * The message sending or reception status.
     */
    public enum Status {
       /** \~chinese 发送/接收成功。 \~english The message is successfully delivered/received.*/

        SUCCESS, 
         /** \~chinese 发送/接收失败。 \~english The message fails to be delivered/received.*/
        FAIL, 
       /** \~chinese 正在发送/接收。 \~english The message is being delivered/received.*/
        INPROGRESS,
       /** \~chinese 消息已创建待发送。 \~english The message is created to be sent.*/

        CREATE
    }
    
    EMMessageBody body;
    
    /**
     * \~chinese
     * 获取消息的发送/接收状态。
     * 
     * @return 消息的发送/接收状态。
     * 
     * \~english
     * Gets the message sending or reception status.
     * 
     * @return  The message sending or reception status.
     */
    public Status status() {
    	EMAMessageStatus st = emaObject._status();
    	Status s = Status.CREATE;
    	switch (st) {
    	    case NEW :
                s = Status.CREATE;
    	        break;
    	    case SUCCESS :
                s = Status.SUCCESS;
    	        break;
    	    case FAIL :
                s = Status.FAIL;
    	        break;
    	    case DELIVERING :
                s = Status.INPROGRESS;
    	        break;
    	}
    	return s;
    }
    
    /**
     * \~chinese
     * 聊天类型枚举类。
     *
     *\~english
     * The chat types.
     */
    public enum ChatType {
        /**
         * \~chinese
         * 单聊。
         *
         * \~english
         * One-to-one chat.
         */
        Chat, 
        /**
         * \~chinese
         * 群聊。
         *
         * \~english
         * Group chat.
         */
        GroupChat,
        /**
         * \~chinese
         * 聊天室。
         *
         * \~english
         * Chat room chat.
         */
        ChatRoom
    }
    
    public EMMessage(EMAMessage message) {
        emaObject = message;
    }
    
    /**
     * \~chinese
     * 设置消息发送或接收的状态。
     * 
     * @param status 消息的状态。
     * 
     * \~english
     * Sets the message sending or reception status.

     * @param status The message sending or reception status.
     */
    public void setStatus(Status status) {
    	EMAMessageStatus s = EMAMessageStatus.SUCCESS;
    	switch (status) {
            case CREATE:
                s = EMAMessageStatus.NEW;
    	        break;
            case SUCCESS:
                s = EMAMessageStatus.SUCCESS;
    	        break;
            case FAIL:
                s = EMAMessageStatus.FAIL;
    	        break;
            case INPROGRESS:
                s = EMAMessageStatus.DELIVERING;
    	        break;
    	}
    	emaObject.setStatus(s.ordinal());
    }
    
    /**
     * \~chinese
     * 获取消息类型。
     * 
     * @return  消息类型。
     * 
     * \~english
     * Gets the chat message type.
     * 
     * @return  The chat message type.
     */
    public Type getType() {
    	List<EMAMessageBody> bodies = emaObject.bodies();
    	if (bodies.size() > 0) {
    		int type = bodies.get(0).type();
    		//Log.d(TAG, "type:" + type);
    		if (type == Type.TXT.ordinal()) {
    			return Type.TXT;
    		} else if (type == Type.IMAGE.ordinal()) {
    			return Type.IMAGE;
    		} else if (type == Type.CMD.ordinal()) {
    			return Type.CMD;
    		} else if (type == Type.FILE.ordinal()) {
    			return Type.FILE;
    		} else if (type == Type.VIDEO.ordinal()) {
    			return Type.VIDEO;
    		} else if (type == Type.VOICE.ordinal()) {
    			return Type.VOICE;
    		} else if (type == Type.LOCATION.ordinal()) {
    			return Type.LOCATION;
    		} else if (type == Type.CUSTOM.ordinal()) {
    		    return Type.CUSTOM;
    		} else if (type == Type.COMBINE.ordinal()) {
    		    return Type.COMBINE;
            }
    	}
        return Type.TXT;
    }
    
    /**
     * \~chinese
     * 获取消息体。
     * 
     * @return  消息体。
     * 
     * \~english
     * Gets the message body.
     * 
     * @return  The message body.
     */
    public EMMessageBody getBody() {
        if (body != null) {
            return body;
        }
    	List<EMAMessageBody> bodies = emaObject.bodies();
    	if (bodies.size() > 0) {
    		EMAMessageBody emaBody = bodies.get(0);
    		if (emaBody instanceof EMATextMessageBody) {
    			body = new EMTextMessageBody((EMATextMessageBody)emaBody);
    		} else if (emaBody instanceof EMACmdMessageBody) {
    			body = new EMCmdMessageBody((EMACmdMessageBody)emaBody);
    		} else if (emaBody instanceof EMAVideoMessageBody) {
    			body = new EMVideoMessageBody((EMAVideoMessageBody)emaBody);
    		} else if (emaBody instanceof EMAVoiceMessageBody) {
    			body = new EMVoiceMessageBody((EMAVoiceMessageBody)emaBody);
    		} else if (emaBody instanceof EMAImageMessageBody) {
    			body = new EMImageMessageBody((EMAImageMessageBody)emaBody);
            } else if (emaBody instanceof EMACombineMessageBody) {
                body = new EMCombineMessageBody((EMACombineMessageBody) emaBody);
    		} else if (emaBody instanceof EMALocationMessageBody) {
    			body = new EMLocationMessageBody((EMALocationMessageBody)emaBody);
    		} else if (emaBody instanceof EMAFileMessageBody) {
    			body = new EMNormalFileMessageBody((EMAFileMessageBody)emaBody);
    		} else if (emaBody instanceof EMACustomMessageBody) {
    		    body = new EMCustomMessageBody((EMACustomMessageBody) emaBody);
            }
        	return body;
    	}

    	return null;
    }
    
    /**
     * \~chinese
     * 获取消息的服务器时间戳。
     * 
     * 该时间戳为服务器接收消息的时间。
     * 
     * @return  服务器接收消息的时间戳。
     * 
     * \~english
     * Gets the Unix timestamp when the server receives the message.
     * 
     * @return  The Unix timestamp when the server receives the message.
     */
    public long getMsgTime() {
    	return emaObject.timeStamp();
    }
    
    
    /**
     * \~chinese
     * 设置消息的服务器时间戳。
     * 
     * 该时间戳为服务器接收消息的时间。
     * 
     * @param msgTime   服务器接收消息的时间戳。
     * 
     * \~english
     * Sets the Unix timestamp when the server receives the message.
     * 
     * @param msgTime   The Unix timestamp when the server receives the message.
     */
    public void setMsgTime(long msgTime) {
		emaObject.setTimeStamp(msgTime);
	}
    
    /**
     * \~chinese
     * 获取消息的本地时间戳。
     * 
     * 该时间戳为消息的本地创建时间戳。
     * 
     * @return  消息的本地时间戳。
     * 
     * \~english
     * Gets the local timestamp of the message.
     * 
     * This method gets the timestamp when the message is created locally.
     * 
     * @return The local timestamp of the message.
     */
    public long localTime() {
        return emaObject.getLocalTime();
    }
    
    /**
     * \~chinese
     * 设置消息的本地时间戳。
     * 
     * 该时间戳为消息的本地创建时间戳。
     * 
     * @param serverTime    消息的本地时间戳。
     * 
     * \~english
     * Sets the local timestamp of the message.
     * 
     * This method sets the timestamp when the message is created locally.
     * 
     * @param serverTime    The local timestamp of the message.
     */
    public void setLocalTime(long serverTime) {
        emaObject.setLocalTime(serverTime);
    }

    /**
     * \~chinese
     * 消息是否需要群组已读回执。
     * 
     * @return 消息是否需要群组已读回执
     * - `true`：需要已读回执；
     * - `false`：不需要已读回执。
     *
     * \~english
     * Sets whether read receipts are required for group messages.
     * 
     * @return Whether read receipts are required for group messages:
     *        - `true`: Yes;
     *        - `false`: No.
     */
    public boolean isNeedGroupAck() {return emaObject.isNeedGroupAck(); }


    /**
     * \~chinese
     * 设置群组消息是否需要已读回执。
     * 
     * @param need 群组消息是否需要已读回执:
     *             - `true`：需要；
     *             - `false`：不需要。
     *
     * \~english
     * Sets whether read receipts are required for group messages.
     * 
     * @param need  Whether read receipts are required for group messages.
     *             - `true`: Yes; 
     *             - `false`: No.
     */
    public void setIsNeedGroupAck(boolean need) {emaObject.setIsNeedGroupAck(need); }

    /**
     * \~chinese
     * 群消息已读人数。
     * 
     * @return 消息已读人数。
     *
     * \~english
     * Gets the number of members that have read the group message.
     * 
     * @return The number of members that have read the group message.
     */
    public int groupAckCount() { return emaObject.groupAckCount(); }

    /**
     * \~chinese
     * 设置群消息已读人数。
     * 
     * @param count  群消息已读人数。
     *
     * \~english
     * Sets the number of members that have read the group message.
     * 
     * @param count  The number of members that have read the group message.
     */
    public void setGroupAckCount(int count) { emaObject.setGroupAckCount(count); }

    /**
     * \~chinese
     * 设置消息是否是子区消息。
     *
     * @param isChatThreadMessage - `true`表示设置成子区消息；
     *                            - `false`表示不设置为子区消息。
     * 
     * \~english
     * Sets whether the message is a threaded message.
     * 
     * @param isChatThreadMessage - `true` means set to a threaded message.
     *                            - `false` means not set as a threaded message.
     */
    public void setIsChatThreadMessage(boolean isChatThreadMessage) {
        emaObject.setIsChatThreadMessage(isChatThreadMessage);
    }

    /**
     * \~chinese
     * 获取消息是否是子区消息。
     *
     * @return 返回是否是子区消息的结果。
     *         - `true`表示是子区消息；
     *         - `false`表示不是子区消息。
     *
     * \~english
     * Gets whether the message is a threaded message.
     * 
     * @return Whether the message is a threaded message.
     *         - `true`: Yes.
     *         - `false`: No.
     */
    public boolean isChatThreadMessage() {
        return emaObject.isChatThreadMessage();
    }

    /**
     * \~chinese
     * 获取子区概览信息。
     * 
     * 子区概览信息仅在创建子区后携带。
     * 
     * @return 子区概览信息。
     *
     * \~english
     * Gets the overview of the message thread.
     * 
     * The overview of the message thread exists only after you creates a message thread.
     * 
     * @return The overview of the message thread.
     */
    public EMChatThread getChatThread() {
        EMAThreadInfo thread = emaObject.getChatThread();
        if(thread == null) {
            return null;
        }
        return new EMChatThread(thread);
    }

	/**
	 * \~chinese
     * 创建一个发送消息。
     * 
     * @param type 消息类型。
     * @return 消息对象。
     * 
     * \~english
     * Creates a message instance for sending.
     * 
     * @param type The message type.
     * @return The message instance.
     */
    public static EMMessage createSendMessage(Type type) {
    	EMAMessage _msg = EMAMessage.createSendMessage(self(), "", null, ChatType.Chat.ordinal());
        EMMessage msg = new EMMessage(_msg);
        return msg;
    }
    
    /**
     * \~chinese
     * 创建一个接收消息。
     * 
     * @param type 消息类型。
     * @return 消息对象。
     * 
     * \~english
     * Creates a received message instance.
     * 
     * @param type The message type.
     * @return The message instance.
     */
    public static EMMessage createReceiveMessage(Type type) {
    	EMAMessage _msg = EMAMessage.createReceiveMessage("", self(), null, ChatType.Chat.ordinal());
        EMMessage msg = new EMMessage(_msg);
        msg.setTo(EMSessionManager.getInstance().currentUser.getUsername());
        return msg;
    }
    
    /**
     * \~chinese
     * 创建一个文本发送消息。
     * 
     * @param content 文本内容。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     * @deprecated 已废弃。请用 {@link #createTextSendMessage(String, String)} 代替。
     * 
     * \~english
     * Creates a text message for sending.
     * 
     * @param content The text content.
     * @param username The message sender is determined by the conversation type:
     *      *                 - One-to-one chat: The peer user ID.
     *      *                 - Group chat: The group ID.
     *      *                 - Thread conversation: The message thread ID.
     *      *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     * @deprecated Deprecated. Use {@link #createTextSendMessage(String, String)} instead.
     */
    @Deprecated
    public static EMMessage createTxtSendMessage(String content, String username){
        if (content.length() > 0) {
        	EMMessage msg = EMMessage.createSendMessage(Type.TXT);
        	EMTextMessageBody txtBody = new EMTextMessageBody(content);
        	msg.addBody(txtBody);
        	msg.setTo(username);
            return msg;
        }
        EMLog.e(TAG, "text content size must be greater than 0");
        return null;
    }

    /**
     * \~chinese
     * 创建一个文本发送消息。
     * 
     * @param content 文本内容。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a text message for sending.
     * 
     * @param content The text content.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     */
    public static EMMessage createTextSendMessage(String content, String username){
        if (content.length() > 0) {
        	EMMessage msg = EMMessage.createSendMessage(Type.TXT);
        	EMTextMessageBody txtBody = new EMTextMessageBody(content);
        	msg.addBody(txtBody);
        	msg.setTo(username);
            return msg;
        }
        EMLog.e(TAG, "text content size must be greater than 0");
        return null;
    }

    /**
     * \~chinese
     * 创建一个语音发送消息。
     * 
     * @param filePath 语音文件路径。
     * @param timeLength 语音时间长度，单位为秒。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates a voice message for sending.
     * 
     * @param filePath The path of the voice file.
     * @param timeLength The voice duration in seconds.
     * @param username  The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     *
     */
    public static EMMessage createVoiceSendMessage(String filePath, int timeLength, String username){
        return createVoiceSendMessage(EMFileHelper.getInstance().formatInUri(filePath), timeLength, username);
    }

    /**
     * \~chinese
     * 创建一个语音发送消息。
     * 
     * @param filePath 语音文件 URI。
     * @param timeLength 语音时间长度，单位为秒。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a voice message for sending.
     * 
     * @param filePath The URI of the voice file.
     * @param timeLength The voice duration in seconds.
     * @param username  The user ID of the message recipient or group ID.
     * @return The message instance.
     */
    public static EMMessage createVoiceSendMessage(Uri filePath, int timeLength, String username){
        if (!EMFileHelper.getInstance().isFileExist(filePath)) {
            EMLog.e(TAG, "voice file does not exsit");
            return null;
        }
        EMMessage message = EMMessage.createSendMessage(Type.VOICE);
        // 如果是群聊，需设置 chattype，默认是单聊。
        EMVoiceMessageBody body = new EMVoiceMessageBody(filePath, timeLength);
        message.addBody(body);

        message.setTo(username);
        return message;
    }

    /**
     * \~chinese
     * 创建一个图片发送消息。
     * 
     * @param filePath 图片路径。
     * @param sendOriginalImage 是否发送原图：
     *     - `true`：发送原图。
     *     - `false`：若图片小于 100 KB，发送原图；若图片大于等于 100 KB, 发送压缩后的图片。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates an image message for sending.
     * 
     * @param filePath The image path.
     * @param sendOriginalImage Whether to send the original image.
     *     - `true`: Yes.
     *     - `false`: If the image is smaller than 100 KB, the SDK sends the original image. If the image is equal to or greater than 100 KB, the SDK will compress it before sending the compressed image.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     *
     */
    public static EMMessage createImageSendMessage(String filePath, boolean sendOriginalImage, String username){
        return createImageSendMessage(EMFileHelper.getInstance().formatInUri(filePath), sendOriginalImage, username);
    }

    /**
     * \~chinese
     * 创建一个图片发送消息。
     * 
     * @param imgUri 图片 URI。
     * @param sendOriginalImage 是否发送原图：
     *     - `true`：发送原图。
     *     - `false`：若图片小于 100 KB，发送原图；若图片大于等于 100 KB, 发送压缩后的图片。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates an image message for sending.
     * 
     * @param imgUri The URI of the image.
     * @param sendOriginalImage Whether to send the original image.
     *                           - `true`: Yes.
     *                           - `false`: If the image is smaller than 100 KB, the SDK sends the original image. If the image is equal to or greater than 100 KB, the SDK will compress it before sending the compressed image.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     */
    public static EMMessage createImageSendMessage(Uri imgUri, boolean sendOriginalImage, String username){
        if(!EMFileHelper.getInstance().isFileExist(imgUri)) {
            EMLog.e(TAG, "image file does not exsit");
            return null;
        }
        // create and add the image message in view.
        final EMMessage message = EMMessage.createSendMessage(Type.IMAGE);
        message.setTo(username);
        EMImageMessageBody body = new EMImageMessageBody(imgUri);
        // 默认超过 100k 的图片会压缩后发给对方，可以设置成发送原图。
        body.setSendOriginalImage(sendOriginalImage);
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * 创建一个视频发送消息。
     * 
     * @param videofilePath 视频文件路径。
     * @param imageThumbPath 视频第一帧图缩略图。
     * @param timeLength 视频时间长度，单位为秒。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates a video message for sending.
     * 
     * @param videofilePath The path of the video file.
     * @param imageThumbPath The path of thumbnail of the first frame of the video.
     * @param timeLength The video duration in seconds.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     *
     */
    public static EMMessage createVideoSendMessage(String videofilePath, String imageThumbPath, int timeLength, String username){
        return createVideoSendMessage(EMFileHelper.getInstance().formatInUri(videofilePath)
                , EMFileHelper.getInstance().formatInUri(imageThumbPath)
                , timeLength
                , username);
    }

    /**
     * \~chinese
     * 创建一个视频发送消息。
     * 
     * @param videofilePath 视频文件路径。
     * @param imageThumbPath 视频第一帧图缩略图。
     * @param timeLength 视频时间长度，单位为秒。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a video message for sending.
     * 
     * @param videofilePath The path of the video file.
     * @param imageThumbPath The path of the thumbnail of the first frame of the video.
     * @param timeLength The video duration in seconds.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     */
    public static EMMessage createVideoSendMessage(Uri videofilePath, String imageThumbPath, int timeLength, String username){
        return createVideoSendMessage(videofilePath, EMFileHelper.getInstance().formatInUri(imageThumbPath), timeLength, username);
    }

    /**
     * \~chinese
     * 创建一个视频发送消息。
     * 
     * @param videofilePath 视频文件路径。
     * @param imageThumbPath 视频第一帧图缩略图。
     * @param timeLength 视频时间长度，单位为秒。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a video message for sending.
     * 
     * @param videofilePath The path of the video file.
     * @param imageThumbPath The path of the thumbnail of the first frame of the video.
     * @param timeLength The video duration in seconds.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return  The message instance.
     */
    public static EMMessage createVideoSendMessage(Uri videofilePath, Uri imageThumbPath, int timeLength, String username){
        if (!EMFileHelper.getInstance().isFileExist(videofilePath)) {
            EMLog.e(TAG, "video file does not exist");
            return null;
        }
        EMMessage message = EMMessage.createSendMessage(Type.VIDEO);
        message.setTo(username);
        EMVideoMessageBody body = new EMVideoMessageBody(videofilePath, imageThumbPath, timeLength
                , EMFileHelper.getInstance().getFileLength(videofilePath));
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * 创建一个位置发送消息。
     * 
     * @param latitude 纬度。
     * @param longitude 经度。
     * @param locationAddress 位置详情。
     * @param buildingName 建筑物名称。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates a location message for sending.
     * 
     * @param latitude The latitude.
     * @param longitude The longitude.
     * @param locationAddress The location details.
     * @param buildingName The name of the building.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     */
    public static EMMessage createLocationSendMessage(double latitude, double longitude, String locationAddress, String buildingName, String username){
        EMMessage message = EMMessage.createSendMessage(Type.LOCATION);
        EMLocationMessageBody locBody = new EMLocationMessageBody(locationAddress, latitude, longitude, buildingName);
        message.addBody(locBody);
        message.setTo(username);
        
        return message;
    }

    /**
     * \~chinese
     * 创建一个位置发送消息。
     * 
     * @param latitude 纬度。
     * @param longitude 经度。
     * @param locationAddress 位置详情。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a location message for sending.
     * 
     * @param latitude The latitude.
     * @param longitude The longitude.
     * @param locationAddress The location details.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     */
    public static EMMessage createLocationSendMessage(double latitude, double longitude, String locationAddress, String username){
        EMMessage message = EMMessage.createSendMessage(Type.LOCATION);
        EMLocationMessageBody locBody = new EMLocationMessageBody(locationAddress, latitude, longitude);
        message.addBody(locBody);
        message.setTo(username);

        return message;
    }
    
    /**
     * \~chinese
     * 创建一个普通文件发送消息。
     * 
     * @param filePath 文件路径。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates a file message for sending.
     * 
     * @param filePath The file path.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     */
    public static EMMessage createFileSendMessage(String filePath, String username){
        return createFileSendMessage(EMFileHelper.getInstance().formatInUri(filePath), username);
    }

    /**
     * \~chinese
     * 创建一个普通文件发送消息。
     * 
     * @param filePath 文件路径。
     * @param username 消息接收方的设置取决于会话类型：
     *                 - 单聊：对方用户 ID。
     *                 - 群聊：群组 ID。
     *                 - 子区会话：子区 ID。
     *                 - 聊天室聊天：聊天室 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a file message for sending.
     * 
     * @param filePath The file path.
     * @param username The message sender is determined by the conversation type:
     *                 - One-to-one chat: The peer user ID.
     *                 - Group chat: The group ID.
     *                 - Thread conversation: The message thread ID.
     *                 - Chat room chat: The chat room ID.
     * @return The message instance.
     */
    public static EMMessage createFileSendMessage(Uri filePath, String username){
        if(!EMFileHelper.getInstance().isFileExist(filePath)){
            EMLog.e(TAG, "file does not exist");
            return null;
        }
        // 创建一个文件消息
        EMMessage message = EMMessage.createSendMessage(Type.FILE);

        message.setTo(username);
        // add message body
        EMNormalFileMessageBody body = new EMNormalFileMessageBody(filePath);
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * 创建一个合并转发消息的发送消息。
     *
     * 对于不支持合并转发消息的 SDK 版本，该类消息会被解析为文本 {@link Type#TXT} 消息，消息内容为 `compatibleText` 携带的内容，其他字段会被忽略。
     *
     * @param title             合并消息的标题。该字段可以设置为 `null` 或者空字符串。
     * @param summary           合并消息的概要。该字段可以设置为 `null` 或者空字符串。
     * @param compatibleText    合并消息的兼容信息。该字段可以设置为 `null` 或者空字符串。该字段用于需要兼容不支持合并转发消息的版本。
     * @param messageIdList     合并消息的消息 ID 列表。列表不可为`null`或者空，最多可包含 300 个消息 ID。调用 {@link EMChatManager#sendMessage(EMMessage)} 方法发送消息会检查该字段的设置。
     * @param userId            消息接收方。该字段的设置取决于会话类型：
     *                          - 单聊：对方用户 ID。
     *                          - 群聊：群组 ID。
     *                          - 子区会话：子区 ID。
     *                          - 聊天室聊天：聊天室 ID。
     * @return                  合并转发消息对象。
     *
     * \~english
     * Creates a combined message for sending.
     *
     * The SDKs that do not support combined messages will parse this type of message as a text message {@link Type#TXT} by resolving the value of `compatibleText` into the content of the text message and igoring other fields. 
     * and the content of the message will be from `compatibleText`, and other fields will be ignored.
     *
     * @param title             The title of the combined message. It can be `null` or an empty string ("").
     * @param summary           The summary of the combined message. It can be `null` or an empty string ("").
     * @param compatibleText    The compatible text of the combined message. It can be `null` or an empty string ("").
     * @param messageIdList     The list cannot be `null` or empty. It can contain a maximum of 300 message IDs.
     *                        When you call {@link EMChatManager#sendMessage(EMMessage)} to send a message, the SDK will check the setting of this field.
     * @param userId            The message recipient. The field setting is determined by the conversation type:
     *                          - One-to-one chat: The peer user ID.
     *                          - Group chat: The group ID.
     *                          - Thread conversation: The message thread ID.
     *                          - Chat room chat: The chat room ID.
     * @return                  The combined message instance.
     */
    public static EMMessage createCombinedSendMessage(String title, String summary, String compatibleText, List<String> messageIdList, String userId){
        EMMessage message = EMMessage.createSendMessage(Type.COMBINE);

        message.setTo(userId);
        // add message body
        EMCombineMessageBody body = new EMCombineMessageBody(title, summary, compatibleText, messageIdList);
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * （建议方法）设置消息体。
     * 
     * @param body 消息体。
     *
     * \~english
     * （Recommended）Sets the message body. 
     * 
     * @param body The message body.
     */
    public void setBody(EMMessageBody body) {
        if(this.body != body) {
            this.body = body;
            emaObject.clearBodies();
            emaObject.addBody(body.emaObject);
        }
    }

    /**
     * \~chinese
     * 添加消息体。
     * 
     * 本方法与设置消息体 {@link EMMessage#setBody(EMMessageBody)} 的方法效果相同。推荐使用后者。
     * 
     * 只支持一条消息添加一个消息体。
     * 
     * @param body 消息体。
     * 
     * \~english
     * Adds a message body. 
     * 
     * This method is equivalent to {@link EMMessage#setBody(EMMessageBody)} which sets the message body. The latter is recommended.
     * 
     * Only one message body can be added for one message.
     * 
     * @param body The message body.
     */
    public void addBody(EMMessageBody body) {
        this.body = body;
        emaObject.addBody(body.emaObject);
    }
    
    /**
     * \~chinese
     * 获取消息发送方的用户 ID。
     * 
     * @return 用户 ID。
     * 
     * \~english
     * Gets the user ID of the message sender.
     * 
     * @return  The user ID of the message sender.
     */
    public String getFrom() {
        return emaObject.from();
    }
    
    /**
     * \~chinese
     * 设置消息发送方的用户 ID。
     * 
     * @param from  发送方的用户 ID。
     * 
     * \~english
     * Sets the user ID of the message sender.
     * 
     * @param from The user ID of the message sender.
     */
    public void setFrom(String from) {
		emaObject.setFrom(from);
		// The reason for using "getTo ()" here is to prevent "from" as "conversationId" when the message is a group type
		if (!self().equals(from) && getTo() != null && getTo() != "" && getTo().equals(self())) {
			emaObject.setConversationId(from);
		}
	}

    /**
     * \~chinese
     * 获取消息撤回者的用户 ID。
     * 
     * @return 撤回者用户 ID。
     *
     * \~english
     * Gets the ID of the user that recalls the message.
     * 
     * @return  The ID of the user that recalls the message.
     */
    public String getRecaller(){
        return emaObject.getRecaller();
    }


    /**
     * \~chinese
     * 设置消息的接收方的用户 ID。
     * 
     * @param to    消息接收方的用户 ID。
     * 
     * \~english
     * Sets the user ID of the message recipient.
     * 
     * @param to    The user ID of the message recipient.
     */
	public void setTo(String to) {
        emaObject.setTo(to);
        // Removing the logic to check if it's you is to prevent you from being able to message yourself
        emaObject.setConversationId(to);
	}

	/**
	 * \~chinese
     * 获取消息接收者的用户名。
     * 
     * @return  接收者的用户名。
     * 
     * \~english
     * Gets the user ID of the message recipient.
     * 
     * @return  The user ID of the message recipient.
     */
    public String getTo() {
        return emaObject.to();
    }
    
    /**
     * \~chinese
     * 获取消息的 ID。
     * 
     * @return 消息 ID。
     * 
     * \~english
     * Gets the message ID.
     * 
     * @return  The message ID.
     */
    public String getMsgId() {
        return emaObject.msgId();
    }
    
    /**
     * \~chinese
     * 设置本地消息 ID。
     * 
     * @param msgId 消息 ID。
     * 
     * \~english
     * Sets the local message ID.
     * 
     * @param msgId  Sets the local message ID.
     */
    public void setMsgId(String msgId){
    	emaObject.setMsgId(msgId);
    }

    /**
     * The XXXListener doesn't need CallbackHolder, you will be responsible for adding and removing the listener reference.
     * The message's callback function is different from listener, user add callback, and won't remove the reference.
     * Old implementation for Message's callback is simply hold the reference, which leads to memory leak.
     * SDK caches hold reference of message, message hold callback, callback hold Activity's this reference, so activities will never be released.
     *
     * Solution:
     * 1. change the message's callback to weak reference.
     * 2. when sendMessage or downloadMessageAttachment make the reference be strong one.
     * 3. when callback be executed, onSuccess or onError, release the reference.
     *
     * Declaim:
     * 1. don't set message be strong when user call setMessageStatusCallback, user may just attach the callback, there will not be onError or onSuccess to be called
     */
    static class EMCallbackHolder implements EMCallBack {

        EMCallbackHolder(EMCallBack callback) {
            weak = new WeakReference<EMCallBack>(callback);
        }

        synchronized void update(EMCallBack callback) {
            if (strong != null) {
                strong = callback;
                return;
            }
            weak = new WeakReference<EMCallBack>(callback);
        }

        synchronized void makeItStrong() {
            if (strong != null) {
                return;
            }
            if (weak != null && weak.get() != null) {
                strong = weak.get();
            }
        }

        synchronized void release() {
            if (strong == null) {
                return;
            }
            weak = new WeakReference<EMCallBack>(strong);
            strong = null;
        }

        synchronized EMCallBack getRef() {
            if (strong != null) {
                return strong;
            }
            if (weak != null) {
                EMCallBack callBack = weak.get();
                if (callBack == null) {
                    EMLog.d(TAG, "getRef weak:" + callBack);
                }
                return callBack;
            }
            return null;
        }

        public void onSuccess() {
            if (innerCallback != null) {
                innerCallback.onSuccess();
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onSuccess();
                release();
            } else {
                EMLog.d(TAG, "CallbackHolder getRef: null");
            }
        }

        public void onError(int code, String error) {
            if (innerCallback != null) {
                innerCallback.onError(code, error);
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onError(code, error);
                release();
            } else {
                EMLog.d(TAG, "CallbackHolder getRef: null");
            }
        }

        public void onProgress(int progress, String status) {
            if (innerCallback != null) {
                innerCallback.onProgress(progress, status);
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onProgress(progress, status);
            }
        }

        // strong & weak are used for external callback reference
        // inner callback is used by chat manager sendMessage, to handle changes of msgId and msg time
        private EMCallBack strong;
        private WeakReference<EMCallBack> weak;
        EMCallBack innerCallback = null;
    }

    EMCallbackHolder messageStatusCallBack;

    synchronized void setInnerCallback(EMCallBack callback) {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.innerCallback = callback;
        } else {
            messageStatusCallBack = new EMCallbackHolder(null);
            messageStatusCallBack.innerCallback = callback;
        }
        setCallback(messageStatusCallBack);
    }

    /**
     * \~chinese
     * 设置消息状态变化的回调。
     * 
     * 设置消息状态回调，以便刷新界面。
     * 
     * @param callback  消息状态改变的回调。
     * 
     * \~english
     * Sets the message status change callback.
     * 
     * After the callback is triggered, the UI will be refreshed.
     * 
     * @param callback  The callback triggered when the message status changes.
     */
    public synchronized void setMessageStatusCallback(EMCallBack callback) {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.update(callback);
        } else {
            messageStatusCallBack = new EMCallbackHolder(callback);
        }
        setCallback(messageStatusCallBack);
    }


    void setCallback(final EMCallbackHolder holder) {
        this.emaObject.setCallback(new EMACallback(new EMCallBack(){

            @Override
            public void onSuccess() {
                if(holder != null){
                    holder.onSuccess();
                }
            }

            @Override
            public void onProgress(int progress, String status) {
                if(holder != null){
                    holder.onProgress(progress, status);
                }
            }

            @Override
            public void onError(int code, String error) {
                if(holder != null){
                    holder.onError(code, error);
                }
            }
        }));
    }

    void makeCallbackStrong() {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.makeItStrong();
        }
    }


    public String toString() {
        String sb = "msg{from:" + getFrom() +
                ", to:" + getTo() +
                " body:" + getBody();
        return sb;
    }
    
    /**
     * \~chinese
     * 设置消息的 Bool 类型的扩展属性。
     * 
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the Bool type.
     * 
     * @param attribute  The attribute name.
     * @param value  The attribute value.
     */
    public void setAttribute(String attribute, boolean value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }
    
    /**
     * \~chinese
     * 设置消息的 Int 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the Int type.
     * 
     * @param attribute The attribute name.
     * @param value The attribute value.
     */
    public void setAttribute(String attribute, int value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }


    /**
     * \~chinese
     * 设置消息的 Long 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the Long type.
     * 
     * @param attribute  The attribute name.
     * @param value  The attribute value.
     */
    public void setAttribute(String attribute, long value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }

    /**
     * \~chinese
     * 设置消息的 Float 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param value  属性值。
     *
     * \~english
     * Sets a message extension attribute of the Float type.
     * 
     * @param attribute  The attribute name.
     * @param value  The attribute value.
     */
    public void setAttribute(String attribute, float value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }

    /**
     * \~chinese
     * 设置消息的 Double 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param value  属性值。
     *
     * \~english
     * Sets a message extension attribute of the Double type.
     * 
     * @param attribute  The attribute name.
     * @param value  The attribute value.
     */
    public void setAttribute(String attribute, double value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }

    
    /**
     * \~chinese
     * 设置消息的 JSONObject 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the JSONObject type.
     * 
     * @param attribute  The attribute name.
     * @param value The attribute value.
     */
    public void setAttribute(String attribute, JSONObject value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setJsonAttribute(attribute, value.toString());
    }
    
    /**
     * \~chinese
     * 设置消息的 JSONArray 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the JSONArray type.
     * 
     * @param attribute The attribute name.
     * @param value The attribute value.
     */
    public void setAttribute(String attribute, JSONArray value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setJsonAttribute(attribute, value.toString());
    }
    
    /**
     * \~chinese
     * 设置消息的 String 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the String type.
     * 
     * @param attribute The attribute name.
     * @param value The attribute value.
     */
    public void setAttribute(String attribute, String value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }
    
    /**
     * \~chinese
     * 获取 Bool 类型的消息扩展属性。
     * 
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。详见 {@link com.hyphenate.EMError}。
     * 
     * \~english
     * Gets a message extension attribute of the Bool type.
     * 
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException  A description of the exception. See {@link com.hyphenate.EMError}.
     */
    public boolean getBooleanAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " can not be null or empty");
        }
        AtomicBoolean val = new AtomicBoolean();
        boolean exists = emaObject.getBooleanAttribute(attribute, false, val);
        if (!exists) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " not found");
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取 Bool 类型的消息扩展属性。
     * 
     * @param attribute 属性名。
     * @param defaultValue 属性的默认值。
     * @return 属性值。
     * 
     * \~english
     * Gets a message extension attribute of the Bool type.
     * 
     * @param attribute The attribute name.
     * @param defaultValue  The default value of the attribute.
     * @return The attribute value.
     */
    public boolean getBooleanAttribute(String attribute, boolean defaultValue){
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicBoolean val = new AtomicBoolean();
        boolean exists = emaObject.getBooleanAttribute(attribute, false, val);
        if (!exists) {
            return defaultValue;
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取消息的 Int 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param defaultValue 默认值。
     * @return 属性值。
     * 
     * \~english
     * Gets a message extension attribute of the Int type.
     * 
     * @param attribute The attribute name.
     * @param defaultValue The default value of the attribute.
     * @return The attribute value.
     */
    public int getIntAttribute(String attribute,int defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicInteger val = new AtomicInteger();
        boolean exists = emaObject.getIntAttribute(attribute, -1, val);
        if (!exists) {
            return defaultValue;
        }
        return val.intValue();
    }
    
    /**
     * \~chinese
     * 获取消息的 Long 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param defaultValue 属性的默认值。
     * @return 属性值。
     * 
     * \~english
     * Gets a message extension attribute of the Long type.
     * 
     * @param attribute The attribute name.
     * @param defaultValue The default value of the attribute.
     * @return The attribute value.
     */
    public long getLongAttribute(String attribute,long defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicLong val = new AtomicLong();
        boolean exists = emaObject.getLongAttribute(attribute, -1, val);
        if (!exists) {
            return defaultValue;
        }
        return val.longValue();
    }

    /**
     * \~chinese
     * 获取消息的 Float 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param defaultValue 属性的默认值。
     * @return 属性值。
     *
     * \~english
     * Gets a message extension attribute of the Float type.
     * 
     * @param attribute The attribute name.
     * @param defaultValue The default value of the attribute.
     * @return The attribute value.
     */
    public float getFloatAttribute(String attribute,float defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicReference<Float> val =new AtomicReference();
        boolean exists = emaObject.getFloatAttribute(attribute, defaultValue, val);
        if (!exists) {
            return defaultValue;
        }
        return val.get();
    }
    /**
     * \~chinese
     * 获取消息的 Double 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param defaultValue 属性的默认值。
     * @return 属性值。
     *
     * \~english
     * Gets a message extension attribute of the Double type.
     * 
     * @param attribute The attribute name.
     * @param defaultValue The default value of the attribute.
     * @return The attribute value.
     */
    public double getDoubleAttribute(String attribute,double defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicReference<Double> val =new AtomicReference<>();
        boolean exists = emaObject.getDoubleAttribute(attribute, defaultValue, val);
        if (!exists) {
            return defaultValue;
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取消息的 Int 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。详见 {@link com.hyphenate.EMError}。
     * 
     * \~english
     * Gets a message extension attribute of the Int type.
     * 
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException A description of the exception. See {@link com.hyphenate.EMError}.
     */
    public int getIntAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " can not be null or empty");
        }
        AtomicInteger val = new AtomicInteger();
        boolean exists = emaObject.getIntAttribute(attribute, -1, val);
        if (!exists) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " not found");
        }        
        return val.intValue();
    }
    
    /**
     * \~chinese
     * 获取消息的 Long 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。详见 {@link com.hyphenate.EMError}。
     * 
     * \~english
     * Gets a message extension attribute of the Long type.
     * 
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException A description of the exception. See {@link com.hyphenate.EMError}.
     */
    public long getLongAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " can not be null or empty");
        }
        AtomicLong val = new AtomicLong();
        boolean exists = emaObject.getLongAttribute(attribute, -1, val);
        if (!exists) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " not found");
        }        
        return val.longValue();
    }
    /**
     * \~chinese
     * 获取消息的 Float 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。详见 {@link com.hyphenate.EMError}。
     *
     * \~english
     * Gets a message extension attribute of the Float type.
     * 
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException A description of the exception. See {@link com.hyphenate.EMError}.
     */
    public float getFloatAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " can not be null or empty");
        }
        AtomicReference<Float> val =new AtomicReference();
        boolean exists = emaObject.getFloatAttribute(attribute, -1f, val);
        if (!exists) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " not found");
        }
        return val.get();
    }
    /**
     * \~chinese
     * 获取消息的 Double 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。详见 {@link com.hyphenate.EMError}。
     *
     * \~english
     * Gets a message extension attribute of the double type.
     * 
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException A description of the exception. See {@link com.hyphenate.EMError}.
     */
    public double getDoubleAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " can not be null or empty");
        }
        AtomicReference<Double> val =new AtomicReference<>();
        boolean exists = emaObject.getDoubleAttribute(attribute, -1d, val);
        if (!exists) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " not found");
        }
        return val.get();
    }
    /**
     * \~chinese
     * 获取消息的 String 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。详见 {@link com.hyphenate.EMError}。
     * 
     * \~english
     * Gets a message extension attribute of the String type.
     * 
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException A description of the exception. See {@link com.hyphenate.EMError}.
     */
    public String getStringAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " can not be null or empty");
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getStringAttribute(attribute, "", val);
        if (!exists) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " not found");
        }        
        return val.toString();
    }
    
    /**
     * \~chinese
     * 获取消息的 String 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @param defaultValue 默认值。
     * @return 属性值。
     * 
     * \~english
     * Gets a message extension attribute of the String type.
     * 
     * @param attribute The attribute name.
     * @param defaultValue The default value of the attribute.
     * @return The attribute value.
     */
    public String getStringAttribute(String attribute, String defaultValue){
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getStringAttribute(attribute, "", val);
        if (!exists) {
            return defaultValue;
        }        
        return val.toString();
    }
    
    
    /**
     * \~chinese
     * 获取消息的 JSONObject 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。详见 {@link com.hyphenate.EMError}。
     * 
     * \~english
     * Gets a message extension attribute of the JSONObject type.
     * 
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException A description of the exception. See {@link com.hyphenate.EMError}.
     */
    public JSONObject getJSONObjectAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " can not be null or empty");
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getJsonAttribute(attribute, "{}", val);
        if (exists) {
            try {
                JSONObject jsonObj = new JSONObject(val.toString());
                return jsonObj;
            } catch (JSONException e) {
                throw new HyphenateException(EMError.GENERAL_ERROR, e.getMessage());
            }
        }
        throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " not found");
    }
    
    /**
     * \~chinese
     * 获取消息的 JSONArray 类型扩展属性。
     * 
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。详见 {@link com.hyphenate.EMError}。
     * 
     * \~english
     * Gets a message extension attribute of the JSONArray type.
     * 
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException A description of the exception. See {@link com.hyphenate.EMError}.
     */
    public JSONArray getJSONArrayAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " can not be null or empty");
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getJsonAttribute(attribute, "[]", val);
        if (exists) {
            try {
                JSONArray jsonArray = new JSONArray(val.toString());
                return jsonArray;
            } catch (JSONException e) {
                throw new HyphenateException(EMError.GENERAL_ERROR, e.getMessage());
            }
        }
        throw new HyphenateException(EMError.GENERAL_ERROR, "attribute " + attribute + " not found");
    }
    
    /**
     * \~chinese
     * 获取聊天类型。
     * 
     * @return  聊天类型。
     * 
     * \~english
     * Gets the chat type.
     * 
     * @return The chat type.
     */
    public ChatType getChatType() {
    	EMAMessage.EMAChatType t = emaObject.chatType();
    	ChatType type = ChatType.Chat;
    	if (t == EMAMessage.EMAChatType.SINGLE) {
    		type = ChatType.Chat;
    	} else if (t == EMAMessage.EMAChatType.GROUP) {
    		type = ChatType.GroupChat;
    	} else {
    		type = ChatType.ChatRoom;
    	}
    	return type;
    }
    
    /**
     * \~chinese
     * 设置聊天类型。
     * 
     * @param chatType  聊天类型，默认为单聊。详见 {@link ChatType}。
     *
     * \~english
     * Sets the chat type.
     * 
     * @param chatType  The chat type. See {@link ChatType}. 
     */
    public void setChatType(ChatType chatType) {
    	EMAMessage.EMAChatType t = EMAMessage.EMAChatType.SINGLE;
    	if (chatType == ChatType.Chat) {
    		t = EMAMessage.EMAChatType.SINGLE;
    	} else if (chatType == ChatType.GroupChat) {
    		t = EMAMessage.EMAChatType.GROUP;
    	} else {
    		t = EMAMessage.EMAChatType.CHATROOM;
    	}
    	emaObject.setChatType(t);
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(getMsgId());
        out.writeLong(emaObject.getNativeHandler());
    }
    
    public static final Creator<EMMessage> CREATOR
        = new Creator<EMMessage>() {
        public EMMessage createFromParcel(Parcel in) {
            EMMessage msg = null;
            try {
                msg = new EMMessage(in);
            } catch (HyphenateException e) {
                e.printStackTrace();
            }
            return msg;
        }

        public EMMessage[] newArray(int size) {
            return new EMMessage[size];
        }
    };
    
	@SuppressWarnings("CloneDoesntCallSuperClone")
    @Override
    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    /**
     * \~chinese
     * 获取对方是否已读。 
     * 
     * @return 消息是否已读：
     *         -`true`：已读。
     *         -`false`：未读。
     * 
     * \~english
     * Gets whether the message is read.
     * 
     * @return whether the message is read:
     *         -`true`: Yes.
     *         -`false`: No.
     */
	public boolean isAcked() {
		return emaObject.isAcked();
	}

    /**
     * \~chinese
     * 设置对方是否已读。
     * 
     * 该方法由 SDK 调用，而非开发者。
     * 
     * @param isAcked  消息是否已读：
     *                 -`true`：已读。
     *                 -`false`：未读。
     * 
     * \~english
     * Sets whether the message is read.
     * 
     * The method is supposed to be called by the SDK instead of you.
     * 
     * @param isAcked  Whether the message is read:
     *         -`true`: Yes.
     *         -`false`: No.
     */
	public void setAcked(boolean isAcked) {
		emaObject.setIsAcked(isAcked);
	}

    /**
     * \~chinese
     * 获取消息是否成功送达。
     * 
     * @return 消息是否成功送达。
     *         - `true`：已送达。
     *         - `false`：未送达。
     * 
     * \~english
     * Gets whether the message is successfully delivered.
     * 
     * @return Whether the message is successfully delivered:
     *         -`true`: Yes.
     *         -`false`: No.
     */
	public boolean isDelivered() {
		return emaObject.isDeliverAcked();
	}

    /**
     * \~chinese
     * 设置消息是否成功送达。
     * 
     * 该方法由 SDK 调用，而非开发者。
     * 
     * @param isDelivered 消息是否成功送达。
     *         - `true`：已送达。
     *         - `false`：未送达。
     *
     * \~english
     * Sets whether the message is successfully delivered.
     * 
     * The method is supposed to be called by the SDK instead of you.
     * 
     * @param isDelivered Whether the message is successfully delivered:
     *         -`true`: Yes.
     *         -`false`: No.
     */
	public void setDelivered(boolean isDelivered) {
		emaObject.setIsDeliverAcked(isDelivered);
	}

    /**
     * \~chinese
     * 检查消息是否未读。
     * 
     * @return 消息是否未读。
     * - `true`: 未读；
     * - `false`: 已读。
     *
     * \~english
     * Checks whether the message is unread.
     * 
     * @return Whether the message is unread. 
     * -`true`: Yes. 
     * -`false`: No.
     */
	public boolean isUnread() {
		return !emaObject.isRead();
	}

    /**
     * \~chinese
     * 设置消息是否未读。
     *
     * 建议在会话中，使用 {@link EMConversation#markAllMessagesAsRead()}。
     * 
     * @param unread  消息是否未读：
     * - `true`:设置消息为未读；
     * - `false`:设置消息为已读。
     *
     * \~english
     * Sets whether the message is unread. 
     * 
     * We recommend you to use {@link EMConversation#markAllMessagesAsRead()} to mark all messages in a conversation as read.
     * 
     * @param unread Whether the message is unread. 
     * - `true`: Yes.
     * - `false`: No.
     * 
     */
	public void setUnread(boolean unread) {
	    EMLog.d(TAG, "setUnread unread: " + unread + " msgId: " + getMsgId());
		emaObject.setIsRead(!unread);
	}

	/**
	 * \~chinese
	 * 获取语音消息是否已听。
     * 
	 * @return 语音是否已听：
     *         - `true`：已听；
     *         - `false`：未听。
	 * 
	 * \~english
	 * Gets whether the voice message is listened.
     * 
     * @return  Whether the voice message is listened:
     *          - `true`: Yes.
     *          - `false`: No.
	 */
	public boolean isListened() {
		return emaObject.isListened();
	}

	/**
	 * \~chinese
	 * 设置语音消息是否已听。
     * 
     * 该方法由 SDK 调用，而非开发者。
     * 
	 * @param isListened 语音消息是否已听：
     *                   - `true`：已听。
     *                   - `false`：未听。
     * 
	 * \~english
	 * Sets whether the voice message is listened.
     * 
     * The method is supposed to be called by the SDK instead of you.
     * 
	 * @param isListened  Whether the voice message is listened:
     *          - `true`: Yes.
     *          - `false`: No.
	 */
	public void setListened(boolean isListened) {
	    emaObject.setListened(isListened);
	}
	
	/**
	 * \~chinese
	 * 获取对端用户的 ID：
     *            - 单聊：对端用户的 ID；
     *            - 群聊：群组 ID；
     *            - 子区会话：子区 ID;
     *            - 聊天室：聊天室 ID。
     * 
	 * @return 对端用户的 ID。
	 * 
	 * \~english
	 * Gets the user ID of the other party:
     *            - One-to-one chat: The user ID of the peer user.
     *            - Group chat: The group ID.
     *            - Thread conversation: The message thread ID.
     *            - Chat room chat: The chat room ID.
     * 
     * @return The user ID of the peer user.
	 */
	public String getUserName() {
		String username = "";
		if (getFrom() != null && getFrom().equals(EMClient.getInstance().getCurrentUser())) {
			username = getTo();
		} else {
			username = getFrom();
		}
		return username;
	}

    /**
     * \~chinese
     * 设置消息已送达标记。
     * 
     * 开发者不要调用，由 SDK 内部调用。
     * 
     * @param isDeliverAcked 
     *                       - `true`：消息已送达。
     *                       - `false`：消息未送达。
     *
     * \~english
     * Sets the message deliver ack.
     * 
     * The method is supposed to be called by the SDK instead of you.
     * 
     * @param isDeliverAcked - `true`: The message is delivered to the other party.
     *                       - `false`: The message has not been delivered to the other party.
     */
	public void setDeliverAcked(boolean isDeliverAcked) {
	    emaObject.setIsDeliverAcked(isDeliverAcked);
	}

    /**
     * \~chinese
     * 消息中的附件的上传或者下载进度。
     * 
     * 消息附件的缩略图不涉及进度信息。
     * 
     * @return 消息中的附件的上传或者下载进度，取值范围为 [0,100]。
     *
     * \~english
     * The progress for uploading or downloading a message attachment.
     * 
     * The upload or download progress is unavailable for the thumbnail of the message attachments.
     * 
     * @return The progress for uploading or downloading the message attachment. The value range is [0,100].
     */
	public int progress() {
	    return emaObject.progress();
	}

    /**
     * \~chinese
     * 设置消息中的附件的上传或者下载进度。
     * 
     * 对于 app 开发者来说，通常不需要主动设置进度。详见 {@link EMMessage#progress()}。
     * 
     * @param progress 消息中的附件的上传或者下载进度，取值范围为 [0,100]。
     *
     * \~english
     * Sets the progress for uploading or downloading a message attachment.
     * 
     * Usually, you do not need to set the progress value. See {@link EMMessage#progress()}.
     * 
     * @param progress The progress for uploading or downloading the message attachment. The value range is [0,100].
     */
    public void setProgress(int progress) {
        emaObject.setProgress(progress);
    }

	/**
	 * \~chinese
	 * 消息方向。
     * - SEND：该消息是当前用户发送出去的。
     * - RECEIVE：该消息是当前用户接收到的。
     * 
     * @return 发送或接收，详见 {@link Direct}。
	 * 
	 * \~english
	 * The message direction.
     * - SEND: This message is sent from the local client.
     * - RECEIVE: The message is received by the local client.
     *
     * @return The message direction. See {@link Direct}.
	 */
	public Direct direct() {
	    if (emaObject.direction() == EMADirection.SEND) {
	        return Direct.SEND;
	    } else {
	        return Direct.RECEIVE;
	    }
	}
	
	/**
	 * \~chinese
	 * 设置消息方向。
     * 
	 * @param dir 消息的方向，详见 {@link Direct}。
	 * 
	 * \~english
	 * Sets the message direction.
     * 
	 * @param dir  The message direction. See {@link Direct}.
	 */
	public void setDirection(Direct dir) {
	    emaObject.setDirection(dir.ordinal());
	}

    /**
     * \~chinese
     * 获取会话 ID。
     * 
     * @return  会话 ID。
     *
     * \~english
     * Gets the conversation ID.
     * 
     * @return The conversation ID.
     */
	public String conversationId() {
	    return emaObject.conversationId();
	}

    static String self() {
        String myself = EMClient.getInstance().getCurrentUser();
        if (myself == null) {
            //currentUser is only set after login successfully
            //since it is not when not logged in, use last login user instead
            myself = EMSessionManager.getInstance().getLastLoginUser();
        }
        return myself;
    }

    private EMMessage(Parcel in) throws HyphenateException {
        String msgId = in.readString();
        long nativeHandler = in.readLong();
        EMMessage msg = EMClient.getInstance().chatManager().getMessage(nativeHandler);
        if (msg == null) {
            msg = EMClient.getInstance().chatManager().getMessage(msgId);
            if(msg == null) {
                throw new HyphenateException("EMMessage constructed from parcel failed");
            }
        }
        emaObject = msg.emaObject;
    }

    /**
     * \~chinese
     * 获取消息包含的全部扩展字段。
     *
     * @return  消息的全部扩展字段，以 Map<String, Object> 形式返回。返回 Entry.value 的具体类型有 Boolean, Integer, Long, Float, Double, String。
     *          输入 JsonObject 的属性，map 结构返回时属性是 String。
     *
     * \~english
     * Gets all message extension fields. The return type is Map<String, Object>.
     *
     * @return  All message extension fields which are returned as a Map collection, Map<String, Object>. The Map collection contains key-value pairs where the key is the type (Boolean, Integer, Long, Float, Double, or String) of the extension field, and the value is the name of the extension field.
     *          If JsonObject is passed as the extension field type, it is returned as String. 
     */
    public Map<String, Object> ext() {
        return emaObject.ext();
    }

    /**
     * \~chinese
     * 获取消息包含的全部扩展字段。
     *
     * @return  消息的全部扩展字段，以 Map<String, Object> 形式返回，其中 Entry.value 可为 Boolean、Integer、Long、Float、Double、String、JsonObject、或 JsonArray 类型。
     *
     * \~english
     * Gets all extension fields of the message.
     *
     * @return  All extension fields in the message are returned as a Map collection of the Map<String, Object> type.
     *          The Map collection contains key-value pairs where the key is the data type (Boolean, Integer, Long, Float, Double, String, JsonObject, or JsonArray) of an extension field,
     *          and the value is the field name.
     */
    public Map<String, Object> getAttributes() {
        return emaObject.getAttributes();
    }


    /**
     * \~chinese
     * 获取 Reaction 列表。
     * 
     * @return  Reaction 列表。
     *
     * \~english
     * Gets the list of Reactions.
     * 
     * @return The list of Reactions.
     */
    public List<EMMessageReaction> getMessageReaction() {
        List<EMAMessageReaction> messageReactions = emaObject.reactionList();
        if (messageReactions.size() > 0) {
            List<EMMessageReaction> reactionList = new ArrayList<>(messageReactions.size());
            EMMessageReaction reaction;
            for (EMAMessageReaction emaMessageReaction : messageReactions) {
                reaction = new EMMessageReaction(emaMessageReaction);
                reactionList.add(reaction);
            }
            return reactionList;
        }
        return null;
    }

    /**
     * \~chinese
     * 是否为在线消息。
     * 
     * @note
     * 该字段标记服务器下发消息时判断用户是在线还是离线的状态。
     *
     * 该字段为服务器下发字段，不在本地数据库中存储。对于从数据库读取的消息或拉取的漫游消息，该字段的值默认为 `true`。
     *
     * @return  是否为在线消息。
     * - （默认）`true`：是。
     * - `false`：否。
     * 
     *
     * \~english
     *  Whether the message gets delivered to an online user.
     * 
     * @note
     * This field is issued by the server to indicate whether the recipient is online when the message is delivered.
     *
     * This field is not stored in the local database. The value of this field is `true` by default for messages read from the database or pulled from the server.
     *
     * @return Whether the recipient is online when the message is delivered.
     * - (Default)`true`: Yes.
     * - `false`: No. The recipient is offline.
     */
    public boolean isOnlineState() {
        return  emaObject.isOnlineState();
    }


    /**
     * \~chinese
     * 设置消息优先级  chatroom message priority
     * @param priority
     *
     * \~english
     * Set message priority  chatroom message priority
     * @param priority
     */
    public void setPriority(EMChatRoomMessagePriority priority){
       emaObject.setPriority(priority);
    }

    public enum EMChatRoomMessagePriority{
        /**
         *\~chinese
         * 高。
         *
         *\~english
         * High.
         */
        PriorityHigh,
        /**
         *\~chinese
         * 中。
         *
         *\~english
         * Normal.
         */
        PriorityNormal,
        /**
         *\~chinese
         * 低。
         *
         *\~english
         * Lower.
         */
        PriorityLow,
    }

    /**
     * \~chinese
     * 判断消息是否只投递在线用户。
     * 
     * @return - `true`：只投递在线用户。
     * - `false`：投递所有用户。
     *
     * \~english
     * Checks whether this message is only delivered to online users.
     * 
     * @return - `true`: The message is delivered only to online users.
     * - `false`: The message is delivered to all users, regardless of whether they are online or offline.
     */
    public boolean isDeliverOnlineOnly() {
        return emaObject.isDeliverOnlineOnly();
    }

    /**
     * \~chinese
     * 设置消息是否只投递给在线用户。
     *
     * @param onlineOnly    消息是否只投递给在线用户：
     *                      - `true`：只有消息接收方在线时才能投递成功。若接收方离线，则消息会被丢弃。
     *                      - `false`（默认）：如果用户在线，则直接投递；如果用户离线，消息会在用户上线时投递。
     *
     * \~english
     * Sets whether the current msg is only delivered to online users.
     *
     * @param onlineOnly    Whether the message is delivered only when the recipient(s) is/are online:
     *                      - `true`：The message is delivered only when the recipient(s) is/are online. If the recipient(s) is/are offline, the message is discarded.
     *                      - (Default) `false`：The message is delivered when the recipient(s) is/are online. If the recipient(s) is/are offline, the message will not be delivered to them until they get online.
     */
    public void deliverOnlineOnly(boolean onlineOnly) {
        emaObject.deliverOnlineOnly(onlineOnly);
    }

    /**
     *  \~chinese
     * 设置定向消息接收方。
     * 
     * 该方法适用于群组和聊天室。若创建消息后不调用该方法，则消息发送给群组或聊天室的所有成员。
     * 
     * @param receiverList 定向消息接收方。若传入 `null`，则消息发送给群组或聊天室的所有成员。
     *
     *  \~english
     *  Sets the recipient list of a targeted message.
     * 
     *  This method is used only for groups and chat rooms. If this method is not called after message created, messages are sent to all members in the group or chat room.
     * 
     * @param receiverList The recipient list of a targeted message. If you pass in `null`, the messages are sent to all members in the group or chat room.
     */
    public void setReceiverList(List<String> receiverList){
        emaObject.setReceiverList(receiverList);
    }

    /**
     * \~chinese
     * 定向消息的接收方。
     * 
     * @return 定向消息的接收方。
     *
     * \~english
     *  
     * The recipient list of a targeted message.
     * 
     *  @return The recipient list of a targeted message.
     */
    public List<String> receiverList(){
        return emaObject.receiverList();
    }
}
