/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import com.hyphenate.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.adapter.message.EMAVideoMessageBody;
import com.hyphenate.util.EMFileHelper;
import com.hyphenate.util.EMLog;

/**
 * \~chinese
 * 视频消息体。
 * ```java
 *     EMVideoMessageBody body = new EMVideoMessageBody(videoFilePath,thumbPath, duration, filelength);
 * ```
 *
 * \~english
 * The video message body.
 * ```java
 *     EMVideoMessageBody body = new EMVideoMessageBody(videoFilePath, thumbPath, duration, filelength);
 * ```
 */
public class EMVideoMessageBody extends EMFileMessageBody implements Parcelable {
    private static final String TAG = EMVideoMessageBody.class.getSimpleName();

    public EMVideoMessageBody() {
        super("", EMAMessageBody.EMAMessageBodyType_VIDEO);
    }

    public EMVideoMessageBody(EMAVideoMessageBody body) {
        super(body);
    }

    /**
     * \~chinese
     * 创建视频消息体。
     *
     * @param videoFilePath 视频文件路径。
     * @param thumbPath     缩略图路径。
     * @param duration      视频时长，单位是秒。
     * @param filelength    视频文件大小。
     *
     * \~english
     * Creates a video message body.
     *
     * @param videoFilePath The path of the video file.
     * @param thumbPath     The thumbnail URI.
     * @param duration      The video duration in seconds.
     * @param filelength    The size of the video file.
     */
    public EMVideoMessageBody(String videoFilePath, String thumbPath, int duration, long filelength) {
        this(EMFileHelper.getInstance().formatInUri(videoFilePath), EMFileHelper.getInstance().formatInUri(thumbPath), duration, filelength);
        EMLog.d(EMClient.TAG, "EMVideoMessageBody thumbPath = "+thumbPath);
    }

    /**
     * \~chinese
     * 创建视频消息体。
     *
     * @param videoFilePath 视频缩略图的资源标识符。
     * @param thumbPath     缩略图的资源标识符。
     * @param duration      视频时长，单位为秒。
     * @param filelength    视频文件大小。
     *
     * \~english
     * Creates a video message body.
     *
     * @param videoFilePath The URI of the video thumbnail.
     * @param thumbPath     The thumbnail URI.
     * @param duration      The video duration in seconds.
     * @param filelength    The size of the video file.
     */
    public EMVideoMessageBody(Uri videoFilePath, Uri thumbPath, int duration, long filelength) {
        super(videoFilePath, EMAMessageBody.EMAMessageBodyType_VIDEO);
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(thumbPath));
        ((EMAVideoMessageBody) emaObject).setDuration(duration);
        ((EMAVideoMessageBody) emaObject).setDisplayName(EMFileHelper.getInstance().getFilename(videoFilePath));
        ((EMAVideoMessageBody) emaObject).setFileLength(filelength);
        EMLog.d("videomsg", "create video, message body for:" + videoFilePath + " filename = "+EMFileHelper.getInstance().getFilename(videoFilePath));
        EMLog.d(EMClient.TAG, "EMVideoMessageBody thumbPath = "+thumbPath);
    }

    /**
     * create video message body from incoming xmpp message json
     */
    EMVideoMessageBody(String fileName, String remoteUrl, String thumbnailUrl, int length) {
        super(fileName, EMAMessageBody.EMAMessageBodyType_VIDEO);
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(thumbnailUrl);
        ((EMAVideoMessageBody) emaObject).setLocalPath(fileName);
        ((EMAVideoMessageBody) emaObject).setRemotePath(remoteUrl);
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(thumbnailUrl);
        ((EMAVideoMessageBody) emaObject).setFileLength(length);

    }

    /**
     * \~chinese
     * 获取视频文件的大小。
     *
     * @return 视频文件的大小。
     *
     * \~english
     * Gets the size of the video file in bytes.
     *
     * @return The size of the video file.
     */
    public long getVideoFileLength() {
        return ((EMAVideoMessageBody) emaObject).fileLength();
    }

    /**
     * \~chinese
     * 设置视频文件的文件大小。
     * 用户在做数据库迁移的时候可能使用。
     * @param filelength 视频文件的大小。
     *
     * \~english
     * Sets the size of the video file in bytes.
     * This method may be used during database migration.
     * @param filelength The size of the video file.
     */
    public void setVideoFileLength(long filelength) {
        ((EMAVideoMessageBody) emaObject).setFileLength(filelength);
    }

    /**
     *  \~chinese
     *  获取缩略图在服务器的路径。
     * @return 缩略图在服务器的路径。
     *
     *  \~english
     *  Gets the URL of the thumbnail on the server.
     * @return The URL of the thumbnail on the server.
     */
    public String getThumbnailUrl() {
        return ((EMAVideoMessageBody) emaObject).thumbnailRemotePath();
    }

    /**
     *  \~chinese
     *  设置缩略图在服务器的路径。
     * @param thumbnailUrl  缩略图在服务器的路径。
     *
     *  \~english
     *  Sets the URL of the thumbnail on the server.
     *  @param thumbnailUrl The URL of the thumbnail on the server.
     */
    public void setThumbnailUrl (String thumbnailUrl) {
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    /**
     * \~chinese
     * 设置视频缩略图的尺寸。
     * @param width     视频缩略图的宽度。
     * @param height    视频缩略图的高度。
     *
     * \~english
     * Sets the size of the video thumbnail.
     * @param width     The thumbnail's width.
     * @param height    The thumbnail's height.
     */
    public void setThumbnailSize(int width, int height) {
        ((EMAVideoMessageBody)emaObject).setSize(width, height);
    }
    /**
     * \~chinese
     * 获取视频缩略图的宽度。
     * @return  视频缩略图的宽度。
     *
     * \~english
     * Gets the width of the video thumbnail.
     * @return  The width of video thumbnail.
     */
    public int getThumbnailWidth() {
        return ((EMAVideoMessageBody)emaObject).width();
    }

    /**
     * \~chinese
     * 获取视频缩略图的高度。
     * @return  视频缩略图的高度。
     *
     * \~english
     * Gets the height of the video thumbnail.
     * @return  The height of the video thumbnail.
     */
    public int getThumbnailHeight() {
        return ((EMAVideoMessageBody)emaObject).height();
    }

    /**
     *  \~chinese
     *  获取视频缩略图的本地绝对路径或者资源标识符的字符串形式。
     *
     *  @return 视频缩略图的本地路径。
     *
     *  \~english
     *  Gets the local absolute path of the video thumbnail or its URI in string format.
     *
     *  @return The local path of the video thumbnail.
     */
    public String getLocalThumb() {
        return EMFileHelper.getInstance().formatOutLocalUrl(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
    }

    /**
     *  \~chinese
     *  获取缩略图的本地资源标识符。
     *
     *  @return 缩略图的本地资源标识符。
     *
     *  \~english
     *  Gets the local URI of the video thumbnail.
     *
     *  @return The local URI of the video thumbnail.
     */
    public Uri getLocalThumbUri() {
        return EMFileHelper.getInstance().formatOutUri(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
    }

    /**
     * \~chinese
     * 设置视频缩略图路径。
     * @param localThumbPath  视频缩略图路径，可以是缩略图的绝对路径或资源标识符。
     *
     * \~english
     * Sets the path of the video thumbnail.
     * @param localThumbPath The path of the video thumbnail, which can be the absolute path or URI of the video thumbnail.
     */
    public void setLocalThumb(String localThumbPath) {
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(localThumbPath));
    }

    /**
     * \~chinese
     * 设置视频缩略图的路径。
     * @param localThumbPath    视频缩略图的资源标识符。
     *
     * \~english
     * Sets the path of the video thumbnail.
     * @param localThumbPath    The URI of video thumbnail.
     */
    public void setLocalThumb(Uri localThumbPath) {
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(EMFileHelper.getInstance().formatInUriToString(localThumbPath));
    }

    /**
     * \~chinese
     * 获取视频时长, 单位为秒。
     *
     * @return 视频时长, 单位为秒。
     *
     * \~english
     * Gets the video duration in seconds.
     *
     * @return  The video duration in seconds.
     */
    public int getDuration() {
        return (int) ((EMAVideoMessageBody) emaObject).duration();
    }

    public String toString() {
        return "video: " + ((EMAVideoMessageBody) emaObject).displayName() +
                ", localUrl: " + ((EMAVideoMessageBody) emaObject).getLocalUrl() +
                ", remoteUrl: " + ((EMAVideoMessageBody) emaObject).getRemoteUrl() +
                ", thumbnailUrl: " + ((EMAVideoMessageBody) emaObject).thumbnailLocalPath() +
                ", length: " + ((EMAVideoMessageBody) emaObject).fileLength();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        // members from FileMessageBody
        dest.writeString(((EMAVideoMessageBody) emaObject).displayName());
        dest.writeString(((EMAVideoMessageBody) emaObject).getLocalUrl());
        dest.writeString(((EMAVideoMessageBody) emaObject).getRemoteUrl());
        dest.writeString(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
        dest.writeString(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
        dest.writeInt(((EMAVideoMessageBody) emaObject).duration());
        dest.writeLong(((EMAVideoMessageBody) emaObject).fileLength());
        dest.writeInt(((EMAVideoMessageBody) emaObject).width());
        dest.writeInt(((EMAVideoMessageBody) emaObject).height());
    }

    public static final Parcelable.Creator<EMVideoMessageBody> CREATOR = new Creator<EMVideoMessageBody>() {

        @Override
        public EMVideoMessageBody[] newArray(int size) {
            return new EMVideoMessageBody[size];
        }

        @Override
        public EMVideoMessageBody createFromParcel(Parcel in) {

            return new EMVideoMessageBody(in);
        }
    };

    private EMVideoMessageBody(Parcel in) {
        super("", EMAMessageBody.EMAMessageBodyType_VIDEO);
        // members from FileMessageBody
        ((EMAVideoMessageBody) emaObject).setDisplayName(in.readString());
        ((EMAVideoMessageBody) emaObject).setLocalPath(in.readString());
        ((EMAVideoMessageBody) emaObject).setRemotePath(in.readString());
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(in.readString());
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(in.readString());
        ((EMAVideoMessageBody) emaObject).setDuration(in.readInt());
        ((EMAVideoMessageBody) emaObject).setFileLength(in.readLong());
        int width = in.readInt();
        int height = in.readInt();
        ((EMAVideoMessageBody) emaObject).setSize(width, height);
    }

    /**
     *  \~chinese
     *  获取视频缩略图的密钥。
     *  下载视频缩略图时需要密匙做校验。
     * @return 视频缩略图的密钥。
     *
     *  \~english
     *  Gets the secret key of the video thumbnail.
     *  The secret key is used for validation during download of the video thumbnail.
     * @return The secret key of the video thumbnail.
     */
    public String getThumbnailSecret() {
        return ((EMAVideoMessageBody) emaObject).thumbnailSecretKey();
    }

    /**
     * \~chinese
     * 设置视频缩略图的密钥。
     * @param secret 视频缩略图的密钥。
     *
     * \~english
     * Sets the secret key of the video thumbnail.
     * @param secret The secret key of the video thumbnail.
     */
    public void setThumbnailSecret(String secret) {
        ((EMAVideoMessageBody) emaObject).setThumbnailSecretKey(secret);
    }

    /**
     * \~chinese
     * 获取视频缩略图的下载状态。
     *
     * @return  视频缩略图的下载状态。
     *
     * \~english
     * Gets the download status of the video thumbnail.
     *
     * @return  The download status of the video thumbnail.
     */
    public EMDownloadStatus thumbnailDownloadStatus() {
        EMADownloadStatus _status = ((EMAVideoMessageBody) emaObject).thumbnailDownloadStatus();
        switch (_status) {
            case DOWNLOADING:
                return EMDownloadStatus.DOWNLOADING;
            case SUCCESSED:
                return EMDownloadStatus.SUCCESSED;
            case FAILED:
                return EMDownloadStatus.FAILED;
            case PENDING:
                return EMDownloadStatus.PENDING;
        }
        return EMDownloadStatus.SUCCESSED;
    }

    /**
     * \~chinese
     * 设置视频缩略图的下载状态。
     *
     * @param status  视频缩略图的下载状态。
     *
     * \~english
     * Sets the download status of the video thumbnail.
     *
     * @param status  The download status of the video thumbnail.
     */
    public void setThumbnailDownloadStatus(EMDownloadStatus status) {
        ((EMAVideoMessageBody) emaObject).setThumbnailDownloadStatus(EMADownloadStatus.valueOf(status.name()));
    }
}
