package com.hyphenate.chat;

import android.os.AsyncTask;

import com.hyphenate.EMError;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMATranslateManager;
import com.hyphenate.chat.adapter.EMATranslateResult;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * \~chinese
 * 翻译信息管理类，负责对 SDK 本地数据库中保存的翻译信息进行查找、增加、修改和删除。
 * @deprecated 使用 {@link EMChatManager#fetchSupportLanguages(EMValueCallBack)} {@link EMChatManager#translateMessage(EMMessage, List, EMValueCallBack)} 替代。
 *
 * \~english
 * The translation information management class, which is responsible for searching for, adding, modifying, and deleting translation information that is saved in the SDK local database.
 * @deprecated use {@link EMChatManager#fetchSupportLanguages(EMValueCallBack)} {@link EMChatManager#translateMessage(EMMessage, List, EMValueCallBack)} instead.
 */
@Deprecated
public class EMTranslationManager {
    private static final String TAG = EMTranslationManager.class.getSimpleName();

    public final static int MaxTranslationTextSize = 5000;

    static private EMTranslationManager instance = null;

    private EMTranslator mTranslator;

    private AtomicBoolean mInitializing = new AtomicBoolean(false);
    private AtomicBoolean mInitialized = new AtomicBoolean(false);

    private List<EMLanguage> mLanguageList;

    EMATranslateManager emaObject;
    static final int MaxCacheSize = 10000;
    private CacheManager mCacheManager;

    EMTranslationManager(EMATranslateManager translateManager){
        emaObject = translateManager;
        mLanguageList = new ArrayList<>();
        mCacheManager = new CacheManager(MaxCacheSize);
    }

    /**
     * \~chinese
     * 初始化翻译参数。
     *
     * @param params 参数对象。
     *
     * \~english
     * Initializes translation parameters.
     *
     * @param params The parameter objects.
     */
    public void init(EMTranslateParams params) {
        if( !EMClient.getInstance().isLoggedInBefore())
            return;

        if(mInitializing.compareAndSet(false, true)) {
            AsyncTask.execute(new Runnable() {
                @Override
                public void run() {
                    cleanCache();
                    loadIds(params.LoadCount);
                    mInitialized.set(true);
                    mTranslator = new EMTranslator(params);
                    mLanguageList = mTranslator.getSupportedLanguages();
                }
            });
        }
    }

    /**
     * \~chinese
     * 初始化是否完成。
     *
     * @return boolean - `true`：是；
     * - `false`：否。
     *
     * \~english
     * Whether the initialization is completed.
     *
     * @return boolean - `true`: Yes; 
     * - `false`: No.
     */
    public boolean isInitialized() {
        return mInitialized.get();
    }

    /**
     * \~chinese
     * 获取支持翻译的语言列表。
     *
     * @return 支持翻译的语言列表。
     *
     * \~english
     * Gets the list of supported languages for translation.
     *
     * @return The list of supported languages for translation.
     */
    public List<EMLanguage> getSupportedLanguages() {
        if(!mInitialized.get())
            return new ArrayList<>();

        return mLanguageList;
    }

    /**
     * \~chinese
     * 翻译文本。
     *
     * @param messageId 消息 ID。
     * @param conversationId 会话 ID。
     * @param messageText 需要翻译的文本。
     * @param targetLanguageCode 微软翻译目标语言对应的 code。
     * 
     * @return 翻译记录。
     *
     * \~english
     * Translates the text.
     *
     * @param messageId The message ID.
     * @param conversationId The conversation ID.
     * @param messageText The text to be translated.
     * @param targetLanguageCode The code of the target language in Microsoft Translation Service.
     * 
     * @return The translation record.
     */
    public void translate(String messageId, String conversationId, String messageText, String targetLanguageCode, EMValueCallBack<EMTranslationResult> callback) {
        EMTranslationResult result;

        if(! mInitializing.get()) {
            callback.onError(EMError.TRANSLATE_NOT_INIT, "EMTranslationManager is not initialized");
            return;
        }

        if(messageText.length() > MaxTranslationTextSize) {
            callback.onError(EMError.TRANSLATE_INVALID_PARAMS, "Text exceeds limit");
            return;
        }

        if(isTranslationResultForMessage(messageId)) {
            result = getTranslationResult(messageId);
        }else{
            result = new EMTranslationResult(messageId);
            result.setConversationId(conversationId);
        }

        mTranslator.translate(messageText, targetLanguageCode, (TranslationText, ErrorText) -> {
            if (!TranslationText.isEmpty()) {
                result.setShowTranslation(true);
                result.setTranslatedText(TranslationText);
                result.setTranslateCount(result.translateCount() + 1);
                updateTranslationResult(result);
                callback.onSuccess(result);
            }else{
                callback.onError(EMError.TRANSLATE_FAIL, ErrorText);
            }
        });
    }

    /**
     * \~chinese
     * 根据消息 ID 删除翻译记录。
     *
     * @param messageId 消息 ID。
     *
     * \~english
     * Deletes translation records by the message ID.
     *
     * @param messageId The message ID.
     */
    public void removeTranslationResult(String messageId) {
        removeTranslationResults(Arrays.asList(messageId));
    }

    /**
     * \~chinese
     * 根据消息 ID 批量删除翻译记录。
     *
     * @param messageIds 消息 ID 数组。
     *
     * \~english
     * Batch delete translation records by the message ID.
     *
     * @param messageIds The message ID array.
     */
    public void removeTranslationResults(List<String> messageIds) {
        if(! mInitializing.get())
            return;

        deleteTranslationResults(messageIds);
    }

    /**
     * \~chinese
     * 删除所有的翻译记录。
     *
     * \~english
     * Deletes all translation records.
     *
     */
    public void clearTranslations() {
        if(! mInitializing.get())
            return;

        deleteAll();
    }

    /**
     * \~chinese
     * 根据会话 ID 删除翻译记录。
     *
     * @param conversationId 会话 ID。
     *
     * \~english
     * Deletes translation records by the conversation ID.
     *
     * @param conversationId The conversation ID.
     */
    public void removeResultsByConversationId(String conversationId) {
        if(! mInitializing.get())
            return;
        removeTranslationsByConversationId(conversationId);
        mCacheManager.removeByConversationId(conversationId);
    }

    /**
     * \~chinese
     * 根据消息 ID 获取翻译记录。
     *
     * @param messageId 消息 ID。
     *
     * \~english
     * Gets translation records by message ID.
     *
     * @param messageId The message ID.
     */
    public EMTranslationResult getTranslationResult(String messageId) {
        if(! mInitializing.get())
            return null;

        if(!mCacheManager.check(messageId))
            return null;

        if(mCacheManager.get(messageId) != null)
            return mCacheManager.get(messageId);

        EMTranslationResult result =  getTranslationResultByMsgId(messageId);

        return result;
    }

    /**
     * \~chinese
     * 判断缓存里是否有翻译记录。
     *
     * @param messageId 消息 ID。
     *
     * \~english
     * Checks whether there are translation records in the cache.
     *
     * @param messageId The message ID.
     */
    public boolean isTranslationResultForMessage(String messageId) {
        if(!mInitialized.get())
            return false;

        return isMessageResult(messageId);
    }

    /**
     * \~chinese
     * 重置参数，清空缓存。
     *
     * \~english
     * Resets parameters to clear the cache.
     */
    public void logout() {
        if(!mInitialized.get())
            return;

        mInitializing.set(false);
        mInitialized.set(false);

        cleanCache();
        mLanguageList.clear();
    }

    private void loadIds(int loadCount) {
        List<EMTranslationResult> results = loadTranslateResults(loadCount);
        if(results.size() > 0) {
            for(EMTranslationResult result : results) {
                mCacheManager.add(result);
            }
        }
    }

    /**
     * \~chinese
     * 更新翻译记录。
     *
     * @param result 翻译记录对象。
     *
     * \~english
     * Updates the translation records.
     *
     * @param result The translation record object.
     */
    public void updateTranslationResult(EMTranslationResult result) {
        String messageId = result.msgId();

        if(mCacheManager.check(messageId)) {
            removeTranslationResult(messageId);
        }

        mCacheManager.add(result);
        updateTranslate(result);
    }

    private void deleteTranslationResults(List<String> messageIds) {
        List<String> toBeDeletedMsgIds = new ArrayList<>();
        for(String messageId : messageIds) {
            if(mCacheManager.check(messageId))
                toBeDeletedMsgIds.add(messageId);
        }

        removeTranslationsByMsgId(toBeDeletedMsgIds);
        mCacheManager.removeByMsgIds(toBeDeletedMsgIds);
    }

    private void deleteAll() {
        removeAllTranslations();
        mCacheManager.clear();
    }

    private boolean isMessageResult(String messageId) {
        return mCacheManager.check(messageId);
    }

    private void cleanCache() {
        mCacheManager.clear();
    }

    // 调用底层实现
    private boolean updateTranslate(EMTranslationResult result){
        return emaObject.updateTranslation(result.emaObject);
    }

    private EMTranslationResult getTranslationResultByMsgId(String msgId){

        EMATranslateResult aResult = emaObject.getTranslationResultByMsgId(msgId);
        if(aResult == null){
            return new EMTranslationResult(msgId);
        }
        return new EMTranslationResult(aResult);
    }

    private List<EMTranslationResult> loadTranslateResults(int count){
        List<EMATranslateResult> list = emaObject.loadTranslateResults(count);
        List<EMTranslationResult> results = new ArrayList<>();
        if(list.size() > 0){
            for(int i = 0; i < list.size(); i++){
                EMATranslateResult aResult = list.get(i);
                EMTranslationResult result = new EMTranslationResult(aResult);
                results.add(result);
            }
            return results;
        }else {
            return results;
        }
    }

    private boolean removeTranslationsByMsgId(List<String> msgIds){
        return emaObject.removeTranslationsByMsgId(msgIds);
    }

    private boolean removeTranslationsByConversationId(String conversationId){
        return emaObject.removeTranslationsByConversationId(conversationId);
    }

    private boolean removeAllTranslations(){
        return emaObject.removeAllTranslations();
    }
}

@Deprecated
class CacheManager {
    private TranslationCache mTranslationCache;
    private Set<Integer> mIdCache;

    CacheManager(int maxCacheSize) {
        mTranslationCache =  new TranslationCache(maxCacheSize);
        mIdCache = new HashSet<Integer>();
    }

    void add(EMTranslationResult result) {
        synchronized (CacheManager.class) {
            String id = result.msgId();
            mTranslationCache.put(id, result);

            int value = id2Value(id);
            mIdCache.add(value);
        }
    }

    void removeByMsgIds(List<String> messageIds) {
        synchronized (CacheManager.class) {
            for(String messageId : messageIds) {
                mTranslationCache.remove(messageId);

                int value = id2Value(messageId);
                mIdCache.remove(value);
            }
        }
    }

    void removeByConversationId(String conversationId) {
        mTranslationCache.removeByConversationId(conversationId);
    }

    EMTranslationResult get(String messageId) {
        EMTranslationResult result;
        synchronized (CacheManager.class) {
            result = mTranslationCache.get(messageId);
        }

        return result;
    }

    boolean check(String messageId) {
        boolean isExist = false;
        synchronized (CacheManager.class) {
            int value = id2Value(messageId);
            isExist = mIdCache.contains(value);
        }

        return isExist;
    }

    void clear() {
        synchronized (CacheManager.class) {
            mTranslationCache.clear();
            mIdCache.clear();
        }
    }

    private int id2Value(String messageId) {
        //get message timestamp part and return as 32 bit int
        long value = Long.parseLong(messageId);

        value >>= 22;
        value &= 0xffffffff;

        return (int)value;
    }
}

@Deprecated
class TranslationCache extends LRUCache<String, EMTranslationResult> {
    private HashMap<String, List<String>> mConversationMap;


    public TranslationCache(int maxSize) {
        super(maxSize);
        mConversationMap = new HashMap<>();
    }

    @Override
    public void clear() {
        super.clear();
        mConversationMap.clear();
    }

    @Override
    public void remove(String key) {
        EMTranslationResult result = get(key);
        List<String> messageList = mConversationMap.get(result.conversationId());
        if(messageList != null)
            messageList.remove(key);

        super.remove(key);

    }

    @Override
    public void put(String key, EMTranslationResult value) {
        super.put(key, value);

        if(mConversationMap.get(value.conversationId()) == null)
            mConversationMap.put(value.conversationId(), new ArrayList<>());

        List<String> messageIdList = mConversationMap.get(value.conversationId());
        assert messageIdList != null;
        messageIdList.add(key);
    }

    public void removeByConversationId(String conversationId) {
        List<String> messageList = mConversationMap.get(conversationId);

        if(messageList != null) {
            for(String messageId : messageList) {
                super.remove(messageId);
            }
        }
    }
}

@Deprecated
class LRUCache<K, V>{

    // Define Node with pointers to the previous and next items and a key, value pair
    class Node<T, U> {
        Node<T, U> previous;
        Node<T, U> next;
        T key;
        U value;

        public Node(Node<T, U> previous, Node<T, U> next, T key, U value){
            this.previous = previous;
            this.next = next;
            this.key = key;
            this.value = value;
        }
    }

    private HashMap<K, Node<K, V>> cache;
    private Node<K, V> leastRecentlyUsed;
    private Node<K, V> mostRecentlyUsed;
    private int maxSize;
    private int currentSize;

    public LRUCache(int maxSize){
        this.maxSize = maxSize > 1 ? maxSize : 2;;
        this.currentSize = 0;
        leastRecentlyUsed = new Node<K, V>(null, null, null, null);
        mostRecentlyUsed = leastRecentlyUsed;
        cache = new HashMap<K, Node<K, V>>();
    }

    public void clear() {
        this.currentSize = 0;
        leastRecentlyUsed = new Node<K, V>(null, null, null, null);
        mostRecentlyUsed = leastRecentlyUsed;
        cache.clear();
    }

    public void remove(K key) {
        Node<K, V> tempNode = cache.get(key);
        if(tempNode == null)
            return;

        cache.remove(key);
        currentSize--;

        if(tempNode.key == mostRecentlyUsed.key) {
            mostRecentlyUsed = mostRecentlyUsed.previous;
            if(mostRecentlyUsed != null)
                mostRecentlyUsed.next = null;
            else
                leastRecentlyUsed = null;
        } else if (tempNode.key == leastRecentlyUsed.key) {
            leastRecentlyUsed = leastRecentlyUsed.next;
            if(leastRecentlyUsed != null)
                leastRecentlyUsed.previous = null;
            else
                mostRecentlyUsed = null;
        } else {
            tempNode.next.previous = tempNode.previous;
            tempNode.previous.next = tempNode.next;
        }
    }

    public boolean contains(K key) {
        return cache.containsKey(key);
    }

    public V get(K key){
        Node<K, V> tempNode = cache.get(key);
        if (tempNode == null){
            return null;
        } else if (tempNode.key == mostRecentlyUsed.key){
            // If MRU leave the list as it is
            return mostRecentlyUsed.value;
        }

        // Get the next and previous nodes
        Node<K, V> nextNode = tempNode.next;
        Node<K, V> previousNode = tempNode.previous;

        if (tempNode.key == leastRecentlyUsed.key){
            // If at the left-most, we update LRU
            nextNode.previous = null;
            leastRecentlyUsed = nextNode;
        } else if (tempNode.key != mostRecentlyUsed.key){
            // If we are in the middle, we need to update the items before and after our item
            previousNode.next = nextNode;
            nextNode.previous = previousNode;
        }

        // Finally move our item to the MRU
        tempNode.previous = mostRecentlyUsed;
        mostRecentlyUsed.next = tempNode;
        mostRecentlyUsed = tempNode;
        mostRecentlyUsed.next = null;

        return tempNode.value;

    }

    public void put(K key, V value){
        if (cache.containsKey(key)){
            return;
        }

        // Put the new node at the right-most end of the linked-list
        Node<K, V> myNode = new Node<K, V>(mostRecentlyUsed, null, key, value);
        mostRecentlyUsed.next = myNode;
        cache.put(key, myNode);
        mostRecentlyUsed = myNode;

        // Delete the left-most entry and update the LRU pointer
        if (currentSize == maxSize){
            cache.remove(leastRecentlyUsed.key);
            leastRecentlyUsed = leastRecentlyUsed.next;
            leastRecentlyUsed.previous = null;
            return;
        }

        currentSize++;
    }
}

