/************************************************************
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2013-2015 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import static com.hyphenate.chat.EMMessage.self;

import android.graphics.BitmapFactory.Options;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMConversationListener;
import com.hyphenate.EMError;
import com.hyphenate.EMMessageListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.EMConversation.EMConversationType;
import com.hyphenate.chat.EMMessage.Status;
import com.hyphenate.chat.EMMessage.Type;
import com.hyphenate.chat.adapter.EMAChatManager;
import com.hyphenate.chat.adapter.EMAChatManagerListener;
import com.hyphenate.chat.adapter.EMAConversation;
import com.hyphenate.chat.adapter.EMAConversation.EMAConversationType;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAGroupReadAck;
import com.hyphenate.chat.adapter.EMAMessageReaction;
import com.hyphenate.chat.adapter.EMAMessageReactionChange;
import com.hyphenate.chat.adapter.EMAReactionManager;
import com.hyphenate.chat.adapter.EMAReactionManagerListener;
import com.hyphenate.chat.adapter.message.EMAFileMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.core.EMAdvanceDebugManager;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.notification.core.EMNotificationHelper;
import com.hyphenate.util.EMFileHelper;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.ImageUtils;
import com.hyphenate.util.PathUtil;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * \~chinese
 * 聊天管理类，该类负责管理会话（加载，删除等）、发送消息、下载消息附件等。
 * 
 * 发送文文本消息示例如下：
 * 
 * ```java
 *	EMMessage message = EMMessage.createTxtSendMessage(content, toChatUsername);
 * 	EMClient.getInstance().chatManager().sendMessage(message);
 * ```
 *
 * \~english
 * The chat manager class, responsible for sending and receiving messages, managing conversations (including loading and deleting conversations), and downloading attachments.
 *
 * The sample code for sending a text message is as follows:
 * 
 * ```java
 * 	EMMessage message = EMMessage.createTxtSendMessage(content, toChatUsername);
 * 	EMClient.getInstance().chatManager().sendMessage(message);
 * ```
 */
public class EMChatManager {

	EMAChatManager emaObject;
	EMAReactionManager emaReactionObject;

	private static final String TAG = "EMChatManager";

	private final static String INTERNAL_ACTION_PREFIX = "em_";

	EMClient mClient;

	Map<String, EMConversation.MessageCache> caches = new Hashtable<String, EMConversation.MessageCache>();

	protected EMChatManager(){}

	protected EMChatManager(EMClient client, EMAChatManager manager, EMAReactionManager reactionManager) {
		mClient = client;

		emaObject = manager;
		emaObject.addListener(chatManagerListenerImpl);

		emaReactionObject = reactionManager;
		emaReactionObject.addListener(mReactionManagerListenerImpl);
	}

	private List<EMMessageListener> messageListeners = new CopyOnWriteArrayList<EMMessageListener>();
	private List<EMConversationListener> conversationListeners = Collections.synchronizedList(new ArrayList<EMConversationListener>());

	EMAChatManagerListener chatManagerListenerImpl = new EMAChatManagerListener() {
//		private final Object lockObj = new Object();
//
//        List<EMMessageListener> cloneSyncedList(List<EMMessageListener> list) {
//            if (list == null) {
//                return new ArrayList<>();
//            } else {
//                synchronized (lockObj) {
//                    return list.subList(0, list.size());
//                }
//            }
//        }

		@Override
		public void onReceiveMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new EMMessage(msg));
					}

					List<EMMessage> remainingMsgs = new ArrayList<EMMessage>();
					for (EMMessage msg : msgs) {
				    	/*if(msg.getChatType() == EMMessage.ChatType.ChatRoom){
                            EMChatRoom room = EMClient.getInstance().chatroomManager().getChatRoom(msg.conversationId());
                            if(room == null){
                            	continue;
                            }
				    	}*/

	                    EMConversation conv = getConversation(msg.conversationId(), EMConversation.msgType2ConversationType(msg.getFrom(), msg.getChatType()), false);
	                    if(conv == null){
	                    	EMLog.d(TAG, "no conversation");
	                    	continue;
	                    }
	                    
	                    // Command messages are not put into the cache.
	                    if (msg.getType() != Type.CMD) {
	                        conv.getCache().addMessage(msg);
	                    }
	                    remainingMsgs.add(msg);
				    }
				    
				    if(remainingMsgs.size() <= 0){
						EMLog.d(TAG, "no remainingMsgs");
				    	return;
				    }

					for (EMMessageListener l : messageListeners) {
						try { // Do not break the loop if one listener has a problem
							EMLog.d(TAG, "onMessageReceived： " + l);
							l.onMessageReceived(remainingMsgs);
						} catch (Exception e) {
							EMLog.d(TAG, "onMessageReceived has problem: " + e.getMessage());
							e.printStackTrace();
						}
					}
				}
			});
		}

		@Override
		public void onReceiveCmdMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						EMMessage message = new EMMessage(msg);
						String action = ((EMCmdMessageBody)message.getBody()).action();
						if(isAdvanceDebugMessage(action)) {
							EMAdvanceDebugManager.getInstance().handleDebugMessage(message, EMAdvanceDebugManager.Type.valueOf(action));
						} else {
							msgs.add(message);
						}
					}
                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onCmdMessageReceived(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }

				}
			});
		}

		@Override
		public void onMessageStatusChanged(final EMAMessage message, final EMAError error) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    try {
                        EMMessage msg = new EMMessage(message);
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageChanged(msg, null);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
	    public void onMessageAttachmentsStatusChanged(final EMAMessage message, final EMAError error) {
            try {
                EMMessage msg = new EMMessage(message);
                for (EMMessageListener l : messageListeners) {
                    l.onMessageChanged(msg, null);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
		}
		
		
        @Override
        public void onReceiveRecallMessages(final List<EMAMessage> messages) {
            mClient.executeOnMainQueue(new Runnable() {

                @Override
                public void run() {
                    List<EMMessage> msgs = new ArrayList<EMMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new EMMessage(msg));
                        getConversation(msg.conversationId()).getCache().removeMessage(msg.msgId());
                    }

                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageRecalled(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
        }
		@Override
		public void onReceiveHasReadAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    List<EMMessage> msgs = new ArrayList<EMMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new EMMessage(msg));
                    }

                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageRead(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveHasDeliveredAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new EMMessage(msg));
					}

                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageDelivered(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveReadAckForConversation(String fromUsername, String toUsername) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					EMLog.d(TAG, "onReceiveConversationHasReadAcks");
					synchronized (conversationListeners) {
						if(!TextUtils.equals(fromUsername, EMClient.getInstance().getCurrentUser())) {
							//1v1聊天中收到对方发送的channel ack消息，将本地的会话的发送消息置为已读（对方已读）,需要重新load数据
							EMConversation conversation = EMClient.getInstance().chatManager().getConversation(fromUsername);
							if(conversation != null) {
								conversation.loadMoreMsgFromDB(null, conversation.getAllMessages().size());
							}
						}
						try {
							for (EMConversationListener l : conversationListeners) {
								l.onConversationRead(fromUsername, toUsername);
							}
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				}
			});
		}

		@Override
		public void onUpdateConversationList(final List<EMAConversation> conversations) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					EMLog.d(TAG, "onUpdateConversationList");
                    synchronized (conversationListeners) {
	                    try {
	                        for (EMConversationListener l : conversationListeners) {
	                            l.onCoversationUpdate();
	                        }
	                    } catch (Exception e) {
		                    e.printStackTrace();
	                    }
                    }
				}
			});
		}

		@Override
		public void onReceivePrivateMessages(List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onPrivateMessages");
					for (EMAMessage msg : messages) {
						EMMessage message = new EMMessage(msg);
						EMNotificationHelper.getInstance().analyzeCmdMessage(message);
					}
				}
			});
		}

		@Override
		public void onReceiveReadAcksForGroupMessage(List<EMAGroupReadAck> acks) {
            mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onReceiveReadAcksForGroupMessage");

					List<EMGroupReadAck> groupAcks = new ArrayList<EMGroupReadAck>();
					for (EMAGroupReadAck ack : acks) {
						groupAcks.add(new EMGroupReadAck(ack));
					}

					try {
						for (EMMessageListener l : messageListeners) {
							l.onGroupMessageRead(groupAcks);
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			});
		}

		@Override
		public void onUpdateGroupAcks() {
        	mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onUpdateGroupAcks");
					try {
						for (EMMessageListener l : messageListeners) {
							l.onReadAckForGroupMessageUpdated();
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			});
		}
	};

	EMAReactionManagerListener mReactionManagerListenerImpl = new EMAReactionManagerListener() {
		@Override
		public void onMessageReactionDidChange(List<EMAMessageReactionChange> reactionChangeList) {
			mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onMessageReactionDidChange");
					List<EMMessageReactionChange> list = new ArrayList<>(reactionChangeList.size());
					EMMessageReactionChange reactionChange;
					for (EMAMessageReactionChange emaReactionChange : reactionChangeList) {
						reactionChange = new EMMessageReactionChange(emaReactionChange);
						list.add(reactionChange);
					}
					for (EMMessageListener l : messageListeners) {
						try {
							l.onReactionChanged(list);
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				}
			});

		}
	};

	private boolean isAdvanceDebugMessage(String action){
		if(action.startsWith(INTERNAL_ACTION_PREFIX)){
			try {
				EMAdvanceDebugManager.Type.valueOf(action);
				return true;
			} catch (Exception e) {
				//if it is not a field of Type, throws exception
				e.printStackTrace();
			}
		}
		return false;
	}

	/**
	 * \~chinese
	 * 发送消息。
	 *
	 * 如果是语音，图片类等有附件的消息，SDK 会自动上传附件。
	 *
	 * 可以通过 {@link EMOptions#setAutoTransferMessageAttachments(boolean)} 设置是否上传到聊天服务器。
	 * 
	 * 发送消息的状态，可以通过设置 {@link EMMessage#setMessageStatusCallback(EMCallBack)} 进行监听。
	 * 
	 * @param msg    要发送的消息，必填。
	 *
	 * \~english
	 * Sends a message。
	 *  
	 * For a voice or image message or a message with an attachment, the SDK will automatically upload the attachment.
	 *
	 * You can determine whether to upload the attachment to the chat sever by setting {@link EMOptions#setAutoTransferMessageAttachments(boolean)}.
	 *
	 * To listen for the message sending status, call {@link EMMessage#setMessageStatusCallback(EMCallBack)}.
	 * 
	 * @param msg    The message object to be sent. Make sure to set the param. 
	 */
	public void sendMessage(final EMMessage msg) {
		msg.makeCallbackStrong();

		boolean createConv = msg.getType() != Type.CMD;
		final EMConversation conv = getConversation(msg.conversationId(), EMConversation.msgType2ConversationType(msg.getTo(), msg.getChatType()), createConv, msg.isChatThreadMessage());

		//Add the message at first
		if (conv != null) {
			boolean exists = conv.getCache().getMessage(msg.getMsgId()) != null;
			if (!exists) {
				// The message sending time is lastMsgTime + 1 ms
				long lastMsgTime = System.currentTimeMillis();
				EMMessage lastMsg = conv.getLastMessage();
				if (lastMsg != null) {
					lastMsgTime = lastMsgTime < lastMsg.getMsgTime() ? lastMsg.getMsgTime() : lastMsgTime;
				}
				msg.setMsgTime(lastMsgTime + 1);

				conv.getCache().addMessage(msg);
			}
		}

		class HandleError {
			HandleError(int code, String desc) {
				EMMessage.EMCallbackHolder holder = msg.messageStatusCallBack;
				if (holder != null) {
					holder.onError(code, desc);
				}
			}
		}

    mClient.executeOnSendQueue(new Runnable() {
            
        @Override
        public void run() {
				try {
					String originPath = null;
					// If do not use Chat Server, do not deal with the logic of pictures and videos
					if(mClient.getOptions().getAutoTransferMessageAttachments()) {
						// If message body is image, check scale request, set size, and file name
						if (msg.getType() == Type.IMAGE) {
							//the default status is fail status,
							//which lead to message show fail and then show success in need scale image
							msg.setStatus(Status.INPROGRESS);
							EMImageMessageBody body = (EMImageMessageBody) msg.getBody();
							if (body == null) {
								new HandleError(EMError.GENERAL_ERROR, "Message body can not be null");
								return;
							}

							String localUri = body.getLocalUrl();
							String remoteUrl = body.getRemoteUrl();
							if (TextUtils.isEmpty(remoteUrl)){
								if(!EMFileHelper.getInstance().isFileExist(localUri)) {
									new HandleError(EMError.FILE_INVALID, "File not exists or can not be read");
									return;
								}
							}
							if(!body.isSendOriginalImage() && EMFileHelper.getInstance().isFileExist(localUri)) {
								String scaledImagePath = ImageUtils.getScaledImageByUri(mClient.getContext(), localUri);
								if(!TextUtils.equals(scaledImagePath, localUri)) {
									originPath = localUri;
									long originalSize = EMFileHelper.getInstance().getFileLength(localUri);
									long scaledSize = EMFileHelper.getInstance().getFileLength(scaledImagePath);
									if (originalSize == 0) {
										EMLog.d(TAG, "original image size:" + originalSize);
										new HandleError(EMError.FILE_INVALID, "original image size is 0");
										return;
									}
									EMLog.d(TAG, "original image size:" + originalSize + " scaled image size:" + scaledSize
											+ " ratio:" + (int) (scaledSize / originalSize) + "%");
									localUri = scaledImagePath;
									body.setLocalUrl(EMFileHelper.getInstance().formatInUri(localUri));
									body.setFileLength(scaledSize);
								}
								body.setFileName(EMFileHelper.getInstance().getFilename(localUri));
							}
							// get image width and height
							Options options = ImageUtils.getBitmapOptions(mClient.getContext(), localUri);
							if(options != null) {
								int width = options.outWidth;
								int height = options.outHeight;
								body.setSize(width, height);
							}
						}else if (msg.getType() == Type.VIDEO){
							msg.setStatus(Status.INPROGRESS);
							EMVideoMessageBody body = (EMVideoMessageBody) msg.getBody();
							if (body == null) {
								new HandleError(EMError.GENERAL_ERROR, "Message body can not be null");
								return;
							}
							Uri filePathUri = body.getLocalUri();
							String fileRemoteUrl = body.getRemoteUrl();
							if (TextUtils.isEmpty(fileRemoteUrl)){
								if(!EMFileHelper.getInstance().isFileExist(filePathUri)) {
									new HandleError(EMError.FILE_INVALID, "File not exists or can not be read");
									return;
								}
							}
							// get video width and height
							String thumbPath = EMFileHelper.getInstance().getFilePath(body.getLocalThumbUri());
							if(!TextUtils.isEmpty(thumbPath)) {
								Options options = ImageUtils.getBitmapOptions(thumbPath);
								int width = options.outWidth;
								int height = options.outHeight;
								body.setThumbnailSize(width, height);
							}
						}
					}

					String oldId = msg.getMsgId();
					//set callback to replace old id
					setMessageSendCallback(msg, conv, oldId, originPath);
					emaObject.sendMessage(msg.emaObject);
				} catch (Exception e) {
					e.printStackTrace();
					new HandleError(EMError.GENERAL_ERROR, "send message failed");
				}
			}
		});
	}

	private void setMessageSendCallback(final EMMessage msg,
										final EMConversation conv,
										final String oldId,
										final String originImagePath) {
		if (msg == null) {
			return;
		}

		msg.setInnerCallback(new EMCallBack(){

			@Override
			public void onSuccess() {
				if (conv != null) {
					conv.getCache().removeMessage(oldId);
					conv.getCache().addMessage(msg);
				}
				if (originImagePath != null){
					if(!EMFileHelper.getInstance().isFileExist(originImagePath)) {
					    return;
					}
					if (msg.getBody() instanceof EMImageMessageBody) {
						String scaleImagePath = ((EMImageMessageBody)msg.getBody()).getLocalUrl();
						EMLog.d(TAG, "origin: + " + originImagePath + ", scale:" + scaleImagePath);
						//if scaleImagePath is not origin image path, should delete scale image file
						if(scaleImagePath != null && !scaleImagePath.equals(originImagePath)){
							boolean isDeleted = EMFileHelper.getInstance().deletePrivateFile(scaleImagePath);
							EMLog.d(TAG, "Deleted the scale image file: "+ isDeleted + " the scale image file path: "+scaleImagePath);
						}
						((EMImageMessageBody)msg.getBody()).setLocalUrl(originImagePath);
						updateMessage(msg);
					}
				}
			}

			@Override
			public void onProgress(int progress, String status) {
			}

			@Override
			public void onError(int code, String error) {
			}
		});
	}

	/**
	 * \~chinese
	 * 发送会话的已读回执，该方法只针对单聊会话。
	 * 
	 * 该方法会通知服务器将此会话未读数设置为 0，消息发送方将会收到 {@link EMConversationListener#onConversationRead(String, String)} 回调。
	 * 在多端多设备登录下，接收方的其他设备也会收到该回调。
	 *
	 * 群消息已读回执，详见 {@link #ackGroupMessageRead(String, String, String)}。
	 * 
	 *
	 * @param conversationId			会话 ID。
	 * @throws HyphenateException		可能抛出以下异常：{@link EMError#USER_NOT_LOGIN}、{@link EMError#SERVER_NOT_REACHABLE}、
	 * 									{@link EMError#MESSAGE_INVALID} 等，详见 {@link EMError}。
	 *
	 * \~english
	 * Sends the conversation read receipt to the server.
	 * 
	 * This method is only for one-to-one chat conversations.
	 *
	 * This method will inform the sever to set the unread message count of the conversation to `0`.
	 * The message sender will receive the {@link EMConversationListener#onConversationRead(String, String)} callback.
	 * The message recipient that has logged in to multiple devices, will also receive the callback.
	 *
	 * If you want to send a read receipt for a group message, call {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * @param conversationId			The conversation ID.
	 * @throws HyphenateException	    The possible exceptions are as follows: {@link EMError#USER_NOT_LOGIN}, {@link EMError#SERVER_NOT_REACHABLE}, and
	 * 									{@link EMError#MESSAGE_INVALID}. See {@link EMError}.
	 */
	public void ackConversationRead(String conversationId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.sendReadAckForConversation(conversationId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 发送消息的已读回执。
	 * 
	 * 该方法仅适用于单聊会话，仅在 {@link EMOptions#setRequireAck(boolean)} 为 `true` 时生效。
	 *
	 * 发送群消息已读回执，详见 {@link #ackGroupMessageRead(String, String, String)}。
	 *
	 * 推荐进入聊天页面的时调用 {@link #ackConversationRead(String)} ，其他情况下调用该方法以减少调用频率。
	 *
	 * @param to					接收方的用户 ID。
	 * @param messageId				消息的 ID。
	 * @throws HyphenateException 	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Sends the read receipt for a message to the server.
	 *
	 * **Note**
	 * 
	 * This method applies to one-to-one chats only.
	 * 
	 * This method only takes effect if you set {@link EMOptions#setRequireAck(boolean)} as `true`.
	 *
	 * To send the read receipt for a group message, call {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * We recommend that you call {@link #ackConversationRead(String)} when opening the chat page, and call this method in other cases to reduce the number of method calls.
	 *
	 * @param to			The message for which the read receipt is to be sent.
	 * @param messageId		The message ID.
	 * @throws HyphenateException  A description of the exception, see  {@link EMError}.
	 */
	public void ackMessageRead(String to, String messageId) throws HyphenateException {
		EMOptions chatOptions = EMClient.getInstance().getChatConfigPrivate().getOptions();
		if (!chatOptions.getRequireAck()) {
			EMLog.d(TAG, "As the chat option SetRequireAck is set to false, the read receipt is not sent.");
			return;
		}
		if(TextUtils.isEmpty(to)) {
			EMLog.e(TAG, "The to parameter cannot be null.");
		    return;
		}
		EMAMessage msg = emaObject.getMessage(messageId);
		if (msg != null) {
			emaObject.sendReadAckForMessage(msg);
		} else { // just for Xinju since there is no local storage
			EMAMessage _msg = EMAMessage.createReceiveMessage("", self(), null, EMMessage.ChatType.Chat.ordinal());
			_msg.setMsgId(messageId);
			_msg.setFrom(to);
			//set conversationId because native code need it
			_msg.setConversationId(to);

			emaObject.sendReadAckForMessage(_msg);
		}
	}

	/**
	 * \~chinese
	 * 发送群消息已读回执。
	 * 
	 * **注意**
	 *
	 * 设置了 {@link EMOptions#setRequireAck(boolean)} 和 {@link EMMessage#setIsNeedGroupAck(boolean)} 后才能调用该方法。
	 *
	 * 发送单聊消息已读回执，详见 {@link #ackMessageRead(String, String)}
	 *
	 * 会话已读回执，详见 {@link #ackConversationRead(String)}。
	 *
	 * @param to					会话 ID。
	 * @param messageId				消息 ID。
	 * @param ext					扩展信息。用户通过定义关键字指定动作或命令。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Sends the read receipt for a group message to the server.
	 * 
	 * **Note**
	 *
	 * You can only call this method after setting {@link EMOptions#setRequireAck(boolean)} and {@link EMMessage#setIsNeedGroupAck(boolean)} to `true`.
	 * 
	 * To send the read recipient for a one-to-one chat message to the server, call {@link #ackMessageRead(String, String)}.

	 * To send the conversation read receipt to the server, call {@link #ackConversationRead(String)}.
	 *
	 * @param to					The conversation ID.
	 * @param messageId				The message ID.
	 * @param ext					The extension information, which is a custom keyword that specifies a custom action or command.
	 * @throws HyphenateException	A description of the exception, see {@link EMError}.
	 */
	public void ackGroupMessageRead(String to, String messageId, String ext) throws HyphenateException {
		EMOptions chatOptions = EMClient.getInstance().getChatConfigPrivate().getOptions();
		if (!chatOptions.getRequireAck()) {
			EMLog.d(TAG, "chat option reqire ack set to false. skip send out ask msg read");
			return;
		}
		EMAMessage msg = emaObject.getMessage(messageId);
		if (msg != null) {
			if (msg.isNeedGroupAck()) {
				emaObject.sendReadAckForGroupMessage(msg, ext);
			} else {
				EMLog.d(TAG, "normal group message, do not ack it");
			}
		}
	}
	
    /**
     * \~chinese
     * 撤回发送成功的消息。
	 *
	 * 若消息发送和撤回时接收方为离线状态，接收方会收到消息撤回回调 {@link EMMessageListener#onMessageRecalled}，不会收到该消息。
     *
	 * 同步方法，会阻塞当前线程。
	 *
     * @param message 消息对象。
	 *
	 * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Recalls the sent message.
	 *
	 * If the recipient is offline when the message is delivered and recalled, the recipient only receives the callback {@link EMMessageListener#onMessageRecalled} instead of the message.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param message The message instance.
	 *
	 * @throws HyphenateException A description of the exception. See {@link EMError}.
     */
    public void recallMessage(EMMessage message) throws HyphenateException {
        EMAError error = new EMAError();
        if (message == null) {
            throw new HyphenateException(EMError.MESSAGE_INVALID, "The message was not found");
        }
        emaObject.recallMessage(message.emaObject, error);
        handleError(error);
        // 指定是否是Thread消息
		EMConversation conversation = getConversation(message.getTo(), EMConversation.msgType2ConversationType(message.getMsgId(), message.getChatType()), true, message.isChatThreadMessage());
		if(conversation != null) {
		    conversation.getCache().removeMessage(message.getMsgId());
		}
    }

	/**
	 * \~chinese
	 * 撤回发送成功的消息。
	 *
	 * 异步方法。
	 *
	 * @param message 	消息对象。
	 * @param callback	EMCallBack 类的对象，详见 {@link EMCallBack}。
	 *
	 * \~english
	 * Recalls the sent message.
	 *
	 * This is an asynchronous method.
	 *
	 * @param message	The message object.
	 * @param callback  A object of the EMCallBack class. See {@link EMCallBack}.
	 */
	public void aysncRecallMessage(final EMMessage message,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    recallMessage(message);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 获取指定 ID 的消息对象。
	 *
	 * @param messageId		消息 ID。
	 * @return  根据指定 ID 获取的消息对象，如果消息不存在会返回空值。
	 *
	 * \~english
	 * Gets the message by message ID.
	 *
	 * @param messageId 	The message ID.
	 * @return  The message object with the specified ID. The SDK returns `null` if the message does not exist.
	 */
	public EMMessage getMessage(String messageId) {
		synchronized (caches) {
			for(EMConversation.MessageCache cache : caches.values()) {
				EMMessage msg = cache.getMessage(messageId);
				if (msg != null) {
					return msg;
				}
			}
		}
		EMAMessage message =  emaObject.getMessage(messageId);
		if (message == null) {
			return null;
		}
		EMMessage msg = new EMMessage(message);
		return msg;
	}


	/**
	 * \~chinese
	 * 获取指定 ID 的会话对象。
	 * 
	 * @param id   会话 ID。
	 * @return     根据指定会话 ID 找到的会话对象，如果没有找到会返回空值。
	 *
	 * \~english
	 * Gets the conversation object by conversation ID.
	 * 
	 * The SDK will return `null` if the conversation is not found.
	 *
	 * @param id 	The conversation ID.
	 * @return 		The conversation with the specified ID. The SDK returns `null` if the conversation is not found.
	 */
	public EMConversation getConversation(String id){
		EMAConversation conversation = emaObject.conversationWithType(id, EMAConversationType.CHAT, false, false);
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.GROUPCHAT, false, false);
		}
		// 判断完群组会话后，再检查Thread会话
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.GROUPCHAT, false, true);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.CHATROOM, false, false);
		}
		// 判断完聊天室会话后，再检查Thread会话
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.CHATROOM, false, true);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.DISCUSSIONGROUP, false, false);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.HELPDESK, false, false);
		}

		return conversation == null ? null : new EMConversation(conversation);
	}

	/**
	 * \~chinese
	 * 根据会话 ID 以及会话类型获取会话。
	 * 
	 * 没有找到返回空值。
	 *
	 * @param id 		会话 ID。
	 *                  - 单聊：对端用户的用户 ID；
	 *                  - 群聊：群组 ID；
	 *                  - 聊天室：聊天室 ID；
	 *                  - 子区：子区 ID。
	 * @param type 		会话类型，详见 {@link EMConversationType}。
	 * @return          根据指定 ID 以及会话类型找到的会话对象，未找到会返回空值。
	 *
	 * \~english
	 * Gets the conversation by conversation ID and type.
	 * 
	 * The SDK will return `null` if the conversation is not found.
	 *
	 * @param id 		The conversation ID:
	 *                  - One-to-one chat: The ID of the peer user;
	 *                  - Group chat: The group ID;
	 *                  - Chat room chat: The chat room ID;
	 *                  - Message thread: The message thread ID.
	 * @param type  	The conversation type. See {@link EMConversationType}
	 * @return 			The conversation object found by ID and type. The SDK returns `null` if the conversation is not found.
	 */
	public EMConversation getConversation(String id, EMConversationType type) {
		EMConversation conversation = getConversation(id, type, false);
		// 先检查非Thread会话，再检查Thread会话
		if(conversation == null) {
		    return getConversation(id, type, false, true);
		}
		return conversation;
	}

	/**
	 * \~chinese
	 * 根据会话 ID 和会话类型获取会话。
	 *
	 * @param username 			会话 ID。
	 *                          - 单聊：对端用户的用户 ID；
	 *                          - 群聊：群组 ID；
	 *                          - 聊天室：聊天室 ID；
	 *                          - 子区：子区 ID。
	 * @param type 				会话类型，详见 {@link EMConversationType}。
	 * @param createIfNotExists 未找到指定会话时是否创建一个新会话。
	 *                          - `true`：是；
	 *                          - `false`：否。
	 * @return					找到的会话对象，如果未找到，返回空值。
	 *
	 * \~english
	 * Gets the conversation object by user ID or group ID and conversation type.
	 *
	 * @param username 			The conversation ID:
	 *                          - One-to-one chat: The ID of the peer user;
	 *                          - Group chat: The group ID;
	 *                          - Chat room chat: The chat room ID;
	 *                          - Message thread: The message thread ID.
	 * @param type 				The conversation type. See {@link EMConversationType}.
	 * @param createIfNotExists Whether to create a conversation if the specified conversation is not found:
	 *                          -  `true`: Yes;
	 *                          -  `false`: No.
	 * @return					The retrieved conversation object. The SDK returns `null` if the conversation is not found.
	 */
	public EMConversation getConversation(String username, EMConversationType type, boolean createIfNotExists) {
		EMConversation conversation = getConversation(username, type, createIfNotExists, false);
		// 先检查非Thread会话，再检查Thread会话
		if(conversation == null) {
		    return getConversation(username, type, createIfNotExists, true);
		}
		return conversation;
	}

	/**
	 * \~chinese
	 * 根据会话 ID 以及会话类型获取会话。
	 *
	 *
	 * @param username 			会话 ID。
	 *                          - 单聊：对端用户的用户 ID；
	 *                          - 群聊：群组 ID；
	 *                          - 聊天室：聊天室 ID；
	 *                          - 子区：子区 ID。
	 * @param type 				会话类型，详见 {@link EMConversationType}。
	 * @param createIfNotExists 未找到指定会话时是否创建一个新会话。
	 *                          - `true`：是；
	 * 							- `false`：否。
	 * @param isChatThread 		是否查找子区会话。
	 *                          - `true`：是；
	 * 							- `false`：否。
	 * @return					找到的会话对象。如果未找到，返回空值。
	 *
	 * \~english
	 * Get conversation object by conversation id and conversation type.
	 *
	 * If the conversation is not found, you can create a new object based on the value of CreateIFNotExists or an empty object.
	 *
	 * @param username 			The conversation ID.
	 * *                        - One-to-one chat: The ID of the peer user;
	 *                          - Group chat: The group ID;
	 *                          - Chat room chat: The chat room ID;
	 *                          - Message thread: The message thread ID.
	 * @param type 				The conversation type. See {@link EMConversationType}.
	 * @param createIfNotExists Whether to create a conversation if the specified conversation is not found:
	 *                          - `true`: Yes;
	 *                          -  `false`: No.
	 * @param isChatThread 		Whether to search for the chat thread conversation.
	 *                          - `true`: Yes;
	 * 							- `false`: No.
	 * @return					The retrieved conversation object. The SDK returns `null` if the conversation is not found.
	 */
	public EMConversation getConversation(String username, EMConversationType type, boolean createIfNotExists, boolean isChatThread) {
		EMAConversationType t = EMAConversationType.CHAT;
		switch (type) {
			case Chat:
				t = EMAConversationType.CHAT;
		        break;
			case GroupChat:
				t = EMAConversationType.GROUPCHAT;
		        break;
			case ChatRoom:
				t = EMAConversationType.CHATROOM;
		        break;
			case DiscussionGroup:
				t = EMAConversationType.DISCUSSIONGROUP;
		        break;
			case HelpDesk:
				t = EMAConversationType.HELPDESK;
		        break;
		}
		EMAConversation conversation = emaObject.conversationWithType(username, t, createIfNotExists, isChatThread);
		if (conversation == null) {
			return null;
		}
		Log.d(TAG, "convID:" + conversation.conversationId());
		return new EMConversation(conversation);
	}


	/**
	 * \~chinese
	 * 将所有会话都设成已读。
	 * 
	 * 该方法仅适用于本地会话。
	 *
	 * \~english
	 * Marks all conversations as read.
	 * 
	 * This method is for the local conversations only.
	 */
	public void markAllConversationsAsRead() {
		List<EMAConversation> conversations = emaObject.loadAllConversationsFromDB();
		for (EMAConversation conversation : conversations) {
			conversation.markAllMessagesAsRead(true);
		}
	}

	/**
	 * \~chinese
	 * 获取未读消息数。
	 *
	 * @return	未读消息数。
	 *
	 * \~english
	 * Gets the unread message count.
	 *
	 * @return	The count of unread messages.
	 */
	public int getUnreadMessageCount()
	{
		List<EMAConversation> conversations = emaObject.getConversations();
		int unreadCount = 0;
		for (EMAConversation conversation : conversations) {
			if (conversation._getType() != EMAConversationType.CHATROOM) {
				unreadCount += conversation.unreadMessagesCount();
			}
		}
		return unreadCount;
	}

	/**
	 * \~chinese
	 * 保存消息到内存和本地数据库。
	 * 
	 * 命令消息（类型见：{@link Type#CMD}，消息体见：{@link EMCmdMessageBody}）不会在本地保存。
	 *
	 * @param message	待存储的消息。
	 *
	 * \~english
	 * Saves the message to the memory and local database.
	 * 
	 * The command messages (type: {@link Type#CMD}, message body: {@link EMCmdMessageBody}) are not saved locally.
	 *
	 * @param message	The message to store.
	 */
	public void saveMessage(EMMessage message) {
		EMMessage.ChatType type = message.getChatType();
		EMConversationType t = EMConversationType.Chat;
		switch (type) {
			case Chat:
				t = EMConversationType.Chat;
				break;
			case GroupChat:
				t = EMConversationType.GroupChat;
				break;
			case ChatRoom:
				t = EMConversationType.ChatRoom;
				break;
		}
		String convId = message.getTo();
		//for group, chatroom, conversation id is group id for both receive and send message
		if (t == EMConversationType.Chat && message.direct() == EMMessage.Direct.RECEIVE) {
			convId = message.getFrom();
		}
		if (message.getType() == Type.CMD) {
			return;
		}
		EMConversation conv = getConversation(convId, t, true, message.isChatThreadMessage());
		if(conv == null) {
			EMLog.e(TAG, "Failed to save message because conversation is null, convId: "+convId);
		    return;
		}
		// when send message out, appendMessage will update time to lastMsgTime + 1ms
		conv.insertMessage(message);
	}

	/**
	 * \~chinese
	 * 更新本地消息。
	 * 
	 * 该方法会同时更新本地内存和数据库中的消息。
	 *
	 * @param message 要更新的消息对象。
	 *
	 * \~english
	 * Updates the local message.
	 * 
	 * This method updates the message in both the memory and the local database at the same time.
	 *
	 * @param message The message object to update.
	 */
	public boolean updateMessage(EMMessage message) {
		String id = message.direct() == EMMessage.Direct.RECEIVE ? message.getFrom() : message.getTo();
		if (message.getType() == Type.CMD) {
			return false;
		}
		EMConversation conv = getConversation(message.conversationId(), EMConversation.msgType2ConversationType(id, message.getChatType()), true, message.isChatThreadMessage());
		return conv.updateMessage(message);
	}

	/**
	 * \~chinese
	 * 下载消息的附件。
	 * 
	 * 若附件自动下载失败，也可以调用此方法下载。
	 *
	 * @param msg	要下载附件的消息。
	 *
	 * \~english
	 * Downloads the message attachment.
	 * 
	 * You can also call this method if the attachment fails to be downloaded automatically.
	 *
	 * @param msg 	The ID of the message with the attachment to be downloaded.
	 */
	public void downloadAttachment(final EMMessage msg) {
        if (msg == null) {
            return;
        }
        msg.makeCallbackStrong();
		checkContentAttachmentExist(msg);
		emaObject.downloadMessageAttachments(msg.emaObject);
	}

	/**
	 * \~chinese
	 * 检查消息是否存在附件。
	 *
	 * @param msg  消息对象。
	 * 
	 * \~english
	 * Checks whether the attachment exists in the message.
	 * 
	 * @param msg  The message object.
	 */
	private void checkContentAttachmentExist(EMMessage msg) {
		List<EMAMessageBody> bodies = msg.emaObject.bodies();
		if(bodies != null && !bodies.isEmpty()) {
		    for (EMAMessageBody body: bodies) {
		    	switch (body.type()) {
		    	    case EMAMessageBody.EMAMessageBodyType_IMAGE :
		    	    case EMAMessageBody.EMAMessageBodyType_VIDEO :
		    	    case EMAMessageBody.EMAMessageBodyType_VOICE :
		    	    case EMAMessageBody.EMAMessageBodyType_FILE :
						EMAFileMessageBody fileBody = (EMAFileMessageBody) body;
						String localUrl = fileBody.getLocalUrl();
						EMLog.d(TAG, "download before check path = "+localUrl);
						if(!EMFileHelper.getInstance().isFileExist(localUrl)) {
							String filename = fileBody.displayName();
							String newLocalPath = null;
							switch (body.type()) {
							    case EMAMessageBody.EMAMessageBodyType_IMAGE :
									newLocalPath = PathUtil.getInstance().getImagePath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_VIDEO :
									newLocalPath = PathUtil.getInstance().getVideoPath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_VOICE :
									newLocalPath = PathUtil.getInstance().getVoicePath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_FILE :
									newLocalPath = PathUtil.getInstance().getFilePath()+File.separator+filename;
							        break;
							}
							if(!TextUtils.isEmpty(newLocalPath)) {
							    fileBody.setLocalPath(newLocalPath);
							    updateMessage(msg);
								EMLog.d(TAG, "download:create new path , path is "+newLocalPath);
							}
						}
						break;
		    	}
			}
		}
	}

	/**
	 * \~chinese
	 * 下载消息的缩略图。
	 *
	 * @param msg	要下载缩略图的消息。只有图片消息和视频消息有缩略图。
	 *
	 * \~english
	 * Downloads the message thumbnail.
	 *
	 * @param msg  The message with the thumbnail to be downloaded. Only the image messages and video messages have a thumbnail.
	 */
	public void downloadThumbnail(final EMMessage msg) {
        msg.makeCallbackStrong();
		emaObject.downloadMessageThumbnail(msg.emaObject);
	}

	/**
	 * \~chinese
	 * 将消息导入本地数据库。
	 *
	 * 你只能将你发送或接收的消息导入本地数据库。
	 *
	 * 推荐一次导入 1,000 条以内的数据。
	 *
	 * @param msgs 需要导入数据库的消息。
	 *
	 * \~english
	 * Imports messages to the local database.
	 * 
	 * You can only import messages that you sent or received.
	 * 
	 * It is recommended that you import at most 1,000 messages each time.
	 *
	 * @param msgs The messages to import.
	 */
	public synchronized void importMessages(List<EMMessage> msgs) {
		List<EMAMessage> dummy = new ArrayList<EMAMessage>();
		for (EMMessage msg : msgs) {
			dummy.add(msg.emaObject);
		}
		EMClient.getInstance().getChatConfigPrivate().importMessages(dummy);
	}

	/**
	 * \~chinese
	 * 获取指定类型的所有会话。
	 *
	 * @param type 	会话类型，详见 {@link EMConversationType}。
	 * @return 		指定类型的会话列表。
	 *
	 * \~english
	 * Gets the list of conversations by conversation type.
	 *
	 * @param type	The conversation type. See {@link EMConversationType}.
	 * @return 		The list of conversations of the specified type.
	 */
	public List<EMConversation> getConversationsByType(EMConversationType type) {
		List<EMAConversation> conversations = emaObject.getConversations();
		List<EMConversation> result = new ArrayList<EMConversation>();
		for (EMAConversation conv : conversations) {
			if (type.ordinal() == conv._getType().ordinal()) {
				result.add(new EMConversation(conv));
			}
		}
		return result;
	}

	/**
	 * \~chinese
	 * 获取本地当前所有会话。
     *
     * 该方法会先从内存中获取，如果未找到任何会话，从本地数据库获取。
	 *
	 * @return 返回本地内存或者数据库中所有的会话。
	 *
	 * \~english
	 * Gets all local conversations.
	 * 
	 * Conversations will be first retrieved from the memory. If no conversation is found, the SDK retrieves from the local database.
	 *
	 * @return The retrieved conversations.
	 */
	public Map<String, EMConversation> getAllConversations() {
		List<EMAConversation> conversations = emaObject.getConversations();
		Hashtable<String, EMConversation> result = new Hashtable<String, EMConversation>();
		for (EMAConversation conversation : conversations) {
			/*if (conversation._getType() != EMAConversationType.CHATROOM) {
				result.put(conversation.conversationId(), new EMConversation(conversation));
			}*/
			result.put(conversation.conversationId(), new EMConversation(conversation));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取会话列表。
     *
     * 该功能需联系商务开通。开通后，用户默认可拉取 7 天内的 10 个会话（每个会话包含最新一条历史消息）。如需调整会话数量或时间限制请联系商务经理。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @return 返回当前用户的会话列表。
	 *
	 * \~english
	 * Gets the conversation list from the server.
	 *
	 * **Note**
     *
	 * - This is a synchronous method and blocks the current thread.
     * - To use this function, you need to contact our business manager to activate it.
     * - After this function is activated, users can pull 10 conversations within 7 days by default (each conversation contains the latest historical message).
     * - If you want to adjust the number of conversations or time limit, contact our business manager.
	 *
	 * @return    The conversation list of the current user.
	 */
	public Map<String, EMConversation> fetchConversationsFromServer() throws HyphenateException {
		EMAError error = new EMAError();

		List<EMAConversation> conversations = emaObject.fetchConversationsFromServer(error);
		//emaObject.fetchConversationsFromServer(error);
		handleError(error);

		Hashtable<String, EMConversation> result = new Hashtable<String, EMConversation>();
		for (EMAConversation conversation : conversations) {
			result.put(conversation.conversationId(), new EMConversation(conversation));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取会话列表。
     *
     * 该功能需联系商务开通。开通后，用户默认可拉取 7 天内的 10 个会话（每个会话包含最新一条历史消息）。如需调整会话数量或时间限制请联系商务经理。
	 *
	 * 异步方法。
	 *
	 * @return	当前用户的会话列表。
	 *
	 * \~english
	 * Gets the conversation list from the server.
	 *
	 * **Note**
     *
	 * - This is an asynchronous method.
     * - To use this function, you need to contact our business manager to activate it.
     * - After this function is activated, users can pull 10 conversations within 7 days by default (each conversation contains the latest historical message).
     * - If you want to adjust the number of conversations or time limit, contact our business manager.
	 *
	 * @return	The conversation list of the current user.
	 */
	public void asyncFetchConversationsFromServer(final EMValueCallBack<Map<String, EMConversation>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchConversationsFromServer());
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从数据库加载本地所有的会话到内存中。
	 *
	 * 一般在登录成功后使用此方法，可以加快会话列表的加载速度。
	 *
	 * \~english
	 * Loads all conversations from the local database into the memory.
	 *
	 * Generally, this method is called upon successful login to speed up the loading of the conversation list.
	 */
	public void loadAllConversations() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~chinese
	 * 删除指定会话及其本地历史消息。
	 *
	 * @param username 			会话 ID。
	 * @param deleteMessages	删除会话时是否同时删除本地的历史消息。
	 *                          - `true`：删除；
	 *                          - `false`：不删除。
	 * @return 					会话是否删除成功。
	 *                          - `true`：是；
	 *                          - `false`： 否。
	 *
	 * \~english
	 * Deletes a conversation and its local messages from the local database.
	 *
	 * @param username 			The conversation ID.
	 * @param deleteMessages 	Whether to delete the historical messages with the conversation.
	 *                          - `true`: Yes.
     *                          - `false`: No.
	 * @return 					Whether the conversation is successfully deleted.
     *                          - `true`: Yes.
     *                          - `false`: No.
	 */
	public boolean deleteConversation(String username, boolean deleteMessages) {
		EMConversation conv = getConversation(username);
		if (conv == null) {
			return false;
		}
		if (!deleteMessages) {
			conv.clear();
		} else {
			conv.clearAllMessages();
		}
		emaObject.removeConversation(username, deleteMessages, conv.isChatThread());
		return true;
	}

	/**
	 * \~chinese
	 * 删除服务端的指定会话及其历史消息。
	 *
	 * 异步方法。
	 *
	 * @param username 			会话 ID。
	 * @param type              会话类型，详见 {@link EMConversationType}。
	 * @param isDeleteServerMessages	删除会话时是否同时删除历史消息。
	 *                                  - `true`：是；
     *                                  - `false`: 否。
	 * @param deleteCallBack  删除服务端会话及其历史消息成功与否的回调。
	 *                        - `true`：删除成功；
	 *                        - `false`：删除失败。
	 *
	 * \~english
	 * Deletes the specified conversation and its historical messages from the server.
	 *
	 * This is an asynchronous method.
	 *
	 * @param username 			The conversation ID.
	 * @param type              The conversation type. See {@link EMConversationType}.
	 * @param isDeleteServerMessages 	Whether to delete the historical messages with the conversation.
	 *                                  - `true`: Yes
	 *                                  - `false`: No
	 * @param deleteCallBack  The callback for the deletion of the conversation and its historical messages.
	 *                          - `true`: Success;
	 *                          - `false`: Failure.
	 */
	public void deleteConversationFromServer(String username,EMConversationType type, boolean isDeleteServerMessages,EMCallBack deleteCallBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				EMAConversationType t = EMAConversationType.CHAT;
				switch (type) {
					case Chat:
						t = EMAConversationType.CHAT;
				        break;
					case GroupChat:
						t = EMAConversationType.GROUPCHAT;
				        break;
					case ChatRoom:
						t = EMAConversationType.CHATROOM;
				        break;
					case DiscussionGroup:
						t = EMAConversationType.DISCUSSIONGROUP;
				        break;
					case HelpDesk:
						t = EMAConversationType.HELPDESK;
				        break;
				}
				EMAError error=emaObject.deleteConversationFromServer(username,t, isDeleteServerMessages);
				if(error.errCode()==EMAError.EM_NO_ERROR) {
					deleteCallBack.onSuccess();
				}else{
					deleteCallBack.onError(error.errCode(), error.errMsg());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 注册消息监听器。
	 *
	 * 接收到新消息等回调可以通过设置此方法进行监听。
	 *
	 * 在不需要消息监听器时，可调用 {@link #removeMessageListener(EMMessageListener)} 方法将其移除。
	 *
	 * @param listener 要注册的消息监听，详见 {@link EMMessageListener}。
	 *
	 * \~english
	 *
	 * Adds the message listener.
	 * 
	 * After a message listener is added, you can listen for new messages when they arrive.
	 *
	 * You can call {@link #removeMessageListener(EMMessageListener)} to remove the message listener when it is not required.
	 *
	 * @param listener The message listener to add. See {@link EMMessageListener}.
	 */
	public void addMessageListener(EMMessageListener listener) {
		if(listener == null){
			EMLog.d(TAG, "addMessageListener: listener is null");
			return;
		}

		if(!messageListeners.contains(listener)){
			EMLog.d(TAG, "add message listener: " + listener);
			messageListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除消息监听器。
	 * 
	 * 调用 {@link #addMessageListener(EMMessageListener)} 添加消息监听后再调用本方法移除。
	 *
	 * @param listener 要移除的监听器，详见 {@link EMMessageListener}。
	 *
	 * \~english
	 * Removes the message listener.
	 * 
	 * After adding a message listener with {@link #addMessageListener(EMMessageListener)}, you can call this method to remove it when it is not required.
	 * 
	 * @param listener The message listener to remove.
	 */
	public void removeMessageListener(EMMessageListener listener) {
		if(listener == null){
			return;
		}

		messageListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 注册会话监听器。
	 * 
	 * 会话监听器可监听会话变化及会话已读回执。
	 * 
	 * 在不需要会话监听器时，可调用 {@link #removeConversationListener(EMConversationListener)} 方法移除。
	 *
	 * @param listener 要添加的会话监听，详见 {@link EMConversationListener}。
	 *
	 * \~english
	 * Adds the conversation listener. 
	 * 
	 * After a conversation listener is added, you can listen for conversation changes and conversation read receipts.
	 * 
	 * You can call the {@link #removeConversationListener(EMConversationListener)} method to remove the conversation listener when it is not required.
	 *
	 * @param listener The conversation listener to add. See {@link EMConversationListener}.
	 */
	public void addConversationListener(EMConversationListener listener){
		if(!conversationListeners.contains(listener)){
			conversationListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除会话监听器。
	 * 
	 * 调用 {@link #addConversationListener(EMConversationListener)} 添加会话监听器后再调用本方法移除。
	 *
	 * @param listener  要移除的会话监听器，详见 {@link EMConversationListener}。
	 *
	 * \~english
	 * Removes the conversation listener.
	 * 
	 * After adding a conversation listener with {@link #addConversationListener(EMConversationListener)}, you can call this method to remove it when it is not required.
	 *
	 * @param listener The conversation listener to remove.
	 */
	public void removeConversationListener(EMConversationListener listener){
		if(listener == null){
			return;
		}

		conversationListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 将语音消息设置为已听。
	 *
	 * @param message 要设置的消息对象。
	 *
	 * \~english
	 * Marks a voice message as listened.
	 *
	 * @param message The message object.
	 */
	public void setVoiceMessageListened(EMMessage message)
	{
		message.setListened(true);
		updateMessage(message);
	}

	void onLogout() {
		caches.clear();
	}

	synchronized void  loadAllConversationsFromDB() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~chinese
	 * 将数据库中指定联系人的相关信息变更成另外一个联系人。
	 * 
	 * **注意**
	 * - 消息列表，会话列表，联系人列表和黑名单列表等相关信息会发生变化。
	 * - 该方法不会更新内存中数据。
	 *
	 * @param from		原联系人的用户 ID。
	 * @param changeTo	新联系人的用户 ID。
	 * @return 			返回更新结果，任何列表更新失败，均会返回 `false`。
	 *
	 * \~english
	 * Changes the data of one contact in the local database to those of the other contact.
	 * 
	 * **Note**
	 * - This method brings changes to data such as the message list, the conversation list, contact list, the block list in the database.
	 * - This method does not update the data stored in the memory.
	 *
	 * @param from		The user ID of the original contact.
	 * @param changeTo  The user ID of the new contact.
	 * @return 			The operation result.
	 *                  - `true`: Success.
	 *                  - `false`: Failure.
	 */
	public boolean updateParticipant(String from, String changeTo) {
		return emaObject.updateParticipant(from, changeTo);
	}

	/**
	 * \~chinese
	 * 从服务器分页获取群组消息已读回执详情。
	 * 
	 * 发送群组消息回执，可调用 {@link #ackGroupMessageRead(String, String, String)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param msgId 				消息 ID。
	 * @param pageSize 				每页期望返回的群消息已读数。取值范围[1,50]。
	 * @param startAckId 			查询起始的已读回执 ID。该参数设置后，SDK 从指定的已读回执 ID 开始，按服务器接收已读回执的时间的逆序获取。
	 *                              若该参数为空，SDK 从最新的已读回执开始按服务器接收回执时间的逆序获取。
	 * @return 						返回消息列表（不包含查询起始 ID 的已读回执）和用于继续获取群消息回执的 cursor。
	 * @throws HyphenateException 	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Uses the pagination to get read receipts for a group message from the server.
	 * 
	 * To send a read receipt for a group message, you can call {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param msgId 				The message ID.
	 * @param pageSize 				The number of read receipts for the group message that you expect to get on each page. The value range is [1,50].
	 * @param startAckId 			The starting read receipt ID for query. After this parameter is set, the SDK retrieves read receipts, from the specified one, in the reverse chronological order of when the server receives them.
	 *                              If this parameter is set as `null` or an empty string, the SDK retrieves read receipts, from the latest one, in the reverse chronological order of when the server receives them.
	 * @return 						The list of retrieved read receipts (excluding the one with the starting ID) and the cursor for the next query.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMCursorResult<EMGroupReadAck> fetchGroupReadAcks(String msgId, int pageSize, String startAckId) throws HyphenateException {
		EMAError error = new EMAError();
        EMCursorResult<EMGroupReadAck> cusorResult = new EMCursorResult<>();

		EMMessage msg = getMessage(msgId);
		String groupId = null;
		if (msg.getChatType() == EMMessage.ChatType.GroupChat && msg.isNeedGroupAck()) {
			groupId = msg.conversationId();
		} else {
			EMLog.e(TAG, "not group msg or don't need ack");
			return cusorResult;
		}

		EMCursorResult<EMAGroupReadAck> _cusorResult = emaObject.fetchGroupReadAcks(msgId, groupId, error, pageSize, startAckId);
		handleError(error);
		cusorResult.setCursor(_cusorResult.getCursor());

		List<EMGroupReadAck> groupReadAcks = new ArrayList<>();

		for(EMAGroupReadAck _ack: _cusorResult.getData()) {
			groupReadAcks.add(new EMGroupReadAck(_ack));
		}
		cusorResult.setData(groupReadAcks);
		return cusorResult;
	}

	/**
	 * \~chinese
	 * 从服务器分页获取群组消息已读回执详情。
	 * 
	 * 发送群组消息回执，可调用 {@link #ackGroupMessageRead(String, String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param msgId 		消息 ID。
	 * @param pageSize 		每页获取的群消息已读回执数。取值范围为 [1,50]。
	 * @param startAckId 	查询起始的已读回执 ID。该参数设置后，SDK 从指定的已读回执 ID 开始，按服务器接收已读回执的时间的逆序获取。
	 *                      若该参数为空，SDK 从最新的已读回执开始按服务器接收回执时间的逆序获取。
	 * @param callBack    	结果回调。
	 *                      - 调用成功，触发 {@link EMValueCallBack#onSuccess(Object)} 回调，返回查询的已读回执列表（不包含查询起始 ID 的已读回执）和下次查询的 cursor。
	 *                      - 调用失败，触发 {@link EMValueCallBack#onError(int, String)} 回调报错。
	 *
	 * \~english
	 * Uses the pagination to get read receipts for the group message from the server.
	 * 
	 * To send a read receipt for a group message, you can call {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param msgId 		The message ID.
	 * @param pageSize 		The number of read receipts for the group message that you expect to get on each page. The value range is [1,50].
	 * @param startAckId 	The starting read receipt ID for query. After this parameter is set, the SDK retrieves read receipts, from the specified one, in the reverse chronological order of when the server receives them.
	 *                      If this parameter is set as `null` or an empty string, the SDK retrieves read receipts, from the latest one, in the reverse chronological order of when the server receives them.
	 * @param callBack 		The result callback:
	 *                      - If the call succeeds, the callback {@link EMValueCallBack#onSuccess(Object)} is triggered to return the list of retrieved read receipts (excluding the one with the starting ID) and the cursor for the next query.
	 *                      - If the call fails, the callback {@link EMValueCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void asyncFetchGroupReadAcks(final String msgId, final int pageSize,
										final String startAckId, final EMValueCallBack<EMCursorResult<EMGroupReadAck>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupReadAcks(msgId, pageSize, startAckId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器分页获取指定会话的历史消息。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param conversationId 		会话 ID。
	 * @param type 					会话类型, 详见 {@link EMConversationType}。
	 * @param pageSize 				每页期望获取的消息条数。取值范围为 [1,50]。
	 * @param startMsgId 			开始获取的消息 ID。该参数设置后，SDK 从指定的消息 ID 开始，按服务器接收消息的时间的逆序获取。
	 *                              若该参数为空，SDK 从最新的消息开始，按服务器接收消息的时间的逆序获取。
	 * @return 						消息列表（不包含查询起始 ID 的消息）和下次查询的 cursor。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Uses the pagination to get historical messages of the conversation from the server.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param conversationId 		The conversation ID.
	 * @param type 					The conversation type. See {@link EMConversationType}.
	 * @param pageSize 				The number of messages that you expect to get on each page. The value range is [1,50].
	 * @param startMsgId 			The starting message ID for query. After this parameter is set, the SDK retrieves messages, from the specified one, in the reverse chronological order of when the server receives them.
	 *                              If this parameter is set as `null` or an empty string, the SDK retrieves messages, starting from the latest one, in the reverse chronological order of when the server receives them.
	 * @return 					    The list of retrieved messages (excluding the one with the starting ID) and the cursor for the next query.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMCursorResult<EMMessage> fetchHistoryMessages(String conversationId, EMConversationType type, int pageSize,
	                                 String startMsgId) throws HyphenateException {
		return fetchHistoryMessages(conversationId, type, pageSize, startMsgId, EMConversation.EMSearchDirection.UP);
	}

	/**
	 * \~chinese
	 * 从服务器分页获取指定会话的历史消息。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param conversationId 		会话 ID。
	 * @param type 					会话类型, 详见 {@link EMConversationType}。
	 * @param pageSize 				每页期望获取的消息条数。取值范围为 [1,50]。
	 * @param startMsgId 			查询的起始消息 ID。该参数设置后，SDK 从指定的消息 ID 开始，按消息检索方向获取。如果传入消息的 ID 为空，SDK 忽略该参数，按搜索方向查询消息。
	 *                              - 若 `direction` 为 `UP`，SDK 从最新消息开始，按照服务器接收消息时间的逆序获取；
 *                                  - 若 `direction` 为 `DOWN`，SDK 从最早消息开始，按照服务器接收消息时间的正序获取。
	 * @param direction				消息检索方向。详见 {@link EMConversation.EMSearchDirection}。
	 *                              - （默认）`UP`：按照服务器接收消息时间的逆序获取；
	 *                              - `DOWN`：按照服务器接收消息时间的正序获取。
	 * @return 						消息列表（不包含查询起始 ID 的消息）和下次查询的 cursor。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Uses the pagination to get historical messages of the conversation from the server.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param conversationId 		The conversation ID.
	 * @param type 					The conversation type. See {@link EMConversationType}.
	 * @param pageSize 				The number of messages you expect to get on each page. The value range is [1,50].
	 * @param startMsgId 			The starting message ID for query. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                              If this parameter is set as "null" or an empty string, the SDK retrieves messages according to the message search direction while ignoring this parameter.
	 *                              - If `direction` is set as `UP`, the SDK retrieves messages, starting from the latest one, in the reverse chronological order of when the server receives them.
     *                              - If `direction` is set as `DOWN`, the SDK retrieves messages, starting from the oldest one, in the chronological order of the time the server receives them.
	 * @param direction             The message search direction. See {@link EMConversation.EMSearchDirection}.
	 *                              - (Default) `UP`: The SDK retrieves messages in the reverse chronological order of when the server receives them;
	 * 								- `DOWN`: The SDK retrieves messages in the chronological order of the time the server receives them.
	 *
	 * @return 					    The list of retrieved messages (excluding the one with the starting ID) and the cursor for the next query.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMCursorResult<EMMessage> fetchHistoryMessages(String conversationId,
														  EMConversationType type,
														  int pageSize,
														  String startMsgId,
														  EMConversation.EMSearchDirection direction) throws HyphenateException {
		EMAError error = new EMAError();
		EMAConversation.EMASearchDirection d = direction == EMConversation.EMSearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;
		EMCursorResult<EMAMessage> _cursorResult = emaObject.fetchHistoryMessages(conversationId,
				type.ordinal(), pageSize, startMsgId, d, error);
		handleError(error);
		EMCursorResult<EMMessage> cursorResult = new EMCursorResult<>();
		cursorResult.setCursor(_cursorResult.getCursor());
		List<EMMessage> msgs = new ArrayList<>();
		for (EMAMessage _msg : _cursorResult.getData()) {
			msgs.add(new EMMessage(_msg));
		}
		cursorResult.setData(msgs);
		return cursorResult;
	}

	/**
	 * \~chinese
	 * 从服务器分页获取指定会话的历史消息。
	 *
	 * 异步方法。
	 *
	 * @param conversationId 	会话 ID。
	 * @param type 				会话类型，详见 {@link EMConversationType}。
	 * @param pageSize 			每页期望获取的消息条数。取值范围为 [1,50]。
	 * @param startMsgId 		查询的起始消息 ID。该参数设置后，SDK 从指定的消息 ID 开始，按服务器接收消息的时间的逆序获取。
	 *                          若该参数为空，SDK 从最新的消息开始，按服务器接收消息的时间的逆序获取。
	 * @param callBack 			结果回调，成功则执行 {@link EMValueCallBack#onSuccess(Object)}，返回消息列表（不包含查询起始 ID 的消息）和下次查询的 cursor；
	 *                          失败则执行 {@link EMValueCallBack#onError(int, String)}。
	 *
	 *
	 * \~english
	 * Uses the pagination to get the historical messages of the conversation from the server.
	 *
	 * This is an asynchronous method.
	 *
	 * @param conversationId 	The conversation ID.
	 * @param type 				The conversation type. See {@link EMConversationType}.
	 * @param pageSize 			The number of messages that you expect to get on each page. The value range is [1,50].
	 * @param startMsgId 		The starting message ID for query. After this parameter is set, the SDK retrieves messages, starting from the specified one, in the reverse chronological order of when the server receives them.
	 *                          If this parameter is set as "null" or an empty string, the SDK retrieves messages, starting from the latest one, in the reverse chronological order of when the server receives them.
	 * @param callBack 			The result callback:
	 *                          - If the call succeeds, the callback {@link EMValueCallBack#onSuccess(Object)} is triggered to return the list of retrieved messages (excluding the one with the starting ID) and the cursor for the next query;
	 *                          - If the call fails, the callback {@link EMValueCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void asyncFetchHistoryMessage(final String conversationId, final EMConversationType type, final int pageSize,
	                                final String startMsgId, final EMValueCallBack<EMCursorResult<EMMessage>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchHistoryMessages(conversationId, type, pageSize, startMsgId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器分页获取指定会话的历史消息。
	 *
	 * 异步方法。
	 *
	 * @param conversationId 	会话 ID。
	 * @param type 				会话类型，详见 {@link EMConversationType}。
	 * @param pageSize 			每页期望获取的消息数。取值范围为 [1,50]。
	 * @param startMsgId 		查询的起始消息 ID。该参数设置后，SDK 从指定的消息 ID 开始，按消息搜索方向获取。
	 *                          如果传入消息的 ID 为空，SDK 忽略该参数，按搜索方向查询消息。
	  *                         - 若 `direction` 为 `UP`，SDK 从最新消息开始，按照服务器接收消息时间的逆序获取；
 *                              - 若 `direction` 为 `DOWN`，SDK 从最早消息开始，按照服务器接收消息时间的正序获取。
	 * @param direction         消息搜索方向。详见 {@link EMConversation.EMSearchDirection}。
	 *                          - `UP`：按照消息中的时间戳的逆序查询；
	 * 							- `DOWN`：按照消息中的时间戳的正序查询；
	 * @param callBack 			结果回调：
	 *                          - 调用成功，触发 {@link EMValueCallBack#onSuccess(Object)}，返回消息列表（不包含查询起始 ID 的消息）和下次查询的 cursor；
	 *                          - 调用失败，触发 {@link EMValueCallBack#onError(int, String)} 报错。
	 *
	 *
	 * \~english
	 * Uses the pagination to get the messages in the conversation from the server.
	 *
	 * This is an asynchronous method.
	 *
	 * @param conversationId 	The conversation ID.
	 * @param type 				The conversation type. See {@link EMConversationType}.
	 * @param pageSize 			The number of messages that you expect to get on each page. The value range is [1,50].
	 * @param startMsgId        The starting message ID for query. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                          If this parameter is set as "null" or an empty string, the SDK retrieves messages according to the message search direction while ignoring this parameter.
	  *                         - If `direction` is set as `UP`, the SDK retrieves messages, starting from the latest one, in the reverse chronological order of when the server receives them.
     *                          - If `direction` is set as `DOWN`, the SDK retrieves messages, starting from the oldest one, in the chronological order of the time the server receives them.
	 * @param direction         The message search direction. See {@link EMConversation.EMSearchDirection}.
	 *                          - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 							- `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
	 * @param callBack 			The result callback:
	 *                          - If the call succeeds, the callback {@link EMValueCallBack#onSuccess(Object)} is triggered to return the list of retrieved messages (excluding the one with the starting ID) and the cursor for the next query;
	 *                          - If the call fails, the callback {@link EMValueCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void asyncFetchHistoryMessage(final String conversationId,
										 final EMConversationType type,
										 final int pageSize,
										 final String startMsgId,
										 final EMConversation.EMSearchDirection direction,
										 final EMValueCallBack<EMCursorResult<EMMessage>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchHistoryMessages(conversationId, type, pageSize, startMsgId, direction));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	private void handleError(EMAError error) throws HyphenateException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new HyphenateException(error);
		}
	}

	private static final int LIST_SIZE = 512;

	/**
	 * \~chinese
	 * 从本地数据库获取指定会话的一定数量的特定类型的消息。
	 *
	 * @param type       消息类型。详见 {@link Type}。
	 * @param timeStamp  查询的起始消息 Unix 时间戳，单位为毫秒。该参数设置后，SDK 从指定时间戳的消息开始，按消息搜索方向获取。
	 *                   如果该参数设置为负数，SDK 从当前时间开始搜索。
	 * @param maxCount   每次获取的最大消息数。取值范围为 [1,400]。
	 * @param from       搜索来自指定用户或者群组的消息，一般为会话 ID。
	 * @param direction  消息搜索方向。详见 {@link EMConversation.EMSearchDirection}。
	 *                    - `UP`：按照消息中的时间戳的逆序查询；
	 * 				      - `DOWN`：按照消息中的时间戳的正序查询。
	 * @return           消息列表（不包含查询起始时间戳对应的消息）。若未查找到任何消息，返回空列表。
	 *
	 * \~english
	 * Retrieves messages of a certain type in the conversation from the local database.
	 *
	 * @param type       The message type. See {@link Type}.
	 * @param timeStamp  The starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If you set this parameter as a negative value, the SDK retrieves messages, starting from the current time, in the descending order of the timestamp included in them.
	 * @param maxCount   The maximum number of messages to retrieve each time. The value range is [1,400].
	 * @param from       The user ID of the message sender or group ID for retrieval. Usually, it is the conversation ID.
	 * @param direction  The message search direction. See {@link EMConversation.EMSearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
	 * @return           The list of retrieved messages (excluding the one with the starting timestamp). If no message is obtained, an empty list is returned.
	 */
	public List<EMMessage> searchMsgFromDB(Type type, long timeStamp, int maxCount, String from, EMConversation.EMSearchDirection direction) {
		EMAConversation.EMASearchDirection d = direction == EMConversation.EMSearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;

		List<EMAMessage> msgs = emaObject.searchMessages(type.ordinal(), timeStamp, maxCount, from, d);
		// to avoid resizing issue of array list, used linked list when size > 512
		List<EMMessage> result;
		if (msgs.size() > LIST_SIZE) {
			result = new LinkedList<EMMessage>();
		} else {
			result = new ArrayList<EMMessage>();
		}
		for (EMAMessage msg : msgs) {
			if (msg != null) {
				result.add(new EMMessage(msg));
			}
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从本地数据库获取指定会话中包含特定关键字的消息。
	 *
	 * @param keywords   查询关键字。
	 * @param timeStamp  查询的起始消息 Unix 时间戳，单位为毫秒。该参数设置后，SDK 从指定时间戳的消息开始，按消息搜索方向获取。
	 *                   如果该参数设置为负数，SDK 从当前时间开始搜索。
	 * @param maxCount   每次获取的最大消息数。取值范围为 [1,400]。
	 * @param from       查询来自指定用户或者群组的消息，一般为会话 ID。
	 * @param direction  消息搜索方向。详见 {@link EMConversation.EMSearchDirection}。
	 *                    - `UP`：按照消息中的时间戳的逆序查询；
	 * 				      - `DOWN`：按照消息中的时间戳的正序查询。
	 * @return           消息列表（不包含查询起始时间戳对应的消息）。若未查找到任何消息，返回空列表。
	 *
	 * \~english
	 * Retrieves messages with keywords in the conversation in the local database.
	 *
	 * @param keywords   The keywords for query.
	 * @param timeStamp  The starting Unix timestamp in the message for query. The unit is millisecond. After this parameter is set, the SDK retrieves messages, starting from the specified one, according to the message search direction.
	 *                   If you set this parameter as a negative value, the SDK retrieves messages, starting from the current time, in the descending order of the timestamp included in them.
	 * @param maxCount   The maximum number of messages to retrieve each time. The value range is [1,400].
	 * @param from       The user ID of the message sender or group ID for retrieval. Usually, it is the conversation ID.
	 * @param direction  The message search direction. See {@link EMConversation.EMSearchDirection}.
	 *                   - `UP`: The SDK retrieves messages in the descending order of the timestamp included in them.
	 * 					 - `DOWN`: The SDK retrieves messages in the ascending order of the timestamp included in them.
	 * @return            The list of retrieved messages (excluding the one with the starting timestamp). If no message is obtained, an empty list is returned.
	 */
	public List<EMMessage> searchMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMConversation.EMSearchDirection direction) {
		EMAConversation.EMASearchDirection d = direction == EMConversation.EMSearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;

		List<EMAMessage> msgs = emaObject.searchMessages(keywords, timeStamp, maxCount, from, d);
		// to avoid resizing issue of array list, used linked list when size > 512
		List<EMMessage> result;
		if (msgs.size() > LIST_SIZE) {
			result = new LinkedList<EMMessage>();
		} else {
			result = new ArrayList<EMMessage>();
		}
		for (EMAMessage msg : msgs) {
			if (msg != null) {
				result.add(new EMMessage(msg));
			}
		}
		return result;
	}

	/**
	 * \~chinese
	 * 删除指定时间戳之前的本地历史消息。
	 *
	 * @param timeStamp 			查询的消息的起始 Unix 时间戳，单位为毫秒。
	 * @param callback 				删除结果回调，详见 {@link EMCallBack}。
	 *
	 * \~english
	 * Deletes local historical messages with a Unix timestamp before a specified one.
	 *
	 * @param timeStamp 			The starting Unix timestamp in the message for query. This unit is millisecond.
	 * @param callback 				The result callback. See {@link EMCallBack}.
	 */
	public void deleteMessagesBeforeTimestamp(long timeStamp, EMCallBack callback){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				boolean result = emaObject.removeMessagesBeforeTimestamp(timeStamp);
				if(result){
					for(EMConversation.MessageCache cache : caches.values()){
						for(EMMessage message : cache.getAllMessages()){
							if(message.getMsgTime() < timeStamp){
								cache.removeMessage(message.getMsgId());
							}
						}
					}
					callback.onSuccess();
				}else{
					callback.onError(EMError.DATABASE_ERROR, "Database operation failed");
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 举报非法消息。
	 *
	 * 异步方法。
	 *
	 * @param msgId				非法消息的 ID。
	 * @param reportTarget 		非法消息标签，如涉政或涉恐。
	 * @param reportReason 		举报原因。
	 * @param callBack 			结果回调，详见 {@link EMCallBack}：
	 *                          - 调用成功，触发 {@link EMCallBack#onSuccess()} 回调；
	 *                          - 调用失败，触发 {@link EMCallBack#onError(int, String)} 回调。
	 *
	 * \~english
	 * Reports an inappropriate message.
	 *
	 * This is an asynchronous method.
	 *
	 * @param msgId				The ID of the inappropriate message.
	 * @param reportTarget		The message content tag. For example, the message is related to pornography or terrorism.
	 * @param reportReason		The reason for reporting the message.
	 * @param callBack 			The result callback. See {@link EMCallBack}.
	 *                          - If the call succeeds, {@link EMCallBack#onSuccess()} is triggered;
	 *                          - If the call fails, {@link EMCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void asyncReportMessage(String msgId, String reportTarget, String reportReason,EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					reportMessage(msgId, reportTarget, reportReason);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}
	/**
	 * \~chinese
	 * 举报非法消息。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param msgId				非法消息的 ID。
	 * @param reportTarget 		非法消息标签，如涉政或涉恐。
	 * @param reportReason 		举报原因。
	 * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Reports an inappropriate message.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param msgId				The ID of the inappropriate message.
	 * @param reportTarget		The message content tag. For example, the message is related to pornography or terrorism.
	 * @param reportReason		The reason for reporting the message.
	 *
	 * @throws HyphenateException A description of the exception. See {@link EMError}.
	 */
	public void reportMessage(String msgId, String reportTarget, String reportReason ) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.reportMessage(msgId,reportTarget,reportReason,error);
		handleError(error);
	}

	/**
	 * 获取翻译服务支持的语言。
	 *
	 * @param callBack 结果回调，详见 {@link EMCallBack}。
	 *                 - 调用成功，触发 {@link EMCallBack#onSuccess()} 返回获取的语言；
	 *                 - 调用失败，触发 {@link EMCallBack#onError(int, String)} 报错。
	 *
	 * \~english
	 * Gets all languages supported by the translation service.
	 *
	 * @param callBack The result callback. See {@link EMCallBack}.
	 *                 - If the call succeeds, {@link EMCallBack#onSuccess()} is triggered to return the obtained languages;
	 *                 - If the call fails, {@link EMCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void fetchSupportLanguages(EMValueCallBack<List<EMLanguage>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					List<EMLanguage> languages = new ArrayList<>();
					EMAError error = new EMAError();
					List<List<String>> result = emaObject.fetchSupportLanguages(error);
					handleError(error);
					for(List<String> list : result){
						languages.add(new EMLanguage(list.get(0), list.get(1), list.get(2)));
					}
					callBack.onSuccess(languages);
				} catch (HyphenateException e){
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 翻译一条文本消息。
	 *
	 * @param message 要翻译的消息对象。
	 * @param languages 目标语言代码列表。
	 * @param callBack 结果回调，详见 {@link EMCallBack}。
	 *                 - 调用成功，触发 {@link EMCallBack#onSuccess()} 返回译文；
	 *                 - 调用失败，触发 {@link EMCallBack#onError(int, String)} 报错。
	 *
	 * \~english
	 * Translates a text message.
	 *
	 * @param message The message object for translation.
	 * @param languages The list of target language codes.
	 * @param callBack The result callback. See {@link EMCallBack}.
	 *                 - If the call succeeds, {@link EMCallBack#onSuccess()} is triggered to return the translation;
	 *                 - If the call fails, {@link EMCallBack#onError(int, String)} is triggered to report an error.
	 */
	public void translateMessage(EMMessage message, List<String> languages, EMValueCallBack<EMMessage> callBack){
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try{
					EMAError error = new EMAError();
					EMAMessage emaMessage = emaObject.translateMessage(message.emaObject, languages, error);
					handleError(error);
					callBack.onSuccess(new EMMessage(emaMessage));
				}catch(HyphenateException e){
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}
	/**
	 * \~chinese
	 * 添加 Reaction。
	 *
	 * 同步方法。
	 *
	 * @param messageId 消息 ID。
	 * @param reaction  要添加的消息 Reaction。
	 * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Adds a Reaction.
	 *
	 * This is a synchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The Reaction content.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void addReaction(final String messageId, final String reaction) throws HyphenateException {
		EMAError error = new EMAError();
		emaReactionObject.addReaction(messageId, reaction, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 添加 Reaction。
	 *
	 * 异步方法。
	 *
	 * @param messageId 消息 ID。
	 * @param reaction  要添加的消息 Reaction。
	 * @param callback  处理结果回调，失败会返回失败原因。详见 {@link EMCallBack}。
	 *
	 * \~english
	 * Adds a Reaction.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The message Reaction to add.
	 * @param callback  The result callback which contains the error information if the method fails. See {@link EMCallBack}.
	 */
	public void asyncAddReaction(final String messageId, final String reaction, final EMCallBack callback) {
		EMClient.getInstance().execute(() -> {
			try {
				addReaction(messageId, reaction);
				if (null != callback) {
					callback.onSuccess();
				}
			} catch (HyphenateException e) {
				EMLog.e(TAG, "asyncAddReaction error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				if (null != callback) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 删除 Reaction。
	 *
	 * 同步方法。
	 *
	 * @param messageId 消息 ID。
	 * @param reaction  要删除的 Reaction。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Deletes a Reaction.
	 *
	 * This is a synchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The Reaction content to delete.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void removeReaction(final String messageId, final String reaction) throws HyphenateException {
		EMAError error = new EMAError();
		emaReactionObject.removeReaction(messageId, reaction, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 删除 Reaction。
	 *
	 * 异步方法。
	 *
	 * @param messageId 消息 ID。
	 * @param reaction  要删除的 Reaction。
	 * @param callback  处理结果回调，详见 {@link EMCallBack}。
	 *
	 * \~english
	 * Deletes a Reaction.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The Reaction to delete.
	 * @param callback  The result callback which contains the error information if the method fails. See {@link EMCallBack}.
	 */
	public void asyncRemoveReaction(final String messageId, final String reaction, final EMCallBack callback) {
		EMClient.getInstance().execute(() -> {
			try {
				removeReaction(messageId, reaction);
				if (null != callback) {
					callback.onSuccess();
				}
			} catch (HyphenateException e) {
				EMLog.e(TAG, "asyncRemoveReaction error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				if (null != callback) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取 Reaction 列表。
	 *
	 * 同步方法。
	 *
	 * @param messageIdList 消息 ID。
	 * @param chatType      会话类型，仅支持单聊（{@link EMMessage.ChatType#Chat}）和群聊（{@link EMMessage.ChatType#GroupChat}）。
	 * @param groupId       群组 ID，该参数仅对群聊生效。
	 * @return map          指定消息 ID 对应的 Reaction 列表（`EMMessageReaction` 的用户列表 `UserList` 为概要数据，只包含前三个用户信息）。
	 *
	 * \~english
	 * Gets the list of Reactions.
	 *
	 * This is a synchronous method.
	 *
	 * @param messageIdList  The message ID.
	 * @param chatType       The conversation type. Only the one-to-one chats ({@link EMMessage.ChatType#Chat} and group chats ({@link EMMessage.ChatType#GroupChat}) are allowed.
	 * @param groupId        The group ID, which is invalid only for group chats.
	 * @return map           The Reaction list under the specified message ID. The `UserList` of `EMMessageReaction` indicates the summary data, which only contains the information of the first three users.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public Map<String, List<EMMessageReaction>> getReactionList(final List<String> messageIdList, final EMMessage.ChatType chatType, final String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		String chatTypeStr = "chat";
		if (EMMessage.ChatType.Chat == chatType) {
			chatTypeStr = "chat";
		} else if (EMMessage.ChatType.GroupChat == chatType) {
			chatTypeStr = "groupchat";
		}

		Map<String, List<EMAMessageReaction>> emaReactionMap = emaReactionObject.getReactionList(messageIdList, chatTypeStr, groupId, error);
		handleError(error);

		Map<String, List<EMMessageReaction>> results = new HashMap<>(emaReactionMap.size());
		if (emaReactionMap.size() > 0) {
			List<EMMessageReaction> reactionList;
			for (Map.Entry<String, List<EMAMessageReaction>> entry : emaReactionMap.entrySet()) {
				reactionList = new ArrayList<>(entry.getValue().size());
				for (EMAMessageReaction emaMessageReaction : entry.getValue()) {
					if (null != emaMessageReaction) {
						reactionList.add(new EMMessageReaction(emaMessageReaction));
					}
				}
				results.put(entry.getKey(), reactionList);
			}
		}
		return results;
	}

	/**
	 * \~chinese
	 * 获取 Reaction 列表。
	 *
	 * 异步方法。
	 *
	 * @param messageIdList 消息 ID。
	 * @param chatType      会话类型，仅支持单聊（{@link EMMessage.ChatType#Chat} ）和群聊（{@link EMMessage.ChatType#GroupChat}）。
	 * @param groupId       群组 ID，该参数只对群聊生效。
	 * @param callback      处理结果回调，包含消息 ID 对应的 Reaction 列表（EMMessageReaction 的用户列表 `UserList` 为概要数据，只包含前三个用户信息）。
	 *
	 * \~english
	 * Gets the list of Reactions.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageIdList  The message ID.
	 * @param chatType       The conversation type. Only one-to-one chats ({@link EMMessage.ChatType#Chat} and group chats ({@link EMMessage.ChatType#GroupChat}) are allowed.
	 * @param groupId        The group ID, which is invalid only for group chats.
	 * @param callback       The Reaction list under the specified message ID. The `UserList` of `EMMessageReaction` indicates the summary data, which only contains the information of the first three users.
	 */
	public void asyncGetReactionList(final List<String> messageIdList, final EMMessage.ChatType chatType, final String groupId, final EMValueCallBack<Map<String, List<EMMessageReaction>>> callback) {
		EMClient.getInstance().execute(() -> {
			try {
				if (null != callback) {
					callback.onSuccess(getReactionList(messageIdList, chatType, groupId));
				}
			} catch (HyphenateException e) {
				EMLog.e(TAG, "asyncGetReactionList error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				callback.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

	/**
	 * \~chinese
	 * 获取 Reaction 详细信息。
	 *
	 * 同步方法。
	 *
	 * @param messageId    消息 ID。
	 * @param reaction     消息 Reaction 内容。
	 * @param cursor       查询的起始游标位置。
	 * @param pageSize     每页期望获取的 Reaction 数。
	 * @return             获取结果 {@link EMCursorResult}，包含 cursor 和 Reaction 列表。若 `cursor` 为空表示数据已全部获取。
	 *
	 * \~english
	 * Gets the Reaction details.
	 *
	 * This is a synchronous method.
	 *
	 * @param messageId    The message ID.
	 * @param reaction     The Reaction content.
	 * @param cursor       The cursor position from which to start querying data.
	 * @param pageSize     The number of Reactions you expect to get on each page.
	 * @return   The query result {@link EMCursorResult}, which contains the list of obtained Reactions and the cursor for the next query. If the cursor is `null`, all data is obtained.
	 * @throws HyphenateException	A description of the exception, see {@link EMError}.
	 */
	public EMCursorResult<EMMessageReaction> getReactionDetail(final String messageId, final String reaction, final String cursor, final int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAMessageReaction> result = emaReactionObject.getReactionDetail(messageId, reaction, cursor, pageSize, error);
		handleError(error);
		EMCursorResult<EMMessageReaction> cursorResult = new EMCursorResult<>();
		cursorResult.setCursor(result.getCursor());
		if (null != result.getData()) {
			List<EMMessageReaction> messageReactionList = new ArrayList<>(result.getData().size());
			for (EMAMessageReaction emaMessageReaction : result.getData()) {
				if (null != emaMessageReaction) {
					messageReactionList.add(new EMMessageReaction(emaMessageReaction));
				}
			}
			cursorResult.setData(messageReactionList);
		}
		return cursorResult;
	}

	/**
	 * \~chinese
	 * 获取 Reaction 详细信息。
	 *
	 * 异步方法。
	 *
	 * @param messageId   消息 ID。
	 * @param reaction    消息 Reaction。
	 * @param cursor      查询的起始游标位置。
	 * @param pageSize    每页期望获取的 Reaction 数。
	 * @param callback    处理结果回调，包含 cursor 和 EMMessageReaction 列表（仅使用该列表第一个数据即可）。
	 *
	 * \~english
	 * Gets the Reaction details.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageId    The message ID.
	 * @param reaction     The Reaction content.
	 * @param cursor       The cursor position from which to start querying data.
	 * @param pageSize     The number of Reactions you expect to get on each page.
	 * @param callback     The query result {@link EMCursorResult}, which contains the cursor for the next query and EMMessageReaction list (only the first data entry is used).
	 */
	public void asyncGetReactionDetail(final String messageId, final String reaction, final String cursor, final int pageSize, final EMValueCallBack<EMCursorResult<EMMessageReaction>> callback) {
		EMClient.getInstance().execute(() -> {
			try {
				if (null != callback) {
					callback.onSuccess(getReactionDetail(messageId, reaction, cursor, pageSize));
				}
			} catch (HyphenateException e) {
				EMLog.e(TAG, "asyncGetReactionDetail error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				callback.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

}
