/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate;

import com.hyphenate.chat.EMChatRoom;

import java.util.List;
import java.util.Map;

/**
 * \~chinese
 * 聊天室事件监听接口。
 * 
 * 可通过 {@link com.hyphenate.chat.EMChatRoomManager} 添加或移除聊天室事件监听器。
 * 
 * 添加聊天室事件监听器：
 * ```java
 * EMChatRoomManager manager = EMClient.getInstance().chatroomManager();
 * manager.addChatRoomChangeListener(mChatRoomChangeListener);
 * ```
 *
 * 移除聊天室事件监听器：
 * ```java
 * EMChatRoomManager manager = EMClient.getInstance().chatroomManager();
 * manager.removeChatRoomListener(mChatRoomChangeListener);
 * ```
 *
 * \~english
 * The chat room change listener.
 * 
 * You can add or remove a chat room change listener by using {@link com.hyphenate.chat.EMChatRoomManager}.
 * 
 * Add a chat room change listener:
 * ```java
 * EMChatRoomManager manager = EMClient.getInstance().chatroomManager();
 * manager.addChatRoomChangeListener(mChatRoomChangeListener);
 * ```
 *
 * Remove a chat room change listener:
 * ```java
 * EMChatRoomManager manager = EMClient.getInstance().chatroomManager();
 * manager.removeChatRoomListener(mChatRoomChangeListener);
 * ```
 */
public interface EMChatRoomChangeListener {
  /**
   * \~chinese
   * 聊天室被解散。
   * 
   * 聊天室的所有成员会收到该事件。
   * 
   * @param roomId   聊天室 ID。
   * @param roomName 聊天室名称。
   *
   * \~english
   * Occurs when the chat room is destroyed.
   * 
   * All chat room members receive this event.
   * 
   * @param roomId   The chat room ID.
   * @param roomName The chat room name.
   */
  void onChatRoomDestroyed(final String roomId, final String roomName);

  /**
   * \~chinese
   * 有新成员加入聊天室。
   * 
   * 聊天室的所有成员（除新成员外）会收到该事件。
   * 
   * @param roomId      聊天室 ID。
   * @param participant 新成员。
   *
   * \~english
   * Occurs when a member joins the chat room.
   * 
   * All chat room members, except the new member, receive this event.
   * 
   * @param roomId      The chat room ID.
   * @param participant The new member.
   */
  void onMemberJoined(final String roomId, final String participant);

  /**
   * \~chinese
   * 有成员主动退出聊天室。
   * 
   * 聊天室的所有成员（除退出的成员）会收到该事件。
   * 
   * @param roomId      聊天室 ID。
   * @param roomName    聊天室名称。
   * @param participant 退出的成员。
   * 
   * \~english
   * Occurs when a member exits the chat room.
   * 
   * All chat room members, except the member exiting the chat room, receive this event.
   * 
   * @param roomId      The chat room ID.
   * @param roomName    The chat room name.
   * @param participant The member exiting the chat room.
   */
  void onMemberExited(final String roomId, final String roomName, final String participant);

  /**
   * \~chinese
   * 有成员被移出聊天室。
   * 
   * 被移出的成员收到该事件。
   * 
   *
   * @param reason      成员被移出聊天室的原因：
   *                    - xxx BE_KICKED：该用户被聊天室管理员移除；
   *                    - xxx BE_KICKED_FOR_OFFLINE：该用户由于当前设备断网被服务器移出聊天室。
   * 
   * @param roomId      聊天室 ID。
   * @param roomName    聊天室名称。
   * @param participant 被移除人员。
   *
   * \~english
   * Occurs when a member is removed from a chat room.
   * 
   * The member that is kicked out of the chat room receive this event.
   *
   * @param reason      The reason why the member is removed from the chat room:
   *                    - xxx BE_KICKED: The member is removed by the chat room owner.
   *                    - xxx BE_KICKED_FOR_OFFLINE: The member is disconnected from the server, probably due to network interruption.
   * @param roomId      The chat room ID.
   * @param roomName    The chat room name.
   * @param participant The member removed from a chat room.
   */
  void onRemovedFromChatRoom(final int reason, final String roomId, final String roomName, final String participant);

  /**
   * \~chinese
   * 有成员被禁言。
   * 
   * 被添加的成员收到该事件。禁言期间成员不能发送发消息。
   *
   * @param chatRoomId 聊天室 ID。
   * @param mutes      被禁言的成员。
   * @param expireTime 禁言失效的 Unix 时间戳，单位为毫秒。预留参数。
   *
   * \~english
   * Occurs when the chat room member(s) is/are added to the mute list.
   * 
   * The muted members receive this event. The muted members cannot send message during the mute duration.
   *
   * @param chatRoomId The chat room ID.
   * @param mutes      The muted number(s).
   * @param expireTime The Unix timestamp when the mute expires. The unit is millisecond. Reserved parameter.
   */
  void onMuteListAdded(final String chatRoomId, final List<String> mutes, final long expireTime);

  /**
   * \~chinese
   * 有成员从禁言列表中移除。
   * 
   * 被解除禁言的成员会收到该事件。
   *
   * @param chatRoomId 聊天室 ID。
   * @param mutes      从禁言列表中移除的成员名单。
   *
   * \~english
   * Occurs when the chat room member(s) is/are removed from the mute list.
   * 
   * The members that are removed from the mute list receive this event.
   *
   * @param chatRoomId The chat room ID.
   * @param mutes      The member(s) removed from the mute list.
   */
  void onMuteListRemoved(final String chatRoomId, final List<String> mutes);

  /**
   * \~chinese
   * 有成员加入白名单。
   * 
   * 被添加的成员收到该事件。
   *
   * @param chatRoomId 聊天室 ID。
   * @param whitelist  白名单中增加的成员名单。
   *
   * \~english
   * Occurs when the chat room member(s) is/are added to the allow list.
   * 
   * The members added to the allow list receive this event.
   *
   * @param chatRoomId The chat room ID.
   * @param whitelist  The member(s) added to the allow list.
   */
  void onWhiteListAdded(final String chatRoomId, final List<String> whitelist);

  /**
   * \~chinese
   * 有成员被移出白名单。
   * 
   * 被移出白名单的成员会收到该事件。
   *
   * @param chatRoomId 聊天室 ID。
   * @param whitelist  移出白名单的成员。
   *
   * \~english
   * Occurs when the chat room member(s) is/are removed from the allow list.
   * 
   * The members that are removed from the allow list receive this event.
   *
   * @param chatRoomId The chat room ID.
   * @param whitelist  The member(s) removed from the allow list.
   */
  void onWhiteListRemoved(final String chatRoomId, final List<String> whitelist);

  /**
   * \~chinese
   * 全员禁言状态有变更。
   * 
   * 聊天室所有成员会收到该事件。
   *
   * @param chatRoomId 聊天室 ID。
   * @param isMuted    是否开启了全员禁言。
   *
   * \~english
   * Occurs when all members in the chat room are muted or unmuted.
   * 
   * All chat room members receive this event.
   *
   * @param chatRoomId The chat room ID.
   * @param isMuted    Whether all chat room members are muted or unmuted.
   */
  void onAllMemberMuteStateChanged(final String chatRoomId, final boolean isMuted);

  /**
   * \~chinese
   * 有成员被设置为管理员。
   * 
   * 被添加的管理员会收到该事件。
   *
   * @param chatRoomId 聊天室 ID。
   * @param admin      被设置为管理员的成员。
   *
   * \~english
   * Occurs when a chat room member is set as an admin.
   * 
   * The member set as the chat room admin receives this event.
   *
   * @param chatRoomId The chat room ID.
   * @param admin      The chat room member set as an admin.
   */
  void onAdminAdded(final String chatRoomId, final String admin);

  /**
   * \~chinese
   * 有成员被移出管理员列表。
   * 
   * 被移出的管理员会收到该事件。
   *
   * @param chatRoomId 聊天室 ID。
   * @param admin      被移出管理员列表的成员。
   *
   * \~english
   * Occurs when the chat room member(s) is/are removed from the admin list.
   * 
   * The admin removed from the admin list receives this event.
   *
   * @param chatRoomId The chat room ID.
   * @param admin      The member(s) removed from the admin list.
   */
  void onAdminRemoved(final String chatRoomId, final String admin);

  /**
   * \~chinese
   * 聊天室所有者变更。
   * 
   * 聊天室所有成员会收到该事件。
   *
   * @param chatRoomId 聊天室 ID。
   * @param newOwner   新的聊天室所有者。
   * @param oldOwner   原聊天室所有者。
   *
   * \~english
   *  Occurs when the chat room owner is changed.
   * 
   *  The chat room owner receives this event.
   *
   * @param chatRoomId The chat room ID.
   * @param newOwner   The new chat room owner.
   * @param oldOwner   The previous chat room owner.
   */
  void onOwnerChanged(final String chatRoomId, final String newOwner, final String oldOwner);

  /**
   * \~chinese
   * 聊天室公告有变更。
   * 
   * 聊天室的所有成员会收到该事件。
   * 
   * @param chatRoomId   聊天室 ID。
   * @param announcement 更新的公告内容。
   *
   * \~english
   * Occurs when the chat room announcement changes.
   * 
   * All chat room members receive this event.
   * 
   * @param chatRoomId   The chat room ID.
   * @param announcement The modified chat room announcement.
   */
  void onAnnouncementChanged(String chatRoomId, String announcement);

  /**
   * \~chinese
   * 聊天室信息有更新。
   * 
   * 聊天室的所有成员会收到该事件。
   * 
   * @param chatRoom 聊天室对象。
   *
   * \~english
   * Occurs when the chat room specifications changes.
   * 
   * All chat room members receive this event.
   * 
   * @param chatRoom The chat room instance.
   */
  default void onSpecificationChanged(EMChatRoom chatRoom) {}

  /**
   * \~chinese
   * 聊天室自定义属性（key-value）有更新。
   * 
   * 聊天室所有成员会收到该事件。
   *
   * @param chatRoomId   聊天室 ID。
   * @param attributeMap 聊天室自定义属性。
   * @param from         操作者的用户 ID。
   *
   * \~english
   * The custom chat room attribute(s) is/are updated.
   * 
   * All chat room members receive this event.
   *
   * @param chatRoomId   The chat room ID.
   * @param attributeMap The map of custom chat room attributes.
   * @param from         The user ID of the operator.
   */
  default void onAttributesUpdate(String chatRoomId, Map<String, String> attributeMap, String from) {}

  /**
   * \~chinese
   * 聊天室自定义属性被移除。
   * 
   * 聊天室所有成员会收到该事件。
   *
   * @param chatRoomId   聊天室 ID。
   * @param keyList      聊天室自定义属性 key 列表。
   * @param from         操作者用户 ID。
   *
   * \~english
   * The custom chat room attribute(s) is/are removed.
   * 
   * All chat room members receive this event.
   *
   * @param chatRoomId   The chat room ID.
   * @param keyList      The list of custom chat room attributes key.
   * @param from         The user ID of the operator.
   */

    default void onAttributesRemoved(String chatRoomId, List<String> keyList , String from){}

}
