/************************************************************
  *  * EaseMob CONFIDENTIAL 
  * __________________ 
  * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
  *  
  * NOTICE: All information contained herein is, and remains 
  * the property of EaseMob Technologies.
  * Dissemination of emaObject information or reproduction of emaObject material 
  * is strictly forbidden unless prior written permission is obtained
  * from EaseMob Technologies.
  */
package com.hyphenate.chat;

import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import com.hyphenate.EMCallBack;
import com.hyphenate.chat.adapter.EMACallback;
import com.hyphenate.chat.adapter.EMAMessageReaction;
import com.hyphenate.chat.adapter.EMAThreadInfo;
import com.hyphenate.chat.adapter.message.EMACmdMessageBody;
import com.hyphenate.chat.adapter.message.EMACustomMessageBody;
import com.hyphenate.chat.adapter.message.EMAFileMessageBody;
import com.hyphenate.chat.adapter.message.EMAImageMessageBody;
import com.hyphenate.chat.adapter.message.EMALocationMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.chat.adapter.message.EMAMessage.EMADirection;
import com.hyphenate.chat.adapter.message.EMAMessage.EMAMessageStatus;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.adapter.message.EMATextMessageBody;
import com.hyphenate.chat.adapter.message.EMAVideoMessageBody;
import com.hyphenate.chat.adapter.message.EMAVoiceMessageBody;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMFileHelper;
import com.hyphenate.util.EMLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

/**
 * \~chinese
 * 消息对象，代表一条发送或接收到的消息。
 * 
 * 例如：
 * 构造一条文本发送消息：
 * ```java
 *     EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
 *     msg.setTo("user1");
 *     TextMessageBody body = new TextMessageBody("hello from hyphenate sdk");
 *     msg.addBody(body);
 * ```
 * 
 * 构造一条图片消息：
 * ```java
 *      EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
 *      msg.setTo("user1");
 *      ImageMessageBody body = new ImageMessageBody(imageFileUrl);
 *      msg.addBody(body);
 * ```
 * 
 * \~english
 * The message instance, which represents a sent/received message.

 * 
 * For example:
 * Constructs a text message to send:
 * ```java
 *     EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
 *     msg.setTo("user1");
 *     TextMessageBody body = new TextMessageBody("hello from hyphenate sdk");
 *     msg.addBody(body);
 * ```
 * 
 * Constructs a new received text message:
 * ```java
 *      EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
 *      msg.setTo("user1");
 *      ImageMessageBody body = new ImageMessageBody(imageFileUrl);
 *      msg.addBody(body);
 * ```
 */
public class EMMessage extends EMBase<EMAMessage> implements Parcelable, Cloneable{
    static final String ATTR_ENCRYPTED = "isencrypted";
    
    private final static String TAG = "msg";

    /**
     * \~chinese
     * 消息类型枚举类。
     * 包含：文本，图片，视频，位置，语音，文件，命令（透传），用户自定义消息。
     * 
     * \~english
     * The enumeration of the message type.
     */
    public enum Type {
       /**
        * \~chinese
        * 文本消息。
        * 
        * \~english
        * Text message.
        */
        TXT, 
       /**
        * \~chinese
        * 图片消息。
        * 
        * \~english
        * Image message.
        */
        IMAGE,
       /**
        * \~chinese
        * 视频消息。
        * 
        * \~english
        * Video message.
        */ 
        VIDEO, 
       /**
        * \~chinese
        * 位置消息。
        * 
        * \~english
        * Location message.
        */ 
        LOCATION, 
        /**
        * \~chinese
        * 语音消息。
        * 
        * \~english
        * Voice message.
        */ 
        VOICE, 
        /**
        * \~chinese
        * 附件消息。
        * 
        * \~english
        * File message.
        */ 
        FILE, 
        /**
        * \~chinese
        * 命令消息（透传消息）。
        * 
        * \~english
        * Command message.
        */ 
        CMD, 
        /**
        * \~chinese
        * 用户自定义消息。
        * 
        * \~english
        * Customized message.
        */ 
        CUSTOM
    }
    
    /**
     * \~chinese
     * 消息的方向类型枚举类。
     * 区分是发送消息还是接收到的消息。
     * 
     * \~english
     * The enumeration of the message direction.
     * Whether the message is sent or received.
     */
    public enum Direct {
      /* \~chinese 该消息是当前用户发送出去的。\~english This message is sent from the local client.*/
        SEND, 
     /* \~chinese 该消息是当前用户接收到的。 \~english The message is received by the local client.*/
        RECEIVE
    }
    
    /**
     * \~chinese
     * 消息的发送/接收状态枚举类。
     * 包括如下状态：成功，失败，发送/接收过程中，创建成功待发送。
     * 
     * \~english
     * The enumeration of the message sending/reception status.
     * The states include success, failure, being sent/being received, and created to be sent.
     */
    public enum Status {
       /** \~chinese 发送/接收成功。 \~english The message is successfully delivered/received.*/

        SUCCESS, 
         /** \~chinese 发送/接收失败。 \~english The message fails to be delivered/received.*/
        FAIL, 
       /** \~chinese 正在发送/接收。 \~english The message is being delivered/receiving.*/
        INPROGRESS,
       /** \~chinese 消息已创建待发送。 \~english The message is created to be sent.*/

        CREATE
    }
    
    EMMessageBody body;
    
    /**
     * \~chinese
     * 获取消息的发送/接收状态。
     * @return 消息的发送/接收状态。
     * 
     * \~english
     * Gets the message sending/reception status.
     * @return  The message sending/reception status.
     */
    public Status status() {
    	EMAMessage.EMAMessageStatus st = emaObject._status();
    	Status s = Status.CREATE;
    	switch (st) {
    	    case NEW :
                s = Status.CREATE;
    	        break;
    	    case SUCCESS :
                s = Status.SUCCESS;
    	        break;
    	    case FAIL :
                s = Status.FAIL;
    	        break;
    	    case DELIVERING :
                s = Status.INPROGRESS;
    	        break;
    	}
    	return s;
    }
    
    /**
     * \~chinese
     * 聊天类型枚举类。
     * 包括：单聊，群聊，聊天室。
     *
     *\~english
     * The enumeration of the chat type.
     * There are three chat types: one-to-one chat, group chat, and chat room.
     */
    public enum ChatType {
        /**
         * \~chinese
         * 单聊。
         *
         * \~english
         * One-to-one chat.
         */
        Chat, 
        /**
         * \~chinese
         * 群聊。
         *
         * \~english
         * Group chat.
         */
        GroupChat,
        /**
         * \~chinese
         * 聊天室。
         *
         * \~english
         * Chat room.
         */
        ChatRoom
    }
    
    public EMMessage(EMAMessage message) {
        emaObject = message;
    }
    
    /**
     * \~chinese
     * 设置消息发送或接收的状态。
     * @param status 消息的状态。
     * 
     * \~english
     * Sets the message sending or reception status.

     * @param status The message sending or reception status.
     */
    public void setStatus(Status status) {
    	EMAMessage.EMAMessageStatus s = EMAMessageStatus.SUCCESS;
    	switch (status) {
            case CREATE:
                s = EMAMessageStatus.NEW;
    	        break;
            case SUCCESS:
                s = EMAMessageStatus.SUCCESS;
    	        break;
            case FAIL:
                s = EMAMessageStatus.FAIL;
    	        break;
            case INPROGRESS:
                s = EMAMessageStatus.DELIVERING;
    	        break;
    	}
    	emaObject.setStatus(s.ordinal());
    }
    
    /**
     * \chinese
     * 获取消息类型。
     * @return  消息类型。
     * 
     * \~english
     * Gets the chat message type.
     * @return  The chat message type.
     */
    public Type getType() {
    	List<EMAMessageBody> bodies = emaObject.bodies();
    	if (bodies.size() > 0) {
    		int type = bodies.get(0).type();
    		//Log.d(TAG, "type:" + type);
    		if (type == Type.TXT.ordinal()) {
    			return Type.TXT;
    		} else if (type == Type.IMAGE.ordinal()) {
    			return Type.IMAGE;
    		} else if (type == Type.CMD.ordinal()) {
    			return Type.CMD;
    		} else if (type == Type.FILE.ordinal()) {
    			return Type.FILE;
    		} else if (type == Type.VIDEO.ordinal()) {
    			return Type.VIDEO;
    		} else if (type == Type.VOICE.ordinal()) {
    			return Type.VOICE;
    		} else if (type == Type.LOCATION.ordinal()) {
    			return Type.LOCATION;
    		} else if (type == Type.CUSTOM.ordinal()) {
    		    return Type.CUSTOM;
            }
    	}
        return Type.TXT;
    }
    
    /**
     * \~chinese
     * 获取消息体。
     * @return  消息体。
     * 
     * \~english
     * Gets the message body.
     * @return  The message body.
     */
    public EMMessageBody getBody() {
        if (body != null) {
            return body;
        }
    	List<EMAMessageBody> bodies = emaObject.bodies();
    	if (bodies.size() > 0) {
    		EMAMessageBody emaBody = bodies.get(0);
    		if (emaBody instanceof EMATextMessageBody) {
    			body = new EMTextMessageBody((EMATextMessageBody)emaBody);
    		} else if (emaBody instanceof EMACmdMessageBody) {
    			body = new EMCmdMessageBody((EMACmdMessageBody)emaBody);
    		} else if (emaBody instanceof EMAVideoMessageBody) {
    			body = new EMVideoMessageBody((EMAVideoMessageBody)emaBody);
    		} else if (emaBody instanceof EMAVoiceMessageBody) {
    			body = new EMVoiceMessageBody((EMAVoiceMessageBody)emaBody);
    		} else if (emaBody instanceof EMAImageMessageBody) {
    			body = new EMImageMessageBody((EMAImageMessageBody)emaBody);
    		} else if (emaBody instanceof EMALocationMessageBody) {
    			body = new EMLocationMessageBody((EMALocationMessageBody)emaBody);
    		} else if (emaBody instanceof EMAFileMessageBody) {
    			body = new EMNormalFileMessageBody((EMAFileMessageBody)emaBody);
    		} else if (emaBody instanceof EMACustomMessageBody) {
    		    body = new EMCustomMessageBody((EMACustomMessageBody) emaBody);
            }
        	return body;
    	}

    	return null;
    }
    
    /**
     * \~chinese
     * 获取消息的时间戳(服务器时间)。
     * @return  消息的时间戳(服务器时间)。
     * 
     * \~english
     * Gets the Unix timestamp (server time) of the message.
     * @return  The Unix timestamp (server time) of the message.
     */
    public long getMsgTime() {
    	return emaObject.timeStamp();
    }
    
    
    /**
     * \~chinese
     * 设置消息的时间戳(服务器时间)。
     * @param msgTime   消息的时间戳(服务器时间)。
     * 
     * \~english
     * Sets the Unix timestamp (server time) of the message.
     * @param msgTime   The Unix timestamp (server time) of the message.
     */
    public void setMsgTime(long msgTime) {
		emaObject.setTimeStamp(msgTime);
	}
    
    /**
     * \~chinese
     * 获取消息的本地时间戳。
     * @return  消息的本地时间戳。
     * 
     * \~english
     * Gets the local timestamp of the message.
     * @return The local timestamp of the message.
     */
    public long localTime() {
        return emaObject.getLocalTime();
    }
    
    /**
     * \~chinese
     * 设置消息的本地时间戳。
     * @param serverTime    消息的本地时间戳。
     * 
     * \~english
     * Sets the local timestamp of the message.
     * @param serverTime    The local timestamp of the message.
     */
    public void setLocalTime(long serverTime) {
        emaObject.setLocalTime(serverTime);
    }

    /**
     * \~chinese
     * 消息是否需要群组已读回执。
     * @return - `true`：需要已读回执；
     * - `false`：不需要已读回执。
     *
     * \~english
     * Sets whether read receipts are required for group messages.
     * @return - `true`: Read receipts are required; 
     * - `false`: Read receipts are NOT required.
     */
    public boolean isNeedGroupAck() {return emaObject.isNeedGroupAck(); }


    /**
     * \~chinese
     * 设置消息是否需要群组已读回执。
     * @param need - `true`：需要已读回执；
     * - `false`：不需要。
     *
     * \~english
     * Sets whether read receipts are required for group messages.
     * @param need - `true`: Read receipts are required; 
     * - `false`: Read receipts are NOT required.
     */
    public void setIsNeedGroupAck(boolean need) {emaObject.setIsNeedGroupAck(need); }

    /**
     * \~chinese
     * 群消息已读人数。
     * @return 消息已读人数。
     *
     * \~english
     * Gets the number of members that have read the group message.
     * @return The number of members that have read the group message.
     */
    public int groupAckCount() { return emaObject.groupAckCount(); }

    /**
     * \~chinese
     * 设置群消息已读人数。
     * @param count  群消息已读人数。
     *
     * \~english
     * Sets the number of members that have read the group message.
     * @param count  The number of members that have read the group message.
     */
    public void setGroupAckCount(int count) { emaObject.setGroupAckCount(count); }

    /**
     * \~chinese
     * 设置是否是 Thread 消息
     * @param isChatThreadMessage
     */
    public void setIsChatThreadMessage(boolean isChatThreadMessage) {
        emaObject.setIsChatThreadMessage(isChatThreadMessage);
    }

    /**
     * \~chinese
     * 获取消息是否是 Thread 消息
     * @return
     */
    public boolean isChatThreadMessage() {
        return emaObject.isChatThreadMessage();
    }

    /**
     * \~chinese
     * 获取子区概览信息，仅当消息创建子区后携带。
     * @return 子区概览信息。
     *
     * \~chinese
     * Get Chat Thread overview information, which is carried only after Chat Thread is created in the message.
     * @return Thread overview information.
     */
    public EMChatThread getChatThread() {
        EMAThreadInfo thread = emaObject.getChatThread();
        if(thread == null) {
            return null;
        }
        return new EMChatThread(thread);
    }

	/**
	 * \~chinese
     * 创建一个发送消息。
     * @param type 消息类型。
     * @return 消息对象。
     * 
     * \~english
     * Creates a message instance for sending.
     * @param type The message type.
     * @return The message instance.
     */
    public static EMMessage createSendMessage(Type type) {
    	EMAMessage _msg = EMAMessage.createSendMessage(self(), "", null, ChatType.Chat.ordinal());
        EMMessage msg = new EMMessage(_msg);
        return msg;
    }
    
    /**
     * \~chinese
     * 创建一个接收消息。
     * @param type 消息类型。
     * @return 消息对象。
     * 
     * \~english
     * Creates a new received message instance.
     * @param type The message type.
     * @return The message instance.
     */
    public static EMMessage createReceiveMessage(Type type) {
    	EMAMessage _msg = EMAMessage.createReceiveMessage("", self(), null, ChatType.Chat.ordinal());
        EMMessage msg = new EMMessage(_msg);
        msg.setTo(EMSessionManager.getInstance().currentUser.getUsername());
        return msg;
    }
    
    /**
     * \~chinese
     * 创建一个文本发送消息。
     * @param content 文本内容。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     * @deprecated 使用 {@link #createTextSendMessage(String, String)} 代替。
     * 
     * \~english
     * Creates a text message for sending.
     * @param content The text content.
     * @param username The ID of the message recipient(user or group).
     * @return The message instance.
     * @deprecated Use {@link #createTextSendMessage(String, String)} to replace.
     */
    @Deprecated
    public static EMMessage createTxtSendMessage(String content, String username){
        if (content.length() > 0) {
        	EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
        	EMTextMessageBody txtBody = new EMTextMessageBody(content);
        	msg.addBody(txtBody);
        	msg.setTo(username);
            return msg;
        }
        EMLog.e(TAG, "text content size must be greater than 0");
        return null;
    }

    /**
     * \~chinese
     * 创建一个文本发送消息。
     * @param content 文本内容。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a text message for sending.
     * @param content The text content.
     * @param username The ID of the message recipient(user or group).
     * @return The message instance.
     */
    public static EMMessage createTextSendMessage(String content, String username){
        if (content.length() > 0) {
        	EMMessage msg = EMMessage.createSendMessage(EMMessage.Type.TXT);
        	EMTextMessageBody txtBody = new EMTextMessageBody(content);
        	msg.addBody(txtBody);
        	msg.setTo(username);
            return msg;
        }
        EMLog.e(TAG, "text content size must be greater than 0");
        return null;
    }

    /**
     * \~chinese
     * 创建一个语音发送消息。
     * @param filePath 语音文件路径。
     * @param timeLength 语音时间长度(单位秒)。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates a voice message for sending.
     * @param filePath The path of the voice file.
     * @param timeLength The voice duration in seconds.
     * @param username  The message recipient (user or group) ID.
     * @return The message instance.
     *
     */
    public static EMMessage createVoiceSendMessage(String filePath, int timeLength, String username){
        return createVoiceSendMessage(EMFileHelper.getInstance().formatInUri(filePath), timeLength, username);
    }

    /**
     * \~chinese
     * 创建一个语音发送消息。
     * @param filePath 语音文件 URI。
     * @param timeLength 语音时间长度(单位秒)。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a voice message for sending.
     * @param filePath The URI of the voice file.
     * @param timeLength The voice duration in seconds.
     * @param username  The message recipient (user or group) ID.
     * @return The message instance.
     */
    public static EMMessage createVoiceSendMessage(Uri filePath, int timeLength, String username){
        if (!EMFileHelper.getInstance().isFileExist(filePath)) {
            EMLog.e(TAG, "voice file does not exsit");
            return null;
        }
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.VOICE);
        // 如果是群聊，需设置 chattype，默认是单聊。
        EMVoiceMessageBody body = new EMVoiceMessageBody(filePath, timeLength);
        message.addBody(body);

        message.setTo(username);
        return message;
    }

    /**
     * \~chinese
     * 创建一个图片发送消息。
     * @param filePath 图片路径。
     * @param sendOriginalImage 是否发送原图。
     * -  (默认）`false`：大于 100K 的图片 SDK 会进行压缩；
     * - `true`：发送原图。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates an image message for sending.
     * @param filePath The image path.
     * @param sendOriginalImage Whether to send the original image.
     * -  (Default)`false`: For an image greater than 100 KB, the SDK will compress it.
     * - `true`: Send the original image.
     * @param username The message recipient (user or group) ID.
     * @return The message instance.
     *
     */
    public static EMMessage createImageSendMessage(String filePath, boolean sendOriginalImage, String username){
        return createImageSendMessage(EMFileHelper.getInstance().formatInUri(filePath), sendOriginalImage, username);
    }

    /**
     * \~chinese
     * 创建一个图片发送消息。
     * @param imgUri 图片 URI。
     * @param sendOriginalImage 是否发送原图。 
     * -  (默认）`false`: 大于 100K 的图片 SDK 会进行压缩；
     * - `true`: 发送原图。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates an image message for sending.
     * @param imgUri The URI of the image.
     * @param sendOriginalImage Whether to send the original.
     * - (Default)`false`: For an image greater than 100 KB, the SDK will compress it.
     * - `true`: Send the original image.
     * @param username The recipient(user or group) ID.
     * @return The message instance.
     */
    public static EMMessage createImageSendMessage(Uri imgUri, boolean sendOriginalImage, String username){
        if(!EMFileHelper.getInstance().isFileExist(imgUri)) {
            EMLog.e(TAG, "image file does not exsit");
            return null;
        }
        // create and add the image message in view.
        final EMMessage message = EMMessage.createSendMessage(EMMessage.Type.IMAGE);
        message.setTo(username);
        EMImageMessageBody body = new EMImageMessageBody(imgUri);
        // 默认超过 100k 的图片会压缩后发给对方，可以设置成发送原图。
        body.setSendOriginalImage(sendOriginalImage);
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * 创建一个视频发送消息。
     * @param videofilePath 视频文件路径。
     * @param imageThumbPath 视频第一帧图缩略图。
     * @param timeLength 视频时间长度(单位秒)。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates a video message instance for sending.
     * @param videofilePath The path of the video file.
     * @param imageThumbPath The path of the thumbnail.
     * @param timeLength The video duration in seconds.
     * @param username The message recipient(user or group) ID.
     * @return The message instance.
     *
     */
    public static EMMessage createVideoSendMessage(String videofilePath, String imageThumbPath, int timeLength, String username){
        return createVideoSendMessage(EMFileHelper.getInstance().formatInUri(videofilePath)
                , EMFileHelper.getInstance().formatInUri(imageThumbPath)
                , timeLength
                , username);
    }

    /**
     * \~chinese
     * 创建一个视频发送消息。
     * @param videofilePath 视频文件路径。
     * @param imageThumbPath 视频第一帧图缩略图。
     * @param timeLength 视频时间长度(单位秒)。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a video message instance for sending.
     * @param videofilePath The path of the video file.
     * @param imageThumbPath The path of the thumbnail of the first frame of video.
     * @param timeLength The video duration in seconds.
     * @param username The message recipient(user or group) ID.
     * @return The message instance.
     */
    public static EMMessage createVideoSendMessage(Uri videofilePath, String imageThumbPath, int timeLength, String username){
        return createVideoSendMessage(videofilePath, EMFileHelper.getInstance().formatInUri(imageThumbPath), timeLength, username);
    }

    /**
     * \~chinese
     * 创建一个视频发送消息。
     * @param videofilePath 视频文件路径。
     * @param imageThumbPath 视频第一帧图缩略图。
     * @param timeLength 视频时间长度(单位秒)。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a video message instance for sending.
     * @param videofilePath The path of the video file.
     * @param imageThumbPath The path of the thumbnail of the first frame of video.
     * @param timeLength The video duration in seconds.
     * @param username The message recipient(user or group) ID.
     * @return  The message instance.
     */
    public static EMMessage createVideoSendMessage(Uri videofilePath, Uri imageThumbPath, int timeLength, String username){
        if (!EMFileHelper.getInstance().isFileExist(videofilePath)) {
            EMLog.e(TAG, "video file does not exist");
            return null;
        }
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.VIDEO);
        message.setTo(username);
        EMVideoMessageBody body = new EMVideoMessageBody(videofilePath, imageThumbPath, timeLength
                , EMFileHelper.getInstance().getFileLength(videofilePath));
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * 创建一个位置发送消息。
     * @param latitude 纬度。
     * @param longitude 经度。
     * @param locationAddress 位置详情。
     * @param buildingName 建筑物名称。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates a location message for sending.
     * @param latitude The latitude.
     * @param longitude The longitude.
     * @param locationAddress The location details.
     * @param buildingName The name of the building.
     * @param username The message recipient (user or group) ID.
     * @return The message instance.
     */
    public static EMMessage createLocationSendMessage(double latitude, double longitude, String locationAddress, String buildingName, String username){
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.LOCATION);
        EMLocationMessageBody locBody = new EMLocationMessageBody(locationAddress, latitude, longitude, buildingName);
        message.addBody(locBody);
        message.setTo(username);
        
        return message;
    }

    /**
     * \~chinese
     * 创建一个位置发送消息。
     * @param latitude 纬度。
     * @param longitude 经度。
     * @param locationAddress 位置详情。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a location message for sending.
     * @param latitude The latitude.
     * @param longitude The longitude.
     * @param locationAddress The location details.
     * @param username The message recipient (user or group) ID.
     * @return The message instance.
     */
    public static EMMessage createLocationSendMessage(double latitude, double longitude, String locationAddress, String username){
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.LOCATION);
        EMLocationMessageBody locBody = new EMLocationMessageBody(locationAddress, latitude, longitude);
        message.addBody(locBody);
        message.setTo(username);

        return message;
    }
    
    /**
     * \~chinese
     * 创建一个普通文件发送消息。
     * @param filePath 文件路径。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     * 
     * \~english
     * Creates a message to send a regular file.
     * @param filePath The file path.
     * @param username The message recipient (user or group) ID.
     * @return The message instance.
     */
    public static EMMessage createFileSendMessage(String filePath, String username){
        return createFileSendMessage(EMFileHelper.getInstance().formatInUri(filePath), username);
    }

    /**
     * \~chinese
     * 创建一个普通文件发送消息。
     * @param filePath 文件路径。
     * @param username 消息接收人或群 ID。
     * @return 消息对象。
     *
     * \~english
     * Creates a message instance to send a regular file.
     * @param filePath The file path.
     * @param username The message recipient (user or group) ID.

     * @return The message instance.
     */
    public static EMMessage createFileSendMessage(Uri filePath, String username){
        if(!EMFileHelper.getInstance().isFileExist(filePath)){
            EMLog.e(TAG, "file does not exist");
            return null;
        }
        // 创建一个文件消息
        EMMessage message = EMMessage.createSendMessage(EMMessage.Type.FILE);

        message.setTo(username);
        // add message body
        EMNormalFileMessageBody body = new EMNormalFileMessageBody(filePath);
        message.addBody(body);
        return message;
    }

    /**
     * \~chinese
     * （建议方法）设置消息体。
     * @param body 消息体。
     *
     * \~english
     * （Recommended）Sets the message body. 
     * @param body The message body.
     */
    public void setBody(EMMessageBody body) {
        if(this.body != body) {
            this.body = body;
            emaObject.clearBodies();
            emaObject.addBody(body.emaObject);
        }
    }

    /**
     * \~chinese
     * 添加消息体。与（建议方法）设置消息体 {@link EMMessage#setBody(com.hyphenate.chat.EMMessageBody)} 的方法效果一样。
     * 只支持一条消息添加一个消息体。
     * @param body 消息体。
     * 
     * \~english
     * Adds a message body. We recommend you use {@link EMMessage#setBody(com.hyphenate.chat.EMMessageBody)}.
     * Only support to add one now.
     * @param body The message body.
     */
    public void addBody(EMMessageBody body) {
        this.body = body;
        emaObject.addBody(body.emaObject);
    }
    
    /**
     * \~chinese
     * 获取消息发送者的用户名。
     * @return 用户名。
     * 
     * \~english
     * Gets the user ID of the message sender.
     * @return  The user ID.
     */
    public String getFrom() {
        return emaObject.from();
    }
    
    /**
     * \~chinese
     * 设置消息发送者 ID。
     * @param from  发送者 ID。
     * 
     * \~english
     * Sets the user ID of the message sender.
     * @param from The user ID of the message sender.
     */
    public void setFrom(String from) {
		emaObject.setFrom(from);
		// The reason for using "getTo ()" here is to prevent "from" as "conversationId" when the message is a group type
		if (!self().equals(from) && getTo() != null && getTo() != "" && getTo().equals(self())) {
			emaObject.setConversationId(from);
		}
	}

    /**
     * \~chinese
     * 获取消息撤回者的用户 ID。
     * @return 撤回者用户 ID。
     *
     * \~english
     * Gets the ID of the user that recalls the message.
     * @return  The ID of the user that recalls the message.
     */
    public String getRecaller(){
        return emaObject.getRecaller();
    }


    /**
     * \~chinese
     * 设置消息的接收者用户 ID。
     * @param to    消息接收者用户 ID。
     * 
     * \~english
     * Sets the user ID of the message recipient.
     * @param to    The user ID of the message recipient.
     */
	public void setTo(String to) {
        emaObject.setTo(to);
        // Removing the logic to check if it's you is to prevent you from being able to message yourself
        emaObject.setConversationId(to);
	}

	/**
	 * \~chinese
     * 获取消息接收者的用户名。
     * @return  接收者的用户名。
     * 
     * \~english
     * Gets the user ID of the message recipient.
     * @return  The user ID of the message recipient.
     */
    public String getTo() {
        return emaObject.to();
    }
    
    /**
     * \~chinese
     * 获取消息的 ID。
     * @return 消息的 ID。
     * 
     * \~english
     * Gets the message ID.
     * @return  The message ID.
     */
    public String getMsgId() {
        return emaObject.msgId();
    }
    
    /**
     * \~chinese
     * 设置本地消息 ID。
     * @param msgId 消息 ID。
     * 
     * \~english
     * Sets the local message ID.
     * @param msgId  Sets the local message ID.
     */
    public void setMsgId(String msgId){
    	emaObject.setMsgId(msgId);
    }

    /**
     * The XXXListener doesn't need CallbackHolder, you will be responsible for adding and removing the listener reference.
     * The message's callback function is different from listener, user add callback, and won't remove the reference.
     * Old implementation for Message's callback is simply hold the reference, which leads to memory leak.
     * SDK caches hold reference of message, message hold callback, callback hold Activity's this reference, so activities will never be released.
     *
     * Solution:
     * 1. change the message's callback to weak reference.
     * 2. when sendMessage or downloadMessageAttachment make the reference be strong one.
     * 3. when callback be executed, onSuccess or onError, release the reference.
     *
     * Declaim:
     * 1. don't set message be strong when user call setMessageStatusCallback, user may just attach the callback, there will not be onError or onSuccess to be called
     */
    static class EMCallbackHolder implements EMCallBack {

        EMCallbackHolder(EMCallBack callback) {
            weak = new WeakReference<EMCallBack>(callback);
        }

        synchronized void update(EMCallBack callback) {
            if (strong != null) {
                strong = callback;
                return;
            }
            weak = new WeakReference<EMCallBack>(callback);
        }

        synchronized void makeItStrong() {
            if (strong != null) {
                return;
            }
            if (weak != null && weak.get() != null) {
                strong = weak.get();
            }
        }

        synchronized void release() {
            if (strong == null) {
                return;
            }
            weak = new WeakReference<EMCallBack>(strong);
            strong = null;
        }

        synchronized EMCallBack getRef() {
            if (strong != null) {
                return strong;
            }
            if (weak != null) {
                EMCallBack callBack = weak.get();
                if (callBack == null) {
                    EMLog.d(TAG, "getRef weak:" + callBack);
                }
                return callBack;
            }
            return null;
        }

        public void onSuccess() {
            if (innerCallback != null) {
                innerCallback.onSuccess();
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onSuccess();
                release();
            } else {
                EMLog.d(TAG, "CallbackHolder getRef: null");
            }
        }

        public void onError(int code, String error) {
            if (innerCallback != null) {
                innerCallback.onError(code, error);
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onError(code, error);
                release();
            } else {
                EMLog.d(TAG, "CallbackHolder getRef: null");
            }
        }

        public void onProgress(int progress, String status) {
            if (innerCallback != null) {
                innerCallback.onProgress(progress, status);
            }
            EMCallBack callback = getRef();
            if (callback != null) {
                callback.onProgress(progress, status);
            }
        }

        // strong & weak are used for external callback reference
        // inner callback is used by chat manager sendMessage, to handle changes of msgId and msg time
        private EMCallBack strong;
        private WeakReference<EMCallBack> weak;
        EMCallBack innerCallback = null;
    }

    EMCallbackHolder messageStatusCallBack;

    synchronized void setInnerCallback(EMCallBack callback) {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.innerCallback = callback;
        } else {
            messageStatusCallBack = new EMCallbackHolder(null);
            messageStatusCallBack.innerCallback = callback;
        }
        setCallback(messageStatusCallBack);
    }

    /**
     * \~chinese
     * 设置消息状态改变的回调。
     * 设置消息状态回调，以便刷新界面。
     * @param callback  消息状态改变的回调。
     * 
     * \~english
     * Sets the message status change callback.
     * Your app should set emaObject callback to get the message status and then refresh the UI accordingly.
     * @param callback  The callback when the message status changed.
     */
    public synchronized void setMessageStatusCallback(EMCallBack callback) {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.update(callback);
        } else {
            messageStatusCallBack = new EMCallbackHolder(callback);
        }
        setCallback(messageStatusCallBack);
    }


    void setCallback(final EMCallbackHolder holder) {
        this.emaObject.setCallback(new EMACallback(new EMCallBack(){

            @Override
            public void onSuccess() {
                if(holder != null){
                    holder.onSuccess();
                }
            }

            @Override
            public void onProgress(int progress, String status) {
                if(holder != null){
                    holder.onProgress(progress, status);
                }
            }

            @Override
            public void onError(int code, String error) {
                if(holder != null){
                    holder.onError(code, error);
                }
            }
        }));
    }

    void makeCallbackStrong() {
        EMCallbackHolder callbackHolder = messageStatusCallBack;
        if (callbackHolder != null) {
            callbackHolder.makeItStrong();
        }
    }


    public String toString() {
        String sb = "msg{from:" + getFrom() +
                ", to:" + getTo() +
                " body:" + getBody();
        return sb;
    }
    
    /**
     * \~chinese
     * 设置消息的 boolean 类型扩展属性。
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message's extension attribute of the boolean type.
     * @param attribute  The attribute name.
     * @param value  The attribute value.
     */
    public void setAttribute(String attribute, boolean value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }
    
    /**
     * \~chinese
     * 设置消息的 int 类型扩展属性。
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the integer type.
     * @param attribute The attribute name.
     * @param value The attribute value.
     */
    public void setAttribute(String attribute, int value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }


    /**
     * \~chinese
     * 设置消息的 long 类型扩展属性。
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the long type.
     * @param attribute  The attribute name.
     * @param value  The attribute value.
     */
    public void setAttribute(String attribute, long value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }

    /**
     * \~chinese
     * 设置消息的 float 类型扩展属性。
     * @param attribute 属性名。
     * @param value  属性值。
     *
     * \~english
     * Sets a message extension attribute of the float type.
     * @param attribute  The attribute name.
     * @param value  The attribute value.
     */
    public void setAttribute(String attribute, float value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }

    /**
     * \~chinese
     * 设置消息的 double 类型扩展属性。
     * @param attribute 属性名。
     * @param value  属性值。
     *
     * \~english
     * Sets a message extension attribute of the double type.
     * @param attribute  The attribute name.
     * @param value  The attribute value.
     */
    public void setAttribute(String attribute, double value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }

    
    /**
     * \~chinese
     * 设置消息的 JSONObject 类型扩展属性。
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the JsonObject type.
     * @param attribute  The attribute name.
     * @param value The attribute value.
     */
    public void setAttribute(String attribute, JSONObject value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setJsonAttribute(attribute, value.toString());
    }
    
    /**
     * \~chinese
     * 设置消息的 JSONArray 类型扩展属性。
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a message extension attribute of the JSONArray type.
     * @param attribute The attribute name.
     * @param value The attribute value.
     */
    public void setAttribute(String attribute, JSONArray value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setJsonAttribute(attribute, value.toString());
    }
    
    /**
     * \~chinese
     * 设置消息的 string 类型扩展属性。
     * @param attribute 属性名。
     * @param value  属性值。
     * 
     * \~english
     * Sets a string type extension attributes of the message.
     * @param attribute The attribute name.
     * @param value The attribute value.
     */
    public void setAttribute(String attribute, String value) {
        if (attribute == null || attribute.equals("")) {
            return;
        }
        emaObject.setAttribute(attribute, value);
    }
    
    /**
     * \~chinese
     * 获取 boolean 类型的消息扩展属性。
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。
     * 
     * \~english
     * Gets a message extension attribute of the boolean type.
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException  When exceptions occur, you can troubleshoot issues by referring to the error code and the error description.
     */
    public boolean getBooleanAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicBoolean val = new AtomicBoolean();
        boolean exists = emaObject.getBooleanAttribute(attribute, false, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取 boolean 类型扩展属性。
     * @param attribute 属性名。
     * @param defaultValue 默认值。
     * @return 属性值。
     * 
     * \~english
     * Gets a message extension attribute of the boolean type.
     * @param attribute The attribute name.
     * @param defaultValue  The default value you want.
     * @return The attribute value.
     */
    public boolean getBooleanAttribute(String attribute, boolean defaultValue){
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicBoolean val = new AtomicBoolean();
        boolean exists = emaObject.getBooleanAttribute(attribute, false, val);
        if (!exists) {
            return defaultValue;
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取消息的 int 类型扩展属性。
     * @param attribute 属性名。
     * @param defaultValue 默认值。
     * @return 属性值。
     * 
     * \~english
     * Gets a message extension attribute of the integer type.
     * @param attribute The attribute name.
     * @param defaultValue The default value you want.
     * @return The attribute value.
     */
    public int getIntAttribute(String attribute,int defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicInteger val = new AtomicInteger();
        boolean exists = emaObject.getIntAttribute(attribute, -1, val);
        if (!exists) {
            return defaultValue;
        }
        return val.intValue();
    }
    
    /**
     * \~chinese
     * 获取消息的 long 类型扩展属性。
     * @param attribute 属性名。
     * @param defaultValue 默认值。
     * @return 属性值。
     * 
     * \~english
     * Gets a message extension attribute of the long integer type.
     * @param attribute The attribute name.
     * @param defaultValue The default value you want.
     * @return The attribute value.
     */
    public long getLongAttribute(String attribute,long defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicLong val = new AtomicLong();
        boolean exists = emaObject.getLongAttribute(attribute, -1, val);
        if (!exists) {
            return defaultValue;
        }
        return val.longValue();
    }

    /**
     * \~chinese
     * 获取消息的 float 类型扩展属性。
     * @param attribute 属性名。
     * @param defaultValue 默认值。
     * @return 属性值。
     *
     * \~english
     * Gets a message extension attribute of the float type.
     * @param attribute The attribute name.
     * @param defaultValue The default value you want.
     * @return The attribute value.
     */
    public float getFloatAttribute(String attribute,float defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicReference<Float> val =new AtomicReference();
        boolean exists = emaObject.getFloatAttribute(attribute, defaultValue, val);
        if (!exists) {
            return defaultValue;
        }
        return val.get();
    }
    /**
     * \~chinese
     * 获取消息的 double 类型扩展属性。
     * @param attribute 属性名。
     * @param defaultValue 默认值。
     * @return 属性值。
     *
     * \~english
     * Gets a message extension attribute of the double type.
     * @param attribute The attribute name.
     * @param defaultValue The default value you want.
     * @return The attribute value.
     */
    public double getDoubleAttribute(String attribute,double defaultValue) {
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        AtomicReference<Double> val =new AtomicReference<>();
        boolean exists = emaObject.getDoubleAttribute(attribute, defaultValue, val);
        if (!exists) {
            return defaultValue;
        }
        return val.get();
    }
    
    /**
     * \~chinese
     * 获取消息的 int 类型扩展属性。
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。
     * 
     * \~english
     * Gets an extension attribute of the integer type.
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException When exceptions occur, you can troubleshoot issues by referring to the error code and the error description.
     */
    public int getIntAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicInteger val = new AtomicInteger();
        boolean exists = emaObject.getIntAttribute(attribute, -1, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }        
        return val.intValue();
    }
    
    /**
     * \~chinese
     * 获取 long 类型扩展属性
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。
     * 
     * \~english
     * Gets a message extension attribute of the long integer type.
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException When exceptions occur, you can troubleshoot issues by referring to the error code and the error description.
     */
    public long getLongAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicLong val = new AtomicLong();
        boolean exists = emaObject.getLongAttribute(attribute, -1, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }        
        return val.longValue();
    }
    /**
     * \~chinese
     * 获取 float 类型扩展属性
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。
     *
     * \~english
     * Gets a message extension attribute of the float type.
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException When exceptions occur, you can troubleshoot issues by referring to the error code and the error description.
     */
    public float getFloatAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicReference<Float> val =new AtomicReference();
        boolean exists = emaObject.getFloatAttribute(attribute, -1f, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }
        return val.get();
    }
    /**
     * \~chinese
     * 获取 double 类型扩展属性
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。
     *
     * \~english
     * Gets a message extension attribute of the double type.
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException When exceptions occur, you can troubleshoot issues by referring to the error code and the error description.
     */
    public double getDoubleAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        AtomicReference<Double> val =new AtomicReference<>();
        boolean exists = emaObject.getDoubleAttribute(attribute, -1d, val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }
        return val.get();
    }
    /**
     * \~chinese
     * 获取 String 类型扩展属性。
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。
     * 
     * \~english
     * Gets a string type extension attribute
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException The error code and the description of the cause of exception.
     */
    public String getStringAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            throw new HyphenateException("attribute " + attribute + " can not be null or empty");
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getStringAttribute(attribute, "", val);
        if (!exists) {
            throw new HyphenateException("attribute " + attribute + " not found");
        }        
        return val.toString();
    }
    
    /**
     * \~chinese
     * 获取 String 类型扩展属性。
     * @param attribute 属性名。
     * @param defaultValue 默认值。
     * @return 属性值。
     * 
     * \~english
     * Gets a string type extension attribute
     * @param attribute The attribute name.
     * @param defaultValue The default value you want.
     * @return The attribute value.
     */
    public String getStringAttribute(String attribute, String defaultValue){
        if (attribute == null || attribute.equals("")) {
            return defaultValue;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getStringAttribute(attribute, "", val);
        if (!exists) {
            return defaultValue;
        }        
        return val.toString();
    }
    
    
    /**
     * \~chinese
     * 获取 JSONObject 类型扩展属性
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。
     * 
     * \~english
     * Gets the JSONObject type attribute.
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException When exceptions occur, you can troubleshoot issues by referring to the error code and the error description.
     */
    public JSONObject getJSONObjectAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            return null;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getJsonAttribute(attribute, "{}", val);
        if (exists) {
            try {
                JSONObject jsonObj = new JSONObject(val.toString());
                return jsonObj;
            } catch (JSONException e) {
            }
        }
        throw new HyphenateException("attribute " + attribute + " not found");
    }
    
    /**
     * \~chinese
     * 获取 JSONArray 类型扩展属性。
     * @param attribute 属性名。
     * @return 属性值。
     * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和异常原因。
     * 
     * \~english
     * Gets a JSONArray type attribute.
     * @param attribute The attribute name.
     * @return The attribute value.
     * @throws HyphenateException When exceptions occur, you can troubleshoot issues by referring to the error code and the error description.
     */
    public JSONArray getJSONArrayAttribute(String attribute) throws HyphenateException {
        if (attribute == null || attribute.equals("")) {
            return null;
        }
        StringBuilder val = new StringBuilder();
        boolean exists = emaObject.getJsonAttribute(attribute, "[]", val);
        if (exists) {
            try {
                JSONArray jsonArray = new JSONArray(val.toString());
                return jsonArray;
            } catch (JSONException e) {
            }
        }
        throw new HyphenateException("attribute " + attribute + " not found");
    }
    
    /**
     * \~chinese
     * 获取聊天类型。
     * @return  聊天类型。
     * 
     * \~english
     * Gets the chat type.
     * @return The chat type.
     */
    public ChatType getChatType() {
    	EMAMessage.EMAChatType t = emaObject.chatType();
    	ChatType type = ChatType.Chat;
    	if (t == EMAMessage.EMAChatType.SINGLE) {
    		type = ChatType.Chat;
    	} else if (t == EMAMessage.EMAChatType.GROUP) {
    		type = ChatType.GroupChat;
    	} else {
    		type = ChatType.ChatRoom;
    	}
    	return type;
    }
    
    /**
     * \~chinese
     * 设置聊天类型。
     * 默认为单聊 {@link ChatType#Chat}。
     * @param chatType  聊天类型，详见 {@link ChatType}。
     *
     * \~english
     * Sets the chat type.
     * The default value is one-to-one chat(single chat) {@link ChatType#Chat}.
     * @param chatType  The chat type, see {@link ChatType}.
     */
    public void setChatType(ChatType chatType) {
    	EMAMessage.EMAChatType t = EMAMessage.EMAChatType.SINGLE;
    	if (chatType == ChatType.Chat) {
    		t = EMAMessage.EMAChatType.SINGLE;
    	} else if (chatType == ChatType.GroupChat) {
    		t = EMAMessage.EMAChatType.GROUP;
    	} else {
    		t = EMAMessage.EMAChatType.CHATROOM;
    	}
    	emaObject.setChatType(t);
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(getMsgId());
    }
    
    public static final Parcelable.Creator<EMMessage> CREATOR
        = new Parcelable.Creator<EMMessage>() {
        public EMMessage createFromParcel(Parcel in) {
            EMMessage msg = null;
            try {
                msg = new EMMessage(in);
            } catch (HyphenateException e) {
                e.printStackTrace();
            }
            return msg;
        }

        public EMMessage[] newArray(int size) {
            return new EMMessage[size];
        }
    };
    
	@SuppressWarnings("CloneDoesntCallSuperClone")
    @Override
    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    /**
     * \~chinese
     * 获取对方是否已读。
     * @return 是否已读。
     *         -`true` 表示已读。
     * 
     * \~english
     * Gets whether the other party has read the message.
     * @return Whether the other party has read the message. `true`: the other party has read the message.
     */
	public boolean isAcked() {
		return emaObject.isAcked();
	}

    /**
     * \~chinese
     * 设置对方是否已读。
     * 开发者不应该调用此方法，由 SDK 进行调用。
     * @param isAcked  -`true`: 消息已读。
     * 
     * \~english
     * Sets whether the message has been read by the other party.
     * The method is supposed to be called by the SDK instead of you.
     * @param isAcked  - `true`:the message has been read by the other party.
     */
	public void setAcked(boolean isAcked) {
		emaObject.setIsAcked(isAcked);
	}

    /**
     * \~chinese
     * 获取消息是否已到达对方。
     * @return 消息是否已到达对方。- `true`:已送达。
     * 
     * \~english
     * Gets the delivery receipt, which is to check whether the other party has received the message.
     * @return Whether the other party has received the message. `true`:the message has been delivered to the other party.
     */
	public boolean isDelivered() {
		return emaObject.isDeliverAcked();
	}

    /**
     * \~chinese
     * 设置消息是否已经送达对方。
     * 开发者不要调用，由 SDK 内部调用。
     * @param isDelivered - `true`: 已送达。
     *
     * \~english
     * Sets whether the message has been delivered to the other party.
     * The method is supposed to be called by the SDK instead of you.
     * @param isDelivered - `true`: The message has been delivered to the other party.
     */
	public void setDelivered(boolean isDelivered) {
		emaObject.setIsDeliverAcked(isDelivered);
	}

    /**
     * \~chinese
     * 检查消息是否未读。
     * @return 消息是否未读。
     * - `true`: 消息未读；
     * - `false`: 消息已读。
     *
     * \~english
     * Checks whether the message is unread.
     * @return Whether the message is unread. 
     * -`true`: The message is unread. 
     * -`false`: The message is read.
     */
	public boolean isUnread() {
		return !emaObject.isRead();
	}

    /**
     * \~chinese
     * 设置消息为未读。
     * 
     * 参考：
     * 建议在会话中，使用 {@link EMConversation#markAllMessagesAsRead()}。
     * 
     * @param unread 
     * - `true`:设置消息为未读；
     * - `false`:设置消息为已读。
     *
     * \~english
     * Sets the message as unread. 
     * - `true`: Sets the message as unread.
     * - `false`: Sets the message as read.
     * 
     * Reference:
     * We recommend you to use {@link EMConversation#markAllMessagesAsRead()} in a conversation.
     * 
     * @param unread 
     * - `true`: Sets the message as unread.
     * - `false`: Sets the message as read.
     * 
     */
	public void setUnread(boolean unread) {
		emaObject.setIsRead(!unread);
	}

	/**
	 * \~chinese
	 * 获取语音是否已听。
	 * @return - `true`：消息已听。
     *         - `false`：消息未听。
	 * 
	 * \~english
	 * Gets whether the voice message has been listened.
     * @return  - `true`: The voice message has been listened.
     *          - `false`: The voice message has NOT been listened.
	 */
	public boolean isListened() {
		return emaObject.isListened();
	}

	/**
	 * \~chinese
	 * 设置语音是否已听。
     * 
     * 开发者不要调用，由 SDK 内部调用。
     * 
	 * @param isListened - `true`：设置消息为已听。
     *                   - `false`：设置消息为未听。
     * 
	 * \~english
	 * Sets whether the voice message has been listened.
     * 
     * The method is supposed to be called by the SDK instead of you.
	 * @param isListened  - `true`: Sets the voice message as listened.
     *                    - `false`: Sets the voice message as NOT listened.
	 */
	public void setListened(boolean isListened) {
	    emaObject.setListened(isListened);
	}
	
	/**
	 * \~chinese
	 * 获取通话对象的用户 ID。
	 * @return 对方的用户 ID。
	 * 
	 * \~english
	 * Gets the other party's ID.
     * @return The other party's user ID.
	 */
	public String getUserName() {
		String username = "";
		if (getFrom() != null && getFrom().equals(EMClient.getInstance().getCurrentUser())) {
			username = getTo();
		} else {
			username = getFrom();
		}
		return username;
	}

    /**
     * \~chinese
     * 设置消息已送达标记。
     * 开发者不要调用，由 SDK 内部调用。
     * @param isDeliverAcked - `true`：消息已送达。
     *                       - `false`：消息未送达。
     *
     * \~english
     * Sets the message deliver ack.
     * The method is supposed to be called by the SDK instead of you.
     * @param isDeliverAcked - `true`: The message is delivered to the other party.
     *                       - `false`: The message has not been delivered to the other party.
     */
	public void setDeliverAcked(boolean isDeliverAcked) {
	    emaObject.setIsDeliverAcked(isDeliverAcked);
	}

    /**
     * \~chinese
     * 消息包含附件的上传或者下载进度，值的范围 0-100。
     * 消息附件的缩略图不涉及进度信息。
     * @return 进度值。
     *
     * \~english
     * The upload or download progress value of the message attachment in the message body. The value ranges between 0-100.
     * For the thumbnail of the message attachment, it doesn't has progress information.
     * @return The progress value.
     */
	public int progress() {
	    return emaObject.progress();
	}

    /**
     * \~chinese
     * 设置消息包含附件的上传或者下载进度，值的范围 0-100。
     * 对于 app 开发者来说，通常不需要主动设置进度。参考：{@link EMMessage#progress()}
     * @param progress 上传或者下载进度。
     *
     * \~english
     * Sets the upload or download progress value of the message attachment. The value ranges between 0-100.
     * Usually you don't need to set progress value.
     * @param progress 
     */
    public void setProgress(int progress) {
        emaObject.setProgress(progress);
    }

	/**
	 * \~chinese
	 * 消息方向。
     * @return 发送或接收，详见 {@link Direct}。
	 * 
	 * \~english
	 * The message direction.
     * @return SEND or RECEIVE, see {@link Direct}.
	 */
	public Direct direct() {
	    if (emaObject.direction() == EMADirection.SEND) {
	        return Direct.SEND;
	    } else {
	        return Direct.RECEIVE;
	    }
	}
	
	/**
	 * \~chinese
	 * 设置消息的方向。
	 * @param dir 消息的方向，详见 {@link Direct}。
	 * 
	 * \~english
	 * Sets the message direction.
	 * @param dir  The message direction, see {@link Direct}.
	 */
	public void setDirection(Direct dir) {
	    emaObject.setDirection(dir.ordinal());
	}

    /**
     * \~chinese
     * 获取会话 ID。
     * @return  会话 ID。
     *
     * \~english
     * Gets the conversation ID.
     * @return The conversation ID.
     */
	public String conversationId() {
	    return emaObject.conversationId();
	}

    static String self() {
        String myself = EMClient.getInstance().getCurrentUser();
        if (myself == null) {
            //currentUser is only set after login successfully
            //since it is not when not logged in, use last login user instead
            myself = EMSessionManager.getInstance().getLastLoginUser();
        }
        return myself;
    }

    private EMMessage(Parcel in) throws HyphenateException {
        String msgId = in.readString();
        EMMessage msg = EMClient.getInstance().chatManager().getMessage(msgId);
        if (msg == null) {
            throw new HyphenateException("EMMessage constructed from parcel failed");
        }
        emaObject = msg.emaObject;
    }

    /**
     * \~chinese
     * 获取消息包含的全部扩展字段，并以 Map<String, Object> 形式返回。
     *
     * @return  消息的扩展字段，以 Map 形式返回，返回 Entry.value 的具体类型有 Boolean, Integer, Long, Float, Double, String。
     *          输入 JsonObject 的属性，map 结构返回时属性是 String。
     *
     * \~english
     * Gets all of the message‘s extension. The return type is Map<String, Object>.
     *
     * @return  Returns the message's extension. The return type is Map<String, Object>.
     *          The object can be Boolean, Integer, Long, Float, Double, String.
     *          If the input is JsonObject or JsonArray, which use setAttribute(String attribute,
     *          JSONObject json) to passed in, the Map.Entry.value type is String.
     */
    public Map<String, Object> ext() {
        return emaObject.ext();
    }

    /**
     * \~chinese
     * 获取 Reaction 列表。
     * @return  Reaction 列表。
     *
     * \~english
     * Gets the list of Reactions.
     * @return The list of Reactions.
     */
    public List<EMMessageReaction> getMessageReaction() {
        List<EMAMessageReaction> messageReactions = emaObject.reactionList();
        if (messageReactions.size() > 0) {
            List<EMMessageReaction> reactionList = new ArrayList<>(messageReactions.size());
            EMMessageReaction reaction;
            for (EMAMessageReaction emaMessageReaction : messageReactions) {
                reaction = new EMMessageReaction(emaMessageReaction);
                reactionList.add(reaction);
            }
            return reactionList;
        }
        return null;
    }

    /**
     * \~chinese
     * 是否为在线消息。
     * @return  是否在线。
     *
     * \~english
     * Is it an online message.
     * @return Whether is online.
     */
    public boolean isOnlineState() {
        return  emaObject.isOnlineState();
    }
}
