/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.EMGroupChangeListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMACallback;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAGroup;
import com.hyphenate.chat.adapter.EMAGroup.EMGroupLeaveReason;
import com.hyphenate.chat.adapter.EMAGroupManager;
import com.hyphenate.chat.adapter.EMAGroupManagerListener;
import com.hyphenate.chat.adapter.EMAGroupSetting;
import com.hyphenate.chat.adapter.EMAMucShareFile;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;


/**
 * \~chinese
 * 群组管理, 用于管理群组的创建，删除，成员管理等操作。
 *
 * \~english
 * The group manager for management of group creation and deletion and member management.
 */
public class EMGroupManager {

    /**
     * \~chinese
	 * 群组类型枚举类。
	 *
	 * \~english
	 * The group styles.
     */
	public enum EMGroupStyle {

		/**
		 * \~chinese
		 * 私有群组，创建完成后，只允许群主邀请用户加入。
		 *
		 * \~english
		 * Private groups. Only the group owner can invite users to join.
		 */
		EMGroupStylePrivateOnlyOwnerInvite,

		/**
		 * \~chinese
		 * 私有群组，创建完成后，只允许群主和群成员邀请用户加入。
		 *
		 * \~english
		 * Private groups. Both the group owner and group members can invite users to join.
		 */
		EMGroupStylePrivateMemberCanInvite,

		/**
		 * \~chinese
		 * 公开群组，创建完成后，只允许群主邀请用户加入; 非群成员用户需发送入群申请，群主同意后才能入群。
		 *
		 * \~english
		 * Public groups. Only the owner can invite users to join. 
		 * A user can join a group only after the owner approves the user's group request;
		 */
		EMGroupStylePublicJoinNeedApproval,

		/**
		 * \~chinese
		 * 公开群组，创建完成后，允许非群组成员加入，无需群主同意.
		 *
		 * \~english
		 * Public groups. A user can join a group without the group owner approving user's group request.
		 */
		EMGroupStylePublicOpenJoin
	}

	EMAGroupManager emaObject;

	private static String TAG = "group";

	List<EMGroupChangeListener> groupChangeListeners;

	EMAGroupManagerListener listenerImpl = new EMAGroupManagerListener() {

		@Override
		public void onReceiveInviteFromGroup(String groupId, String groupName, String inviter, String inviteMessage) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onInvitationReceived(groupId, groupName, inviter, inviteMessage);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		/**
		 * \~chinese
		 * 受邀人接受群组邀请的回调。
		 * @param  invitee  受邀加入群组的用户。
		 * @return NA
		 * 
		 * \~english
		 * Occurs when the user accepts the invitation to join the group.
		 * @param  invitee The user invited to join the group.
		 * @return NA
		 */
		@Override
		public void onReceiveInviteAcceptionFromGroup(EMAGroup group, String invitee) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onInvitationAccepted(group.groupId(), invitee, "");
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		/**
		 * \~chinese
		 * 受邀人拒绝群组邀请的回调。
		 * @param  invitee  受邀加入群组的用户。
		 * @param  reason   拒绝理由。
		 * @return NA
		 * 
		 * \~english
		 * Occurs when the user declines to join the group.
		 * @param  invitee The user invited to join the group.
		 * @param  reason  The reason of declining.
		 * @return NA
		 */
		@Override
		public void onReceiveInviteDeclineFromGroup(EMAGroup group, String invitee, String reason) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onInvitationDeclined(group.groupId(), invitee, "");
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAutoAcceptInvitationFromGroup(EMAGroup group, String inviter, String inviteMessage) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAutoAcceptInvitationFromGroup(group.groupId(), inviter, inviteMessage);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onLeaveGroup(EMAGroup group, int reason) {
			EMClient.getInstance().chatManager().caches.remove(group.groupId());
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						if (reason == EMGroupLeaveReason.BE_KICKED.ordinal()) {
							listener.onUserRemoved(group.groupId(), group.groupSubject());
						} else {
							listener.onGroupDestroyed(group.groupId(), group.groupSubject());
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onReceiveJoinGroupApplication(EMAGroup group, String from, String message) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinReceived(group.groupId(), group.groupSubject(), from, message);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onReceiveAcceptionFromGroup(EMAGroup group) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinAccepted(group.groupId(), group.groupSubject(), group.getOwner());
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onReceiveRejectionFromGroup(String groupId, String reason) {
			EMGroup group = EMGroupManager.this.getGroup(groupId);
			String groupName = group == null ? "" : group.groupSubject();
			String decliner = group == null ? "" : group.getOwner();

			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinDeclined(groupId, groupName, decliner, reason);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUpdateMyGroupList(List<EMAGroup> groups) {

		}

		@Override
		public void onAddMutesFromGroup(EMAGroup group, final List<String> muteMembers, long muteExpire) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMuteListAdded(group.groupId(), muteMembers, muteExpire);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onRemoveMutesFromGroup(EMAGroup group, final List<String> banPostList) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMuteListRemoved(group.groupId(), banPostList);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

        public void onWhiteListAdded(EMAGroup group, List<String> members) {
            synchronized (groupChangeListeners) {
                try {
                    for(EMGroupChangeListener listener : groupChangeListeners) {
                        listener.onWhiteListAdded(group.groupId(), members);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        public void onWhiteListRemoved(EMAGroup group, List<String> members) {
            synchronized (groupChangeListeners) {
                try {
                    for(EMGroupChangeListener listener : groupChangeListeners) {
                        listener.onWhiteListRemoved(group.groupId(), members);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        public void onAllMemberMuteStateChanged(EMAGroup group, boolean isMuted) {
            synchronized (groupChangeListeners) {
                try {
                    for(EMGroupChangeListener listener : groupChangeListeners) {
                        listener.onAllMemberMuteStateChanged(group.groupId(), isMuted);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

		@Override
		public void onAddAdminFromGroup(EMAGroup group, String admin) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAdminAdded(group.groupId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onRemoveAdminFromGroup(EMAGroup group, String admin) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAdminRemoved(group.groupId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAssignOwnerFromGroup(EMAGroup group, String newOwner, String oldOwner) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onOwnerChanged(group.groupId(), newOwner, oldOwner);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onMemberJoined(EMAGroup group, String member) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMemberJoined(group.groupId(), member);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onMemberExited(EMAGroup group, String member) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMemberExited(group.groupId(), member);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAnnouncementChanged(EMAGroup group, String announcement) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAnnouncementChanged(group.groupId(), announcement);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUploadShareFileFromGroup(EMAGroup group, EMAMucShareFile shareFile) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onSharedFileAdded(group.groupId(), new EMMucSharedFile(shareFile));
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onDeleteShareFileFromGroup(EMAGroup group, String fileId) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onSharedFileDeleted(group.groupId(), fileId);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUpdateSpecificationFromGroup(EMAGroup group) {
			synchronized (groupChangeListeners) {
				for (EMGroupChangeListener listener : groupChangeListeners) {
					try {
						if (null != group) {
							listener.onSpecificationChanged(new EMGroup(group));
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

		@Override
		public void onStateChangedFromGroup(EMAGroup group, boolean isDisabled) {
		}
	};

	EMClient mClient;
	EMGroupManager(EMClient client, EMAGroupManager groupManager) {
		emaObject = groupManager;
		mClient = client;
		groupChangeListeners = Collections.synchronizedList(new ArrayList<EMGroupChangeListener>());
		emaObject.addListener(listenerImpl);
		EMClient.getInstance().chatManager();
	}

	/**
	 * \~chinese
	 * 获取当前(内存)用户的所有群组。
	 * 在调用此方法之前，可调用 {@link #loadAllGroups()}，提前将数据从群组加载到内存；
	 * 如果未调用 {@link #loadAllGroups()}，首次会从数据库中加载群组数据，之后再从内存中加载。
	 *
	 * @return 群组列表。
	 *
	 * \~english
	 * Gets all groups of the current user (from the cache).
	 * Before a call to this method, call {@link #loadAllGroups()} to load data from the group to the cache.
	 * If {@link #loadAllGroups()} is not called, this method will load data from the database before
	 * loading from the cache.
	 *
	 * @return The group list.
	 */
	public List<EMGroup> getAllGroups() {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.allMyGroups(error);
		List<EMGroup> result = new ArrayList<EMGroup>();
		for (EMAGroup group : groups) {
			result.add(new EMGroup(group));
		}
		return Collections.unmodifiableList(result);
	}

	/**
	 * \~chinese
	 * 根据群组 ID，从内存中获得群组对象。
	 *
	 * @param groupId	群组ID。
	 * @return 			返回群组对象。如果群组不存在，返回 null。
	 *
	 * \~english
	 * Gets the group instance from the cache by group ID.
	 *
	 * @param groupId	The group ID.
	 * @return 			The group instance. Returns null if the group does not exist.
	 */
	public EMGroup getGroup(String groupId) {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.allMyGroups(error);
		for (EMAGroup group : groups) {
			if (group.groupId().equals(groupId)) {
				return new EMGroup(group);
			}
		}
		return null;
	}

	/**
	 * \~chinese
	 * 创建群组。
	 * 群组创建成功后，会更新内存及数据库中的数据，多端多设备会收到相应的通知事件，将群组更新到内存及数据库中。
	 * 可通过设置 {@link com.hyphenate.EMMultiDeviceListener}监听相关事件，事件回调函数为 
	 * {@link com.hyphenate.EMMultiDeviceListener#onGroupEvent(int, String, List)},
	 * 第一个参数为事件，建群事件为 {@link com.hyphenate.EMMultiDeviceListener#GROUP_CREATE}。
	 * 异步方法见 {@link #asyncCreateGroup(String, String, String[], String, EMGroupOptions, EMValueCallBack)}
	 *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupName     		群组的名字，非必要，不设置传 null。
     * @param desc          		群组描述，非必要，不设置传 null。
	 * @param allMembers    		群成员数组。群主 ID 非必要。该参数不可为 null。
	 * @param reason        		邀请群成员加入时的邀请信息，非必要，不设置传 null。
	 * @param option        		群的设置，非必要，不可为 null。详见 {@link EMGroupOptions}。
	 *                      		群组的其他选项如下：
	 *                      		- 群最大成员数，默认值为 200；
	 *                      		- 群类型，详见 {@link EMGroupManager.EMGroupStyle}，默认为 {@link EMGroupStyle#EMGroupStylePrivateOnlyOwnerInvite}；
	 *                      		- 邀请进群是否需要对方同意，默认为 false，即邀请后直接进群；
	 *                      		- 群组详情扩展。
	 *
	 * @return 						返回创建成功的群对象。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Creates a group instance.
	 * After the group is created, the data in the cache and database will be updated and multiple devices will receive the notification event and
	 * update the group to the cache and database. 
	 * You can set {@link com.hyphenate.EMMultiDeviceListener} to listen on the event. The event callback function
	 * is {@link com.hyphenate.EMMultiDeviceListener#onGroupEvent(int, String, List)}, where the first parameter is the event,
	 * for example, {@link com.hyphenate.EMMultiDeviceListener#GROUP_CREATE} for the group creation event.
	 * For the asynchronous method, see {@link #asyncCreateGroup(String, String, String[], String, EMGroupOptions, EMValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *

	 * @param groupName     		The group name. It is optional. Pass null if you do not want to set this parameter.
     * @param desc          		The group description. It is optional. Pass null if you do not want to set this parameter.
	 * @param allMembers    		The group member array. The group owner ID is optional. This parameter can not be null.
	 * @param reason        		The group joining invitation. It is optional. Pass null if you do not want to set this parameter.
	 * @param option        		The options for creating a group. They are optional and can not be null. See {@link EMGroupOptions}.
	 *                      		The options are as follows:
	 *                      		- The maximum number of group members. The default value is 200.
	 *                      		- The group style. See {@link EMGroupManager.EMGroupStyle}. The default value is {@link EMGroupStyle#EMGroupStylePrivateOnlyOwnerInvite}.
	 *                      		- Whether to ask for permission when inviting a user to joing the group. The default value is false, indicating that invitees are automaticall added to the group without their permission.
	 *                      		- The group detail extensions.
	 *
	 * @return 						The created group instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
	 */
	public EMGroup createGroup(String groupName, String desc, String[] allMembers, String reason, EMGroupOptions option) throws HyphenateException{
		int style = EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE;

		switch(option.style) {
			case EMGroupStylePrivateOnlyOwnerInvite:
				style = EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE;
				break;

			case EMGroupStylePrivateMemberCanInvite:
				style = EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE;
				break;

			case EMGroupStylePublicJoinNeedApproval:
				style = EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL;
				break;

			case EMGroupStylePublicOpenJoin:
				style = EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_OPEN;
				break;
		}

		return createGroup(style, groupName, desc, allMembers, option.maxUsers, reason, option.inviteNeedConfirm, option.extField);
	}

	/**
	 * \~chinese
	 * 创建群组。
     * 群组创建成功后，会更新内存及数据库中的数据，多端多设备会收到相应的通知事件，然后将群组更新到内存及数据库中。
	 * 可通过设置 {@link com.hyphenate.EMMultiDeviceListener} 监听相关事件，事件回调函数为
	 * {@link com.hyphenate.EMMultiDeviceListener#onGroupEvent(int, String, List)},
	 * 第一个参数为事件，建群事件为 {@link com.hyphenate.EMMultiDeviceListener#GROUP_CREATE}。
	 *
	 * 异步方法。
	 *
	 * @param groupName     群组的名字，非必要，不设置传 null。
     * @param desc          群组描述，非必要，不设置传 null。
	 * @param allMembers    群成员数组。群主 ID非必要。该参数不可为 null。
     * @param reason        成员入群的邀请信息，非必要，不设置传 null。
	 * @param option        群组设置，非必要，不可为 null。详见 {@link EMGroupOptions}。
	 *                      群组的其他选项如下：
	 *                      - 群组最大成员数，默认值为 200；
	 *                      - 群组类型，详见 {@link EMGroupManager.EMGroupStyle}，默认为 {@link EMGroupStyle#EMGroupStylePrivateOnlyOwnerInvite}；
	 *                      - 邀请入群是否需要对方同意，默认为 `false`，即邀请后直接入群；
	 *                      - 群详情扩展。
	 *
     * @param callback		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回创建成功的群组对象;
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Creates a group instance.
     * After the group is created, the data in the cache and database will be updated and multiple devices will receive the notification event and
	 * update the group to the cache and database. You can set {@link com.hyphenate.EMMultiDeviceListener} to listen on the event. The event callback function
	 * is {@link com.hyphenate.EMMultiDeviceListener#onGroupEvent(int, String, List)}, where the first parameter is the event, for example,
	 * {@link com.hyphenate.EMMultiDeviceListener#GROUP_CREATE} for the group creation event.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupName     The group name. It is optional. Pass null if you do not want to set this parameter.
     * @param desc          The group description. It is optional. Pass null if you do not want to set this parameter.
	 * @param allMembers    The group member array. The group owner ID is optional. This parameter can not be null.
	 * @param reason        The group joining invitation. It is optional. Pass null if you do not want to set this parameter.
	 * @param option        The options for creating a group. They are optional and can not be null. See {@link EMGroupOptions}.
	 *                      The options are as follows:
	 *                      - The maximum number of group members. The default value is 200.
	 *                      - The group style. See {@link EMGroupManager.EMGroupStyle}. The default value is {@link EMGroupStyle#EMGroupStylePrivateOnlyOwnerInvite}.
	 *                      - Whether to ask for permission when inviting a user to joining the group. The default value is false, indicating that invitees are automaticall added to the group without their permission.
	 *                      - The group detail extensions.
	 *
     * @param callback		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and returns the created group object;
	 * 						If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
     */
	public void asyncCreateGroup(final String groupName,
								 final String desc,
								 final String[] allMembers,
								 final String reason,
								 final EMGroupOptions option,
								 final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMGroup group = createGroup(groupName, desc, allMembers, reason, option);
					callback.onSuccess(group);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

    /// @cond
	// Private method.
    /**
     *  Creates a group.
     *
     *  @param style                The public or private group.
     *  @param groupName            The group name.
     *  @param desc                 The group description.
     *  @param allMembers           The group members, excluding the owner and creator.
     *  @param maxUsers             The maximum group member capacity.
     *  @param reason               The invitation message.
     *  @param inviteNeedConfirm    Whether to need invitation confirmation.
     *  @param extension            The group extension information.
     *  @return                     The created group.
     */
	 /// @endcond
    private EMGroup createGroup(int style,
                                String groupName,
                                String desc,
                                String[] allMembers,
                                int maxUsers,
                                String reason,
                                boolean inviteNeedConfirm,
                                String extension) throws HyphenateException {
        EMAGroupSetting setting = new EMAGroupSetting(style, maxUsers, inviteNeedConfirm, extension);
        List<String> newMembers = new ArrayList<String>();
        Collections.addAll(newMembers, allMembers);
        EMAError error = new EMAError();
        EMAGroup group = emaObject.createGroup(groupName, desc, reason, setting, newMembers, inviteNeedConfirm, error);
        handleError(error);
        return new EMGroup(group);
    }

	/**
	 * \~chinese
	 * 从数据库中加载所有群组。
	 *
	 * \~english
	 * Loads all local groups from the database.
	 */
	public synchronized void loadAllGroups() {
		emaObject.loadAllMyGroupsFromDB();
	}

	/**
	 * \~chinese
	 * 解散群组。
	 * 仅群主可调用此方法。
	 * 异步方法见 {@link #asyncDestroyGroup(String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Destroys the group instance.
	 * Only the group owner can call this method.
	 * For the asynchronous method, see {@link #asyncDestroyGroup(String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void destroyGroup(final String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.destroyGroup(groupId, error);
		EMClient.getInstance().chatManager().caches.remove(groupId);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 解散群组。
	 * 仅群主可调用此方法。
	 * 同步方法见 {@link #destroyGroup(String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
     * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
     *
	 * \~english
	 * Destroys the group instance.
	 * Only the group owner can call this method.
	 * For the synchronous method, see {@link #destroyGroup(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
     * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 					if this call fails, calls {@link EMCallBack#onError(int, String)}.
     */
	public void asyncDestroyGroup(final String groupId, final EMCallBack callback){
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					destroyGroup(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 向群组中添加新成员。
	 * 仅群组创建者和管理员可调用此方法。
	 * 异步方法见 {@link #asyncAddUsersToGroup(String, String[], EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId 				群组 ID。
	 * @param newmembers			要添加的新成员数组。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Adds users to the group.
	 * Only the group creator or admin can call this method.
	 * For the asynchronous method, see {@link #asyncAddUsersToGroup(String, String[], EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param newmembers			The array of new members to add.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void addUsersToGroup(String groupId, String[] newmembers) throws HyphenateException {
		addUsersToGroup(groupId, newmembers, "welcome");
	}

	/**
	 * \~chinese
	 * 向群组中添加新成员。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncAddUsersToGroup(String, String[], EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId               群组 ID。
	 * @param newmembers            要添加的新成员数组。
	 * @param welcomeMessage   		邀请信息。
	 * @throws HyphenateException   如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Adds users to the group.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncAddUsersToGroup(String, String[], EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId                The group ID.
	 * @param newmembers             The array of new members to add.
	 * @param welcomeMessage		 The welcome message.
	 * @throws HyphenateException    A description of the exception. See {@link EMError}.
	 */
	public void addUsersToGroup(String groupId, String[] newmembers, String welcomeMessage) throws HyphenateException {
		EMAError error = new EMAError();
		List<String> newmembers2 = new ArrayList<String>();
		Collections.addAll(newmembers2, newmembers);
		emaObject.addGroupMembers(groupId, newmembers2, welcomeMessage, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 向群组中添加新成员。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #addUsersToGroup(String, String[])}。
	 *
	 * 异步方法。
	 *
	 * @param groupId               群组的 ID。
	 * @param newmembers            要添加的新成员数组。
	 * @param callback				结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 								失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Adds users to the group.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #addUsersToGroup(String, String[])}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId		The group ID.
	 * @param newmembers	The array of new members to add.
	 * @param callback		The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 						If this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncAddUsersToGroup(final String groupId,
									 final String[] newmembers,
									 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					addUsersToGroup(groupId, newmembers);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将群成员移出群组。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncRemoveUserFromGroup(String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组的 ID。
	 * @param username				要删除成员的用户名。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes a member from the group.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncRemoveUserFromGroup(String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param username   			The user ID of the member to be removed.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void removeUserFromGroup(final String groupId, final String username) throws HyphenateException {
		List<String> members = new ArrayList<String>();
		EMAError error = new EMAError();
		members.add(username);
		emaObject.removeGroupMembers(groupId, members, error);
		handleError(error);
		emaObject.fetchGroupSpecification(groupId, error, true);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 将群成员移出群组。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #removeUserFromGroup(String, String)}
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
	 * @param username	要删除成员的用户名。
     * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
     *
	 * \~english
	 * Removes a member from the group.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #removeUserFromGroup(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId   The group ID.
	 * @param username  The user ID of the member to be removed.
     * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 					if this call fails, calls {@link EMCallBack#onError(int, String)}.
     */
	public void asyncRemoveUserFromGroup(final String groupId,
										 final String username,
										 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					removeUserFromGroup(groupId, username);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将群成员移出群组。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #removeUsersFromGroup(String, List)}}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @param members				要删除成员的用户名列表。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes members from the group.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #removeUsersFromGroup(String, List)}}.
	 *
	 * Only the group owner or admin can call this method.
	 *
	 * @param groupId				The group ID.
	 * @param members   			The user IDs of members to be removed.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void removeUsersFromGroup(final String groupId, final List<String> members) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.removeGroupMembers(groupId, members, error);
		handleError(error);
		emaObject.fetchGroupSpecification(groupId, error, true);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 从群组中删除成员。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #removeUsersFromGroup(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId   群组 ID。
	 * @param members   要删除成员的用户名列表。
     * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					败时回调 {@link EMCallBack#onError(int, String)}。
     *
	 * \~english
	 * Removes members from the group.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #removeUsersFromGroup(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param members   The user IDs of members to be removed.
     * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 					if this call fails, calls {@link EMCallBack#onError(int, String)}.
     */
	public void asyncRemoveUsersFromGroup(final String groupId,
										 final List<String> members,
										 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					removeUsersFromGroup(groupId, members);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 当前登录用户退出群组。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Leaves a group.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void leaveGroup(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.leaveGroup(groupId, error);
		EMClient.getInstance().chatManager().caches.remove(groupId);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 当前登录用户退出群组。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
     * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
     *
	 * \~english
	 * Leaves a group.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
     * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 					If this call fails, calls {@link EMCallBack#onError(int, String)}.
     */
	public void asyncLeaveGroup(final String groupId, final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					leaveGroup(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的详细信息。
	 * 该方法不获取成员。如需获取成员，使用 {@link #fetchGroupMembers(String, String, int)}。
	 * 异步方法见 {@link #asyncGetGroupFromServer(String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @return EMGroup				返回群组对象。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets group information from the server.
	 * This method does not get member information. If member information is required, call {@link #fetchGroupMembers(String, String, int)}.
	 * For the asynchronous method, see {@link #asyncGetGroupFromServer(String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @return EMGroup				The group instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMGroup getGroupFromServer(String groupId) throws HyphenateException {
		if(groupId == null || groupId.isEmpty())
			throw new HyphenateException(EMError.GROUP_INVALID_ID, "group id is null or empty");
		EMAError error = new EMAError();
		EMAGroup group = emaObject.fetchGroupSpecification(groupId, error, false);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的详细信息。
	 * 参数 fetchMembers 为 true 的时候，获取群成员。如需获取更多成员，使用 {@link #fetchGroupMembers(String, String, int)}。
	 * 只有群成员才能获取群组的成员列表。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @param fetchMembers 			是否获取成员列表，默认获取 200 成员的列表。
	 * @return EMGroup 				返回更新后的群组对象。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets group information from the server.
	 * If fetchMembers is set to true, a member list is also fetched. If more members need to be fetched, call {@link #fetchGroupMembers(String, String, int)}.
	 * Only members in the group can get the member list of the group.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param fetchMembers 			Whether to get group members. By default, a list of 200 members is fetched.
	 * @return EMGroup				The updated group instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMGroup getGroupFromServer(String groupId, boolean fetchMembers) throws HyphenateException {
		if(groupId == null || groupId.isEmpty())
			throw new HyphenateException(EMError.GROUP_INVALID_ID, "group id is null or empty");
		EMAError error = new EMAError();
		EMAGroup group = emaObject.fetchGroupSpecification(groupId, error, fetchMembers);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的详细信息。
	 * 同步方法见 {@link #getGroupFromServer(String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
     * @param callback 	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets group information from the server.
	 * For the synchronous method, see {@link #getGroupFromServer(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
     * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 					if this call fails, calls {@link EMCallBack#onError(int, String)}.
     */
	public void asyncGetGroupFromServer(final String groupId,
										final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMGroup group = getGroupFromServer(groupId);
					callback.onSuccess(group);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器端获取当前用户的所有群组。
	 * 此操作只返回群组列表，不包含所有成员的信息。如果要更新某个群组包括成员的全部信息，需要再调用 {@link #getGroupFromServer(String groupId)}。
	 * @return 群组列表。
	 * @throws HyphenateException 如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets all groups from the server.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * @return The group list. 
	 * @throws HyphenateException A description of the exception. See {@link EMError}.
	 */
	synchronized List<EMGroup> getGroupsFromServer() throws HyphenateException {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.fetchAllMyGroups(error);
		handleError(error);
		List<EMGroup> result = new ArrayList<EMGroup>();
		for (EMAGroup group : groups) {
			result.add(new EMGroup(group));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 以分页方式从服务器获取当前用户的所有群组。
	 * 此操作只返回群组列表，不包括群组的所有成员信息。如果要更新某个群组的全部成员信息，需要调用 {@link #getGroupFromServer(String groupId)}。
	 *
	 * @param pageIndex 当前页码，从 1 开始。
	 * @param pageSize  每页返回的数组数。
	 * @return          群组列表。如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 * @throws HyphenateException 
	 *
	 * \~english
	 * Gets all groups of the current user from the server with pagination。
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId).}. 
	 *
	 * @param pageIndex  The page number, starting from 1.
	 * @param pageSize   The number of groups per page.
	 * @return           The group list.  
	 * @throws HyphenateException A description of the exception. See {@link EMError}.
	 */
	synchronized List<EMGroup> getGroupsFromServer(int pageIndex, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.fetchAllMyGroupsWithPage(pageIndex, pageSize, error);
		handleError(error);
		List<EMGroup> result = new ArrayList<EMGroup>();
		for (EMAGroup group : groups) {
			result.add(new EMGroup(group));
		}
		return result;
	}


	/**
	 * \~chinese
	 * 从服务器端获取当前用户的所有群组。
	 * 此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用 {@link #getGroupFromServer(String groupId)}。
	 * @param callback  包含群组列表。
	 *
	 * \~english
	 * Gets all groups of the current user from the server.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 *
	 * @param callback  Contains the group list.
	 */
	void asyncGetGroupsFromServer(final EMValueCallBack<List<EMGroup>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<EMGroup> groups = getGroupsFromServer();
					callback.onSuccess(groups);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取当前用户的所有群组。
	 * 此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}。
	 * 异步方法见 {@link #asyncGetJoinedGroupsFromServer(EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @return 						当前用户已加入的群组列表。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets all groups of the current user from the server.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For the asynchronous method, see {@link #asyncGetJoinedGroupsFromServer(EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @return 						The list of groups that the current user joins.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public synchronized List<EMGroup> getJoinedGroupsFromServer() throws HyphenateException{
		return getGroupsFromServer();
	}

	/**
	 * \~chinese
	 * 以分页方式从服务器获取当前用户已加入的群组。
	 * 此操作只返回群组列表，不包含群组的所有成员信息。如果要更新某个群组包括成员的全部信息，需要再调用 {@link #getGroupFromServer(String groupId)}。
	 * 异步方法见 {@link #asyncGetJoinedGroupsFromServer(int, int, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param pageIndex 			当前页码，从 1 开始。
	 * @param pageSize				每页返回的群组数。
	 * @return 						返回分页获取的群组列表。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets all groups of the current user from the server with pagination.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For the asynchronous method, see {@link #asyncGetJoinedGroupsFromServer(int, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param pageIndex 			The page number, starting from 1.
	 * @param pageSize				The number of groups per page.
	 * @return 						The group list on the next page.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public synchronized List<EMGroup> getJoinedGroupsFromServer(int pageIndex, int pageSize) throws HyphenateException{
		return getGroupsFromServer(pageIndex, pageSize);
	}

	/**
	 * \~chinese
	 * 从服务器获取当前用户的所有群组。
	 * 此操作只返回群组列表，并不获取群组的所有成员信息。如果要更新某个群组包括成员的全部信息，需要再调用 {@link #getGroupFromServer(String groupId)}。
	 * 同步方法见 {@link #getJoinedGroupsFromServer()}。
	 *
	 * 异步方法。
	 *
	 * @param callback  结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回已加入的群组列表
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets all groups of the current user from the server.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For a synchronous method, see {@link #getJoinedGroupsFromServer()}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param callback 	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and 
	 *                  returns the list of groups that the user has joined;
	 * 					if the call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncGetJoinedGroupsFromServer(final EMValueCallBack<List<EMGroup>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<EMGroup> groups = getJoinedGroupsFromServer();
					callback.onSuccess(groups);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 以分页方式从服务器获取当前用户已加入的群组。
	 * 此操作只返回群组列表，不包含群组的所有成员信息。如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}。
	 * 同步方法见 {@link #getJoinedGroupsFromServer(int, int)}。
	 *
	 * 异步方法。
	 *
	 * @param pageIndex	当前页码，从 1 开始。
	 * @param pageSize	每页返回的群组数。
	 * @param callback 	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}， 返回分页获取的群组列表
	 *                  失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 *
	 * \~english
	 * Gets all groups of the current user from the server with pagination.
	 * This method returns a group list which does not contain member information. If you want to update information of a group to include its member information, call {@link #getGroupFromServer(String groupId)}.
	 * For the synchronous method, see {@link #getJoinedGroupsFromServer(int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param pageIndex	The page number, starting from 1.
	 * @param pageSize	The number of groups per page.
     * @param callback 	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and returns the group list on the next page;
	 *                  if this call fails, calls {@link EMValueCallBack#onError(int, String)};
	 * 					
	 */
	public void asyncGetJoinedGroupsFromServer(final int pageIndex,
											   final int pageSize,
											   final EMValueCallBack<List<EMGroup>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<EMGroup> groups = getJoinedGroupsFromServer(pageIndex, pageSize);
					callback.onSuccess(groups);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 以分页方式从服务器获取公开群组。
	 * 异步方法见 {@link #asyncGetPublicGroupsFromServer(int, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param pageSize				每页返回的公开群组数。
	 * @param cursor				从这个游标位置开始取数据，首次获取数据时传 null 即可。
	 * @return						分页获取结果 {@link EMCursorResult}，包含用于下次获取数据的 cursor 以及群组 list 的对象，
	 * 								如果是最后一页，cursor 返回结果为空字符串。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Gets public groups from the server with pagination.
	 * For the asynchronous method, see {@link #asyncGetPublicGroupsFromServer(int, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param pageSize  			The number of public groups per page.
	 * @param cursor    			The cursor position from which to start to get data next time. Sets the parameter as null for the first time.
	 * @return  		    		The result of {@link EMCursorResult}, including the cursor for getting data next time and the group list.
	 * 								For the last page, the return value of cursor is an empty string.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
	 */
	public EMCursorResult<EMGroupInfo> getPublicGroupsFromServer(int pageSize, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMGroupInfo> result = emaObject.fetchPublicGroupsWithCursor(cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 以分页方式从服务器获取公开群组。
	 * 同步方法见 {@link #getPublicGroupsFromServer(int, String)}。
	 *
	 * 异步方法。
	 *
	 * @param pageSize		每页返回的公开群组数。
	 * @param cursor		从这个游标位置开始取数据，首次获取数据时传 null 即可。
     * @param callback      结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      返回分页获取结果 {@link EMCursorResult}，包含用于下次获取数据的 cursor 以及群组list的对象。
	 * 						如果是最后一页，cursor 返回结果为空字符串；
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets public groups from the server with pagination.
	 * For the synchronous method, see {@link #getPublicGroupsFromServer(int, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param pageSize  	The number of public groups per page.
	 * @param cursor    	The cursor position from which to start getting data next time. Sets the parameter as null for the first time.
	 * @param callback      The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} 
	 *                      and returns the result of {@link EMCursorResult}), including the cursor for getting data next time and the group list.
	 * 						For the last page, the return value of cursor is an empty string.
	 * 						If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncGetPublicGroupsFromServer(final int pageSize,
											   final String cursor,
											   final EMValueCallBack<EMCursorResult<EMGroupInfo>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMCursorResult<EMGroupInfo> result = getPublicGroupsFromServer(pageSize, cursor);
					callback.onSuccess(result);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 当前登录用户加入公开群。
	 * 如果是自由加入的公开群，直接进入群组；需要验证的公开群需要等待群主同意。详见 {@link EMGroupStyle}。
	 * 异步方法见 {@link #asyncJoinGroup(String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Joins a public group.
	 * For a group that requires no authentication，users can join it freely without the need of having permission. 
	 * For a group that requires authentication, users need to wait for the owner to agree before joing the group. For details, see {@link EMGroupStyle}.
	 * For the asynchronous method, see {@link #asyncJoinGroup(String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void joinGroup(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup _group = emaObject.fetchGroupSpecification(groupId, error, false);
		handleError(error);
		if (_group.groupSetting() == null) {
			throw new HyphenateException();
		}
		if (_group.groupSetting().style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_OPEN) {
			emaObject.joinPublicGroup(groupId, error);
			handleError(error);
			return;
		}
		if (_group.groupSetting().style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL) {
			emaObject.applyJoinPublicGroup(groupId, mClient.getCurrentUser(), "hello", error);
			handleError(error);
			return;
		}
	}

	/**
	 * \~chinese
	 * 当前登录用户加入公开群。
	 * 如果是自由加入的公开群，直接进入群组；需要验证的公开群需要等待群主同意。详见 {@link EMGroupStyle}。
	 * 同步方法见 {@link #joinGroup(String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
	 * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Joins a public group.
	 * For a group that requires no authentication，users can join it freely without the need of having permission.
	 * For a group that requires authentication, users need to wait for the owner to agree before joing the group. For details, see {@link EMGroupStyle}.
	 * For the synchronous method, see {@link #joinGroup(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncJoinGroup(final String groupId,
							   final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					joinGroup(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改群组名称。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncChangeGroupName(String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId 				要修改名称的群组的 ID。
	 * @param changedGroupName 		修改后的群组名称。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Changes the group name.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncChangeGroupName(String, String, EMCallBack)}.
	 * 
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The ID of group whose name is to be changed.
	 * @param changedGroupName 		The new group name.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void changeGroupName(String groupId, String changedGroupName) throws HyphenateException{
		EMAError error = new EMAError();
		emaObject.changeGroupSubject(groupId, changedGroupName, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 修改群组名称。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #changeGroupName(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId 			需要修改名称的群组的 ID。
	 * @param changedGroupName 	修改后的群组名称。
	 * @param callback			结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 							失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Changes the group name.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #changeGroupName(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 			The ID of group whose name is to be changed.
	 * @param changedGroupName 	The new group name.
	 * @param callback			The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 							if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncChangeGroupName(final String groupId,
									 final String changedGroupName,
									 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					changeGroupName(groupId, changedGroupName);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改群描述。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncChangeGroupDescription(String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId 				群组 ID。
	 * @param changedDescription 	修改后的群描述。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Changes the group description.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncChangeGroupDescription(String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The group ID.
	 * @param changedDescription 	The new group description.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void changeGroupDescription(String groupId, String changedDescription)
			throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.changeGroupDescription(groupId, changedDescription, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 修改群描述。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #changeGroupDescription(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId 				群组 ID。
	 * @param changedDescription 	修改后的群描述。
     * @param callBack              结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 	                   			失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Changes the group description.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #changeGroupDescription(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 				The group ID.
	 * @param changedDescription 	The new group description.
     * @param callBack				The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *								if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncChangeGroupDescription(final String groupId,
											final String changedDescription,
											final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override public void run() {
				try {
					changeGroupDescription(groupId, changedDescription);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 接受入群邀请。
	 * 异步方法见 {@link #asyncAcceptInvitation(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @param inviter				邀请者 ID。
	 * @return 						用户已经同意邀请的群组对象。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Accepts a group invitation.
	 * For the asynchronous method, see {@link #asyncAcceptInvitation(String, String, EMValueCallBack)}。
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param  groupId 				The group ID.
	 * @param  inviter 				The user who initiates the invitation.
	 * @return 						The group instance which the user has accepted the invitation to join.
	 * @throws HyphenateException	A description of the exception. see {@link EMError}.
	 */
	public EMGroup acceptInvitation(String groupId, String inviter) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.acceptInvitationFromGroup(groupId, inviter == null ? "" : inviter, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 接受入群邀请。
	 * 同步方法见 {@link #acceptInvitation(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
	 * @param inviter	邀请者 ID。
	 * @param callback  结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Accepts a group invitation.
	 * For the synchronous method, see {@link #acceptInvitation(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param inviter 	The inviter ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)}
	 *                  and returns the updated group instance;
	 *					if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncAcceptInvitation(final String groupId,
									  final String inviter,
									  final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMGroup group = acceptInvitation(groupId, inviter);
					callback.onSuccess(group);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 拒绝入群邀请。
	 * 异步方法见 {@link #asyncDeclineInvitation(String, String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @param inviter				邀请者 ID。
	 * @param reason				拒绝理由。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Declines a group invitation.
	 * For the asynchronous method, see {@link #asyncDeclineInvitation(String, String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The group ID.
	 * @param inviter 				The inviter.
	 * @param reason				The reason for declining.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void declineInvitation(
			String groupId,
			String inviter,
			String reason) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.declineInvitationFromGroup(groupId, inviter == null ? "" : inviter, reason == null ? "" : reason, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 拒绝入群邀请。
     * 同步方法见 {@link #declineInvitation(String, String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
	 * @param inviter	邀请人 ID。
	 * @param reason	拒绝理由。
	 * @param callback  结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Declines a group invitation.
	 * For the synchronous method, see {@link #declineInvitation(String, String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param inviter 	The inviter ID.
	 * @param reason	The reason for declining.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncDeclineInvitation(final String groupId,
									   final String inviter,
									   final String reason,
									   final EMCallBack callback) {

		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					declineInvitation(groupId, inviter, reason);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 批准入群申请。
     * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncAcceptApplication(String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param username 				申请人 ID。
	 * @param groupId  				群组 ID。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Approves a group request.
     * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncAcceptApplication(String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param username 				The ID of the user who sends a request to join the group.
	 * @param groupId  				The group ID.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void acceptApplication(String username, String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.acceptJoinGroupApplication(groupId, username, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 批准入群申请。
     * 仅群主和管理员可调用此方法。 
	 * 同步方法见 {@link #acceptApplication(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param username 	申请人 ID。
	 * @param groupId  	群组 ID。
	 * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Approves a group request.
     * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #acceptApplication(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param username 	The ID of the user who sends the request to join the group.
	 * @param groupId  	The group ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)} if it does not.
	 */
	public void asyncAcceptApplication(final String username,
									   final String groupId,
									   final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					acceptApplication(username, groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 拒绝入群申请。
     * 仅群组创建者和管理员可调用此方法。
	 * 异步方法见 {@link #asyncDeclineApplication(String, String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param username 				申请人 ID。
	 * @param groupId  				群组 ID。
	 * @param reason   				拒绝理由。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Declines a group request.
     * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncDeclineApplication(String, String, String, EMCallBack)}.
	 *
	 * Only the group owner or admin can call this method.
	 *
	 * @param username 				The ID of the user who sends the request for join the group.
	 * @param groupId  				The group ID.
	 * @param reason   				The reason of declining.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void declineApplication(String username, String groupId, String reason) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.declineJoinGroupApplication(groupId, username, reason, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 拒绝入群申请。
     * 仅群组创建者和管理员可调用此方法。
	 * 同步方法见 {@link #declineApplication(String, String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param username 	申请人 ID。
	 * @param groupId  	群组 ID。
	 * @param reason   	拒绝理由。
     * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
     *
	 * \~english
	 * Declines a group request.
     * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #declineApplication(String, String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param username 	The ID of the user who sends the request for join the group.
	 * @param groupId  	The group ID.
	 * @param reason   	The reason of declining.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncDeclineApplication(final String username,
                                        final String groupId,
                                        final String reason,
                                        final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					declineApplication(username, groupId, reason);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 邀请用户加入群组。
     * 前提：群组类型是 {@link EMGroupStyle#EMGroupStylePrivateMemberCanInvite}，即允许群成员邀请用户。
	 * 异步方法见 {@link #asyncInviteUser(String, String[], String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @param beInvitedUsernames	受邀用户的数组。
	 * @param reason				邀请原因。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Invites users to join a group.
	 * Note: The group style is {@link EMGroupStyle#EMGroupStylePrivateMemberCanInvite}, which allows group
	 * members to invite users.
	 * For the asynchronous method, see {@link #asyncInviteUser(String, String[], String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param beInvitedUsernames   	The array of users to be invited.
	 * @param reason               	The reason for invitation.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void inviteUser(String groupId, String[] beInvitedUsernames, String reason) throws HyphenateException {
		EMAError error = new EMAError();
		List<String> members = new ArrayList<String>();
		Collections.addAll(members, beInvitedUsernames);
		emaObject.addGroupMembers(groupId, members, reason, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 群成员邀请用户入群。
	 * 前提：群组类型是 {@link EMGroupStyle#EMGroupStylePrivateMemberCanInvite}，即允许群成员邀请用户。
	 * 同步方法见 {@link #inviteUser(String, String[], String)}。
	 *
	 * 异步方法。
     *
	 * @param groupId				群组 ID。
	 * @param beInvitedUsernames	受邀用户的数组。
	 * @param reason				邀请原因。
	 * @param callback				结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 								失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Invites users to join a group.
	 * Note: The group style is {@link EMGroupStyle#EMGroupStylePrivateMemberCanInvite}, which allows group
	 * members to invite users.
	 * For the synchronous method, see {@link #inviteUser(String, String[], String)}。
	 *
	 * The is an asynchronous method.
     *
	 * @param groupId				The group ID.
	 * @param beInvitedUsernames   	The array of users to be invited.
	 * @param reason               	The reason for invitation.
	 * @param callback				The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *								if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncInviteUser(final String groupId,
                                final String[] beInvitedUsernames,
                                final String reason,
                                final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					inviteUser(groupId, beInvitedUsernames, reason);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 申请加入群组。
	 * 前提：群组类型为 {@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}，即需要验证的公开群组。
     * 异步方法见 {@link #asyncApplyJoinToGroup(String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组ID。
	 * @param reason				申请加入的原因。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Requests to join a group.
     * Note: The group style is {@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}, which is a public group 
	 * requiring authentication.
	 * For an asynchronous method, see {@link #asyncApplyJoinToGroup(String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param reason    			The reason for requesting to join the group.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void applyJoinToGroup(String groupId, String reason) throws HyphenateException {
		String userName = mClient.getCurrentUser();
		EMAError error = new EMAError();
		emaObject.applyJoinPublicGroup(groupId, userName, reason, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 申请加入群组。
	 * 前提：群组类型为 {@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}，即需要需要验证的公开群。
     * 同步方法见 {@link #applyJoinToGroup(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
	 * @param reason    申请加入的原因。
	 * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Requests to join a group.
     * Note: The group style is {@link EMGroupStyle#EMGroupStylePublicJoinNeedApproval}, which is a public group requiring authentication.
	 * For the synchronous method, see {@link #applyJoinToGroup(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param reason    The reason for requesting to joining the group.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncApplyJoinToGroup(final String groupId,
                                      final String reason,
                                      final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					applyJoinToGroup(groupId, reason);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 屏蔽群消息。
	 * 被屏蔽群消息的用户仍是群成员，但无法接收群消息。
	 * 异步方法见 {@link #asyncBlockGroupMessage(String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
     *
	 * @param groupId				群组 ID。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 *\~english
	 * Blocks group messages.
     * The user that blocks group messages is still a group member, but can't receive group messages.
	 * For the asynchronous method, see {@link #asyncBlockGroupMessage(String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void blockGroupMessage(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.blockGroupMessage(groupId, error);
		handleError(error);
	}


	/**
	 * \~chinese
	 * 屏蔽群消息。
	 * 被屏蔽群消息的用户仍是群成员，但无法接收群消息。
     * 同步方法见 {@link #blockGroupMessage(String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
	 * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 *\~english
	 * Blocks group messages.
     * The user that blocks group messages is still a group member, but can't receive group messages.
	 * For the synchronous method, see {@link #blockGroupMessage(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncBlockGroupMessage(final String groupId, final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					blockGroupMessage(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 取消屏蔽群消息。
	 * 异步方法见 {@link #asyncUnblockGroupMessage(String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Unblocks group messages.
	 * For the asynchronous method, see {@link #asyncUnblockGroupMessage(String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void unblockGroupMessage(String groupId) throws HyphenateException {
		EMLog.d(TAG, "try to unblock group msg:" + groupId);
		EMAError error = new EMAError();
		emaObject.unblockGroupMessage(groupId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 取消屏蔽群消息。
     * 同步方法见 {@link #unblockGroupMessage(String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
	 * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Unblocks group messages.
     * For the synchronous method, see {@link #unblockGroupMessage(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncUnblockGroupMessage(final String groupId,
                                         final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					unblockGroupMessage(groupId);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将用户加入群组黑名单。
	 * 先将用户移出群，再加入黑名单。加入黑名单的用户无法加入群。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncBlockUser(String, String, EMCallBack)}。
	 *
	 * 
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId 				群组 ID。
	 * @param username 				要加入黑名单的用户名。
     * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 *\~english
	 * Adds the user to the group blocklist. 
	 * Users will be first removed from the group they have joined before being added to the group blocklist. The users on the group blocklist can not join the group again. 
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncBlockUser(String, String, EMCallBack)}.
	 *
	 *
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param username 				The user to be added to the blocklist.
     * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void blockUser(String groupId, String username) throws HyphenateException {
		EMLog.d(TAG, "block user for groupid:" + groupId + " username:" + username);
		EMAError error = new EMAError();
		String reason = "";
		List<String> members = new ArrayList<String>();
		members.add(username);
		emaObject.blockGroupMembers(groupId, members, error, reason);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 将用户加入群组黑名单。
	 * 先将用户移出群，再加入黑名单。加入黑名单的用户无法加入群。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #blockUser(String, String)}。
	 *
	 *
	 * 异步方法。
     *
	 * @param groupId 	群组 ID。
	 * @param username 	要加入黑名单的用户名。
	 * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 *\~english
	 * Adds the user to the group blocklist.
	 * Users will be first removed from the group they have joined before being added to the group blocklist. The users on the group blocklist can not join the group again. 
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #blockUser(String, String)}.
     *
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param username 	The user to be added to the blocklist.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					If this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncBlockUser(final String groupId,
                               final String username,
                               final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					blockUser(groupId, username);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
     * 将用户加入群组黑名单。
	 * 先将用户移出群，再加入黑名单。加入黑名单的用户无法加入群。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncBlockUsers(String, List, EMCallBack)}。
	 *
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId 				群组 ID。
	 * @param members 				要加入黑名单的用户列表。
     * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
     * Adds the user to the group blocklist.
	 * Users will be first removed from the group they have joined before being added to the group blocklist. The users on the group blocklist can not join the group again.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncBlockUsers(String, List, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param members 				The list of users to be added to the blocklist.
     * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void blockUsers(String groupId, List<String> members) throws HyphenateException {
		EMLog.d(TAG, "block user for groupid:" + groupId + " members:" + members == null ? "":members.toString());
		EMAError error = new EMAError();
		String reason = "";
		emaObject.blockGroupMembers(groupId, members, error, reason);
		handleError(error);
	}

	/**
	 * \~chinese
     * 将用户加入群组黑名单。
	 * 先将用户移出群，再加入黑名单。加入黑名单的用户无法加入群。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #blockUsers(String, List)}。
	 *
	 *
	 * 异步方法。
	 *
	 * @param groupId 	群组ID。
	 * @param members 	要加入黑名单的用户列表。
	 * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
     * Adds the user to the group blocklist.
	 * Users will be first removed from the group they have joined before being added to the group blocklist. The users on the group blocklist can not join the group again.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #blockUsers(String, List)}.
	 *
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param members 	The list of users to be blocked.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncBlockUsers(final String groupId,
                               final List<String> members,
                               final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					blockUsers(groupId, members);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将用户从群组黑名单中移除。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncUnblockUser(String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId 				群组 ID。
	 * @param username 				要从黑名单中移除的用户。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes a user from the group blocklist.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncUnblockUser(String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param username				The user to be removed from the group blocklist.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void unblockUser(String groupId, String username) throws HyphenateException{
		EMLog.d(TAG, "unblock user groupid:" + groupId + " username:" + username);
		EMAError error = new EMAError();
		List<String> members = new ArrayList<String>();
		members.add(username);
		emaObject.unblockGroupMembers(groupId, members, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 将用户从群组黑名单移除。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #unblockUser(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId 	群组 ID。
	 * @param username 	要从黑名单中移除的用户。
	 * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes a user from the group blocklist.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #unblockUser(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param username	The user to be removed from the group blocklist.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncUnblockUser(final String groupId,
                                 final String username,
                                 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					unblockUser(groupId, username);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将用户从群组黑名单移除。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncBlockUsers(String, List, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId 				群组 ID。
	 * @param members 				要从黑名单中移除的用户列表。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes users from the group blocklist.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncBlockUsers(String, List, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @param members				The users to be removed from the group blocklist.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void unblockUsers(String groupId, List<String> members) throws HyphenateException{
		EMLog.d(TAG, "unblock user groupid:" + groupId + " members:" + members == null ? "":members.toString());
		EMAError error = new EMAError();
		emaObject.unblockGroupMembers(groupId, members, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 将用户从群组黑名单移除。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #unblockUsers(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId 	群组 ID。
	 * @param members 	要从黑名单中移除的用户列表。
	 * @param callback	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes users from the group blocklist.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #unblockUsers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param members	The users to be removed from the bloclist.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncUnblockUsers(final String groupId,
                                 final List<String> members,
                                 final EMCallBack callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					unblockUsers(groupId, members);
					callback.onSuccess();
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取群组的黑名单列表, 默认最多取 200 个.
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncGetBlockedUsers(String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId				群组 ID。
	 * @return List					返回黑名单列表。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets the group blocklist that can contain a maximum of 200 users by default.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncGetBlockedUsers(String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @return List					The group blocklist.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public List<String> getBlockedUsers(String groupId) throws HyphenateException {
		return getBlockedUsers(groupId, 0, 200);
	}

	/**
	 * \~chinese
	 * 以分页方式获取群组的黑名单。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncGetBlockedUsers(String, int, int, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
     *
	 * @param groupId 				群组 ID。
	 * @param pageIndex 			当前页码，从 1 开始。
	 * @param pageSize 				每页返回的群组数。
	 * @return List					返回黑名单列表。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * 
	 * \~english
	 * Gets the group blocklist from server with pagination.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncGetBlockedUsers(String, int, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId               The group ID.
	 * @param pageIndex            	The page number, starting from 1.
	 * @param pageSize              The number of groups per page.
	 * @return List					The group blocklist.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public List<String> getBlockedUsers(String groupId, int pageIndex, int pageSize) throws HyphenateException{
		EMLog.d(TAG, "get blocked users of the group: " + groupId);
		EMAError error = new EMAError();
		List<String> members = emaObject.fetchGroupBlackList(groupId, pageIndex, pageSize, error);
		handleError(error);
		return members;
	}

	/**
	 * \~chinese
	 * 获取群组的黑名单列表, 默认最多取 200 个。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #getBlockedUsers(String)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId 	群组 ID。
	 * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回群组的黑名单列表；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the group blocklist that can contain a maximum of 200 users by default.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #getBlockedUsers(String)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                  if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 *					
	 */
	public void asyncGetBlockedUsers(final String groupId,
                                     final EMValueCallBack<List<String>> callback) {
		asyncGetBlockedUsers(groupId, 0, 200, callback);
	}

	/**
	 * \~chinese
	 * 以分页方式获取群组的黑名单。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #getBlockedUsers(String, int, int)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId     群组 ID。
	 * @param pageIndex   当前页码，从 1 开始。
	 * @param pageSize    每页返回的群组数。
	 * @param callback	  结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                    返回黑名单列表；
	 * 					  失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the group blocklist with pagination.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #getBlockedUsers(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId     The group ID.
	 * @param pageIndex   The page number, starting from 1.
	 * @param pageSize    The number of groups per page.
	 * @param callback	  The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and returns the group blocklist;
	 *                    if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 *					  
	 */
	public void asyncGetBlockedUsers(final String groupId,
                                     final int pageIndex,
                                     final int pageSize,
                                     final EMValueCallBack<List<String>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					List<String> users = getBlockedUsers(groupId, pageIndex, pageSize);
					callback.onSuccess(users);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 注册群变动事件监听。
	 * 注册后，需要配合 {@link #removeGroupChangeListener(EMGroupChangeListener)} 使用。
	 *
	 * @param listener	注册的群组事件监听。
	 *
	 * \~english
	 * Registers a group change listener.
	 * The registered listener needs to be used together with {@link #removeGroupChangeListener(EMGroupChangeListener)}.
	 *
	 * @param listener	The group event listener to be registered.
	 */
	public void addGroupChangeListener(EMGroupChangeListener listener) {
		EMLog.d(TAG, "add group change listener:" + listener.getClass().getName());
		if (!groupChangeListeners.contains(listener)) {
			groupChangeListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除群组变化监听器。
	 * 该方法在注册 {@link #addGroupChangeListener(EMGroupChangeListener)}后调用。
     *
	 * @param listener	要移除的群组监听器。
	 *
	 * \~english
	 * Removes a group change listener.
     * This method removes a group change listener registered with {@link #addGroupChangeListener(EMGroupChangeListener)}.
	 *
	 * @param listener 	The group event listener to be removed.
	 */
	public void removeGroupChangeListener(EMGroupChangeListener listener) {
		if (listener != null) {
			EMLog.d(TAG, "remove group change listener:" + listener.getClass().getName());
			groupChangeListeners.remove(listener);
		}
	}


	// ============================= group_reform new add api begin =============================
	/**
	 * \~chinese
	 * 以分页方式获取群组成员列表。
	 * 返回的结果中，当 EMCursorResult.getCursor() 为空字符串 ("") 时，表示没有更多数据。
	 * 异步方法见 {@link #asyncFetchGroupMembers(String, String, int, EMValueCallBack)}
	 *
     * 同步方法，会阻塞当前线程。
	 *
	 * 例如:
	 * ```java
	 *     EMCursorResult result = fetchGroupMembers(groupId, cursor, pageSize);    // search 1
	 * 	   result = fetchGroupMembers(groupId, result.getCursor(), pageSize);       // search 2
	 * ```
     *
	 * @param groupId				群组 ID。
	 * @param cursor				从这个游标位置开始取数据，首次获取数据时传 null 即可。
	 * @param pageSize				每页返回的群组成员数。
	 * @return						分页获取结果 {@link EMCursorResult}，包含用于下次获取数据的 cursor 以及群组成员 list，
	 * 								如果是最后一页，cursor 返回结果为空字符串。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
     * Gets a group's member list with pagination.
	 * When EMCursorResult.getCursor() is an empty string ("") in the result, there is no more data.
     * For the asynchronous method, see {@link #asyncFetchGroupMembers(String, String, int, EMValueCallBack)}.
	 *
     * Synchronization method will block the current thread.
	 * 
	 * For example:
	 * ```java
	 *     EMCursorResult result = fetchGroupMembers(groupId, cursor, pageSize);    // search 1
	 * 	   result = fetchGroupMembers(groupId, result.getCursor(), pageSize);       // search 2
	 * ```
     *
	 * @param groupId				The group ID.
	 * @param cursor				The cursor position from which to start to get data next time. Sets the parameter as null for the first time.
	 * @param pageSize				The number of group members per page.
	 * @return 						The result of {@link EMCursorResult}, 
	 *                              including the cursor for getting data next time and the group member list.
	 *                              For the last page, the return value of cursor is an empty string.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}
	 */
	public EMCursorResult<String> fetchGroupMembers(String groupId, String cursor, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<String> result = emaObject.fetchGroupMembers(groupId, cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 以分页方式获取群组成员列表。
	 * 同步方法见 {@link #fetchGroupMembers(String, String, int)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
	 * @param cursor	从这个游标位置开始取数据，首次获取数据时传 null 即可。
	 * @param pageSize	每页返回的群组成员数。
	 * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 					返回分页获取结果 {@link EMCursorResult}，包含用于下次获取数据的 cursor 以及群组成员集合的对象，
	 * 					如果是最后一页，cursor 返回结果为空字符串；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets a group's member list with pagination.
	 * For the synchronous method, see {@link #fetchGroupMembers(String, String, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId	The group ID.
	 * @param cursor	The number of group members per page.
	 * @param pageSize	The cursor position from which to start to get data next time. Sets the parameter as null for the first time.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and 
	 *                  returns the result of {@link EMCursorResult}, including the cursor for getting data next time and the group member list.
	 *               	For the last page, the return value of cursor is an empty string.
	 *					If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchGroupMembers(final String groupId,
                                       final String cursor,
                                       final int pageSize,
                                       final EMValueCallBack<EMCursorResult<String>> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMCursorResult<String> result = fetchGroupMembers(groupId, cursor, pageSize);
					callback.onSuccess(result);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 移交群组所有权。
     * 仅群主可调用此方法。
	 * 异步方法见 {@link #asyncChangeOwner(String, String, EMValueCallBack)}。
	 *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId				群组 ID。
	 * @param newOwner				新的群主。
	 * @return						返回更新后的群组对象
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息详见 {@link EMError}
	 *
	 * \~english
     * Transfers the group ownership.
	 * Only the group owner can call this method.
	 * For the asynchronous method, see {@link #asyncChangeOwner(String, String, EMValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param newOwner				The new owner ID.
	 * @return						The updated group instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMGroup changeOwner(String groupId, String newOwner) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.transferGroupOwner(groupId, newOwner, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 移交群组所有权。
	 * 仅群主可调用此方法。
     * 同步方法见 {@link #changeOwner(String, String)}
	 *
	 * 异步方法。
	 *
	 * @param groupId	群组 ID。
	 * @param newOwner	新的群主。
	 * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Transfers the group ownership.
	 * Only the group owner can call this method.
	 * For the synchronous method, see {@link #changeOwner(String, String)}。
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param newOwner	The new owner ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and
	 *                  returns the updated group instance;
	 *					if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeOwner(final String groupId,
                                 final String newOwner,
                                 final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMGroup group = changeOwner(groupId, newOwner);
					callback.onSuccess(group);
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 添加群组管理员。
	 * 仅群主可调用此方法，admin 无权限。
	 * 异步方法见 {@link #asyncAddGroupAdmin(String, String, EMValueCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId				群组 ID。
	 * @param admin					新增的管理员 ID。
	 * @return						返回更新后的群组对象。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Adds a group admin.
	 * Only the group owner can call this method and admin can not.
	 * For the asynchronous method, see {@link #asyncAddGroupAdmin(String, String, EMValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param admin					The admin ID to add.
	 * @return						The updated group instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMGroup addGroupAdmin(final String groupId, final String admin) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.addGroupAdmin(groupId, admin, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 添加群组管理员。
	 * 仅群主可调用此方法，admin 无权限。
	 * 同步方法见 {@link #addGroupAdmin(String, String)}。
	 *
	 * 异步方法。
     *
	 * @param groupId	群组 ID。
	 * @param admin		新增加的管理员 ID。
	 * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
     * Adds a group admin.
	 * Only the group owner can call this method and admin can not.
	 * For the synchronous method, see {@link #addGroupAdmin(String, String)}
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param admin		The new owner ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and 
	 *                  returns the updated group instance;
	 *					If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncAddGroupAdmin(final String groupId,
                                   final String admin,
                                   final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(addGroupAdmin(groupId, admin));
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 删除群组管理员。
	 * 仅群主可调用此方法。
	 * 异步方法见 {@link #asyncRemoveGroupAdmin(String, String, EMValueCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId				群组 ID。
	 * @param admin					删除的管理员 ID。
	 * @return						返回更新后的群组对象。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes a group admin.
	 * Only the group owner can call this method.
	 * For the asynchronous method, see {@link #asyncRemoveGroupAdmin(String, String, EMValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param admin					The admin ID to remove.
	 * @return						The updated group instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMGroup removeGroupAdmin(String groupId, String admin) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.removeGroupAdmin(groupId, admin, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 删除群组管理员。
	 * 仅群主可调用此方法。
	 * 同步方法见 {@link #removeGroupAdmin(String, String)}。
	 *
	 * 异步方法。
     *
	 * @param groupId	群组 ID。
	 * @param admin		删除的管理员 ID。
	 * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes a group admin.
	 * Only the group owner can call this method.
	 * For the synchronous method, see {@link #removeGroupAdmin(String, String)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param admin		The admin ID to remove.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and
	 *                  returns the updated group instance;
	 *					if this call succeeds, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncRemoveGroupAdmin(final String groupId,
                                      final String admin,
                                      final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(removeGroupAdmin(groupId, admin));
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将一组成员禁言。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #aysncMuteGroupMembers(String, List, long, EMValueCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId				群组 ID。
	 * @param muteMembers 			要禁言的用户列表。
	 * @param duration 				禁言时间，单位是毫秒。
	 * @return						返回更新后的群组对象。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
     * Mutes group members.
     * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #aysncMuteGroupMembers(String, List, long, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param muteMembers 			The list of members to be muted.
	 * @param duration 				The mute duration in milliseconds.
	 * @return						The updated group instance.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMGroup muteGroupMembers(String groupId, List<String> muteMembers, long duration) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.muteGroupMembers(groupId, muteMembers, duration, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 将一组成员禁言。
	 * 仅群主和管理员可调用此方法。
     * 同步方法见 {@link #muteGroupMembers(String, List, long)}。
	 *
	 * 异步方法。
	 *
	 * @param groupId		群组 ID。
	 * @param muteMembers	要禁言的用户列表。
	 * @param duration 		禁言时间，单位是毫秒。
	 * @param callback		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      返回更新后的群组对象；
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
     * Mutes group members.
     * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #muteGroupMembers(String, List, long)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId		The group ID.
	 * @param muteMembers 	The list of members to be muted.
	 * @param duration 		The mute duration in milliseconds.
	 * @param callback		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and 
	 *                      returns the updated group instance;
	 *						If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void aysncMuteGroupMembers(final String groupId,
                                      final List<String> muteMembers,
                                      final long duration, final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(muteGroupMembers(groupId, muteMembers, duration));
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解除禁言。
     * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncUnMuteGroupMembers(String, List, EMValueCallBack)}。
	 *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId				群组 ID。
	 * @param members				要解除禁言的用户列表。
	 * @return						返回更新后的群组对象。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Unmutes group members.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncUnMuteGroupMembers(String, List, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
     * @param groupId				The group ID.
     * @param members				The list of members to be muted.
     * @return 						The updated group instance.
     * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMGroup unMuteGroupMembers(String groupId, List<String> members) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.unMuteGroupMembers(groupId, members, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 解除禁言。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #unMuteGroupMembers(String, List)}。
	 *
	 * 异步方法。
     *
	 * @param groupId	群组 ID。
	 * @param members	要解除禁言的用户列表。
	 * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Unmutes group members.
	 * Only the group owner or admin can call this method.
	 * For the synchronious method, see {@link #unMuteGroupMembers(String, List)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param members	The list of members to be unmuted.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and 
	 *                  returns the updated group instance;
	 *					if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncUnMuteGroupMembers(final String groupId,
                                        final List<String> members,
                                        final EMValueCallBack<EMGroup> callback) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callback.onSuccess(unMuteGroupMembers(groupId, members));
				} catch (HyphenateException e) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取群组的禁言列表。
	 * 仅聊天室创建者和管理员可调用此方法。
	 * 异步方法见 {@link #asyncFetchGroupMuteList(String, int, int, EMValueCallBack)}
     *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId				群组 ID。
	 * @param pageNum				当前页码，从 1 开始。
	 * @param pageSize				每页返回的禁言成员数。
	 * @return 						返回的包含禁言成员 ID 及其禁言时长的 Map，其中每一项的 key 是禁言的成员 ID，value 是禁言时间，单位是毫秒。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets the mutelist of the group from the server.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncFetchGroupMuteList(String, int, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId				The group ID.
	 * @param pageNum				The page number, starting from 1.
	 * @param pageSize				The cursor position from which to start to get data next time.
	 * @return 						A map which contains the muted member ID and mute duration, where the key of each entry
	 * 								is the ID of a muted member and the value is the mute duration in milliseconds.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public Map<String, Long> fetchGroupMuteList(String groupId, int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		Map<String, Long> muteMembers = emaObject.fetchGroupMutes(groupId, pageNum, pageSize, error);
		handleError(error);
		return muteMembers;
	}

	/**
	 * \~chinese
	 * 获取群组的禁言列表。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #fetchGroupMuteList(String, int, int)}。
	 *
	 * 异步方法。
     *
	 * @param groupId	群组 ID。
	 * @param pageNum	当前页码，从 1 开始。
	 * @param pageSize	每页返回的禁言成员数。
	 * @param callBack 	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回的包含禁言成员 ID 及其禁言时长的 Map，其中每一项的 key 是禁言的成员 ID，value 是禁言时长，单位是毫秒;
	 *                  失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the mutelist of the group from the server.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #fetchGroupMuteList(String, int, int)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param pageNum	The page number, starting from 1.
	 * @param pageSize	The number of muted members per page.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and
	 *                  The map which contains the muted member ID and mute duration, where the key of each entry
	 * 					is the ID of a muted member and the value is the mute duration in milliseconds.
	 *                  If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 *
	 */
	public void asyncFetchGroupMuteList(final String groupId,
                                        final int pageNum,
                                        final int pageSize,
                                        final EMValueCallBack<Map<String, Long>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupMuteList(groupId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 以分页方式从服务器获群组黑名单。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncFetchGroupBlackList(String, int, int, EMValueCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId       		群组 ID。
	 * @param pageNum       		当前页码，从 1 开始。
	 * @param pageSize      		每页返回的黑名单上的用户数。
	 * @return						返回分页获取的群组黑名单列表。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets the blocklist of group from the server with pagination.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncFetchGroupBlackList(String, int, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId       		The group ID.
	 * @param pageNum       		The page number, starting from 1.
	 * @param pageSize      		The number of users on the blocklist per page.
	 * @return						The group blocklist on the next page.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public List<String> fetchGroupBlackList(String groupId, int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		List<String> muteMembers = emaObject.fetchGroupBlackList(groupId, pageNum, pageSize, error);
		handleError(error);
		return muteMembers;
	}

	/**
	 * \~chinese
	 * 以分页从服务器获群组黑名单。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见 {@link #fetchGroupBlackList(String, int, int)}。
	 *
	 * 异步方法。
     *
	 * @param groupId       群组 ID。
	 * @param pageNum       当前页码，从 1 开始。
	 * @param pageSize      每页返回的黑名单上的用户数。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      返回分页获取的黑名单列表；
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the blocklist of group from the server with pagination.
	 * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #fetchGroupBlackList(String, int, int)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId       The group ID.
	 * @param pageNum       The page number, starting from 1.
	 * @param pageSize      The number of users on the blocklist per page.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and
	 *                      returns the group blocklist on the next page;
	 *						if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchGroupBlackList(final String groupId,
                                         final int pageNum,
                                         final int pageSize,
                                         final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupBlackList(groupId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 添加白名单。
	 * 仅群组创建者和管理员可调用此方法。
	 *
	 * 异步方法。 
	 *
	 * @param groupId 	群组 ID。
	 * @param members 	要添加的成员列表。
	 * @param callBack  结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。 
	 *
	 * \~english
	 * Adds members to the allowlist.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param members 	The members to be added to the allowlist.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void addToGroupWhiteList(final String groupId, final List<String> members, final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					emaObject.addToWhiteList(groupId, members, error);
					handleError(error);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 移除白名单。
	 * 仅群组创建者和管理员可调用此方法。
	 *
	 * 异步方法。
	 *
	 * @param groupId 	群组 ID。
	 * @param members 	需要从白名单中移除的成员 ID 列表。
	 * @param callBack  结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes members from the allowlist.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param members 	The members to be removed from the allowlist.
	 * @param callBack  The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void removeFromGroupWhiteList(final String groupId, final List<String> members, final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					emaObject.removeFromWhiteList(groupId, members, error);
					handleError(error);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 检查自己是否在群组白名单中。
	 *
	 * 异步方法。
	 *
	 * @param groupId 	群组 ID。
	 * @param callBack	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回是否在群组白名单的布尔值；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets whether the member is on the allowlist.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and
	 *                  returns a Boolean value to indicate whether the current user is on the group blocklist;
	 *					if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void checkIfInGroupWhiteList(final String groupId, EMValueCallBack<Boolean> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfInWhiteList(groupId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});

	}

	/**
	 * \~chinese
	 * 获取群组白名单列表。
	 * 仅聊天室创建者和管理员可调用此方法。
	 *
	 * 异步方法。
	 *
	 * @param groupId 	群组 ID。
	 * @param callBack	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回白名单列表；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the allowlist of group from the server.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and
	 *                  return the group blocklist;
	 *					if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void fetchGroupWhiteList(final String groupId, final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					List<String> result = emaObject.fetchGroupWhiteList(groupId, error);
					handleError(error);
					callBack.onSuccess(result);
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}


    /**
     * \~chinese
     * 设置全员禁言。
	 * 仅群主和管理员可调用此方法。
	 *
	 * 异步方法。
	 *
     * @param groupId 	群组 ID。
	 * @param callBack	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
     *
     * \~english
     * Mutes all members.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
     * @param groupId 	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and 
	 *                  returns the updated group instance;
	 *					if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
     */
    public void muteAllMembers(final String groupId, final EMValueCallBack<EMGroup> callBack) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMAGroup group = emaObject.muteAllMembers(groupId, error);
                    handleError(error);
                    callBack.onSuccess(new EMGroup(group));
                } catch (HyphenateException e) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 解除全员禁言。
	 * 仅群主和管理员可调用此方法。
	 *
	 * 异步方法。
	 *
     * @param groupId 	群组 ID。
	 * @param callBack	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回更新后的群组对象；
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
     *
     * \~english
     * Unmutes all members.
	 * Only the group owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
     * @param groupId 	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and
	 *                  returns the updated group instance;
	 *					if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
     */
    public void unmuteAllMembers(final String groupId, final EMValueCallBack<EMGroup> callBack) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMAGroup group = emaObject.unmuteAllMembers(groupId, error);
                    handleError(error);
                    callBack.onSuccess(new EMGroup(group));
                } catch (HyphenateException e) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 更新群公告。
	 * 仅群主和管理员可调用此方法。
	 * 异步方法见 {@link #asyncUpdateGroupAnnouncement(String, String, EMCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId 				群组 ID。
	 * @param announcement 			公告内容。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Updates the group announcement.
	 * Only the group owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncUpdateGroupAnnouncement(String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The group ID.
	 * @param announcement 			The group announcement.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void updateGroupAnnouncement(String groupId, String announcement) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.updateGroupAnnouncement(groupId, announcement, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 更新群公告。
	 * 仅群主和管理员可调用此方法。
	 * 同步方法见{@link #updateGroupAnnouncement(String, String)}。
	 *
	 * 异步方法。
     *
	 * @param groupId 		群组 ID。
	 * @param announcement 	公告内容。
	 * @param callBack		结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 						失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Updates the group announcement.
     * Only the group owner or admin can call this method.
	 * For the synchronous method, see {@link #updateGroupAnnouncement(String, String)}
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 		The group ID.
	 * @param announcement 	The group announcement.
	 * @param callBack 		The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 						if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncUpdateGroupAnnouncement(final String groupId,
                                             final String announcement,
                                             final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateGroupAnnouncement(groupId, announcement);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取群组公告。
	 * 群成员可调用此方法。
     * 异步方法见 {@link #asyncFetchGroupAnnouncement(String, EMValueCallBack)}。
	 *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId 				群组 ID。
	 * @return						群组公告。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets the group announcement from the server.
	 * Group members can call this method.
	 * For the asynchronous method, see {@link #asyncFetchGroupAnnouncement(String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId				The group ID.
	 * @return						The group announcement.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public String fetchGroupAnnouncement(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.fetchGroupAnnouncement(groupId, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取群公告。
	 * 群成员可调用此方法。
	 * 同步方法见 {@link #fetchGroupAnnouncement(String)}。
	 *
	 * 异步方法。
     *
	 * @param groupId 	群组 ID。
	 * @param callBack	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the group announcement from the server.
	 * Group members can call this method.
	 * For the synchronous method, see {@link #fetchGroupAnnouncement(String)}.
	 *
	 * This is an asynchronous method.
     *
	 * @param groupId	The group ID.
	 * @param callBack	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 					if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchGroupAnnouncement(final String groupId,
                                            final EMValueCallBack<String> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupAnnouncement(groupId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 上传共享文件至群组。
	 * 注意：callback 只做进度回调用。
	 * 异步方法见 {@link #asyncUploadGroupSharedFile(String, String, EMCallBack)}。
	 *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId 				群组 ID。
	 * @param filePath 				文件本地路径。
	 * @param callBack 				上传文件进度回调。
	 * @return EMMucSharedFile		返回共享文件属性对象。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Uploads the shared file to the group.
	 * Note: The callback is only used for progress callback.
	 * For the asynchronous method, see {@link #asyncUploadGroupSharedFile(String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId 				The group ID.
	 * @param filePath 				The local file path.
	 * @param callBack 				The file upload progress callback.
	 * @return						The attribute object of the shared file.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMMucSharedFile uploadGroupSharedFile(String groupId, String filePath, EMCallBack callBack) throws HyphenateException {
		EMAError error = new EMAError();
		EMACallback aCallback = new EMACallback(callBack);
		EMAMucShareFile aFile = emaObject.uploadGroupShareFile(groupId, filePath, aCallback, error);
		handleError(error);
		callBack.onSuccess();

		return new EMMucSharedFile(aFile);
	}


	/**
	 * \~chinese
	 * 上传共享文件至群组。
	 * 同步方法见 {@link #uploadGroupSharedFile(String, String, EMCallBack)}。
	 *
	 * 异步方法。
     *
	 * @param groupId 	群组 ID。
	 * @param filePath 	文件本地路径。
	 * @param callBack 	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 *                  上传文件进度回调 {@link EMCallBack#onProgress(int, String)}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 *
	 * \~english
	 * Uploads the shared file to the group.
     * For the synchronous method, see {@link #uploadGroupSharedFile(String, String, EMCallBack)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param filePath 	The local file path.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 *                  Calls {@link EMCallBack#onSuccess()} to view the upload progress. 
	 */
	public void asyncUploadGroupSharedFile(final String groupId,
                                           final String filePath,
                                           final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMMucSharedFile file = uploadGroupSharedFile(groupId, filePath, callBack);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的共享文件列表。
	 * 异步方法见 {@link #asyncFetchGroupSharedFileList(String, int, int, EMValueCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId 				群组 ID。
	 * @param pageNum 				当前页码，从 1 开始。
	 * @param pageSize 				每页返回的共享文件数。
	 * @return						返回共享文件列表。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets the shared files of group from the server.
     * For the asynchronous method, see {@link #asyncFetchGroupSharedFileList(String, int, int, EMValueCallBack)}.
	 *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId 				The group ID.
	 * @param pageNum 				The page number, starting from 1.
	 * @param pageSize 				The number of shared files per page.
	 * @return						The shared files.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public List<EMMucSharedFile> fetchGroupSharedFileList(String groupId, int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		List<EMAMucShareFile> files = emaObject.fetchGroupShareFiles(groupId, pageNum, pageSize, error);
		handleError(error);
		List<EMMucSharedFile> result = new ArrayList<EMMucSharedFile>();
		for(EMAMucShareFile file : files){
			result.add(new EMMucSharedFile(file));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取群共享文件列表。
	 * 同步方法见 {@link #fetchGroupSharedFileList(String, int, int)}。
	 *
	 * 异步方法。
     *
	 * @param groupId 	群组 ID。
	 * @param pageNum 	当前页码，从 1 开始。
	 * @param pageSize 	每页返回的共享文件数。
	 * @param callBack 	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  返回共享文件列表;
	 * 					失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * 需注意以下两点：
 
     *  - pageSize 是这次调用期望返回的记录数，如当前在最后一页，返回的数量小于 pageSize 的值；
     *  - pageNum 是当前页码，方便服务器分页查询返回。
	 *    对于数据量未知且很大的情况，分页获取，服务器会根据每次的 pageSize 和 pageNum 的设置返回数据。

	 * \~english
	 * Gets the shared file list from the server.
     * For the synchronous method, see {@link #fetchGroupSharedFileList(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param pageNum 	The page number, starting from 1.
	 * @param pageSize 	The number of shared files per page.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and
	 *                  returns the shared files; if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 *
	 *  Note the following:
     * - pageSize specifies the number of shared files expected to return for this call. 
	 *   For the last page, the actual number of returned shared files is less than the value of pageSize.
     * - pageNum specifies the number of the current page on which the returned data is presented. 
	 *   For a large but unknown quantity of data, the server will return data as specified by pageSize and pageNum.
				
	 */
	public void asyncFetchGroupSharedFileList(final String groupId,
                                              final int pageNum,
                                              final int pageSize,
                                              final EMValueCallBack<List<EMMucSharedFile>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupSharedFileList(groupId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 删除群组指定的共享文件。
	 * 成员可以删除自己上传的文件，群组拥有者或者群组管理员可以删除所有的共享文件。
	 * 异步方法见 {@link #asyncDeleteGroupSharedFile(String, String, EMCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId 				群组 ID。
	 * @param fileId 				文件 ID。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Removes a shared file of the group.
     * Group members can delete their own uploaded files. The group owner or admin can delete all shared files.
	 * For the asynchronous method, see {@link #asyncDeleteGroupSharedFile(String, String, EMCallBack)}.
	 *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId 				The group ID.
	 * @param fileId 				The shared file ID.
	 * @throws HyphenateException	Only the group owner or admin can call this method. See {@link EMError}.
	 */
	public void deleteGroupSharedFile(String groupId, String fileId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.deleteGroupShareFile(groupId, fileId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 删除群组指定的共享文件。
	 * 成员可以删除自己上传的文件，群组拥有者或者群组管理员可以删除所有的共享文件。
	 * 同步方法见 {@link #deleteGroupSharedFile(String, String)}。
	 *
	 * 异步方法。
     *
	 * @param groupId 	群组 ID。
	 * @param fileId 	文件 ID。
	 * @param callBack 	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes the shared file of the group.
     * Group members can delete their own uploaded files. The group owner or admin can delete all shared files.
	 * For the synchronous method, see {@link #deleteGroupSharedFile(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId 	The group ID.
	 * @param fileId 	The shared file ID.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncDeleteGroupSharedFile(final String groupId,
                                           final String fileId,
                                           final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					deleteGroupSharedFile(groupId, fileId);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 下载群组中指定的共享文件。
	 * 注意：callback 只做进度回调用。
	 * 异步方法见 {@link #asyncDownloadGroupSharedFile(String, String, String, EMCallBack)}。
     *
     * 同步方法，会阻塞当前线程。
     *
	 * @param groupId 				群组 ID。
	 * @param fileId 				文件 ID。
	 * @param savePath 				文件保存路径。
	 * @param callBack 				下载文件进度回调。
	 * @throws HyphenateException	如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Downloads the shared file of the group.
	 * Note: The callback is only used for progress callback.
	 * For the asynchronous method, see {@link #asyncDownloadGroupSharedFile(String, String, String, EMCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param groupId 				The group ID.
	 * @param fileId 				The ID of the shared file.
	 * @param savePath 				The local file path.
	 * @param callBack 				The fiel downlod progress callback.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void downloadGroupSharedFile(String groupId,
                                        String fileId,
                                        String savePath,
                                        EMCallBack callBack) throws HyphenateException {
		EMAError error = new EMAError();
		EMACallback aCallback = new EMACallback(callBack);
		emaObject.downloadGroupShareFile(groupId, fileId, savePath, aCallback, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 下载群组中指定的共享文件。
	 * 同步方法见 {@link #downloadGroupSharedFile(String, String, String, EMCallBack)}。
	 *
	 * 异步方法。
     *
	 * @param groupId   群组 ID。
	 * @param fileId    共享文件 ID。
	 * @param savePath  文件保存路径。
	 * @param callBack 	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 					下载文件进度回调 {@link EMCallBack#onProgress(int, String)}，
	 * 					失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Downloads the shared file of the group.
     * For synchronous method, see {@link #downloadGroupSharedFile(String, String, String, EMCallBack)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param groupId   The group ID.
	 * @param fileId    The shared file ID.
	 * @param savePath  The file path.
	 * @param callBack 	The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *					if this call fails, calls {@link EMCallBack#onError(int, String)}. 
	 *	                Calls {@link EMCallBack#onProgress(int, String)} to view the file download progress.
	 */
	public void asyncDownloadGroupSharedFile(final String groupId,
											 final String fileId,
											 final String savePath,
											 final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					downloadGroupSharedFile(groupId, fileId, savePath, callBack);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	// ============================= group_reform new add api end =============================

	/**
	 * \~chinese
	 * 更新群组扩展字段。
	 * 仅群主和管理员可调用此方法。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param groupId       群组 ID。
	 * @param extension     群组扩展字段。
	 * @throws HyphenateException   如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
	 *
	 * \~english
	 * Updates the group extension field.
	 * Only the group owner or admin can call this method.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param groupId       		The group ID.
	 * @param extension     		The group extension field.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public EMGroup updateGroupExtension(String groupId, String extension) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.updateGroupExtension(groupId, extension, error);
		handleError(error);
		return new EMGroup(group);
	}

	void onLogout() {
	}

	private void handleError(EMAError error)  throws HyphenateException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new HyphenateException(error);
		}
	}
}
