/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import static com.hyphenate.util.EasyUtils.convertToCerts;
import static com.hyphenate.util.EasyUtils.getSystemDefaultTrustManager;

import android.annotation.TargetApi;
import android.app.Activity;
import android.app.Application;
import android.app.Application.ActivityLifecycleCallbacks;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.http.X509TrustManagerExtensions;
import android.os.Build;
import android.os.Bundle;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.text.TextUtils;
import android.util.Log;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMConnectionListener;
import com.hyphenate.EMError;
import com.hyphenate.EMMultiDeviceListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatClient;
import com.hyphenate.chat.adapter.EMAChatClient.EMANetwork;
import com.hyphenate.chat.adapter.EMAConnectionListener;
import com.hyphenate.chat.adapter.EMADeviceInfo;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAMultiDeviceListener;
import com.hyphenate.chat.adapter.EMANetCallback;
import com.hyphenate.chat.core.EMAdvanceDebugManager;
import com.hyphenate.chat.core.EMChatConfigPrivate;
import com.hyphenate.chat.core.EMPreferenceUtils;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.monitor.EMNetworkMonitor;
import com.hyphenate.notification.core.EMNotificationHelper;
import com.hyphenate.push.EMPushConfig;
import com.hyphenate.push.EMPushHelper;
import com.hyphenate.push.EMPushType;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.NetUtils;
import com.hyphenate.util.PathUtil;
import com.hyphenate.util.Utils;

import org.json.JSONException;
import org.json.JSONObject;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Pattern;

import javax.net.ssl.X509TrustManager;

import internal.com.getkeepsafe.relinker.ReLinker;

/**
  * \~chinese
  * 该类是 Chat SDK 的入口，负责登录、退出及连接管理等，由此可以获得其他模块的入口。
  * 
  * ```java
  * EMChatManager chatManager = EMClient.getInstance().chatManager();
  * ```
  *
  * \~english
  * The EMClient, which is the entry point of the Chat SDK. You can log in, log out, and access other functionalities such as group and chatroom with this class. 
  *  
  * ```java
  * EMChatManager chatManager = EMClient.getInstance().chatManager();
  * ```
  */
public class EMClient {
    public final static String TAG = "EMClient";
    static private EMClient instance = null;
    static boolean libraryLoaded = false;

    private EMGroupManager groupManager;
    private EMChatRoomManager chatroomManager;
    private EMChatManager chatManager;
    private EMContactManager contactManager;
    private EMUserInfoManager userInfoManager;
    private EMPushManager pushManager;
    private EMTranslationManager translationManager;
    private volatile EMPresenceManager presenceManager;
    private EMChatThreadManager threadManager;

    private EMAChatClient emaObject;
    private Context mContext;
    private ExecutorService executor = null;
    private ExecutorService mainQueue = Executors.newSingleThreadExecutor();
    private ExecutorService sendQueue = Executors.newSingleThreadExecutor();
    private EMEncryptProvider encryptProvider = null;
    private boolean sdkInited = false;
    private EMChatConfigPrivate mChatConfigPrivate;
    private List<EMConnectionListener> connectionListeners = Collections.synchronizedList(new ArrayList<EMConnectionListener>());

	private MyConnectionListener connectionListener;
	private EMSmartHeartBeat smartHeartbeat = null;
    private List<EMMultiDeviceListener> multiDeviceListeners = Collections.synchronizedList(new ArrayList<EMMultiDeviceListener>());
    private MyMultiDeviceListener multiDeviceListenerImpl;
	private WakeLock wakeLock;

	private ConnectivityManager connManager;
	private EMANetwork currentNetworkType = EMANetwork.NETWORK_NONE;
    private boolean mIsLoginWithAgoraToken = false;

	public final static String VERSION = "3.9.4";

	/**
     * \~chinese
     * 请确保 EMClient 由 IM SDK 来初始化。
     * 
     * \~english
	 * Make sure that this EMClient is initialized by the Chat SDK.
	 *
	 */
	private EMClient() {
	}

	public static EMClient getInstance() {
		if(instance == null){
		    synchronized (EMClient.class) {
		        if(instance == null) {
                    instance = new EMClient();
		        }
            }
		}
		return instance;
	}


    /**
    * \~chinese
    * 初始化 SDK。
    * 
    * 请确保在主进程中进行初始化。
    * 
    * @param context 上下文，不可为空。
    * @param options 配置项，不可为空，见 {@link EMOptions}。
    *
    * \~english
    * Initializes the SDK.
    * 
    * Make sure to initialize the SDK in the main thread.
    *
    * @param context Make sure to set the param.
    * @param options The configurations. Make sure to set the param, see {@link EMOptions}.
    */
    public void init(Context context, EMOptions options) {
        if(sdkInited){
            return;
        }

        final EMTimeTag tag = new EMTimeTag();
        tag.start();

        mContext = context.getApplicationContext();
        connManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        //if android sdk >= 14,register callback to know app enter the background or foreground
        registerActivityLifecycleCallbacks();

        loadLibrary();

        mChatConfigPrivate = new EMChatConfigPrivate();
        mChatConfigPrivate.load(context, options);
        options.setConfig(mChatConfigPrivate);

        // multi-device is enabled by setDeviceUuid
        // config setDeviceUuid before EMAChatClient.create being called
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(context);
        getChatConfigPrivate().setDeviceUuid(deviceFactory.getDeviceUuid().toString());
        getChatConfigPrivate().setDeviceName(Build.MANUFACTURER + Build.MODEL);
        getChatConfigPrivate().setDid(getDidInfo());
        getChatConfigPrivate().setServiceId(UUID.randomUUID().toString());

        // Init push relevant.
        EMPushConfig pushConfig = options.getPushConfig();
        if (pushConfig == null) {
            pushConfig = new EMPushConfig.Builder(mContext).build();
        }
        // Merge EMOptions to EMPushConfig
        EMPushConfig.Builder builder = new EMPushConfig.Builder(mContext, pushConfig);
        if (TextUtils.isEmpty(pushConfig.getFcmSenderId())) {
            builder.enableFCM(options.getFCMNumber());
        }
        if (TextUtils.isEmpty(pushConfig.getMiAppId()) || TextUtils.isEmpty(pushConfig.getMiAppKey())) {
            EMChatConfigPrivate.EMMipushConfig miPushConfig = options.getMipushConfig();
            if (miPushConfig != null) {
                builder.enableMiPush(miPushConfig.appId, miPushConfig.appKey);
            }
        }
        EMPushHelper.getInstance().init(context, builder.build());
        emaObject = EMAChatClient.create(mChatConfigPrivate.emaObject);
        // sandbox
        // rest 103.241.230.122:31080 im-msync 103.241.230.122:31097
//        mChatConfigPrivate.setChatServer("118.193.28.212");
//        mChatConfigPrivate.setChatPort(31097);
//        mChatConfigPrivate.setRestServer("https://118.193.28.212:31443");
//        mChatConfigPrivate.enableDnsConfig(false);

        connectionListener = new MyConnectionListener();
        emaObject.addConnectionListener(connectionListener);

        multiDeviceListenerImpl = new MyMultiDeviceListener();
        emaObject.addMultiDeviceListener(multiDeviceListenerImpl);

        executor = Executors.newCachedThreadPool();

        // init all the managers
        initManagers();

        mIsLoginWithAgoraToken =EMSessionManager.getInstance().getIsLoginWithAgoraToken();

        final String lastLoginUser = EMSessionManager.getInstance().getLastLoginUser();

        EMLog.e(TAG, "is autoLogin : " + options.getAutoLogin());
        EMLog.e(TAG, "lastLoginUser : " + lastLoginUser);
        EMLog.e(TAG, "hyphenate SDK is initialized with version : " + getChatConfigPrivate().getVersion());

        PowerManager pm = (PowerManager) mContext.getSystemService(Context.POWER_SERVICE);
        wakeLock = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "emclient");
        EMNotificationHelper.getInstance().onInit(context);
        sdkInited = true;

        mContext.registerReceiver(connectivityBroadcastReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
        onNetworkChanged();//主动更新一次，上面注册由于异步原因不会及时更新.数据上报需要

        if (options.getAutoLogin() && isLoggedInBefore()) {
                final String lastLoginToken = EMSessionManager.getInstance().getLastLoginToken();
                final String lastLoginPwd = EMSessionManager.getInstance().getLastLoginPwd();

                //sessionManager.login(lastLoginUser, lastLoginPwd,false, null);
                EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);

                final EMCallBack callback = new EMCallBack() {

                    @Override
                    public void onSuccess() {
                        EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);
                        Log.d(TAG, "hyphenate login onSuccess");
                        tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
                    }

                    @Override
                    public void onError(int code, String error) {
                        Log.d(TAG, "hyphenate login onError");
                        tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
                    }

                    @Override
                    public void onProgress(int progress, String status) {
                    }
                };

                this.execute(new Runnable() {

                    @Override
                    public void run() {
                        // to ensure database has been open before login,
                        // login takes long time before user can see the result
                        getChatConfigPrivate().openDatabase(lastLoginUser);
                        groupManager().loadAllGroups();
                        chatManager().loadAllConversationsFromDB();

                        _login(lastLoginUser,
                                EMSessionManager.getInstance().isLastLoginWithToken() ? lastLoginToken : lastLoginPwd,
                                callback,
                                /*autoLogin*/ true,
                                /*loginWithToken*/ EMSessionManager.getInstance().isLastLoginWithToken());
                    }
                });
        } else {
            tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
        }
    }

    private String getDidInfo() {
        String manufacturer =Build.MANUFACTURER;
        String model = Build.MODEL;
        String hardware = Build.HARDWARE;
        int sdkInt = Build.VERSION.SDK_INT;
        String osVersion = Build.VERSION.RELEASE;

        String did=manufacturer+"/"
                +model+"/"
                +hardware+"/"
                +sdkInt+"/"
                +osVersion;
        return did;
    }

    /**
     * \~chinese
     * 创建账号。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username 用户名，长度不超过 64 个字符。请确保你对该参数设值。支持的字符包括英文字母（a-z），数字（0-9），下划线（_），英文横线（-），英文句号（.）。该参数不区分大小写，大写字母会被自动转为小写字母。如果使用正则表达式设置该参数，则可以将表达式写为：^[a-zA-Z0-9_-]+$。
     * @param password 密码，长度不超过 64 个字符。请确保你对该参数设值。
     *
     * @throws HyphenateException 如果该方法调用失败，会包含调用失败的原因。例如 username（账号），（password）密码为空，或者账号不符合要求。
     *                           
     *
     * \~english
     * Register a new user with your chat network.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The username. The maximum length is 64 characters. Ensure that you set this parameter. Supported characters include the 26 English letters (a-z), the ten numbers (0-9), the underscore (_), the hyphen (-), and the English period (.). This parameter is case insensitive, and upper-case letters are automatically changed to low-case ones. If you want to set this parameter as a regular expression, set it as ^[a-zA-Z0-9_-]+$. 
     * @param password The password. The maximum length is 64 characters. Ensure that you set this parameter.
     *                 
     *
     * @throws HyphenateException A description of the issue that caused this call to fail. For example, the user account or password is null, or the account is illegal.
     *                 
     */
    public void createAccount(String username, String password) throws HyphenateException {
        username = username.toLowerCase();

        Pattern pattern = Pattern.compile("^[a-zA-Z0-9_.-]+$");
        boolean find = pattern.matcher(username).find();
        if (!find) {
            throw new HyphenateException(EMError.USER_ILLEGAL_ARGUMENT, "illegal user name");
        }
        EMAError error = emaObject.createAccount(username, password);
        handleError(error);
    }

    /**
     * \~chinese
     * 使用密码登录服务器。
     * 
     * 参考：
     * 通过 token 登录服务器的方法见 {@link #loginWithToken(String, String, EMCallBack)}。
     *
     * 异步方法。
     *
     * @param id          用户 ID。           不能为空。
     * @param password    用户密码。           不能为空。
     * @param callback    EMCallback 回调函数。   不能为空，登录的结果将通过 callback 返回。
     *
     * @throws IllegalArgumentException 用户 ID，密码，callback 及 App Key 为空会抛出异常。
     *
     * \~english
     * An app user logs in to the chat server with a password.
     * 
     * Reference:
     * An app user logs in to to the chat server by user ID and token, see {@link #loginWithToken(String, String, EMCallBack)}.
     *
     * This is an asynchronous method.
     *
     * @param id 		The unique chat user ID, the same as username.            Make sure to set the param.
     * @param password 	The password.                       Make sure to set the param.
     * @param callback 	The login callback.                 Make sure to set the param. The result is returned via the callback.
     *
     * @throws IllegalArgumentException  A description of the issue that caused this call to fail. For example, the user ID, password, callback or App Key is null.
     */
    public void login(String id, String password, final EMCallBack callback) throws IllegalArgumentException {

        if (callback == null) {
            throw new IllegalArgumentException("callback is null!");
        }

        if (id == null || password == null || id.equals("") || password.equals("")) {
            throw new IllegalArgumentException("username or password is null or empty!");
        }

        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            throw new IllegalArgumentException("please setup your App Key  either in AndroidManifest.xml or through the EMOptions");
        }

        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        id = id.toLowerCase();
        initLoginWithAgoraData(false,"",0);
        _login(id, password, callback, false, false);
    }

    /**
     * \~chinese
     * 通过 token 登录服务器。该方法支持自动登录。
     * 
     * 参考：
     * 通过密码登录服务器，见 {@link #login(String, String, EMCallBack)}。
     *
     * 异步方法。
     *
     * @param username 用户 ID。            不能为空。
     * @param token    验证 token。         不能为空。
     * @param callback EMCallback 回调函数。 不能为空，登录的结果将通过 callback 返回。
     *
     * @throws RuntimeException          App Key 为空回报运行异常。
     * @throws IllegalArgumentException  用户 ID，token 及 callback 为空，会抛出异常。
     *
     * \~english
     * An app user logs in to the chat server by user ID and token.
     * 
     * Reference:
     * There is another method to login to the chat server, which is by user ID and password, see {@link #login(String, String, EMCallBack)}.
     *
     * This is an asynchronous method.
     *
     * @param username The user ID.                       Make sure to set the param.
     * @param token    The password for this user ID.     Make sure to set the param.
     * @param callback The login callback.                Make sure to set the param. The result of login is returned via the callback.
     *
     * @throws RuntimeException          If the app key is null, the SDK throws a RuntimeException.
     * @throws IllegalArgumentException  A description of the issue that caused this call to fail. For example, the ID, token or callback is null.
     */
    public void loginWithToken(String username, String token, final EMCallBack callback) {
        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            throw new RuntimeException("please setup your App Ke y either in AndroidManifest.xml or through the EMOptions");
        }

        if (callback == null) {
            throw new IllegalArgumentException("callback is null!");
        }

        if (username == null || token == null || username.equals("") || token.equals("")) {
            throw new IllegalArgumentException("username or password is null or empty!");
        }

        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        username = username.toLowerCase();
        initLoginWithAgoraData(false,"",0);
        _login(username, token, callback, false, true);
    }

    /**
     * \~chinese
     * 通过用户 ID 和声网 Agora token 登录 chat 服务器。该方法支持自动登录。
     * 
     * 参考：
     * 通过用户 ID 和密码登录 chat 服务器，见{@link #login(String, String, EMCallBack)}。
     *
     * 异步方法。
     *
     * @param username      用户 ID。                不能为空。
     * @param agoraToken    验证 Agora token。        不能为空。
     * @param callback      EMCallback 回调函数。     不能为空。登录的结果将通过 callback 返回。
     *
     * @throws RuntimeException          App Key 为空会报运行异常。
     * @throws IllegalArgumentException  用户 ID，token 及 callback 为空，会抛出异常。
     *
     * \~english
     * An app user logs in to the chat server by user ID and Agora token. This method supports automatic login.
     * 
     * Reference:
     * Another method to login to chat server is to login with user ID and token, see {@link #login(String, String, EMCallBack)}.
     *
     * This an asynchronous method.
     *
     * @param username      The user ID.             Make sure to set the param.
     * @param agoraToken    The Agora token.         Make sure to set the param.
     * @param callback      The callback.            Make sure to set the param. The result of login is returned via the callback.
     *
     * @throws RuntimeException          If the App Key is null, the SDK will throws a RuntimeException.
     * @throws IllegalArgumentException  A description of the exception. For example, the user ID, AgoraToken or callback is null.
     */
    public void loginWithAgoraToken(String username, String agoraToken, final EMCallBack callback){
        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            throw new RuntimeException("please setup your App Key  either in AndroidManifest.xml or through the EMOptions");
        }

        if (callback == null) {
            throw new IllegalArgumentException("callback is null!");
        }

        if (TextUtils.isEmpty(username) ||TextUtils.isEmpty(agoraToken)) {
            throw new IllegalArgumentException("username or agoraToken is null or empty!");
        }

        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "sdk not initialized");
            return;
        }
        //换取环信 Token
        execute(new Runnable() {
            @Override
            public void run() {
                getChatToken(username.toLowerCase(),agoraToken,callback);
            }
        });
    }

    private synchronized void getChatToken(String username, String agoraToken, final EMCallBack callback){
        EMAError error = new EMAError();
        String response = emaObject.getChatTokenbyAgoraToken(agoraToken,error);
        if (error.errCode() == EMError.EM_NO_ERROR) {
            EMLog.d(TAG, "getChatTokenbyAgoraToken success");
            if(response != null && response.length() > 0){
                try {
                    JSONObject object = new JSONObject(response);
                    if(object != null){
                        String chatToken = object.optString("access_token");
                        String expireTimestamp = object.optString("expire_timestamp");
                        long tokenAvailablePeriod=Long.valueOf(expireTimestamp)-System.currentTimeMillis();
                        if (TextUtils.isEmpty(chatToken)) {
                            throw new Exception("chatToken  is null or empty!");
                        }

                        initLoginWithAgoraData(true,expireTimestamp,tokenAvailablePeriod);
                        //登录
                        _login(username.toLowerCase(), chatToken, callback, false, true);
                    }
                }catch (Exception e){
                    EMLog.e(TAG, "getChatTokenbyAgoraToken Exception:"+e.getMessage());
                    callback.onError(EMError.GENERAL_ERROR,"getChatTokenbyAgoraToken Exception:"+e.getMessage());
                }
            }else{
                EMLog.e(TAG, "getChatTokenbyAgoraToken response is null");
              callback.onError(EMError.GENERAL_ERROR,"getChatTokenbyAgoraToken response is null or empty!");
            }
        } else {
            callback.onError(error.errCode(), error.errMsg());
            EMLog.e(TAG, "getChatTokenbyAgoraToken failed error:" + error.errCode() + "  errorMessage:" + error.errMsg());
        }
    }

    private void initLoginWithAgoraData(boolean isLoginWithAgoraToken, String expireTimestamp, long tokenAvailablePeriod) {
        mIsLoginWithAgoraToken =isLoginWithAgoraToken;
        if(isLoginWithAgoraToken) {
            //存储
            EMSessionManager.getInstance().setLoginWithAgoraData(isLoginWithAgoraToken,expireTimestamp,tokenAvailablePeriod);
        }else{
            EMSessionManager.getInstance().clearLoginWithAgoraTokenData();
        }
    }

    /**
     * \~chinese
     *
     * 通知 token 过期，会通过 EMClient 回调给 connectionListener 过期事件。
     *
     * @param response 请求返回的结果。
     *
     * \~english
     *
     * The notification that the token is expired, will be called back to the connectionListener expiration event through EMClient.
     *
     * @param response The request result, which includes the description of the issue that cause the method fails.
     */
    public void notifyTokenExpired(String response){
        //只在声网 Token 方式登录下才通知。
        if(connectionListener!=null&&mIsLoginWithAgoraToken) {
            //解析
            try {
                JSONObject object = new JSONObject(response);
                if(object!=null) {
                    String errorDescription = object.optString("error_description");
                    EMLog.e(TAG,"notifyTokenExpired--errorDescription:"+errorDescription);
                    if(errorDescription.contains("milliseconds ago")||errorDescription.contains("has expired")||
                            errorDescription.contains("Unable to authenticate due to expired access Token")) {
                        connectionListener.onTokenNotification(401);
                        EMLog.e(TAG,"notifyTokenExpired--onTokenNotification(401) ");
                    }
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }
    /**
     * \~chinese
     *
     * 当用户在声网 token 登录状态时，且在 {@link EMConnectionListener} 实现类中收到 token 即将过期事件的回调通知可以调用这个 API 来更新 token，避免因 token 失效产生的未知问题。
     *
     * @param newAgoraToken 声网 token。
     *
     * \~english
     *
     * When a user is in the Agora token login state and receives a callback notification of the token is to be expired in the {@link EMConnectionListener} implementation class,
     * this API can be called to update the token to avoid unknown problems caused by the token invalidation.
     *
     * @param newAgoraToken The new token.
     */
    public void renewToken(String newAgoraToken){
        if(mIsLoginWithAgoraToken&& emaObject.isLoggedIn()) {//只有处于声网登录方式且处于登录状态时才更新Token
            execute(new Runnable() {
                @Override
                public void run() {
                    getChatToken(getCurrentUser(), newAgoraToken, new EMCallBack() {
                        @Override
                        public void onSuccess() {
                        }

                        @Override
                        public void onError(int code, String error) {
                        }

                        @Override
                        public void onProgress(int progress, String status) {
                        }
                    });
                }
            });
        }else{
           EMLog.e(TAG,"the method  excepted to be called when login by agoraToken and login state is loggeIn");
        }

    }

    /**
     * \~chinese
     * 退出登录并返回退出结果。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param unbindToken 是否解绑 token。 
     *                    - `true` 表示需要退出时解绑设备 `token`。 
     *                    - `false` 表示退出时不解绑设备 `token`。
     * @return - 返回值为 0，即 {@link EMError#EM_NO_ERROR}，表示退出登录成功；
     *         - 其他值为退出失败，详见 {@link EMError}。
     *
     * \~english
     * An app user logs out and returns the result.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param unbindToken Whether to unbind the token. 
     *                    - `true` means to unbind the device token when logout.
     *                    - `false` means to not unbind the device token when logout.
     * @return - Returns {@link EMError#EM_NO_ERROR} if the user successfully logs out. 
     *         - Or returns the description of the cause to the failure, see {@link EMError}.
     */
    public int logout(boolean unbindToken) {
        if (!emaObject.isLogout()) {
            String pushToken = EMPreferenceUtils.getInstance().getPushToken();
            String pushNotifierName = EMPreferenceUtils.getInstance().getPushNotifierName();
            if(!TextUtils.isEmpty(pushToken) && !TextUtils.isEmpty(pushNotifierName)) {
                try {
                    pushManager.unBindDeviceToken();
                } catch (HyphenateException e) {
                    return EMError.USER_UNBIND_DEVICETOKEN_FAILED;
                }
            }else {
                boolean result = EMPushHelper.getInstance().unregister(unbindToken);
                if (!result) {
                    return EMError.USER_UNBIND_DEVICETOKEN_FAILED;
                }
            }
        } else {
            EMPushHelper.getInstance().unregister(false);
            EMLog.e(TAG, "already logout, skip unbind token");
        }

        logout();

        return EMError.EM_NO_ERROR;
    }

    /**
     * \~chinese
     * 登出聊天服务器。
     * 
     * 同步方法，会阻塞当前线程。
     * 
     * 如果需要异步请参考 {@link EMClient#logout(EMCallBack callback)}。
     *
     * \~english
     * An app user logs out from the chat server.
     * 
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference:
     * To use the asynchronous method, call {@link EMClient#logout(EMCallBack callback)}.
     */
    void logout() {
        EMLog.d(TAG, " SDK Logout");

        try {
            if (connectivityBroadcastReceiver != null) {
                mContext.unregisterReceiver(connectivityBroadcastReceiver);
            }
        } catch (Exception e) {
        }

        EMSessionManager.getInstance().clearLastLoginUser();
        EMSessionManager.getInstance().clearLastLoginToken();
        EMSessionManager.getInstance().clearLoginWithAgoraTokenData();

        // ============ gaode code start ===============
        // EMGDLocation.getInstance().onDestroy();
        // ============ gaode code end ===============

        if (smartHeartbeat != null) {
            smartHeartbeat.stop();
        }

        if (wakeLock.isHeld()) {
            wakeLock.release();
        }

        // make sure we clear the username and pwd in a sync way
        if (emaObject != null) {
            emaObject.logout();
        }
        if (chatManager != null) {
            chatManager.onLogout();
        }
        if (groupManager != null) {
            groupManager.onLogout();
        }
        if (contactManager != null) {
            contactManager.onLogout();
        }
        if (chatroomManager != null) {
            chatroomManager.onLogout();
        }

        // clear all memory cache
        try {
            EMAdvanceDebugManager.getInstance().onDestroy();
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (EMChatConfigPrivate.isDebugTrafficMode()) {
            EMNetworkMonitor.stopTrafficStat();
        }
    }

    /**
     * \~chinese
     * 登出聊天服务器。
     *
     * 异步方法。
     *
     * @param unbindToken 是否解绑 token。   
     *                    - `true` 表示需要退出时解绑设备 `token`；
     *                    - `false` 表示退出时不解绑设备 `token`。
     * @param callback    方法完成的回调，包含调用失败的原因。   
     *
     * \~english
     * An app user logs out the chat server.
     *
     * This is an asynchronous method.
     *
     * @param unbindToken  Whether to unbind token. 
     *                     - `true` means to unbind the device token when logout.
     *                     - `false` means to not unbind the device token when logout.
     * @param callback     The completion callback, which contains the error message if the method fails.
     */
    public void logout(final boolean unbindToken, final EMCallBack callback) {
        new Thread() {
            @Override
            public void run() {
                int error = logout(unbindToken);

                if (error != EMError.EM_NO_ERROR) {
                    if (callback != null) {
                        callback.onError(error, "faild to unbind device token");
                    }
                } else {
                    if (callback != null) {
                        callback.onSuccess();
                    }
                }
            }
        }.start();
    }

    /**
     * \~chinese
     * 登出聊天服务器。
     * 
     * 异步方法。
     *
     * @param callback    方法完成的回调，包含调用失败的原因。 
     *
     * \~english
     * An user logs out the chat server.
     * 
     * This is an asynchronous method.
     *
     * @param callback     The completion callback, which contains the error message if the method fails.
     */
    void logout(final EMCallBack callback) {
        Thread logoutThread = new Thread() {
            @Override
            public void run() {
                if (callback != null) {
                    callback.onProgress(0, null);
                }

                logout();

                if (callback != null) {
                    callback.onSuccess();
                }
            }
        };
        logoutThread.setPriority(Thread.MAX_PRIORITY - 1);
        logoutThread.start();
    }

    /**
     * \~chinese
     * 修改 App Key，注意只有在未登录状态才能修改 App Key。修改 App Key 是为了方便你切换其他 App Key，切换后可以使用切换后的 App Key 测试，除了登出外，没有其他的限制。
     * 
     * 也可以通过 {@link EMOptions#setAppKey(String)} 设置 App Key，同样只有在未登录状态才能修改。
     *
     * @param appkey  App Key，应用的唯一标识，不可为空。
     *
     * \~english
     * Update the App Key, which is the unique identifier used to access Agora Chat.
     *  
     * You retrieve the new App Key from Agora Console. 
     * 
     * As this key controls all access to Agora Chat for your app, you can only update the key when the current user is logged out.
     * 
     * Also, you can set App Key by the following method when logged out {@link EMOptions#setAppKey(String)}.
     *
     * @param appkey  The App Key, make sure to set the param.
     */
    public void changeAppkey(String appkey) throws HyphenateException {
        EMAError error = emaObject.changeAppkey(appkey);
        if (error.errCode() == EMAError.EM_NO_ERROR) {
            this.getOptions().updatePath(appkey);
        }
        handleError(error);
    }

    /**
     * \~chinese
     * 设置 Chat 服务器连接监听。
     *
     * @param listener Chat 服务器连接监听。 
     *                 - {@link EMConnectionListener#onConnected()} 表示与 chat 服务器连接成功；
     *                 - {@link EMConnectionListener#onDisconnected(int)} 表示与 chat 服务器断开连接。参数表示错误码，详见 {@link EMError}。
     *
     * \~english
     * Adds the connection listener of chat server.
     *
     * @param listener The chat server connection listener. {@link EMConnectionListener#onConnected()} indicates a successful connection to chat server, {@link EMConnectionListener#onDisconnected(int)} indicates a failure connection to chat server, it contains the error code, see {@link EMError}.
     */
    public void addConnectionListener(final EMConnectionListener listener) {
        if (listener == null) {
            return;
        }

        synchronized (connectionListeners) {
            if (!connectionListeners.contains(listener)) {
                connectionListeners.add(listener);
            }
        }

        execute(new Runnable() {

            @Override
            public void run() {
                if (isConnected()) {
                    listener.onConnected();
                } else {
                    listener.onDisconnected(EMError.NETWORK_ERROR);
                }
            }
        });
    }

    /**
     * \~chinese
     * 移除 Chat 服务器连接监听。
     *
     * @param listener  Chat 服务器连接监听。
     *
     * \~english
     * Removes the chat server connection listener.
     *
     * @param listener  The chat server connection listener.
     */
    public void removeConnectionListener(final EMConnectionListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (connectionListeners) {
            connectionListeners.remove(listener);
        }
    }

    /**
     * \~chinese
     * 获取群组管理类，需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 群组管理类。
     *
     * \~english
     * Gets the `GroupManager` class. Be sure to call it after the EMClient has been initialized, see {@link #init(Context, EMOptions)}.
     *
     * @return The `GroupManager` class.
     */
    public EMGroupManager groupManager() {
        if (groupManager == null) {
            synchronized (EMClient.class) {
                if(groupManager == null) {
                    groupManager = new EMGroupManager(this, emaObject.getGroupManager());
                }
            }
        }
        return groupManager;
    }

    /**
     * \~chinese
     * 获取推送管理类，需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 推送管理类。
     *
     * \~english
     * Gets the `PushManager` class. Be sure to call it after the EMClient has been initialized, see {@link #init(Context, EMOptions)}.
     *
     * @return The `PushManager` class.
     */
    public EMPushManager pushManager() {
        if (pushManager == null) {
            synchronized (EMClient.class) {
                if(pushManager == null) {
                    pushManager = new EMPushManager(this, emaObject.getPushMnager());
                }
            }
        }
        return pushManager;
    }

    /**
     * \~chinese
     * 获取聊天室管理类，该方法需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 聊天室管理类。
     *
     * \~english
     * Gets the `ChatRoomManager` class. Make sure to call it after the EMClient has been initialized, see {@link #init(Context, EMOptions)}.
     *
     * @return The `ChatRoomManager` class.
     */
    public EMChatRoomManager chatroomManager() {
        if (chatroomManager == null) {
            synchronized (EMClient.class) {
                if(chatroomManager == null) {
                    chatroomManager = new EMChatRoomManager(this, emaObject.getChatRoomManager());
                }
            }
        }
        return chatroomManager;
    }

    /**
     * \~chinese
     * 获取聊天管理类，需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 聊天管理类。
     *
     * \~english
     * Gets the `ChatManager` class. Make sure to call it after EMClient has been initialized, see {@link #init(Context, EMOptions)}
     *
     * @return The `ChatManager` class.
     */
    public EMChatManager chatManager() {
        if (chatManager == null) {
            synchronized (EMClient.class) {
                if(chatManager == null) {
                    chatManager = new EMChatManager(this, emaObject.getChatManager(), emaObject.getReactionManager());
                }
            }
        }
        return chatManager;
    }


    /**
     * \~chinese
     * 获取用户信息管理类，需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 用户信息管理类。
     *
     * \~english
     * Gets the `EMUserInfoManager` class. Make sure to call it after EMClient has been initialized, see {@link #init(Context, EMOptions)}.
     *
     * @return The `EMUserInfoManager` class.
     */
    public EMUserInfoManager userInfoManager() {
        if (userInfoManager == null) {
            synchronized (EMClient.class) {
                if(userInfoManager == null) {
                    userInfoManager = new EMUserInfoManager(emaObject.getUserInfoManager());
                }
            }
        }
        return userInfoManager;
    }



    /**
     * \~chinese
     * 获取联系人管理类，需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return 联系人管理类。
     *
     * \~english
     * Gets the `ContactManager` class. Make sure to call it after the EMClient has been initialized, see {@link #init(Context, EMOptions)}.
     *
     * @return The `ContactManager` class.
     */
    public EMContactManager contactManager() {
        if (contactManager == null) {
            synchronized (EMClient.class) {
                if(contactManager == null) {
                    contactManager = new EMContactManager(this, emaObject.getContactManager());
                }
            }
        }
        return contactManager;
    }

    public EMTranslationManager translationManager(){
        if(translationManager == null){
            synchronized (EMClient.class) {
                if(translationManager == null){
                    translationManager = new EMTranslationManager(emaObject.getTranslateManager());
                }
            }
        }
        return translationManager;
    }

    public EMPresenceManager presenceManager(){
        if(presenceManager == null){
            synchronized (EMClient.class) {
                if(presenceManager == null){
                    presenceManager = new EMPresenceManager(emaObject.getPresenceManager());
                }
            }
        }
        return presenceManager;
    }

    /**
     * \~chinese
     * 获取子区管理类，需要在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @return Thread 管理类。
     *
     * \~english
     * Gets the `EMChatThreadManager` class. Make sure to call it after EMClient has been initialized, see {@link #init(Context, EMOptions)}.
     *
     * @return The `EMChatThreadManager` class.
     */
    public EMChatThreadManager chatThreadManager(){
        if(threadManager == null){
            synchronized (EMClient.class) {
                if(threadManager == null){
                    threadManager = new EMChatThreadManager(this, emaObject.getThreadManager());
                }
            }
        }
        return threadManager;
    }

    public Context getContext() {
        return mContext;
    }

    /**
     * \~chinese
     * 获取当前登录用户的用户名。
     *
     * @return 当前登录的用户。
     *
     * \~english
     * Gets the current login user ID.
     *
     * @return The current login user ID.
     */
    public synchronized String getCurrentUser() {
        if (EMSessionManager.getInstance().currentUser == null ||
                EMSessionManager.getInstance().currentUser.username == null ||
                EMSessionManager.getInstance().currentUser.username.equals("")) {
            return EMSessionManager.getInstance().getLastLoginUser();
        }
        return EMSessionManager.getInstance().currentUser.username;
    }

    /**
     * \~chinese
     * 根据用户名和密码获取 token。
     *
     * @param username 已经注册过的用户名。
     * @param password 密码。
     * @param callBack 结果回调。   
     *                 - {@link EMValueCallBack#onSuccess(Object)} 表示请求成功的回调，onSuccess 中的参数即为返回的 token；
     *                 - {@link EMValueCallBack#onError(int, String)} 表示请求失败的回调，第一个参数是错误码，第二个参数是错误描述。
     *
     * \~english
     * Fetches token by username and password.
     *
     * @param username The registered username.
     * @param password The password.
     * 
     * @param callBack The result callback：  
     *                 - {@link EMValueCallBack#onSuccess(Object)} indicates a successful callback，the parameter in onSuccess is the token;
     *                 - {@link EMValueCallBack#onError(int, String)} indicates a failed callback, in which the first parameter is the error code, and the second parameter is the error description.
     */
    public void getUserTokenFromServer(final String username, final String password, final EMValueCallBack<String> callBack) {
        execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                final String token = emaObject.getUserTokenFromServer(username, password, error);

                if (callBack == null) {
                    return;
                }

                if (error.errCode() == EMError.EM_NO_ERROR) {
                    callBack.onSuccess(token);
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~chinese
     * 返回是否登录过。
     * 
     * 如果登录成功后未调用 `logout` 方法，这个方法的返回值一直是 `true`。
     * 
     * 参考：
     * 如果需要判断当前是否连接到服务器，请使用 {@link #isConnected()} 方法。
     *
     * ```java
     * if(EMClient.getInstance().isLoggedInBefore()){
     *     // enter main activity
     * }else{
     *     // enter login activity
     * }
     * ```
     *
     * @return 返回是否登录过的结果。   
     *         - `true` 表示之前登录过。
     *         - `false` 表示之前未登录过或者已经调用过 {@link #logout()} 方法。
     *
     * \~english
     * Checks whether the user has logged in before and did not log out.
     * 
     * Reference:
     * If you need to check whether the SDK is connected to the server, please use {@link #isConnected()}.
     *
     * ```java
     * if(EMClient.getInstance().isLoggedInBefore()){
     *     // Enter the main activity.
     * }else{
     *     // Enter the login activity.
     * }
     * ```
     *
     * @return The result of whether the user has logged in before. 
     *         - `true` means that the user has logged in before,
     *         - `false` means that the user has not login before or has called {@link #logout()} method.
     */
    public boolean isLoggedInBefore() {
        EMSessionManager sessionMgr = EMSessionManager.getInstance();
        String user = sessionMgr.getLastLoginUser();
        // no need decrypt password or token when user empty
        if (TextUtils.isEmpty(user)) return false;

        String pwd = sessionMgr.getLastLoginPwd();
        String token = sessionMgr.getLastLoginToken();

        if (user != null  && !user.isEmpty() &&
                ((pwd != null   && !pwd.isEmpty()) ||
                 (token != null && !token.isEmpty())) ) {
            return true;
        }

        return false;
    }


    /**
     * \~chinese
     * 检查 SDK 是否连接到 Chat 服务器。
     *
     * @return 返回是否连接到 Chat 服务器的结果。
     *         - `true` 表示已经连接到 Chat 服务器。
     *         - `false` 表示没有连接到 Chat 服务器。
     *
     * \~english
     * Checks whether the SDK is connected to the chat server.
     *
     * @return Returns the result whether the SDK is connected to the chat server.
     *         - `true` means that the SDK is connected to the chat server.
     *         - `false` means not.
     */
    public boolean isConnected() {
        return emaObject.isConnected();
    }

    /**
     *\~chinese
     * 检查用户是否登录。
     *
     * @return 返回是否登录的结果。
     *         - `true` 表示已经登录。
     *         - `false` 表示没有登录。
     * \~english
     * Checks whether the user is logged in.
     *
     * @return Returns the result of whether the user has logged in.
     *         - `true` indicates that the user has logged in;
     *         - `false` indicates not.
     */
    public boolean isLoggedIn() {
        return emaObject.isLoggedIn();
    }

    /**
     * \~chinese
     * 设置是否输出调试信息，在 EMClient 初始化完成后调用，详见 {@link #init(Context, EMOptions)}。
     *
     * @param debugMode - `true`：SDK 会在 log 里输出调试信息；
     *                  -  `false`：不会输出调试信息。
     *
     * \~english
     * Sets whether to output the debug info. Make sure to call the method after the EMClient has been initialized, see {@link #init(Context, EMOptions)}.
     * 
     * The SDK will out put debug info if you set the debugMode as `true`.
     *
     * @param debugMode  - `true`: The SDK will output the debug info;
     *                   - `false`: Do not output the debug info.
     */
    public void setDebugMode(boolean debugMode) {
        if (sdkInited) {
            //if you have set debugmode, use it.
            String mode = EMAdvanceDebugManager.getInstance().getDebugMode();
            if (mode != null)
                debugMode = Boolean.parseBoolean(mode);
        }
        EMLog.debugMode = debugMode;
        getChatConfigPrivate().setDebugMode(debugMode);
    }

    /**
     * \~chinese
     * 更新当前用户的推送昵称，用于推送时在通知栏展示。
     * 
     * @deprecated 从 V3.3.3 废弃，使用 {@link EMPushManager#updatePushNickname(String)} 代替。
     * 
     * 需要注意用户属性中的昵称跟这里推送昵称是独立的。
     * 
     * 建议从用户服务器获取到用户信息后，将用户属性中的昵称更新到 Chat 服务器；
     * 当个人信息中昵称发生变化的时候，请同步更新到 Chat 服务器，防止出现显示差异。
     *
     * @param nickname 推送昵称。不能为空，长度不超过 100 个字符。
     * 
     *
     * \~english
     * Updates the current user's push notification display nickname, which is used to display in the notification bar when pushing.
     * 
     * @deprecated From V3.3.3, use {@link EMPushManager#updatePushNickname(String)} instead.
     * 
     * Notice that the nickname in user profiles is different from this PushNickname.
     * 
     * We recommend you to update the push notification display nickname to the chat server after getting the user information from the user server;
     * When the nickname of user information change, make sure to update to the chat server to prevent display discrepancies.
     *
     * @param nickname Make sure to set the param, the data length is not more than 100 characters.
     * 
     */
    @Deprecated
    public boolean updateCurrentUserNick(String nickname) throws IllegalArgumentException, HyphenateException {
        return pushManager().updatePushNickname(nickname);
    }

    /**
     * \~chinese
     * 上传本地的日志。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param callback 预留参数。
     *
     * \~english
     * Uploads local log file. 
     *
     * The information in the debug log is used by our engineers to fix errors and improve system performance.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param callback Reserved parameter.
     */
    public void uploadLog(EMCallBack callback) {
        chatManager().emaObject.uploadLog();
    }


    /**
     * \~chinese
     * 获取 EMOptions。
     *
     * \~english
     * Gets EMOptions.
     */
    public EMOptions getOptions() {
        return mChatConfigPrivate.getOptions();
    }


    /**
     * \~chinese
     * 压缩 log 文件，并返回压缩后的文件路径。强烈建议方法完成之后删除该压缩文件。
     *
     * @return 压缩后的 log 文件路径。
     * @throws HyphenateException 压缩文件失败报的错。
     *
     * \~english
     * Compresses the debug log into a gzip archive.
     * Best practice is to delete this debug archive as soon as it is no longer used.

     * @return The path of the compressed gz file.
     * @throws HyphenateException  A description of the cause of the exception if the method fails.
     */
    public String compressLogs() throws HyphenateException {
        EMAError error = new EMAError();
        String path = emaObject.compressLogs(error);
        handleError(error);
        return path;
    }

    /**
     * \~chinese
     * 添加多设备监听的接口。
     *
     * @param listener 见 {@link EMMultiDeviceListener}, {@link EMMultiDeviceListener#onContactEvent(int, String, String)} 联系人事件回调，
     *                 {@link EMMultiDeviceListener#onGroupEvent(int, String, List)} 群组事件回调。
     *
     * \~english
     * Adds the multi-device listener.
     * 
     * @param listener See {@link EMMultiDeviceListener}, {@link EMMultiDeviceListener#onContactEvent(int, String, String)} is the contact event callback,
     *                 {@link EMMultiDeviceListener#onGroupEvent(int, String, List)} is the group event callback.
     */
    public void addMultiDeviceListener(EMMultiDeviceListener listener) {
        multiDeviceListeners.add(listener);
    }

    /**
     * \~chinese
     * 删除多设备监听的接口。
     * 
     * @param listener 见 {@link EMMultiDeviceListener}。
     *
     * \~english
     * Removes the multi-device listener.
     * @param listener See {@link EMMultiDeviceListener}.
     */
    public void removeMultiDeviceListener(EMMultiDeviceListener listener) {
        multiDeviceListeners.remove(listener);
    }

    /**
     * \~chinese
     * 获取指定账号下登录的在线设备列表。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username  用户名。
     * @param password  密码。
     * @return 在线设备列表。
     * @throws HyphenateException 具体错误详见 {@link EMError}。
     *
     * \~english
     * Gets all the information about the logged in devices under the specified account.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID you want to get the device information.
     * @param password The password.
     * @return The list of the online devices.
     * @throws HyphenateException A description of the exception, see {@link EMError}.
     */
    public List<EMDeviceInfo> getLoggedInDevicesFromServer(String username, String password) throws HyphenateException {
        EMAError error = new EMAError();
        List<EMADeviceInfo> devices = emaObject.getLoggedInDevicesFromServer(username, password, error);
        handleError(error);
        List<EMDeviceInfo> result = new ArrayList<>();
        for (EMADeviceInfo info : devices) {
            result.add(new EMDeviceInfo(info));
        }
        return result;
    }

    /**
     * \~chinese
     * 在指定账号下，根据设备 ID，将指定设备下线, 设备 ID：{@link EMDeviceInfo#getResource()}
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username  账户名称。
     * @param password  该账户密码。
     * @param resource  设备 ID, 见 {@link EMDeviceInfo#getResource()}。
     * @throws HyphenateException 如果有异常会在此抛出，包括错误码和错误信息，详见 {@link EMError}。
     *
     * \~english
     * Force the specified account to logout from the specified device, to fetch the device ID: {@link EMDeviceInfo#getResource()}.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The account you want to force logout.
     * @param password The account's password.
     * @param resource The device ID, see {@link EMDeviceInfo#getResource()}.
     * @throws HyphenateException A description of the exception if the method fails, see {@link EMError}.
     */
    public void kickDevice(String username, String password, String resource) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickDevice(username, password, resource, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 将指定账号下的所有设备都踢下线。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param username 用户名。
     * @param password 密码。
     * @throws HyphenateException 如果有异常会抛出，包括错误码和错误描述，错误码详见 {@link EMError}。
     *
     * \~english
     * Kicks out all the devices logged in under the specified account.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The account you want to log out from all the devices.
     * @param password The account's password.
     * @throws HyphenateException A description of the exception, see {@link EMError}.
     */
    public void kickAllDevices(String username, String password) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickAllDevices(username, password, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 上传 FCM token 至服务器。
     * 
     * token 可以被上传的前提条件有：
     * 1.被上传的 token 不为空；
     * 2.当前已有用户登录；
     * 3.当前设备支持 Google 推送；
     * 4.Google 推送类型为 FCM {@link com.hyphenate.push.EMPushType#FCM}。
     * 
     * 设置推送类型为 FCM 的接口为 {@link EMOptions#setFCMNumber(String)}，即设置了 FCM number 则推送类型为 FCM。
     *
     * @param fcmToken 要上传的 token。
     *
     * \~english
     * Uploads the FCM token to the chat server.
     * 
     * The token can be uploaded when all the following conditions are met:
     * 1.The token is not empty;
     * 2.The user have logged in;
     * 3.The device support Google PUSH service;
     * 4.The push type is FCM {@link com.hyphenate.push.EMPushType#FCM}.
     * 
     * You can set with {@link EMOptions#setFCMNumber(String)}.
     * 
     * @param fcmToken The token you want to upload.
     */
    public void sendFCMTokenToServer(String fcmToken) {
        EMLog.d(TAG, "sendFCMTokenToServer: " + fcmToken);
        if (TextUtils.isEmpty(fcmToken)) {
            return;
        }

        // If no user logged in, stop upload the fcm Token.
        String userName = getCurrentUser();
        if (TextUtils.isEmpty(userName)) {
            EMLog.i(TAG, "No user login currently, stop upload the token.");
            return;
        }
        // let's check if the push service available
        EMPushType pushType = EMPushHelper.getInstance().getPushType();
        EMLog.i(TAG, "pushType: " + pushType);
        if (pushType == EMPushType.FCM) {
            EMPushHelper.getInstance().onReceiveToken(pushType, fcmToken);
        }
    }

    /**
     * \~chinese
     * 发送华为推送 token 到服务器。
     * @param appId    华为 AppId。
     * @param token    华为推送 token。
     * @deprecated     请使用 {@link #sendHMSPushTokenToServer(String)} 代替。
     *
     * \~english
     * Sends the Huawei devices token to the server.
     * @param appId   The Huawei appId.
     * @param token   The Huawei device token.
     * @Deprecated    Please use {@link #sendHMSPushTokenToServer(String)} instead.
     */
    @Deprecated
    public void sendHMSPushTokenToServer(String appId, String token){
        sendHMSPushTokenToServer(token);
    }

    /**
     * \~chinese
     * 发送华为推送 token 到服务器。
     * @param token    华为推送 token。
     *
     * \~english
     * Sends the Huawei devices token to the server.
     * @param token    The Huawei device token.
     */
    public void sendHMSPushTokenToServer(String token){
        if (EMPushHelper.getInstance().getPushType() == EMPushType.HMSPUSH) {
            EMPushHelper.getInstance().onReceiveToken(EMPushType.HMSPUSH, token);
        }
    }

	//---------------------------private methods ---------------------------------

	private void initManagers(){
		// invoke constructor before login, to listener PB event
		EMHttpClient.getInstance().onInit(mChatConfigPrivate);
		chatManager();
		contactManager();
		groupManager();
		chatroomManager();
        presenceManager();
        chatThreadManager();

		setNatvieNetworkCallback();
		EMSessionManager.getInstance().init(this, emaObject.getSessionManager());
	}

	void _login(final String username, final String code, final EMCallBack callback, final boolean autoLogin, final boolean loginWithToken) {
        if (getChatConfigPrivate() == null || sdkInited == false) {
            callback.onError(EMError.GENERAL_ERROR, "");
            return;
        }

        EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid());

        // convert to lowercase
        execute(new Runnable() {

            @Override
            public void run() {
                EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid() + " threadName:" + Thread.currentThread().getName() + " ID:" + Thread.currentThread().getId());

                if (username == null) {
                    callback.onError(EMError.INVALID_USER_NAME, "Invalid user name");
                    return;
                }

                EMAError error = new EMAError();
                emaObject.login(username, code, autoLogin, loginWithToken, error);

                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    EMSessionManager.getInstance().setLastLoginUser(username);
                    EMAError emaError = new EMAError();
                    if (emaError.errCode() == EMAError.EM_NO_ERROR) {
                        if (/*mChatConfigPrivate.getUsingSQLCipher() || */loginWithToken) {
                            String token = emaObject.getUserToken(false, emaError);
                            EMSessionManager.getInstance().setLastLoginToken(token);
                            EMSessionManager.getInstance().setLastLoginWithToken(true);
                            EMSessionManager.getInstance().clearLastLoginPwd();
                            if(mIsLoginWithAgoraToken) {
                                EMSessionManager.getInstance().startCountDownTokenAvailableTime(connectionListener);
                            }
                        } else {
                            EMSessionManager.getInstance().setLastLoginPwd(code);
                            EMSessionManager.getInstance().setLastLoginWithToken(false);
                            EMSessionManager.getInstance().clearLastLoginToken();
                        }
                    }

                    onNewLogin();
                    EMPushHelper.getInstance().register();
                    // set EMConferenceManager parameters, lite jar build need to remove this line
//                    conferenceManager().set(getAccessToken(), getOptions().getAppKey(), getCurrentUser());
                    callback.onSuccess();
                } else {
                    callback.onError(error.errCode(), error.errMsg());
                    if(error.errCode()==EMAError.USER_ALREADY_LOGIN&& mIsLoginWithAgoraToken) {
                        //进程杀死又立马开启，会走到这里，renewToken 也会走这里。
                        //更新 Token
                        emaObject.renewToken(code);
                        EMSessionManager.getInstance().setLastLoginToken(code);
                        //此时计时器可能失效需要重新开启。
                        EMSessionManager.getInstance().startCountDownTokenAvailableTime(connectionListener);
                    }
                }

                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    if (getOptions().isEnableStatistics()) {
                        setPresence(getLocationString(autoLogin));
                    } else {
                        EMLog.d(TAG, "statistics is not enabled");
                    }
                }

                if (error.errCode() == EMAError.USER_AUTHENTICATION_FAILED) {
                    EMSessionManager.getInstance().clearLastLoginPwd();
                    EMSessionManager.getInstance().clearLastLoginToken();
                }
            }
        });
    }

    /**
     * \~chinese
     * 检查 FCM 推送是否可用。
     *
     * @return  - `true` 表示 FCM 推送可用；
     *          - `false` 表示 FCM 推送不可用。
     *
     * \~english
     * Checks whether the FCM push is available.
     *
     * @return  - `true`: the FCM push is available; 
     *          - `false`: not available.
     */
    public boolean isFCMAvailable() {
        return EMPushHelper.getInstance().getPushType() == EMPushType.FCM;
    }

    void onNewLogin() {
        EMLog.d(TAG, "on new login created");

        String username = EMSessionManager.getInstance().getLastLoginUser();

        PathUtil.getInstance().initDirs(getChatConfigPrivate().getAppKey(), username, mContext);

        EMAdvanceDebugManager.getInstance().onInit(mChatConfigPrivate);

        if (smartHeartbeat == null) {
            smartHeartbeat = EMSmartHeartBeat.create(mContext);
        }

        if (getChatConfigPrivate().emaObject.hasHeartBeatCustomizedParams()) {
            smartHeartbeat.setCustomizedParams(getChatConfigPrivate().emaObject.getWifiHeartBeatCustomizedParams(),
                    getChatConfigPrivate().emaObject.getMobileHeartBeatCustomizedParams());
        }
        smartHeartbeat.onInit();
        if (getOptions().getFixedInterval() != -1) {
            smartHeartbeat.setFixedInterval(getOptions().getFixedInterval());
        }
    }

    /**
     * \~chinese
     * 从内存中获取身份认证 token。
     * 
     * 在上传下载附件（语音，图片，文件等）时必须添加到请求 header 中，当出现任何异常时将返回 null，
     * 可通过判断是否为 null 来检测是否有问题；如果为 null，在打开 EMLog 日志时，可以看到异常原因。
     *
     * 也可以通过 {@link EMOptions#getAccessToken()} 获取，该方法需要从服务端获取 token，可以调用 {@link EMOptions#getAccessToken(boolean)}，
     * 并传入 `true`。
     *
     * @return 身份认证 token。
     *
     * \~english
     * Gets the access token from the memory.
     * When uploading or downloading attachment (voice, image, file and so on), the token should be added to the request header;
     * 
     * The SDK returns null when any exceptions occurs.
     * 
     * You can detect whether there is a exception by check the token. If the token is null, you can check the EMLog file
     * to find the exception.
     *
     * You can also get token by calling {@link EMOptions#getAccessToken(boolean)} and enter true.
     *
     * @return  The access token.
     */
    public String getAccessToken() {
        return getChatConfigPrivate().getAccessToken();
    }

    /**
     * \~chinese
     * 判断 SDK 是否已经初始化完毕。
     * @return  - `true`：SDK 已经初始化完毕。
     *          - `false`：SDK 尚未初始化完毕。
     *
     * \~english
     * Checks whether the SDK has been initialized.
     * @return   - `true`: The SDK has been initialized.
     *           - `false`: The SDK has not been initialized.
     */
    public boolean isSdkInited() {
        return sdkInited;
    }

    private boolean _loadLibrary(final String library, boolean trace) {
        try {
            ReLinker.loadLibrary(mContext, library);
            return true;
        } catch (Throwable e) {
            if (trace) {
                e.printStackTrace();
            }
        }
        return false;
    }

    private boolean _loadLibrary(final String library) {
        return _loadLibrary(library, true);
    }

    private void loadLibrary() {
        if (libraryLoaded == false) {
            //if (_loadLibrary("sqlcipher", false) || _loadLibrary("sqlite")) {}
            //_loadLibrary("sqlite");
            _loadLibrary("cipherdb", false);
//            _loadLibrary("hyphenate_av");
//            _loadLibrary("hyphenate_av_recorder");
//            ReLinker.loadLibrary(mContext, "crypto");
//            ReLinker.loadLibrary(mContext, "ssl");
            ReLinker.loadLibrary(mContext, "hyphenate");
            libraryLoaded = true;
        }
    }

    class MyConnectionListener extends EMAConnectionListener {

        @Override
        public void onConnected() {
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        try {
                            for (EMConnectionListener listener : connectionListeners) {
                                listener.onConnected();
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onDisconnected(final int errCode) {
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        switch (errCode) {
                            case EMError.USER_REMOVED:
                                EMSessionManager.getInstance().clearLastLoginUser();
                                EMSessionManager.getInstance().clearLastLoginToken();
                                EMSessionManager.getInstance().clearLastLoginPwd();
                                break;
                            case EMError.USER_LOGIN_ANOTHER_DEVICE:
                            case EMError.USER_BIND_ANOTHER_DEVICE:
                            case EMError.USER_DEVICE_CHANGED:
                            case EMError.SERVER_SERVICE_RESTRICTED:
                            case EMError.USER_LOGIN_TOO_MANY_DEVICES:
                            case EMError.USER_KICKED_BY_CHANGE_PASSWORD:
                            case EMError.USER_KICKED_BY_OTHER_DEVICE:
                                EMSessionManager.getInstance().clearLastLoginToken();
                                EMSessionManager.getInstance().clearLastLoginPwd();
                                if(isSdkInited()) {
                                    // clear push token info
                                    EMPreferenceUtils.getInstance().setPushNotifierName("");
                                    EMPreferenceUtils.getInstance().setPushToken("");
                                }
                                break;
                        }

                        try {
                            for (EMConnectionListener listener : connectionListeners) {
                                listener.onDisconnected(errCode);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }

                        if ((errCode == EMError.USER_LOGIN_ANOTHER_DEVICE) ||
                                (errCode == EMError.USER_REMOVED) ||
                                (errCode == EMError.USER_BIND_ANOTHER_DEVICE) ||
                                (errCode == EMError.USER_DEVICE_CHANGED) ||
                                (errCode == EMError.SERVER_SERVICE_RESTRICTED) ||
                                (errCode == EMError.USER_LOGIN_TOO_MANY_DEVICES) ||
                                (errCode == EMError.USER_KICKED_BY_CHANGE_PASSWORD) ||
                                (errCode == EMError.USER_KICKED_BY_OTHER_DEVICE)){
                            try {
                                for (EMConnectionListener listener : connectionListeners) {
                                    listener.onLogout(errCode);
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    }
				}
			});
		}

        @Override
        public boolean verifyServerCert(List<String> certschain,String domain) {
            if (certschain == null) {
                EMLog.d(TAG, "List<String> certschain : null ");
                return false;
            }
            if(TextUtils.isEmpty(domain)) {
                EMLog.d(TAG, "domain is empty or null ");
                return false;
            }
            EMLog.d(TAG, "domain = " + domain);
            X509Certificate[] certsArray = convertToCerts(certschain);
            try {
                X509TrustManager x509TrustManager = getSystemDefaultTrustManager();
                X509TrustManagerExtensions managerExtensions = new X509TrustManagerExtensions(x509TrustManager);
                managerExtensions.checkServerTrusted(certsArray, certsArray[0].getType(),domain);
            } catch (Exception e) {
                e.printStackTrace();
                EMLog.e(TAG, e.getMessage());
                EMLog.d(TAG, "List<String> certschain :" + certschain.toString());

                return false;
            }
            return true;
        }
        public void onTokenNotification(int code) {
            //从EMSessionManager::onDisconnect、com.hyphenate.chat.EMSessionManager.checkTokenAvailability、EMClient#notifyTokenExpired中通知过来
            execute(new Runnable() {
                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        try {

                            if(code==EMError.TOKEN_EXPIRED||code==401) {
                                logout();
                                for (EMConnectionListener listener : connectionListeners) {
                                    EMLog.d(TAG,"MyConnectionListener onTokenExpired code: "+code);
                                    listener.onTokenExpired();
                                }
                            }else{
                                for (EMConnectionListener listener : connectionListeners) {
                                    EMLog.d(TAG,"MyConnectionListener onTokenWillExpire code: "+code);
                                    listener.onTokenWillExpire();
                                }
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                            EMLog.e(TAG,"MyConnectionListener onTokenNotification Exception: "+e.getMessage());
                        }
                    }
                }
            });
        }
    }

    class MyMultiDeviceListener extends EMAMultiDeviceListener {
        @Override
        public void onContactEvent(final int event, final String target, final String ext) {
            EMLog.d(TAG, "onContactEvent:" + event + " target:" + target + " ext:" + ext);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (EMMultiDeviceListener listener : multiDeviceListeners) {
                                listener.onContactEvent(event, target, ext);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onGroupEvent(final int event, final String target, final List<String> usernames) {
            EMLog.d(TAG, "onGroupEvent:" + event + " target:" + target + " usernames:" + usernames);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (EMMultiDeviceListener listener : multiDeviceListeners) {
                                listener.onGroupEvent(event, target, usernames);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }
    }

    void execute(Runnable runnable) {
        executor.execute(runnable);
    }

    void executeOnMainQueue(Runnable runnable) {
        mainQueue.submit(runnable);
    }

    void executeOnSendQueue(Runnable runnable) {
        sendQueue.submit(runnable);
    }

    // TODO: temp
    public EMChatConfigPrivate getChatConfigPrivate() {
        return mChatConfigPrivate;
    }

    void setNatvieNetworkCallback() {
        EMANetCallback callback = new EMANetCallback() {

            @Override
            public int getNetState() {
                if (!NetUtils.hasDataConnection(mContext)) {
                    return EMANetwork.NETWORK_NONE.ordinal();
                } else {
                    if (NetUtils.isWifiConnected(mContext)
                            || NetUtils.isOthersConnected(mContext)) {
                        return EMANetwork.NETWORK_WIFI.ordinal();
                    } else if (NetUtils.isMobileConnected(mContext)) {
                        return EMANetwork.NETWORK_MOBILE.ordinal();
                    } else if (NetUtils.isEthernetConnected(mContext)) {
                        return EMANetwork.NETWORK_CABLE.ordinal();
                    } else {
                        return EMANetwork.NETWORK_NONE.ordinal();
                    }
                }
            }
        };

        mChatConfigPrivate.emaObject.setNetCallback(callback);
    }

    /**
     * \~chinese
     * 设置应用的加密实现。
     * 如果未设置，SDK 会使用内置的加密算法。
     * 
     * @param provider 加密实现。
     * 
     * \~english
     * Sets the encrypt provider.
     * If you don't set it, the SDK will use the default encrypt provider.
     *
     * @param provider The encrypt provider.
     */
    void setEncryptProvider(EMEncryptProvider provider) {
        this.encryptProvider = provider;
    }

    /**
     * \~chinese
     * 获取 EncryptProvider。如果你未设置，将返回 SDK 内置的 encrypt provider。
     * 
     *  \~english
     * Gets the encrypt provider.
     * 
     * @return The encrypt provider.
     */
    EMEncryptProvider getEncryptProvider() {
        if (encryptProvider == null) {
            EMLog.d(TAG, "encrypt provider is not set, create default");
            encryptProvider = new EMEncryptProvider() {

                public byte[] encrypt(byte[] input, String username) {
                    try {
                        String in = new String(input);
                        return emaObject.getSessionManager().encrypt(in).getBytes();
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

                public byte[] decrypt(byte[] input, String username) {
                    try {
                        String in = new String(input);
                        return emaObject.getSessionManager().decrypt(in).getBytes();
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

            };
        }
        return encryptProvider;
    }

    boolean sendPing(boolean waitPong, long timeout) {
        return emaObject.sendPing(waitPong, timeout);
    }

    void checkTokenAvailability(){
        if(mIsLoginWithAgoraToken) {
            EMSessionManager.getInstance().checkTokenAvailability(connectionListener);
        }
    }

    void forceReconnect() {
        EMLog.d(TAG, "forceReconnect");
        disconnect();
        reconnect();
    }

    void reconnect() {
        EMLog.d(TAG, "reconnect");
        wakeLock.acquire();
        emaObject.reconnect();
        if (wakeLock.isHeld()) {
            wakeLock.release();
        }
    }

    void disconnect() {
        EMLog.d(TAG, "disconnect");
        emaObject.disconnect();
    }

    private void handleError(EMAError error) throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

    
    /**
     *\~chinese
     * 连接变化监听。
     * 
     * 该方法只在重连时运行，会更新重连申请次数。
     * 
     *  \~english 
     * The connectivity change listener.
     * 
     * Only occurs in the reconnection thread and resets the attempts times.
     */
    private BroadcastReceiver connectivityBroadcastReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(final Context context, Intent intent) {
            String action = intent.getAction();
            if (!action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
                EMLog.d(TAG, "skip no connectivity action");
                return;
            }

            EMLog.d(TAG, "connectivity receiver onReceiver");

            EMANetwork networkType;
            NetUtils.Types types = NetUtils.getNetworkTypes(getContext());
            switch (types) {
                case WIFI:
                case OTHERS:
                    networkType = EMANetwork.NETWORK_WIFI;
                    break;
                case MOBILE:
                    networkType = EMANetwork.NETWORK_MOBILE;
                    break;
                case ETHERNET:
                    networkType = EMANetwork.NETWORK_CABLE;
                    break;
                case NONE:
                default:
                    networkType = EMANetwork.NETWORK_NONE;
                    break;
            }

            boolean prevNetworkAvailable = currentNetworkType != EMANetwork.NETWORK_NONE;
            boolean currentNetworkAvailable = networkType != EMANetwork.NETWORK_NONE;
            currentNetworkType = networkType;
            if (prevNetworkAvailable == currentNetworkAvailable) {
               execute(new Runnable() {
                    @Override
                    public void run() {
                        if (smartHeartbeat != null) {
                            // Network availability no change, return.
                            EMLog.i(TAG, "Network availability no change, just return. " + currentNetworkType + ", but check ping");
                            smartHeartbeat.sendPingCheckConnection();
                        }
                    }
                });
                return;
            }

            EMLog.i(TAG, "Network availability changed, notify... " + currentNetworkType);

            execute(new Runnable() {

                @Override
                public void run() {
                    emaObject.onNetworkChanged(currentNetworkType);
                }
            });
        }
    };

    void onNetworkChanged() {
        try {
            if (NetUtils.isWifiConnected(mContext)
                    || NetUtils.isOthersConnected(mContext)) {
                EMLog.d(TAG, "has wifi connection");
                currentNetworkType = EMANetwork.NETWORK_WIFI;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_WIFI);
                return;
            }

            if (NetUtils.isMobileConnected(mContext)) {
                EMLog.d(TAG, "has mobile connection");
                currentNetworkType = EMANetwork.NETWORK_MOBILE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_MOBILE);
                return;
            }

            if (NetUtils.isEthernetConnected(mContext)) {
                EMLog.d(TAG, "has ethernet connection");
                currentNetworkType = EMANetwork.NETWORK_CABLE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_CABLE);
                return;
            }
            currentNetworkType = EMANetwork.NETWORK_NONE;
            EMLog.d(TAG, "no data connection");
            emaObject.onNetworkChanged(EMANetwork.NETWORK_NONE);
            return;
        } catch (Exception e) {
            e.printStackTrace();
            return;
        }
    }

    void onNetworkChanged(EMANetwork network) {
        emaObject.onNetworkChanged(network);
    }

    private AppStateListener appStateListener;
    private List<Activity> resumeActivityList = new ArrayList<>();

    void setAppStateListener(AppStateListener appStateListener) {
        this.appStateListener = appStateListener;
    }

    @TargetApi(14)
    private void registerActivityLifecycleCallbacks() {
        if (Utils.isSdk14()) {
            Object lifecycleCallbacks = new ActivityLifecycleCallbacks() {

                @Override
                public void onActivityStopped(Activity activity) {
                    resumeActivityList.remove(activity);
                    if (resumeActivityList.isEmpty()) {
                        if (appStateListener != null) appStateListener.onBackground();
                    }
                }

                @Override
                public void onActivityResumed(Activity activity) {
                }

                @Override
                public void onActivityCreated(Activity activity, Bundle savedInstanceState) {

                }

                @Override
                public void onActivityStarted(Activity activity) {
                    if(!resumeActivityList.contains(activity)) {
                        resumeActivityList.add(activity);
                        if(resumeActivityList.size() == 1) {
                            if (appStateListener != null) appStateListener.onForeground();
                        }
                    }
                }

                @Override
                public void onActivitySaveInstanceState(Activity activity, Bundle outState) {

                }

                @Override
                public void onActivityPaused(Activity activity) {

                }

                @Override
                public void onActivityDestroyed(Activity activity) {

                }

            };
            ((Application) mContext).registerActivityLifecycleCallbacks((ActivityLifecycleCallbacks) lifecycleCallbacks);
        }
    }

    interface AppStateListener {
        void onForeground();

        void onBackground();
    }

    void setPresence(final String location) {
        if(TextUtils.isEmpty(location)) {
            return;
        }
        execute(new Runnable() {
            @Override
            public void run() {
                emaObject.setPresence(location);
            }
        });
    }

    private String getLocationString(boolean autoLogin) {
        JSONObject jObj = null;
        if (!autoLogin) {
            jObj = getDeviceInfo();
        }
        return jObj == null ? "": jObj.toString();
    }

    public JSONObject getDeviceInfo() {
        JSONObject jobj = new JSONObject();
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(mContext);
        String deviceId = deviceFactory.getDeviceUuid().toString();
        try {
            jobj.put("deviceid", deviceId);
            jobj.put("app-id", mContext.getPackageName());
            jobj.put("hid", EMClient.getInstance().getCurrentUser());
            jobj.put("os", "android");
            jobj.put("os-version", Build.VERSION.RELEASE);
            jobj.put("manufacturer", Build.MANUFACTURER);
            jobj.put("model", Build.MODEL);
        } catch (JSONException e) {
            EMLog.d(TAG, e.getMessage());
        }

        return jobj;
    }

    // For service check.
    private boolean duringChecking = false;

    /**
     * \~chinese
     * 服务诊断接口，流程如下:
     * 
     * 1.校验用户名和密码；
     * 2.从服务端获取 DNS 列表；
     * 3.从服务端获取 token；
     * 4.连接 chat 服务器；
     * 5.断开连接(如果检查前已经有账户登录，则不执行该步骤)。
     * 
     * 如果在诊断过程中产生了错误，该流程将会被打断。
     *
     * @param username 用于服务诊断的用户名，即 user ID，如果已有账户登录，该用户名会被替换为已登录账户的用户名，以防止更改当前已登录账户
     *                 的信息，比如 token 等。
     * @param password 密码，如果已有账户登录，该密码会被替换为已登录账户的密码。
     * @param listener 诊断结果的回调。
     *
     * \~english
     * The service check interface. The schedule is as follows:
     * 
     * 1.Validates the username and password user input.
     * 2.Gets the dns list from the server.
     * 3.Gets the token from the server.
     * 4.Connects to the im server.
     * 5.Logout.(If you call this method after the user logged in, this step will not be in the schedule.)
     * 
     * If there is a error occurs during checking, this schedule will be interrupted.
     *
     * @param username The user inputted for service check. If account has logged in before, this username
     *                 would be changed to the logged in username to avoid update the current logged in
     *                 account data.
     * @param password The password for username. if this is a logged in service check, the password would
     *                 changed to the logged in password.
     * @param listener A listener for service check results callback.
     */
    public void check(String username, String password, final CheckResultListener listener) {
        // Already during checking, just return.
        if (duringChecking) {
            EMLog.i("EMServiceChecker", "During service checking, please hold on...");
            return;
        }

        duringChecking = true;

        // If is logged in before, the username must be current user.
        if (isLoggedInBefore()) {
            username = getCurrentUser();
            EMSessionManager sessionMgr = EMSessionManager.getInstance();
            password = sessionMgr.getLastLoginPwd();
        }

        final String finalUser = username;
        final String finalPwd = password;

        new Thread(new Runnable() {
            @Override
            public void run() {
                /**
                 * Contains account-validation check, get-dns check, get-Token check, login check.
                 * So the {@link EMAChatClient.CheckResultListener#onResult(int, int, String)} callback
                 * will be called four times.
                 */
                emaObject.check(finalUser, finalPwd, new EMAChatClient.CheckResultListener() {
                    @Override
                    public void onResult(final int type, final int result, final String desc) {
                        EMLog.i("EMServiceChecker", "type: " + type + ", result: " + result + ", desc: " + desc);
                        // Notify user once per callback.
                        notifyCheckResult(listener, type, result, desc);

                        // If occur a error during four checks, return.
                        if (result != EMError.EM_NO_ERROR) {
                            duringChecking = false;
                            return;
                        }

                        // If login successfully, logout
                        if (type == EMCheckType.DO_LOGIN) {
                            checkLogout(listener);
                        }
                    }
                });
            }
        }).start();
    }

    private void checkLogout(CheckResultListener listener) {
        // If is not a logged in service check, logout and notify the user.
        if (!isLoggedInBefore()) {
            logout();
            notifyCheckResult(listener, EMCheckType.DO_LOGOUT, EMError.EM_NO_ERROR, "");
        }

        duringChecking = false;
    }

    private void notifyCheckResult(CheckResultListener listener, @EMCheckType.CheckType int type, int result, String desc) {
        if (listener == null) return;
        listener.onResult(type, result, desc);
    }

    public interface CheckResultListener {
        void onResult(@EMCheckType.CheckType int type, int result, String desc);
    }

}
