/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMChatThreadChangeListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAThreadInfo;
import com.hyphenate.chat.adapter.EMAThreadManager;
import com.hyphenate.chat.adapter.EMAThreadManagerListener;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;


/**
 * \~chinese
 * 子区管理类，用于管理子区，包含子区创建和解散以及成员管理等操作。
 *
 * \~english
 * The message thread manager that defines how to manage message threads, including message thread creation, destruction, and member management.
 */
public class EMChatThreadManager {
	private static String TAG = EMChatThreadManager.class.getSimpleName();

	EMAThreadManager emaObject;
	EMClient mClient;

	List<EMChatThreadChangeListener> threadChangeListeners;

	EMChatThreadManager(EMClient client, EMAThreadManager threadManager) {
		emaObject = threadManager;
		mClient = client;
		threadChangeListeners = Collections.synchronizedList(new ArrayList<EMChatThreadChangeListener>());
		emaObject.addListener(listenerImpl);
		EMClient.getInstance().chatManager();
	}

	/**
	 * \~chinese
	 * 注册子区事件监听器，用于监听子区变化，如子区的创建和解散等。
	 *
	 * 你可以调用 {@link #removeChatThreadChangeListener(EMChatThreadChangeListener)} 移除不需要的监听器。
	 *
	 * @param listener	要注册的子区事件监听器。
	 *
	 * \~english
	 * Adds the message thread event listener, which listens for message thread changes, such as the message thread creation and destruction.
	 *
	 * You can call {@link #removeChatThreadChangeListener(EMChatThreadChangeListener)} to remove an unnecessary message thread event listener.
	 *
	 * @param listener The message thread event listener to add.
	 */
	public void addChatThreadChangeListener(EMChatThreadChangeListener listener) {
		if (listener != null && !threadChangeListeners.contains(listener)) {
			threadChangeListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除子区事件监听器。
	 *
	 * 在利用 {@link #addChatThreadChangeListener(EMChatThreadChangeListener)} 注册子区事件监听器后调用此方法。
	 *
	 * @param listener	要移除的子区事件监听器。
	 *
	 * \~english
	 * Removes the message thread event listener.
	 *
	 * After a message thread event listener is added with {@link #addChatThreadChangeListener(EMChatThreadChangeListener)}, you can call this method to remove it when it is not required.
	 *
	 * @param listener  The message thread event listener to remove.
	 */
	public void removeChatThreadChangeListener(EMChatThreadChangeListener listener) {
		if (listener != null && threadChangeListeners.contains(listener)) {
			threadChangeListeners.remove(listener);
		}
	}

	/**
	 * \~chinese
	 * 创建子区。
	 *
	 * 子区所属群组的所有成员均可调用该方法。
	 *
	 * 子区创建成功后，会出现如下情况：
	 *
	 * - 单设备登录时，子区所属群组的所有成员均会收到回调 {@link EMChatThreadChangeListener#onChatThreadCreated(EMChatThreadEvent)}。
	 *   你可通过设置 {@link EMChatThreadChangeListener} 监听相关事件。
	 *
	 * - 多端多设备登录时，各设备会收到事件回调 {@link com.hyphenate.EMMultiDeviceListener#onChatThreadEvent(int, String, List)}。
	 *   该回调方法中第一个参数表示子区事件，例如，子区创建事件为 {@link com.hyphenate.EMMultiDeviceListener#THREAD_CREATE}。
	 *   你可通过设置 {@link com.hyphenate.EMMultiDeviceListener} 监听相关事件。
	 *
	 * @param parentId 父 ID，即群组 ID。
	 * @param messageId 父消息 ID。
	 * @param chatThreadName 要创建的子区的名称。长度不超过 64 个字符。
	 * @param callBack	结果回调：
	 *                  - 成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回创建成功的子区对象；
	 *                  - 失败时回调 {@link EMValueCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~english
	 * Creates a message thread.
	 *
	 * Each member of the chat group where the message thread belongs can call this method.
	 *
	 * Upon the creation of a message thread, the following will occur:
	 *
	 * - In a single-device login scenario, each member of the group to which the message thread belongs will receive the {@link EMChatThreadChangeListener#onChatThreadCreated(EMChatThreadEvent)} callback.
	 *   You can listen for message thread events by setting {@link EMChatThreadChangeListener}.
	 *
	 * - In a multi-device login scenario, the devices will receive the {@link com.hyphenate.EMMultiDeviceListener#onChatThreadEvent(int, String, List)} callback.
	 *   In this callback method, the first parameter indicates a message thread event, for example, {@link com.hyphenate.EMMultiDeviceListener#THREAD_CREATE} for a message thread creation event.
	 *   You can listen for message thread events by setting {@link com.hyphenate.EMMultiDeviceListener}.
	 *
	 * @param parentId 		The parent ID, which is the group ID.
	 * @param messageId 	The ID of the parent message.
	 * @param chatThreadName The name of the new message thread. It can contain a maximum of 64 characters.
	 * @param callBack   	The result callback:
	 *                      - If success, {@link EMValueCallBack#onSuccess(Object)} is triggered to return the new message thread object;
	 *                      - If a failure occurs, {@link EMValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void createChatThread(String parentId, String messageId, String chatThreadName, EMValueCallBack<EMChatThread> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMChatThread thread = createChatThread(parentId, messageId, chatThreadName);
					if(callBack != null) {
					    callBack.onSuccess(thread);
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
					    callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"createChatThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private EMChatThread createChatThread(String parentId, String messageId, String chatThreadName) throws HyphenateException{
		EMAError error = new EMAError();
		EMAThreadInfo thread = emaObject.createThread(parentId, messageId, chatThreadName, error);
		handleError(error);
		return new EMChatThread(thread);
	}

	/**
	 * \~chinese
	 * 从服务器获取子区详情。
	 *
	 * @param chatThreadId 子区 ID。
	 * @param callBack     结果回调：
	 *                     - 成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回子区详情；
	 *                     - 失败时回调 {@link EMValueCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~english
	 * Gets the details of the message thread from the server.
	 *
	 * @param chatThreadId   The message thread ID.
	 * @param callBack       The result callback:
	 *                       - If success, {@link EMValueCallBack#onSuccess(Object)} is triggered to return the message thread details;
	 *                       - If a failure occurs, {@link EMValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getChatThreadFromServer(String chatThreadId, EMValueCallBack<EMChatThread> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMChatThread thread = getChatThreadFromServer(chatThreadId);
					if(callBack != null) {
						callBack.onSuccess(thread);
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getChatThreadFromServer error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private EMChatThread getChatThreadFromServer(String chatThreadId) throws HyphenateException {
		EMAError error = new EMAError();
		EMAThreadInfo thread = emaObject.getThreadFromServer(chatThreadId, error);
		handleError(error);
		return new EMChatThread(thread);
	}

	/**
	 * \~chinese
	 * 加入子区。
	 *
	 * 子区所属群组的所有成员均可调用该方法。
	 *
	 * 多端多设备登录时，注意以下几点：
	 *
	 * - 设备会收到 {@link com.hyphenate.EMMultiDeviceListener#onChatThreadEvent(int, String, List)} 回调。
	 *
	 * - 该回调方法中，第一个参数表示子区事件，例如，加入子区事件为 {@link com.hyphenate.EMMultiDeviceListener#THREAD_JOIN}。
	 *
	 * - 可通过设置 {@link com.hyphenate.EMMultiDeviceListener} 监听相关事件。
	 *
	 * @param chatThreadId 子区 ID。
	 * @param callBack     结果回调：
	 *                     - 成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回子区详情 {@link EMChatThread}，详情中不含成员数量；
	 *                     - 失败时回调 {@link EMValueCallBack#onError(int, String)}，返回错误信息。
	 *                       重复加入会报错，错误码为 {@link com.hyphenate.EMError#USER_ALREADY_EXIST}。
	 *
	 * \~english
	 * Joins a message thread.
	 *
	 * Each member of the group where the message thread belongs can call this method.
	 *
	 * In a multi-device login scenario, note the following:
	 *
	 * - The devices will receive the {@link com.hyphenate.EMMultiDeviceListener#onChatThreadEvent(int, String, List)} callback.
	 *
	 * - In this callback method, the first parameter indicates a message thread event, for example, {@link com.hyphenate.EMMultiDeviceListener#THREAD_JOIN} for a message thread join event.
	 *
	 * - You can listen for message thread events by setting {@link com.hyphenate.EMMultiDeviceListener}.
	 *
	 * @param chatThreadId The message thread ID.
	 * @param callBack     The result callback:
	 *                     - If success, {@link EMValueCallBack#onSuccess(Object)} is triggered to return the message thread details {@link EMChatThread} which do not include the member count.
	 *                     - If a failure occurs, {@link EMValueCallBack#onError(int, String)} is triggered to return an error.
	 *                       If you join a message thread repeatedly, the SDK will return an error with the error code {@link com.hyphenate.EMError#USER_ALREADY_EXIST}.
	 */
	public void joinChatThread(String chatThreadId, EMValueCallBack<EMChatThread> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMChatThread thread = joinChatThread(chatThreadId);
					if(callBack != null) {
					    callBack.onSuccess(thread);
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"joinChatThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private EMChatThread joinChatThread(String chatThreadId) throws HyphenateException {
		EMAError error = new EMAError();
		EMAThreadInfo thread = emaObject.joinThread(chatThreadId, error);
		handleError(error);
		return new EMChatThread(thread);
	}

	/**
	 * \~chinese
	 * 解散子区。
	 *
	 * 只有子区所属群组的群主及管理员可调用该方法。
	 *
	 * **注意**
	 *
	 * - 单设备登录时，子区所属群组的所有成员均会收到 {@link EMChatThreadChangeListener#onChatThreadDestroyed(EMChatThreadEvent)} 回调。
	 *   你可通过设置 {@link EMChatThreadChangeListener} 监听子区事件。
	 *
	 * - 多端多设备登录时，设备会收到 {@link com.hyphenate.EMMultiDeviceListener#onChatThreadEvent(int, String, List)} 回调。
	 *   该回调方法中，第一个参数为子区事件，例如，子区解散事件为 {@link com.hyphenate.EMMultiDeviceListener#THREAD_DESTROY}。
	 *   你可通过设置 {@link com.hyphenate.EMMultiDeviceListener} 监听子区事件。
	 *
	 *
	 * @param chatThreadId 子区 ID。
	 * @param callBack     结果回调：
	 *                     - 成功时回调 {@link EMCallBack#onSuccess()}；
	 * 				       - 失败时回调 {@link EMCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~english
	 * Destroys the message thread.
	 *
	 * Only the owner or admins of the group where the message thread belongs can call this method.
	 *
	 * **Note**
	 *
	 * - In a single-device login scenario, each member of the group to which the message thread belongs will receive the {@link EMChatThreadChangeListener#onChatThreadDestroyed(EMChatThreadEvent)} callback.
	 *   You can listen for message thread events by setting {@link EMChatThreadChangeListener}.
	 *
	 * - In a multi-device login scenario, The devices will receive the {@link com.hyphenate.EMMultiDeviceListener#onChatThreadEvent(int, String, List)} callback.
	 *   In this callback method, the first parameter indicates a message thread event, for example, {@link com.hyphenate.EMMultiDeviceListener#THREAD_DESTROY} for a message thread destruction event.
	 *   You can listen for message thread events by setting {@link com.hyphenate.EMMultiDeviceListener}.
	 *
	 * @param chatThreadId  The message thread ID.
	 * @param callBack      The result callback:
	 *                      - If success, {@link EMCallBack#onSuccess()} is triggered.
	 * 				        - If a failure occurs, {@link EMCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void destroyChatThread(String chatThreadId, EMCallBack callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					destroyChatThread(chatThreadId);
					if(callBack != null) {
					    callBack.onSuccess();
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"destroyChatThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private void destroyChatThread(String chatThreadId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.destroyThread(chatThreadId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 退出子区。
	 *
	 * 子区中的所有成员均可调用该方法。
	 *
	 * 多设备登录情况下，注意以下几点：
	 *
	 * - 各设备会收到 {@link com.hyphenate.EMMultiDeviceListener#onChatThreadEvent(int, String, List)} 回调。
	 *
	 * - 该回调方法中第一个参数表示事件，退出子区事件为 {@link com.hyphenate.EMMultiDeviceListener#THREAD_LEAVE}。
	 *
	 * - 你可通过设置 {@link com.hyphenate.EMMultiDeviceListener} 监听相关事件；
	 *
	 * @param chatThreadId 要退出的子区的 ID。
	 * @param callBack     结果回调：
	 *                     - 成功时回调 {@link EMCallBack#onSuccess()}；
	 *                     - 失败时回调 {@link EMCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~english
	 * Leaves a message thread.
	 *
	 * Each member in the message thread can call this method.
	 *
	 * In a multi-device login scenario, note the following:
	 *
	 * - The devices will receive the {@link com.hyphenate.EMMultiDeviceListener#onChatThreadEvent(int, String, List)} callback.
	 *
	 * - In this callback method, the first parameter indicates a message thread event, for example, {@link com.hyphenate.EMMultiDeviceListener#THREAD_LEAVE} for a message thread leaving event.
	 *
	 * - You can listen for message thread events by setting {@link com.hyphenate.EMMultiDeviceListener}.
	 *
	 * @param chatThreadId The ID of the message thread that the current user wants to leave.
	 * @param callBack     The result callback:
	 *                     - If success, {@link EMCallBack#onSuccess()} is triggered;
	 * 				       - If a failure occurs, {@link EMCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void leaveChatThread(String chatThreadId, EMCallBack callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					leaveChatThread(chatThreadId);
					if(callBack != null) {
					    callBack.onSuccess();
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"leaveChatThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private void leaveChatThread(String chatThreadId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.leaveThread(chatThreadId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 修改子区名称。
	 *
	 * 只有子区所属群主、群管理员及子区创建者可调用该方法。
	 *
	 * 子区所属群组的成员会收到 {@link EMChatThreadChangeListener#onChatThreadUpdated(EMChatThreadEvent)} 回调。
	 *
	 * 你可通过设置 {@link EMChatThreadChangeListener} 监听子区事件。
	 *
	 * @param chatThreadId		子区 ID。
	 * @param chatThreadName	子区的新名称。长度不超过 64 个字符。
	 * @param callBack		    结果回调：
	 *                          - 成功时回调 {@link EMCallBack#onSuccess()}；
	 * 							- 失败时回调 {@link EMCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~english
	 * Changes the name of the message thread.
	 *
	 * Only the owner or admins of the group where the message thread belongs and the message thread creator can call this method.
	 *
	 * Each member of the group to which the message thread belongs will receive the {@link EMChatThreadChangeListener#onChatThreadUpdated(EMChatThreadEvent)} callback.
	 *
	 * You can listen for message thread events by setting {@link EMChatThreadChangeListener}.
	 *
	 * @param chatThreadId      The message thread ID.
	 * @param chatThreadName    The new message thread name. It can contain a maximum of 64 characters.
	 * @param callBack          The result callback:
	 *                          - If success, {@link EMCallBack#onSuccess()} is triggered.
	 * 				            - If a failure occurs, {@link EMCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void updateChatThreadName(String chatThreadId, String chatThreadName, EMCallBack callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateChatThreadName(chatThreadId, chatThreadName);
					if(callBack != null) {
						callBack.onSuccess();
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"changeChatThreadName error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private void updateChatThreadName(String chatThreadId, String chatThreadName) throws HyphenateException{
		EMAError error = new EMAError();
		emaObject.updateChatThreadName(chatThreadId, chatThreadName, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 移除子区成员。
	 *
	 * 只有子区所属群主、群管理员及子区创建者可调用该方法。
	 *
	 * 被移出的成员会收到 {@link EMChatThreadChangeListener#onChatThreadUserRemoved(EMChatThreadEvent)} 回调。
	 *
	 * 你可通过设置 {@link EMChatThreadChangeListener} 监听子区事件。
	 *
	 *
	 * @param chatThreadId	子区 ID。
	 * @param member	    被移出子区的成员的用户 ID。
	 * @param callBack	    结果回调。
	 *                      - 成功时回调 {@link EMCallBack#onSuccess()}；
	 * 					    - 失败时回调 {@link EMCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~english
	 * Removes a member from the message thread.
	 *
	 * Only the owner or admins of the group where the message thread belongs and the message thread creator can call this method.
	 *
	 * The removed member will receive the {@link EMChatThreadChangeListener#onChatThreadUserRemoved(EMChatThreadEvent)} callback.
	 *
	 * You can listen for message thread events by setting {@link EMChatThreadChangeListener}.
	 *
	 * @param chatThreadId  The message thread ID.
	 * @param member        The user ID of the member to be removed from the message thread.
	 * @param callBack      The result callback.
	 *                      - If success, {@link EMCallBack#onSuccess()} is triggered.
	 * 				        - If a failure occurs, {@link EMCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void removeMemberFromChatThread(String chatThreadId, String member, EMCallBack callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					removeMemberFromChatThread(chatThreadId, member);
					if(callBack != null) {
						callBack.onSuccess();
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"removeMemberFromThread error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private void removeMemberFromChatThread(String chatThreadId, String member) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.removeMemberFromThread(chatThreadId, member, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 分页获取子区成员。
	 *
	 * 子区所属群组的所有成员均可调用该方法。
	 *
	 * @param chatThreadId	子区 ID。
	 * @param limit		    每页期望返回的成员数。取值范围为 [1,50]。
	 * @param cursor	    开始获取数据的游标位置，首次调用方法时传 `null` 或空字符串，按成员加入子区时间的正序获取数据。
	 * @param callBack	    结果回调：
	 *                      - 成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回获取结果 {@link EMCursorResult}，包含成员列表以及用于下次获取数据的游标；
	 *                      - 失败时回调 {@link EMValueCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~chinese
	 * Gets a list of members in the message thread with pagination.
	 *
	 * Each member of the group to which the message thread belongs can call this method.
	 *
	 * @param chatThreadId The message thread ID.
	 * @param limit        The number of members that you expect to get on each page. The value range is [1,50].
	 * @param cursor       The position from which to start getting data. At the first method call, if you set `cursor` to `null` or an empty string, the SDK will get data in the chronological order of when members join the message thread.
	 * @param callBack     The result callback:
	 *                     - If success, {@link EMValueCallBack#onSuccess(Object)} is triggered to return the result {@link EMCursorResult}, including the message thread member list and the cursor for the next query.
	 *                     - If a failure occurs, {@link EMValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getChatThreadMembers(String chatThreadId, int limit, String cursor, EMValueCallBack<EMCursorResult<String>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMCursorResult<String> cursorResult = getChatThreadMembers(chatThreadId, limit, cursor);
					if(callBack != null) {
					    callBack.onSuccess(cursorResult);
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getThreadMembers error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private EMCursorResult<String> getChatThreadMembers(String chatThreadId, int limit, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<String> cursorResult = emaObject.fetchThreadMembers(chatThreadId, limit, cursor, error);
		handleError(error);

		return cursorResult;
	}

	/**
	 * \~chinese
	 * 分页从服务器获取当前用户加入的子区列表。
	 *
	 * @param limit		每页期望返回的子区数。取值范围为 [1,50]。
	 * @param cursor	开始获取数据的游标位置。首次调用方法时传 `null` 或空字符串，按用户加入子区时间的倒序获取数据。
	 * @param callBack	结果回调：
	 *                  - 成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回分页获取结果 {@link EMCursorResult}，包含子区列表以及用于下次获取数据的游标；
	 * 		            - 失败时回调 {@link EMValueCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~english
	 * Uses the pagination to get the list of message threads that the current user has joined.
	 *
	 * This method gets data from the server.
	 *
	 * @param limit     The number of message threads that you expect to get on each page. The value range is [1,50].
	 * @param cursor    The position from which to start getting data. At the first method call, if you set `cursor` to `null` or an empty string, the SDK will get data in the reverse chronological order of when the user joins the message threads.
	 * @param callBack  The result callback:
	 *                  - If success, {@link EMValueCallBack#onSuccess(Object)} is triggered to return the result {@link EMCursorResult}, including the message thread list and the cursor for the next query.
	 *                  - If a failure occurs, {@link EMValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getJoinedChatThreadsFromServer(int limit, String cursor, EMValueCallBack<EMCursorResult<EMChatThread>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMCursorResult<EMChatThread> result = getJoinedChatThreadsFromServer(limit, cursor);
					if(callBack != null) {
						callBack.onSuccess(result);
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getJoinedThreadsFromServer error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private EMCursorResult<EMChatThread> getJoinedChatThreadsFromServer(int limit, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAThreadInfo> cursorResult = emaObject.getJoinedThreadsFromServer(limit, cursor, error);
		handleError(error);
		EMCursorResult<EMChatThread> result = new EMCursorResult<>();
		List<EMChatThread> threadList = new ArrayList<>();
		List<EMAThreadInfo> data = cursorResult.getData();
		for(int i = 0; i < data.size(); i++) {
		  	threadList.add(new EMChatThread(data.get(i)));
		}
		result.setData(threadList);
		result.setCursor(cursorResult.getCursor());
		return result;
	}

	/**
	 * \~chinese
	 * 分页从服务器获取当前用户加入指定群组的子区列表。
	 *
	 * @param parentId  父 ID，即群组 ID。
	 * @param limit		每页期望返回的子区数。取值范围为 [1,50]。
	 * @param cursor	开始取数据的游标位置。首次调用方法时传 `null` 或空字符串，按用户加入子区时间的倒序获取数据。
	 * @param callBack  结果回调：
	 *                  - 成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回获取结果 {@link EMCursorResult}，包含子区列表以及下次查询的游标。
	 *        			- 失败时回调 {@link EMValueCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~english
	 * Use the pagination to get the list of message threads that the current user has joined in the specified group.
	 *
	 * This method gets data from the server.
	 *
	 * @param parentId  The parent ID, which is the group ID.
	 * @param limit     The number of message threads that you expect to get on each page. The value range is [1,50].
	 * @param cursor    The position from which to start getting data. At the first method call, if you set `cursor` to `null` or an empty string, the SDK will get data in the reverse chronological order of when the user joins the message threads.
	 * @param callBack  The result callback:
	 *                  - If success, {@link EMValueCallBack#onSuccess(Object)} is triggered to return the result {@link EMCursorResult}, including the message thread list and the cursor for the next query.
	 *                  - If a failure occurs, {@link EMValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getJoinedChatThreadsFromServer(String parentId, int limit, String cursor, EMValueCallBack<EMCursorResult<EMChatThread>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMCursorResult<EMChatThread> result = getJoinedChatThreadsFromServer(parentId, limit, cursor);
					if(callBack != null) {
						callBack.onSuccess(result);
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getJoinedThreadsFromServer error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private EMCursorResult<EMChatThread> getJoinedChatThreadsFromServer(String parentId, int limit, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAThreadInfo> cursorResult = emaObject.getJoinedThreadsFromServer(parentId, limit, cursor, error);
		handleError(error);
		EMCursorResult<EMChatThread> result = new EMCursorResult<>();
		List<EMChatThread> threadList = new ArrayList<>();
		List<EMAThreadInfo> data = cursorResult.getData();
		for(int i = 0; i < data.size(); i++) {
			threadList.add(new EMChatThread(data.get(i)));
		}
		result.setData(threadList);
		result.setCursor(cursorResult.getCursor());
		return result;
	}

	/**
	 * \~chinese
	 * 分页从服务器端获取指定群组的子区列表。
	 *
	 * @param parentId	父 ID，即群组 ID。
	 * @param limit		每页期望返回的子区数。取值范围为 [1,50]。
	 * @param cursor	开始取数据的游标位置。首次获取数据时传 `null` 或空字符串，按子区创建时间的倒序获取数据。
	 * @param callBack  结果回调：
	 *                  - 成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回获取结果 {@link EMCursorResult}，包含子区列表和下次查询的游标。
	 * 					- 失败时回调 {@link EMValueCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~english
	 * Use the pagination to get the list of message threads in the specified group.
	 *
	 * This method gets data from the server.
	 *
	 * @param parentId  The parent ID, which is the group ID.
	 * @param limit     The number of message threads that you expect to get on each page. The value range is [1,50].
	 * @param cursor    The position from which to start getting data. At the first method call, if you set `cursor` to `null` or an empty string, the SDK will get data in the reverse chronological order of when message threads are created.
	 * @param callBack  The result callback:
	 *                  - If success, {@link EMValueCallBack#onSuccess(Object)} is triggered to return the result {@link EMCursorResult}, including the message thread list and the cursor for the next query.
	 *                  - If a failure occurs, {@link EMValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getChatThreadsFromServer(String parentId, int limit, String cursor, EMValueCallBack<EMCursorResult<EMChatThread>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMCursorResult<EMChatThread> result = getChatThreadsFromServer(parentId, limit, cursor);
					if(callBack != null) {
					    callBack.onSuccess(result);
					}
				} catch (HyphenateException e) {
					if(callBack != null) {
						callBack.onError(e.getErrorCode(), e.getDescription());
					}else {
						EMLog.e(TAG,"getThreadsFromServer error: "+e.getErrorCode() + " " +e.getDescription());
					}
				}
			}
		});
	}

	private EMCursorResult<EMChatThread> getChatThreadsFromServer(String parentId, int limit, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAThreadInfo> cursorResult = emaObject.getThreadsFromServer(parentId, limit, cursor, error);
		handleError(error);
		EMCursorResult<EMChatThread> result = new EMCursorResult<>();
		List<EMChatThread> threadList = new ArrayList<>();
		List<EMAThreadInfo> data = cursorResult.getData();
		for(int i = 0; i < data.size(); i++) {
			threadList.add(new EMChatThread(data.get(i)));
		}
		result.setData(threadList);
		result.setCursor(cursorResult.getCursor());
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器批量获取指定子区中的最新一条消息。
	 *
	 * @param chatThreadIds 要查询的子区 ID 列表，每次最多可传 20 个子区。
	 * @param callBack	    结果回调：
	 *                      - 成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回一个 Map 集合，键为子区 ID，值是对应子区最新消息；
	 *                  	- 失败时回调 {@link EMValueCallBack#onError(int, String)}，返回错误信息。
	 *
	 * \~chinese
	 * Gets the last reply in the specified message threads from the server.
	 *
	 * @param chatThreadIds The list of message thread IDs to query. You can pass a maximum of 20 message thread IDs each time.
	 * @param callBack      The result callback:
	 *                      - If success, {@link EMValueCallBack#onSuccess(Object)} is triggered to return a Map collection that contains key-value pairs where the key is the message thread ID and the value is the last threaded reply.
	 * 	                    - If a failure occurs, {@link EMValueCallBack#onError(int, String)} is triggered to return an error.
	 */
	public void getChatThreadLatestMessage(List<String> chatThreadIds, EMValueCallBack<Map<String, EMMessage>> callBack) {
		mClient.execute(()-> {
			Map<String, EMMessage> messageMap = null;
			try {
				messageMap = getChatThreadLatestMessage(chatThreadIds);
				if(callBack != null) {
					callBack.onSuccess(messageMap);
				}
			} catch (HyphenateException e) {
				if(callBack != null) {
				    callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	private Map<String, EMMessage> getChatThreadLatestMessage(List<String> chatThreadIds) throws HyphenateException {
		EMAError error = new EMAError();
		Map<String, EMAMessage> result = emaObject.getThreadsLatestMessage(chatThreadIds, error);
		handleError(error);
		Map<String, EMMessage> map = new HashMap<>();
		Iterator<Map.Entry<String, EMAMessage>> iterator = result.entrySet().iterator();
		while (iterator.hasNext()) {
			Map.Entry<String, EMAMessage> item = iterator.next();
			String key = item.getKey();
			EMMessage value = new EMMessage(item.getValue());
			map.put(key, value);
		}
		return map;
	}

	private void handleError(EMAError error)  throws HyphenateException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new HyphenateException(error);
		}
	}

	private EMAThreadManagerListener listenerImpl = new EMAThreadManagerListener() {
		@Override
		public void onThreadNameUpdated(EMAThreadInfo event) {

		}

		@Override
		public void onLeaveThread(EMAThreadInfo event, int reason) {
			synchronized (threadChangeListeners) {
				for (EMChatThreadChangeListener listener : threadChangeListeners) {
					try {
						if(reason == EMAThreadInfo.LeaveReason.BE_KICKED.ordinal()) {
							listener.onChatThreadUserRemoved(new EMChatThreadEvent(event));
						}else {
							listener.onChatThreadDestroyed(new EMChatThreadEvent(event));
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}

		@Override
		public void onMemberJoined(EMAThreadInfo event) {

		}

		@Override
		public void onMemberExited(EMAThreadInfo event) {

		}

		@Override
		public void onThreadNotifyChange(EMAThreadInfo event) {
			synchronized (threadChangeListeners) {
				for (EMChatThreadChangeListener listener : threadChangeListeners) {
					try {
						switch (event.getType()) {
							case CREATE:
								listener.onChatThreadCreated(new EMChatThreadEvent(event));
						        break;
							case UPDATE:
							case UPDATE_MSG:
								listener.onChatThreadUpdated(new EMChatThreadEvent(event));
							    break;
							case DELETE:
								listener.onChatThreadDestroyed(new EMChatThreadEvent(event));
							    break;
						}
					} catch (Exception e) {
						EMLog.e(TAG, e.getMessage());
					}
				}
			}
		}
	};
}
