/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMChatRoomChangeListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatRoom;
import com.hyphenate.chat.adapter.EMAChatRoomManager;
import com.hyphenate.chat.adapter.EMAChatRoomManagerListener;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.HyphenateException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * \~chinese
 * 聊天室管理类，负责聊天室加入和退出、聊天室列表获取以及成员权限管理等。
 * 比如，加入聊天室：
 * ```java
 *  EMClient.getInstance().chatroomManager().joinChatRoom(conversationId, new EMValueCallBack<EMChatRoom>() {
 *
 *     	public void onSuccess(EMChatRoom value) {
 *     	    //成功加入聊天室后的处理逻辑。
 *     	}
 *
 *     	public void onError(int error, String errorMsg) {
 *     	    //加入聊天室失败后的处理逻辑。
 *     	}
 * 	});
 * ```
 *
 * \~english
 * The chat room manager, which manages users joining and existing the chat room and getting the chat room list, and manages member privileges.
 * For example, joining a chat room:
 * ```java
 *   EMClient.getInstance().chatroomManager().joinChatRoom(conversationId, new EMValueCallBack<EMChatRoom>() {
 *    	public void onSuccess(EMChatRoom value) {
 *    	    //Processing logic for successful chat room joining.
 *        }
 *    	public void onError(int error, String errorMsg) {
 *    	    //Processing logic for chat room joining failures.
 *        }
 * 	});
 * ```
 */
public class EMChatRoomManager {
    EMAChatRoomManager emaObject;
	private List<EMChatRoomChangeListener> chatRoomListeners = Collections.synchronizedList(new ArrayList<EMChatRoomChangeListener>());
	private ExecutorService threadPool = null;
	private List<EMChatRoom> chatRooms = Collections.synchronizedList(new ArrayList<EMChatRoom>());
	
	EMClient mClient;
	public EMChatRoomManager(EMClient client, EMAChatRoomManager manager) {
	    emaObject = manager;
	    emaObject.addListener(chatRoomListenerImpl);
		mClient = client;
		threadPool = Executors.newCachedThreadPool();
	}

	/**
	 * \~chinese
	 * 注册聊天室事件监听对象。
	 * 聊天室被销毁、成员的加入和退出、禁言和加入白名单等操作均可通过设置 {@link EMChatRoomChangeListener} 进行监听。
	 *
	 * 利用本方法注册聊天室事件监听对象后，可调用 {@link #removeChatRoomListener(EMChatRoomChangeListener)} 将其移除。
	 *
	 * @param listener 聊天室事件监听对象，详见 {@link EMChatRoomChangeListener}。
	 *
	 * \~english
	 * Registers a chat room event listener.
	 * Chat room destruction, member entry and exit, mute, and allowlist entry addition among other operations can be listened by setting
	 * {@link EMChatRoomChangeListener}.
	 *
	 * Chat room event listeners registered with this method can be removed by calling {@link #removeChatRoomListener(EMChatRoomChangeListener)}.
	 *
	 * @param listener A chat room listener. See {@link EMChatRoomChangeListener}.
	 */
	public void addChatRoomChangeListener(EMChatRoomChangeListener listener) {
		chatRoomListeners.add(listener);
	}

    /**
     * \~chinese
     * 移除聊天室事件监听对象。
	 * 利用 {@link #addChatRoomChangeListener(EMChatRoomChangeListener)} 注册聊天室事件监听对象后，可调用本方法将其移除。目前，推荐使用
	 * {@link #removeChatRoomListener(EMChatRoomChangeListener)} 移除聊天室事件监听对象。
	 *
	 * @param listener 	要移除的聊天室监听对象。
	 * @deprecated  	已废弃，请用 {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} 代替。
	 *
	 * \~english
	 * Removes a chat room event listener.
	 * This method removes a chat room event listener registered with {@link #addChatRoomChangeListener(EMChatRoomChangeListener)}.
	 * Currently, {@link #removeChatRoomListener(EMChatRoomChangeListener)} is recommended to remove a chat room event listener.
	 *
	 * @param listener	The chat room listener to be removed.
	 * @deprecated  	Deprecated. Please use {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} instead.
	 */
    @Deprecated
	public void removeChatRoomChangeListener(EMChatRoomChangeListener listener) {
	    removeChatRoomListener(listener);
    }

	/**
	 * \~chinese
	 * 移除聊天室事件监听对象。
	 * 利用 {@link #addChatRoomChangeListener(EMChatRoomChangeListener)} 注册聊天室事件监听对象后，可调用本方法将其移除。
	 *
	 * @param listener	要移除的聊天室事件监听对象。
	 *
	 * \~english
	 * Removes the chat room event listener.
	 * This method removes the chat room event listener registered with {@link #addChatRoomChangeListener(EMChatRoomChangeListener)}.
	 *
	 * @param listener	The chat room event listener to be removed.
	 */
	public void removeChatRoomListener(EMChatRoomChangeListener listener)
	{
		chatRoomListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 加入聊天室。
	 * 退出聊天室调用 {@link #leaveChatRoom(String)}。
	 *
	 * 异步方法。
	 *
	 * @param roomId 	聊天室 ID。
	 * @param callback	聊天室加入结果回调。加入成功，回调 {@link EMValueCallBack#onSuccess(Object)} 并返回聊天室对象。
	 *                  加入失败，回调 {@link EMValueCallBack#onError(int, String)}，其中第一个参数为错误参数，第二各参数为错误信息。
	 * 
	 * \~english
     * Joins the chat room.
	 * To exit the chat room, call {@link #leaveChatRoom(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param roomId 	The ID of the chat room to join.
	 * @param callback 	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and returns the chat room object;
	 *                  if this call fails, calls {@link EMValueCallBack#onError(int, String)}, where the first parameter is the error code and
	 *                  the second is the error message.
	 */
	public void joinChatRoom(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
		threadPool.submit(new Runnable() {                                         

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMAChatRoom r = EMChatRoomManager.this.emaObject.joinChatRoom(roomId, error);
				EMChatRoom room = new EMChatRoom(r);
				if (error.errCode() == EMAError.EM_NO_ERROR || error.errCode() == EMAError.CHATROOM_ALREADY_JOINED) {
					callback.onSuccess(room);
				} else {
					callback.onError(error.errCode(), error.errMsg());
				}
			}
		});

	}
	
	/**
	 * \~chinese
	 * 离开聊天室。
	 * 利用 {@link #joinChatRoom(String, EMValueCallBack)} 加入聊天室后，离开时调用此方法。
	 *
	 * 异步方法。
	 *
	 * @param roomId 	聊天室 ID。
	 *
	 * \~english
	 * Leaves a chat room.
	 * This method is called for a member to leave a chat room the members joins using {@link #joinChatRoom(String, EMValueCallBack)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param roomId 	The ID of the chat room to leave.
	 */
	public void leaveChatRoom(final String roomId) {
		EMChatRoom room = getChatRoom(roomId);
		if (room == null) {
		    return;
		}

        EMOptions options = EMClient.getInstance().getOptions();

        boolean allowed = options.isChatroomOwnerLeaveAllowed();
        String owner = room.getOwner();
        if(!allowed && owner.equals(EMSessionManager.getInstance().getLastLoginUser())){
            return;
        }

        // The following is a temporary workaround to delete the conversation in case of leaving the room,
        // but deleteConversation could be a time-consuming operation.
		boolean delete = options.isDeleteMessagesAsExitChatRoom();
        if(delete) {
			EMClient.getInstance().chatManager().deleteConversation(roomId,true);
        }
		threadPool.submit(new Runnable() {

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMChatRoomManager.this.emaObject.leaveChatRoom(roomId, error);
			}
		});
	}

	/**
	 * \~chinese
	 * 以分页的方式从服务器获取聊天室数据。
	 * 此方法为同步方法，也可使用异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param pageNum 				当前页码，从 1 开始。
	 * @param pageSize 				每页返回的记录数。
	 * @return  					分页获取结果 {@link EMPageResult}。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * 需注意以下两点：
	 *  - pageSize 是这次调用期望返回的记录数，如当前在最后一页，返回的数量小于等于 pageSize 的值；
	 *  - pageNum 是当前页码，方便服务器分页查询返回。
	 *    对于数据量未知且很大的情况，分页获取，服务器会根据每次的 pageSize 和 pageNum 的设置返回数据。

	 * \~english
	 * Gets chat room data from the server with pagination.
	 * This is a synchronous method. Also, the asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)} can be used.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param pageNum 				The page number, starting from 1.
	 * @param pageSize 				The number of records per page.
	 * @return      				Chat room data. See {@link EMPageResult}.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
	 *
	 * Note the following:
     * - pageSize specifies the number of records expected to return for this call. 
	 *   For the last page, the actual number of returned records is less than the value of pageSize.
     * - pageNum specifies the number of the current page on which the returned data is presented. 
	 *   For a large but unknown quantity of data, the server will return data with pagination as specified by pageSize and pageNum.

	 */
	public EMPageResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		EMPageResult<EMAChatRoom> result = emaObject.fetchChatroomsWithPage(pageNum, pageSize, error);
		handleError(error);

		List<EMAChatRoom> rooms = result.getData();
		int count = result.getPageCount();

		EMPageResult<EMChatRoom> ret = new EMPageResult<EMChatRoom>();
		List<EMChatRoom> data = new ArrayList<EMChatRoom>();
		for (EMAChatRoom room : rooms) {
			data.add(new EMChatRoom(room));
		}
		ret.setPageCount(count);
		ret.setData(data);

		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}

	/**
	 * \~chinese
	 * 以分页的方式从服务器获取聊天室数据。
	 * 返回的结果中，当 EMCursorResult.getCursor() 为空字符串 ("") 时，表示数据获取完毕。
	 * 此方法为耗时操作，可以调用异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param pageSize 				每页返回的记录数。
	 * @param cursor 				从游标位置开始取数据。
	 * @return   					分页获取结果 {@link EMCursorResult}。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 * @deprecated 					已废弃，请用 {@link #fetchPublicChatRoomsFromServer(int, int)} 代替。
	 *
	 * 
	 * 
	 * 
	 * \~english
	 * Gets chat room data from the server with pagination.
	 * When EMCursorResult.getCursor() is an empty string ("") amid the result, all data is fetched.
	 * As this method is time-consuming, the asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)} can be used.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param pageSize 				The number of records per page.
	 * @param cursor 				The cursor position from which to start getting data.
	 * @return 					Chat room data. See {@link EMCursorResult}.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
	 * @deprecated 					Deprecated. Please use {@link #fetchPublicChatRoomsFromServer(int, int)} instead.
	 


	 */
	@Deprecated
	public EMCursorResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageSize, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAChatRoom> cursorResult = emaObject.fetchChatroomsWithCursor(cursor, pageSize, error);
		handleError(error);
		
		EMCursorResult<EMChatRoom> ret = new EMCursorResult<EMChatRoom>();
		List<EMChatRoom> data = new ArrayList<EMChatRoom>();
		for (EMAChatRoom room : cursorResult.getData()) {
			data.add(new EMChatRoom(room));
		}
		ret.setCursor(cursorResult.getCursor());
		ret.setData(data);
		
		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}
	
	/**
     * \~chinese
	 * 以分页的方式从服务器获取聊天室。
	 * 返回的结果中，当 EMCursorResult.getCursor() 为空字符串 ("") 时，表示没有更多数据。
	 *
	 * 异步方法。
	 *
	 * @param pageSize 	每页返回的记录数。
	 * @param cursor 	从游标位置开始取数据。
	 * @param callback	结果回调，成功回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * @deprecated 		使用 {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)} 代替。
	 *
	 * \~english
	 * Gets chat room data from the server with pagination.
	 * When EMCursorResult.getCursor() is an empty string ("") amid the result, all data is fetched.
	 *
	 * This is an asynchronous method.
	  *
	 * @param pageSize 	The number of records per page.
	 * @param cursor 	The cursor position from which to start getting data.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                  If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 *
	 * @deprecated 		Deprecated. Please use {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)} instead.
	 * 
	      */
	@Deprecated
	public void asyncFetchPublicChatRoomsFromServer(final int pageSize, final String cursor, final EMValueCallBack<EMCursorResult<EMChatRoom>> callback) {
	    EMClient.getInstance().execute(new Runnable() {
	        
	        @Override
            public void run() {
                try {
                    EMCursorResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageSize, cursor);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
	
    
    /**
     * \~chinese
     * 以分页的方式从服务器获取聊天室。
	 *
     * 异步方法。
	 *
     * @param pageNum 	当前页数，从 1 开始。
     * @param pageSize 	每页返回的记录数。
     * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 * 需注意以下两点：
     *  - pageSize 是这次调用期望返回的记录数，如当前在最后一页，返回的数量小于等于 pageSize 的值；
     *  - pageNum 是当前页码，方便服务器分页查询返回。
	 *    对于数据量未知且很大的情况，分页获取，服务器会根据每次的 pageSize 和 pageNum 的设置返回数据。

     *
     * \~english
     * Gets chat room data from the server with pagination.
	 *
     * This is an asynchronous method.
     *
     * @param pageNum 	The page number, starting from 1.
     * @param pageSize 	The number of records per page.
     * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)}.
	 *                  If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
     * Note the following:
     * - pageSize specifies the number of records expected to return for this call. 
	 *   For the last page, the actual number of returned records is no greater than the value of pageSize.
     * - pageNum specifies the number of the current page on which the returned data is presented. 
	 *   For a large but unknown quantity of data, the server will return data with pagination as specified by pageSize and pageNum.

	 */
	public void asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final EMValueCallBack<EMPageResult<EMChatRoom>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMPageResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageNum, pageSize);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 从服务器获取聊天室详情，默认不取成员列表。
	 * 异步方法见 {@link #asyncFetchChatRoomFromServer(String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param roomId				聊天室 ID。
	 * @return EMChatRoom			返回聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，错误码详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Gets details of a chat room from the server, excluding the member list by default.
	 * For the asynchronous method, see {@link #asyncFetchChatRoomFromServer(String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param roomId				The chat room ID.
	 * @return EMChatRoom			The chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
	 */
	public EMChatRoom fetchChatRoomFromServer(String roomId) throws HyphenateException {
		return fetchChatRoomFromServer(roomId, false);
	}
	
	/**
	 * \~chinese
	 * 从服务器获取聊天室详情。
	 * 如果需要取成员列表，默认最多取 200 个成员，超出部分，请调用 {@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param roomId 		        聊天室ID。
	 * @param fetchMembers 			是否需要获取聊天室成员。
	 * @return EMChatRoom			返回的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Gets details of a chat room from the server.
	 * The member list, if required, can contain at most 200 members by default. For more members, call {@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param roomId 				The chat room ID.
	 * @param fetchMembers 			Whether to get chat room members.
	 * @return EMChatRoom			The chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
	 */
 	public EMChatRoom fetchChatRoomFromServer(String roomId, boolean fetchMembers)throws HyphenateException {
 	   EMAError error = new EMAError();
       EMAChatRoom room = emaObject.fetchChatroomSpecification(roomId, error, fetchMembers);
       handleError(error);
       return new EMChatRoom(room);
 	}
	
	/**
     * \~chinese
     * 从服务器获取聊天室详情，默认不取成员列表。
	 * 同步方法见 {@link #fetchChatRoomFromServer(String)}。
	 *
     * 异步方法。
	 *
     * @param roomId	聊天室 ID。
	 * @param callback	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                  失败时回调 {@link EMValueCallBack#onError(int, String)}。
     *
     * \~english
     * Gets details of a chat room from the server, excluding the member list by default.
     * For the synchronous method, see {@link #fetchChatRoomFromServer(String)}.
	 *
	 * This is an asynchronous method.
	 *
     * @param roomId	The chat room ID.
	 * @param callback	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                  if this call fails, calls{@link EMValueCallBack#onError(int, String)}.
     */
    public void asyncFetchChatRoomFromServer(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMChatRoom chatRoom = fetchChatRoomFromServer(roomId);
                    callback.onSuccess(chatRoom);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });   
    }
	
	/**
	 * \~chinese
	 * 从内存中获取聊天室。
	 *
	 * @param roomId		聊天室 ID。
     * @return				返回聊天室对象。如果内存中不存在聊天室对象，返回 null。
	 *
	 * \~english
	 * Gets the chat room in the cache.
	 *
	 * @param roomId		The chat room ID.
	 * @return 				The chat room instance. Returns null if the chat room is not found in the cache.
	 */
	public EMChatRoom getChatRoom(String roomId) {
		EMAChatRoom room = emaObject.getChatroom(roomId);
		if(room == null){
			return null;
		}
		return new EMChatRoom(room);
	}
	
	/**
	 * \~chinese
	 * 获取当前内存的聊天室列表。
	 * 注：调用以下任何一种方法后，调用该方法才能有返回数据：
	 * （1）{@link #fetchPublicChatRoomsFromServer(int, int)} 或其异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}
	 * （2）{@link #fetchPublicChatRoomsFromServer(int, String)} 或其异步方法 {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}
	 *   返回的数据是以上方法最后一次分页获取的数据。
	 *
	 * @return	返回 EMChatRoomManager 维护的聊天室列表。
	 * @deprecated
	 * 
	 * \~english
	 * Gets the list of chat rooms in the cache.
	 * Note: Data can be returned upon the call of this method only after one of the following methods is called:
	 * - {@link #fetchPublicChatRoomsFromServer(int, int)} or its asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, int, EMValueCallBack)}
	 * - {@link #fetchPublicChatRoomsFromServer(int, String)} or its asynchronous method {@link #asyncFetchPublicChatRoomsFromServer(int, String, EMValueCallBack)}
	 *  The returned data is the data last fetched by using one of the preceding methods.
	 * @return	The list of chat rooms maintained by EMChatRoomManager.
	 * @deprecated
	 */
	@Deprecated
	public List<EMChatRoom> getAllChatRooms() {
		return Collections.unmodifiableList(chatRooms);
	}

	EMAChatRoomManagerListener chatRoomListenerImpl = new EMAChatRoomManagerListener() {

		@Override
		public void onLeaveChatRoom(EMAChatRoom chatroom, int leaveReason) {
            EMClient.getInstance().chatManager().caches.remove(chatroom.getId());
		    synchronized (chatRoomListeners) {
			    try {
			        for (EMChatRoomChangeListener listener : chatRoomListeners) {
			            if (leaveReason == DESTROYED) {
			                listener.onChatRoomDestroyed(chatroom.getId(), chatroom.getName());
			            } else {
							listener.onRemovedFromChatRoom(leaveReason, chatroom.getId(), chatroom.getName(), EMClient.getInstance().getCurrentUser());
						}
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberJoinedChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
				    for (EMChatRoomChangeListener listener : chatRoomListeners) {
					    listener.onMemberJoined(chatroom.getId(), member);
				    }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberLeftChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
			        for (EMChatRoomChangeListener listener : chatRoomListeners) {
			            listener.onMemberExited(chatroom.getId(), chatroom.getName(), member);
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		public void onAddMuteList(EMAChatRoom chatRoom, List<String> mutes, long expireTime) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListAdded(chatRoom.getId(), mutes, expireTime);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveMutes(EMAChatRoom chatRoom, List<String> mutes) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListRemoved(chatRoom.getId(), mutes);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListAdded(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListAdded(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListRemoved(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListRemoved(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAllMemberMuteStateChanged(EMAChatRoom chatRoom, final boolean isMuted) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAllMemberMuteStateChanged(chatRoom.getId(), isMuted);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAddAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminAdded(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminRemoved(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onOwnerChanged(EMAChatRoom chatRoom, String newOwner, String oldOwner) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onOwnerChanged(chatRoom.getId(), newOwner, oldOwner);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAnnouncementChanged(EMAChatRoom chatRoom, String announcement) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAnnouncementChanged(chatRoom.getId(), announcement);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onUpdateSpecificationFromChatroom(EMAChatRoom chatroom) {
			synchronized (chatRoomListeners) {
				for (EMChatRoomChangeListener listener : chatRoomListeners) {
					try {
						if (null != chatroom) {
							listener.onSpecificationChanged(new EMChatRoom(chatroom));
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
		}
	};


	/**
	 * \~chinese
	 * 创建聊天室。
	 * 异步方法见 {@link #asyncCreateChatRoom(String, String, String, int, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param subject           	聊天室名称。
	 * @param description       	聊天室描述。
	 * @param welcomeMessage    	邀请成员加入聊天室的消息。
	 * @param maxUserCount      	允许加入聊天室的最大成员数。
	 * @param members           	邀请加入聊天室的成员列表。
	 * @return EMChatRoom 		创建成功的聊天室对象。
     * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Creates a chat room.
	 * For the asynchronous method, see {@link #asyncCreateChatRoom(String, String, String, int, List, EMValueCallBack)}. 
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param subject           	The chat room subject.
	 * @param description       	The chat room description.
	 * @param welcomeMessage    	A welcome message inviting members to join the chat room.
	 * @param maxUserCount      	The maximum number of members allowed to join the chat room.
	 * @param members           	The list of members invited to join the chat room.
	 * @return EMChatRoom		The chat room instance created successfully.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public EMChatRoom createChatRoom(String subject, String description, String welcomeMessage,
	                                 int maxUserCount, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.createChatRoom(subject, description, welcomeMessage, EMChatRoom.EMChatRoomStyle.EMChatRoomStylePublicOpenJoin.ordinal(), maxUserCount, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 创建聊天室。
	 * 同步方法见 {@link #createChatRoom(String, String, String, int, List)}。
	 *
	 * 异步方法。
	 *
	 * @param subject           聊天室名称。
	 * @param description       聊天室描述。
	 * @param welcomeMessage    邀请成员加入聊天室的消息。
	 * @param maxUserCount      允许加入聊天室的最大成员数。
	 * @param members           邀请加入聊天室的成员列表。
	 * @param callBack          结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)},
	 *                          失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Creates a chat room.
	 * For the synchronous method, see {@link #createChatRoom(String, String, String, int, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param subject           The chat room subject.
	 * @param description       The chat room description.
	 * @param welcomeMessage    A welcome message inviting members to join the chat room.
	 * @param maxUserCount      The maximum number of members allowed to join the chat room.
	 * @param members           The list of members invited to join the chat room.
	 * @param callBack			The completion callback. If this call succeeds calls {@link EMValueCallBack#onSuccess(Object)};
	 *                          if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncCreateChatRoom(final String subject, final String description, final String welcomeMessage,
	                                final int maxUserCount, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(createChatRoom(subject, description, welcomeMessage, maxUserCount, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 销毁聊天室。
	 * 仅聊天室所有者可调用此方法。
	 * 异步方法见 {@link #asyncDestroyChatRoom(String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Destroys a chat room.
	 * Only the chat room owner can call this method.
	 * For the asynchronous method, see {@link #asyncDestroyChatRoom(String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public void destroyChatRoom(String chatRoomId)
			throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.destroyChatroom(chatRoomId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 销毁聊天室。
	 * 仅聊天室所有者可调用此方法。
	 * 同步方法见 {@link #destroyChatRoom(String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param callBack	结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 *                      失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Destroys a chat room.
	 * Only the chat room owner can call this method.
	 * For the synchronous method, see {@link #destroyChatRoom(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 *                      if this call fails, calls {@link EMCallBack#onError(int, String)}.
	 */
	public void asyncDestroyChatRoom(final String chatRoomId, final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					destroyChatRoom(chatRoomId);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改聊天室标题。
	 * 仅聊天室所有者可调用此方法。
	 * 异步方法见 {@link #asyncChangeChatRoomSubject(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param newSubject			新的聊天室名称。
	 * @return			       返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Changes the chat room subject.
	 * Only the chat room owner can call this method.
	 * For the asynchronous method, see {@link #asyncChangeChatRoomSubject(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param newSubject			The new subject of the chat room.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}
     */
	public EMChatRoom changeChatRoomSubject(String chatRoomId, String newSubject)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomSubject(chatRoomId, newSubject, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 修改聊天室标题。
	 * 仅聊天室所有者可调用此方法。
	 * 同步方法见 {@link #changeChatRoomSubject(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param newSubject	新的聊天室名称。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Changes the chat room subject.
	 * Only the chat room owner can call this method.
	 * For the synchronous method, see {@link #changeChatRoomSubject(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	A chat room ID.
	 * @param newSubject	The new subject of the chat room.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeChatRoomSubject(final String chatRoomId, final String newSubject, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatRoomSubject(chatRoomId, newSubject));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改聊天室描述信息。
	 * 仅聊天室所有者可调用此方法。
	 * 异步方法见 {@link #asyncChangeChatroomDescription(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param newDescription		新的聊天室描述。
	 * @return						修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见{@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Modifies the chat room description.
	 * Only the chat room owner can call this method.
	 * For the asynchronous method, see {@link #asyncChangeChatroomDescription(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param newDescription		The new description of the chat room.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public EMChatRoom changeChatroomDescription(String chatRoomId, String newDescription)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomDescription(chatRoomId, newDescription, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 修改聊天室描述信息。
	 * 仅聊天室所有者可调用此方法。
	 * 同步方法见 {@link #changeChatroomDescription(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId		聊天室 ID。
	 * @param newDescription	新的聊天室描述。
	 * @param callBack   		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                          失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Modifies the chat room description.
	 * Only the chat room owner can call this method.
	 * For the synchronous method, see {@link #changeChatroomDescription(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId		The chat room ID.
	 * @param newDescription	The new description of the chat room.
	 * @param callBack			The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                          if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeChatroomDescription(final String chatRoomId, final String newDescription, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatroomDescription(chatRoomId, newDescription));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取聊天室成员列表。
	 * 返回的结果中，当 EMCursorResult.getCursor() 为空字符串 ("") 时，表示没有更多数据。
	 * 异步方法见 {@link #asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param cursor				从这个游标位置开始取数据。
	 * @param pageSize				每页返回的成员数。
	 * @return						分页获取结果 {@link EMCursorResult}。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 * 请注意以下几点：
	 *  - 每次调用只返回一页的数据。首次调用 cursor 传空值，从最新数据开始获取；
	 *    然后根据服务器返回的数据，其中一个字段是 cursor，保存在本地，下次调用接口时，会把更新的 cursor 传入作为获取数据的标志位置。
     *  - pageSize 是这次接口调用期望返回的数据条目数，如当前在最后一页，返回的数据量小于等于 pageSize 的值；
     *  - 对于数据量未知且很大的情况，分页获取，服务器会根据每次的 pageSiz 和 cursor 设置返回数据。

	 * 
	 * \~english
	 * Gets the chat room member list.
	 * When EMCursorResult.getCursor() is an empty string ("") amid the result, all data is fetched.
	 * For the asynchronous method, see {@link #asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param cursor				The cursor position from which to start getting data.
	 * @param pageSize				The number of members per page. 
	 * @return						The list of chat room members. See {@link EMCursorResult}.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     Note the following:
     *  - One page of data is returned each time this method is called. 
	 *    During the first call to this method, null is passed to cursor and the latest records are returned. 
	 *    Amid the returned data, cursor is a field saved locally and the updated parameter will be passed to be the position from which to obtain data.
     *  - pageSize specifies the number of the current page on which the returned data is presented. 
	 *    For the last page, the actual number of returned records is no greater than the value of pageSize.
     *  - For a large but unknown quantity of data, the server will return data as specified by pageSize and the cursor.


	 */
	public EMCursorResult<String> fetchChatRoomMembers(String chatRoomId, String cursor, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult result = emaObject.fetchChatroomMembers(chatRoomId, cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 获取聊天室成员列表。
	 * 返回的结果中，当 EMCursorResult.getCursor() 为空字符串 ("") 时，表示没有更多数据。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param cursor		从这个游标位置开始取数据。
	 * @param pageSize		每页返回的成员数。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 * 需注意以下几点：
     *  - 每次调用只返回一页的数据。首次调用 cursor 传空值，从最新数据开始获取；然后根据服务器返回的数据，其中一个字段是 cursor，保存在本地，下次调用接口时，会把更新的 cursor 传入作为获取数据的标志位置。
     *  - pageSize 是这次接口调用期望返回的记录数，如当前在最后一页，返回的数据量小于等于 pageSize 的值；
     *  - 对于数据量未知且很大的情况，分页获取，服务器会根据每次的 pageSize 和 cursor 设置返回数据。


	 * \~english
	 * Gets the chat room member list.
	 * When EMCursorResult.getCursor() is an empty string ("") amid the result, all data is fetched.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param cursor		The cursor position from which to get data.
	 * @param pageSize		The number of members per page.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 * Note the following:
     * - One page of data is returned each time this method is called. 
	 *   During the first call to this method, null is passed to cursor and the latest records are returned. 
	 *   Amid the returned data, cursor is a field saved locally and the updated parameter will be passed to be the position from which to obtain data.
     * - pageSize specifies the number of the current page on which the returned data is presented. 
	 *   For the last page, the actual number of returned records is no greater than the value of pageSize.
     * - For a large but unknown quantity of data, the server will return data as specified by pageSize and the cursor.


	 */
	public void asyncFetchChatRoomMembers(final String chatRoomId, final String cursor, final int pageSize, final EMValueCallBack<EMCursorResult<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMembers(chatRoomId, cursor, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 禁止聊天室成员发言。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 异步方法见 {@link #asyncMuteChatRoomMembers(String, List, long, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param muteMembers 			禁言的成员列表。
	 * @param duration 		        禁言时长，单位是毫秒。
	 * @return			        返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Mutes members in a chat room.
	 * Only the chat room owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncMuteChatRoomMembers(String, List, long, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param muteMembers 			The list of members to be muted.
	 * @param duration 			The mute duration in milliseconds.
	 * @return				The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public EMChatRoom muteChatRoomMembers(String chatRoomId, List<String> muteMembers, long duration)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.muteChatroomMembers(
				chatRoomId, muteMembers, duration, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 禁止聊天室成员发言。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 同步方法见 {@link #muteChatRoomMembers(String, List, long)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param muteMembers   禁言的用户列表。
	 * @param duration 	禁言时长，单位是毫秒。
	 * @param callBack	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Mutes members in a chat room.
	 * Only the chat room owner or admin can call this method.
	 * For the synchronous method, see {@link #muteChatRoomMembers(String, List, long)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param muteMembers 	The list of members to be muted.
	 * @param duration 		The mute duration in milliseconds.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncMuteChatRoomMembers(final String chatRoomId, final List<String> muteMembers, final long duration, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(muteChatRoomMembers(chatRoomId, muteMembers, duration));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解除禁言。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 异步方法见 {@link #asyncUnMuteChatRoomMembers(String, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param members			解除禁言的用户列表。
	 * @return				返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Unmutes members in a chat room.
	 * Only the chat room owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncUnMuteChatRoomMembers(String, List, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members				The list of members to be unmuted.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public EMChatRoom unMuteChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unmuteChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 解除禁言。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 同步方法见 {@link #unMuteChatRoomMembers(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param members       解除禁言的用户列表。
	 * @param callBack	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Unmutes members in a chat room.
	 * Only the chat room owner or admin can call this method.
	 * For the synchronous method, see {@link #unMuteChatRoomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	A chat room ID.
	 * @param members		The list of members to be unmuted.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncUnMuteChatRoomMembers(final String chatRoomId, final List<String> members,
	                                             final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unMuteChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 转移聊天室的所有权。
	 * 仅聊天室所有者可调用此方法。
	 * 异步方法见 {@link #asyncChangeOwner(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param newOwner				新的聊天室拥有者 ID。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Transfers the chat room ownership.
	 * Only the chat room owner can call this method.
	 * For the asynchronous method, see {@link #asyncChangeOwner(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			A chat room ID.
	 * @param newOwner				The ID of the new chat room owner.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
	 */
	public EMChatRoom changeOwner(String chatRoomId, String newOwner)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.transferChatroomOwner(chatRoomId, newOwner, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 转移聊天室的所有权。
	 * 仅聊天室所有者可调用此方法。
	 * 同步方法见 {@link #changeOwner(String, String)}
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param newOwner		新的聊天室所有者 ID。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。 
	 *
	 * \~english
	 * Transfers the chat room ownership.
	 * Only the chat room owner can call this method.
	 * For the synchronous method, see {@link #changeOwner(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param newOwner		The ID of the new chat room owner.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncChangeOwner(final String chatRoomId, final String newOwner, final EMValueCallBack<EMChatRoom> callBack)
			throws HyphenateException {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeOwner(chatRoomId, newOwner));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 添加聊天室管理员。
	 * 仅聊天室所有者可调用此方法。
	 * 异步方法见 {@link #asyncAddChatRoomAdmin(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param admin					要设置的管理员 ID。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Adds a chat room admin.
	 * Only the chat room owner can call this method.
	 * For the asynchronous method, see {@link #asyncAddChatRoomAdmin(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param admin					The ID of the chat room admin to be added.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public EMChatRoom addChatRoomAdmin(String chatRoomId, String admin)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.addChatroomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 添加聊天室管理员。
	 * 仅聊天室所有者可调用此方法。
	 * 同步方法见 {@link #addChatRoomAdmin(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param admin			要设置的管理员 ID。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Adds a chat room admin.
	 * Only the chat room owner can call this method.
	 * For the asynchronous method, see {@link #addChatRoomAdmin(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param admin			The ID of the chat room admin to be added.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncAddChatRoomAdmin(final String chatRoomId, final String admin, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(addChatRoomAdmin(chatRoomId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 移除聊天室管理员权限。
	 * 仅聊天室所有者可调用此方法。
	 * 异步方法见 {@link #asyncRemoveChatRoomAdmin(String, String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param admin					要移除管理员权限的 ID。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Removes privileges of a chat room admin.
	 * Only the chat room owner can call this method.
	 * For the asynchronous method, see {@link #asyncRemoveChatRoomAdmin(String, String, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param admin					The ID of admin whose privileges are to be removed.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public EMChatRoom removeChatRoomAdmin(String chatRoomId, String admin)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 移除聊天室管理员权限。
	 * 仅聊天室所有者可调用此方法。
	 * 同步方法见 {@link #removeChatRoomAdmin(String, String)}}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param admin			要移除管理员权限的 ID。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes the privileges of a chat room admin.
	 * Only the chat room owner can call this method.
	 * For the synchronous method, see {@link #removeChatRoomAdmin(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param admin			The ID of admin whose privileges are to be removed.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomAdmin(final String chatRoomId, final String admin,
	                                final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomAdmin(chatRoomId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取聊天室禁言列表。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 异步方法见 {@link #asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param pageNum				当前页码，从 1 开始。
	 * @param pageSize				每页返回的禁言成员数。
	 * @return  					返回的包含禁言成员 ID 及其禁言时长的 Map，其中每一项的 key 是禁言的成员 ID，value 是禁言时长，单位是毫秒。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 * 
	 * 需注意以下两点：
 
     *  - pageSize 是这次调用期望返回的记录数，如当前在最后一页，返回的数量小于等于 pageSize 的值；
     *  - pageNum 是当前页码，方便服务器分页查询返回。
	 *    对于数据量未知且很大的情况，分页获取，服务器会根据每次的 pageSize 和 pageNum 的设置返回数据。


	 *
	 * \~english
	 * Gets the list of members who are muted in the chat room from the server.
	 * Only the chat room owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID
	 * @param pageNum				The page number, starting from 1.
	 * @param pageSize				The number of muted members per page.
	 * @return 						A map which contains the muted member ID and mute duration, where the key of each entry
	 * 								is the ID of a muted member and the value is the mute duration in milliseconds.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}
     * Note the following:
     * - pageSize specifies the number of muted members expected to return for this call. 
	 *   For the last page, the actual number of returned records is no greater than the value of pageSize.
     * - pageNum specifies the number of the current page on which the returned data is presented. 
	 *   For a large but unknown quantity of data, the server will return data as specified by pageSize and pageNum.

	 
	 
	 */
	public Map<String, Long> fetchChatRoomMuteList(String chatRoomId, int pageNum, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		Map<String, Long> members = emaObject.fetchChatRoomMuteList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return members;
	}

	/**
	 * \~chinese
	 * 获取聊天室禁言列表。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 同步方法见 {@link #fetchChatRoomMuteList(String, int, int)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param pageNum		当前页码，从 1 开始。
	 * @param pageSize		每页返回的禁言成员数。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 * 需注意以下两点：
 
     *  - pageSize 是这次调用期望返回的记录数，如当前在最后一页，返回的数量小于等于 pageSize 的值；
     *  - pageNum 是当前页码，方便服务器分页查询返回。
	 *    对于数据量未知且很大的情况，分页获取，服务器会根据每次的 pageSize 和 pageNum 的设置返回数据。
 
	 *
	 * \~english
	 * Gets the list of members who are muted in the chat room from the server.
	 * Only the chat room owner or admin can call this method.
	 * For the synchronous method, see {@link #fetchChatRoomMuteList(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param pageNum		The page number, starting from 1.
	 * @param pageSize		The number of muted members per page.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 * Note the following:
     * - pageSize specifies the number of muted members expected to return for this call. 
	 *   For the last page, the actual number of returned records is no greater than the value of pageSize.
     * - pageNum specifies the number of the current page on which the returned data is presented. 
	 *   For a large but unknown quantity of data, the server will return data as specified by pageSize and pageNum.
 
	 */
	public void asyncFetchChatRoomMuteList(final String chatRoomId, final int pageNum, final int pageSize,
	                                       final EMValueCallBack<Map<String, Long>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMuteList(chatRoomId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员移出聊天室。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 异步方法见 {@link #asyncRemoveChatRoomMembers(String, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param members			要移出聊天室的用户列表。
	 * @return			        返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Removes members from a chat room.
	 * Only the chat room owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncRemoveChatRoomMembers(String, List, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members			The list of members to be removed.
	 * @return				The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public EMChatRoom removeChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 将成员移出聊天室。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 同步方法见 {@link #removeChatRoomMembers(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID
	 * @param members		要移出的用户列表。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes members from a chat room.
	 * Only the chat room owner or admin can call this method.
	 * For the synchronous method, see {@link #removeChatRoomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param members		The list of members to be removed.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncRemoveChatRoomMembers(final String chatRoomId, final List<String> members,
	                                       final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员添加到聊天室黑名单。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 对于添加到聊天室黑名单的成员，请注意以下几点：
	 * - 成员添加到黑名单的同时，将被服务器移出聊天室。
	 * - 可通过 {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} 回调通知，
	 * 第一个参数是具体原因，加入黑名单返回的原因是 {@link EMAChatRoomManagerListener#BE_KICKED}。
	 * - 添加到黑名单的成员禁止再次加入到聊天室。
	 * 异步方法见 {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param members				要加入黑名单的成员列表。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Adds members to the chat room's blocklist.
	 * Only the chat room owner or admin can call this method.
	 * For members added to blocklist, note the following:
	 * 1. A member, once added to the chat room blocklist, will be removed from the chat room by the server.
	 * 2. The method {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} occurs, 
	 *  where the first parameter is the reason which is {@link EMAChatRoomManagerListener#BE_KICKED}.  
	 * 3. Members added to the blocklist are banned from rejoining the chat room.
	 * For the asynchronous method, see {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members				The list of members to be added to blocklist.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public EMChatRoom blockChatroomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.blockChatroomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 将成员添加到聊天室黑名单。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 对于加入聊天室黑名单的成员，请注意以下几点：
	 * - 成员加入黑名单的同时，将被服务器移出聊天室。
	 * - 可通过 {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} 回调通知，
	 * 第一个参数是具体的原因，加入黑名单返回的原因是 {@link EMAChatRoomManagerListener#BE_KICKED}。
	 * - 加入黑名单的成员禁止再次加入到聊天室。
	 * 同步方法见{@link #blockChatroomMembers(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param members		要加入黑名单的成员列表。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Adds members to the chat room's blocklist.
	 * Only the chat room owner or admin can call this method.
	 * For members added to blacklist, note the following:
	 * - A member, once added to the chat room blocklist, will be removed from the chat room by the server.
	 * - The method {@link EMChatRoomChangeListener#onRemovedFromChatRoom(int, String, String, String)} can be called to send a callback notification,
	 *    where the first parameter is the reason which is {@link EMAChatRoomManagerListener#BE_KICKED}. 
	 * - Members added to the blocklist are banned from rejoining the chat room.
	 * For the synchronous method, see {@link #blockChatroomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param members		The list of members to be added to blocklist.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncBlockChatroomMembers(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(blockChatroomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从聊天室黑名单中移除成员。
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 异步方法见 {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param members				要移除黑名单的成员列表。
	 * @return						返回修改后的聊天室对象。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Removes members from the chat room's blocklist.
	 * Only the chat room owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncBlockChatroomMembers(String, List, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param members				The list of members to be removed from the blocklist.
	 * @return						The modified chat room instance.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
     */
	public EMChatRoom unblockChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unblockChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 从聊天室黑名单中移除成员。
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 同步方法见 {@link #unblockChatRoomMembers(String, List)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param members		要移除黑名单的成员列表。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes members from the chat room's blocklist.
	 * Only the chat room owner or admin can call this method.
	 * For the synchronous method, see {@link #unblockChatRoomMembers(String, List)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param members		The list of members to be removed from the blocklist.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 *                      if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncUnBlockChatRoomMembers(final String chatRoomId, final List<String> members,
	                                        final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unblockChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 以分页的形式获取聊天室黑名单列表。
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 异步方法见 {@link #asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 * 
	 * 需注意以下两点：
 
     *  - pageSize 是这次调用期望返回的记录数，如当前在最后一页，返回的数量小于等于 pageSize 的值；
     *  - pageNum 是当前页码，方便服务器分页查询返回。
	 *    对于数据量未知且很大的情况，分页获取，服务器会根据每次的 pageSize 和 pageNum 的设置返回数据。

	 *
	 * @param chatRoomId			聊天室 ID。
	 * @param pageNum				当前页码，从 1 开始。
	 * @param pageSize				每页返回的黑名单中的用户数。
	 * @return						返回聊天室黑名单列表。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述详见 {@link com.hyphenate.EMError}。
	 *
	 * \~english
	 * Gets the chat room blocklist with pagination.
	 * Only the chat room owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId			The chat room ID.
	 * @param pageNum				The page number, starting from 1.
	 * @param pageSize				The number of users on the blocklist per page.
	 * @return						The chat room's blocklist.
	 * @throws HyphenateException	A description of the exception. See {@link com.hyphenate.EMError}.
	 * 
	 * Note the following:
     * - pageSize specifies the number of records expected to return for this call. 
	 *   For the last page, the actual number of returned records is no greater than the value of pageSize.
     * - pageNum specifies the number of the current page on which the returned data is presented. 
	 *   For a large but unknown quantity of data, the server will return data as specified by pageSize and pageNum.

	 * 
     */
	public List<String> fetchChatRoomBlackList(String chatRoomId, int pageNum, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		List<String> blockList = emaObject.fetchChatRoomBlackList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return blockList;
	}

	/**
	 * \~chinese
	 * 以分页的形式获取聊天室黑名单列表。
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 同步方法见 {@link #fetchChatRoomBlackList(String, int, int)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId	聊天室 ID。
	 * @param pageNum		当前页码，从 1 开始。
	 * @param pageSize		每页返回的黑名单上的用户数。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回聊天室黑名单列表；
	 *                      失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 * 需注意以下两点：
     *  - pageSize 是这次调用期望返回的记录数，如当前在最后一页，返回的数量小于等于 pageSize 的值；
     *  - pageNum 是当前页码，方便服务器分页查询返回。对于数据量未知且很大的情况，分页获取，服务器会根据每次的 pageSize 和 pageNum 的设置返回数据。

	 *
	 * \~english
	 * Gets the chat room blocklist with pagination.
	 * Only the chat room owner or admin can call this method.
	 * For the synchronous method, see {@link #fetchChatRoomBlackList(String, int, int)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId	The chat room ID.
	 * @param pageNum		The page number, starting from 1.
	 * @param pageSize		The number of users on the blocklist per page.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and returns the chat room's blocklist;
	 * 						If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 Note the following:
     * - pageSize specifies the number of records expected to return for this call. 
	 *   For the last page, the actual number of returned records is no greater than the value of pageSize.
     * - pageNum specifies the number of the current page on which the returned data is presented. 
	 *   For a large but unknown quantity of data, the server will return data as specified by pageSize and pageNum.

	 */
	public void asyncFetchChatRoomBlackList(final String chatRoomId, final int pageNum, final int pageSize,
	                                        final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomBlackList(chatRoomId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员添加到白名单。
	 * 仅聊天室所有者或管理员可调用此方法。
	 * 聊天室拥有者或者管理员执行 {@link #muteAllMembers(String, EMValueCallBack)} 时，加入白名单的成员不受影响。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 			聊天室 ID。
	 * @param members 				加入白名单的成员列表。
	 * @param callBack				结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 								失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Adds members to the allowlist.
	 * Only the chat room owner or admin can call this method.
	 * For members added to the allowlist, {@link #muteAllMembers(String, EMValueCallBack)}, when executed by the chat room owner or admin, does not work.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 			The chat room ID.
	 * @param members 				The list of members to be added to the allowlist.
	 * @param callBack				The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 								if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void addToChatRoomWhiteList(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.addToWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员从白名单移除。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 成员从白名单移除后，将受到 {@link #muteAllMembers(String, EMValueCallBack)} 功能的影响。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 			聊天室 ID。
	 * @param members 				移除白名单的用户列表。
	 * @param callBack 				结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 								失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Removes members from the blocklist.
	 * Only the chat room owner or admin can call this method.
	 * For members removed from the blocklist, {@link #muteAllMembers(String, EMValueCallBack)} works.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 			The chat room ID.
	 * @param members 				The list of members to be removed from the blocklist.
	 * @param callBack              The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)}; 
	 *                              if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void removeFromChatRoomWhiteList(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.removeFromWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 检查成员自己是否加入了白名单。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack 		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回是否在白名单中；
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Checks whether the member is on the blocklist.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} to show whether the member is on the blocklist;
	 * 						if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void checkIfInChatRoomWhiteList(final String chatRoomId, EMValueCallBack<Boolean> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfInWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});

	}

	/**
	 * \~chinese
	 * 从服务器获取白名单列表。
	 * 仅聊天室所有者和管理员可调用此方法。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the blocklist from the server.
	 * Only the chat room owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 						If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void fetchChatRoomWhiteList(final String chatRoomId, final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					List<String> result = emaObject.fetchChatRoomWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(result);
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 设置全员禁言。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 聊天室拥有者、管理员及加入白名单的用户不受影响。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack 		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Mutes all members.
	 * Only the chat room owner or admin can call this method.
	 * This method does not work for the chat room owner, admin, and members added to the blocklist.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId  	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 						if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void muteAllMembers(final String chatRoomId, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.muteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解除所有成员的禁言状态。
	 * 仅聊天室所有者和管理员可调用此方法。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack 		结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Unmutes all members.
	 * Only the chat room owner or admin can call this method.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack		The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)};
	 * 						If this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void unmuteAllMembers(final String chatRoomId, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.unmuteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 更新聊天室公告。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 异步方法见 {@link #asyncUpdateChatRoomAnnouncement(String, String, EMCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId 			聊天室 ID。
	 * @param announcement 			公告内容。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Updates the chat room announcement.
	 * Only the chat room owner or admin can call this method.
	 * For the asynchronous method, see {@link #asyncUpdateChatRoomAnnouncement(String, String, EMCallBack)}.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param chatRoomId 			The chat room ID.
	 * @param announcement 			The announcement content.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void updateChatRoomAnnouncement(String chatRoomId, String announcement) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.updateChatRoomAnnouncement(chatRoomId, announcement, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 更新聊天室公告。
	 * 仅聊天室所有者和管理员可调用此方法。
	 * 同步方法见 {@link #updateChatRoomAnnouncement(String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param announcement 	公告内容。
	 * @param callBack 		结果回调，成功时回调 {@link EMCallBack#onSuccess()}，
	 * 						失败时回调 {@link EMCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Updates the chat room announcement.
	 * Only the chat room owner or admin can call this method.
	 * For the synchronous method, see {@link #updateChatRoomAnnouncement(String, String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param chatRoomId 	The chat room ID
	 * @param announcement 	The announcement content.
	 * @param callBack 		The completion callback. If this call succeeds, calls {@link EMCallBack#onSuccess()};
	 * 						if this call fails, calls {@link EMCallBack#onError(int, String)}. 
	 */
	public void asyncUpdateChatRoomAnnouncement(final String chatRoomId, final String announcement, final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateChatRoomAnnouncement(chatRoomId, announcement);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室公告内容。
	 * 异步方法见 {@link #asyncFetchChatRoomAnnouncement(String, EMValueCallBack)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param chatRoomId 			聊天室 ID。
	 * @return						聊天室公告。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Gets the chat room announcement from the server.
	 * For the asynchronous method, see {@link #asyncFetchChatRoomAnnouncement(String, EMValueCallBack)}.
	 * 
	 * This is a synchronous method and blocks the current thread.
	 * 
	 * @param chatRoomId 			The chat room ID.
	 * @return						The chat room announcement.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}
	 */
	public String fetchChatRoomAnnouncement(String chatRoomId) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.fetchChatRoomAnnouncement(chatRoomId, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室公告内容。
	 * 同步方法见 {@link #fetchChatRoomAnnouncement(String)}。
	 *
	 * 异步方法。
	 *
	 * @param chatRoomId 	聊天室 ID。
	 * @param callBack  	结果回调，成功时回调 {@link EMValueCallBack#onSuccess(Object)}，返回聊天室公告；
	 * 						失败时回调 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Gets the chat room announcement from the server.
	 * For the synchronous method, see {@link #fetchChatRoomAnnouncement(String)}.
	 * 
	 * This is an asynchronous method.
	 * 
	 * @param chatRoomId 	The chat room ID.
	 * @param callBack  	The completion callback. If this call succeeds, calls {@link EMValueCallBack#onSuccess(Object)} and returns the chat room announcement;
	 * 						if this call fails, calls {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchChatRoomAnnouncement(final String chatRoomId, final EMValueCallBack<String> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomAnnouncement(chatRoomId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	// ============================= chatRoom_reform new add api end
	private void handleError(EMAError error)  throws HyphenateException {
	    if (error.errCode() != EMAError.EM_NO_ERROR) {
	        throw new HyphenateException(error);
	    }
	}
	
	void onLogout() {
	    chatRoomListeners.clear();
	}
}
