/************************************************************
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2013-2015 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import static com.hyphenate.chat.EMMessage.self;

import android.graphics.BitmapFactory.Options;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMConversationListener;
import com.hyphenate.EMError;
import com.hyphenate.EMMessageListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.EMConversation.EMConversationType;
import com.hyphenate.chat.EMMessage.Status;
import com.hyphenate.chat.EMMessage.Type;
import com.hyphenate.chat.adapter.EMAChatManager;
import com.hyphenate.chat.adapter.EMAChatManagerListener;
import com.hyphenate.chat.adapter.EMAConversation;
import com.hyphenate.chat.adapter.EMAConversation.EMAConversationType;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAGroupReadAck;
import com.hyphenate.chat.adapter.EMAMessageReaction;
import com.hyphenate.chat.adapter.EMAMessageReactionChange;
import com.hyphenate.chat.adapter.EMAReactionManager;
import com.hyphenate.chat.adapter.EMAReactionManagerListener;
import com.hyphenate.chat.adapter.message.EMAFileMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.core.EMAdvanceDebugManager;
import com.hyphenate.cloud.EMCloudOperationCallback;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.notification.core.EMNotificationHelper;
import com.hyphenate.util.EMFileHelper;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.ImageUtils;
import com.hyphenate.util.PathUtil;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * \~chinese
 * 聊天管理类，该类负责管理会话（加载，删除等）、发送消息、下载消息附件等。
 * 
 * 比如，发送一条文本消息：
 * 
 * ```java
 *	EMMessage message = EMMessage.createTxtSendMessage(content, toChatUsername);
 * 	EMClient.getInstance().chatManager().sendMessage(message);
 * ```
 *
 * \~english
 * The chat manager. This class is responsible for managing conversations.
 * (such as: load, delete), sending messages, downloading attachments and so on.
 * 
 * Such as, send a text message:
 * 
 * ```java
 * 	EMMessage message = EMMessage.createTxtSendMessage(content, toChatUsername);
 * 	EMClient.getInstance().chatManager().sendMessage(message);
 * ```
 */
public class EMChatManager {

	EMAChatManager emaObject;
	EMAReactionManager emaReactionObject;

	private static final String TAG = "EMChatManager";

	private final static String INTERNAL_ACTION_PREFIX = "em_";

	EMClient mClient;

	Map<String, EMConversation.MessageCache> caches = new Hashtable<String, EMConversation.MessageCache>();

	protected EMChatManager(){}

	protected EMChatManager(EMClient client, EMAChatManager manager, EMAReactionManager reactionManager) {
		mClient = client;

		emaObject = manager;
		emaObject.addListener(chatManagerListenerImpl);

		emaReactionObject = reactionManager;
		emaReactionObject.addListener(mReactionManagerListenerImpl);
	}

	private List<EMMessageListener> messageListeners = new CopyOnWriteArrayList<EMMessageListener>();
	private List<EMConversationListener> conversationListeners = Collections.synchronizedList(new ArrayList<EMConversationListener>());

	EMAChatManagerListener chatManagerListenerImpl = new EMAChatManagerListener() {
//		private final Object lockObj = new Object();
//
//        List<EMMessageListener> cloneSyncedList(List<EMMessageListener> list) {
//            if (list == null) {
//                return new ArrayList<>();
//            } else {
//                synchronized (lockObj) {
//                    return list.subList(0, list.size());
//                }
//            }
//        }

		@Override
		public void onReceiveMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new EMMessage(msg));
					}

					List<EMMessage> remainingMsgs = new ArrayList<EMMessage>();
					for (EMMessage msg : msgs) {
				    	/*if(msg.getChatType() == EMMessage.ChatType.ChatRoom){
                            EMChatRoom room = EMClient.getInstance().chatroomManager().getChatRoom(msg.conversationId());
                            if(room == null){
                            	continue;
                            }
				    	}*/

	                    EMConversation conv = getConversation(msg.conversationId(), EMConversation.msgType2ConversationType(msg.getFrom(), msg.getChatType()), false);
	                    if(conv == null){
	                    	EMLog.d(TAG, "no conversation");
	                    	continue;
	                    }
	                    
	                    // CMD not put into cache
	                    if (msg.getType() != Type.CMD) {
	                        conv.getCache().addMessage(msg);
	                    }
	                    remainingMsgs.add(msg);
				    }
				    
				    if(remainingMsgs.size() <= 0){
						EMLog.d(TAG, "no remainingMsgs");
				    	return;
				    }

					for (EMMessageListener l : messageListeners) {
						try { // do not break the loop if one listener has problem
							EMLog.d(TAG, "onMessageReceived： " + l);
							l.onMessageReceived(remainingMsgs);
						} catch (Exception e) {
							EMLog.d(TAG, "onMessageReceived has problem: " + e.getMessage());
							e.printStackTrace();
						}
					}
				}
			});
		}

		@Override
		public void onReceiveCmdMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						EMMessage message = new EMMessage(msg);
						String action = ((EMCmdMessageBody)message.getBody()).action();
						if(isAdvanceDebugMessage(action)) {
							EMAdvanceDebugManager.getInstance().handleDebugMessage(message, EMAdvanceDebugManager.Type.valueOf(action));
						} else {
							msgs.add(message);
						}
					}
                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onCmdMessageReceived(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }

				}
			});
		}

		@Override
		public void onMessageStatusChanged(final EMAMessage message, final EMAError error) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    try {
                        EMMessage msg = new EMMessage(message);
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageChanged(msg, null);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
	    public void onMessageAttachmentsStatusChanged(final EMAMessage message, final EMAError error) {
            try {
                EMMessage msg = new EMMessage(message);
                for (EMMessageListener l : messageListeners) {
                    l.onMessageChanged(msg, null);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
		}
		
		
        @Override
        public void onReceiveRecallMessages(final List<EMAMessage> messages) {
            mClient.executeOnMainQueue(new Runnable() {

                @Override
                public void run() {
                    List<EMMessage> msgs = new ArrayList<EMMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new EMMessage(msg));
                        getConversation(msg.conversationId()).getCache().removeMessage(msg.msgId());
                    }

                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageRecalled(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
        }
		@Override
		public void onReceiveHasReadAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    List<EMMessage> msgs = new ArrayList<EMMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new EMMessage(msg));
                    }

                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageRead(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveHasDeliveredAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new EMMessage(msg));
					}

                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageDelivered(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveReadAckForConversation(String fromUsername, String toUsername) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					EMLog.d(TAG, "onReceiveConversationHasReadAcks");
					synchronized (conversationListeners) {
						if(!TextUtils.equals(fromUsername, EMClient.getInstance().getCurrentUser())) {
							//1v1聊天中收到对方发送的channel ack消息，将本地的会话的发送消息置为已读（对方已读）,需要重新load数据
							EMConversation conversation = EMClient.getInstance().chatManager().getConversation(fromUsername);
							if(conversation != null) {
								conversation.loadMoreMsgFromDB(null, conversation.getAllMessages().size());
							}
						}
						try {
							for (EMConversationListener l : conversationListeners) {
								l.onConversationRead(fromUsername, toUsername);
							}
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				}
			});
		}

		@Override
		public void onUpdateConversationList(final List<EMAConversation> conversations) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					EMLog.d(TAG, "onUpdateConversationList");
                    synchronized (conversationListeners) {
	                    try {
	                        for (EMConversationListener l : conversationListeners) {
	                            l.onCoversationUpdate();
	                        }
	                    } catch (Exception e) {
		                    e.printStackTrace();
	                    }
                    }
				}
			});
		}

		@Override
		public void onReceivePrivateMessages(List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onPrivateMessages");
					for (EMAMessage msg : messages) {
						EMMessage message = new EMMessage(msg);
						EMNotificationHelper.getInstance().analyzeCmdMessage(message);
					}
				}
			});
		}

		@Override
		public void onReceiveReadAcksForGroupMessage(List<EMAGroupReadAck> acks) {
            mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onReceiveReadAcksForGroupMessage");

					List<EMGroupReadAck> groupAcks = new ArrayList<EMGroupReadAck>();
					for (EMAGroupReadAck ack : acks) {
						groupAcks.add(new EMGroupReadAck(ack));
					}

					try {
						for (EMMessageListener l : messageListeners) {
							l.onGroupMessageRead(groupAcks);
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			});
		}

		@Override
		public void onUpdateGroupAcks() {
        	mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onUpdateGroupAcks");
					try {
						for (EMMessageListener l : messageListeners) {
							l.onReadAckForGroupMessageUpdated();
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			});
		}
	};

	EMAReactionManagerListener mReactionManagerListenerImpl = new EMAReactionManagerListener() {
		@Override
		public void onMessageReactionDidChange(List<EMAMessageReactionChange> reactionChangeList) {
			mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onMessageReactionDidChange");
					List<EMMessageReactionChange> list = new ArrayList<>(reactionChangeList.size());
					EMMessageReactionChange reactionChange;
					for (EMAMessageReactionChange emaReactionChange : reactionChangeList) {
						reactionChange = new EMMessageReactionChange(emaReactionChange);
						list.add(reactionChange);
					}
					for (EMMessageListener l : messageListeners) {
						try {
							l.onReactionChanged(list);
						} catch (Exception e) {
							e.printStackTrace();
						}
					}
				}
			});

		}
	};

	private boolean isAdvanceDebugMessage(String action){
		if(action.startsWith(INTERNAL_ACTION_PREFIX)){
			try {
				EMAdvanceDebugManager.Type.valueOf(action);
				return true;
			} catch (Exception e) {
				//if it is not a field of Type, throws exception
				e.printStackTrace();
			}
		}
		return false;
	}

	/**
	 * \~chinese
	 * 发送消息。
	 * 
	 * 参考：
	 * 如果是语音，图片类等有附件的消息，SDK 会自动上传附件。
	 * 可以通过 {@link EMOptions#setAutoTransferMessageAttachments(boolean)} 设置是否上传到聊天服务器。
	 * 
	 * 发送消息的状态，可以通过设置 {@link EMMessage#setMessageStatusCallback(EMCallBack)} 进行监听。
	 * 
	 * @param msg    待发送消息对象，不能为空。
	 *
	 * \~english
	 * Sends a message。
	 *  
	 * Reference:
	 * If the message is voice, picture and other message with attachment, the SDK will automatically upload the attachment.
	 * You can set whether to upload the attachment to the chat sever by {@link EMOptions#setAutoTransferMessageAttachments(boolean)}. 
	 *
	 * To listen for the status of sending messages, call {@link EMMessage#setMessageStatusCallback(EMCallBack)}.
	 * 
	 * @param msg    The message object to be sent. Make sure to set the param. 
	 */
	public void sendMessage(final EMMessage msg) {
		msg.makeCallbackStrong();

		boolean createConv = msg.getType() != Type.CMD;
		final EMConversation conv = getConversation(msg.conversationId(), EMConversation.msgType2ConversationType(msg.getTo(), msg.getChatType()), createConv, msg.isChatThreadMessage());

		//add the message at first
		if (conv != null) {
			boolean exists = conv.getCache().getMessage(msg.getMsgId()) != null;
			if (!exists) {
				// sent message time is lastMsgTime + 1ms
				long lastMsgTime = System.currentTimeMillis();
				EMMessage lastMsg = conv.getLastMessage();
				if (lastMsg != null) {
					lastMsgTime = lastMsgTime < lastMsg.getMsgTime() ? lastMsg.getMsgTime() : lastMsgTime;
				}
				msg.setMsgTime(lastMsgTime + 1);

				conv.getCache().addMessage(msg);
			}
		}

		class HandleError {
			HandleError(int code, String desc) {
				EMMessage.EMCallbackHolder holder = msg.messageStatusCallBack;
				if (holder != null) {
					holder.onError(code, desc);
				}
			}
		}

    mClient.executeOnSendQueue(new Runnable() {
            
        @Override
        public void run() {
				try {
					String originPath = null;
					// If do not use Chat Server, do not deal with the logic of pictures and videos
					if(mClient.getOptions().getAutoTransferMessageAttachments()) {
						// If message body is image, check scale request, set size, and file name
						if (msg.getType() == Type.IMAGE) {
							//the default status is fail status,
							//which lead to message show fail and then show success in need scale image
							msg.setStatus(Status.INPROGRESS);
							EMImageMessageBody body = (EMImageMessageBody) msg.getBody();
							if (body == null) {
								new HandleError(EMError.GENERAL_ERROR, "Message body can not be null");
								return;
							}

							String localUri = body.getLocalUrl();
							String remoteUrl = body.getRemoteUrl();
							if (TextUtils.isEmpty(remoteUrl)){
								if(!EMFileHelper.getInstance().isFileExist(localUri)) {
									new HandleError(EMError.FILE_INVALID, "File not exists or can not be read");
									return;
								}
							}
							if(!body.isSendOriginalImage() && EMFileHelper.getInstance().isFileExist(localUri)) {
								String scaledImagePath = ImageUtils.getScaledImageByUri(mClient.getContext(), localUri);
								if(!TextUtils.equals(scaledImagePath, localUri)) {
									originPath = localUri;
									long originalSize = EMFileHelper.getInstance().getFileLength(localUri);
									long scaledSize = EMFileHelper.getInstance().getFileLength(scaledImagePath);
									if (originalSize == 0) {
										EMLog.d(TAG, "original image size:" + originalSize);
										new HandleError(EMError.FILE_INVALID, "original image size is 0");
										return;
									}
									EMLog.d(TAG, "original image size:" + originalSize + " scaled image size:" + scaledSize
											+ " ratio:" + (int) (scaledSize / originalSize) + "%");
									localUri = scaledImagePath;
									body.setLocalUrl(EMFileHelper.getInstance().formatInUri(localUri));
									body.setFileLength(scaledSize);
								}
								body.setFileName(EMFileHelper.getInstance().getFilename(localUri));
							}
							// get image width and height
							Options options = ImageUtils.getBitmapOptions(mClient.getContext(), localUri);
							if(options != null) {
								int width = options.outWidth;
								int height = options.outHeight;
								body.setSize(width, height);
							}
						}else if (msg.getType() == Type.VIDEO){
							msg.setStatus(Status.INPROGRESS);
							EMVideoMessageBody body = (EMVideoMessageBody) msg.getBody();
							if (body == null) {
								new HandleError(EMError.GENERAL_ERROR, "Message body can not be null");
								return;
							}
							Uri filePathUri = body.getLocalUri();
							String fileRemoteUrl = body.getRemoteUrl();
							if (TextUtils.isEmpty(fileRemoteUrl)){
								if(!EMFileHelper.getInstance().isFileExist(filePathUri)) {
									new HandleError(EMError.FILE_INVALID, "File not exists or can not be read");
									return;
								}
							}
							// get video width and height
							String thumbPath = EMFileHelper.getInstance().getFilePath(body.getLocalThumbUri());
							if(!TextUtils.isEmpty(thumbPath)) {
								Options options = ImageUtils.getBitmapOptions(thumbPath);
								int width = options.outWidth;
								int height = options.outHeight;
								body.setThumbnailSize(width, height);
							}
						}
					}

					String oldId = msg.getMsgId();
					//set callback to replace old id
					setMessageSendCallback(msg, conv, oldId, originPath);
					emaObject.sendMessage(msg.emaObject);
				} catch (Exception e) {
					e.printStackTrace();
					new HandleError(EMError.GENERAL_ERROR, "send message failed");
				}
			}
		});
	}

	private void setMessageSendCallback(final EMMessage msg,
										final EMConversation conv,
										final String oldId,
										final String originImagePath) {
		if (msg == null) {
			return;
		}

		msg.setInnerCallback(new EMCallBack(){

			@Override
			public void onSuccess() {
				if (conv != null) {
					conv.getCache().removeMessage(oldId);
					conv.getCache().addMessage(msg);
				}
				if (originImagePath != null){
					if(!EMFileHelper.getInstance().isFileExist(originImagePath)) {
					    return;
					}
					if (msg.getBody() instanceof EMImageMessageBody) {
						String scaleImagePath = ((EMImageMessageBody)msg.getBody()).getLocalUrl();
						EMLog.d(TAG, "origin: + " + originImagePath + ", scale:" + scaleImagePath);
						//if scaleImagePath is not origin image path, should delete scale image file
						if(scaleImagePath != null && !scaleImagePath.equals(originImagePath)){
							boolean isDeleted = EMFileHelper.getInstance().deletePrivateFile(scaleImagePath);
							EMLog.d(TAG, "Deleted the scale image file: "+ isDeleted + " the scale image file path: "+scaleImagePath);
						}
						((EMImageMessageBody)msg.getBody()).setLocalUrl(originImagePath);
						updateMessage(msg);
					}
				}
			}

			@Override
			public void onProgress(int progress, String status) {
			}

			@Override
			public void onError(int code, String error) {
			}
		});
	}

	/**
	 * \~chinese
	 * 发送会话的已读回执，该方法只针对单聊会话。
	 * 
	 * 该方法会通知服务器将此会话未读数设置为 0，对话方（包含多端多设备）将会在下面这个回调方法中接收到回调：
	 *
	 * {@link EMConversationListener#onConversationRead(String, String)} 。
	 * 
	 * 
	 * 参考：
	 * 群消息已读回执，详见 {@link #ackGroupMessageRead(String, String, String)}。
	 * 
	 *
	 * @param conversationId			会话 ID。
	 * @throws HyphenateException		以下为可能抛出的异常：{@link EMError#USER_NOT_LOGIN}、{@link EMError#SERVER_NOT_REACHABLE}、
	 * 									{@link EMError#MESSAGE_INVALID} 等，详见 {@link EMError}。
	 *
	 * \~english
	 * Sends the conversation read receipt to the server. This method is only for one-to-one chat conversation.
	 * 
	 * This method will inform the sever to set the unread message count of the conversation to 0, and conversationist (with multiple devices) will receive
	 * a callback method from {@link EMConversationListener#onConversationRead(String, String)}.
	 *
	 * Reference：
	 * If you want to send group message read receipt to the sever, see {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * @param conversationId			The conversation ID.
	 * @throws HyphenateException	    The possible exceptions are as follows: {@link EMError#USER_NOT_LOGIN}, {@link EMError#SERVER_NOT_REACHABLE},
	 * 									{@link EMError#MESSAGE_INVALID} and so on, see {@link EMError}.
	 */
	public void ackConversationRead(String conversationId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.sendReadAckForConversation(conversationId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 发送消息的已读回执。
	 * 
	 * 该方法仅适用于单聊会话。
	 * 
	 * 注意：如果设置 {@link EMOptions#setRequireAck(boolean)} 为 `false`，当前方法将失效。
	 *
	 * 发送群消息已读回执，详见 {@link #ackGroupMessageRead(String, String, String)}。
	 *
	 * 建议：在进入聊天页面的时调用 {@link #ackConversationRead(String)} ，其他场景再调用此方法，可以大量减少调用此方法的次数。
	 *
	 * @param to					接收方的用户名。
	 * @param messageId				消息的 ID。
	 * @throws HyphenateException 	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Sends the read receipt to the server. 
	 * 
	 * This method applies to one-to-one chats only.
	 * 
	 * **Warning** This method only takes effect if you set {@link EMOptions#setRequireAck(boolean)} as `true`.
	 *
	 * Reference:
	 * To send the group message read receipt, call {@link #ackGroupMessageRead(String, String, String)}.
	 *
	 * We recommend that you call {@link #ackConversationRead(String)} when entering a chat page, and call this method in other cases to reduce the number of method calls.
	 *
	 * @param to			The receiver of the read receipt.
	 * @param messageId		The message ID.
	 * @throws HyphenateException  A description of the exception, see  {@link EMError}.
	 */
	public void ackMessageRead(String to, String messageId) throws HyphenateException {
		EMOptions chatOptions = EMClient.getInstance().getChatConfigPrivate().getOptions();
		if (!chatOptions.getRequireAck()) {
			EMLog.d(TAG, "chat option reqire ack set to false. skip send out ask msg read");
			return;
		}
		if(TextUtils.isEmpty(to)) {
			EMLog.e(TAG, "The parameter of to should not be null");
		    return;
		}
		EMAMessage msg = emaObject.getMessage(messageId);
		if (msg != null) {
			emaObject.sendReadAckForMessage(msg);
		} else { // just for Xinju since there is no local storage
			EMAMessage _msg = EMAMessage.createReceiveMessage("", self(), null, EMMessage.ChatType.Chat.ordinal());
			_msg.setMsgId(messageId);
			_msg.setFrom(to);
			//set conversationId because native code need it
			_msg.setConversationId(to);

			emaObject.sendReadAckForMessage(_msg);
		}
	}

	/**
	 * \~chinese
	 * 发送群消息已读回执。
	 * 
	 * 前提条件：设置了 {@link EMOptions#setRequireAck(boolean)} 和 {@link EMMessage#setIsNeedGroupAck(boolean)}。
	 *
	 * 参考：
	 * 发送单聊消息已读回执，详见 {@link #ackMessageRead(String, String)} ;
	 * 会话已读回执，详见 {@link #ackConversationRead(String)}。
	 *
	 * @param to					会话 ID。
	 * @param messageId				消息 ID。
	 * @param ext					扩展信息。`ext` 属性是用户自己定义的关键字，接收后，解析出自定义的字符串，可以自行处理。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Sends the group message receipt to the server.
	 * 
	 * You can only call the method after setting the following method: {@link EMOptions#setRequireAck(boolean)} and {@link EMMessage#setIsNeedGroupAck(boolean)}.
	 *
	 * Reference:
	 * To send the one-to-one chat message receipt to server, call {@link #ackMessageRead(String, String)}; 
	 * To send the conversation receipt to the server, call {@link #ackConversationRead(String)}.
	 * 
	 * @param to					The conversation ID.
	 * @param messageId				The message ID.
	 * @param ext					The extension information. Developer self-defined command string that can be used for specifying custom action/command.
	 * @throws HyphenateException	A description of the exception, see {@link EMError}.
	 */
	public void ackGroupMessageRead(String to, String messageId, String ext) throws HyphenateException {
		EMOptions chatOptions = EMClient.getInstance().getChatConfigPrivate().getOptions();
		if (!chatOptions.getRequireAck()) {
			EMLog.d(TAG, "chat option reqire ack set to false. skip send out ask msg read");
			return;
		}
		EMAMessage msg = emaObject.getMessage(messageId);
		if (msg != null) {
			if (msg.isNeedGroupAck()) {
				emaObject.sendReadAckForGroupMessage(msg, ext);
			} else {
				EMLog.d(TAG, "normal group message, do not ack it");
			}
		}
	}
	
    /**
     * \~chinese
     * 撤回发送成功的消息。
     *
	 * 同步方法，会阻塞当前线程。
	 *
     * @param message 消息对象。
	 *
	 * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Recalls the sent message.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param message The message instance.
	 *
	 * @throws HyphenateException A description of the exception. See {@link EMError}.
     */
    public void recallMessage(EMMessage message) throws HyphenateException {
        EMAError error = new EMAError();
        if (message == null) {
            throw new HyphenateException(EMError.MESSAGE_INVALID, "The message was not found");
        }
        emaObject.recallMessage(message.emaObject, error);
        handleError(error);
        // 指定是否是Thread消息
		EMConversation conversation = getConversation(message.getTo(), EMConversation.msgType2ConversationType(message.getMsgId(), message.getChatType()), true, message.isChatThreadMessage());
		if(conversation != null) {
		    conversation.getCache().removeMessage(message.getMsgId());
		}
    }

	/**
	 * \~chinese
	 * 撤回发送成功的消息。
	 *
	 * 异步方法。
	 *
	 * @param message 	消息对象。
	 * @param callback	一个 EMCallBack 类型的对象，详见 {@link EMCallBack}。
	 *
	 * \~english
	 * Recalls the sent message.
	 *
	 * This is an asynchronous method.
	 *
	 * @param message	The message object.
	 * @param callback  An instance of the EMCallBack class, see {@link EMCallBack}.
	 */
	public void aysncRecallMessage(final EMMessage message,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    recallMessage(message);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 获取指定 ID 的消息对象。
	 *
	 * @param messageId		消息 ID。
	 * @return  根据指定 ID 获取的消息对象，如果消息不存在会返回空值。
	 *
	 * \~english
	 * Fetches messages by message ID.
	 *
	 * @param messageId 	The message ID.
	 * @return  The message object obtained by the specified ID. Returns null if the message doesn't exist.
	 */
	public EMMessage getMessage(String messageId) {
		synchronized (caches) {
			for(EMConversation.MessageCache cache : caches.values()) {
				EMMessage msg = cache.getMessage(messageId);
				if (msg != null) {
					return msg;
				}
			}
		}
		EMAMessage message =  emaObject.getMessage(messageId);
		if (message == null) {
			return null;
		}
		EMMessage msg = new EMMessage(message);
		return msg;
	}


	/**
	 * \~chinese
	 * 获取定义 ID 的会话对象。
	 * 
	 * 没有找到会返回空值。
	 * 
	 * @param id   会话 ID。
	 * @return     根据指定会话 ID 找到的会话对象，如果没有找到会返回空值。
	 *
	 * \~english
	 * Gets the conversation object by the specified ID.
	 * 
	 * The SDK wil return null if the conversation is not found.
	 *
	 * @param id 	The conversation ID.
	 * @return 		The existing conversation found by conversation ID. Returns null if the conversation is not found.
	 */
	public EMConversation getConversation(String id){
		EMAConversation conversation = emaObject.conversationWithType(id, EMAConversationType.CHAT, false, false);
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.GROUPCHAT, false, false);
		}
		// 判断完群组会话后，再检查Thread会话
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.GROUPCHAT, false, true);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.CHATROOM, false, false);
		}
		// 判断完聊天室会话后，再检查Thread会话
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.CHATROOM, false, true);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.DISCUSSIONGROUP, false, false);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.HELPDESK, false, false);
		}

		return conversation == null ? null : new EMConversation(conversation);
	}

	/**
	 * \~chinese
	 * 根据会话 ID 以及会话类型获取会话。
	 * 
	 * 没有找到返回空值。
	 *
	 * @param id 		会话 ID。
	 * @param type 		会话类型。
	 * @return          根据指定 ID 以及会话类型找到的会话对象，没有找到会返回空值。
	 *
	 * \~english
	 * Gets the conversation by conversation ID and conversation type.
	 * 
	 * The SDK wil return null if the conversation is not found.
	 *
	 * @param id 		The conversation ID.
	 * @param type  	The conversation type, see {@link EMConversationType}
	 * @return 			The conversation object found according to the id and type. Returns null if the conversation is not found.
	 */
	public EMConversation getConversation(String id, EMConversationType type) {
		EMConversation conversation = getConversation(id, type, false);
		// 先检查非Thread会话，再检查Thread会话
		if(conversation == null) {
		    return getConversation(id, type, false, true);
		}
		return conversation;
	}

	/**
	 * \~chinese
	 * 根据用户或群组 ID 以及会话类型获取会话。
	 * 
	 * 没有找到的话，可以根据 createIfNotExists 的值返回一个新建对象或者空对象。
	 *
	 * @param username 			会话 ID。
	 * @param type 				会话类型。
	 * @param createIfNotExists 没找到相应会话时是否自动创建。
	 *                          - `true` 表示没有找到相应会话时会自动创建会话；
	 *                          - `false` 表示没有找到相应会话时不创建会话。
	 * @return					根据指定 ID 以及会话类型找到的会话对象，如果没有找到会返回空值。
	 *
	 * \~english
	 * Get the conversation object by conversation ID and conversation type.
	 * 
	 * If the conversation is not found, you can create a new object based on the value of CreateIFNotExists or an empty object.
	 *
	 * @param username 			The conversation ID.
	 * @param type 				The conversation type, see {@link EMConversationType}.
	 * @param createIfNotExists Whether to create a conversation if not find the specified conversation. 
	 *                          - `true` means create one;
	 *                          -  `false` means not.
	 * @return					The conversation object found according to the ID and type. Returns null if the conversation is not found.
	 */
	public EMConversation getConversation(String username, EMConversationType type, boolean createIfNotExists) {
		EMConversation conversation = getConversation(username, type, createIfNotExists, false);
		// 先检查非Thread会话，再检查Thread会话
		if(conversation == null) {
		    return getConversation(username, type, createIfNotExists, true);
		}
		return conversation;
	}

	/**
	 * \~chinese
	 * 根据用户或群组id以及会话类型获取会话。
	 *
	 * 没有找到的话，根据createIfNotExists的值返回一个新建对象或者空对象。
	 *
	 * @param username 			会话ID。
	 * @param type 				会话类型。
	 * @param createIfNotExists 没找到相应会话时是否自动创建。
	 *                          - `true` 表示没有找到相应会话时会自动创建会话；
	 * 							- `false` 表示没有找到相应会话时不创建会话。
	 * @param isChatThread 			是否查找子区会话。
	 *                          - `true` 表示查找子区会话；
	 * 							- `false` 表示不查找子区会话。
	 * @return					根据指定 ID 以及会话类型找到的会话对象，如果没有找到会返回空值。
	 *{@link EMChatManager#getConversation(String, EMConversationType, boolean, boolean)}
	 * \~english
	 * Get conversation object by conversation id and conversation type.
	 *
	 * If the conversation is not found, you can create a new object based on the value of CreateIFNotExists or an empty object.
	 *
	 * @param username 			The conversation ID.
	 * @param type 				The conversation type, see {@link EMConversationType}.
	 * @param createIfNotExists Whether to create a conversation if not find the specified conversation.
	 *                          - `true` means create one;
	 *                          -  `false` means not.
	 * @param isChatThread 			Whether to search chat thread conversation.
	 *                          - `true` means search chat thread conversation；
	 * 							- `false` means not.
	 * @return					The conversation object found according to the ID and type. Returns null if the conversation is not found.
	 */
	public EMConversation getConversation(String username, EMConversationType type, boolean createIfNotExists, boolean isChatThread) {
		EMAConversationType t = EMAConversationType.CHAT;
		switch (type) {
			case Chat:
				t = EMAConversationType.CHAT;
		        break;
			case GroupChat:
				t = EMAConversationType.GROUPCHAT;
		        break;
			case ChatRoom:
				t = EMAConversationType.CHATROOM;
		        break;
			case DiscussionGroup:
				t = EMAConversationType.DISCUSSIONGROUP;
		        break;
			case HelpDesk:
				t = EMAConversationType.HELPDESK;
		        break;
		}
		EMAConversation conversation = emaObject.conversationWithType(username, t, createIfNotExists, isChatThread);
		if (conversation == null) {
			return null;
		}
		Log.d(TAG, "convID:" + conversation.conversationId());
		return new EMConversation(conversation);
	}


	/**
	 * \~chinese
	 * 把所有的会话都设成已读。
	 * 
	 * 这里针对的是本地会话。
	 *
	 * \~english
	 * Marks all messages as read.
	 * 
	 * This method is for the local conversations only.
	 */
	public void markAllConversationsAsRead() {
		List<EMAConversation> conversations = emaObject.loadAllConversationsFromDB();
		for (EMAConversation conversation : conversations) {
			conversation.markAllMessagesAsRead(true);
		}
	}

	/**
	 * \~chinese
	 * 获取未读消息数。
	 * @return		未读消息数。
	 *
	 * @deprecated  使用 {@link EMChatManager#getUnreadMessageCount()} 替代。
	 *
	 * \~english
	 * Gets the unread message count.
	 * @return		The count of unread messages.
	 *
	 * @deprecated  Use {@link EMChatManager#getUnreadMessageCount()} instead.
	 */
	@Deprecated
	public int getUnreadMsgsCount() {
		return getUnreadMessageCount();
	}

	/**
	 * \~chinese
	 * 获取未读消息数。
	 * @return	未读消息数。
	 *
	 * \~english
	 * Gets the unread message count.
	 * @return	The count of unread messages.
	 */
	public int getUnreadMessageCount()
	{
		List<EMAConversation> conversations = emaObject.getConversations();
		int unreadCount = 0;
		for (EMAConversation conversation : conversations) {
			if (conversation._getType() != EMAConversationType.CHATROOM) {
				unreadCount += conversation.unreadMessagesCount();
			}
		}
		return unreadCount;
	}

	/**
	 * \~chinese
	 * 保存消息到本地。
	 * 比如系统提示消息会存到内存中的会话和数据库。
	 * 
	 * 注意：
	 * CMD 类型数据不保存在本地。
	 *
	 * @param message	待存储的消息。
	 *
	 * \~english
	 * Saves the message to the memory and local database.
	 * 
	 * Notes:
	 * The CMD message will not be stored in the database.
	 *
	 * @param message	The message to store.
	 */
	public void saveMessage(EMMessage message) {
		EMMessage.ChatType type = message.getChatType();
		EMConversationType t = EMConversationType.Chat;
		switch (type) {
			case Chat:
				t = EMConversationType.Chat;
				break;
			case GroupChat:
				t = EMConversationType.GroupChat;
				break;
			case ChatRoom:
				t = EMConversationType.ChatRoom;
				break;
		}
		String convId = message.getTo();
		//for group, chatroom, conversation id is group id for both receive and send message
		if (t == EMConversationType.Chat && message.direct() == EMMessage.Direct.RECEIVE) {
			convId = message.getFrom();
		}
		if (message.getType() == Type.CMD) {
			return;
		}
		EMConversation conv = getConversation(convId, t, true, message.isChatThreadMessage());
		if(conv == null) {
			EMLog.e(TAG, "Failed to save message because conversation is null, convId: "+convId);
		    return;
		}
		// when send message out, appendMessage will update time to lastMsgTime + 1ms
		conv.insertMessage(message);
	}

	/**
	 * \~chinese
	 * 更新本地消息。
	 * 
	 * 会更新本地内存和数据库。
	 *
	 * @param message 要更新的消息对象。
	 *
	 * \~english
	 * Updates the local message.
	 * 
	 * Will update the memory and the local database at the same time.
	 *
	 * @param message The message objects to update.
	 */
	public boolean updateMessage(EMMessage message) {
		String id = message.direct() == EMMessage.Direct.RECEIVE ? message.getFrom() : message.getTo();
		if (message.getType() == Type.CMD) {
			return false;
		}
		EMConversation conv = getConversation(message.conversationId(), EMConversation.msgType2ConversationType(id, message.getChatType()), true, message.isChatThreadMessage());
		return conv.updateMessage(message);
	}

	/**
	 * \~chinese
	 * 下载消息的附件。
	 * 
	 * 未成功下载的附件，可调用此方法再次下载。
	 *
	 * @param msg	要下载附件的消息。
	 *
	 * \~english
	 * Downloads the message attachment.
	 * 
	 * You can call the method again if the attachment download fails.
	 *
	 * @param msg 	The message to be download the attachment.
	 */
	public void downloadAttachment(final EMMessage msg) {
        if (msg == null) {
            return;
        }
        msg.makeCallbackStrong();
		checkContentAttachmentExist(msg);
		emaObject.downloadMessageAttachments(msg.emaObject);
	}

	/**
	 * \~chinese
	 * 检查附件是否存在。
	 * @param msg  消息对象。
	 * 
	 * \~chinese
	 * Check whether the attachment exists.
	 * 
	 * @param msg  The message object.
	 */
	private void checkContentAttachmentExist(EMMessage msg) {
		List<EMAMessageBody> bodies = msg.emaObject.bodies();
		if(bodies != null && !bodies.isEmpty()) {
		    for (EMAMessageBody body: bodies) {
		    	switch (body.type()) {
		    	    case EMAMessageBody.EMAMessageBodyType_IMAGE :
		    	    case EMAMessageBody.EMAMessageBodyType_VIDEO :
		    	    case EMAMessageBody.EMAMessageBodyType_VOICE :
		    	    case EMAMessageBody.EMAMessageBodyType_FILE :
						EMAFileMessageBody fileBody = (EMAFileMessageBody) body;
						String localUrl = fileBody.getLocalUrl();
						EMLog.d(TAG, "download before check path = "+localUrl);
						if(!EMFileHelper.getInstance().isFileExist(localUrl)) {
							String filename = fileBody.displayName();
							String newLocalPath = null;
							switch (body.type()) {
							    case EMAMessageBody.EMAMessageBodyType_IMAGE :
									newLocalPath = PathUtil.getInstance().getImagePath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_VIDEO :
									newLocalPath = PathUtil.getInstance().getVideoPath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_VOICE :
									newLocalPath = PathUtil.getInstance().getVoicePath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_FILE :
									newLocalPath = PathUtil.getInstance().getFilePath()+File.separator+filename;
							        break;
							}
							if(!TextUtils.isEmpty(newLocalPath)) {
							    fileBody.setLocalPath(newLocalPath);
							    updateMessage(msg);
								EMLog.d(TAG, "download:create new path , path is "+newLocalPath);
							}
						}
						break;
		    	}
			}
		}
	}

	/**
	 * \~chinese
	 * 下载消息的缩略图。
	 *
	 * @param msg	要下载缩略图的消息，一般图片消息和视频消息有缩略图。
	 *
	 * \~english
	 * Downloads the thumbnail if the msg is not downloaded before or the download failed.
	 *
	 * @param msg  	The message object.
	 */
	public void downloadThumbnail(final EMMessage msg) {
        msg.makeCallbackStrong();
		emaObject.downloadMessageThumbnail(msg.emaObject);
	}

	/**
	 * \~chinese
	 * 向消息数据库导入多条聊天记录。
	 * 在调用此函数时要保证，消息的发送方或者接收方是当前用户。
	 * 已经对函数做过速度优化，推荐一次导入 1,000 条以内的数据。
	 *
	 * @param msgs 需要导入数据库的消息。
	 *
	 * \~english
	 * Imports messages to the local database.
	 * 
	 * Make sure the message sender or receiver is the current user before option.
	 * 
	 * Recommends import less than 1,000 messages per operation.
	 *
	 * @param msgs The message list.
	 */
	public synchronized void importMessages(List<EMMessage> msgs) {
		List<EMAMessage> dummy = new ArrayList<EMAMessage>();
		for (EMMessage msg : msgs) {
			dummy.add(msg.emaObject);
		}
		EMClient.getInstance().getChatConfigPrivate().importMessages(dummy);
	}

	/**
	 * \~chinese
	 * 获取指定会话类型所有的会话。
	 *
	 * @param type 	会话类型，详见 {@link EMConversationType}。
	 * @return 		返回指定会话类型的会话列表。
	 *
	 * \~english
	 * Gets the list of conversations by conversation type.
	 *
	 * @param type	The conversation type, see {@link EMConversationType}.
	 * @return 		The list of conversation in specified type.
	 */
	public List<EMConversation> getConversationsByType(EMConversationType type) {
		List<EMAConversation> conversations = emaObject.getConversations();
		List<EMConversation> result = new ArrayList<EMConversation>();
		for (EMAConversation conv : conversations) {
			if (type.ordinal() == conv._getType().ordinal()) {
				result.add(new EMConversation(conv));
			}
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器下载文件。
	 *
	 * @param remoteUrl 	服务器上的远程文件路径。
	 * @param localFilePath 本地要生成的文件路径。
	 * @param headers 		请求头。
	 * @param callback 		下载结果回调，详见 {@link EMCallBack}。
	 *
	 * @deprecated 使用 {@link #downloadAttachment(EMMessage)} 代替。
	 *
	 * \~english
	 * Downloads the attachment files from the server.
	 *
	 * @param remoteUrl 		The remote file url.
	 * @param localFilePath 	The local file path.
	 * @param headers 			The Http Request Headers.
	 * @param callback 			The download status callback, see {@link EMCallBack}.
	 *
	 * @deprecated Use {@link #downloadAttachment(EMMessage)} instead.
	 */
	@Deprecated
	public void downloadFile(final String remoteUrl,
							 final String localFilePath,
							 final Map<String, String> headers,
							 final EMCallBack callback) {
		/*
		if (!headers.containsKey("Authorization")) {
			headers.put("Authorization", EMChatConfig.getInstance().getAccessToken());
		}
		int responseCode = EMARHttpAPI.download(remoteUrl, localFilePath, headers);

		if (200 <= responseCode && responseCode <= 299) {
			if (callback != null) {
				callback.onSuccess();
			}
		} else {
			callback.onError(responseCode, "");
		}
		*/

		EMHttpClient.getInstance().downloadFile(remoteUrl, localFilePath, headers, new EMCloudOperationCallback(){

			@Override
			public void onSuccess(String result) {
				if(callback != null){
					callback.onSuccess();
				}
			}

			@Override
			public void onError(String msg) {
				if(callback != null){
					callback.onError(EMError.GENERAL_ERROR, msg);
				}
			}

			@Override
			public void onProgress(int progress) {
				if(callback != null){
					callback.onProgress(progress, null);
				}
			}

		});
	}

	/**
	 * \~chinese
	 * 从本地获取当前所有的会话。
	 * 
	 * 先从内存中加载，如果内存中没有再从数据库中加载。
	 *
	 * @return 返回本地内存或者数据库中所有的会话。
	 *
	 * \~english
	 * Gets all conversations from the local database.
	 * 
	 * Conversations will be loaded from memory first, if no conversation is found then the SDk loads from the local database.
	 *
	 * @return Returns all the conversations from the memory or local database.
	 */
	public Map<String, EMConversation> getAllConversations() {
		List<EMAConversation> conversations = emaObject.getConversations();
		Hashtable<String, EMConversation> result = new Hashtable<String, EMConversation>();
		for (EMAConversation conversation : conversations) {
			/*if (conversation._getType() != EMAConversationType.CHATROOM) {
				result.put(conversation.conversationId(), new EMConversation(conversation));
			}*/
			result.put(conversation.conversationId(), new EMConversation(conversation));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取会话列表。
	 * 该功能需联系商务开通，开通后，用户默认可拉取 7 天内的 10 个会话（每个会话包含最新一条历史消息），如需调整会话数量或时间限制请联系商务经理。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @return 返回当前用户的会话列表。
	 *
	 * \~english
	 * Fetches the conversation list from the server.
	 * The default maximum return is 100.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @return    Returns the conversation list of the current user.
	 */
	public Map<String, EMConversation> fetchConversationsFromServer() throws HyphenateException {
		EMAError error = new EMAError();

		List<EMAConversation> conversations = emaObject.fetchConversationsFromServer(error);
		//emaObject.fetchConversationsFromServer(error);
		handleError(error);

		Hashtable<String, EMConversation> result = new Hashtable<String, EMConversation>();
		for (EMAConversation conversation : conversations) {
			result.put(conversation.conversationId(), new EMConversation(conversation));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取会话列表。
	 * 该功能需联系商务开通，开通后，用户默认可拉取 7 天内的 10 个会话（每个会话包含最新一条历史消息），如需调整会话数量或时间限制请联系商务经理。
	 *
	 * 异步方法。
	 *
	 * @return	返回当前用户的会话列表。
	 *
	 * \~english
	 * Fetches conversations from the server.
	 * The default maximum return is 100.
	 *
	 * This is an asynchronous method.
	 *
	 * @return	Returns the conversation list of the current log-in user.
	 */
	public void asyncFetchConversationsFromServer(final EMValueCallBack<Map<String, EMConversation>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchConversationsFromServer());
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从数据库加载本地所有的会话到内存中。
	 * 一般在登录成功后使用此方法，可以加快会话列表的加载速度。
	 *
	 * \~english
	 * Loads all conversations from the local database into memory.
	 * Generally used after a successful login to speed up the loading of the conversation list.
	 */
	public void loadAllConversations() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~chinese
	 * 删除指定 ID 的对话和本地的聊天记录。
	 * 如果将 `deleteMessages` 设为 `true`，删除会话的同时也会删除本地的聊天记录。
	 *
	 * @param username 			会话 ID。
	 * @param deleteMessages	是否同时删除本地的聊天记录。`
	 *                          - true` 表示删除；
	 *                          - `false` 表示不删除。
	 * @return 					删除会话结果。
	 *                          - `true` 代表删除成功；
	 *                          - `false` 代表删除失败。
	 *
	 * \~english
	 * Deletes conversation and messages from the local database.
	 * 
	 * If you set `deleteMessages` to `true`, delete the local chat history when delete the conversation.
	 *
	 * @param username 			The conversation ID.
	 * @param deleteMessages 	Whether to delete the chat history when delete the conversation.
	 *                          - `True` means to delete the chat history when delete the conversation;
	 *                          - `false` means not.
	 * @return 					The result of deleting.
	 *                          - `True` means success;
	 *                          - `false` means failure.
	 */
	public boolean deleteConversation(String username, boolean deleteMessages) {
		EMConversation conv = getConversation(username);
		if (conv == null) {
			return false;
		}
		if (!deleteMessages) {
			conv.clear();
		} else {
			conv.clearAllMessages();
		}
		emaObject.removeConversation(username, deleteMessages, conv.isChatThread());
		return true;
	}

	/**
	 * \~chinese
	 * 删除服务端的指定 ID 的对话和聊天记录，内部异步操作。
	 *
	 * @param username 			会话 ID。
	 * @param type              会话类型 {@link EMConversationType}
	 * @param isDeleteServerMessages	是否同时删除服务端的聊天记录。
	 *                                  - `true`：删除服务端会话的同时也会删除服务端的聊天记录。
	 *                                  - `false` 表示删除服务端会话的同时不删除服务端的聊天记录。
	 * @param deleteCallBack  删除服务端会话与记录成功与否的回调。
	 *                        - `true` 代表删除成功；
	 *                        - `false` 代表删除失败。
	 *
	 * \~english
	 * Deletes the conversation of a specified ID and it's chat records on the server.
	 * This operation is performed asynchronously.
	 *
	 * @param username 			Conversation ID.
	 * @param type              conversation type  {@link EMConversationType}
	 * @param isDeleteServerMessages 	Whether to delete the server chat records when delete conversation.
	 *                                  - `True` means to delete the chat history when delete the conversation;
	 *                                  - `false` means not.
	 * @param deleteCallBack  The callback to delete the server session and record successfully.
	 *                          - `True` means success;
	 *                          - `false` means failure.
	 */
	public void deleteConversationFromServer(String username,EMConversationType type, boolean isDeleteServerMessages,EMCallBack deleteCallBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				EMAConversationType t = EMAConversationType.CHAT;
				switch (type) {
					case Chat:
						t = EMAConversationType.CHAT;
				        break;
					case GroupChat:
						t = EMAConversationType.GROUPCHAT;
				        break;
					case ChatRoom:
						t = EMAConversationType.CHATROOM;
				        break;
					case DiscussionGroup:
						t = EMAConversationType.DISCUSSIONGROUP;
				        break;
					case HelpDesk:
						t = EMAConversationType.HELPDESK;
				        break;
				}
				EMAError error=emaObject.deleteConversationFromServer(username,t, isDeleteServerMessages);
				if(error.errCode()==EMAError.EM_NO_ERROR) {
					deleteCallBack.onSuccess();
				}else{
					deleteCallBack.onError(error.errCode(), error.errMsg());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 注册消息监听。
	 *
	 * 接受到新消息等回调可以通过设置此方法进行监听，详见 {@link EMMessageListener}。
	 *
	 * 如果需要移除监听，可以和 {@link #removeMessageListener(EMMessageListener)} 方法配合使用。
	 *
	 * @param listener 要注册的消息监听，详见 {@link EMMessageListener}。
	 *
	 * \~english
	 * Adds the message listener.
	 * 
	 * Receives new messages and so on can set the method to listen, see {@link EMMessageListener}.
	 *
	 * @param listener The message listener which is used to listen the incoming messages, see {@link EMMessageListener}
	 */
	public void addMessageListener(EMMessageListener listener) {
		if(listener == null){
			EMLog.d(TAG, "addMessageListener: listener is null");
			return;
		}

		if(!messageListeners.contains(listener)){
			EMLog.d(TAG, "add message listener: " + listener);
			messageListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除消息监听。
	 * 
	 * 需要先添加 {@link #addMessageListener(EMMessageListener)} 监听，再调用本方法。
	 *
	 * @param listener 要移除的监听，详见 {@link EMMessageListener}。
	 *
	 * \~english
	 * Removes the message listener.
	 * 
	 * You should call this method after set {@link #addMessageListener(EMMessageListener)} .
	 * 
	 * @param listener The message listener to be removed.
	 */
	public void removeMessageListener(EMMessageListener listener) {
		if(listener == null){
			return;
		}

		messageListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 注册会话监听。
	 * 
	 * 用于监听会话变化及监听会话已读回执，详见 {@link EMConversationListener}。
	 * 
	 * 需要配合 {@link #removeConversationListener(EMConversationListener)} 使用。
	 *
	 * @param listener 要添加的会话监听，详见 {@link EMConversationListener}。
	 *
	 * \~english
	 * Adds the conversation listener. 
	 * 
	 * The method is used to listen the change of conversation and listen the receipt of the conversation message, see {@link EMConversationListener}.
	 * 
	 * After set this method, if you want to remove the conversation listener, you can call the method of {@link #removeConversationListener(EMConversationListener)} to remove it.
	 *
	 * @param listener The conversation listener to be set, see {@link EMConversationListener}.
	 */
	public void addConversationListener(EMConversationListener listener){
		if(!conversationListeners.contains(listener)){
			conversationListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除会话监听。
	 * 
	 * 如果想要移除会话监听，需要先添加 {@link #addConversationListener(EMConversationListener)} 后调用此方法进行移除。
	 *
	 * @param listener  准备移除的会话监听，详见 {@link EMConversationListener}。
	 *
	 * \~english
	 * Removes the conversation listener.
	 * 
	 * Make sure to use the {@link #addConversationListener(EMConversationListener)} method first if you want to delete the conversation listener.
	 *
	 * @param listener The conversation listener to be removed.
	 */
	public void removeConversationListener(EMConversationListener listener){
		if(listener == null){
			return;
		}

		conversationListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 将消息设置为已听。
	 * 一般用于语音消息。
	 *
	 * @param message 要设置的消息对象。
	 * @deprecated  请使用 {@link EMChatManager#setVoiceMessageListened(EMMessage)} 替代。
	 *
	 * \~english
	 * Sets the message status as listened.
	 * Usually used for voice messages.
	 *
	 * @param message The message object.
	 * @deprecated  Use {@link EMChatManager#setVoiceMessageListened(EMMessage)} instead.
	 */
	@Deprecated
	public void setMessageListened(EMMessage message) {
		setVoiceMessageListened(message);
	}

	/**
	 * \~chinese
	 * 将消息设置为已听。
	 * 一般用于语音消息。
	 *
	 * @param message 要设置的消息对象。
	 *
	 * \~english
	 * Sets the message status as listened.
	 * Usually used for voice messages.
	 *
	 * @param message The message object.
	 */
	public void setVoiceMessageListened(EMMessage message)
	{
		message.setListened(true);
		updateMessage(message);
	}

	void onLogout() {
		caches.clear();
	}

	synchronized void  loadAllConversationsFromDB() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~chinese
	 * 将数据库中的某个联系人相关信息变更成另外一个联系人。
	 * 
	 * 与变更相关的表单包含消息表单，会话表单，联系人表单，黑名单表单。
	 * 
	 * 注意：该操作不会更新内存中数据。
	 *
	 * @param from		更换前的用户 ID。
	 * @param changeTo	更换后的用户 ID。
	 * @return 			返回更新结果，任何表单更新失败，都会返回 `false`。
	 *
	 * \~english
	 * Updates the database records, changes username from 'from' to 'changeTo'.
	 * The method will effect on other counts, such as the message table, the conversation table, contacts, the blocklist table.
	 * 
	 * Note: this operation does not update the data stored in memory cache.
	 *
	 * @param from		The user ID before replace.
	 * @param changeTo  The user ID after replace.
	 * @return 			The operation result. If any update operations on those tables failed, the SDK will return `false`.
	 */
	public boolean updateParticipant(String from, String changeTo) {
		return emaObject.updateParticipant(from, changeTo);
	}

	/**
	 * \~chinese
	 * 从服务器获取群组消息回执详情。
	 * 
	 * 分页获取。
	 * 
	 * 参考：
	 * 发送群组消息回执，详见 {@link #ackGroupMessageRead(String, String, String)}。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param msgId 				需要获取回执的消息 ID。
	 * @param pageSize 				每次获取群消息已读回执的条数。
	 * @param startAckId 			已读回执的 ID，可以为空。如果为空，从最新的回执向前开始获取。
	 * @return 						返回消息列表和用于继续获取群消息回执的 Cursor。
	 * @throws HyphenateException 	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Fetches the receipt details for group messages from the server.
	 * 
	 * Fetches by page.
	 * 
	 * Reference:
	 * If you want to send group message read receipt, see {@link #ackConversationRead(String)}
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param msgId 				The message ID which needs to fetch receipt.
	 * @param pageSize 				The number of records per page.
	 * @param startAckId 			The start ID for fetching receipts, can be null. If you set it as null, the fetching will start from the server‘s latest receipt.
	 * @return 						The list of returned ack from the server and the cursor for next fetch action.
	 * @throws HyphenateException	A description of the exception, see {@link EMError}.
	 */
	public EMCursorResult<EMGroupReadAck> fetchGroupReadAcks(String msgId, int pageSize, String startAckId) throws HyphenateException {
		EMAError error = new EMAError();
        EMCursorResult<EMGroupReadAck> cusorResult = new EMCursorResult<>();

		EMMessage msg = getMessage(msgId);
		String groupId = null;
		if (msg.getChatType() == EMMessage.ChatType.GroupChat && msg.isNeedGroupAck()) {
			groupId = msg.conversationId();
		} else {
			EMLog.e(TAG, "not group msg or don't need ack");
			return cusorResult;
		}

		EMCursorResult<EMAGroupReadAck> _cusorResult = emaObject.fetchGroupReadAcks(msgId, groupId, error, pageSize, startAckId);
		handleError(error);
		cusorResult.setCursor(_cusorResult.getCursor());

		List<EMGroupReadAck> groupReadAcks = new ArrayList<>();

		for(EMAGroupReadAck _ack: _cusorResult.getData()) {
			groupReadAcks.add(new EMGroupReadAck(_ack));
		}
		cusorResult.setData(groupReadAcks);
		return cusorResult;
	}

	/**
	 * \~chinese
	 * 从服务器获取群组消息回执详情。
	 * 
	 * 分页获取。
	 * 
	 * 参考：
	 * 发送群组消息回执，详见 {@link #ackGroupMessageRead(String, String, String)}。
	 *
	 * 异步方法。
	 *
	 * @param msgId 		消息 ID。
	 * @param pageSize 		每页获取群消息已读回执的条数。
	 * @param startAckId 	已读回执的 ID，如果为空，从最新的回执向前开始获取。
	 * @param callBack    	结果回调，成功执行 {@link EMValueCallBack#onSuccess(Object)}，失败执行 {@link EMValueCallBack#onError(int, String)}。
	 *
	 * \~english
	 * Fetches the ack details for group messages from server.
	 * 
	 * Fetches by page.
	 * 
	 * Reference:
	 * If you want to send group message receipt, see {@link #ackConversationRead(String)}.
	 *
	 * This is an asynchronous method.
	 *
	 * @param msgId 		The message ID.
	 * @param pageSize 		The number of records per page.
	 * @param startAckId 	The start ID for fetch receipts, can be null. If you set it as null, the SDK will start from the server's latest receipt.
	 * @param callBack 		The result callback, if successful, the SDK will execute the method {@link EMValueCallBack#onSuccess(Object)},
	 *                      if the call failed, the SDK will execute the method {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchGroupReadAcks(final String msgId, final int pageSize,
										final String startAckId, final EMValueCallBack<EMCursorResult<EMGroupReadAck>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupReadAcks(msgId, pageSize, startAckId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取历史消息。
	 * 
	 * 分页获取。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param conversationId 		会话 ID。
	 * @param type 					会话类型, 详见 {@link EMConversationType}。
	 * @param pageSize 				每页获取的条数。
	 * @param startMsgId 			漫游消息的开始消息 ID，如果为空，从最新的消息向前开始获取。
	 * @return 						返回消息列表和用于继续获取历史消息的 Cursor。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Fetches history messages of the conversation from the server.
	 * 
	 * Fetches by page.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param conversationId 		The conversation ID.
	 * @param type 					The conversation type which select to fetch roam message, see {@link EMConversationType}
	 * @param pageSize 				The number of records per page.
	 * @param startMsgId 			The start search roam message, if the param is empty, fetches from the server latest message.
	 * @return 					    Returns the messages and the cursor for next fetch action.
	 * @throws HyphenateException	A description of the exception, see {@link EMError}.
	 */
	public EMCursorResult<EMMessage> fetchHistoryMessages(String conversationId, EMConversationType type, int pageSize,
	                                 String startMsgId) throws HyphenateException {
		return fetchHistoryMessages(conversationId, type, pageSize, startMsgId, EMConversation.EMSearchDirection.UP);
	}

	/**
	 * \~chinese
	 * 从服务器获取历史消息。
	 *
	 * 分页获取。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param conversationId 		会话 ID。
	 * @param type 					会话类型, 详见 {@link EMConversationType}。
	 * @param pageSize 				每页获取的条数。
	 * @param startMsgId 			漫游消息的开始消息 ID，如果为空，根据 direction , 当检索方向为 {@link com.hyphenate.chat.EMConversation.EMSearchDirection#UP} 时，
	 *                              从最新的消息向前开始获取；当检索方向为 {@link com.hyphenate.chat.EMConversation.EMSearchDirection#DOWN} 时，
	 *                              从最旧的消息向后开始获取。
	 * @param direction				漫游消息检索方向，{@link com.hyphenate.chat.EMConversation.EMSearchDirection#UP}为检索
	 *                              时间戳消息小于开始消息时间戳的消息，{@link com.hyphenate.chat.EMConversation.EMSearchDirection#DOWN}
	 *                              为检索时间戳消息大于开始消息时间戳的消息。
	 * @return 						返回消息列表和用于继续获取历史消息的 Cursor。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Fetches history messages of the conversation from the server.
	 *
	 * Fetches by page.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param conversationId 		The conversation ID.
	 * @param type 					The conversation type which select to fetch roam message, see {@link EMConversationType}
	 * @param pageSize 				The number of records per page.
	 * @param startMsgId 			The start search roam message, if the param is empty, direction is {@link com.hyphenate.chat.EMConversation.EMSearchDirection#UP} ,
	 *                              fetches from the server latest message,  direction is {@link com.hyphenate.chat.EMConversation.EMSearchDirection#DOWN} ,
	 *                              fetches from the server oldest message.
	 * @param direction             Roaming message retrieval direction, {@link com.hyphenate.chat.EMConversation.EMSearchDirection#UP}
	 *                              is the retrieval time stamp message less than the start message time stamp,
	 * 								{@link com.hyphenate.chat.EMConversation.EMSearchDirection#DOWN} is a message whose retrieval timestamp
	 * 								is greater than the timestamp of the start message.
	 * @return 					    Returns the messages and the cursor for next fetch action.
	 * @throws HyphenateException	A description of the exception, see {@link EMError}.
	 */
	public EMCursorResult<EMMessage> fetchHistoryMessages(String conversationId,
														  EMConversationType type,
														  int pageSize,
														  String startMsgId,
														  EMConversation.EMSearchDirection direction) throws HyphenateException {
		EMAError error = new EMAError();
		EMAConversation.EMASearchDirection d = direction == EMConversation.EMSearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;
		EMCursorResult<EMAMessage> _cursorResult = emaObject.fetchHistoryMessages(conversationId,
				type.ordinal(), pageSize, startMsgId, d, error);
		handleError(error);
		EMCursorResult<EMMessage> cursorResult = new EMCursorResult<>();
		cursorResult.setCursor(_cursorResult.getCursor());
		List<EMMessage> msgs = new ArrayList<>();
		for (EMAMessage _msg : _cursorResult.getData()) {
			msgs.add(new EMMessage(_msg));
		}
		cursorResult.setData(msgs);
		return cursorResult;
	}

	/**
	 * \~chinese
	 * 从服务器获取历史消息。
	 * 
	 * 分页获取。
	 *
	 * 异步方法。
	 *
	 * @param conversationId 	会话 ID。
	 * @param type 				会话类型，详见 {@link EMConversationType}。
	 * @param pageSize 			每页获取的条数。
	 * @param startMsgId 		漫游消息的开始消息 ID，如果为空，从最新的消息向前开始获取。
	 * @param callBack 			结果回调，成功则执行 {@link EMValueCallBack#onSuccess(Object)},
	 *                          返回消息列表和用于继续获取历史消息的 Cursor; 失败则执行 {@link EMValueCallBack#onError(int, String)}。
	 *
	 *
	 * \~english
	 * Fetches the message history of the conversation from the server.
	 * 
	 * Fetches by page.
	 *
	 * This is an asynchronous method.
	 *
	 * @param conversationId 	The conversation id which select to fetch roam message.
	 * @param type 				The conversation type which select to fetch roam message.
	 * @param pageSize 			The number of records per page.
	 * @param startMsgId 		The start search roam message, if empty start from the server latest message.
	 * @param callBack 			The result callback, if successful, the SDK will execute the method {@link EMValueCallBack#onSuccess(Object)},
	 *                          which returns messages from server and cursor for next fetch action; if the call failed, the SDK will execute the method {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchHistoryMessage(final String conversationId, final EMConversationType type, final int pageSize,
	                                final String startMsgId, final EMValueCallBack<EMCursorResult<EMMessage>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchHistoryMessages(conversationId, type, pageSize, startMsgId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取历史消息。
	 *
	 * 分页获取。
	 *
	 * 异步方法。
	 *
	 * @param conversationId 	会话 ID。
	 * @param type 				会话类型，详见 {@link EMConversationType}。
	 * @param pageSize 			每页获取的条数。
	 * @param startMsgId 		漫游消息的开始消息 ID，如果为空，根据 direction , 当检索方向为 {@link com.hyphenate.chat.EMConversation.EMSearchDirection#UP} 时，
	 *                          从最新的消息向前开始获取；当检索方向为 {@link com.hyphenate.chat.EMConversation.EMSearchDirection#DOWN} 时，
	 *                          从最旧的消息向后开始获取。
	 * @param direction         漫游消息检索方向，{@link com.hyphenate.chat.EMConversation.EMSearchDirection#UP}为检索
	 * 							时间戳消息小于开始消息时间戳的消息，{@link com.hyphenate.chat.EMConversation.EMSearchDirection#DOWN}
	 * 							为检索时间戳消息大于开始消息时间戳的消息。
	 * @param callBack 			结果回调，成功则执行 {@link EMValueCallBack#onSuccess(Object)},
	 *                          返回消息列表和用于继续获取历史消息的 Cursor; 失败则执行 {@link EMValueCallBack#onError(int, String)}。
	 *
	 *
	 * \~english
	 * Fetches the message history of the conversation from the server.
	 *
	 * Fetches by page.
	 *
	 * This is an asynchronous method.
	 *
	 * @param conversationId 	The conversation id which select to fetch roam message.
	 * @param type 				The conversation type which select to fetch roam message.
	 * @param pageSize 			The number of records per page.
	 * @param startMsgId        The start search roam message, if the param is empty, direction is {@link com.hyphenate.chat.EMConversation.EMSearchDirection#UP} ,
	 *                          fetches from the server latest message,  direction is {@link com.hyphenate.chat.EMConversation.EMSearchDirection#DOWN} ,
	 *                          fetches from the server oldest message.
	 * @param direction         Roaming message retrieval direction, {@link com.hyphenate.chat.EMConversation.EMSearchDirection#UP}
	 * 							is the retrieval time stamp message less than the start message time stamp,
	 * 							{@link com.hyphenate.chat.EMConversation.EMSearchDirection#DOWN} is a message whose retrieval timestamp
	 * 							is greater than the timestamp of the start message.
	 * @param callBack 			The result callback, if successful, the SDK will execute the method {@link EMValueCallBack#onSuccess(Object)},
	 *                          which returns messages from server and cursor for next fetch action; if the call failed, the SDK will execute the method {@link EMValueCallBack#onError(int, String)}.
	 */
	public void asyncFetchHistoryMessage(final String conversationId,
										 final EMConversationType type,
										 final int pageSize,
										 final String startMsgId,
										 final EMConversation.EMSearchDirection direction,
										 final EMValueCallBack<EMCursorResult<EMMessage>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchHistoryMessages(conversationId, type, pageSize, startMsgId, direction));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	private void handleError(EMAError error) throws HyphenateException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new HyphenateException(error);
		}
	}

	private static final int LIST_SIZE = 512;

	/**
	 * \~chinese
	 * 根据传入的参数从本地存储中搜索指定数量的消息。
	 * 
	 * 注意：当 maxCount 非常大时，需要考虑内存消耗，目前限制一次最多搜索 400 条数据。
	 *
	 * @param type       消息类型，文本、图片、语音、视频等，详见 {@link EMMessage.Type}。
	 * @param timeStamp  搜索消息的时间点，Unix 时间戳。
	 * @param maxCount   搜索结果的最大条数。
	 * @param from       搜索来自某人或者某群的消息，一般是指会话 ID。
	 * @return           消息列表。
	 *
	 * \~english
	 * Searches messages from the local database according the parameters.
	 * 
	 * Note: Be cautious about the memory usage when the maxCount is large, currently the limited number is 400 entries at a time.
	 *
	 * @param type       The message type, TXT、VOICE、IMAGE、VIDEO and so on, see {@link EMMessage.Type}.
	 * @param timeStamp  The Unix timestamp when searching, in milliseconds.
	 * @param maxCount   The max number of message to search.
	 * @param from       A user ID or a group ID searches for messages, usually refers to the conversation ID.
	 * @return           The list of messages.
	 */
	public List<EMMessage> searchMsgFromDB(EMMessage.Type type, long timeStamp, int maxCount, String from, EMConversation.EMSearchDirection direction) {
		EMAConversation.EMASearchDirection d = direction == EMConversation.EMSearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;

		List<EMAMessage> msgs = emaObject.searchMessages(type.ordinal(), timeStamp, maxCount, from, d);
		// to avoid resizing issue of array list, used linked list when size > 512
		List<EMMessage> result;
		if (msgs.size() > LIST_SIZE) {
			result = new LinkedList<EMMessage>();
		} else {
			result = new ArrayList<EMMessage>();
		}
		for (EMAMessage msg : msgs) {
			if (msg != null) {
				result.add(new EMMessage(msg));
			}
		}
		return result;
	}

	/**
	 * \~chinese
	 * 根据传入的参数从本地存储中搜索指定数量的消息。
	 * 
	 * 注意：当 maxCount 非常大时，需要考虑内存消耗，目前限制一次最多搜索 400 条数据。
	 *
	 * @param keywords   搜索消息中的关键词。
	 * @param timeStamp  搜索消息的时间点。
	 * @param maxCount   一次搜索结果的最大条数。
	 * @param from       搜索来自某人或者某群的消息，一般是指会话 ID。
	 * @return           消息列表。
	 *
	 * \~english
	 * Searches messages from the database according to the parameters.
	 * 
	 * Note: Cautious about the memory usage when the maxCount is large, currently the limited number is 400 entries at a time.
	 *
	 * @param keywords   The keywords in message.
	 * @param timeStamp  The timestamp for search, Unix timestamp, in milliseconds.
	 * @param maxCount   The max number of message to search at a time.
	 * @param from       A user ID or a group ID searches for messages, usually refers to the conversation ID.
	 * @return           The list of messages.
	 */
	public List<EMMessage> searchMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMConversation.EMSearchDirection direction) {
		EMAConversation.EMASearchDirection d = direction == EMConversation.EMSearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;

		List<EMAMessage> msgs = emaObject.searchMessages(keywords, timeStamp, maxCount, from, d);
		// to avoid resizing issue of array list, used linked list when size > 512
		List<EMMessage> result;
		if (msgs.size() > LIST_SIZE) {
			result = new LinkedList<EMMessage>();
		} else {
			result = new ArrayList<EMMessage>();
		}
		for (EMAMessage msg : msgs) {
			if (msg != null) {
				result.add(new EMMessage(msg));
			}
		}
		return result;
	}

	/**
	 * \~chinese
	 * 删除时间戳之前的本地消息记录。
	 *
	 * @param timeStamp 			Unix 时间戳，单位为毫秒。
	 * @param callback 				删除结果回调，详见 {@link EMCallBack}。
	 *
	 * \~english
	 * Deletes the local message record before the timestamp.
	 *
	 * @param timeStamp 			The Unix timestamp, in milliseconds.
	 * @param callback 				The result callback，see {@link EMCallBack}.
	 */
	public void deleteMessagesBeforeTimestamp(long timeStamp, EMCallBack callback){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				boolean result = emaObject.removeMessagesBeforeTimestamp(timeStamp);
				if(result){
					for(EMConversation.MessageCache cache : caches.values()){
						for(EMMessage message : cache.getAllMessages()){
							if(message.getMsgTime() < timeStamp){
								cache.removeMessage(message.getMsgId());
							}
						}
					}
					callback.onSuccess();
				}else{
					callback.onError(EMError.DATABASE_ERROR, "Database operation failed");
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 举报违规消息
	 * @param msgId				违规的消息id。
	 * @param reportTarget 				举报类型(例如：涉黄、涉恐)
	 * @param reportReason 				举报原因。
	 * @param callBack 				执行上报结果 成功执行 {@link EMCallBack#onSuccess()}，失败执行{@link EMCallBack#onError(int, String)}
	 *
	 * \~english
	 * Report violation message
	 *
	 * @param msgId				Violation Message ID
	 * @param reportTarget				Report type (For example: involving pornography and terrorism)
	 * @param reportReason				Report Reason
	 * @param callBack 				The result callback，see {@link EMCallBack}.
	 */
	public void asyncReportMessage(String msgId, String reportTarget, String reportReason,EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					reportMessage(msgId, reportTarget, reportReason);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}
	/**
	 * \~chinese
	 * 举报违规消息。
	 *
	 * 同步方法，会阻塞当前线程。
	 *
	 * @param msgId				违规的消息id。
	 * @param reportTarget 				举报类型(例如：涉黄、涉恐)
	 * @param reportReason 				举报原因。
	 * @throws HyphenateException 如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Report violation message
	 *
	 * @param msgId				Violation Message ID
	 * @param reportTarget				Report type (For example: involving pornography and terrorism)
	 * @param reportReason				Report Reason
	 *
	 * @throws HyphenateException A description of the exception. See {@link EMError}.
	 */
	public void reportMessage(String msgId, String reportTarget, String reportReason ) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.reportMessage(msgId,reportTarget,reportReason,error);
		handleError(error);
	}

	/**
	 * 获取翻译服务支持的语言
	 * @param callBack 完成回调
	 *
	 * \~english
	 * Fetch all languages what the translate service support
	 * @param callBack Complete the callback
	 */
	public void fetchSupportLanguages(EMValueCallBack<List<EMLanguage>> callBack) {
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try {
					List<EMLanguage> languages = new ArrayList<>();
					EMAError error = new EMAError();
					List<List<String>> result = emaObject.fetchSupportLanguages(error);
					handleError(error);
					for(List<String> list : result){
						languages.add(new EMLanguage(list.get(0), list.get(1), list.get(2)));
					}
					callBack.onSuccess(languages);
				} catch (HyphenateException e){
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 翻译消息
	 * @param message 消息对象
	 * @param languages 要翻译的目标语言code列表
	 * @param callBack 完成回调
	 *
	 * \~english
	 * translate a message
	 * @param message The message object
	 * @param languages List of target language codes to translate
	 * @param callBack Complete the callback
	 */
	public void translateMessage(EMMessage message, List<String> languages, EMValueCallBack<EMMessage> callBack){
		mClient.execute(new Runnable() {
			@Override
			public void run() {
				try{
					EMAError error = new EMAError();
					EMAMessage emaMessage = emaObject.translateMessage(message.emaObject, languages, error);
					handleError(error);
					callBack.onSuccess(new EMMessage(emaMessage));
				}catch(HyphenateException e){
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}
	/**
	 * \~chinese
	 * 添加 Reaction。
	 *
	 * 同步方法。
	 *
	 * @param messageId 消息 ID。
	 * @param reaction  消息 Reaction 内容。
	 *
	 * \~english
	 * Adds a reaction.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The reaction content.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void addReaction(final String messageId, final String reaction) throws HyphenateException {
		EMAError error = new EMAError();
		emaReactionObject.addReaction(messageId, reaction, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 添加 Reaction。
	 *
	 * 异步方法。
	 *
	 * @param messageId 消息 ID。
	 * @param reaction  消息 Reaction。
	 * @param callback  处理结果回调，详见 {@link EMCallBack}。
	 *
	 * \~english
	 * Adds a reaction.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The message reaction.
	 * @param callback  The result callback which contains the error information if the method fails.
	 */
	public void asyncAddReaction(final String messageId, final String reaction, final EMCallBack callback) {
		EMClient.getInstance().execute(() -> {
			try {
				addReaction(messageId, reaction);
				if (null != callback) {
					callback.onSuccess();
				}
			} catch (HyphenateException e) {
				EMLog.e(TAG, "asyncAddReaction error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				if (null != callback) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 删除 Reaction。
	 *
	 * 同步方法。
	 *
	 * @param messageId 消息 ID。
	 * @param reaction  消息 Reaction。
	 * @throws HyphenateException	如果有异常会在这里抛出，包含错误码和错误描述，详见 {@link EMError}。
	 *
	 * \~english
	 * Deletes a reaction.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The message reaction.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public void removeReaction(final String messageId, final String reaction) throws HyphenateException {
		EMAError error = new EMAError();
		emaReactionObject.removeReaction(messageId, reaction, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 删除 Reaction。
	 *
	 * 异步方法。
	 *
	 * @param messageId 消息 ID。
	 * @param reaction  消息 Reaction。
	 * @param callback  处理结果回调，详见 {@link EMCallBack}。
	 *
	 * \~english
	 * Deletes a reaction.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageId The message ID.
	 * @param reaction  The reaction content.
	 * @param callback  The result callback which contains the error information if the method fails.
	 */
	public void asyncRemoveReaction(final String messageId, final String reaction, final EMCallBack callback) {
		EMClient.getInstance().execute(() -> {
			try {
				removeReaction(messageId, reaction);
				if (null != callback) {
					callback.onSuccess();
				}
			} catch (HyphenateException e) {
				EMLog.e(TAG, "asyncRemoveReaction error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				if (null != callback) {
					callback.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取 Reaction 列表。
	 *
	 * 同步方法。
	 *
	 * @param messageIdList 消息 ID。
	 * @param chatType      会话类型，仅支持单聊（ {@link EMMessage.ChatType#Chat} ）和群聊（{@link EMMessage.ChatType#GroupChat}）。
	 * @param groupId       群组 ID，该参数只在群聊生效。
	 * @return map          指定消息 ID 对应的 Reaction 列表（EMMessageReaction 的 UserList 为概要数据，只包含前三个用户信息）。
	 *
	 * \~english
	 * Gets the list of Reactions.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param messageIdList  The message ID.
	 * @param chatType       The chat type. Only one-to-one chat ({@link EMMessage.ChatType#Chat} and group chat ({@link EMMessage.ChatType#GroupChat}) are allowed.
	 * @param groupId        The group ID, which is invalid only when the chat type is group chat.
	 * @return map           The Reaction list under the specified message ID（The UserList of EMMessageReaction is the summary data, which only contains the information of the first three users）.
	 * @throws HyphenateException	A description of the exception. See {@link EMError}.
	 */
	public Map<String, List<EMMessageReaction>> getReactionList(final List<String> messageIdList, final EMMessage.ChatType chatType, final String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		String chatTypeStr = "chat";
		if (EMMessage.ChatType.Chat == chatType) {
			chatTypeStr = "chat";
		} else if (EMMessage.ChatType.GroupChat == chatType) {
			chatTypeStr = "groupchat";
		}

		Map<String, List<EMAMessageReaction>> emaReactionMap = emaReactionObject.getReactionList(messageIdList, chatTypeStr, groupId, error);
		handleError(error);

		Map<String, List<EMMessageReaction>> results = new HashMap<>(emaReactionMap.size());
		if (emaReactionMap.size() > 0) {
			List<EMMessageReaction> reactionList;
			for (Map.Entry<String, List<EMAMessageReaction>> entry : emaReactionMap.entrySet()) {
				reactionList = new ArrayList<>(entry.getValue().size());
				for (EMAMessageReaction emaMessageReaction : entry.getValue()) {
					if (null != emaMessageReaction) {
						reactionList.add(new EMMessageReaction(emaMessageReaction));
					}
				}
				results.put(entry.getKey(), reactionList);
			}
		}
		return results;
	}

	/**
	 * \~chinese
	 * 获取 Reaction 列表。
	 *
	 * 异步方法。
	 *
	 * @param messageIdList 消息 ID。
	 * @param chatType      会话类型，仅支持单聊（ {@link EMMessage.ChatType#Chat} ）和群聊（{@link EMMessage.ChatType#GroupChat}）。
	 * @param groupId       群组 ID，该参数只在群聊生效。
	 * @param callback      处理结果回调，包含消息 ID 对应的 Reaction 列表（EMMessageReaction 的用户列表为概要数据，只包含前三个用户信息）。
	 *
	 * \~english
	 * Gets the list of Reactions.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageIdList  The message ID.
	 * @param chatType       The chat type. Only one-to-one chat ({@link EMMessage.ChatType#Chat} and group chat ({@link EMMessage.ChatType#GroupChat}) are allowed.
	 * @param groupId        The group ID, which is invalid only when the chat type is group chat.
	 * @param callback       The result callback, which contains the Reaction list under the specified message ID（The user list of EMMessageReaction is the summary data, which only contains the information of the first three users）.
	 */
	public void asyncGetReactionList(final List<String> messageIdList, final EMMessage.ChatType chatType, final String groupId, final EMValueCallBack<Map<String, List<EMMessageReaction>>> callback) {
		EMClient.getInstance().execute(() -> {
			try {
				if (null != callback) {
					callback.onSuccess(getReactionList(messageIdList, chatType, groupId));
				}
			} catch (HyphenateException e) {
				EMLog.e(TAG, "asyncGetReactionList error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				callback.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

	/**
	 * \~chinese
	 * 获取 Reaction 详细信息。
	 *
	 * 同步方法。
	 *
	 * @param messageId    消息 ID。
	 * @param reaction     消息 Reaction 内容。
	 * @param cursor       查询 cursor。
	 * @param pageSize     每页获取的 Reaction 条数。
	 * @param cursor       返回 cursor 和 EMMessageReaction 列表（仅使用该列表第一个数据即可）。cursor 为空值说明数据已全部获取到。
	 * @return EMCursorResult  包含 cursor(cursor 为空表示数据已全部获取)和 EMMessageReaction 列表。
	 *
	 * \~english
	 * Gets the reaction details.
	 *
	 * This is a synchronous method and blocks the current thread.
	 *
	 * @param messageId    The message ID.
	 * @param reaction     The reaction content.
	 * @param cursor       The cursor position from which to get Reactions.
	 * @param pageSize     The number of Reactions you expect to get on each page.
	 * @return EMCursorResult  The result callback, which contains the reaction list obtained from the server and the cursor for the next query. Returns null if all the data is fetched.
	 * @throws HyphenateException	A description of the exception, see {@link EMError}.
	 */
	public EMCursorResult<EMMessageReaction> getReactionDetail(final String messageId, final String reaction, final String cursor, final int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAMessageReaction> result = emaReactionObject.getReactionDetail(messageId, reaction, cursor, pageSize, error);
		handleError(error);
		EMCursorResult<EMMessageReaction> cursorResult = new EMCursorResult<>();
		cursorResult.setCursor(result.getCursor());
		if (null != result.getData()) {
			List<EMMessageReaction> messageReactionList = new ArrayList<>(result.getData().size());
			for (EMAMessageReaction emaMessageReaction : result.getData()) {
				if (null != emaMessageReaction) {
					messageReactionList.add(new EMMessageReaction(emaMessageReaction));
				}
			}
			cursorResult.setData(messageReactionList);
		}
		return cursorResult;
	}

	/**
	 * \~chinese
	 * 获取 Reaction 详细信息。
	 *
	 * 异步方法。
	 *
	 * @param messageId   消息 ID。
	 * @param reaction    消息 Reaction。
	 * @param cursor      查询 cursor。
	 * @param pageSize    每页获取的 Reaction 条数。
	 * @param callback    处理结果回调，包含 cursor 和 EMMessageReaction 列表（仅使用该列表第一个数据即可）。
	 *
	 * \~english
	 * Gets the reaction details.
	 *
	 * This is an asynchronous method.
	 *
	 * @param messageId    The message ID.
	 * @param reaction     The reaction content.
	 * @param cursor       The query cursor.
	 * @param pageSize     The number of Reactions you expect to get on each page.
	 * @param callback     The result callback, which contains the reaction list obtained from the server and the cursor for the next query. Returns null if all the data is fetched.
	 */
	public void asyncGetReactionDetail(final String messageId, final String reaction, final String cursor, final int pageSize, final EMValueCallBack<EMCursorResult<EMMessageReaction>> callback) {
		EMClient.getInstance().execute(() -> {
			try {
				if (null != callback) {
					callback.onSuccess(getReactionDetail(messageId, reaction, cursor, pageSize));
				}
			} catch (HyphenateException e) {
				EMLog.e(TAG, "asyncGetReactionDetail error code:" + e.getErrorCode() + ",error message:" + e.getDescription());
				callback.onError(e.getErrorCode(), e.getDescription());
			}
		});
	}

}
