/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate;

import com.hyphenate.chat.EMGroup;
import com.hyphenate.chat.EMMucSharedFile;

import java.util.List;

/**
 *  \~chinese
 *  群组事件监听器。
 *  监听群组相关的事件，例如，加群申请，群主同意或者拒绝加群，被踢群等事件。
 *  
 *      注册群组事件监听：
 * ```java
 *      EMClient.getInstance().groupManager().addGroupChangeListener(mGroupChangeListener);
 * ```
 *
 *      解注册群组事件监听：
 * ```java
 *      EMClient.getInstance().groupManager().removeGroupChangeListener(mGroupChangeListener);
 * ```
 *  
 *
 *  \~english
 *  The group change listener.
 *  Listens for group events such as requesting to join a group, approving or declining a group request, and kicking a user out of a group.
 * 
 *  
 *      Registers a group change listener:
 * ```java
 *      EMClient.getInstance().groupManager().addGroupChangeListener(mGroupChangeListener);
 * ```
 *
 *      Unregisters a group change listener:
 * ```java
 *      EMClient.getInstance().groupManager().removeGroupChangeListener(mGroupChangeListener);
 *  ```
 */
public interface EMGroupChangeListener {
	/**
     * \~chinese
	 * 用户收到入群邀请的回调。
	 * @param groupId 	群组 ID。
	 * @param groupName 群组名称。
	 * @param inviter 	邀请人 ID。
	 * @param reason 	邀请原因。
     *
     * \~english
	 * Occurs when the user receives a group invitation.
	 * @param groupId		The group ID.
	 * @param groupName		The group name.
	 * @param inviter		The invitee ID.
	 * @param reason		The reason for invitation.
	 */
	void onInvitationReceived(String groupId, String groupName, String inviter, String reason);
	
	/**
     * \~chinese
	 * 用户申请入群回调。
	 * @param groupId 	群组 ID。
	 * @param groupName 群组名称。
	 * @param applicant 申请人 ID。
	 * @param reason 	申请加入原因。
     *
     * \~english
     * Occurs when the group owner or administrator receives a group request from a user.
	 *
	 * @param groupId		The group ID.
	 * @param groupName		The group name.
	 * @param applicant		The ID of the user requesting to join the group.
	 * @param reason		The reason for requesting to join the group.
	 */
	void onRequestToJoinReceived(String groupId, String groupName, String applicant, String reason);

	/**
     * \~chinese
	 * 接受入群申请回调。
	 * @param groupId	群组 ID。
	 * @param groupName 群组名称。
	 * @param accepter 	接受人 ID.
     *
     * \~english
	 * Occurs when a group request is accepted.
	 *
	 * @param groupId 		The group ID.
	 * @param groupName 	The group name.
	 * @param accepter 		The ID of the user that accepts the group request.
	 */
	void onRequestToJoinAccepted(String groupId, String groupName, String accepter);

	/**
     * \~chinese
	 * 拒绝入群申请回调。
	 * @param groupId 	群组 ID。
	 * @param groupName 群组名称。
	 * @param decliner 	拒绝人 ID。
	 * @param reason 	拒绝理由
     *
     * \~english
     * Occurs when a group request is declined.
	 *
	 * @param groupId 		The group ID.
	 * @param groupName 	The group name.
	 * @param decliner 		The ID of the user that declines the group request.
	 * @param reason 		The reason for declining.
	 */
	void onRequestToJoinDeclined(String groupId, String groupName, String decliner, String reason);

	/**
     * \~chinese
	 * 接受入群邀请回调。
	 * @param groupId		群组 ID。
	 * @param invitee		受邀人 ID。
	 * @param reason		接受理由
     *
     * \~english
     * Occurs when a group invitation is accepted.
	 *
	 * @param groupId 		The group ID.
	 * @param invitee 		The invitee ID.
	 * @param reason		The reason for acceptance.
	 */
	void onInvitationAccepted(String groupId, String invitee, String reason);
    
	/**
     * \~chinese
	 * 拒绝群组邀请回调。
	 * @param groupId		群组 ID。
	 * @param invitee		受邀人 ID。
	 * @param reason 		拒绝理由。
     *
     * \~english
     * Occurs when a group invitation is declined.
	 *
	 * @param groupId 		The group ID.
	 * @param invitee		The invitee ID.
	 * @param reason 		The reason for declining.
	 */
	void onInvitationDeclined(String groupId, String invitee, String reason);
    
	/**
     * \~chinese
	 * 当前登录用户被管理员移出群组回调。
	 * @param groupId 		群组 ID。
	 * @param groupName		群组名称。
     *
     * \~english
     * Occurs when the current user is removed from the group by the group admin.
	 *
	 * @param groupId 		The group ID.
	 * @param groupName		The group name.
	 */
	void onUserRemoved(String groupId, String groupName);
    
    /**
     * \~chinese
     * 群组解散回调。
     * SDK 会先删除本地的这个群组，然后通过此回调通知应用此群组被删除。
     * @param groupId 	群组 ID。
     * @param groupName 群组名称。
     *
     * \~english
     * Occurs when a group is destroyed.
     * The SDK will remove the group from the local database and local cache before notifying the app of the group removal.
	 *
	 * @param groupId		The group ID.
	 * @param groupName 	The group name.
     */
	void onGroupDestroyed(String groupId, String groupName);
    
    /**
     * \~chinese
     * 自动同意入群申请回调。
     * SDK 会先加入这个群组，并通过此回调通知应用。
     * 具体设置，参考 {@link com.hyphenate.chat.EMOptions#setAutoAcceptGroupInvitation(boolean value)}。
     * 
     * @param groupId			群组 ID。
     * @param inviter			邀请者 ID。
     * @param inviteMessage		邀请信息。
     *
     * \~english
     * Occurs when the group invitation is accepted automatically.
	 * The SDK will join the group before notifying the app of the acceptance of the group invitation.
     * For settings, see {@link com.hyphenate.chat.EMOptions#setAutoAcceptGroupInvitation(boolean value)}.
	 *
	 * @param groupId			The group ID.
	 * @param inviter			The inviter ID.
	 * @param inviteMessage		The invitation message.
     */
	void onAutoAcceptInvitationFromGroup(String groupId, String inviter, String inviteMessage);

	/**
	 * \~chinese
	 * 有成员被禁言。
	 * 注意：禁言不同于加入黑名单。用户禁言后，将无法在群中发送消息，但可查看群组中的消息，而黑名单中的用户无法查看和发送群组消息。
	 *
	 * @param groupId 	群组 ID。
	 * @param mutes 	禁言的成员列表。
	 *              	Map.entry.key 是禁言成员 ID，Map.entry.value 是禁言时长。
	 *
	 * \~english
	 * Occurs when one or more group members are muted.
	 * Note: The mute function is different from a blocklist. A user, when muted, can still see group messages, but cannot send messages in the group. However, a user on the blocklist can neither see nor send group messages.
	 *
	 * @param groupId		The group ID.
	 * @param mutes 		The member(s) added to the mute list.
	 *	 			        Map.entry.key is the muted username; Map.entry.value is the mute duration in milliseconds.
	 */
	void onMuteListAdded(String groupId, final List<String> mutes, final long muteExpire);

	/**
	 * \~chinese
	 * 有成员被解除禁言。
	 * 注意：禁言不同于加入黑名单。用户禁言后，将无法在群中发送消息，但可查看群组中的消息，而黑名单中的用户无法查看和发送群组消息。
	 *
	 * @param groupId 	群组 ID。
	 * @param mutes 	有成员从群组禁言列表中移除。
	 *
	 * \~english
	 * Occurs when one or more group members are unmuted.
	 * Note: The mute function is different from a blocklist. A user, when muted, still can see group messages, but cannot send messages in the group. However, a user on the blocklist can neither see nor send group messages.
	 *
	 * @param groupId		The group ID.
	 * @param mutes 		The member(s) removed from the mute list.
	 */
	void onMuteListRemoved(String groupId, final List<String> mutes);

	/**
	 * \~chinese
	 * 添加白名单回调。
	 *
	 * @param groupId    	群组 ID。
	 * @param whitelist     要添加的成员列表。
	 *
	 * \~english
	 * Occurs when one or more group members are added to the allowlist.
	 *
	 * @param groupId    	The group ID.
	 * @param whitelist     The member(s) added to the allowlist.
	 */
	void onWhiteListAdded(final String groupId, final List<String> whitelist);

	/**
	 * \~chinese
	 * 移除白名单回调。
	 *
	 * @param groupId    	群组 ID。
	 * @param whitelist     从白名单中移除的成员列表
	 *
	 * \~english
	 * Occurs when one or more members are removed from the allowlist.
	 *
	 * @param groupId    	The group ID.
	 * @param whitelist     Member(s) removed from the allowlist.
	 */
	void onWhiteListRemoved(final String groupId, final List<String> whitelist);

	/**
	 * \~chinese
	 * 全员禁言状态变化回调。
	 *
	 * @param groupId    	群组 ID。
	 * @param isMuted       是否开启了全员禁言。
	 *
	 * \~english
	 * Occurs when all group members are muted or unmuted.
	 *
	 * @param groupId    	The group ID.
	 * @param isMuted       Whether all group members are muted or unmuted. true: all group members are muted; false: all group members are unmuted.
	 */
	void onAllMemberMuteStateChanged(final String groupId, final boolean isMuted);

	/**
	 * \~chinese
	 * 成员设置为管理员回调。
	 *
	 * @param groupId 		群组 ID。
	 * @param administrator 设置为管理员的成员。
	 *
	 * \~english
	 * Occurs when a member is set as an admin.
	 *
	 * @param groupId		The group ID.
	 * @param administrator The member that is set as an admin.
	 */
	void onAdminAdded(String groupId, String administrator);

	/**
	 * \~chinese
	 * 取消成员的管理员权限回调。
	 * @param groupId 		群组 ID。
	 * @param administrator 被取消管理员权限的成员。
	 *
	 * \~english
	 * Occurs when a member's admin privileges are removed.
	 *
	 * @param groupId 		The group ID.
	 * @param administrator The member whose admin privileges are removed.
	 */
	void onAdminRemoved(String groupId, String administrator);

	/**
	 * \~chinese
	 * 转移群主权限回调。
	 * @param groupId 	群组 ID。
	 * @param newOwner 	新群主。
	 * @param oldOwner 	原群主。
	 *
	 * \~english
	 * Occurs when the group ownership is transferred.
	 * @param groupId 		The group ID.
	 * @param newOwner 		The new owner.
	 * @param oldOwner 		The previous owner.
	 */
	void onOwnerChanged(String groupId, String newOwner, String oldOwner);
	
	/**
     * \~chinese
     * 新成员加入群组回调。
     * 
     * @param groupId 	群组 ID。
     * @param member 	新成员 ID。
     *
     * \~english
     * Occurs when a member joins a group.
     * 
     * @param groupId  The group ID.
     * @param member   The ID of the new member.
     */
    void onMemberJoined(final String groupId, final String member);

    /**
     * \~chinese
     * 群组成员主动退出回调。
     * 
     * @param groupId 	群组 ID。
     * @param member 	退群的成员 ID。
     * 
     * \~english
     * Occurs when a member proactively leaves the group.
     * 
     * @param groupId   The group ID.
     * @param member  	The member leaving the group.
      */
    void onMemberExited(final String groupId,  final String member);

	/**
	 * \~chinese
	 * 群公告更新回调。
	 * @param groupId      群组 ID。
	 * @param announcement 更新后的公告内容。
	 *
	 * \~english
	 * Occurs when the announcement is updated.
	 * @param groupId The group ID.
	 * @param announcement The updated announcement content.
	 */
	void onAnnouncementChanged(String groupId, String announcement);

	/**
	 * \~chinese
	 * 群组添加共享文件回调。
	 * @param groupId    群组 ID。
	 * @param sharedFile 添加的共享文件。
	 *
	 * \~english
	 * Occurs when a shared file is added to a group.
	 * @param groupId The group ID.
	 * @param sharedFile The new shared file.
	 */
	void onSharedFileAdded(String groupId, EMMucSharedFile sharedFile);

	/**
	 * \~chinese
	 * 群组删除共享文件回调。
	 * @param groupId 群组 ID。
	 * @param fileId  删除的共享文件的 ID。
	 *
	 * \~english
	 * Occurs when a shared file is removed from a group.
	 * @param groupId The group ID.
	 * @param fileId  The ID of the removed shared file.
	 */
	void onSharedFileDeleted(String groupId, String fileId);

	/**
	 * \~chinese
	 * 群组信息更新回调。
	 * @param group 群组。
	 *
	 * \~english
	 * Occurs when group info updated.
	 * @param group The group.
	 */
	default void onSpecificationChanged(EMGroup group){}
}
