/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate;

import com.hyphenate.chat.EMChatRoom;

import java.util.List;

/**
 * \~chinese
 * 聊天室事件监听接口。
 * 
 * 通过 {@link com.hyphenate.chat.EMChatRoomManager} 进行注册和解注册，具体如下：
 * 
 *     注册聊天室事件监听：
 *     ```java
 *     EMChatRoomManager manager = EMClient.getInstance().chatroomManager();
 *     manager.addChatRoomChangeListener(mChatRoomChangeListener);
 *     ```
 *
 *     解注册聊天室事件监听：
 *     ```java
 *     EMChatRoomManager manager = EMClient.getInstance().chatroomManager();
 *     manager.removeChatRoomListener(mChatRoomChangeListener);
 *     ```
 *
 * \~english
 * The chat room change listener.
 * 
 * Register/unregister it by {@link com.hyphenate.chat.EMChatRoomManager}:
 * 
 *     Register chat room change listener:
 *     ```java
 *     EMChatRoomManager manager = EMClient.getInstance().chatroomManager();
 *     manager.addChatRoomChangeListener(mChatRoomChangeListener);
 *     ```
 *
 *     Unregister chat room change listener:
 *     ```java
 *     EMChatRoomManager manager = EMClient.getInstance().chatroomManager();
 *     manager.removeChatRoomListener(mChatRoomChangeListener);
 *     ```
 */
public interface EMChatRoomChangeListener {
    /**
     * \~chinese
     * 聊天室被解散。
     * 
     * @param roomId        聊天室 ID。
     * @param roomName      聊天室名称。
     *
     * \~english
     * Occurs when the chat room is destroyed.
     * 
     * @param roomId        The chatroom ID.
     * @param roomName      The chatroom subject.
     */
    void onChatRoomDestroyed(final String roomId, final String roomName);

    /**
     * \~chinese
     * 聊天室加入新成员事件。
     * 
     * @param roomId        聊天室 ID。
     * @param participant   新成员 username。
     *
     * \~english
     * Occurs when a member join the chatroom.
     * 
     * @param roomId        The chatroom ID.
     * @param participant   The new member's username.
     */
    void onMemberJoined(final String roomId, final String participant);

    /**
     * \~chinese
     * 聊天室成员主动退出事件。
     * 
     * @param roomId        聊天室 ID。
     * @param roomName      聊天室名字。
     * @param participant   退出的成员 ID。
     * 
     * \~english
     * Occurs when a member leaves the chatroom.
     * 
     * @param roomId       The chatroom ID.
     * @param roomName     The name of the chatroom.
     * @param participant  The member who leaves the chatroom.
      */
    void onMemberExited(final String roomId, final String roomName, final String participant);

    /**
     * \~chinese
     * 聊天室人员被移除。
     *
     * @param reason        用户被移出聊天室的原因：
     *                        - xxx BE_KICKED：该用户被聊天室管理员移除；
     *                        - xxxBE_LICKED)FOR_OFFLINE：该用户由于当前设备断网被服务器移出聊天室。
     * 
     * @param roomId        聊天室 ID。
     * @param roomName      聊天室名字。
     * @param participant   被移除人员 ID。
     *
     * \~english
     * Occurs when a member is dismissed from a chat room.
     *
     * @param reason        The reason why the user is removed from the chatroom:
     *                       - xxxBE_KICKED: The user is removed by the chatroom owner.
     *                       - xxxBE_KICKED_FOR_OFFINE: The user is disconnected from the server, probably due to network interruption.
     * @param roomId        The chatroom ID.
     * @param roomName      The name of the chatroom.
     * @param participant   The member is dismissed from a chat room.
     */
    void onRemovedFromChatRoom(final int reason, final String roomId, final String roomName, final String participant);

    /**
     * \~chinese
     * 有成员被禁言。
     * 禁言期间成员不能发送发消息。
     *
     * @param chatRoomId    聊天室 ID。
     * @param mutes         禁言的成员。
     * @param expireTime    禁言有效期，单位是毫秒。
     *
     * \~english
     * Occurs when there are chat room member(s) muted (added to mute list),
     * The muted members are not allowed to post message temporarily based on muted time duration.
     *
     * @param chatRoomId    The chatroom ID.
     * @param mutes         The members to be muted.
     * @param expireTime    The mute duration.
     */
    void onMuteListAdded(final String chatRoomId, final List<String> mutes, final long expireTime);

    /**
     * \~chinese
     * 有成员从禁言列表中移除。
     *
     * @param chatRoomId    聊天室 ID。
     * @param mutes         从禁言列表中移除的成员名单。
     *
     * \~english
     * Occurs when there are chat room member(s) unmuted (removed from mute list).
     *
     * @param chatRoomId    The chatroom ID.
     * @param mutes         The member(s) muted is removed from the mute list.
     */
    void onMuteListRemoved(final String chatRoomId, final List<String> mutes);

    /**
     * \~chinese
     * 白名单成员增加。
     *
     * @param chatRoomId    聊天室 ID。
     * @param whitelist     白名单中增加的成员名单。
     *
     * \~english
     * Occurs when the chat room member(s) is added to the allowlist.
     *
     * @param chatRoomId    The chatroom ID.
     * @param whitelist     The member(s) to be added to the allowlist.
     */
    void onWhiteListAdded(final String chatRoomId, final List<String> whitelist);

    /**
     * \~chinese
     * 白名单成员减少。
     *
     * @param chatRoomId    聊天室 ID。
     * @param whitelist     白名单中移除的成员名单。
     *
     * \~english
     * Occurs when the chat room member(s) is removed from the allowlist.
     *
     * @param chatRoomId    The chatroom ID.
     * @param whitelist     The member(s) is removed from the allowlist.
     */
    void onWhiteListRemoved(final String chatRoomId, final List<String> whitelist);

    /**
     * \~chinese
     * 全员禁言状态的改变。
     *
     * @param chatRoomId    聊天室 ID。
     * @param isMuted       是否开启了全员禁言。
     *
     * \~english
     * Occurs when all members in the chat room are muted or unmuted.
     *
     * @param chatRoomId    The chatroom ID.
     * @param isMuted       Whether all chat room members is muted or unmuted.
     */
    void onAllMemberMuteStateChanged(final String chatRoomId, final boolean isMuted);

    /**
     * \~chinese
     * 有成员提升为管理员权限。
     *
     * @param chatRoomId    聊天室 ID。
     * @param admin         被提升管理员的成员。
     *
     * \~english
     * Occurs when a member has been changed to an admin.
     *
     * @param  chatRoomId   The chatroom ID.
     * @param  admin        The member who has been changed to an admin.
     */
    void onAdminAdded(final String chatRoomId, final String admin);

    /**
     * \~chinese
     * 移除管理员权限。
     *
     * @param  chatRoomId   聊天室 ID。
     * @param  admin        被移除的管理员。
     *
     * \~english
     * Occurs when an admin is been removed.
     *
     * @param  chatRoomId   The chatroom ID.
     * @param  admin        The member whose admin permission is removed.
     */
    void onAdminRemoved(final String chatRoomId, final String admin);

    /**
     * \~chinese
     * 转移聊天室拥有者。
     *
     * @param chatRoomId    聊天室 ID。
     * @param newOwner      新所有者。
     * @param oldOwner      原聊天室所有者。
     *
     * \~english
     * Occurs when the chat room ownership has been transferred.
     *
     * @param chatRoomId    The chatroom ID.
     * @param newOwner      The new owner.
     * @param oldOwner      The previous owner.
     */
    void onOwnerChanged(final String chatRoomId, final String newOwner, final String oldOwner);


    /**
     * \~chinese
     * 聊天室公告更改事件。
     * @param chatRoomId    聊天室 ID。
     * @param announcement  更新的公告内容。
     *
     * \~english
     * Occurs when the announcement changed.
     * @param chatRoomId    The chatroom ID.
     * @param announcement  The changed announcement.
     */
    void onAnnouncementChanged(String chatRoomId, String announcement);

    /**
     * \~chinese
     * 聊天室信息更新回调。
     * @param chatRoom    聊天室。
     *
     * \~english
     * Occurs when the chatroom specification changed.
     * @param chatRoom    The chatroom.
     */
    default void onSpecificationChanged(EMChatRoom chatRoom){}
}
