package com.hyphenate.util;

import android.content.Context;
import android.net.Uri;
import android.text.TextUtils;

import com.hyphenate.chat.EMClient;
import com.hyphenate.chat.EMOptions;

import java.io.File;

/**
 * {@link EMFileHelper}类为文件帮助类，实现了判断文件是否存在，获取文件绝对路径，获取文件名称
 * ，获取文件大小及获取文件类型等方法。
 * {@link EMFileHelper}类可以让开发者实现如下自定义的逻辑：
 * （1）判断文件是否存在{@link IFilePresenter#isFileExist(Context, Uri)}
 *  (2) 获取文件名称{@link IFilePresenter#getFilename(Context, Uri)}
 *  (3) 获取文件绝对路径{@link IFilePresenter#getFilePath(Context, Uri)}
 *  (4) 获取文件大小{@link IFilePresenter#getFileLength(Context, Uri)}
 *  (5) 获取文件类型{@link IFilePresenter#getFileMimeType(Context, Uri)}
 * 通过方法{@link #setFileHelper(IFilePresenter)}添加实现了{@link IFilePresenter}接口的对象即可
 *
 * 注：在{@link EMFileHelper}中标注了需要在{@link EMClient#init(Context, EMOptions)}后调用的方法
 * ，请务必在SDK初始化后调用，否则context会为空。
 */
public class EMFileHelper {
    private static class EMFileHelperInstance {
        private static final EMFileHelper instance = new EMFileHelper();
    }

    private Context mContext;
    private IFilePresenter mHelper;

    private EMFileHelper() {
        mContext = EMClient.getInstance().getContext();
        mHelper = new FilePresenterImpl();
    }

    public static EMFileHelper getInstance() {
        return EMFileHelperInstance.instance;
    }

    /**
     * 设置自定义的FilePresenter
     * @param presenter
     */
    public void setFileHelper(IFilePresenter presenter) {
        this.mHelper = presenter;
    }

    /**
     * 判断文件是否存在，需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param fileUri
     * @return
     */
    public boolean isFileExist(Uri fileUri) {
        return mHelper.isFileExist(mContext, fileUri);
    }

    /**
     * 判断文件是否存在，需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param stringUri
     * @return
     */
    public boolean isFileExist(String stringUri) {
        if (TextUtils.isEmpty(stringUri)) {
            return false;
        }
        return isFileExist(Uri.parse(stringUri));
    }

    /**
     * 判断文件是否存在
     * @param context
     * @param fileUri
     * @return
     */
    public boolean isFileExist(Context context, Uri fileUri) {
        return mHelper.isFileExist(context, fileUri);
    }

    /**
     * 判断文件是否存在
     * @param context
     * @param stringUri
     * @return
     */
    public boolean isFileExist(Context context, String stringUri) {
        if (TextUtils.isEmpty(stringUri)) {
            return false;
        }
        return isFileExist(context, Uri.parse(stringUri));
    }

    /**
     * 获取文件名称
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param fileUri
     * @return
     */
    public String getFilename(Uri fileUri) {
        return mHelper.getFilename(mContext, fileUri);
    }

    /**
     * 获取文件名称
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param filePath
     * @return
     */
    public String getFilename(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return "";
        }
        return getFilename(Uri.parse(filePath));
    }

    /**
     * 获取文件路径
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param fileUri
     * @return
     */
    public String getFilePath(Uri fileUri) {
        return mHelper.getFilePath(mContext, fileUri);
    }

    /**
     * 获取文件路径
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param filePath
     * @return
     */
    public String getFilePath(String filePath)
    {
        if (TextUtils.isEmpty(filePath)) {
            return filePath;
        }
        return getFilePath(Uri.parse(filePath));
    }

    /**
     * 获取文件路径
     * @param fileUri
     * @return
     */
    public String getFilePath(Context context, Uri fileUri) {
        return mHelper.getFilePath(context, fileUri);
    }

    /**
     * 获取文件路径
     * @param filePath
     * @return
     */
    public String getFilePath(Context context, String filePath)
    {
        if (TextUtils.isEmpty(filePath)) {
            return filePath;
        }
        return getFilePath(context, Uri.parse(filePath));
    }

    /**
     * 获取文件大小
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param fileUri
     * @return
     */
    public long getFileLength(Uri fileUri) {
        return mHelper.getFileLength(mContext, fileUri);
    }

    /**
     * 获取文件大小
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param filePath
     * @return
     */
    public long getFileLength(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return 0;
        }
        return getFileLength(Uri.parse(filePath));
    }

    /**
     * 获取文件类型
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param fileUri
     * @return
     */
    public String getFileMimeType(Uri fileUri) {
        return mHelper.getFileMimeType(mContext, fileUri);
    }

    /**
     * 删除文件（可以获取到绝对路径）
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param filePath
     * @return
     */
    public boolean deletePrivateFile(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return false;
        }
        if (!isFileExist(filePath)) {
            return false;
        }
        filePath = getFilePath(Uri.parse(filePath));
        if (!TextUtils.isEmpty(filePath))
        {
            File file = new File(filePath);
            if (file.exists()) {
                return file.delete();
            }
        }
        return false;
    }

    /**
     * 格式化输入到SDK内部的文件的Uri，并输出未Uri
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param fileUri
     * @return
     */
    public Uri formatInUri(Uri fileUri) {
        if (fileUri == null) {
            return null;
        }
        if ((VersionUtils.isTargetQ(mContext)) && (UriUtils.uriStartWithContent(fileUri))) {
            return fileUri;
        }
        String path = getFilePath(fileUri);
        if (!TextUtils.isEmpty(path)) {
            fileUri = Uri.parse(path);
        }
        return fileUri;
    }

    /**
     * 格式化输入到SDK的文件，并输出未Uri
     * @param file
     * @return
     */
    public Uri formatInUri(File file) {
        if (file == null) {
            return null;
        }
        return Uri.parse(file.getAbsolutePath());
    }

    /**
     * 格式化输入到SDK内部的文件的Uri，并输出未Uri
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param filePath
     * @return
     */
    public Uri formatInUri(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return null;
        }
        return formatInUri(Uri.parse(filePath));
    }

    /**
     * 将格式化后的Uri转为string
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param uri
     * @return
     */
    public String formatInUriToString(Uri uri) {
        uri = formatInUri(uri);
        if (uri == null) {
            return "";
        }
        return uri.toString();
    }

    /**
     * 格式化文件，并最终转为Uri的string样式
     * @param file
     * @return
     */
    public String formatInUriToString(File file) {
        Uri fileUri = formatInUri(file);
        if (fileUri == null) {
            return "";
        }
        return fileUri.toString();
    }

    /**
     * 格式化传入的路径，并最终转为Uri的string样式
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param filePath
     * @return
     */
    public String formatInUriToString(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return "";
        }
        return formatInUriToString(Uri.parse(filePath));
    }

    /**
     * 格式化输出的路径，如果可以获取到绝对路径，则优先返回绝对路径
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param filePath
     * @return
     */
    public String formatOutLocalUrl(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return filePath;
        }
        String path = getFilePath(filePath);
        if (!TextUtils.isEmpty(path)) {
            return path;
        }
        return filePath;
    }

    /**
     * 格式化输出的Uri
     * 注：需要在{@link EMClient#init(Context, EMOptions)}后调用
     * @param filePath
     * @return
     */
    public Uri formatOutUri(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return null;
        }
        Uri fileUri = Uri.parse(filePath);
        if ((VersionUtils.isTargetQ(mContext)) && (UriUtils.uriStartWithContent(fileUri))) {
            return fileUri;
        }
        String str = getFilePath(fileUri);
        if (!TextUtils.isEmpty(str)) {
            fileUri = Uri.fromFile(new File(str));
        }
        return fileUri;
    }

    /**
     * 将Uri转成string类型
     * @param fileUri
     * @return
     */
    public String uriToString(Uri fileUri) {
        if (fileUri == null) {
            return "";
        }
        return fileUri.toString();
    }

    public static class FilePresenterImpl implements IFilePresenter {

        @Override
        public boolean isFileExist(Context context, Uri fileUri) {
            return UriUtils.isFileExistByUri(context, fileUri);
        }

        @Override
        public String getFilename(Context context, Uri fileUri) {
            return UriUtils.getFileNameByUri(context, fileUri);
        }

        @Override
        public String getFilePath(Context context, Uri fileUri) {
            return UriUtils.getFilePath(context, fileUri);
        }

        @Override
        public long getFileLength(Context context, Uri fileUri) {
            return UriUtils.getFileLength(context, fileUri);
        }

        @Override
        public String getFileMimeType(Context context, Uri fileUri) {
            return UriUtils.getMimeType(context, fileUri);
        }
    }

    public interface IFilePresenter {
        /**
         * 用于判断文件是否存在
         * @param context
         * @param fileUri
         */
        boolean isFileExist(Context context, Uri fileUri);

        /**
         * 获取文件名
         * @param context
         * @param fileUri
         */
        String getFilename(Context context, Uri fileUri);

        /**
         * 获取文件绝对路径
         * @param context
         * @param fileUri
         */
        String getFilePath(Context context, Uri fileUri);

        /**
         * 获取文件大小
         * @param context
         * @param fileUri
         * @return
         */
        long getFileLength(Context context, Uri fileUri);

        /**
         * 获取文件类型
         * @param context
         * @param fileUri
         * @return
         */
        String getFileMimeType(Context context, Uri fileUri);
    }
}

