package com.hyphenate.push;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;

import com.hyphenate.util.EMLog;

import java.util.ArrayList;

public class EMPushConfig {
    private static final String TAG = "EMPushConfig";

    private String fcmSenderId;
    private String hwAppId;
    private String miAppId;
    private String miAppKey;
    private String mzAppId;
    private String mzAppKey;
    private String oppoAppKey;
    private String oppoAppSecret;
    private String vivoAppId;
    private String vivoAppKey;
    private ArrayList<EMPushType> enabledPushTypes;

    private EMPushConfig() {
    }

    public String getFcmSenderId() {
        return fcmSenderId;
    }

    public String getHwAppId() {
        return hwAppId;
    }

    public String getMiAppId() {
        return this.miAppId;
    }

    public String getMiAppKey() {
        return this.miAppKey;
    }

    public String getMzAppId() {
        return this.mzAppId;
    }

    public String getMzAppKey() {
        return this.mzAppKey;
    }

    public String getOppoAppKey() {
        return this.oppoAppKey;
    }

    public String getOppoAppSecret() {
        return this.oppoAppSecret;
    }

    public String getVivoAppId() {
        return vivoAppId;
    }

    public String getVivoAppKey() {
        return vivoAppKey;
    }

    public ArrayList<EMPushType> getEnabledPushTypes() {
        return this.enabledPushTypes;
    }

    @Override
    public String toString() {
        return "EMPushConfig{" +
                "fcmSenderId='" + fcmSenderId + '\'' +
                ", hwAppId='" + hwAppId + '\'' +
                ", miAppId='" + miAppId + '\'' +
                ", miAppKey='" + miAppKey + '\'' +
                ", mzAppId='" + mzAppId + '\'' +
                ", mzAppKey='" + mzAppKey + '\'' +
                ", oppoAppKey='" + oppoAppKey + '\'' +
                ", oppoAppSecret='" + oppoAppSecret + '\'' +
                ", vivoAppId='" + vivoAppId + '\'' +
                ", vivoAppKey='" + vivoAppKey + '\'' +
                ", enabledPushTypes=" + enabledPushTypes +
                '}';
    }

    public static class Builder {
        private Context context;

        private String fcmSenderId;
        private String hwAppId;
        private String miAppId;
        private String miAppKey;
        private String mzAppId;
        private String mzAppKey;
        private String oppoAppKey;
        private String oppoAppSecret;
        private String vivoAppId;
        private String vivoAppKey;
        private ArrayList<EMPushType> enabledPushTypes = new ArrayList();

        public Builder(Context context) {
            this.context = context.getApplicationContext();
        }

        public Builder(Context context, EMPushConfig config) {
            this(context);

            if (config == null) {
                return;
            }

            if (config.enabledPushTypes.contains(EMPushType.MIPUSH)) {
                enableMiPush(config.miAppId, config.miAppKey);
            }

            if (config.enabledPushTypes.contains(EMPushType.HMSPUSH)) {
                enableHWPush();
            }

            if (config.enabledPushTypes.contains(EMPushType.VIVOPUSH)) {
                enableVivoPush();
            }

            if (config.enabledPushTypes.contains(EMPushType.OPPOPUSH)) {
                enableOppoPush(config.oppoAppKey, config.oppoAppSecret);
            }

            if (config.enabledPushTypes.contains(EMPushType.MEIZUPUSH)) {
                enableMeiZuPush(config.mzAppId, config.mzAppKey);
            }

            if (config.enabledPushTypes.contains(EMPushType.FCM)) {
                enableFCM(config.fcmSenderId);
            }
        }

        public Builder enableMiPush(String appId, String appKey) {
            if (!TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey)) {
                this.miAppId = appId;
                this.miAppKey = appKey;
                this.enabledPushTypes.add(EMPushType.MIPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appId or appKey can't be empty when enable MI push !");
                return this;
            }
        }

        public Builder enableHWPush() {
            try {
                ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(
                        context.getPackageName(), PackageManager.GET_META_DATA);
                hwAppId = appInfo.metaData.getString("com.huawei.hms.client.appid");

                if(hwAppId != null && hwAppId.contains("=")) {
                    hwAppId = hwAppId.split("=")[1];
                } else {
                    // compatible for user set "value='xxxx'" directly
                    int id = appInfo.metaData.getInt("com.huawei.hms.client.appid");
                    hwAppId = String.valueOf(id);
                }
                this.enabledPushTypes.add(EMPushType.HMSPUSH);
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
            } catch (NullPointerException e) {
                EMLog.e(TAG, "Huawei push must config meta-data: com.huawei.hms.client.appid in AndroidManifest.xml.");
            } catch (ArrayIndexOutOfBoundsException e) {
                EMLog.e(TAG, "Huawei push meta-data: com.huawei.hms.client.appid value must be like this 'appid=xxxxxx'.");
            }
            return this;
        }

        public Builder enableFCM(String senderId) {
            if (!TextUtils.isEmpty(senderId)) {
                this.fcmSenderId = senderId;
                this.enabledPushTypes.add(EMPushType.FCM);
                return this;
            } else {
                EMLog.e(TAG, "senderId can't be empty when enable FCM push !");
                return this;
            }
        }

        public Builder enableMeiZuPush(String appId, String appKey) {
            if (!TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey)) {
                this.mzAppId = appId;
                this.mzAppKey = appKey;
                this.enabledPushTypes.add(EMPushType.MEIZUPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appId or appKey can't be empty when enable MEIZU push !");
                return this;
            }
        }

        public Builder enableOppoPush(String appKey, String appSecret) {
            if (!TextUtils.isEmpty(appKey) && !TextUtils.isEmpty(appSecret)) {
                this.oppoAppKey = appKey;
                this.oppoAppSecret = appSecret;
                this.enabledPushTypes.add(EMPushType.OPPOPUSH);
                return this;
            } else {
                EMLog.e(TAG, "appKey or appSecret can't be empty when enable OPPO push !");
                return this;
            }
        }

        public Builder enableVivoPush() {
            try {
                ApplicationInfo appInfo = context.getPackageManager().getApplicationInfo(
                        context.getPackageName(), PackageManager.GET_META_DATA);
                vivoAppId = appInfo.metaData.getInt("com.vivo.push.app_id") + "";
                vivoAppKey = appInfo.metaData.getString("com.vivo.push.api_key");
                this.enabledPushTypes.add(EMPushType.VIVOPUSH);
            } catch (PackageManager.NameNotFoundException e) {
                EMLog.e(TAG, "NameNotFoundException: " + e.getMessage());
            }
            return this;
        }

        public EMPushConfig build() {
            EMPushConfig pushConfig = new EMPushConfig();
            pushConfig.fcmSenderId = this.fcmSenderId;
            pushConfig.hwAppId = this.hwAppId;
            pushConfig.miAppId = this.miAppId;
            pushConfig.miAppKey = this.miAppKey;
            pushConfig.mzAppId = this.mzAppId;
            pushConfig.mzAppKey = this.mzAppKey;
            pushConfig.oppoAppKey = this.oppoAppKey;
            pushConfig.oppoAppSecret = this.oppoAppSecret;
            pushConfig.vivoAppId = this.vivoAppId;
            pushConfig.vivoAppKey = this.vivoAppKey;
            pushConfig.enabledPushTypes = this.enabledPushTypes;
            return pushConfig;
        }
    }
}
