package com.hyphenate.chat;

import com.hyphenate.EMError;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAUserInfoManager;
import com.hyphenate.util.EMLog;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import com.hyphenate.chat.EMUserInfo.*;

/**
 * Created by lijian on 2021/03/17
 *
 * */

 public class EMUserInfoManager {

    /**
     * Be careful about synchronized usage,
     * there are lots of callback, may lead to concurrent issue
     */
    static final String TAG = "EMUserInfoManager";
    EMAUserInfoManager emaObject;
    private ExecutorService executorService;

    protected EMUserInfoManager(EMAUserInfoManager manager) {
        emaObject = manager;
        executorService = Executors.newCachedThreadPool();
    }

    /**
     * \~chinese
     * 修改自己的用户信息
     *
     * @param userInfo 要修改的用户信息
     * @param callBack 结果回调
     *
     * \~english
     * Update own userInfo.
     *
     * @param callBack result callback
     */
    public void updateOwnInfo(final EMUserInfo userInfo, final EMValueCallBack<String> callBack) {
        EMLog.d(TAG, "start updateOwnInfo");
        if(userInfo == null){
            EMLog.d(TAG, "fetchUserInfoByUserId userInfo is empty");
            callBack.onError(EMError.USER_ILLEGAL_ARGUMENT,"userInfo is empty");
            return;
        }
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                JSONObject object = new JSONObject();
                try {
                    if(userInfo != null){
                        object.putOpt(EMUserInfoType.NICKNAME.getDesc(),userInfo.getNickName());
                        object.putOpt(EMUserInfoType.AVATAR_URL.getDesc(),userInfo.getAvatarUrl());
                        object.putOpt(EMUserInfoType.EMAIL.getDesc(),userInfo.getEmail());
                        object.putOpt(EMUserInfoType.GENDER.getDesc(),userInfo.getGender());
                        object.putOpt(EMUserInfoType.BIRTH.getDesc(),userInfo.getBirth());
                        object.putOpt(EMUserInfoType.PHONE.getDesc(),userInfo.getPhoneNumber());
                        object.putOpt(EMUserInfoType.SIGN.getDesc(),userInfo.getSignature());
                        object.putOpt(EMUserInfoType.EXT.getDesc(),userInfo.getExt());
                    }
                    EMLog.d(TAG, "updateOwnInfo param: " + object.toString());
                    emaObject.updateOwnInfo(object.toString(), error);
                    if (callBack == null) {
                        return;
                    }

                    if (error.errCode() == EMError.EM_NO_ERROR) {
                        EMLog.d(TAG, "updateOwnInfo success");
                        callBack.onSuccess(error.toString());
                    } else {
                        EMLog.e(TAG, "updateOwnInfo failed error:" + error.errCode() + "  errorMessage:" + error.errMsg());
                        callBack.onError(error.errCode(), error.errMsg());
                    }
                }catch (Exception e){
                    e.getStackTrace();
                }
            }
        });
    }


    /**
     * \~chinese
     * 修改自己用户信息中的某个属性
     *
     * @param attribute 用户属性字段
     * @param value 修改后的信息(value为空的时候会把设置的attribute删除)
     * @param callBack 结果回调
     *
     * \~english
     * Update own userInfo.
     *
     * @param callBack result callback
     */
    public void updateOwnInfoByAttribute(final EMUserInfoType attribute, final String value, final EMValueCallBack<String> callBack) {
        EMLog.d(TAG, "start updateOwnInfoByAttribute");
        if(attribute == null){
            EMLog.d(TAG, "fetchUserInfoByUserId attribute is null");
            callBack.onError(EMError.USER_ILLEGAL_ARGUMENT,"attribute is null");
            return;
        }
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                JSONObject object = new JSONObject();
                try {
                    object.putOpt(attribute.getDesc(),value);
                    EMLog.d(TAG, "updateOwnInfoByAttribute param: " + object.toString());
                    String response = emaObject.updateOwnInfo(object.toString(), error);
                    EMLog.d(TAG, "updateOwnInfoByAttribute response: " + response);
                    if (callBack == null) {
                        return;
                    }

                    if (error.errCode() == EMError.EM_NO_ERROR) {
                        EMLog.d(TAG, "updateOwnInfoByAttribute success");
                        callBack.onSuccess(response);
                    } else {
                        EMLog.e(TAG, "updateOwnInfoByAttribute failed error:" + error.errCode() + "  errorMessage:" + error.errMsg());
                        callBack.onError(error.errCode(), error.errMsg());
                    }
                }catch (Exception e){
                    e.getStackTrace();
                }
            }
        });
    }

    /**
     * \~chinese
     * 根据环信ID获取用户信息
     *
     * @param userIds 用户ID列表
     * @param callBack 结果回调
     *
     * \~english
     * Update own userInfo.
     *
     * @param callBack result callback
     */
    public void fetchUserInfoByUserId(final String[] userIds, final EMValueCallBack<Map<String,EMUserInfo>> callBack) {
        EMLog.d(TAG, "start fetchUserInfoByUserId");
        if(userIds == null || userIds.length == 0){
            EMLog.d(TAG, "fetchUserInfoByUserId userIds is empty");
            callBack.onError(EMError.USER_ILLEGAL_ARGUMENT,"userIds is empty");
            return;
        }
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                List<String> userIds2 = new ArrayList<String>();
                Collections.addAll(userIds2, userIds);
                List<String> attributes = new ArrayList<>();
                attributes.add(EMUserInfoType.NICKNAME.getDesc());
                attributes.add(EMUserInfoType.EMAIL.getDesc());
                attributes.add(EMUserInfoType.PHONE.getDesc());
                attributes.add(EMUserInfoType.GENDER.getDesc());
                attributes.add(EMUserInfoType.AVATAR_URL.getDesc());
                attributes.add(EMUserInfoType.SIGN.getDesc());
                attributes.add(EMUserInfoType.BIRTH.getDesc());
                attributes.add(EMUserInfoType.EXT.getDesc());
                EMLog.d(TAG, "fetchUserInfoByUserId param: " + userIds2.toString() + "  " + attributes.toString());
                String response = emaObject.fetchUserInfoByAttribute(userIds2,attributes,error);
                if (callBack == null) {
                    return;
                }
                EMLog.d(TAG, "fetchUserInfoByUserId response: " + response);
                if (error.errCode() == EMError.EM_NO_ERROR) {
                    EMLog.d(TAG, "fetchUserInfoByUserId success");
                    Map<String,EMUserInfo> userInfoMap = new HashMap<>();
                    if(response != null && response.length() > 0){
                        try {
                            JSONObject object = new JSONObject(response);
                            for(String  userId : userIds){
                                JSONObject obj = object.getJSONObject(userId);
                                if(obj != null){
                                    EMUserInfo userInfo = new EMUserInfo();
                                    userInfo.setNickName(obj.optString(EMUserInfoType.NICKNAME.getDesc()));
                                    String gender = obj.optString(EMUserInfoType.GENDER.getDesc());
                                    if(gender != null && gender.length() > 0){
                                        userInfo.setGender(Integer.valueOf(gender));
                                    }
                                    userInfo.setEmail(obj.optString(EMUserInfoType.EMAIL.getDesc()));
                                    userInfo.setPhoneNumber(obj.optString(EMUserInfoType.PHONE.getDesc()));
                                    userInfo.setSignature(obj.optString(EMUserInfoType.SIGN.getDesc()));
                                    userInfo.setAvatarUrl(obj.optString(EMUserInfoType.AVATAR_URL.getDesc()));
                                    userInfo.setExt(obj.optString(EMUserInfoType.EXT.getDesc()));
                                    userInfo.setBirth(obj.optString(EMUserInfoType.BIRTH.getDesc()));
                                    userInfo.setUserId(userId);
                                    userInfoMap.put(userId,userInfo);
                                }
                            }
                        }catch (JSONException e){
                            error.errMsg();
                        }
                        callBack.onSuccess(userInfoMap);
                    }else{
                        EMLog.e(TAG, "fetchUserInfoByUserId response is null");
                        callBack.onSuccess(null);
                    }
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                    EMLog.e(TAG, "fetchUserInfoByUserId failed error:" + error.errCode() + "  errorMessage:" + error.errMsg());
                }
            }
        });
    }


    /**
     * \~chinese
     * 根据环信ID 用户属性获取用户信息
     *
     * @param userIds 用户ID列表
     * @param attributes 用户属性
     * @param callBack 结果回调
     *
     * \~english
     * Update own userInfo.
     *
     * @param callBack result callback
     */
    public void fetchUserInfoByAttribute(final String[] userIds, final EMUserInfoType[] attributes, EMValueCallBack<Map<String,EMUserInfo>> callBack) {
        EMLog.d(TAG, "start fetchUserInfoByAttribute");
        if(userIds == null || userIds.length == 0){
            EMLog.d(TAG, "fetchUserInfoByUserId userIds is empty");
            callBack.onError(EMError.USER_ILLEGAL_ARGUMENT,"userIds is empty");
            return;
        }
        if(attributes == null || attributes.length == 0){
            EMLog.d(TAG, "fetchUserInfoByUserId attributes is empty");
            callBack.onError(EMError.USER_ILLEGAL_ARGUMENT,"attributes is empty");
            return;
        }
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                List<String> userIds2 = new ArrayList<String>();
                Collections.addAll(userIds2, userIds);
                List<String> attributes2 = new ArrayList<String>();
                for(EMUserInfoType type : attributes){
                    attributes2.add(type.getDesc());
                }
                EMLog.d(TAG, "fetchUserInfoByAttribute param: " + userIds2.toString() + "  " + attributes2.toString());
                String response = emaObject.fetchUserInfoByAttribute(userIds2,attributes2,error);
                EMLog.d(TAG, "fetchUserInfoByAttribute response: " + response);
                if (callBack == null) {
                    return;
                }

                if (error.errCode() == EMError.EM_NO_ERROR) {
                    EMLog.d(TAG, "fetchUserInfoByAttribute success");
                    Map<String, EMUserInfo> userInfoMap = new HashMap<>();
                    if (response != null && response.length() > 0) {
                        try {
                            JSONObject object = new JSONObject(response);
                            for (String userId : userIds) {
                                JSONObject obj = object.getJSONObject(userId);
                                if (obj != null) {
                                    EMUserInfo userInfo = new EMUserInfo();
                                    userInfo.setNickName(obj.optString(EMUserInfoType.NICKNAME.getDesc()));
                                    String gender = obj.optString(EMUserInfoType.GENDER.getDesc());
                                    if(gender != null && gender.length() > 0){
                                        userInfo.setGender(Integer.valueOf(gender));
                                    }

                                    userInfo.setEmail(obj.optString(EMUserInfoType.EMAIL.getDesc()));
                                    userInfo.setPhoneNumber(obj.optString(EMUserInfoType.PHONE.getDesc()));
                                    userInfo.setSignature(obj.optString(EMUserInfoType.SIGN.getDesc()));
                                    userInfo.setAvatarUrl(obj.optString(EMUserInfoType.AVATAR_URL.getDesc()));
                                    userInfo.setExt(obj.optString(EMUserInfoType.EXT.getDesc()));
                                    userInfo.setBirth(obj.optString(EMUserInfoType.BIRTH.getDesc()));
                                    userInfo.setUserId(userId);
                                    userInfoMap.put(userId, userInfo);
                                }
                            }
                        } catch (JSONException e) {
                            error.errMsg();
                        }
                        callBack.onSuccess(userInfoMap);
                    } else {
                        EMLog.d(TAG, "fetchUserInfoByAttribute response is null ");
                        callBack.onSuccess(null);
                    }
                }else{
                    callBack.onError(error.errCode(), error.errMsg());
                    EMLog.e(TAG, "fetchUserInfoByAttribute failed error:" + error.errCode() + "  errorMessage:" + error.errMsg());
                }
            }
        });
    }
}
