/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.annotation.TargetApi;
import android.app.Activity;
import android.app.Application;
import android.app.Application.ActivityLifecycleCallbacks;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.os.Build;
import android.os.Bundle;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.provider.Settings.Secure;
import android.text.TextUtils;
import android.util.Log;

import internal.com.getkeepsafe.relinker.ReLinker;
import com.hyphenate.EMCallBack;
import com.hyphenate.EMClientListener;
import com.hyphenate.EMConnectionListener;
import com.hyphenate.EMError;
import com.hyphenate.EMMultiDeviceListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.analytics.EMTimeTag;
import com.hyphenate.chat.adapter.EMAChatClient;
import com.hyphenate.chat.adapter.EMAChatClient.EMANetwork;
import com.hyphenate.chat.adapter.EMAConnectionListener;
import com.hyphenate.chat.adapter.EMADeviceInfo;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAMultiDeviceListener;
import com.hyphenate.chat.adapter.EMANetCallback;
import com.hyphenate.chat.adapter.EMARTCConfigManager;
import com.hyphenate.chat.core.EMAdvanceDebugManager;
import com.hyphenate.chat.core.EMChatConfigPrivate;
import com.hyphenate.chat.core.EMDBManager;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.monitor.EMNetworkMonitor;
import com.hyphenate.notification.core.EMNotificationHelper;
import com.hyphenate.push.EMPushConfig;
import com.hyphenate.push.EMPushHelper;
import com.hyphenate.push.EMPushType;
import com.hyphenate.util.CryptoUtils;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.NetUtils;
import com.hyphenate.util.PathUtil;
import com.hyphenate.util.Utils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Pattern;

/**
 * \~chinese
 * IM SDK的入口，负责登录退出及连接管理等，由此可以获得其他模块的入口
 * <pre>
 *  EMChatManager chatManager = EMClient.getInstance().chatManager();
 * </pre>
 * <p>
 * \~english
 * IM SDK Client, entrance of SDK, used to login, logout, and get access IM modules, such as
 * <pre>
 *  EMChatManager chatManager = EMClient.getInstance().chatManager();
 * </pre>
 */
public class EMClient {
    public final static String TAG = "EMClient";
    static private EMClient instance = null;
    static boolean libraryLoaded = false;

    private EMGroupManager groupManager;
    private EMChatRoomManager chatroomManager;
    private EMChatManager chatManager;
    private EMContactManager contactManager;
    private EMCallManager callManager;
    private EMConferenceManager conferenceManager;
    private EMUserInfoManager userInfoManager;
    private EMPushManager pushManager;

    private EMAChatClient emaObject;
    private Context mContext;
    private ExecutorService executor = null;
    private ExecutorService mainQueue = Executors.newSingleThreadExecutor();
    private ExecutorService sendQueue = Executors.newSingleThreadExecutor();
    private EMEncryptProvider encryptProvider = null;
    private CryptoUtils cryptoUtils = new CryptoUtils();
    private boolean sdkInited = false;
    private EMChatConfigPrivate mChatConfigPrivate;
    private List<EMConnectionListener> connectionListeners = Collections.synchronizedList(new ArrayList<EMConnectionListener>());

	private MyConnectionListener connectionListener;
	private EMSmartHeartBeat smartHeartbeat = null;
	private List<EMClientListener> clientListeners = Collections.synchronizedList(new ArrayList<EMClientListener>());
    private List<EMMultiDeviceListener> multiDeviceListeners = Collections.synchronizedList(new ArrayList<EMMultiDeviceListener>());
    private MyMultiDeviceListener multiDeviceListenerImpl;
	private WakeLock wakeLock;

	private ConnectivityManager connManager;
	private EMANetwork currentNetworkType = EMANetwork.NETWORK_NONE;

	public final static String VERSION = "3.8.3.1";

	/**
	 * preventing client to instantiate this EMClient
	 *
	 */
	private EMClient() {
	}

	public static EMClient getInstance() {
		if(instance == null){
		    synchronized (EMClient.class) {
		        if(instance == null) {
                    instance = new EMClient();
		        }
            }
		}
		return instance;
	}


    /*!
    * \~chinese
    * 初始化SDK。
    *
    * \~english
    * initialize the SDK
    */
    public void init(Context context, EMOptions options) {
        if(sdkInited){
            return;
        }

        final EMTimeTag tag = new EMTimeTag();
        tag.start();

        mContext = context.getApplicationContext();
        connManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        //if android sdk >= 14,register callback to know app enter the background or foreground
        registerActivityLifecycleCallbacks();

        loadLibrary();

        mChatConfigPrivate = new EMChatConfigPrivate();
        mChatConfigPrivate.load(context, options);
        options.setConfig(mChatConfigPrivate);

        // multi-device is enabled by setDeviceUuid
        // config setDeviceUuid before EMAChatClient.create being called
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(context);
        getChatConfigPrivate().setDeviceUuid(deviceFactory.getDeviceUuid().toString());
        getChatConfigPrivate().setDeviceName(Build.MANUFACTURER + Build.MODEL);

        // Init push relevant.
        EMPushConfig pushConfig = options.getPushConfig();
        if (pushConfig == null) {
            pushConfig = new EMPushConfig.Builder(mContext).build();
        }
        // Merge EMOptions to EMPushConfig
        EMPushConfig.Builder builder = new EMPushConfig.Builder(mContext, pushConfig);
        if (TextUtils.isEmpty(pushConfig.getFcmSenderId())) {
            builder.enableFCM(options.getFCMNumber());
        }
        if (TextUtils.isEmpty(pushConfig.getMiAppId()) || TextUtils.isEmpty(pushConfig.getMiAppKey())) {
            EMChatConfigPrivate.EMMipushConfig miPushConfig = options.getMipushConfig();
            if (miPushConfig != null) {
                builder.enableMiPush(miPushConfig.appId, miPushConfig.appKey);
            }
        }
        EMPushHelper.getInstance().init(context, builder.build());
        emaObject = EMAChatClient.create(mChatConfigPrivate.emaObject);
        // sandbox
        // rest 103.241.230.122:31080 im-msync 103.241.230.122:31097
//        mChatConfigPrivate.setChatServer("118.193.28.212");
//        mChatConfigPrivate.setChatPort(31097);
//        mChatConfigPrivate.setRestServer("https://118.193.28.212:31443");
//        mChatConfigPrivate.enableDnsConfig(false);

        connectionListener = new MyConnectionListener();
        emaObject.addConnectionListener(connectionListener);

        multiDeviceListenerImpl = new MyMultiDeviceListener();
        emaObject.addMultiDeviceListener(multiDeviceListenerImpl);

        executor = Executors.newCachedThreadPool();

        cryptoUtils.init(CryptoUtils.ALGORIGHM_AES);

        // init all the managers
        initManagers();

        final String lastLoginUser = EMSessionManager.getInstance().getLastLoginUser();

        EMLog.e(TAG, "is autoLogin : " + options.getAutoLogin());
        EMLog.e(TAG, "lastLoginUser : " + lastLoginUser);
        EMLog.e(TAG, "hyphenate SDK is initialized with version : " + getChatConfigPrivate().getVersion());

        PowerManager pm = (PowerManager) mContext.getSystemService(Context.POWER_SERVICE);
        wakeLock = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "emclient");
        EMNotificationHelper.getInstance().onInit(context);
        sdkInited = true;

        if (options.getAutoLogin() && isLoggedInBefore()) {
                final String lastLoginToken = EMSessionManager.getInstance().getLastLoginToken();
                final String lastLoginPwd = EMSessionManager.getInstance().getLastLoginPwd();

                //sessionManager.login(lastLoginUser, lastLoginPwd,false, null);
                EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);

                final EMCallBack callback = new EMCallBack() {

                    @Override
                    public void onSuccess() {
                        EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);
                        Log.d(TAG, "hyphenate login onSuccess");
                        tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
                    }

                    @Override
                    public void onError(int code, String error) {
                        Log.d(TAG, "hyphenate login onError");
                        tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
                    }

                    @Override
                    public void onProgress(int progress, String status) {
                    }
                };

                this.execute(new Runnable() {

                    @Override
                    public void run() {
                        // to ensure database has been open before login,
                        // login takes long time before user can see the result
                        getChatConfigPrivate().openDatabase(lastLoginUser);
                        groupManager().loadAllGroups();
                        chatManager().loadAllConversationsFromDB();

                        _login(lastLoginUser,
                                EMSessionManager.getInstance().isLastLoginWithToken() ? lastLoginToken : lastLoginPwd,
                                callback,
                                /*autoLogin*/ true,
                                /*loginWithToken*/ EMSessionManager.getInstance().isLastLoginWithToken());
                    }
                });
        } else {
            tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
        }
    }

    /**
     * \~chinese
     * 创建账号
     * @param username
     * @param password
     *
     * \~english
     * create an IM account on server
     * @param username
     * @param password
     */
    public void createAccount(String username, String password) throws HyphenateException {
        username = username.toLowerCase();

        Pattern pattern = Pattern.compile("^[a-zA-Z0-9_-]+$");
        boolean find = pattern.matcher(username).find();
        if (!find) {
            throw new HyphenateException(EMError.USER_ILLEGAL_ARGUMENT, "illegal user name");
        }
        EMAError error = emaObject.createAccount(username, password);
        handleError(error);
    }

    /**
     * \~chinese
     * 登录IM服务器
     *
     * @param id          用户id
     * @param password    用户密码
     * @param callback    EMCallback回调函数
     *
     * @throws IllegalArgumentException
     *
     * \~english
     * login to IM server
     *
     * @param id 		unique IM Login ID
     * @param password 	password for this IM ID
     * @param callback 	login callback
     *
     * @throws IllegalArgumentException
     */
    public void login(String id, String password, final EMCallBack callback) throws IllegalArgumentException {

        if (callback == null) {
            throw new IllegalArgumentException("callback is null!");
        }

        if (id == null || password == null || id.equals("") || password.equals("")) {
            throw new IllegalArgumentException("username or password is null or empty!");
        }

        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            throw new IllegalArgumentException("please setup your appkey either in AndroidManifest.xml or through the EMOptions");
        }

        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        id = id.toLowerCase();
        _login(id, password, callback, false, false);
    }

    public void loginWithToken(String username, String token, final EMCallBack callback) {
        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            throw new RuntimeException("please setup your appkey either in AndroidManifest.xml or through the EMOptions");
        }

        if (callback == null) {
            throw new IllegalArgumentException("callback is null!");
        }

        if (username == null || token == null || username.equals("") || token.equals("")) {
            throw new IllegalArgumentException("username or password is null or empty!");
        }

        if (!sdkInited) {
            callback.onError(EMError.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        username = username.toLowerCase();
        _login(username, token, callback, false, true);
    }

    /**
     * \~chinese
     * 退出
     *
     * @param unbindToken 是否解绑token
     *
     * \~english
     * logout
     *
     * @param unbindToken whether unbind token
     */
    public int logout(boolean unbindToken) {
        if (!emaObject.isLogout()) {
            boolean result = EMPushHelper.getInstance().unregister(unbindToken);
            if (!result) {
                return EMError.USER_UNBIND_DEVICETOKEN_FAILED;
            }
        } else {
            EMPushHelper.getInstance().unregister(false);
            EMLog.e(TAG, "already logout, skip unbind token");
        }

        logout();

        return EMError.EM_NO_ERROR;
    }

    /**
     * \~chinese
     * 同步登出聊天服务器
     * 如果需要异步请参考{@link EMClient#logout(EMCallBack callback)}
     *
     * \~english
     * logout hyphenate IM server synchronously
     * to use asynchronously API, see {@link EMClient#logout(EMCallBack callback)}
     */
    void logout() {
        EMLog.d(TAG, " SDK Logout");

        try {
            if (connectivityBroadcastReceiver != null) {
                mContext.unregisterReceiver(connectivityBroadcastReceiver);
            }
        } catch (Exception e) {
        }

        EMSessionManager.getInstance().clearLastLoginUser();
        EMSessionManager.getInstance().clearLastLoginToken();

        // ============ gaode code start ===============
        // EMGDLocation.getInstance().onDestroy();
        // ============ gaode code end ===============

        if (smartHeartbeat != null) {
            smartHeartbeat.stop();
        }

        if (wakeLock.isHeld()) {
            wakeLock.release();
        }

        // make sure we clear the username and pwd in a sync way
        if (emaObject != null) {
            emaObject.logout();
        }
        if (chatManager != null) {
            chatManager.onLogout();
        }
        if (groupManager != null) {
            groupManager.onLogout();
        }
        if (contactManager != null) {
            contactManager.onLogout();
        }
        if (chatroomManager != null) {
            chatroomManager.onLogout();
        }

        // clear all memory cache
        try {
            EMAdvanceDebugManager.getInstance().onDestroy();
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (EMChatConfigPrivate.isDebugTrafficMode()) {
            EMNetworkMonitor.stopTrafficStat();
        }
    }

    /**
     * \~chinese
     * 异步登出聊天服务器
     *
     * @param unbindToken 是否解绑token
     * @param callback    EMCallback回调
     *
     * \~english
     * logout hyphenate IM server synchronously
     *
     * @param unbindToken  whether unbind token
     * @param callback     EMCallback
     */
    public void logout(final boolean unbindToken, final EMCallBack callback) {
        new Thread() {
            @Override
            public void run() {
                int error = logout(unbindToken);

                if (error != EMError.EM_NO_ERROR) {
                    if (callback != null) {
                        callback.onError(error, "faild to unbind device token");
                    }
                } else {
                    if (callback != null) {
                        callback.onSuccess();
                    }
                }
            }
        }.start();
    }

    /**
     * \~chinese
     * 异步登出聊天服务器
     *
     * @param callback    EMCallback回调
     *
     * \~english
     * logout hyphenate IM server synchronously
     *
     * @param callback     EMCallback
     */
    void logout(final EMCallBack callback) {
        Thread logoutThread = new Thread() {
            @Override
            public void run() {
                if (callback != null) {
                    callback.onProgress(0, null);
                }

                logout();

                if (callback != null) {
                    callback.onSuccess();
                }
            }
        };
        logoutThread.setPriority(Thread.MAX_PRIORITY - 1);
        logoutThread.start();
    }

    /**
     * \~chinese
     * 更改appkey，注意只有在未登录状态才能修改appkey
     *
     * @param appkey
     *
     * \~english
     * change appkey. Can ONLY be changed if not logged in
     *
     * @param appkey
     */
    public void changeAppkey(String appkey) throws HyphenateException {
        EMAError error = emaObject.changeAppkey(appkey);
        if (error.errCode() == EMAError.EM_NO_ERROR) {
            this.getOptions().updatePath(appkey);
        }
        handleError(error);
    }

    /**
     * \~chinese
     * 添加EMConnectionListener
     *
     * @param listener
     *
     * \~english
     * add connection listener
     *
     * @param listener
     */
    public void addConnectionListener(final EMConnectionListener listener) {
        if (listener == null) {
            return;
        }

        synchronized (connectionListeners) {
            if (!connectionListeners.contains(listener)) {
                connectionListeners.add(listener);
            }
        }

        execute(new Runnable() {

            @Override
            public void run() {
                if (isConnected()) {
                    listener.onConnected();
                } else {
                    listener.onDisconnected(EMError.NETWORK_ERROR);
                }
            }
        });
    }

    public void addClientListener(final EMClientListener listener) {
        clientListeners.add(listener);
    }

    public void removeClientListener(final EMClientListener listener) {
        clientListeners.remove(listener);
    }

    /**
     * \~chinese
     * 删除EMConnectionListener
     *
     * @param listener
     *
     * \~english
     * remove connection listener
     *
     * @param listener
     */
    public void removeConnectionListener(final EMConnectionListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (connectionListeners) {
            connectionListeners.remove(listener);
        }
    }

    /**
     * \~chinese
     * 获取EMGroupManager
     *
     * @return EMGroupManager
     *
     * \~english
     * get group manager
     *
     * @return EMGroupManager
     */
    public EMGroupManager groupManager() {
        if (groupManager == null) {
            synchronized (EMClient.class) {
                if(groupManager == null) {
                    groupManager = new EMGroupManager(this, emaObject.getGroupManager());
                }
            }
        }
        return groupManager;
    }

    public EMPushManager pushManager() {
        if (pushManager == null) {
            synchronized (EMClient.class) {
                if(pushManager == null) {
                    pushManager = new EMPushManager(this, emaObject.getPushMnager());
                }
            }
        }
        return pushManager;
    }

    EMARTCConfigManager rtcConfigManager() {
       return emaObject.getRtcconfigManager();
    }


    /**
     * \~chinese
     * 获取EMChatRoomManager
     *
     * @return EMChatRoomManager
     *
     * \~english
     * get chatroom manager
     *
     * @return EMChatRoomManager
     */
    public EMChatRoomManager chatroomManager() {
        if (chatroomManager == null) {
            synchronized (EMClient.class) {
                if(chatroomManager == null) {
                    chatroomManager = new EMChatRoomManager(this, emaObject.getChatRoomManager());
                }
            }
        }
        return chatroomManager;
    }

    /**
     * \~chinese
     * 获取EMChatManager
     *
     * @return EMChatManager
     *
     * \~english
     * get chat manager
     *
     * @return EMChatManager
     */
    public EMChatManager chatManager() {
        if (chatManager == null) {
            synchronized (EMClient.class) {
                if(chatManager == null) {
                    chatManager = new EMChatManager(this, emaObject.getChatManager());
                }
            }
        }
        return chatManager;
    }


    /**
     * \~chinese
     * 获取EMUserInfoManager
     *
     * @return EMUserInfoManager
     *
     * \~english
     * get userInfo manager
     *
     * @return EMUserInfoManager
     */
    public EMUserInfoManager userInfoManager() {
        if (userInfoManager == null) {
            synchronized (EMClient.class) {
                if(userInfoManager == null) {
                    userInfoManager = new EMUserInfoManager(emaObject.getUserInfoManager());
                }
            }
        }
        return userInfoManager;
    }



    /**
     * \~chinese
     * 获取EMContactManager
     *
     * @return EMContactManager
     *
     * \~english
     * get contact manager
     *
     * @return EMContactManager
     */
    public EMContactManager contactManager() {
        if (contactManager == null) {
            synchronized (EMClient.class) {
                if(contactManager == null) {
                    contactManager = new EMContactManager(this, emaObject.getContactManager());
                }
            }
        }
        return contactManager;
    }

    /**
     * \~chinese
     * 获取EMCallManager
     *
     * @return EMCallManager
     *
     * \~english
     * get call manager
     *
     * @return EMCallManager
     */
    public EMCallManager callManager() {
        if (callManager == null) {
            callManager = new EMCallManager(this, emaObject.getCallManager());
        }
        return callManager;
    }

    /**
     * \~chinese
     * 获取EMConferenceManager
     *
     * @return EMConferenceManager
     *
     * \~english
     * get Conference manager
     *
     * @return EMConferenceManager
     */
    public EMConferenceManager conferenceManager(){
        if (conferenceManager == null) {
            conferenceManager = new EMConferenceManager(emaObject.getCallManager());
        }
        return conferenceManager;
    }

    public Context getContext() {
        return mContext;
    }

    /**
     * \~chinese
     * 获取当前登录用户的用户名
     *
     * @return 当前登录的用户
     *
     * \~english
     * get current logged in user id
     *
     * @return current logged in user
     */
    public synchronized String getCurrentUser() {
        if (EMSessionManager.getInstance().currentUser == null ||
                EMSessionManager.getInstance().currentUser.username == null ||
                EMSessionManager.getInstance().currentUser.username.equals("")) {
            return EMSessionManager.getInstance().getLastLoginUser();
        }
        return EMSessionManager.getInstance().currentUser.username;
    }

    /**
     * \~chinese
     * 根据用户名和密码获取token
     *
     * @param username 用户名
     * @param password 密码
     * @param callBack 结果回调
     *
     * \~english
     * Fetch token by username and password.
     *
     * @param callBack result callback
     */
    public void getUserTokenFromServer(final String username, final String password, final EMValueCallBack<String> callBack) {
        execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                final String token = emaObject.getUserTokenFromServer(username, password, error);

                if (callBack == null) {
                    return;
                }

                if (error.errCode() == EMError.EM_NO_ERROR) {
                    callBack.onSuccess(token);
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~chinese
     * 返回是否登录过
     * 登录成功过没调logout方法，这个方法的返回值一直是true
     * 如果需要判断当前是否连接到服务器，请使用{@link #isConnected()}方法
     *
     * <pre>
     * if(EMClient.getInstance().isLoggedInBefore()){
     *     // enter main activity
     * }else{
     *     // enter login activity
     * }
     * </pre>
     *
     * @return
     *
     * \~english
     * used to check if user has been logged in before and did not logout
     * if you need check if connected to server, please use {@link #isConnected()}
     *
     * <pre>
     * if(EMClient.getInstance().isLoggedInBefore()){
     *     // enter main activity
     * }else{
     *     // enter login activity
     * }
     * </pre>
     *
     * @return
     */
    public boolean isLoggedInBefore() {
        EMSessionManager sessionMgr = EMSessionManager.getInstance();
        String user = sessionMgr.getLastLoginUser();
        String pwd = sessionMgr.getLastLoginPwd();
        String token = sessionMgr.getLastLoginToken();

        if (user != null  && !user.isEmpty() &&
                ((pwd != null   && !pwd.isEmpty()) ||
                 (token != null && !token.isEmpty())) ) {
            return true;
        }

        return false;
    }


    /**
     * \~chinese
     * 检查是否连接到聊天服务器
     *
     * @return
     *
     * \~english
     * check if connected to server.
     *
     * @return
     */
    public boolean isConnected() {
        return emaObject.isConnected();
    }

    /**
     * \~chinese
     * debugMode == true 时，sdk 会在log里输出调试信息
     *
     * @param debugMode
     *
     * \~english
     * SDK will out put debug info if debugMode = true
     *
     * @param debugMode
     */
    public void setDebugMode(boolean debugMode) {
        if (sdkInited) {
            //if had set debugmode, use it
            String mode = EMAdvanceDebugManager.getInstance().getDebugMode();
            if (mode != null)
                debugMode = Boolean.parseBoolean(mode);
        }
        EMLog.debugMode = debugMode;
        getChatConfigPrivate().setDebugMode(debugMode);
    }

    /**
     * \~chinese
     * 更新当前用户的nickname
     * 此方法主要为了在苹果推送时能够推送nick而不是userid 一般可以在登陆成功后从自己服务器获取到个人信息，然后拿到nick更新到环信服务器
     * 并且，在个人信息中如果更改个人的昵称，也要把环信服务器更新下nickname 防止显示差异
     *
     * @param nickname 昵称
     * @deprecated use {@link EMPushManager#updatePushNickname(String)} instead
     */
    @Deprecated
    public boolean updateCurrentUserNick(String nickname) throws IllegalArgumentException, HyphenateException {
        return pushManager().updatePushNickname(nickname);
    }

    /**
     * \~chinese
     * 上传本地的log
     *
     * \~english
     * upload local log file
     */
    public void uploadLog(EMCallBack callback) {
        chatManager().emaObject.uploadLog();
    }

    /**
     * \~chinese
     * 获取Robot列表
     *
     * @return list of EMContact
     * @throws HyphenateException
     *
     * \~english
     * get robot list
     * @throws HyphenateException
     */
    public List<EMContact> getRobotsFromServer() throws HyphenateException {
        return EMExtraService.getInstance().getRobotsFromServer();
    }

    /**
     * \~chinese
     * 获取EMOptions
     *
     * \~english
     * get EMOptions
     */
    public EMOptions getOptions() {
        return mChatConfigPrivate.getOptions();
    }


    /**
     * \~chinese
     * 压缩log文件，并返回压缩后的文件路径
     *
     * @return compressed gz file
     * @throws HyphenateException \~english
     *                            compress log files, and return the compressed file
     * @throws HyphenateException
     *
     * \~english
     * compress log folder into zip file, return zip file path
     *
     * @return compressed gz file
     * @throws HyphenateException
     *                            compress log files, and return the compressed file
     * @throws HyphenateException
     */
    public String compressLogs() throws HyphenateException {
        EMAError error = new EMAError();
        String path = emaObject.compressLogs(error);
        handleError(error);
        return path;
    }

    /**
     * \~chinese
     * 添加多设备监听的接口
     * @param listener
     *
     * \~english
     * add multiple devices listener
     * @param listener
     */
    public void addMultiDeviceListener(EMMultiDeviceListener listener) {
        multiDeviceListeners.add(listener);
    }

    /**
     * \~chinese
     * 删除多设备监听的接口
     * @param listener
     *
     * \~english
     * remove multiple devices listener
     * @param listener
     */
    public void removeMultiDeviceListener(EMMultiDeviceListener listener) {
        multiDeviceListeners.remove(listener);
    }

    /**
     * \~chinese
     * 获取账号名下登陆的在线设备列表
     * @return 在线设备列表
     * @throws HyphenateException
     *
     * \~english
     * multi-device: Get all the information about the logged in devices
     */
    public List<EMDeviceInfo> getLoggedInDevicesFromServer(String username, String password) throws HyphenateException {
        EMAError error = new EMAError();
        List<EMADeviceInfo> devices = emaObject.getLoggedInDevicesFromServer(username, password, error);
        handleError(error);
        List<EMDeviceInfo> result = new ArrayList<>();
        for (EMADeviceInfo info : devices) {
            result.add(new EMDeviceInfo(info));
        }
        return result;
    }

    /**
     * \~chinese
     * 根据设备ID，将该设备下线, 设备ID：{@link EMDeviceInfo#getResource()}
     *
     * @param username  账户名称
     * @param password  该账户密码
     * @param resource  设备ID
     * @throws HyphenateException
     *
     * \~english
     * multi-device: Forced to logout the specified logged in device, resource: {@link EMDeviceInfo#getResource()}
     *
     * @param username
     * @param password
     * @param resource
     * @throws HyphenateException
     */
    public void kickDevice(String username, String password, String resource) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickDevice(username, password, resource, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 将该账号下的所有设备都踢下线
     * @throws HyphenateException
     *
     * \~english
     * multi-device: Forced to logout all logged in devices
     * @throws HyphenateException
     */
    public void kickAllDevices(String username, String password) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.kickAllDevices(username, password, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 上传FCM token至服务器
     * <p>
     * token可以被上传的前提条件有：
     * 1.被上传的token不为空
     * 2.当前已有用户登录
     * 3.当前设备支持google推送
     * 4.google推送类型为FCM {@link com.hyphenate.push.EMPushType#FCM}
     * 设置推送类型为FCM的接口为{@link EMOptions#setFCMNumber(String)},即设置了FCM number则推送类型为FCM
     * <p>
     * \~english
     * Upload the FCM token to hyphenate server
     * <p>
     * The token will be uploaded when all the following conditions are met:
     * 1.The token is not empty.
     * 2.User logged in
     * 3.Device support google play service
     * 4.Push type is FCM {@link com.hyphenate.push.EMPushType#FCM},
     * you can set with {@link EMOptions#setFCMNumber(String)}
     *
     * @param fcmToken
     */
    public void sendFCMTokenToServer(String fcmToken) {
        EMLog.i(TAG, "sendFCMTokenToServer: " + fcmToken);
        if (TextUtils.isEmpty(fcmToken)) {
            return;
        }
        // Save the fcm token native.
        String savedToken = EMPushHelper.getInstance().getFCMPushToken();
        if (!fcmToken.equals(savedToken)) {
            EMPushHelper.getInstance().setFCMPushToken(fcmToken);
        }
        // If no user logged in, stop upload the fcm token.
        String userName = getCurrentUser();
        if (TextUtils.isEmpty(userName)) {
            EMLog.i(TAG, "No user login currently, stop upload the token.");
            return;
        }
        // let's check if the push service available
        EMPushType pushType = EMPushHelper.getInstance().getPushType();
        EMLog.i(TAG, "pushType: " + pushType);
        if (pushType == EMPushType.FCM) {
            EMPushHelper.getInstance().onReceiveToken(pushType, fcmToken);
        }
    }

    /**
     * \~chinese
     * 发送华为推送 token 到服务器
     * @param appId 华为 appId
     * @param token 华为推送 token
     *
     * \~english
     * Send Huawei devices token to server
     * @param appId Huawei appId
     * @param token Huawei device token
     *
     * @Deprecated Use {@link #sendHMSPushTokenToServer(String)} for instead.
     */
    @Deprecated
    public void sendHMSPushTokenToServer(String appId, String token){
        sendHMSPushTokenToServer(token);
    }

    public void sendHMSPushTokenToServer(String token){
        if (EMPushHelper.getInstance().getPushType() == EMPushType.HMSPUSH) {
            EMPushHelper.getInstance().onReceiveToken(EMPushType.HMSPUSH, token);
        }
    }

	//---------------------------private methods ---------------------------------

	private void initManagers(){
		// invoke constructor before login, to listener PB event
		EMHttpClient.getInstance().onInit(mChatConfigPrivate);
		chatManager();
		contactManager();
		groupManager();
		chatroomManager();

		setNatvieNetworkCallback();
	}

	void _login(final String username, final String code, final EMCallBack callback, final boolean autoLogin, final boolean loginWithToken) {
        if (getChatConfigPrivate() == null || sdkInited == false) {
            callback.onError(EMError.GENERAL_ERROR, "");
            return;
        }

        EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid());

        // convert to lowercase
        execute(new Runnable() {

            @Override
            public void run() {
                EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid() + " threadName:" + Thread.currentThread().getName() + " ID:" + Thread.currentThread().getId());

                if (username == null) {
                    callback.onError(EMError.INVALID_USER_NAME, "Invalid user name");
                    return;
                }

                EMAError error = new EMAError();
                emaObject.login(username, code, autoLogin, loginWithToken, error);

                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    EMSessionManager.getInstance().setLastLoginUser(username);
                    EMAError emaError = new EMAError();
                    if (emaError.errCode() == EMAError.EM_NO_ERROR) {
                        if (mChatConfigPrivate.getUsingSQLCipher() || loginWithToken) {
                            String token = emaObject.getUserToken(false, emaError);
                            EMSessionManager.getInstance().setLastLoginToken(token);
                            EMSessionManager.getInstance().setLastLoginWithToken(true);
                            EMSessionManager.getInstance().clearLastLoginPwd();
                        } else {
                            EMSessionManager.getInstance().setLastLoginPwd(code);
                            EMSessionManager.getInstance().setLastLoginWithToken(false);
                            EMSessionManager.getInstance().clearLastLoginToken();
                        }
                    }

                    onNewLogin();
                    EMPushHelper.getInstance().register();
                    // set EMConferenceManager parameters, lite jar build need to remove this line
//                    conferenceManager().set(getAccessToken(), getOptions().getAppKey(), getCurrentUser());
                    callback.onSuccess();
                } else {
                    callback.onError(error.errCode(), error.errMsg());
                }

                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    if (getOptions().isEnableStatistics()) {
                        setPresence(getLocationString(autoLogin));
                    } else {
                        EMLog.d(TAG, "statistics is not enabled");
                    }
                }

                if (error.errCode() == EMAError.USER_AUTHENTICATION_FAILED) {
                    EMSessionManager.getInstance().clearLastLoginPwd();
                    EMSessionManager.getInstance().clearLastLoginToken();
                }
            }
        });
    }

    public boolean isFCMAvailable() {
        return EMPushHelper.getInstance().getPushType() == EMPushType.FCM;
    }

    void onNewLogin() {
        EMLog.d(TAG, "on new login created");

        String username = EMSessionManager.getInstance().getLastLoginUser();

        PathUtil.getInstance().initDirs(getChatConfigPrivate().getAppKey(), username, mContext);

        EMDBManager.initDB(username, mChatConfigPrivate);

        EMDBManager.getInstance().setDBMigrateListener(new EMDBManager.DBMigrateListener() {
            @Override
            public void onMigrationFinish(final boolean success) {
                execute(new Runnable() {
                    @Override
                    public void run() {
                        if (clientListeners != null) {
                            synchronized (clientListeners) {
                                for (EMClientListener listener : clientListeners) {
                                    listener.onMigrate2x(success);
                                }
                            }
                        }
                    }
                });
            }
        });

        EMDBManager.getInstance().tryMigrationToOneSDK();

        EMAdvanceDebugManager.getInstance().onInit(mChatConfigPrivate);

        mContext.registerReceiver(connectivityBroadcastReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));

        if (smartHeartbeat == null) {
            smartHeartbeat = EMSmartHeartBeat.create(mContext);
        }

        if (getChatConfigPrivate().emaObject.hasHeartBeatCustomizedParams()) {
            smartHeartbeat.setCustomizedParams(getChatConfigPrivate().emaObject.getWifiHeartBeatCustomizedParams(),
                    getChatConfigPrivate().emaObject.getMobileHeartBeatCustomizedParams());
        }
        smartHeartbeat.onInit();
        if (getOptions().getFixedInterval() != -1) {
            smartHeartbeat.setFixedInterval(getOptions().getFixedInterval());
        }
    }

    /**
     * 获取身份认证权限 在上传下载附件（语音，图片，文件等）时必须添加到请求header中 当出现任何异常时将返回null，
     * 可通过判断是否为null来检测是否有问题 如果为null，在打开EMLog日志时，是可以看到异常原因。
     *
     * @return
     */
    public String getAccessToken() {
        return getChatConfigPrivate().getAccessToken();
    }

    /**
     * \~chinese
     * 判断SDK是否已经初始化完毕
     * @return  SDK是否已经初始化完毕
     *
     * \~english
     * Determine if the SDK has been initialized
     * @return  Whether the SDK has been initialized
     */
    public boolean isSdkInited() {
        return sdkInited;
    }

    private boolean _loadLibrary(final String library, boolean trace) {
        try {
            ReLinker.loadLibrary(mContext, library);
            return true;
        } catch (Throwable e) {
            if (trace) {
                e.printStackTrace();
            }
        }
        return false;
    }

    private boolean _loadLibrary(final String library) {
        return _loadLibrary(library, true);
    }

    private void loadLibrary() {
        if (libraryLoaded == false) {
            //if (_loadLibrary("sqlcipher", false) || _loadLibrary("sqlite")) {}
            _loadLibrary("sqlite");
//            _loadLibrary("hyphenate_av_recorder");
            ReLinker.loadLibrary(mContext, "hyphenate");
            libraryLoaded = true;
        }
    }

    class MyConnectionListener extends EMAConnectionListener {

        @Override
        public void onConnected() {
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        try {
                            for (EMConnectionListener listener : connectionListeners) {
                                listener.onConnected();
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onDisconnected(final int errCode) {
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        if (errCode == EMError.USER_REMOVED) {
                            EMSessionManager.getInstance().clearLastLoginUser();
                            EMSessionManager.getInstance().clearLastLoginToken();
                            EMSessionManager.getInstance().clearLastLoginPwd();
                        } else if ((errCode == EMError.USER_LOGIN_ANOTHER_DEVICE) ||
                                (errCode == EMError.USER_BIND_ANOTHER_DEVICE) ||
                                (errCode == EMError.SERVER_SERVICE_RESTRICTED) ||
                                (errCode == EMError.USER_LOGIN_TOO_MANY_DEVICES) ||
                                (errCode == EMError.USER_KICKED_BY_CHANGE_PASSWORD) ||
                                (errCode == EMError.USER_KICKED_BY_OTHER_DEVICE)) {
                            EMSessionManager.getInstance().clearLastLoginToken();
                            EMSessionManager.getInstance().clearLastLoginPwd();
                        }

                        try {
                            for (EMConnectionListener listener : connectionListeners) {
                                listener.onDisconnected(errCode);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
				}
			});
		}
	}

    class MyMultiDeviceListener extends EMAMultiDeviceListener {
        @Override
        public void onContactEvent(final int event, final String target, final String ext) {
            EMLog.d(TAG, "onContactEvent:" + event + " target:" + target + " ext:" + ext);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (EMMultiDeviceListener listener : multiDeviceListeners) {
                                listener.onContactEvent(event, target, ext);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onGroupEvent(final int event, final String target, final List<String> usernames) {
            EMLog.d(TAG, "onGroupEvent:" + event + " target:" + target + " usernames:" + usernames);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (EMMultiDeviceListener listener : multiDeviceListeners) {
                                listener.onGroupEvent(event, target, usernames);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }
    }

	CryptoUtils getCryptoUtils() {
		return cryptoUtils;
	}

    void execute(Runnable runnable) {
        executor.execute(runnable);
    }

    void executeOnMainQueue(Runnable runnable) {
        mainQueue.submit(runnable);
    }

    void executeOnSendQueue(Runnable runnable) {
        sendQueue.submit(runnable);
    }

    // TODO: temp
    public EMChatConfigPrivate getChatConfigPrivate() {
        return mChatConfigPrivate;
    }

    void setNatvieNetworkCallback() {
        EMANetCallback callback = new EMANetCallback() {

            @Override
            public int getNetState() {
                if (!NetUtils.hasDataConnection(mContext)) {
                    return EMANetwork.NETWORK_NONE.ordinal();
                } else {
                    if (NetUtils.isWifiConnected(mContext)
                            || NetUtils.isOthersConnected(mContext)) {
                        return EMANetwork.NETWORK_WIFI.ordinal();
                    } else if (NetUtils.isMobileConnected(mContext)) {
                        return EMANetwork.NETWORK_MOBILE.ordinal();
                    } else if (NetUtils.isEthernetConnected(mContext)) {
                        return EMANetwork.NETWORK_CABLE.ordinal();
                    } else {
                        return EMANetwork.NETWORK_NONE.ordinal();
                    }
                }
            }
        };

        mChatConfigPrivate.emaObject.setNetCallback(callback);
    }

    /**
     * 设置应用的加密实现 如果未设置，SDK 会使用内置的加密算法
     * <p>
     * set the encrypt provider.
     *
     * @param provider
     */
    void setEncryptProvider(EMEncryptProvider provider) {
        this.encryptProvider = provider;
    }

    /**
     * 获取EncryptProvider 如果未设置, 将返回sdk 内置的encrypt provider
     *
     * @return
     */
    EMEncryptProvider getEncryptProvider() {
        if (encryptProvider == null) {
            EMLog.d(TAG, "encrypt provider is not set, create default");
            encryptProvider = new EMEncryptProvider() {

                public byte[] encrypt(byte[] input, String username) {
                    try {
                        return cryptoUtils.encrypt(input);
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

                public byte[] decrypt(byte[] input, String username) {
                    try {
                        return cryptoUtils.decrypt(input);
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

            };
        }
        return encryptProvider;
    }

    boolean sendPing(boolean waitPong, long timeout) {
        return emaObject.sendPing(waitPong, timeout);
    }

    void forceReconnect() {
        EMLog.d(TAG, "forceReconnect");
        disconnect();
        reconnect();
    }

    void reconnect() {
        EMLog.d(TAG, "reconnect");
        wakeLock.acquire();
        emaObject.reconnect();
        if (wakeLock.isHeld()) {
            wakeLock.release();
        }
    }

    void disconnect() {
        EMLog.d(TAG, "disconnect");
        emaObject.disconnect();
    }

    private void handleError(EMAError error) throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

    /**
     * the connectivity change listener
     * it will only interrupt the reconnection thread and reset the attempts times
     */
    private BroadcastReceiver connectivityBroadcastReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(final Context context, Intent intent) {
            String action = intent.getAction();
            if (!action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
                EMLog.d(TAG, "skip no connectivity action");
                return;
            }

            EMLog.d(TAG, "connectivity receiver onReceiver");

            EMANetwork networkType;
            NetUtils.Types types = NetUtils.getNetworkTypes(getContext());
            switch (types) {
                case WIFI:
                case OTHERS:
                    networkType = EMANetwork.NETWORK_WIFI;
                    break;
                case MOBILE:
                    networkType = EMANetwork.NETWORK_MOBILE;
                    break;
                case ETHERNET:
                    networkType = EMANetwork.NETWORK_CABLE;
                    break;
                case NONE:
                default:
                    networkType = EMANetwork.NETWORK_NONE;
                    break;
            }

            boolean prevNetworkAvailable = currentNetworkType != EMANetwork.NETWORK_NONE;
            boolean currentNetworkAvailable = networkType != EMANetwork.NETWORK_NONE;
            currentNetworkType = networkType;
            if (prevNetworkAvailable == currentNetworkAvailable) {
                // Network availability no change, return.
                EMLog.i(TAG, "Network availability no change, just return. " + currentNetworkType + ", but check ping");
                execute(new Runnable() {
                    @Override
                    public void run() {
                        if (smartHeartbeat != null) {
                            smartHeartbeat.sendPingCheckConnection();
                        }
                    }
                });
                return;
            }

            EMLog.i(TAG, "Network availability changed, notify... " + currentNetworkType);

            execute(new Runnable() {

                @Override
                public void run() {
                    emaObject.onNetworkChanged(currentNetworkType);
                }


            });
        }
    };

    void onNetworkChanged() {
        try {
            if (NetUtils.isWifiConnected(mContext)
                    || NetUtils.isOthersConnected(mContext)) {
                EMLog.d(TAG, "has wifi connection");
                currentNetworkType = EMANetwork.NETWORK_WIFI;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_WIFI);
                return;
            }

            if (NetUtils.isMobileConnected(mContext)) {
                EMLog.d(TAG, "has mobile connection");
                currentNetworkType = EMANetwork.NETWORK_MOBILE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_MOBILE);
                return;
            }

            if (NetUtils.isEthernetConnected(mContext)) {
                EMLog.d(TAG, "has ethernet connection");
                currentNetworkType = EMANetwork.NETWORK_CABLE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_CABLE);
                return;
            }
            currentNetworkType = EMANetwork.NETWORK_NONE;
            EMLog.d(TAG, "no data connection");
            emaObject.onNetworkChanged(EMANetwork.NETWORK_NONE);
            return;
        } catch (Exception e) {
            e.printStackTrace();
            return;
        }
    }

    void onNetworkChanged(EMANetwork network) {
        emaObject.onNetworkChanged(network);
    }

    private AppStateListener appStateListener;
    private List<Activity> resumeActivityList = new ArrayList<>();

    void setAppStateListener(AppStateListener appStateListener) {
        this.appStateListener = appStateListener;
    }

    @TargetApi(14)
    private void registerActivityLifecycleCallbacks() {
        if (Utils.isSdk14()) {
            Object lifecycleCallbacks = new ActivityLifecycleCallbacks() {

                @Override
                public void onActivityStopped(Activity activity) {
                    resumeActivityList.remove(activity);
                    if (resumeActivityList.isEmpty()) {
                        if (appStateListener != null) appStateListener.onBackground();
                    }
                }

                @Override
                public void onActivityResumed(Activity activity) {
                }

                @Override
                public void onActivityCreated(Activity activity, Bundle savedInstanceState) {

                }

                @Override
                public void onActivityStarted(Activity activity) {
                    if(!resumeActivityList.contains(activity)) {
                        resumeActivityList.add(activity);
                        if(resumeActivityList.size() == 1) {
                            if (appStateListener != null) appStateListener.onForeground();
                        }
                    }
                }

                @Override
                public void onActivitySaveInstanceState(Activity activity, Bundle outState) {

                }

                @Override
                public void onActivityPaused(Activity activity) {

                }

                @Override
                public void onActivityDestroyed(Activity activity) {

                }

            };
            ((Application) mContext).registerActivityLifecycleCallbacks((ActivityLifecycleCallbacks) lifecycleCallbacks);
        }
    }

    interface AppStateListener {
        void onForeground();

        void onBackground();
    }

    void setPresence(final String location) {
        if(TextUtils.isEmpty(location)) {
            return;
        }
        execute(new Runnable() {
            @Override
            public void run() {
                emaObject.setPresence(location);
            }
        });
    }

    private String getLocationString(boolean autoLogin) {
        JSONObject jObj = null;
        if (!autoLogin) {
            jObj = getDeviceInfo();
        }
        return jObj == null ? "": jObj.toString();
    }

    public JSONObject getDeviceInfo() {
        JSONObject jobj = new JSONObject();
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(mContext);
        String deviceId = deviceFactory.getDeviceUuid().toString();
        try {
            jobj.put("deviceid", deviceId);
            jobj.put("app-id", mContext.getPackageName());
            jobj.put("hid", EMClient.getInstance().getCurrentUser());
            jobj.put("os", "android");
            jobj.put("os-version", Build.VERSION.RELEASE);
            jobj.put("manufacturer", Build.MANUFACTURER);
            jobj.put("model", Build.MODEL);
        } catch (JSONException e) {
            EMLog.d(TAG, e.getMessage());
        }

        return jobj;
    }

    // For service check.
    private boolean duringChecking = false;

    /**
     * \~chinese
     * 服务诊断接口,流程如下:
     * 1.校验用户名和密码
     * 2.从服务端获取DNS列表
     * 3.从服务端获取Token
     * 4.连接IM服务器
     * 5.断开连接(如果检查前已经有账户登录,则不执行该步骤)
     * <p>
     * 如果在诊断过程中产生了错误,该流程将会被打断.
     *
     * @param username 用于服务诊断的用户名,如果已有账户登录,该用户名会被替换为已登录账户的用户名,以防止更改当前已登录账户
     *                 的信息,比如Token等...
     * @param password 密码,如果已有账户登录,该密码会被替换为已登录账户的密码.
     * @param listener 诊断结果回调
     * <p>
     * \~english
     * Service check interface, here is the schedule:
     * 1.Validate the username and password user input.
     * 2.Get dns list from server.
     * 3.Get token from server.
     * 4.Connect to the im server.
     * 5.logout.(If call this method after user logged in, this step will not in schedule.)
     * <p>
     * If there is a error occurred during checking, this schedule will be broken.
     *
     * @param username User inputted for service check. If account has logged in before, this username
     *                 would be changed to the logged in username to avoid update the current logged in
     *                 account data.
     * @param password The password for username. if this is a logged in service check, the password would
     *                 changed to the logged in password.
     * @param listener A listener for service check results callback.
     */
    public void check(String username, String password, final CheckResultListener listener) {
        // Already during checking, just return.
        if (duringChecking) {
            EMLog.i("EMServiceChecker", "During service checking, please hold on...");
            return;
        }

        duringChecking = true;

        // If is logged in before, the username must be current user.
        if (isLoggedInBefore()) {
            username = getCurrentUser();
            EMSessionManager sessionMgr = EMSessionManager.getInstance();
            password = sessionMgr.getLastLoginPwd();
        }

        final String finalUser = username;
        final String finalPwd = password;

        new Thread(new Runnable() {
            @Override
            public void run() {
                /**
                 * Contains account-validation check, get-dns check, get-token check, login check.
                 * So the {@link EMAChatClient.CheckResultListener#onResult(int, int, String)} callback
                 * will be called four times.
                 */
                emaObject.check(finalUser, finalPwd, new EMAChatClient.CheckResultListener() {
                    @Override
                    public void onResult(final int type, final int result, final String desc) {
                        EMLog.i("EMServiceChecker", "type: " + type + ", result: " + result + ", desc: " + desc);
                        // Notify user once per callback.
                        notifyCheckResult(listener, type, result, desc);

                        // If occur a error during four checks, return.
                        if (result != EMError.EM_NO_ERROR) {
                            duringChecking = false;
                            return;
                        }

                        // If login successfully, logout
                        if (type == EMCheckType.DO_LOGIN) {
                            checkLogout(listener);
                        }
                    }
                });
            }
        }).start();
    }

    private void checkLogout(CheckResultListener listener) {
        // If is not a logged in service check, logout and notify the user.
        if (!isLoggedInBefore()) {
            logout();
            notifyCheckResult(listener, EMCheckType.DO_LOGOUT, EMError.EM_NO_ERROR, "");
        }

        duringChecking = false;
    }

    private void notifyCheckResult(CheckResultListener listener, @EMCheckType.CheckType int type, int result, String desc) {
        if (listener == null) return;
        listener.onResult(type, result, desc);
    }

    public interface CheckResultListener {
        void onResult(@EMCheckType.CheckType int type, int result, String desc);
    }

}
