package com.yy.mobile.framework.revenuesdk.payservice.revenueservice;

import android.os.Handler;
import android.os.Looper;

import com.yy.mobile.framework.revenuesdk.baseapi.RetryPolicy;
import com.yy.mobile.framework.revenuesdk.baseapi.log.RLog;

/**
 * 重试代码，（超时机制由Service保证）
 * by xiongsheng
 */
public abstract class DefaultRetryPolicy implements RetryPolicy {

    private static final String TAG = "DefaultRetryPolicy";
    private Handler mHander;
    /**
     * Request当前重试次数
     */
    private int mCurrentRetryCount;

    /**
     * Request最多重试次数
     */
    private final int mMaxNumRetries;

    private static final int FINISH = 1;
    private static final int RUNNING = 0;
    private static final int CANCEL = -1;
    private volatile int mStatus = RUNNING;
    private final int mIntervalMs;
    private final int mRetryType; // 重试类型：0 默认等间隔， 1 先等间隔再指数延长

    /**
     * 默认的重试次数(1次,不进行请求重试)
     */
    public static final int DEFAULT_MAX_RETRIES = 1;
    /**
     * 默认的超时时间
     */
    public static final int DEFAULT_TIME_OUT_MS = 2500;
    /**
     * 默认间隔时间
     */
    public static final int DEFAULT_INTERVAL_MS = 3000;

    /**
     * 默认重试类型
     */
    public static final int DEFAULT_RETRY_TYPE = 0;

    /**
     * Request当前超时时间
     */
    private int mTimeoutMs;

    /**
     * Request的默认重试策略构造函数
     */
    public DefaultRetryPolicy() {
        this(new Handler(Looper.getMainLooper()), DEFAULT_TIME_OUT_MS, DEFAULT_MAX_RETRIES, DEFAULT_INTERVAL_MS,
                DEFAULT_RETRY_TYPE);
    }

    /**
     * 开发者自定制Request重试策略构造函数
     *
     * @param timeOutMs     超时时间
     * @param maxNumRetries 最大重试次数
     */
    public DefaultRetryPolicy(Handler handler, int timeOutMs, int maxNumRetries,
                              int intervalMs, int retryType) {
        mHander = handler;
        mTimeoutMs = timeOutMs;
        mMaxNumRetries = maxNumRetries;
        mIntervalMs = intervalMs;
        mRetryType = retryType;
    }

    @Override
    public void retry() {
        RLog.info(TAG, "retry mTimeoutMs=" + mTimeoutMs + ", mMaxNumRetries=" + mMaxNumRetries +
                ",mIntervalMs=" + mIntervalMs);
        if (mStatus != RUNNING) {
            return;
        }
        if (hasAttemptRemaining()) {
            if (mRetryType == 1) {
                // 添加重试次数
                mCurrentRetryCount++;
                if (mCurrentRetryCount <= (int) (mMaxNumRetries / 2)) { //一半次数 mIntervalMs
                    mHander.postDelayed(() -> {
                        RLog.info(TAG, "retry currentRetryCount=" + mCurrentRetryCount);
                        onRetry();
                    }, mIntervalMs);
                } else {
                    mHander.postDelayed(() -> {
                        RLog.info(TAG, "retry currentRetryCount=" + mCurrentRetryCount);
                        onRetry();
                    }, 60000); //一半次数 1分钟
                }
            } else {
                // 添加重试次数
                mCurrentRetryCount++;
                mHander.postDelayed(() -> {
                    RLog.info(TAG, "retry currentRetryCount=" + mCurrentRetryCount);
                    onRetry();
                }, mIntervalMs);
            }

        } else {
            retryCountExhaust();
            RLog.info(TAG, "retryCountExhaust");
        }
    }

    public abstract void onRetry();

    public void call() {
        mCurrentRetryCount = 1;
        mStatus = RUNNING;
        mHander.post(() -> {
            onRetry();
        });
    }

    @Override
    public void cancel() {
        mStatus = CANCEL;
        RLog.info(TAG, "cancel");
    }

    public void done() {
        mStatus = FINISH;
        RLog.info(TAG, "done");
    }

    /**
     * 判断当前Request的重试次数是否超过最大重试次数
     */
    private boolean hasAttemptRemaining() {
        return mCurrentRetryCount < mMaxNumRetries;
    }

    public int getCurrentRetryCount() {
        return mCurrentRetryCount;
    }
}