package com.yy.pushsvc.util;

import android.util.Log;

import com.yy.pushsvc.executor.IPushTaskExecutor;
import com.yy.pushsvc.executor.IQueueTaskExecutor;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.RunnableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

public class PushThreadPool {
    private static PushThreadPool sPushThreadPool;
    private ExecutorService executorService;
    private ExecutorService singleExecutorService;
    private ScheduledExecutorService scheduledExecutorService;
    private IPushTaskExecutor taskExecutor;
    private IQueueTaskExecutor singleTaskExecutor;
    private ScheduledExecutorService spareExecutor;

    private PushThreadPool() {
        if (ExecutorProvider.getPushTaskExecutor() == null) {
            this.executorService = Executors.newFixedThreadPool(5);
            this.singleExecutorService = Executors.newSingleThreadExecutor();
            this.scheduledExecutorService = Executors.newScheduledThreadPool(1, new ThreadFactory() {
                public Thread newThread(Runnable r) {
                    Thread thread = new Thread(r);
                    thread.setPriority(1);
                    return thread;
                }
            });
        } else {
            this.taskExecutor = ExecutorProvider.getPushTaskExecutor();
            this.singleTaskExecutor = this.taskExecutor.createAQueueExcuter();
            if (this.singleTaskExecutor == null) {
                this.singleExecutorService = Executors.newSingleThreadExecutor();
            }
        }
    }

    public static PushThreadPool getPool() {
        if (sPushThreadPool == null) {
            synchronized (PushThreadPool.class) {
                if (sPushThreadPool == null) {
                    sPushThreadPool = new PushThreadPool();
                }
            }
        }
        return sPushThreadPool;
    }

    private ScheduledExecutorService getSpareExecutor() {
        if (this.spareExecutor != null) {
            return this.spareExecutor;
        } else {
            synchronized (this) {
                if (this.spareExecutor != null) {
                    return this.spareExecutor;
                } else {
                    this.spareExecutor = Executors.newScheduledThreadPool(1);
                    return this.spareExecutor;
                }
            }
        }
    }

    public void execute(Runnable runnable) {
        if (this.taskExecutor != null) {
            try {
                this.taskExecutor.execute(runnable, 0L);
            } catch (Throwable var3) {
                this.getSpareExecutor().execute(runnable);
            }
        } else {
            this.executorService.execute(runnable);
        }

    }

    public void executeQueue(Runnable runnable) {
        if (this.singleTaskExecutor != null) {
            try {
                this.singleTaskExecutor.execute(runnable, 0L);
            } catch (Throwable var3) {
                this.getSpareExecutor().execute(runnable);
            }
        } else {
            this.singleExecutorService.execute(runnable);
        }

    }

    public <T> Future<T> submit(Callable<T> callable) {
        RunnableFuture<T> ftask = new FutureTask(callable);
        this.execute(ftask);
        return ftask;
    }

    public <T> Future<T> submitQueue(Callable<T> callable) {
        RunnableFuture<T> ftask = new FutureTask<>(callable);
        this.executeQueue(ftask);
        return ftask;
    }

    public void shutdownNow() {
        if (this.executorService != null) {
            this.executorService.shutdownNow();
        }

        if (this.singleExecutorService != null) {
            this.singleExecutorService.shutdownNow();
        }

        if (this.scheduledExecutorService != null) {
            this.scheduledExecutorService.shutdownNow();
        }
        if (this.spareExecutor != null) {
            this.spareExecutor.shutdownNow();
            this.spareExecutor = null;
        }
    }

    public void shutdown() {
        if (this.executorService != null) {
            this.executorService.shutdown();
        }

        if (this.singleExecutorService != null) {
            this.singleExecutorService.shutdown();
        }

        if (this.scheduledExecutorService != null) {
            this.scheduledExecutorService.shutdown();
        }
        if (this.spareExecutor != null) {
            this.spareExecutor.shutdownNow();
            this.spareExecutor = null;
        }
    }

    public void execute(Runnable run, long delay) {
        try {
            if (this.taskExecutor != null) {
                try {
                    this.taskExecutor.execute(run, delay);
                } catch (Throwable var) {
                    this.getSpareExecutor().schedule(run, delay, TimeUnit.MILLISECONDS);
                }
            } else {
                this.scheduledExecutorService.schedule(run, delay, TimeUnit.MILLISECONDS);
            }
        } catch (Throwable var) {
            Log.e("PushThreadPool", "execute: " + var);
        }

    }

}
