package com.yy.pushsvc.simplify;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;

import com.yy.pushsvc.thirdparty.ThirdPartyPushType;
import com.yy.pushsvc.util.PushLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Method;

public class PushDBHelper extends SQLiteOpenHelper {

    public static class PushDeviceInfo {

        public PushDeviceInfo() {

        }

        public PushDeviceInfo(String token, byte[] deviceid, byte[] mac) {
            mToken = token;
            mDeviceID = deviceid;
            mMac = mac;
        }

        public String mToken = "";
        public byte[] mDeviceID = null;
        public byte[] mMac = null;
    }

    public static class PushAccountInfo {
        public int mAppID = -1;
        public String mAccount = "";
        public byte[] mTicket = "".getBytes();
        public boolean mMulti = false;
        public String mAppver = "";
        public byte[] mSysToken = "".getBytes();
        public byte[] mThirdToken = "".getBytes();
    }

    public static class PushRecvMsg {
        public long msgId;
        public String account;
        public String tokentype;
        public String token;

        public PushRecvMsg(long msgid, String acc, String tokenType, String t) {
            msgId = msgid;
            account = acc;
            tokentype = tokenType;
            token = t;
        }
    }

    private static final String TAG = "PushDBHelper";
    private static String mDBName = "com.yy.shortpushsvc.db";
    private static int mDBVer = 5;
    private static volatile PushDBHelper mInstance;

    public static PushDBHelper getInstance(Context context) {
        if (mInstance == null) {
            synchronized (PushDBHelper.class) {
                if (mInstance == null) {
                    mInstance = new PushDBHelper(context);
                }
            }
        }
        return mInstance;
    }

    public PushDBHelper(Context context) {
        super(context, mDBName, null, mDBVer);
    }

    public boolean isTableExist(SQLiteDatabase db, String tabName) {
        boolean result = false;
        if (tabName == null) {
            return false;
        }
        Cursor cursor = null;
        try {
            String sql =
                    "select count(*) as c from sqlite_master where type ='table' and name ='" + tabName.trim() + "' ";
            cursor = db.rawQuery(sql, null);
            if (cursor.moveToNext()) {
                int count = cursor.getInt(0);
                if (count > 0) {
                    result = true;
                }
            }
        } catch (Exception e) {
            PushLog.inst().log("PushDBHelper.isTableExist... error" + e.getMessage());
        } finally {
            if (null != cursor && !cursor.isClosed()) {
                cursor.close();
            }
        }
        return result;
    }

    //return < 0表示出错；return = 0表示表不为空；return = 1表示表为空。
    private int isTableEmpty(String table) {
        if (table == null) {
            return -1;
        }
        int res;
        Cursor cursor = null;
        try {
            SQLiteDatabase db = getWritableDatabase();
            if (db == null) {
                return -1;
            }

            cursor = db.rawQuery("SELECT count(*) FROM " + table, null);


            if (cursor != null && cursor.moveToNext()) {
                int index = cursor.getColumnIndex("count(*)");
                if (index != -1) {
                    int line = cursor.getInt(index);
                    if (line == 0) {
                        res = 1;
                    } else if (line > 0) {
                        res = 0;
                    } else {
                        res = -1;
                    }
                } else {
                    PushLog.inst().log("PushDBHelper.isTableEmpty invalid index=" + index);
                    res = -1;
                }
            } else {
                PushLog.inst().log("PushDBHelper.isTableEmpty is empty");
                res = 1;
            }
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.isTableEmpty can not open db for writeable: " + e.toString());
            res = -1;
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return res;
    }

    private void rebuildDB(SQLiteDatabase db) {
        PushLog.inst().log(TAG + ".rebuildDB");
        if (isTableExist(db, "push_info")) {
            db.execSQL("drop table push_info;");
        }
        if (isTableExist(db, "recv_msg")) {
            db.execSQL("drop table recv_msg;");
        }
        if (isTableExist(db, "str_2_str_table")) {
            db.execSQL("drop table str_2_str_table;");
        }
        if (isTableExist(db, "report_recv_msg_by_http")) {
            db.execSQL("drop table  report_recv_msg_by_http;");
        }


        db.execSQL("create table push_info(token varchar(256), deviceid varchar(64), mac varchar(64));");
        db.execSQL("create table recv_msg(id integer primary key autoincrement, msg_id long, account varchar(64)," +
                "tokentype varchar(64), token varchar(256));");
        db.execSQL(
                "create table str_2_str_table(id integer primary key autoincrement, " +
                        "key_str varchar(128), val_str varchar(256));");
        db.execSQL("create table report_recv_msg_by_http(id integer primary key autoincrement, type varchar(64)," +
                "msgid long, pushid long, stat varchar(64), thirdToken varchar(256));");
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        PushLog.inst().log("PushDBHelper.onCreate ver=" + mDBVer);
        try {
            db.execSQL("create table push_info(token varchar(256), deviceid varchar(64), mac varchar(64));");
            db.execSQL("create table recv_msg(id integer primary key autoincrement, msg_id long, account varchar(64),"
                    + "tokentype varchar(64), token varchar(256));");
            db.execSQL("create table str_2_str_table(id integer primary key autoincrement, key_str varchar(128), " +
                    "val_str varchar(256));");
            db.execSQL("create table report_recv_msg_by_http(id integer primary key autoincrement, type varchar(64), " +
                    "msgid long, pushid long, stat varchar(64), thirdToken varchar(256));");
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.onCreate can not create db error: " + e.toString());
        }
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        PushLog.inst().log("PushDBHelper.onUpgrade oldVer=" + oldVersion + ", newVer=" + newVersion);
        for (int i = oldVersion; i < newVersion; i++) {
            int curVersion = i + 1;
            String methodName = "onUpgrade" + i + "to" + curVersion;

            boolean result;
            try {
                Method method = getClass().getMethod(methodName,
                        SQLiteDatabase.class);
                result = (Boolean) method.invoke(this, db);
                if (!result) {
                    rebuildDB(db);
                    return;
                }
            } catch (Exception e) {
                PushLog.inst().log("PushDBHelper.onUpgrade oldVer=" + oldVersion + ", newVer=" + newVersion + "err=" +
                        e.toString());
                rebuildDB(db);
                return;
            }
        }
    }

    @Override
    public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        try {
            PushLog.inst().log("PushDBHelper.onDowngrade oldVer=" + oldVersion + ", newVer=" + newVersion);
        } catch (SQLException e) {
            PushLog.inst().log("can not drop db, error=" + e.toString());

        }
    }

    public boolean onUpgrade1to2(SQLiteDatabase db) {
        PushLog.inst().log(TAG + ".onUpgrade1to2");
        try {
            db.execSQL("create table if not exists report_recv_msg_by_http(id integer " +
                    "primary key autoincrement, type varchar(64), msgid long, pushid long, stat varchar(64));");
            return true;
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.onUpgrade1to2 can not alert db " + e.toString());
        }
        return false;
    }

    public boolean onUpgrade2to3(SQLiteDatabase db) {
        PushLog.inst().log(TAG + ".onUpgrade2to3");
        try {
            db.execSQL("alter table report_recv_msg_by_http add thirdToken varchar(256);");
            return true;
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.onUpgrade2to3 can not alert db " + e.toString());
        }
        return false;
    }

    public boolean onUpgrade3to4(SQLiteDatabase db) {
        PushLog.inst().log(TAG + ".onUpgrade3to4");
        try {
            db.execSQL("create table push_ctl_info(test integer);");
            db.execSQL("create table push_jni_watcher(pid integer);");
            db.execSQL("create table push_account_info(appid integer primary key, " +
                    "account varchar(64), ticket varchar(64), multi bit, appver varchar(64), " +
                    "thirdpartytoken varchar(256), umengtoken varchar(256));");
            db.execSQL("create table push_start_info(start_time long, times integer);");
            db.execSQL("create table push_receivers(id integer primary key autoincrement, " +
                    "pkg_name varchar(128), class_name varchar(128));");
            db.execSQL("create table push_ctl_info_from_server(version integer, stop integer, " +
                    "not_restart integer, no_crash_report integer);");
            db.execSQL("create table push_svc_alive_time_table(id integer primary key autoincrement, " +
                    "start long, end long);");
            db.execSQL("create table push_svc_net_time_table(id integer primary key autoincrement, " +
                    "type integer, start long, end long);");
            db.execSQL("create table push_svc_tcp_time_table(id integer primary key autoincrement, " +
                    "tcp_connected integer, start long, end long);");
            db.execSQL("create table app_running_status_table(id integer primary key autoincrement, " +
                    "status integer, start long, end long);");
            db.execSQL("create table app_net_access_table(id integer primary key autoincrement, " +
                    "status integer, time long);");
            db.execSQL("create table push_cmd_time_table(cmd varchar(64) primary key, " +
                    "cmd_key long, time long);");
            db.execSQL("create table push_account_bind_token(account varchar(64));");
            db.execSQL("create table push_notification_state_statistics(id integer primary key autoincrement, " +
                    "pushchannel varchar(64), state long, pushid long, msgid long);");
            return true;
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.onUpgrade3to4 can not alert db " + e.toString());
        }
        return false;
    }

    public boolean onUpgrade4to5(SQLiteDatabase db) {
        PushLog.inst().log(TAG + ".onUpgrade4to5");
        try {
            db.execSQL("drop table push_ctl_info;");
            db.execSQL("drop table push_jni_watcher;");
            db.execSQL("drop table push_account_info;");
            db.execSQL("drop table push_start_info;");
            db.execSQL("drop table push_receivers;");
            db.execSQL("drop table push_ctl_info_from_server;");
            db.execSQL("drop table push_svc_alive_time_table;");
            db.execSQL("drop table push_svc_net_time_table;");
            db.execSQL("drop table push_svc_tcp_time_table;");
            db.execSQL("drop table app_running_status_table;");
            db.execSQL("drop table app_net_access_table;");
            db.execSQL("drop table push_cmd_time_table;");
            db.execSQL("drop table push_account_bind_token;");
            db.execSQL("drop table push_notification_state_statistics;");
            return true;
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.onUpgrade3to4 can not alert db " + e.toString());
        }
        return false;
    }

    public synchronized PushDeviceInfo getPushDeviceInfo() {
        Cursor tuple = null;
        try {
            PushDeviceInfo pushInfo = new PushDeviceInfo();
            SQLiteDatabase db = getReadableDatabase();
            if (db == null) {
                return null;
            }

            tuple = db.query("push_info", null, null, null, null, null,
                    null);
            if (tuple.moveToFirst()) {
                pushInfo.mToken = tuple.getString(tuple.getColumnIndex("token"));
                int indexDeviceID = tuple.getColumnIndex("deviceid");
                if (indexDeviceID != -1) {
                    pushInfo.mDeviceID = tuple.getBlob(indexDeviceID);
                } else {
                    pushInfo.mDeviceID = null;
                    PushLog.inst().log("PushDBHelper.getPushDeviceInfo invalid deviceid col=" + indexDeviceID);
                }
                int indexMac = tuple.getColumnIndex("mac");
                if (indexMac != -1) {
                    pushInfo.mMac = tuple.getBlob(indexMac);
                } else {
                    pushInfo.mMac = null;
                    PushLog.inst().log("PushDBHelper.getPushDeviceInfo invalid mac col=" + indexMac);
                }
                PushLog.inst().log("PushDBHelper.getPushDeviceInfo pushDeviceInfo is not null");
                return pushInfo;
            }
            PushLog.inst().log("PushDBHelper.getPushDeviceInfo pushDeviceInfo is null");
            return null;
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.getPushDeviceInfo can not open db for readable: " + e.toString());
            return null;
        } finally {
            if (tuple != null) {
                tuple.close();
            }
        }
    }

    public synchronized void removePushDeviceInfo() {
        SQLiteDatabase db;
        try {
            db = getWritableDatabase();
            if (db == null) {
                return;
            }

            // there shall be only one tuple in the table. delete the previous
            // one.
            db.delete("push_info", null, null); // no sql exception
        } catch (SQLiteException e) {
            PushLog.inst().log("PushDBHelper.removePushDeviceInfo can not open db for writeable: " + e.toString());

        }
    }

    public synchronized void savePushDeviceInfo(PushDeviceInfo info) {
        SQLiteDatabase db;
        try {
            db = getWritableDatabase();
            if (db == null) {
                return;
            }

            // there shall be only one tuple in the table. delete the previous
            // one.
            db.delete("push_info", null, null); // no sql exception
            if (info == null || info.mToken == null || info.mDeviceID == null || info.mMac == null) {
                // if it's null then we do nothing.
                return;
            }

            ContentValues cv = new ContentValues();
            cv.put("token", info.mToken);
            cv.put("deviceid", info.mDeviceID);
            cv.put("mac", info.mMac);
            if (db.insert("push_info", null, cv) < 0) { // no sql exception
                PushLog.inst().log("PushDBHelper.savePushDeviceInfo failed on saving to db");
                return;
            }
            PushLog.inst().log("PushDBHelper.savePushDeviceInfo successfully save to db");
        } catch (SQLiteException e) {
            PushLog.inst().log("PushDBHelper.savePushDeviceInfo can not open db for writeable: " + e.toString());
        }
    }

    public synchronized void saveRecvMsg(PushRecvMsg recvMsg) {
        try {
            SQLiteDatabase db = getWritableDatabase();
            if (db == null) {
                return;
            }

            ContentValues cv = new ContentValues();
            cv.put("msg_id", recvMsg.msgId);
            cv.put("account", recvMsg.account);
            cv.put("tokentype", recvMsg.tokentype);
            cv.put("token", recvMsg.token);

            if (db.insert("recv_msg", null, cv) < 0) {
                PushLog.inst().log("PushDBHelper.saveRecvMsg failed on saving msgid to db");
                return;
            }
            PushLog.inst().log("PushDBHelper.saveRecvMsg successfully save msgid to db, msgid:" + recvMsg.msgId);
            // only 500 records will be saved, remove the oldest records.
            String deleteSqlStr = "delete from recv_msg where (select count(id) from recv_msg) > 500 " +
                    "and id in (select id from recv_msg order by id desc limit (select count(id) from recv_msg) " +
                    "offset 500)";
            db.execSQL(deleteSqlStr);
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.saveRecvMsg can not open db for writeable: " + e.toString());
        }
    }

    public synchronized boolean isDuplicateMsg(long msgID) {
        SQLiteDatabase db;
        Cursor tuple = null;
        try {
            db = getReadableDatabase();
            if (db == null) {
                return false;
            }

            tuple = db.query("recv_msg", null, "msg_id=" + msgID, null,
                    null, null, null);
            return tuple.moveToFirst();
        } catch (SQLiteException e) {
            PushLog.inst().log("PushDBHelper.isDuplicateMsg " + e.getMessage());

        } finally {
            if (tuple != null) {
                tuple.close();
            }
        }
        return false;
    }

    public synchronized boolean hasStrKey(String key) {
        SQLiteDatabase db;
        Cursor tuple = null;
        try {
            db = getReadableDatabase();
            if (db == null) {
                return false;
            }

            tuple = db.query("str_2_str_table", null, "key_str='" + key + "'", null, null, null, null);
            return tuple.moveToFirst();
        } catch (SQLiteException e) {
            PushLog.inst().log("PushDBHelper.hasStrKey " + e.getMessage());
            return false;
        } finally {
            if (tuple != null) {
                tuple.close();
            }
        }
    }

    public synchronized void addStrKey2StrVal(String key, String value) {
        if (key == null || value == null) {
            PushLog.inst().log("PushDBHelper.addStrKey2StrVal invalid args, key=" + key + ", value=" + value);
            return;
        }
        if (hasStrKey(key)) {
            PushLog.inst().log("PushDBHelper.addStrKey2StrVal already have key=" + key);
        }
        try {
            SQLiteDatabase db = getWritableDatabase();
            if (db == null) {
                return;
            }
            PushLog.inst().log("PushDBHelper.addStrKey2StrVal key=" + key + ", value=" + value);
            ContentValues cv = new ContentValues();
            cv.put("key_str", key);
            cv.put("val_str", value);
            if (db.insert("str_2_str_table", null, cv) < 0) { // no sql exception
                PushLog.inst().log("PushDBHelper.addStrKey2StrVal failed on saving to db");
            }
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.addStrKey2StrVal can not open db for writeable: " + e.toString());
        }
    }

    public synchronized void updateStrKey2StrVal(String key, String value) {
        if (key == null || value == null) {
            PushLog.inst().log("PushDBHelper.updateStrKey2StrVal invalid args, key=" + key + ", value=" + value);
            return;
        }
        if (!hasStrKey(key)) {
            PushLog.inst().log("PushDBHelper.updateStrKey2StrVal don't have key=" + key);
        }
        Cursor cursor = null;
        try {
            SQLiteDatabase db = getWritableDatabase();
            if (db == null) {
                return;
            }

            cursor = db.query("str_2_str_table", null, "key_str='" + key + "'", null, null, null, null);
            if (cursor.moveToNext()) {
                // we found it.
                ContentValues cv = new ContentValues();
                cv.put("key_str", key);
                cv.put("val_str", value);
                PushLog.inst().log("PushDBHelper.updateStrKey2StrVal, key=" + key + ", value=" + value);
                db.update("str_2_str_table", cv, "key_str='" + key + "'", null);
            }
        } catch (SQLException e) {
            PushLog.inst().log("PushDBHelper.updateStrKey2StrVal can not open db for writeable: " + e.toString());
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    public synchronized String getStrVal(String key) {
        if (key == null) {
            return null;
        }
        SQLiteDatabase db;
        Cursor tuple = null;
        String value = null;
        try {
            db = getReadableDatabase();
            if (db == null) {
                return null;
            }

            tuple = db.rawQuery("select * from str_2_str_table where key_str='" + key + "';", null);
            if (tuple.moveToNext()) {
                value = tuple.getString(tuple.getColumnIndex("val_str"));
            }
        } catch (SQLiteException e) {
            PushLog.inst().log("PushDBHelper.getStrVal " + e.getMessage());
        } finally {
            if (tuple != null) {
                tuple.close();
            }
        }
        return value;
    }

    public synchronized void addOrUpdateStrKey2StrVal(String key, String value) {
        if (key == null || value == null) {
            return;
        }
        if (hasStrKey(key)) {
            if (!value.equals(getStrVal(key))) {
                updateStrKey2StrVal(key, value);
            } else {
                PushLog.inst().log(TAG + ".addOrUpdateStrKey2StrVal db has key:" + key);
            }
        } else {
            addStrKey2StrVal(key, value);
        }
    }

    public synchronized void recordReportStatistics(String channeltype, long msgid, long pushid, int stat,
                                                    String thirdToken) {
        PushLog.inst().log("PushDBHelper.recordReportStatistics pushchannel:" + channeltype + " stat:" + stat);
        try {
            SQLiteDatabase db = getWritableDatabase();
            if (db == null) {
                return;
            }
            if (convertChannelType(channeltype) == -1) {
                return;
            }
            ContentValues cv = new ContentValues();
            cv.put("type", convertChannelType(channeltype));
            cv.put("msgid", msgid);
            cv.put("pushid", pushid);
            cv.put("stat", stat);
            cv.put("thirdToken", thirdToken);
            if (db.insert("report_recv_msg_by_http", null, cv) < 0) {
                PushLog.inst().log("PushDBHelper.recordReportStatistics failed on saving msgid to db");
                return;
            }
            PushLog.inst().log("PushDBHelper.recordReportStatistics successfully save msgid to db, msgid:" + msgid);
            // only 500 records will be saved, remove the oldest records.
            String deleteSqlStr = "delete from report_recv_msg_by_http where (select count(id) " +
                    "from report_recv_msg_by_http) > 500 and id in (select id from report_recv_msg_by_http " +
                    "order by id desc limit (select count(id) from report_recv_msg_by_http) offset 500)";
            db.execSQL(deleteSqlStr);
        } catch (SQLiteException e) {
            PushLog.inst().log("PushDBHelper.recordReportStatistics, exception:" + e.getMessage());
        }
    }

    public synchronized void clearReportStatistics(long msgid, long state) {
        PushLog.inst().log(TAG + ".clearReportStatistics, msgid : " + msgid + ", state:" + state);
        Cursor tuple = null;
        try {
            SQLiteDatabase db = getWritableDatabase();
            if (db == null) {
                PushLog.inst().log(TAG + ".clearReportStatistics, db is null");
                return;
            }
            tuple = db.query("report_recv_msg_by_http", null,
                    "msgid = " + msgid + " and stat = " + state, null, null,
                    null, null);
            if (tuple.moveToNext()) {
                db.execSQL("delete from report_recv_msg_by_http where msgid = " + msgid + " and " + "stat = " + state);
                PushLog.inst().log(TAG + ".clearReportStatistics, delete msg from db");
            } else {
                PushLog.inst().log(TAG + ".clearReportStatistics, db has no msg");
            }
        } catch (SQLiteException e) {
            PushLog.inst().log(TAG + ".clearReportStatistics, exception:" + e.getMessage());
        } finally {
            if (tuple != null) {
                tuple.close();
            }
        }
    }

    public synchronized JSONArray getUnReportedStatistics() {
        PushLog.inst().log(TAG + ".getUnReportedStatistics ");
        SQLiteDatabase db;
        Cursor cursor = null;
        int msgStatInfoSize = 0;
        JSONArray jsonArray = new JSONArray();
        try {
            db = getWritableDatabase();
            if (db == null) {
                PushLog.inst().log(TAG + ".getUnReportedStatistics , db is null");
            }
            if (db != null) {
                cursor = db.query("report_recv_msg_by_http", null, "stat<>0", null, null, null, null, "0,10");
            }
            while (cursor.moveToNext()) {
                try {
                    JSONObject jsonObject = new JSONObject();
                    jsonObject.put("type", cursor.getInt(cursor.getColumnIndex("type")));
                    jsonObject.put("msgID", cursor.getLong(cursor.getColumnIndex("msgid")));
                    jsonObject.put("pushID", cursor.getLong(cursor.getColumnIndex("pushid")));
                    jsonObject.put("stat", cursor.getInt(cursor.getColumnIndex("stat")));
                    jsonObject.put("thirdToken", cursor.getString(cursor.getColumnIndex("thirdToken")));
                    jsonArray.put(jsonObject);
                    msgStatInfoSize++;
                } catch (JSONException e) {
                    e.printStackTrace();
                    PushLog.inst().log(TAG + ".getUnReportedStatistics JSONException:" + e);
                    return jsonArray;
                }
            }
            PushLog.inst().log(TAG + ".getUnReportedStatistics size = " + msgStatInfoSize);
            return jsonArray;
        } catch (SQLiteException e) {
            PushLog.inst().log(TAG + ".getUnReportedStatistics exception:" + e);
            return jsonArray;
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    private int convertChannelType(String channeltype) {
        switch (channeltype) {
            case ThirdPartyPushType.PUSH_TYPE_XIAOMI:
                return 1;
            case ThirdPartyPushType.PUSH_TYPE_HUAWEI:
                return 2;
            case ThirdPartyPushType.PUSH_TYPE_OPPO:
                return 32;
            case ThirdPartyPushType.PUSH_TYPE_FCM:
                return 128;
            default:
                return -1;
        }
    }
}
