package com.yy.pushsvc.facknotification;

import android.content.Context;
import android.os.Environment;

import com.yy.pushsvc.util.PushLog;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

/**
 * Created by GuanYuFeng
 * on 2020/1/9
 */
public class ScreenCache implements IScreenCache {
    private static final String fileName = "NotificationEntity";
    private static final String TAG = "ScreenCache";
    private static volatile ScreenCache instance;
    private NotificationEntity mEntity;

    private ScreenCache() {
    }

    public static IScreenCache getInstance() {
        if (instance == null) {
            synchronized (ScreenCache.class) {
                if (instance == null) {
                    instance = new ScreenCache();
                }
            }
        }
        return instance;
    }

    public NotificationEntity getEntity(Context context) {
        NotificationEntity notificationEntity = null;
        try {
            if (context == null) {
                return null;
            }
            if (mEntity != null) {
                notificationEntity = mEntity;
            } else {
                notificationEntity = getEntityFromDisk(context);
            }
            clearCache(context);
        } catch (Throwable t) {
            PushLog.inst().log(TAG + ",getEntity(),erro=" + t.toString());
        }
        return notificationEntity;
    }

    public boolean saveEntity(NotificationEntity entity, Context context) {
        try {
            if (mEntity != null && entity != null && mEntity.msgId == entity.msgId) {
                return true;
            }
            if (context == null) {
                return false;
            }
            clearCache(context);
            mEntity = entity;
            saveToDisk(mEntity, context);
        } catch (Throwable t) {
            PushLog.inst().log(TAG + ",saveEntity(),erro=" + t.toString());
        }
        return false;
    }

    private void clearCache(Context context) {
        try {
            mEntity = null;
            File file = new File(getDiskCachePath(context));
            File parentFile = file.getParentFile();
            if (!parentFile.exists()) {
                parentFile.createNewFile();
            }
            if (file.exists()) {
                file.delete();
            }
        } catch (Throwable t) {
            PushLog.inst().log(TAG + ",clearCache()," + t.toString());
        }
    }

    private NotificationEntity getEntityFromDisk(Context context) {
        NotificationEntity entity = null;
        FileInputStream fileIn = null;
        ObjectInputStream objectInputStream = null;
        try {
            fileIn = new FileInputStream(getDiskCachePath(context));
            objectInputStream = new ObjectInputStream(fileIn);
            entity = (NotificationEntity) objectInputStream.readObject();
        } catch (Throwable e) {
            PushLog.inst().log(TAG + ",getEntityFromDisk()," + e.toString());
        } finally {
            try {
                if (fileIn != null) {
                    fileIn.close();
                }
                if (objectInputStream != null) {
                    objectInputStream.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return entity;
    }

    private void saveToDisk(NotificationEntity entity, Context context) {
        ObjectOutputStream objectOutputStream = null;
        FileOutputStream fileOut = null;
        try {
            fileOut = new FileOutputStream(getDiskCachePath(context));
            objectOutputStream = new ObjectOutputStream(fileOut);
            objectOutputStream.writeObject(entity);
            objectOutputStream.flush();
            objectOutputStream.close();
        } catch (Throwable e) {
            PushLog.inst().log(TAG + ",saveToDisk()," + e.toString());
        } finally {
            try {
                if (fileOut != null) {
                    fileOut.close();
                }
                if (objectOutputStream != null) {
                    objectOutputStream.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private String getDiskCachePath(Context context) {
        try {
            if (Environment.MEDIA_MOUNTED.equals(Environment.getExternalStorageState())
                    || !Environment.isExternalStorageRemovable()) {
                return context.getExternalCacheDir().getPath() + "/" + fileName;
            } else {
                return context.getCacheDir().getPath() + "/" + fileName;
            }
        } catch (Throwable t) {
            PushLog.inst().log(TAG + ",getDiskCachePath()," + t.toString());
        }
        return "";
    }
}
