package com.yy.pushglide;

import android.app.Notification;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.widget.ImageView;

import com.yy.pushsvc.template.NotificationHandler;
import com.yy.pushsvc.template.ViewEntity;
import com.yy.pushsvc.util.PushLog;

/**
 * This class is used to display downloaded Bitmap inside an ImageView of a Notification through RemoteViews.
 *
 * <p>
 * Note - For cancellation to work correctly, you must pass in the same instance of this class for every subsequent
 * load.
 * </p>
 */
public class NotificationTarget {

    private final ViewEntity viewEntity;
    private final Context context;
    private final int notificationId;
    private final Notification notification;
    private final int viewId;
    private float radius = 0f;
    private NotificationHandler notificationHandler;
    public String pushId;
    public String url;
    public int width;
    public int height;

    public NotificationTarget(Context context, ViewEntity viewEntity, int viewId,
                              Notification notification,
                              int notificationId, int width, int height, int cornerradius,
                              String pushId,
                              NotificationHandler notificationHandler) {
        this.width = width;
        this.height = height;
        if (context == null) {
            throw new NullPointerException("Context must not be null!");
        }
        if (notification == null) {
            throw new NullPointerException("Notification object can not be null!");
        }
        if (viewEntity == null) {
            throw new NullPointerException("RemoteViews object can not be null!");
        }
        this.context = context;
        this.viewId = viewId;
        this.notification = notification;
        this.notificationId = notificationId;
        this.viewEntity = viewEntity;
        radius = Resources.getSystem().getDisplayMetrics().density * cornerradius;
        this.pushId = pushId;
        this.notificationHandler = notificationHandler;
    }

    public void onLoadStarted() {
        notificationHandler.showNotificationDelay(notificationId, notification,
                notificationHandler.maxImgCount * 40000, false, pushId);
    }

    public void onLoadFailed(Exception e) {
        PushLog.inst().log("NotificationTarget,url=" + url + "/erro=" + e.toString());
        notificationHandler.addComplate(false, url, pushId);
    }

    public void onResourceReady(Bitmap resource) {
        if (viewEntity.isRemoteViews) {
            viewEntity.mRemoteView.setImageViewBitmap(this.viewId, resource);
        } else {
            ImageView iv = viewEntity.mView.findViewById(viewId);
            iv.setImageBitmap(resource);
        }
        notificationHandler.addComplate(true, url, pushId);
    }
}
