#ifndef _AUDIO_CODEC_HEADER_
#define _AUDIO_CODEC_HEADER_

#include <vector>
#include <list>
#include <string>

typedef void(*CodecLogCallback)(const char* pLogStr);
void SetCodecLogFunction(CodecLogCallback logFunc = NULL);
const char* GetAudioCodecVersion();
void InitAudioCodecPlatform();

/*********************************audio_file_decoder.h****file_decoder api*****************************************/
#include "audiocodec/int_type.h"
enum AudioDecoderType
{
	kUnKnown = -1,
	kWAVType = 0,
	kMP3Type = 1,
	kAACType = 2,
	kM4AType = 3,
};

class IAudioFileDecoder
{
public:
	virtual ~IAudioFileDecoder() {}
	virtual void Destroy() = 0;
	virtual bool Open(char* pFileName) = 0;
	virtual int GetPCMData(char* pData, int dataLen) = 0;
	virtual uint32_t Seek(uint32_t timeMS) = 0;
	virtual uint32_t GetTotalPlayLengthMS() = 0;
	virtual int GetSampleRate() = 0;
	virtual int GetChannelCount() = 0;
	virtual void EnableLoopPlay(bool enable) = 0;   // TODO: make it better, add to protected  
	virtual void Flush() = 0;
    virtual uint32_t GetCurrentPositionMS() = 0;
    virtual bool setOutputFormat(int sampleRate, int channel) = 0;

protected:

};

IAudioFileDecoder* CreateAudioFileDecoder(AudioDecoderType codec);



/*********************************audio_localcodec.h****local_codec api*****************************************/

namespace AudioLocalCodec {

enum AudioCodecType {
	UNKNOWN_CODEC = -1,       // Error Codec
	AAC_PLUS_CODEC = 1,       // NET_AACPLUS
	AAC_ELD_CODEC = 4,        // ELD mode
	SILK_CODEC = 2,           // NET_SILK_16K
	SPEEX_CODEC_8 = 0,        // NET_SPEEX_MODE_8
	SPEEX_CODEC_4 = 21,       // NET_SPEEX_MODE_4
	SPEEX_CODEC_2 = 23,       // NET_SPEEX_MODE_2
	AAC_LC_CODEC = 35,        // NET_AAC_128K	
	AAC_AUDIT_CODEC = 36,        // AUDIT	
	WAV_CODEC = 0xFF,         // Wav Codec, not support speech message player, only recorder
};

enum AacCoderType
{
	LC_AAC = 0,
	EPLUS_AAC = 1,
	ELD_AAC = 2,
	AUDIT_AAC = 3,
};

struct CodecParameter {
	CodecParameter() :
	channels(0),
	sampleRate(0),
	bps(0),
	bitRate(0),
	vbr(0),
	timePerEncodedFrame(0),
	samplePerFrame(0),
	quality(0) {}

	int channels;
	int sampleRate;
	int bps;   // bit per sample 
	int bitRate;
	int vbr;
	int timePerEncodedFrame;
	int samplePerFrame;
	int quality;
};

class IAudioEncoder
{
public:
	virtual ~IAudioEncoder(){}
	virtual void Destroy() = 0;
	virtual int  GetEncoderBytes() = 0;
	virtual bool Init() = 0;
	virtual void Uninit() = 0;
	virtual void Flush() = 0;
	virtual int Encode(const char* pInPCM, int inLen, std::string& outData) = 0; 
	virtual int EncodeLoss(std::string& outData) = 0;
	virtual void SetBitRate(int bitRate) = 0;
	virtual void SetComplexity(int complexity) = 0;
};

class IAudioDecoder
{
public:
	virtual ~IAudioDecoder() {}
	virtual void Destroy() = 0;
	virtual bool Init() = 0;
	virtual void Uninit() = 0;
	virtual void Flush() = 0;
	virtual bool Decode(const char* pInData, int inLen, std::string& outPCM) = 0;
	virtual bool DecodeLoss(std::string& outPCM) = 0;
};

IAudioEncoder*  CreateAudioEncoder(AudioCodecType type);
IAudioDecoder*  CreateAudioDecoder(AudioCodecType type);
CodecParameter GetCodecParameter(AudioCodecType type); 

};

/*********************************audio_livecodec.h****live_codec api*****************************************/
enum YYAudioCodec
{
	kAudioCodecUnknown = 0x00,
	kAudioCodecSpeex = 0x01,
	kAudioCodecAMRWB = 0x02,
	kAudioCodecSilk = 0x04,
	kAudioCodecMP3 = 0x08,
	kAudioCodecAAC = 0x10,
	kAudioCodecPCM = 0x20,
	kAudioCodecAACHIGH = 0x40,
	kAudioCodecOPUS = 0x80,
	kAudioCodecEldAAC = 0x100,
	kAudioCodecAACCommon = 0x200,
};

enum AudioLibraryCodecMode
{
	kAudioLibraryDecoder = 0,
	kAudioLibraryEncoder = 1,
};

struct AudioStreamFormat
{
	YYAudioCodec iCodec;
	uint32_t iFlag;

	uint32_t iSampleRate;
	uint32_t iNumOfChannels;
	uint32_t iBitsOfSample;
	uint32_t iBitRate;

	/// how many decoded samples per channel in one audio frame. always 1 for PCM.
	/// may not be zero for non-PCM codec.
	/// for example: EAAC 2 channels, 44.1KHz, iSamplesOfFrame = 2048, 46ms / per frame.
	uint32_t iSamplesOfFrame;

	/// ignored for PCM.
	/// for some codec, like AAC, profile means the encoder's capabilities.
	/// AAC profile:(AAC:1, EAAC:5, AAC_ADTS:101, EAAC_ADTS:105)
	int iProfile;

	/// ignored for PCM.
	/// for some codec, like SPEEX AMRWB AAC, profile means the encoder's qulity(0-10).
	int iQuality;

	/// ignored for PCM.
	/// for some codec, like Speex/Silk, they have constant encoded bitrate for one frame.
	int iBytesOfCodedFrame;

	uint32_t iNumOfFramesInPacket; //how many audio data will be send to encoder once

	int iRawCodecId;

	unsigned GetFrameDuration()
	{
		return (iSamplesOfFrame * 1000 / iSampleRate);
	}

	static bool IsSupportedSampleRate(uint32_t samplerate)
	{
		static const uint32_t supported[] = { 8000, 11025, 16000, 22050, 24000, 32000, 44100, 48000 };
		for (uint32_t i = 0; i < sizeof(supported) / sizeof(supported[0]); ++i)
		if (samplerate == supported[i])
			return true;
		return false;
	}

	static bool IsSupportedSampleBits(uint32_t bits)
	{
		return bits == 8 || bits == 16 || bits == 32;
	}

	uint32_t GetBytesOfInterleavedSample() const
	{
		return iNumOfChannels * iBitsOfSample / 8;
	}

	static bool IsPCMFormatEquals(const AudioStreamFormat &format1, const AudioStreamFormat &format2)
	{
		bool ret = true;
		if (format1.iFlag != format2.iFlag || format1.iNumOfChannels != format2.iNumOfChannels ||
			format1.iBitsOfSample != format2.iBitsOfSample || format1.iSampleRate != format2.iSampleRate)
		{
			ret = false;
		}
		return ret;
	}
};

class CAudioCodec
{
public:
	virtual ~CAudioCodec(){};
	/*
		nProfile  :
		1.kAudioCodecSpeex (nProfile = 0 -> SPEEX_MODEID_NB
		nProfile = 1 -> SPEEX_MODEID_WB
		nProfile = 2 -> SPEEX_MODEID_UWB)
		*/
	virtual bool Init(AudioStreamFormat* audioFormat) = 0;

	/*
		nInputMaxSize:
		*nOutBufferMaxSize: some frame need buflen
		if nFrameSize is zero, *nOutBufferMaxSize returned the buffer size for 8 frames.
		ret: 0 OK
		*/
	virtual int  CalcBufSize(int *nOutBufferMaxSize, int nFrameSize) = 0;

	/*
		pIn     :  when pIn = NULL recovering one frame
		nInLen  :
		pOut    :
		nOutLen :
		ret     : <0 err ; >=0 consumed data len
		*/
	virtual int  Process(const unsigned char* pIn, int nInLen, unsigned char* pOut, int* nOutLen) = 0;

	//0: decoder 1: encoder
	virtual int  CodecMode() = 0;
	virtual int  CodecID() = 0;
	virtual int  CodecLevel() = 0;
	virtual void DeInit() = 0;
	virtual const char* CodecDescribe() { return ""; }
	virtual bool IsAvailable() { return true; }
	virtual int CodecDelay() { return 0; }   // return ms 
	virtual int GetEncoderSize() { return 0; }
};

CAudioCodec* CreateAudioCodec(YYAudioCodec codecType, AudioLibraryCodecMode codecMode);



#endif 
