package com.tencent.imsdk.v2;

import android.text.TextUtils;
import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMLog;
import com.tencent.imsdk.manager.BaseManager;
import com.tencent.imsdk.message.MessageOfflinePushInfo;
import com.tencent.imsdk.signaling.SignalingInfo;
import com.tencent.imsdk.signaling.SignalingListener;
import com.tencent.imsdk.signaling.SignalingManager;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

public class V2TIMSignalingManagerImpl extends V2TIMSignalingManager {
    private static final String TAG = "V2TIMSigMgrImpl";

    private SignalingListener mInternalSignalingListener;
    private CopyOnWriteArrayList<V2TIMSignalingListener> mSignalingListenerList;
    private boolean mHasValidSignalingListener = false;

    private static class V2TIMSignalingManagerImplHolder {
        private static final V2TIMSignalingManagerImpl v2TIMSignalingManagerImpl = new V2TIMSignalingManagerImpl();
    }

    static V2TIMSignalingManagerImpl getInstance() {
        return V2TIMSignalingManagerImplHolder.v2TIMSignalingManagerImpl;
    }

    private V2TIMSignalingManagerImpl() {
        mSignalingListenerList = new CopyOnWriteArrayList<>();
        initSignalingListener();
    }

    private void initSignalingListener() {
        if (mInternalSignalingListener == null) {
            mInternalSignalingListener = new SignalingListener() {
                @Override
                public void onReceiveNewInvitation(
                    String inviteID, String inviter, String groupID, List<String> inviteeList, String data) {
                    synchronized (V2TIMSignalingManagerImpl.this) {
                        for (V2TIMSignalingListener listener : mSignalingListenerList) {
                            if (listener != null) {
                                listener.onReceiveNewInvitation(inviteID, inviter, groupID, inviteeList, data);
                            }
                        }
                    }
                }

                @Override
                public void onInviteeAccepted(String inviteID, String invitee, String data) {
                    synchronized (V2TIMSignalingManagerImpl.this) {
                        for (V2TIMSignalingListener listener : mSignalingListenerList) {
                            if (listener != null) {
                                listener.onInviteeAccepted(inviteID, invitee, data);
                            }
                        }
                    }
                }

                @Override
                public void onInviteeRejected(String inviteID, String invitee, String data) {
                    synchronized (V2TIMSignalingManagerImpl.this) {
                        for (V2TIMSignalingListener listener : mSignalingListenerList) {
                            if (listener != null) {
                                listener.onInviteeRejected(inviteID, invitee, data);
                            }
                        }
                    }
                }

                @Override
                public void onInvitationCancelled(String inviteID, String inviter, String data) {
                    synchronized (V2TIMSignalingManagerImpl.this) {
                        for (V2TIMSignalingListener listener : mSignalingListenerList) {
                            if (listener != null) {
                                listener.onInvitationCancelled(inviteID, inviter, data);
                            }
                        }
                    }
                }

                @Override
                public void onInvitationTimeout(String inviteID, List<String> inviteeList) {
                    synchronized (V2TIMSignalingManagerImpl.this) {
                        for (V2TIMSignalingListener listener : mSignalingListenerList) {
                            if (listener != null) {
                                listener.onInvitationTimeout(inviteID, inviteeList);
                            }
                        }
                    }
                }

                @Override
                public void onInvitationModified(String inviteID, String data) {
                    synchronized (V2TIMSignalingManagerImpl.this) {
                        for (V2TIMSignalingListener listener : mSignalingListenerList) {
                            if (listener != null) {
                                listener.onInvitationModified(inviteID, data);
                            }
                        }
                    }
                }
            };
        }
        SignalingManager.getInstance().setSignalingListener(mInternalSignalingListener);
    }

    void unInit() {
        synchronized (V2TIMSignalingManagerImpl.this) {
            mSignalingListenerList.clear();
            mHasValidSignalingListener = false;
        }
    }

    @Override
    public void addSignalingListener(V2TIMSignalingListener listener) {
        synchronized (V2TIMSignalingManagerImpl.this) {
            if (mSignalingListenerList.contains(listener)) {
                return;
            }
            mSignalingListenerList.add(listener);
            mHasValidSignalingListener = true;
            checkSignalingListener();
        }
    }

    @Override
    public void removeSignalingListener(V2TIMSignalingListener listener) {
        synchronized (V2TIMSignalingManagerImpl.this) {
            mSignalingListenerList.remove(listener);
            if (0 == mSignalingListenerList.size()) {
                mHasValidSignalingListener = false;
            }
            checkSignalingListener();
        }
    }

    private void checkSignalingListener() {
        BaseManager.getInstance().enableSignaling(mHasValidSignalingListener);
    }

    @Override
    public String invite(String invitee, String data, boolean onlineUserOnly, V2TIMOfflinePushInfo offlinePushInfo,
        int timeout, final V2TIMCallback callback) {
        if (TextUtils.isEmpty(invitee)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invitee is empty");
            }
            IMLog.e(TAG, "signaling invite error, invitee is empty");
            return null;
        }

        if (timeout < 0) {
            if (callback != null) {
                IMLog.e(TAG, "signaling invite error, timeout invalid");
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "timeout invalid");
            }
            return null;
        }

        MessageOfflinePushInfo messageOfflinePushInfo = null;
        if (offlinePushInfo != null) {
            messageOfflinePushInfo = offlinePushInfo.getMessageOfflinePushInfo();
        }

        String inviteID = SignalingManager.getInstance().invite(
            invitee, data, onlineUserOnly, messageOfflinePushInfo, timeout, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });

        return inviteID;
    }

    @Override
    public String inviteInGroup(String groupID, List<String> inviteeList, String data, boolean onlineUserOnly,
        int timeout, final V2TIMCallback callback) {
        if (TextUtils.isEmpty(groupID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupID is empty");
            }
            IMLog.e(TAG, "signaling invite error, groupID is empty");
            return null;
        }
        if (inviteeList == null || inviteeList.size() == 0) {
            if (callback != null) {
                IMLog.e(TAG, "signaling invite error, inviteeList is empty");
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "inviteeList is empty");
            }
            return null;
        }

        if (timeout < 0) {
            if (callback != null) {
                IMLog.e(TAG, "signaling invite error, timeout invalid");
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "timeout invalid");
            }
            return null;
        }

        String inviteID = SignalingManager.getInstance().inviteInGroup(
            groupID, inviteeList, data, onlineUserOnly, timeout, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });

        return inviteID;
    }

    @Override
    public void cancel(final String inviteID, final String data, final V2TIMCallback callback) {
        if (TextUtils.isEmpty(inviteID)) {
            if (callback != null) {
                IMLog.e(TAG, "signaling cancel error, inviteID is empty");
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "inviteID is empty");
            }
            return;
        }

        SignalingManager.getInstance().cancel(inviteID, data, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void accept(final String inviteID, final String data, final V2TIMCallback callback) {
        if (TextUtils.isEmpty(inviteID)) {
            IMLog.e(TAG, "signaling accept failed, inviteID is empty");
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "inviteID is null");
            }
            return;
        }

        SignalingManager.getInstance().accept(inviteID, data, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void reject(final String inviteID, final String data, final V2TIMCallback callback) {
        if (TextUtils.isEmpty(inviteID)) {
            IMLog.e(TAG, "signaling reject invite, inviteID is empty");
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "inviteID is null");
            }
            return;
        }

        SignalingManager.getInstance().reject(inviteID, data, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public V2TIMSignalingInfo getSignalingInfo(V2TIMMessage msg) {
        if (msg == null || msg.getElemType() != V2TIMMessage.V2TIM_ELEM_TYPE_CUSTOM) {
            return null;
        }

        SignalingInfo signalingInfo = SignalingManager.getInstance().getSignalingInfo(msg.getMessage());

        if (signalingInfo == null || TextUtils.isEmpty(signalingInfo.getInviteID())
            || TextUtils.isEmpty(signalingInfo.getInviter()) || signalingInfo.getInviteeList() == null
            || signalingInfo.getInviteeList().size() == 0
            || signalingInfo.getActionType() == V2TIMSignalingInfo.SIGNALING_ACTION_TYPE_UNKNOWN) {
            return null;
        }

        V2TIMSignalingInfo v2TIMSignalingInfo = new V2TIMSignalingInfo();
        v2TIMSignalingInfo.setSignalingInfo(signalingInfo);
        return v2TIMSignalingInfo;
    }

    @Override
    public void addInvitedSignaling(final V2TIMSignalingInfo info, final V2TIMCallback callback) {
        if (info == null || TextUtils.isEmpty(info.getInviteID()) || TextUtils.isEmpty(info.getInviter())
            || info.getInviteeList() == null || info.getInviteeList().isEmpty()) {
            IMLog.e(TAG, "addInvitedSignaling, info is invalid");
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "V2TIMSignalingInfo is invalid");
            }
            return;
        }

        SignalingManager.getInstance().addInvitedSignaling(info.getSignalingInfo(), new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void modifyInvitation(final String inviteID, final String data, final V2TIMCallback callback) {
        if (TextUtils.isEmpty(inviteID)) {
            if (callback != null) {
                IMLog.e(TAG, "modifyInvitation error, inviteID is empty");
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "inviteID is empty");
            }
            return;
        }

        SignalingManager.getInstance().modifyInvitation(inviteID, data, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }
}
