package com.tencent.imsdk.message;

import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.common.IMLog;
import com.tencent.imsdk.conversation.ConversationKey;
import com.tencent.imsdk.manager.BaseManager;
import com.tencent.imsdk.relationship.UserInfo;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

public class MessageCenter {
    private static final String TAG = "MessageCenter";

    private Object mLockObject = new Object();
    private MessageListener mMessageListener;
    private CopyOnWriteArrayList<MessageListener> mMessageProxyListenerList;

    private static class MessageCenterHolder {
        private static final MessageCenter messageCenter = new MessageCenter();
    }

    public static MessageCenter getInstance() {
        return MessageCenter.MessageCenterHolder.messageCenter;
    }

    protected MessageCenter() {
        mMessageProxyListenerList = new CopyOnWriteArrayList<>();
    }

    public void init() {
        initMessageListener();
    }

    private void initMessageListener() {
        mMessageListener = new MessageListener() {
            @Override
            public void onReceiveNewMessage(final List<Message> messageList) {
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (mLockObject) {
                            for (MessageListener messageListener : mMessageProxyListenerList) {
                                messageListener.onReceiveNewMessage(messageList);
                            }
                        }
                    }
                });
            }

            @Override
            public void onReceiveC2CMessageReceipt(final List<C2CMessageReceipt> receiptList) {
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (mLockObject) {
                            for (MessageListener messageListener : mMessageProxyListenerList) {
                                messageListener.onReceiveC2CMessageReceipt(receiptList);
                            }
                        }
                    }
                });
            }

            @Override
            public void onReceiveGroupMessageReceipt(final List<GroupMessageReceipt> receiptList) {
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        for (MessageListener messageListener : mMessageProxyListenerList) {
                            messageListener.onReceiveGroupMessageReceipt(receiptList);
                        }
                    }
                });
            }

            @Override
            public void onReceiveMessageRevoked(final Map<MessageKey, Map<UserInfo, String>> revokeInfoMap) {
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (mLockObject) {
                            for (MessageListener messageListener : mMessageProxyListenerList) {
                                messageListener.onReceiveMessageRevoked(revokeInfoMap);
                            }
                        }
                    }
                });
            }

            @Override
            public void onReceiveMessageModified(final List<Message> messageList) {
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (mLockObject) {
                            for (MessageListener messageListener : mMessageProxyListenerList) {
                                messageListener.onReceiveMessageModified(messageList);
                            }
                        }
                    }
                });
            }

            @Override
            public void onReceiveMessageExtensionsChanged(
                final MessageKey messageKey, final List<MessageExtension> extensions) {
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (mLockObject) {
                            for (MessageListener messageListener : mMessageProxyListenerList) {
                                messageListener.onReceiveMessageExtensionsChanged(messageKey, extensions);
                            }
                        }
                    }
                });
            }

            @Override
            public void onReceiveMessageExtensionsDeleted(
                final MessageKey messageKey, final List<MessageExtension> extensions) {
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (mLockObject) {
                            for (MessageListener messageListener : mMessageProxyListenerList) {
                                messageListener.onReceiveMessageExtensionsDeleted(messageKey, extensions);
                            }
                        }
                    }
                });
            }

            @Override
            public void onReceiveMessageReactionsChanged(final List<MessageReactionChangeInfo> changeInfos) {
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (mLockObject) {
                            for (MessageListener messageListener : mMessageProxyListenerList) {
                                messageListener.onReceiveMessageReactionsChanged(changeInfos);
                            }
                        }
                    }
                });
            }
        };
        nativeSetMessageListener(mMessageListener);
    }

    public void addMessageListener(MessageListener messageListener) {
        synchronized (mLockObject) {
            mMessageProxyListenerList.add(messageListener);
        }
    }

    public String sendMessage(Message message, MessageUploadProgressCallback progressCallback, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return null;
        }

        return nativeSendMessage(message, progressCallback, callback);
    }

    public void revokeMessage(MessageKey messageKey, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeRevokeMessage(messageKey, callback);
    }

    public void modifyMessage(Message message, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeModifyMessage(message, callback);
    }

    public void setC2CMessageRead(String userID, long readTime, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetC2CMessageRead(userID, readTime, callback);
    }

    public void setGroupMessageRead(String groupID, long readSequence, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetGroupMessageRead(groupID, readSequence, callback);
    }

    public void getC2CHistoryMessageList(
        String userID, MessageListGetOption messageListGetOption, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetC2CHistoryMessageList(userID, messageListGetOption, callback);
    }

    public void getGroupHistoryMessageList(
        String groupID, MessageListGetOption messageListGetOption, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupHistoryMessageList(groupID, messageListGetOption, callback);
    }

    public String insertLocalMessage(Message message, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return null;
        }

        return nativeInsertLocalMessage(message, callback);
    }

    public void setLocalCustomString(Message message, String localCustomString) {
        nativeSetLocalCustomString(message, localCustomString);
    }

    public void setLocalCustomNumber(Message message, int localCustomNumber) {
        nativeSetLocalCustomNumber(message, localCustomNumber);
    }

    public void deleteLocalMessage(MessageKey messageKey, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDeleteLocalMessage(messageKey, callback);
    }

    public void deleteCloudMessageList(List<MessageKey> messageKeyList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDeleteCloudMessageList(messageKeyList, callback);
    }

    public void clearC2CHistoryMessage(String userID, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeClearC2CHistoryMessage(userID, callback);
    }

    public void clearGroupHistoryMessage(String groupID, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeClearGroupHistoryMessage(groupID, callback);
    }

    public void clearLocalHistoryMessage(
        ConversationKey conversationKey, long beginTime, long endTime, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeClearLocalHistoryMessage(conversationKey, beginTime, endTime, callback);
    }

    public void findMessageByMessageId(List<String> messageIDList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeFindMessageByMessageID(messageIDList, callback);
    }

    public void searchLocalMessage(MessageSearchParam messageSearchParam, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSearchLocalMessage(messageSearchParam, callback);
    }

    public void searchCloudMessage(MessageSearchParam messageSearchParam, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSearchCloudMessage(messageSearchParam, callback);
    }

    public void downloadMessageElement(DownloadParam downloadParam,
        IMCallback<DownloadProgressInfo> progressInfoIMCallback, IMCallback completeCallback) {
        if (!BaseManager.getInstance().isInited()) {
            if (completeCallback != null) {
                completeCallback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDownloadMessageElement(downloadParam, progressInfoIMCallback, completeCallback);
    }

    public void downloadRelayMessageList(Message message, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeDownloadRelayMessageList(message, callback);
    }

    public boolean isMessageSelfRead(MessageKey messageKey) {
        if (!BaseManager.getInstance().isInited()) {
            IMLog.e(TAG, "sdk not ini");
            return false;
        }

        return nativeIsMessageSelfRead(messageKey);
    }

    public boolean isMessagePeerRead(MessageKey messageKey) {
        if (!BaseManager.getInstance().isInited()) {
            IMLog.e(TAG, "sdk not ini");
            return false;
        }

        return nativeIsMessagePeerRead(messageKey);
    }

    public void sendMessageReceipts(List<MessageKey> messageList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSendMessageReceipts(messageList, callback);
    }

    public void getGroupMessageReceipts(List<MessageKey> messageList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupMessageReceipts(messageList, callback);
    }

    public void getGroupMessageReadMembers(Message message, int filter, long nextSeq, int count, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupMessageReadMembers(message, filter, nextSeq, count, callback);
    }

    public void setMessageExtensions(Message message, List<MessageExtension> extensions, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetMessageExtensions(message, extensions, callback);
    }

    public void getMessageExtensions(Message message, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetMessageExtensions(message, callback);
    }

    public void deleteMessageExtensions(Message message, List<String> keys, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDeleteMessageExtensions(message, keys, callback);
    }

    public void addMessageReaction(Message message, String reactionID, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeAddMessageReaction(message, reactionID, callback);
    }

    public void removeMessageReaction(Message message, String reactionID, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeRemoveMessageReaction(message, reactionID, callback);
    }

    public void getMessageReactions(List<MessageKey> messageKeyList, int maxUserCountPerReaction, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetMessageReactions(messageKeyList, maxUserCountPerReaction, callback);
    }

    public void getAllUserListOfMessageReaction(Message message, String reactionID, int nextSeq, int count,
                                                IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetAllUserListOfMessageReaction(message, reactionID, nextSeq, count, callback);
    }

    public void translateText(
        List<String> sourceTextList, String sourceLanguage, String targetLanguage, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeTranslateText(sourceTextList, sourceLanguage, targetLanguage, callback);
    }

    public void convertVoiceToText(String url, String language, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeConvertVoiceToText(url, language, callback);
    }

    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    protected native void nativeSetMessageListener(MessageListener messageListener);

    protected native String nativeSendMessage(
        Message message, MessageUploadProgressCallback messageUploadProgressCallback, IMCallback callback);

    protected native void nativeRevokeMessage(MessageKey messageKey, IMCallback callback);

    protected native void nativeModifyMessage(Message message, IMCallback callback);

    protected native void nativeSetC2CMessageRead(String userID, long readTime, IMCallback callback);

    protected native void nativeSetGroupMessageRead(String groupID, long readSequence, IMCallback callback);

    protected native void nativeGetC2CHistoryMessageList(
        String userID, MessageListGetOption messageListGetOption, IMCallback callback);

    protected native void nativeGetGroupHistoryMessageList(
        String groupID, MessageListGetOption messageListGetOption, IMCallback callback);

    protected native void nativeFindMessageByMessageID(List<String> messageIDList, IMCallback callback);

    protected native void nativeSearchLocalMessage(MessageSearchParam messageSearchParam, IMCallback callback);

    protected native void nativeSearchCloudMessage(MessageSearchParam messageSearchParam, IMCallback callback);

    protected native String nativeInsertLocalMessage(Message message, IMCallback callback);

    protected native void nativeSetLocalCustomString(Message message, String localCustomString);

    protected native void nativeSetLocalCustomNumber(Message message, int localCustomNumber);

    protected native void nativeDeleteLocalMessage(MessageKey messageKey, IMCallback callback);

    protected native void nativeDeleteCloudMessageList(List<MessageKey> messageList, IMCallback callback);

    protected native void nativeClearC2CHistoryMessage(String userID, IMCallback callback);

    protected native void nativeClearGroupHistoryMessage(String groupID, IMCallback callback);

    protected native void nativeClearLocalHistoryMessage(
        ConversationKey conversationKey, long beginTime, long endTime, IMCallback callback);

    protected native void nativeDownloadMessageElement(
        DownloadParam downloadParam, IMCallback progressCallback, IMCallback completeCallback);

    protected native void nativeDownloadRelayMessageList(Message message, IMCallback callback);

    protected native boolean nativeIsMessageSelfRead(MessageKey messageKey);

    protected native boolean nativeIsMessagePeerRead(MessageKey messageKey);

    protected native void nativeSendMessageReceipts(List<MessageKey> messageList, IMCallback callback);

    protected native void nativeGetGroupMessageReceipts(List<MessageKey> messageList, IMCallback callback);

    protected native void nativeGetGroupMessageReadMembers(
        Message message, int filter, long nextSeq, int count, IMCallback callback);

    protected native void nativeSetMessageExtensions(
        Message message, List<MessageExtension> extensions, IMCallback callback);

    protected native void nativeGetMessageExtensions(Message message, IMCallback callback);

    protected native void nativeDeleteMessageExtensions(Message message, List<String> keys, IMCallback callback);

    protected native void nativeAddMessageReaction(Message message, String reactionID, IMCallback callback);

    protected native void nativeRemoveMessageReaction(Message message, String reactionID, IMCallback callback);

    protected native void nativeGetMessageReactions(List<MessageKey> messageKeyList, int maxUserCountPerReaction,
                                                    IMCallback callback);

    protected native void nativeGetAllUserListOfMessageReaction(Message message, String reactionID, int nextSeq,
                                                                int count, IMCallback callback);

    protected native void nativeTranslateText(
        List<String> sourceTextList, String sourceLanguage, String targetLanguage, IMCallback callback);

    protected native void nativeConvertVoiceToText(String url, String language, IMCallback callback);
}
