package com.thunder.livesdk.system;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.telephony.TelephonyManager;

import com.thunder.livesdk.log.ThunderLog;
import com.thunder.livesdk.helper.ThunderNative;

/**
 * Created by xiaojun on 2018/1/12.
 * Copyright (c) 2017 YY Inc. All rights reserved.
 */

public class ThunderNetStateService {

    private static final String INTERNET_PERMISSION = "android.permission.INTERNET";
    private static final String NETWORK_PERMISSION = "android.permission.ACCESS_NETWORK_STATE";
    private Context mContext;
    public static NetworkInfo.DetailedState lastDetailedState = NetworkInfo.DetailedState.IDLE;

    public static final class NetState {
        public static final int SYSNET_WIFI = 0;
        public static final int SYSNET_MOBILE = 1;
        public static final int SYSNET_DISCONNECT = 2;
        public static final int SYSNET_2G = 3;
        public static final int SYSNET_3G = 4;
        public static final int SYSNET_4G = 5;
        public static final int SYSNET_WLAN = 6;
        public static final int SYSNET_5G = 7;
        public static final int SYSNET_CONNECTING = 20;
        public static final int SYSNET_RECONNECTING = 21;
        public static final int SYSNET_UNKNOWN = 127;
    }

    public ThunderNetStateService(Context mContext) {
        this.mContext = mContext;
    }

    public void init() {
        IntentFilter mFilter = new IntentFilter();
        mFilter.addAction(ConnectivityManager.CONNECTIVITY_ACTION);
        try {
            mContext.registerReceiver(mNetReceiver, mFilter);
        } catch (Exception e) {
            ThunderLog.warn(ThunderLog.kLogTagSdk, "ThunderNetStateService registerReceiver failed, error:" + e);
        }
    }

    public void fini() {
        mContext.unregisterReceiver(mNetReceiver);
    }

    private BroadcastReceiver mNetReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
if (ThunderLog.isInfoValid()) {
            ThunderLog.info(ThunderLog.kLogTagSdk, "NetworkStateService onReceive pid %d",
                    Thread.currentThread().getId());
}
            String action = intent.getAction();
            if (action != null && action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
if (ThunderLog.isInfoValid()) {
                ThunderLog.info(ThunderLog.kLogTagSdk, "current network connectivity action");
}
                updateNetInfo(context);
            }
        }
    };

    private static boolean checkHasPermission(Context context, String perm) {
        int result = context.checkCallingOrSelfPermission(perm);
        return result == PackageManager.PERMISSION_GRANTED;
    }

    private static int getMobileNetworkState(Context context) {
        TelephonyManager telephonyManager =
                (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
        if (telephonyManager == null) {
            ThunderLog.warn(ThunderLog.kLogTagSdk, "cannot get TelephonyManager");
            return NetState.SYSNET_MOBILE;
        }
        int netType = telephonyManager.getNetworkType();
        switch (netType) {
            case TelephonyManager.NETWORK_TYPE_GPRS:
            case TelephonyManager.NETWORK_TYPE_GSM:
            case TelephonyManager.NETWORK_TYPE_EDGE:
            case TelephonyManager.NETWORK_TYPE_CDMA:
            case TelephonyManager.NETWORK_TYPE_1xRTT:
            case TelephonyManager.NETWORK_TYPE_IDEN:
                return NetState.SYSNET_2G;
            case TelephonyManager.NETWORK_TYPE_UMTS:
            case TelephonyManager.NETWORK_TYPE_EVDO_0:
            case TelephonyManager.NETWORK_TYPE_EVDO_A:
            case TelephonyManager.NETWORK_TYPE_HSDPA:
            case TelephonyManager.NETWORK_TYPE_HSUPA:
            case TelephonyManager.NETWORK_TYPE_HSPA:
            case TelephonyManager.NETWORK_TYPE_EVDO_B:
            case TelephonyManager.NETWORK_TYPE_EHRPD:
            case TelephonyManager.NETWORK_TYPE_HSPAP:
            case TelephonyManager.NETWORK_TYPE_TD_SCDMA:
                return NetState.SYSNET_3G;
            case TelephonyManager.NETWORK_TYPE_LTE:
            case TelephonyManager.NETWORK_TYPE_IWLAN:
                return NetState.SYSNET_4G;
//            case TelephonyManager.NETWORK_TYPE_NR:
//                return NetState.SYSNET_5G;
            default:
                ThunderLog.warn(ThunderLog.kLogTagSdk, "unknown mobile network type:" + netType);
                return NetState.SYSNET_MOBILE;
        }
    }

    private static void updateNetInfo(Context context) {
        if (context == null) {
            return;
        }

        try {
            if (!(checkHasPermission(context, INTERNET_PERMISSION) &&
                    checkHasPermission(context, NETWORK_PERMISSION))) {
if (ThunderLog.isInfoValid()) {
                ThunderLog.info(ThunderLog.kLogTagSdk, "cannot get permission INTERNET or ACCESS_NETWORK_STATE!!");
}
                return;
            }

            ConnectivityManager connectivityManager =
                    (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            if (connectivityManager == null) {
if (ThunderLog.isInfoValid()) {
                ThunderLog.info(ThunderLog.kLogTagSdk, "cannot get ConnectivityManager!!");
}
                return;
            }
            NetworkInfo info = connectivityManager.getActiveNetworkInfo();

            int netState = NetState.SYSNET_DISCONNECT;
            if (info != null) {
                NetworkInfo.DetailedState detailState = info.getDetailedState();
                lastDetailedState = detailState;
                if (info.isAvailable()) {
                    int nType = info.getType();
                    if (nType == ConnectivityManager.TYPE_MOBILE) {
                        netState = getMobileNetworkState(context);
if (ThunderLog.isInfoValid()) {
                        ThunderLog.info(ThunderLog.kLogTagSdk, "current network TYPE_MOBILE, netState: %d",
                                netState);
}
                    } else if (nType == ConnectivityManager.TYPE_WIFI) {
                        netState = NetState.SYSNET_WIFI;
if (ThunderLog.isInfoValid()) {
                        ThunderLog.info(ThunderLog.kLogTagSdk, "current network wifi");
}
                    } else {
                        ThunderLog.warn(ThunderLog.kLogTagSdk, "current network %s is omitted",
                                info.getTypeName());
                        return;
                    }
                } else {
                    if (detailState == NetworkInfo.DetailedState.CONNECTING) {
                        if (lastDetailedState.ordinal() == NetworkInfo.DetailedState.FAILED.ordinal() ||
                                lastDetailedState.ordinal() == NetworkInfo.DetailedState.BLOCKED.ordinal()) {
                            netState = NetState.SYSNET_RECONNECTING;
                        } else {
                            netState = NetState.SYSNET_CONNECTING;
                        }
                    } else {
                        netState = NetState.SYSNET_DISCONNECT;
                    }
if (ThunderLog.isInfoValid()) {
                    ThunderLog.info(ThunderLog.kLogTagSdk, "current network No usable network!!");
}
                }
            }

            ThunderNative.notifyNetState(netState);
        } catch (Exception exc) {
            ThunderLog.warn(ThunderLog.kLogTagSdk, "update net info error:" + exc);
        }
    }
}
