package com.thunder.livesdk;

import com.yy.mediaframework.model.Rect;

public abstract class ThunderEventHandler {
    /**
     * Callback of sdk error information (error definition not determined)
     *
     * @param error
     */
    public void onError(int error) {}

    /**
     * Notification for joining room successfully. 
     * After joinRoom is called, this notification will be received, indicating that the connection with the server is normal and the interface only available for “joining room successfully” can be called
     * @param room Room ID
     * @param uid User ID
     * @param elapsed This parameter indicates time consumed by joining room, that is, time (in millisecond) elapsed from the calling of joinRoom to the occurrence of event
     */
    public void onJoinRoomSuccess(String room, String uid, int elapsed) {}

    /**
     * Notification for leaving room
     * Call leaveRoom, and you will receive this notification after leaving room normally
     *
     * @param status This function has not been implemented yet
     */
    public void onLeaveRoom(RoomStats status) {}

    /**
     * Notification for authentication service result
     * When required service authentication is configured for the service, you will receive this notification when a media stream uplinks
     * @param bPublish Whether to publish (speak as an anchor)
     * @param result Authentication result: 0 indicates succeeded and other values indicates failed (which are customized by the developer and only passed through by sdk)
     *
     */
    public void onBizAuthResult(boolean bPublish, int result) {}

    /**
     * Notification for authentication service result
     * When required service authentication is configured for the service, you will receive this notification when a media stream uplinks
     * @param bPublish Whether to publish (speak as an anchor)
     * @param bizAuthStreamType type Type of the authenticated stream, refer to LiveBizAuthStreamType
     * @param bizAuthResult Authentication result: 0 indicates succeeded and other values indicates failed (which are customized by the developer and only passed through by sdk)
     *
     */
    public void onBizAuthStreamResult(boolean bPublish, int bizAuthStreamType, int bizAuthResult) {}

    /**
     * Notification for SDK authentication result
     * After calling joinRoom, you will receive user’s authentication notification in the presence of uplink and downlink media data
     *
     * @param result Authentication result
     *               (0-authentication succeeded 10000-internal server error, and try again 10001-no token, and [YYLiveAPI updateToken:] should be called
     *               10002-token validation failed (incorrect digital signature), and the used appSecret may be incorrect
     *               10003-appid in token is inconsistent with appid in authentication 10004-uid in token is inconsistent with uid in authentication 10005-token expired
     *               10006-app does not exist and is not registered on the management background 10007-token about to expire 10008-user banned
     *               {@link ThunderRtcConstant.AuthResult}
     */
    public void onSdkAuthResult(int result) {}

    /**
     * Notification for user banned
     * You will receive this notification when the banning status of user changes
     *
     * @param status Banning status true=banned false=unbanned
     */
    public void onUserBanned(boolean status) {}

    /**
     * Notification for user joining current room
     * After the local user joins room, you will receive this notification while other users join this room. The notification only takes effect in the audio-only mode
     *
     * @param uid     User ID of remote user/anchor
     * @param elapsed The delay (in millisecond) from calling joinRoom to triggering this callback by the local user
     */
    public void onUserJoined(String uid, int elapsed) {}

    /**
     * Notification for user leaving current room
     * After the local user joins room, you will receive this notification while other users leave this room. The notification only takes effect in the audio-only mode
     *
     * @param uid    User ID of remote user/anchor
     * @param reason Offline reasons
     *               1-User leaves voluntarily
     *               2-Timeout dropped connection is caused due to no data packet of the other party received for long time, and may be determined by mistake for the reason that this party does not receive the leaving message of the other party, who leaves voluntarily
     *               3-User identity is switched to audience from anchor (in live streaming mode)
     *               {@link ThunderRtcConstant.UserOfflineReason}
     */
    public void onUserOffline(String uid, int reason) {}

    /**
     * Token Notification for token about to expire
     * You will receive this notification when user’s token is about to expire
     *
     * @param token Token about to become invalid
     */
    public void onTokenWillExpire(byte[] token) {}

    /**
     * Token Notification for token expired
     * You will receive this notification when user’s token is expired
     */
    public void onTokenRequested() {}

    /**
     * Notification for user’s uplink and downlink network quality report, and the network quality of user under corresponding ID is returned
     *
     * @param uid       Indicating that this callback reports the network quality of user using this ID. When uid is 0, the network quality of local user is returned
     * @param txQuality Network uplink quality (0-unknown quality 1-excellent quality 2-subjective user experience almost equal to excellent quality, whereas the bit rate may be slightly lower than that of excellent quality
     *                  3-poor subjective user experience, but no impacts on communication 4-poor but not smooth communication 5-very bad network quality, basically no communication
     *                  6-Network disconnected, and no communication
     *                  {@link ThunderRtcConstant.NetworkQuality}
     * @param rxQuality Network downlink quality (0-unknown quality 1-excellent quality 2-subjective user experience almost equal to excellent quality, whereas the bit rate may be slightly lower than that of excellent quality
     *                  3-poor subjective user experience, but no impacts on communication 4-poor but not smooth communication 5-very bad network quality, basically no communication
     *                  6-Network disconnected, and no communication)
     *                  {@link ThunderRtcConstant.NetworkQuality}
     */
    public void onNetworkQuality(String uid, int txQuality, int rxQuality) {}

    /**
     * Notification for upstream and downstream traffic (periodic notification, notifying once every 2 seconds)
     * You will receive this notification after the user joins the channel
     *
     * @param stats For specific status, see {@link ThunderNotification.RoomStats}
     */
    public void onRoomStats(ThunderNotification.RoomStats stats) {}

    /**
     * Notification for playing volume
     * After setting setAudioVolumeIndication, you will receive this notification while someone speaks in this room
     *
     * @param speakers    User’s volume and capture timestamp
     * @param totalVolume Total volume (after audio mixing) in [0-100]
     *
     */
    public void onPlayVolumeIndication(ThunderEventHandler.AudioVolumeInfo[] speakers, int totalVolume) {}

    /**
     * Callback of capture volume
     *
     * @param totalVolume Energy value of uplink volume in [0-100]
     * @param cpt         Capture timestamp
     * @param micVolume   Only volume energy value captured by microphone in [0-100]
     *                    Disabled by default, enabling/disabling: enableCaptureVolumeIndication
     */
    public void onCaptureVolumeIndication(int totalVolume, int cpt, int micVolume) {}

    /**
     * TODO: The interface does not provide this function
     * Functions have not been implemented yet
     *
     * @param uid
     * @param quality
     * @param delay
     * @param lost
     */
    public void onAudioQuality(String uid, int quality, short delay, short lost) {}

    /**
     * Notification for disconnecting from the server network
     * After calling joinRoom, you will receive this notification while SDK is disconnected with server network
     *
     * TODO: Original remarks: The functions have not been implemented yet, whereas the interface provides this function
     */
    public void onConnectionLost(){}

    /**
     * TODO: The interface does not provide this function
     * Functions have not been implemented yet
     */
    public void onConnectionInterrupted() {}

    /**
     * Callback of audio play data
     *
     * @param data     Data before decoding
     * @param cpt      Capture timestamp
     * @param pts      Play timestamp
     * @param uid      User ID
     * @param duration Duration
     *                 Disabled by default, enabling/disabling: enableAudioDataIndication
     */
    public void onAudioPlayData(byte[] data, long cpt, long pts, String uid, long duration) {}

    /**
     * Callback of audio play spectrum data
     *
     * @param data Numerical range in [0-100]
     *             Disabled by default, enabling/disabling: enableAudioPlaySpectrum
     */
    public void onAudioPlaySpectrumData(byte[] data) {}

    /**
     * @deprecated use {@link ThunderEventHandler.IAudioFrameObserver} instead
     * Callback of audio capture data
     *
     * @param data       Captured pcm data
     * @param dataSize   Data size
     * @param sampleRate Sampling rate of data
     * @param channel    Number of audio tracks of data
     *                   Disabled by default, enabling/disabling: enableCapturePcmDataCallBack
     */
    public void onAudioCapturePcmData(byte[] data, int dataSize, int sampleRate, int channel) {}

    /**
     * @deprecated  use {@link ThunderEventHandler.IAudioFrameObserver} instead
     * Callback of audio rendering data
     *
     * @param data       Captured pcm data
     * @param dataSize   Data size
     * @param sampleRate Sampling rate of data
     * @param channel    Number of audio tracks of data
     * @param duration   Data duration in ms
     *                   Disabled by default, enabling/disabling: enableRenderPcmDataCallBack
     */
    public void onAudioRenderPcmData(byte[] data, int dataSize, long duration, int sampleRate, int channel) {}

    /**
     * Callback of received pass-through protocol message
     *
     * @param data Pass-through protocol message
     * @param uid  uid for sending this message
     */
    public void onRecvUserAppMsgData(byte[] data, String uid) {}

    /**
     * Callback of sending failed status of pass-through protocol
     *
     * @param status Failed status (1-too frequently 2-too large size of sent data 3-publishing failed)
     *               The pass-through frequency specified at present is 2 times/s, and the size of sent data is limited to be <=200Byte
     */
    public void onSendAppMsgDataFailedStatus(int status) {}

    /**
     * Notification for enabling/disabling remote user’s audio streams
     * After calling joinRoom, you will receive this notification when the statuses of audio streams stored in the room, and follow-up audio streams change
     *
     * @param uid  User uid
     * @param stop true: disabled, false: enabled
     */
    public void onRemoteAudioStopped(String uid, boolean stop) {}

    /**
     * Notification for enabling/disabling remote user’s video streams
     * After calling joinRoom, you will receive this notification when the statuses of video streams stored in the room, and follow-up video streams change
     *
     * @param uid  User uid
     * @param stop true: disabled, false: enabled
     */
    public void onRemoteVideoStopped(String uid, boolean stop) {}

    /**
     * this notification is received when a remote user's audio stream changes
     * @param roomId room id
     * @param uid  User uid
     * @param arrive true: remote user audio stream received; false: remote user audio stream not received
     * @remark (1) remote users will receive this notification when calling stopLocalAudioStream
     *         (2) The callback is not periodic and will only be received when the state changes
     */
    public void onRemoteAudioArrived(String roomId, String uid, boolean arrive) {}

    /**
     * this notification is received when a remote user's video stream changes
     * @param roomId room id
     * @param uid  User uid
     * @param arrive true: remote user video stream received; false: remote user video stream not received
     * @remark (1) remote users will receive this notification when calling stopLocalVideoStream
     *         (2) The callback is not periodic and will only be received when the state changes
     */
    public void onRemoteVideoArrived(String roomId, String uid, boolean arrive) {}

    /**
     * Notification for displayed start frame of remote video
     * After calling setRemoteVideoCanvas, you will receive this notification when the video streams are received and displayed in the window
     *
     * @param uid     User ID
     * @param width   Video dimension - width
     * @param height  Video dimension - height
     * @param elapsed Elapsed time, indicating the time (in millisecond) from the calling of joinRoom to the occurrence of event
     */
    public void onRemoteVideoPlay(String uid, int width, int height, int elapsed) {}

    /**
     * Notification for changes of local or remote video’s resolution
     * After calling joinRoom, you will receive this notification while the video’s resolution changes
     *
     * @param uid      User ID
     * @param width    Video dimension - width
     * @param height   Video dimension - height
     * @param rotation Reserved definition, not implemented
     */
    public void onVideoSizeChanged(String uid, int width, int height, int rotation) {}

    /**
     * Notification for successfully sending the start frame of local audio
     * You will receive this notification while the user uplinks the audio stream
     *
     * @param elapsed Elapsed time, indicating the time (in millisecond) from the calling of joinRoom to the occurrence of event
     */
    public void onFirstLocalAudioFrameSent(int elapsed) {}

    /**
     * Notification for successfully sending the start frame of local video
     * You will receive this notification while the user uplinks the video stream
     *
     * @param elapsed Elapsed time, indicating the time (in millisecond) from the calling of joinRoom to the occurrence of event
     */
    public void onFirstLocalVideoFrameSent(int elapsed) {}

    /**
     * Notification for cCDN stream publishing result
     * After the user calls addPublishOriginStreamUrl or addPublishTranscodingStreamUrl for performing stream publishing, you will receive this notification when the status changes
     *
     * @param url       URL for stream publishing
     * @param errorCode For error codes of stream publishing, see {@link ThunderRtcConstant.ThunderPublishCDNErrorCode}
     */
    public void onPublishStreamToCDNStatus(String url, int errorCode) {}

    /**
     * Notification for network status
     * After “initialization”, you will receive this notification when the network type changes
     *
     * @param type For current network status, see {@link ThunderRtcConstant.ThunderNetworkType}
     */
    public void onNetworkTypeChanged(int type) {}

    /**
     * Notification for connection statuses with the server network (including service and avp)
     * After calling joinRoom, you will receive this notification when the connection status between SDK and the server network changes
     * In the thunder mode, only the connection status with avp is required
     * In the thunderbolt mode and in the case of no publishing or subscription (avp will not be connected), only the connection status with service is required
     * In the thunderbolt mode and in the case of publishing or subscription, the connection statuses with service and avp are required
     *
     * @param status {@link ThunderRtcConstant.ThunderConnectionStatus}
     */
    public void onConnectionStatus(int status) {}

    /**
     * Notification for changes of audio device’s capture status
     *
     * @param status {@link ThunderRtcConstant.ThunderAudioDeviceStatus}
     */
    public void onAudioCaptureStatus(int status) {}

    /**
     * Notification for changes of camera’s capture status
     *
     * @param status {@link ThunderRtcConstant.ThunderVideoCaptureStatus}
     */
    public void onVideoCaptureStatus(int status) {}

    /**
     * The callback of local video stream statistics
     * Statistics information of local video stream sent by local device is described during this callback. Callback time: 1. immediate callback when the publishing interface is called; 2. immediate callback on bracket change during the publishing; and 3. periodical callback at an interval of 2s.
     * @param stats Statistics data of local video, see {@link ThunderEventHandler.LocalVideoStats} for detailed definition
     */
    public void onLocalVideoStats(ThunderEventHandler.LocalVideoStats stats) {}

    /**
     * Statistical information callback for local audio streams
     * This callback describes statistical information of audio streams sent by local device, with the callback timing: periodic callback at the interval of 2s
     * @param stats Statistical data of local audios. For details, see {@link ThunderEventHandler.LocalAudioStats}
     */
    public void onLocalAudioStats(ThunderEventHandler.LocalAudioStats stats) {}

    /**
     * Statistics information callback of remote video stream
     * The end-to-end video stream status in calling of remote users is described during this callback, which is triggered once 2s for each remote user/anchor. This callback will be triggered for multiple times every 2 seconds in case that there are multiple remote users/anchors
     * @param uid Remote user/anchor ID
     * @param stats Statistical data of remote video. For details, see {@link ThunderEventHandler.RemoteVideoStats}
     */
    public void onRemoteVideoStatsOfUid(String uid, ThunderEventHandler.RemoteVideoStats stats) {}

    /**
     * Statistics information callback of remote audio stream
     * The statistics of end-to-end audio in calling of remote users is described during this callback, which is triggered once 2s for each remote user/anchor. SDK triggers for multiple times every 2 seconds in case that there are multiple remote users/anchors
     * @param uid Remote user/anchor ID
     * @param stats Statistical data of remote audio. For details, see {@link ThunderEventHandler.RemoteAudioStats}
     */
    public void onRemoteAudioStatsOfUid(String uid, ThunderEventHandler.RemoteAudioStats stats) {}

    /**
     * Reporting audio publishing status of remote user
     * @param uid uid
     * @param state 音频状态，详细定义见 {@link ThunderRtcConstant.RemoteAudioState}
     * @param reason 在该状态的原因，详细定义见 {@link ThunderRtcConstant.RemoteAudioReason}
     * @param elapsed 从调用joinRoom到发生此事件经过的时间（毫秒）
     */
    public void onRemoteAudioStateChangedOfUid(String uid, int state, int reason, int elapsed) {}

    /**
     * 远端用户视频状态改变回调
     * @param uid 远端用户/主播id
     * @param state 视频流状态，详细定义见 {@link ThunderRtcConstant.RemoteVideoState}
     * @param reason 视频流状态改变的原因，详细定义见 {@link ThunderRtcConstant.RemoteVideoReason}
     * @param elapsed 从调用joinRoom到发生此事件经过的时间（毫秒）
     */
    public void onRemoteVideoStateChangedOfUid(String uid, int state, int reason, int elapsed) {}

    /**
     * 已播放远端音频首帧回调
     * @param uid 远端用户 uid
     * @param elapsed 从调用joinRoom到发生此事件经过的时间（毫秒）
     */
    public void onRemoteAudioPlay(String uid, int elapsed) {}

    /**
     * 本地音频状态
     * 本地音频系统状态发生改变时，SDK 会触发回调当前本地音频状态和错误原因
     * @param status 本地音频状态 {@link ThunderRtcConstant.LocalAudioStreamStatus}
     * @param errorReason 本地音频错误原因 {@link ThunderRtcConstant.LocalAudioStreamErrorReason}
     */
    public void onLocalAudioStatusChanged(int status, int errorReason) {}

    /**
     * 本地视频状态
     * 本地视频系统状态发生改变时，SDK 会触发回调当前本地视频状态和错误原因
     * @param status 本地视频状态 {@link ThunderRtcConstant.LocalVideoStreamStatus}
     * @param errorReason 本地视频错误原因 {@link ThunderRtcConstant.LocalVideoStreamErrorReason}
     */
    public void onLocalVideoStatusChanged(int status, int errorReason) {}

    /**
     * CPU/内存使用信息
     * 该回调描述CPU/内存使用情况，加入房间后，每两秒钟回调一次
     * @param stats CPU/内存使用信息，详细定义见 {@link ThunderEventHandler.DeviceStats}
     */
    public void onDeviceStats(ThunderEventHandler.DeviceStats stats) {}

    /**
     * Callback of user role switch
     * This callback will be triggered when a local user calls switchUserRole to switch the role after joining a room and the server returns the success of the switch
     * @param oldRole Role before switching, see details in {@link ThunderRtcConstant.ThunderUserRole}
     * @param newRole Role after the switch, see details {@link ThunderRtcConstant.ThunderUserRole}
     */
    public void onUserRoleChanged(int oldRole, int newRole) {}


    /**
     * @brief 语音路由已变更回调
     * @param  routing 当前语音路由的设备 {@link ThunderRtcConstant.ThunderAudioOutputRouting}
     * @remark 该回调通知是在语音路由发生切换时才会回调，非实时回调
     */
    public void onAudioRouteChanged(int routing) {}

    /**
     * @brief 啸叫检测结果通知回调
     * @param bHowling true:检测到啸叫；false:没有检测到啸叫
     * @remark (1) enableHowlingDetector(true)并且在打开麦克风的状态下才会收到该回调
     *         (2) 回调不是周期的，只有状态变化时，才会收到该回调
     *         (3) enableHowlingDetector(true)的状态下，关闭麦克风再打开麦克风会收到一次该回调
     *         (4) 麦克风打开的状态状态下，从enableHowlingDetector(false)到enableHowlingDetector(true)会收到一次该回调
     */
    public void onHowlingDetectResult(boolean bHowling) {}

    /**
     * @brief 摄像头对焦区域已改变回调。
     * @param rect 镜头内表示对焦的区域
     * @remark (1) 摄像头对焦区域已改变回调。
     *         (2)该回调表示相机的对焦区域发生了改变。
     *         (3)该回调是由本地用户调用 {@link com.thunder.livesdk.ThunderEngine#setCameraFocusPositionInPreview} 方法改变对焦位置触发的。
     */
    public void onCameraFocusAreaChanged(Rect rect) {}

    /**
     * @brief 摄像头曝光区域已改变回调。
     * @param rect 镜头内表示曝光的区域
     * @remark (1) 摄像头曝光区域已改变回调。
     *         (2)该回调表示相机的曝光区域发生了改变。
     *         (3)该回调是由本地用户调用 {@link com.thunder.livesdk.ThunderEngine#setCameraExposurePosition} 方法改变对焦位置触发的。
     */
    public void onCameraExposureAreaChanged(Rect rect) {}

    public static class AudioVolumeInfo {
        public String uid; // User ID
        public int volume; // Volume
        public int pts; // pts
        public int originalVolume; // originalVolume, No gain effect

        public AudioVolumeInfo() {
        }
    }

    /**
     * @brief 漏回声检测结果通知回调
     * @param bEcho true:检测到漏回声；false:没有检测到漏回声
     * @remark (1) enableEchoDetector(true)并且在打开麦克风的状态下才会收到该回调
     *         (2) 回调不是周期的，只有状态变化时，才会收到该回调
     *         (3) enableEchoDetector(true)的状态下，关闭麦克风再打开麦克风会收到一次该回调
     *         (4) 麦克风打开的状态状态下，从enableEchoDetector(false)到enableEchoDetector(true)会收到一次该回调
     */
    public void onEchoDetectResult(boolean bEcho) {}

    /**
     * @brief 设备检测采集音量通知回调
     * @param volume 音量值： 【0~100】
     * @remark (1) StartInputDeviceTest()才会收到该回调
     */
    public void onAudioInputDeviceTestVolume(int volume) {}

    /**
     * @brief 设备检测播放音量通知回调
     * @param volume 音量值： 【0~100】
     * @remark (1) StartOutputDeviceTest(String fileName)才会收到该回调
     */
    public void onAudioOutputDeviceTestVolume(int volume) {}

    /**
     * @brief 私有回调
     * @param key 表示描述具体回调事件的标示 {@link com.thunder.livesdk.ThunderRtcConstant.ThunderPrivateCallbackKey}
     * @param value 标准json串，用以表示需要回调事件的各种参数
     * @remark value表述 的 json 串的格式在供私有回调的说明文档中给出
     */
    public void onParamsCallback(int key, String value){}

    /**
     * @brief 音频录音状态回调
     * @param errorCode 录音状态错误码, 参考{@link ThunderRtcConstant.ThunderAudioRecordStateCode}
     * @param duration 录音文件时长， <=0：无效时长， >0: 有效时长, 单位:ms
     * @remark 触发时机，在startAudioRecord或StopAudioRecord的时候触发
     */
    public void onAudioRecordState(int errorCode, int duration) {}
    public static class RoomStats {
        public int temp;

        public RoomStats() {
        }
    }

    public static class MixAudioInfo {
        public String uid;
        public int volume;    //[0,100]
    }

    public static class MixVideoInfo {
        public String uid;   //User ID
        public int width;    //Original width of this user's video
        public int height;   //Original height of this user's video
        public int cropX;    //X coordinate of the begin point used to crop the original video in video mixing
        public int cropY;    //Y coordinate of the begin point used to crop the original video in video mixing
        public int cropW;    //Width of the original video to be cropped in video mixing
        public int cropH;    //Height of the original video to be cropped in video mixing
        public int layoutX;  //X coordinate of the begin point of this user's video in video mixing canvas
        public int layoutY;  //Y coordinate of the begin point of this user's video in video mixing canvas
        public int layoutW;  //Width of this user's video in video mixing canvas
        public int layoutH;  //Height of this user's video in video mixing canvas
        public int zOrder;   //Layer number of this user's video frame in video mixing. The value range is the integer in [0, 100], and the minimum value is 0, indicating that the image in this region is at the lowest level
        public float alpha;  //Transparency of this user's video frame in video mixing. The value range is [0.0, 1.0]. 0.0 indicates that the image in this region is completely transparent, while 1.0 indicates completely opaque.
    }

    public static class LocalVideoStats {
        public int sentBitrate;            // Actual sending bit rate (unit: Kbps), with the exception of sending bit rate of reloaded videos after packet loss
        public int sentFrameRate;          // Actual sending frame rate (unit: fps), with the exception of sending frame rate of reloaded videos after packet loss
        public int renderOutputFrameRate;  // Output frame rate of local preview (unit: fps)
        public int targetBitRate;          // Target encoding bit rate of current encoder (unit: Kbps). This bit rate is a value estimated by this SDK in accordance with current network status.
        public int targetFrameRate;        // Target encoding frame rate of current coder (unit: fps)
        public int qualityAdaptIndication; // Quality adaptability of local videos since last statistics (based on target frame rate and target bit rate), see ThunderRtcConstant.ThunderVideoQualityAdapt for detailed definition
        public int encoderOutputFrameRate; // Output frame rate of local coder (unit: fps)
        public int encodedBitrate;         // Encoding bit rate of video (Kbps). This parameter includes no encoding bit rate of reloaded videos after packet loss.
        public int encodedFrameWidth;      // Encoded video width (px)
        public int encodedFrameHeight;     // Encoded video height (px)
        public int encodedFrameCount;      // Number of frames sent by video, an accumulated value
        public int encodedType;            // Encoding type, hardware or software encoding, see ThunderRtcConstant.ThunderVideoEncodedType for detailed definition
        public int codecType;              // Codec type, see ThunderRtcConstant.ThunderVideoCodecType for detailed definition
        public int configBitRate;          // Bit rate configured in the background (unit: Kbps)
        public int configFrameRate;        // Frame rate configured in the background
        public int configWidth;            // Video width configured in the background
        public int configHeight;           // Video height configured in the background

        public LocalVideoStats(){
        }
    }

    public static class LocalAudioStats {
        public int encodedBitrate; // 编码码率 Kbps
        public int numChannels;    // Quantity of tracks
        public int sendSampleRate; // Sampling rate sent (unit: Hz)
        public int sendBitrate;    // Average value of bit rate of sent data (unit: Kbps)
        public int enableVad;      // 发送的音频流采用的VAD，0：不开启；1：开启
        public LocalAudioStats() {

        }
    }
	
    public static class RemoteVideoStats {
        public int delay; // Captured play delay
        public int width;  // Width of remote video stream
        public int height;  // Height of remote video stream
        public int receivedBitrate;  // Receiving bit rate (unit: Kbps)
        public int decoderOutputFrameRate;  // Output frame rate of remote video decoder (unit: fps)
        public int rendererOutputFrameRate; // Output frame rate of remote video renderer (unit: fps)
        public int packetLossRate; // Packet loss rate (%) of remote video after network confrontation
        public int rxStreamType; // Types of video stream, including large stream and small stream
        public int totalFrozenTime; // The accumulated time (ms) from the time when a remote user joins a channel to the time of video freezing
        public int frozenRate; // The percentage (%) of the accumulated time (from the time when a remote user joins a channel to the time of video freezing) accounting for total effective time of videos
        public int codecType; // Codec type, H264/H265/VP8, see ThunderRtcConstant.ThunderVideoCodecType for detailed definition
        public int decodedType; // Decoding type, hardware or software decoding, see ThunderRtcConstant.ThunderVideoDecodedType for detailed definition

        public RemoteVideoStats() {
        }
    }

    public static class RemoteAudioStats {
        public int quality; // Quality of audio streams sent by remote users.
                            // 0: Unknown; 1: Excellent; 2: Good; 3: Poor, flawed but does not affect communication; 4: Bad, communication can be made but not smoothly; 5: Very Bad, communication can barely be made; 6: Disconnected, communication can not be made at all
        public int networkTransportDelay;  // Network delay from an audio sending end to an audio receiving end
        public int jitterBufferDelay;  // Delay from a receiving end to network jitter buffer
        public int totalDelay; // 主播采集音频到观众播放音频的总延时
        public int frameLossRate;  // Frame loss rate (%) of remote audio streams
        public int numChannels;  // Quantity of tracks
        public int receivedSampleRate; // Sampling rate (Hz) of remote audios
        public int receivedBitrate; // Average bit rate of remote audios within a statistical period
        public int totalFrozenTime; // The accumulated time (ms) from the time when a remote user joins a channel to the time of audio freezing; during one statistical period, an audio frame loss rate of 4% is recorded as a single freeze
                                    // totalFrozenTime = number of times of audio freezing * 2 * 1000 (ms)
        public int frozenRate; // The percentage (%) of the accumulated time (from the time when a remote user joins a channel to the time of audio freezing) accounting for total effective time of audios

        public RemoteAudioStats() {
        }
    }

    public static class LocalAudioStatusChanged {
        public  int status; // 本地音频状态
        public  int errorReason; // 本地音频错误原因

        public LocalAudioStatusChanged() {
        }
    }

    public static class DeviceStats {
        public double cpuTotalUsage; // 当前系统的 CPU 使用率(%)
        public double cpuAppUsage; // 当前 App 的 CPU 使用率 (%)
        public double memoryAppUsage; // 当前 App 的内存占比 (%)
        public double memoryTotalUsage; // 当前系统的内存占比 (%)

        public DeviceStats() {
        }
    }
}
