package com.thunder.livesdk.video;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.graphics.PointF;
import android.os.Handler;
import android.os.Looper;

import com.thunder.livesdk.ExternalVideoSource;
import com.thunder.livesdk.ThunderConstant;
import com.thunder.livesdk.ThunderDefaultCamera;
import com.thunder.livesdk.ThunderEngine;
import com.thunder.livesdk.ThunderExternalVideoFrame;
import com.thunder.livesdk.ThunderPreviewConfig;
import com.thunder.livesdk.ThunderPublisher;
import com.thunder.livesdk.ThunderRtcConstant;
import com.thunder.livesdk.ThunderScreenCapture;
import com.thunder.livesdk.ThunderVideoCapture;
import com.thunder.livesdk.ThunderVideoEncodeParam;
import com.thunder.livesdk.log.ThunderLog;
import com.thunder.livesdk.helper.ThunderNative;
import com.thunder.livesdk.video.serviceConfig.VideoConfigManager;
import com.thunder.livesdk.video.serviceConfig.VideoLiveConfig;
import com.yy.mediaframework.CameraInterface;
import com.yy.mediaframework.CameraListener;
import com.yy.mediaframework.YMFExternalFrame;
import com.yy.mediaframework.api.YMFVideoEncodeFrame;
import com.yy.mediaframework.Constant;
import com.yy.mediaframework.IPublishListener;
import com.yy.mediaframework.ITextureListener;
import com.yy.mediaframework.YMFLiveAPI;
import com.yy.mediaframework.YYVideoCodec;
import com.yy.mediaframework.base.VideoEncoderType;
import com.yy.mediaframework.base.VideoPublisheParam;
import com.yy.mediaframework.base.YMFLowStreamEncoderConfig;
import com.yy.mediaframework.inteligence.common.ResolutionModifyConfig;
import com.yy.mediaframework.model.Rect;
import com.yy.mediaframework.stat.IYMFBehaviorEventListener;
import com.yy.mediaframework.stat.IYMFExceptionListener;
import com.yy.mediaframework.stat.VideoDataStat;
import com.yy.mediaframework.stat.YMFLiveExceptionStat;
import com.yy.mediaframework.stat.YMFLiveExceptionType;
import com.yy.mediaframework.stat.YMFLiveStatisticManager;
import com.yy.mediaframework.stat.YMFLiveUsrBehaviorStat;
import com.yy.videoplayer.decoder.YYVideoLibMgr;

import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import static com.thunder.livesdk.ThunderExternalVideoSource.THUNDER_EXTERNAL_VIDEO_TYPE_TEXTURE;
import static com.yy.mediaframework.Constant.AnchorStatus.AnchorStatus_Encoding;

public class ThunderVideoPublishEngineImp implements ITextureListener, IPublishListener,
        IYMFBehaviorEventListener, IYMFExceptionListener, CameraListener {

    private static final String TAG = "ThunderVideoPublishEngineImp";
    private YMFLiveAPI mPublisher = null; // 开播库对象
    private static long mCallBackPtr = 0;

    private final static int VIDEO_STAT_FPS = 0;
    private final static int VIDEO_STAT_BITRATE = 1;
    private final static int VIDEO_STATE_RESOLUTION = 2;
    private final static int VIDEO_PREVIEW_FRAME_RATE  = 3;// 预览帧率
    private final static int VIDEO_DYNAMIC_ENC_FRAME_RATE = 4;// 根据动态码率调整的帧率
    private final static int VIDEO_ENCODED_TYPE = 5; // 编码形式

    private ThunderVideoHiidoUtil mVideoPubHiidoUtil = null;
    private ThunderVideoCapture mCapture = null;
    private int mCaptureType = ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_DEFAULT_CAMERA;
    private int mCameraState = ThunderRtcConstant.ThunderVideoCaptureStatus.THUNDER_VIDEO_CAPTURE_STATUS_CLOSE;

    private float mCurZoomFactor = 0f;
    private boolean bStartEncode = false;
    private boolean bUserCallEncode = false;
    private boolean bStartCameraCapture = true;
    private Handler mUiHandler = null;
    /**
     * 开播状态回调
     */
    @Override
    public void onVideoAnchorStatus(Constant.AnchorStatus status) {
        if (!bUserCallEncode && status == AnchorStatus_Encoding) {
            return;
        }
        if (status == AnchorStatus_Encoding) {
            bUserCallEncode = false;
        }
        if (mCallBackPtr != 0) {
            onVideoAnchorStatus(mCallBackPtr, status.ordinal());
        }
    }

    @Override
    public void onVideoFrameProcessTime(float maxTime, float avgTime) {
        if(mCallBackPtr != 0) {
            onVideoFrameProcessTime(mCallBackPtr, avgTime, maxTime);
        }
    }

    /**
     * 视频数据回调函数，传回的是编码后的数据
     */
    @Override
    public void onEncodeFrameData(YMFVideoEncodeFrame videoEncodeFrame) {
        int encodeType = ThunderRtcConstant.ThunderVideoEncodeType.THUNDERVIDEO_ENCODE_TYPE_H264;
        boolean bHardware = false;
        byte[] data  = videoEncodeFrame.data;
        int len = videoEncodeFrame.len;
        int svcTid = videoEncodeFrame.svcTid;
        int svcSid = videoEncodeFrame.svcSid;
        int frameType = videoEncodeFrame.frameType;
        int streamId = videoEncodeFrame.streamId; // 大小流ID标识
        long pts = videoEncodeFrame.pts;
        long dts = videoEncodeFrame.dts;
        VideoEncoderType encoderType = videoEncodeFrame.encodeType;
        byte[] extraData = videoEncodeFrame.extraData;
        int extraDataLen = videoEncodeFrame.extraDataLen;
        switch (encoderType) {
            case SOFT_ENCODER_X264:
                bHardware = false;
                encodeType = ThunderRtcConstant.ThunderVideoEncodeType.THUNDERVIDEO_ENCODE_TYPE_H264;
                break;
            case HARD_ENCODER_H264:
                bHardware = true;
                encodeType = ThunderRtcConstant.ThunderVideoEncodeType.THUNDERVIDEO_ENCODE_TYPE_H264;
                break;
            case SOFT_ENCODER_H265:
                bHardware = false;
                encodeType = ThunderRtcConstant.ThunderVideoEncodeType.THUNDERVIDEO_ENCODE_TYPE_H265;
                break;
            case HARD_ENCODER_H265:
                bHardware = true;
                encodeType = ThunderRtcConstant.ThunderVideoEncodeType.THUNDERVIDEO_ENCODE_TYPE_H265;
                break;
            case DEFAULT:
            case ERROR:
                ThunderLog.warn(ThunderLog.kLogTagVideo, "unknown encoder type" + encoderType.toString());
                break;
            default:
                break;
        }

        if (videoEncodeFrame.yySeiData != null) {
            ThunderNative.sendMediaExtraInfo(pts, videoEncodeFrame.yySeiData);
        }

        if (mCallBackPtr != 0) {
            onVideoEncodedFrame(mCallBackPtr, data, len, pts, dts, frameType, encodeType,
                    bHardware, svcTid, svcSid, streamId,extraData, extraDataLen);
        }
    }

    @Override
    public void onUpdateVideoSizeChanged(long l, int i, int i1) {
        if (mCallBackPtr != 0) {
            onUpdateVideoSizeChanged(mCallBackPtr, l, i, i1);
        }
    }

    @Override
    public void notifyCameraPreviewParameter(int i, int i1, int i2, CameraInterface.CameraResolutionMode cameraResolutionMode) {}
    @Override
    public void notifyCameraOpenFail(String s) {}
    @Override
    public void onDualOpen(boolean b) {}
    @Override
    public void notifyCameraOpenSuccess() {}
    @Override
    public void onDualPictureSwitch() {}
    @Override
    public void reSetEncodingState() {}

    @Override
    public void onCameraFocusAreaChanged(Rect rect) {
        onVideoCaptureFocusChanged(mCallBackPtr, rect.left, rect.top, rect.right - rect.left, rect.top - rect.bottom);
    }

    @Override
    public void onCameraExposureAreaChanged(Rect rect) {
        onVideoCaptureExposureChanged(mCallBackPtr, rect.left, rect.top, rect.right - rect.left, rect.top - rect.bottom);
    }



    /**
     * 采集回调
     * @param textureId
     * @param width
     * @param height
     * @return
     */
    @Override
    public int onTextureCallback(int textureId, int width, int height) {
        if (mCapture == null || mCaptureType != ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_DEFAULT_CAMERA) {
            return textureId;
        }

        ThunderDefaultCamera camera = ThunderEngine.getDefaluteCamera();
        if (camera == null) {
            return textureId;
        }
        if (camera.getCameraDataCallback() == null) {
            return textureId;
        }
        return camera.getCameraDataCallback().onTextureCallback(textureId, width, height);
    }

    public ThunderVideoPublishEngineImp() {
        mUiHandler = new android.os.Handler(Looper.getMainLooper());
        //比拉取配置要早,异步检查硬编兼容性
        final CountDownLatch waitSync = new CountDownLatch(1);
        Runnable runTestVideoEncoderCrash = new Runnable() {
            @Override
            public void run() {
                try {
                    YYVideoCodec.testSupportH264Encode();
                    YYVideoCodec.getSupportH265();
                } catch (Exception e) {
                    ThunderLog.error("ThunderVideoPublishEngineImp", "testVideoEncoderCrash crash " + e.toString());
                }
                waitSync.countDown();
            }
        };
        new Thread(runTestVideoEncoderCrash).start();

        try {
            waitSync.await(500, TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            ThunderLog.error("ThunderVideoPublishEngineImp", "testVideoEncoderCrash waitSync crash " + e.toString());
        }
    }

    @Override
    public void onBehaviorEvent(String name, String val, String oval, int level) {
        // 视频用户行为日志上报
        ThunderNative.makeBehaviorEvent(name, val, oval, level);
    }

    @Override
    public void onVideoLiveAbnormalStateNotification(YMFLiveExceptionType type) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "onVideoLiveAbnormalStateNotification:" + type);
}
        if (mCallBackPtr != 0) {
            onVideoCaptureEncodeEvent(mCallBackPtr, type.getValue());
            if (type == YMFLiveExceptionType.AnchorStatus_CAPTURE_USED_BY_HIGHER_PRIORITY) {
                mCameraState = ThunderRtcConstant.ThunderVideoCaptureStatus.THUNDER_VIDEO_CAPTURE_STATUS_RESTRICTED;
                onVideoCaptureStatus(mCallBackPtr, mCameraState);
            }
        }
    }

    /**
     * 初始化
     */
    public void pubInit() {
        mPublisher = YMFLiveAPI.getInstance();
        mVideoPubHiidoUtil = new ThunderVideoHiidoUtil(YYVideoLibMgr.instance().getAppContext());
//        mVideoPubHiidoUtil.register();
        mPublisher.setPublishListener(this);
        mPublisher.setCameraListener(this);
        YMFLiveUsrBehaviorStat.getInstance().setYMFBehaviorEventListener(this);
        YMFLiveExceptionStat.getInstance().setYMFExceptionListener(this);
    }

    /**
     * 启动videoEngine
     * @return
     */
    public boolean startVideoEngine() {
//        if (mVideoPubHiidoUtil != null) {
//            mVideoPubHiidoUtil.register();
//        }
        return true;
    }

    /**
     * 停止videoEngine
     * @return
     */
    public boolean stopVideoEngine() {
//        if (mVideoPubHiidoUtil != null) {
//            mVideoPubHiidoUtil.unRegister();
//        }
        return true;
    }



    public VideoPublishLowStreamConfig getLowVideoStreamConfig(int width, int height){
        if (mPublisher == null) {
            return null;
        }
        if (width <= 0 || height <= 0) {
            return null;
        }
        YMFLowStreamEncoderConfig config = mPublisher.getLowVideoStreamConfig(width, height);
        VideoPublishLowStreamConfig lowStreamConfig = new VideoPublishLowStreamConfig();
        lowStreamConfig.type = config.mConfigId;
        lowStreamConfig.codecid = config.mEncoderId;
        lowStreamConfig.encodeCodeRate = config.mCodeRate;
        lowStreamConfig.encodeFrameRate = config.mFrameRate;
        lowStreamConfig.minCodeRate = config.mMinCodeRate;
        lowStreamConfig.maxCodeRate = config.mMaxCodeRate;
        lowStreamConfig.encoderParam = config.mEncodeParam;
        lowStreamConfig.resolutionHeight = config.mEncodeHeight;
        lowStreamConfig.resolutionWidth = config.mEncodeWidth;
        lowStreamConfig.transcoding = config.mTranscoding;
        return lowStreamConfig;
    }


    /**
     * 初始化capture
     * @param capture
     */
    public int attachVideoCapture(Object capture, int captureType) {
        if (mPublisher == null) {
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }

        // 相同captureType时，1、当type是摄像头不进行切换；2、当type是外部源和录屏时且capture对象相同时不进行切换
        if (captureType == mCaptureType) {
            if (mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_DEFAULT_CAMERA ||
                (mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_SCREEN_RECORD && mCapture.equals(capture)) ||
                (mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_EXTERNAL_SOURCE && mCapture.equals(capture))) {

                ThunderLog.error(TAG, "attachVideoCapture same captureType curType %d, newType %d",
                        mCaptureType, captureType);
                return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
            }
        }

        // 如果当前再编码状态，此时调用attachVideoCapture 切换编码源，需要先停掉之前的编码
        if (bStartEncode) {
            if (mCapture == null || mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_DEFAULT_CAMERA) {
                // 停摄像头编码
                mPublisher.stopEncodeVideo();
            } else if (mCapture != null && mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_SCREEN_RECORD) {
                // 停录屏编码
                mPublisher.stopEncodeScreen();
                // 停录屏采集
                mPublisher.stopScreenCapture();
            } else if (mCapture != null && mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_EXTERNAL_SOURCE) {
                // 停外部源编码

                mPublisher.stopEncodeOrigin();
                mPublisher.stopOriginCapture();
                mCapture.stopCapture();
            }
        }


        mCaptureType = captureType;
        mCapture = (ThunderVideoCapture) capture;

if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "attach capture: %s, type %d", mCapture.toString(), mCaptureType);
}

        /*** 如果已经处于编码状态，更换编码源后，需要重新启动视频库编码 ***/
        if (bStartEncode) {
            /*** 是否启动海外弱网逻辑***/
            mPublisher.setAbroadNetWorkStrategy(VideoConfigManager.instance().getAbroadNetWorkStrategy());
            mPublisher.notifyChangeVideoSourceState();
            // external source()
            if (mCapture != null &&
                    mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_EXTERNAL_SOURCE) {
                if(((ExternalVideoSource)mCapture).getExternalVideoBuffType() == THUNDER_EXTERNAL_VIDEO_TYPE_TEXTURE){
                    mPublisher.startOriginCapture(true);
                }else {
                    mPublisher.startOriginCapture(false);
                }

                mCapture.startCapture(new ThunderPublisher.IVideoPublisher() {
                    @Override
                    public void pushVideoData(byte[] data, int format, int width, int height, int rotation, long timestamp) {
                        setOriginFrameToEncode(data, format, width, height, rotation, timestamp);
                    }

                    @Override
                    public void pushVideoData(byte[] encodedData, ThunderConstant.ThunderVideoEncodeType type, long dts, long pts) {

                    }

                    @Override
                    public void pushVideoTexture(int textureID, int textureFormat, int width, int height,
                           int rotation, long timeStamp,float[] transform) {
                        setOriginTextureToEncode(textureID, textureFormat, width, height, rotation, timeStamp, transform);
                    }

                    @Override
                    public void pushVideoFrame(ThunderExternalVideoFrame frame) {
                        setExternalFrameToEncode(frame);
                    }
                });

                // 开启外部源编码
                mPublisher.startEncodeOrigin();
            }
            // screen record
            else if (mCapture != null &&
                    mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_SCREEN_RECORD) {
                // 开始录屏采集
                mPublisher.startScreenCapture(((ThunderScreenCapture) mCapture).getMediaProjection());
                // 开始录屏编码
                mPublisher.startEncodeScreen();
            }
            // default camera
            else {
                // 开始摄像头编码
                mPublisher.startEncodeVideo();
            }

        }
        return 0;
    }

    /**
     * camera 开始采集
     * @return
     */
    public int startVideoCapture() {
        mCameraState = mPublisher.startVideoCapture();
        bStartCameraCapture = true;
        if (mCallBackPtr != 0) {
            onVideoCaptureStatus(mCallBackPtr, mCameraState);
        }
        return mCameraState;
    }

    /**
     * camera 停止采集
     * @return
     */
    public int stopVideoCapture() {
        mPublisher.stopVideoCapture();
        bStartCameraCapture = false;
        if (mCallBackPtr != 0 && (mCameraState == ThunderRtcConstant.ThunderVideoCaptureStatus.THUNDER_VIDEO_CAPTURE_STATUS_SUCCESS)) {
            onVideoCaptureStatus(mCallBackPtr,
                    ThunderRtcConstant.ThunderVideoCaptureStatus.THUNDER_VIDEO_CAPTURE_STATUS_CLOSE);
        }
        return 0;
    }

    /**
     * 暂停/恢复 视频采集
     * @param bPauseCapture true：暂停 false：恢复
     * @return
     */
    public int pauseVideoCapture(boolean bPauseCapture) {
        mCameraState = mPublisher.pauseVideoCapture(bPauseCapture);
        // 这里先注释调 暂停/恢复的状态通知，因为视频库暂时没有区分这个和start/stopCapture的回调状态 TDBTAND-475
//        if (mCallBackPtr != 0) {
//            onVideoCaptureStatus(mCallBackPtr, mCameraState);
//        }
        return 0;
    }

    /**
     * camera 开启预览
     * @return
     */
    public int startPreview(final Object toView, int renderMode) {
        if (toView != null && (toView instanceof ThunderPreviewView)) {
            //因为加入YUV模式之后开播的View有3种，这里需要check下（因为拉取配置的原因，可能导致初始化view类型不对）
            if(((ThunderPreviewView)toView).checkViewType(mPublisher.getPreViewType())){
                ThunderLog.warn(TAG, "startPreview: enableVideoPublishBufferProcess error!");
                final CountDownLatch waitSync = new CountDownLatch(1);
                if(Thread.currentThread().getId() == mUiHandler.getLooper().getThread().getId()){
                    ((ThunderPreviewView)toView).changeViewType();
                }else {
                    mUiHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            ((ThunderPreviewView) toView).changeViewType();
                            waitSync.countDown();
                        }
                    });
                    try {
                        waitSync.await();
                    } catch (Exception e) {
                        ThunderLog.error("ThunderVideoPublishEngineImp",
                                "changeViewType waitSync crash " + e.toString());
                    }
                }
            }
            mPublisher.startPreview(((ThunderPreviewView)toView).getSurfaceView(), renderMode);
        } else {
            mPublisher.startPreview(null, renderMode);
if (ThunderLog.isInfoValid()) {
            ThunderLog.info(TAG, "startPreview: invalid view " + toView);
}
        }

        return 0;
    }

    /**
     * camera 停止预览
     * @return
     */
    public int stopPreview() {
        mPublisher.stopPreview();
        return 0;
    }

    /**
     * @brief 设置本地固定的推流画面,只支持摄像头采集和录屏采集场景，外部推流场景不支持
     * @param bitmap 固定推流画面位图
     */
    public int setCaptureReplaceImage(Object bitmap)
    {
        return mPublisher.setCaptureReplaceImage((Bitmap)bitmap);
    }

    /**
     * @brief 获取本地视频截图
     * @return Bitmap: 本地视频截图的位图，在非正常渲染本地视频流时返回null
     * @remark 需要在"开启视频预览"后调用
     */
    public Object captureLocalScreenShot(){
        return mPublisher.captureLocalScreenShot();
    }

    /**
     * 开始编码
     * @return
     */
    public int startEncodeVideo() {
        bUserCallEncode = true;
        /*** 是否启动海外弱网逻辑***/
        mPublisher.setAbroadNetWorkStrategy(VideoConfigManager.instance().getAbroadNetWorkStrategy());
        /*** 先开启采集 再启动编码***/
        if (mCapture != null &&
                mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_EXTERNAL_SOURCE) {
            // 外部源，开采集
            if(((ExternalVideoSource)mCapture).getExternalVideoBuffType() == THUNDER_EXTERNAL_VIDEO_TYPE_TEXTURE){
                mPublisher.startOriginCapture(true);
            }else {
                mPublisher.startOriginCapture(false);
            }
            mCapture.startCapture(new ThunderPublisher.IVideoPublisher() {
                @Override
                public void pushVideoData(byte[] data, int format, int width, int height, int rotation, long timestamp) {
                    setOriginFrameToEncode(data, format, width, height, rotation, timestamp);
                }

                @Override
                public void pushVideoData(byte[] encodedData, ThunderConstant.ThunderVideoEncodeType type, long dts, long pts) {

                }

                @Override
                public void pushVideoTexture(int textureID, int textureFormat, int width, int height,
                                             int rotation, long timeStamp,float[] transform) {
                    setOriginTextureToEncode(textureID, textureFormat, width, height, rotation, timeStamp, transform);
                }

                @Override
                public void pushVideoFrame(ThunderExternalVideoFrame frame) {
                    setExternalFrameToEncode(frame);
                }
            });
            // 外部源开始编码
            mPublisher.startEncodeOrigin();
        } else if (mCapture != null &&
                mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_SCREEN_RECORD) {
            // 录屏开采集
            mPublisher.startScreenCapture(((ThunderScreenCapture) mCapture).getMediaProjection());
            // 录屏开始编码
            mPublisher.startEncodeScreen();
        } else if (mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_DEFAULT_CAMERA) {
            // 默认摄像头开始编码
            mPublisher.startEncodeVideo();
        }
        bStartEncode = true;

        return 0;
    }

    /**
     * 停止编码
     * @return
     */
    public int stopEncodeVideo() {
        if (mCapture != null &&
                mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_EXTERNAL_SOURCE) {
            // 外部源停编码
            mPublisher.stopEncodeOrigin();
            // 外部源停采集
            mPublisher.stopOriginCapture();
            // 通知用户当前外部源采集已停
            mCapture.stopCapture();
        } else if (mCapture != null &&
                mCaptureType == ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_SCREEN_RECORD) {
            // 录屏停编码
            mPublisher.stopEncodeScreen();
            // 录屏停采集
            mPublisher.stopScreenCapture();
        } else {
            // 默认摄像头停编码
            mPublisher.stopEncodeVideo();
        }
        bStartEncode = false;
        return 0;
    }

    /**
     * 设置本地预览视图模式
     * @param scaleMode 视图模式
     */
    public void setPreviewRenderMode(int scaleMode) {
        mPublisher.setPreviewRenderMode(scaleMode);
    }


    /**
     * 切换录屏的状态到picture模式
     * @param bPicMode true: 图片模式， false：正常流模式
     * @param bitmap：图片模式下的图片
     */
    public void changeScreenLiveMode(boolean bPicMode, Object bitmap) {
        if (mPublisher != null) {
            mPublisher.changeScreenLiveMode(bPicMode, (Bitmap) bitmap);
        }
    }

    /**
     * 更新开播参数
     * @return
     */
    public int updateVideoPublishConfig(VideoPublishParams params) {
        VideoPublisheParam param = new VideoPublisheParam();
        param.bLowLatency = params.bLowLatency;
        param.bWebSdkCompatibility = params.bWebSdkCompatibility;
        param.captureFrameRate = params.captureFrameRate;
        param.captureResolutionWidth = params.captureResolutionWidth;
        param.captureResolutionHeight = params.captureResolutionHeight;
        param.screenOrientation = params.screenOrientation;
        param.codecid = params.codecid;
        param.encodeFrameRate = params.encodeFrameRate;
        param.encodeBitrate = params.encodeBitrate * 1000;
        param.encodeMaxBitrate = params.encodeMaxBitrate * 1000;
        param.encodeMinBitrate = params.encodeMinBitrate * 1000;
        param.encodeResolutionWidth = params.encodeResolutionWidth;
        param.encodeResolutionHeight = params.encodeResolutionHeight;
        param.weakNetConfigsIntervalSecs = params.weakNetConfigsIntervalSecs;
        param.encodeType = switchVideoEncodeType(params.codecid);
        param.bEnableLocalDualStreamMode = params.bEnableLocalDualStreamMode;
        param.mDegradationStrategy = params.degradationStrategy;
        param.mCameraCaptureStrategy = params.cameraCaptureStrategy;

        param.encoderParam = params.encoderParam;
        Iterator itor = params.weakNetCfgs.iterator();
        while (itor.hasNext()) {
            VideoPublishWeakNetParam weakNetParam =
                    (VideoPublishWeakNetParam) itor.next();
            ResolutionModifyConfig config =
                    new ResolutionModifyConfig(
                            weakNetParam.resolutionWidth,
                            weakNetParam.resolutionHeight,
                            weakNetParam.minCodeRate * 1000,
                            weakNetParam.maxCodeRate * 1000,
                            weakNetParam.minFrameRate,
                            weakNetParam.maxFrameRate,
                            switchVideoEncodeType(weakNetParam.codecid),
                            weakNetParam.encoderParam);
            param.weakNetConfigs.add(config);
        }

        Iterator lowStreamItor = params.lowStreamCfgs.iterator();
        while (lowStreamItor.hasNext()) {
            VideoPublishLowStreamConfig LowStreamParam =
                    (VideoPublishLowStreamConfig) lowStreamItor.next();
            YMFLowStreamEncoderConfig config =
                    new YMFLowStreamEncoderConfig();
            config.mConfigId = LowStreamParam.type;
            config.mEncoderId = LowStreamParam.codecid;
            config.mEncodeWidth = LowStreamParam.resolutionWidth;
            config.mEncodeHeight =  LowStreamParam.resolutionHeight;
            config.mFrameRate =  LowStreamParam.encodeFrameRate;
            config.mCodeRate = LowStreamParam.encodeCodeRate * 1000;
            config.mMaxCodeRate = LowStreamParam.maxCodeRate * 1000;
            config.mMinCodeRate = LowStreamParam.minCodeRate * 1000;
            config.mTranscoding = LowStreamParam.transcoding;
            config.mEncodeParam = LowStreamParam.encoderParam;
            param.lowStreamConfigs.add(config);
        }

        mPublisher.updatePublisherConfig(param);
        return 0;
    }

    /**
     * 设置前后置摄像头
     * @param position
     * @return
     */
    public int setCameraPosition(int position) {
        ThunderDefaultCamera camera = ThunderEngine.getDefaluteCamera();
        int ret = ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        if (camera != null) {
            ((ThunderPreviewConfig)camera.getCaptureConfig()).cameraPosition = position;
            ret =  mPublisher.setCameraPosition(position);
        } else {
            ThunderLog.warn(TAG, "setCameraPosition get null camera");
        }
        return ret;
    }

    public boolean setCameraTorchOn(boolean isOn) {
        boolean ret = false;
        if (mPublisher != null) {
            ret = mPublisher.setCameraTorchOn(isOn);
        }
        return ret;
    }

    /**
     * @brief 摄像头是否打开
     * @return true: 摄像头打开；false: 摄像头没打开
     */
    public boolean isCameraOpen() {
        boolean ret = false;
        if (mPublisher != null) {
            ret = mPublisher.isCameraOpen();
        }
        return ret;
    }

    /**
     * @brief 摄像头是否支持缩放
     * @return true: 摄像头支持缩放；false: 摄像头不支持缩放
     * @remark (1) 默认判断前置摄像头，如需判断后置摄像头请先调用setCameraPosition进行切换
     *         (2) 切换了摄像头(调用了setCameraPosition)，需要重新调用
     */
    public boolean isCameraZoomSupported() {
        boolean ret = false;
        if (mPublisher != null) {
            ret = mPublisher.isCameraZoomSupported();
        }
        return ret;
    }

    /**
     * @brief 得到摄像头的最大缩放值
     * @return 调用成功返回最大的摄像头缩放值
     * @remark (1) 如果调用失败，返回-1
     *         (2) 默认前置摄像头，如需查询后置摄像头请先调用setCameraPosition进行切换
     *         (3) 切换了摄像头(调用了setCameraPosition)，需要重新调用
     */
    public float getCameraMaxZoomFactor() {
        if (mPublisher == null) {
            return (float) ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }
        return mPublisher.getCameraMaxZoomFactor();
    }

    /**
     * @brief 设置摄像头缩放比例
     * @param zoomFactor 缩放比例
     * @return 调用成功返回设置的缩放比例，如果设置失败返回值<0
     * @remark (1) 如果设置的zoomFactor超过了最大的缩放值，则返回最大的缩放值
     *         (2) 如果设置的zoomFactor超过了最小的缩放值，则返回最小的缩放值
     */
    public int setCameraZoomFactor(float zoomFactor) {
        if (mPublisher == null) {
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }
        return mPublisher.setCameraZoomFactor(zoomFactor);
    }

    /**
     * @brief 摄像头是否支持手电筒功能
     * @return true: 摄像头支持手电筒功能；false: 摄像头不支持手电筒功能
     * @remark (1) 默认判断前置摄像头，如需判断后置摄像头请先调用setCameraPosition进行切换
     *         (2) 切换了摄像头(调用了setCameraPosition)，需要重新判断
     */
    public boolean isCameraTorchSupported() {
        if (mPublisher == null) {
            return false;
        }
        return mPublisher.isCameraTorchSupported();
    };

    /**
     * @brief 摄像头是否支持手动对焦
     * @return true: 摄像头支持手动对焦功能；false: 摄像头不支持手动对焦功能
     * @remark 默认判断前置摄像头是否支持该功能，如需判断后置摄像头请先调用setCameraPosition进行切换
     */
    public boolean isCameraManualFocusPositionSupported() {
        if (mPublisher == null) {
            return false;
        }
        return mPublisher.isCameraManualFocusPositionSupported();
    }

    /**
     * @brief 设置摄像头对焦位置
     * @param posX, posY 对焦位置, MotionEvent中获取的x,y坐标。
     * @return 0:成功, -1:失败
     */
    public int setCameraFocusPosition(float posX, float posY) {
        if (mPublisher == null) {
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }
        return mPublisher.setCameraFocusPosition(posX, posY);
    }

    /**
     * @brief 摄像头是否支持手动曝光
     * @return true: 摄像头支持手动曝光功能；false: 摄像头不支持手动曝光功能
     * @remark 默认判断前置摄像头是否支持该功能，如需判断后置摄像头请先调用setCameraPosition进行切换
     */
    public boolean isCameraManualExposurePositionSupported() {
        if (mPublisher == null) {
            return false;
        }
        return mPublisher.isCameraManualExposurePositionSupported();
    }

    /**
     * @brief 设置摄像头曝光位置
     * @param posX, posY 对焦位置, MotionEvent中获取的x,y坐标。
     * @return 0:成功, -1:失败
     */
    public int setCameraExposurePosition(float posX, float posY) {
        if (mPublisher == null) {
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }
        return mPublisher.setCameraExposurePosition(posX, posY);
    }

    /**
     * @brief 摄像头是否支持人脸对焦功能
     * @return true: 摄像头支持人脸对焦功能；false: 摄像头不支持人脸对焦功能
     * @remark 默认判断前置摄像头是否支持该功能，如需判断后置摄像头请先调用setCameraPosition进行切换
     */
    public boolean isCameraAutoFocusFaceModeSupported() {
        if (mPublisher == null) {
            return false;
        }
        return mPublisher.isCameraAutoFocusFaceModeSupported();
    }

    /**
     * @brief 开启/关闭人脸对焦功能
     * @return 0:成功
     * @remark (1) 默认设置前置摄像头，如需设置后置摄像头请先调用setCameraPosition进行切换
     *         (2) 切换了摄像头(调用了setCameraPosition)，需要重新设置
     */
    public int setCameraAutoFocusFaceModeEnabled(boolean enable) {
        if (mPublisher == null) {
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }
        return mPublisher.setCameraAutoFocusFaceModeEnabled(enable);
    }

    /**
     * @brief 是否前置摄像头
     * @return true: 是前置摄像头；false: 是后置摄像头
     */
    public boolean isFrontCamera() {
        if (mPublisher == null) {
            return false;
        }
        return mPublisher.isFrontCamera();
    }

    public int getVideoCaptureOrientation() {
        if (mPublisher == null) {
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }
        return mPublisher.getCameraOrientation();
    }

    /**
     * 设置水印
     * @param x
     * @param y
     * @param width
     * @param height
     * @param imageUrl
     * @return
     */
    public int setPubWatermark(int x, int y, int width, int height, String imageUrl) {
        int ret = 0;
        Bitmap bitmap = null;
        Bitmap tmpBitmap = null;
        try {
            if (imageUrl != null && !imageUrl.isEmpty()) {
                FileInputStream fis = new FileInputStream(imageUrl);
                tmpBitmap = BitmapFactory.decodeStream(fis);
                fis.close();
                int w = tmpBitmap.getWidth();
                int h = tmpBitmap.getHeight();
                Matrix matrix = new Matrix();
                float scaleW = ((float) width) / w;
                float scaleH = ((float) height) / h;
                matrix.postScale(scaleW, scaleH);
                bitmap = Bitmap.createBitmap(tmpBitmap, 0, 0, w, h, matrix, true);
            }
            ret = mPublisher.setWatermark(bitmap, x, y);
        }
        catch (Exception e) {
            e.printStackTrace();
            if (bitmap != null && !bitmap.isRecycled()) {
                bitmap.recycle();
            }
            if (tmpBitmap != null && !tmpBitmap.isRecycled()) {
                tmpBitmap.recycle();
            }
        }

        return ret;
    }

    /**
     * 动态码率通知
     * @param bitrate
     * @return
     */
    public int onDynamicBitrate(long bitrate) {
        // 传输回调回来的比特率数值单位是 kbps
        mPublisher.setNetworkBitrateSuggest((int) bitrate * 1000);
        return 0;
    }

    /**
     * 后台配置下发的主播端采集时间戳校正值
     * @param adjustVal
     * @return
     */
    public int setPublisherPtsAdjustVal(int adjustVal){
        if (mPublisher != null) {
            mPublisher.setDeltaYYPtsMillions(adjustVal);
        }
        return 0;
    }

    /**ThunderVideoPublishEngineImp
     * 设置镜像模式
     * @param mode
     * @return
     */
    public int setLocalVideoMirrorMode(int mode) {
        if (mPublisher == null) {
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }
        return mPublisher.setLocalVideoMirrorMode(mode);
    }

    /**
     * 获取camera缩放
     * @return
     */
    public float getCameraZoomFactor() {
        return mCurZoomFactor;
    }

    /**
     * 缺I帧回调
     * @return
     */
    public int onRequestIFrame() {
        mPublisher.requestEncodeIFrame();
        return 0;
    }

    /**
     * 设置 ILiveEngineVideoEngine 的 callback
     * @param ptr
     */
    public void setVideoPublishEngineCallBack(long ptr) {
        mCallBackPtr = ptr;
    }

    /**
     * 获取开播库的版本号
     * @return
     */
    public static String getVideoPubLibVersion() {
        return com.thunder.livesdk.BuildConfig.YY_VIDEOLIB_VERSION;
    }

    /**
     * 获取去hiido统计数据
     * @param streamId
     * @return
     */
    public String getAnchorHiidoStatInfo(long streamId) {
        String anchorStat = "";
        if (mVideoPubHiidoUtil != null) {
            anchorStat = mVideoPubHiidoUtil.getAnchorStatInfo();
        }
        return VideoDataStat.getInstance().getAnchorVideoData(streamId) + anchorStat;
    }

    /**
     * 获取视频实时统计数据
     */
    public int getPublishRuntimeInfo(int type, int typeId) {
        switch (type) {
            case VIDEO_STAT_FPS:
                return mPublisher.getVideoPublishInfo(YMFLiveAPI.VideoPublishInfoEnum.FRAME, typeId);
            case VIDEO_STAT_BITRATE:
                return mPublisher.getVideoPublishInfo(YMFLiveAPI.VideoPublishInfoEnum.BITRATE, typeId);
            case VIDEO_STATE_RESOLUTION:
                return mPublisher.getVideoPublishInfo(YMFLiveAPI.VideoPublishInfoEnum.RESOLUTION, typeId);
            case VIDEO_PREVIEW_FRAME_RATE:
                return mPublisher.getVideoPublishInfo(YMFLiveAPI.VideoPublishInfoEnum.PREVIEW_FRAME_RATE, typeId);
            case VIDEO_DYNAMIC_ENC_FRAME_RATE:
                return mPublisher.getVideoPublishInfo(YMFLiveAPI.VideoPublishInfoEnum.DYNAMIC_ENC_FRAME_RATE, typeId);
            case VIDEO_ENCODED_TYPE:
                int encodedType = mPublisher.getVideoPublishInfo(YMFLiveAPI.VideoPublishInfoEnum.ENCODETYPE, typeId);
                if (VideoEncoderType.HARD_ENCODER_H264.ordinal() == encodedType ||
                        VideoEncoderType.HARD_ENCODER_H265.ordinal() == encodedType) {
                    return 1;
                } else if (VideoEncoderType.SOFT_ENCODER_X264.ordinal() == encodedType ||
                        VideoEncoderType.SOFT_ENCODER_H265.ordinal() == encodedType) {
                    return 2;
                } else {
                    return 0; // unknown
                }
            default:
                break;
        }
        return 0;
    }

    public void notifyEncodeBlackList(String blacklist) {
        if (blacklist.isEmpty()) {
            return;
        }
        String[] blockEncode = blacklist.split(",");
        for (int i = 0; i < blockEncode.length; ++i) {
            if (blockEncode[i].equalsIgnoreCase(YYVideoCodec.getH264EncodeName()) ||
                blockEncode[i].equalsIgnoreCase(YYVideoCodec.getH265EncodeName())) {
                if (mPublisher != null) {
if (ThunderLog.isInfoValid()) {
                    ThunderLog.info(TAG, "notifyEncodeBlackList to videoLib, content: %s", blockEncode[i]);
}
                    mPublisher.setHardwareEncoderAvailable(false);
                }
            }
        }
    }

    public void checkPublishVideoConfigAvailable(VideoPublishConfig config) {
        if (config.codecid == VideoLiveConfig.EncodeType.PHONE_CODEC_HW_H265) {
            if (YMFLiveAPI.getSupportH265EncodeProperty() != YYVideoCodec.EncodeSupport.SUPPORTED) {
                config.codecid = ThunderRtcConstant.ThunderVideoEncodeType.THUNDERVIDEO_ENCODE_TYPE_H264;
                ThunderLog.error(TAG, "checkPublishVideoConfigAvailable THUNDERVIDEO_ENCODE_TYPE_H264");
            }
        }
    }

    public int enableVideoPublishBufferProcess(boolean bBufferMode) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG,"enableVideoPublishBufferProcess:" + bBufferMode);
}
        return mPublisher.setYuvCanvasMode(bBufferMode);
    }

    public int setVideoCommonConfigMode(int mode){
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG,"setVideoCommonConfigMode:" + mode);
}
        return mPublisher.setVideoCommonConfigMode(mode);
    }

    public void updateArgoConfig(HashMap<String, String> argoConfigMap) {
        if (argoConfigMap.isEmpty()) {
            return;
        }
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG,"updateArgoConfig:" + argoConfigMap);
}
        mPublisher.setVideoUpCommonConfig(argoConfigMap);
    }

    public void updateVideoEncoderConfig(HashMap<String, String> videoConfigMap) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG,"updateVideoEncoderConfig " + videoConfigMap);
}
        mPublisher.setCustomVideoConfig(videoConfigMap);
        YYVideoLibMgr.instance().updateArgoConfig(videoConfigMap);
    }

    public void destroyPublishEngine() {
        YMFLiveUsrBehaviorStat.getInstance().setYMFBehaviorEventListener(null);
        if (mPublisher != null) {
            mPublisher.destory();
        }
        reset();
    }

    private void reset() {
        mCaptureType = ThunderRtcConstant.LiveEngineCaptureType.THUNDER_CAPTURE_TYPE_DEFAULT_CAMERA;
        mCameraState = ThunderRtcConstant.ThunderVideoCaptureStatus.THUNDER_VIDEO_CAPTURE_STATUS_CLOSE;
        bStartEncode = false;
    }

    private void setExternalFrameToEncode(ThunderExternalVideoFrame frame) {
        if (mPublisher != null && frame != null) {
            YMFExternalFrame ymfExternalFrame = new YMFExternalFrame();
            ymfExternalFrame.data = frame.data;
            ymfExternalFrame.format = frame.format;
            ymfExternalFrame.width = frame.width;
            ymfExternalFrame.height = frame.height;
            ymfExternalFrame.textureID = frame.textureID;
            ymfExternalFrame.textureFormat = frame.textureFormat;
            ymfExternalFrame.rotation = frame.rotation;
            ymfExternalFrame.transform = frame.transform;
            ymfExternalFrame.scaleMode = frame.scaleMode;
            ymfExternalFrame.timeStamp = frame.timeStamp;
            mPublisher.setOriginExternalFrameToEncode(ymfExternalFrame);
        }
    }


    private void setOriginFrameToEncode(final byte[] data, final int format, final int width, final int height,
                                        final int rotation, final long timestamp) {
        if (mPublisher != null) {
            mPublisher.setOriginFrameToEncode(data, format, width, height, rotation, timestamp,
                    ThunderRtcConstant.ThunderExternalVideoRenderMode.THUNDER_RENDER_MODE_CLIP_TO_BOUNDS);
        }
    }

    private void setOriginTextureToEncode(int textureID, int textureFormat, int width, int height,
                                          int rotation, long timeStamp,float[] transform){
        if (mPublisher != null) {
            mPublisher.setOriginTextureToEncode(textureID, textureFormat, width, height,
                    rotation, timeStamp, transform,
                    ThunderRtcConstant.ThunderExternalVideoRenderMode.THUNDER_RENDER_MODE_CLIP_TO_BOUNDS);
        }
    }

    private VideoEncoderType switchVideoEncodeType(int codecid) {
        VideoEncoderType ret = VideoEncoderType.DEFAULT;
        switch (codecid)
        {
            case 200:
                ret = VideoEncoderType.HARD_ENCODER_H264; // 264硬编
                break;
            case 201:
                ret = VideoEncoderType.SOFT_ENCODER_X264; // 264软编
                break;
            case 220:
                ret = VideoEncoderType.HARD_ENCODER_H265; // 265硬编
                break;
            case 221:
                ret = VideoEncoderType.SOFT_ENCODER_H265; // 265软编
                break;
            default:
                ret = VideoEncoderType.DEFAULT;
                break;
        }
        return ret;
    }

    public class VideoPublishParams {

        public boolean bLowLatency; // 是否低延时
        public boolean bWebSdkCompatibility; // 是否支持WebSdk
        public boolean bEnableLocalDualStreamMode; // 是否双流开播
        public int captureFrameRate; // 采集帧率
        public int captureResolutionWidth; // 采集分辨宽
        public int captureResolutionHeight; // 采集分辨高
        public int screenOrientation; // 横竖屏
        public int codecid; // 编码器ID
        public int encodeFrameRate; // 编码帧率
        public int encodeBitrate; // 编码码率 kbps
        public int encodeMaxBitrate; // 编码最大码率 kbps
        public int encodeMinBitrate; // 编码最小码率 kbps
        public int encodeResolutionWidth; // 编码分辨高
        public int encodeResolutionHeight; // 编码分辨高
        public int weakNetConfigsIntervalSecs; // 弱网分辨率切换最小间隔
        public int degradationStrategy; // 弱网降级策略
        public int cameraCaptureStrategy; // 摄像头采集策略
        public String encoderParam; // 编码器配置
        public List<VideoPublishWeakNetParam> weakNetCfgs = new ArrayList<>(); //弱网参数
        public List<VideoPublishLowStreamConfig> lowStreamCfgs = new ArrayList<>(); // 小流配置

    }

    // 视频开播弱网配置参数
    public class VideoPublishWeakNetParam {
        public int resolutionWidth; // (弱网)分辨宽
        public int resolutionHeight; // (弱网)分辨高
        public int minCodeRate;  // (弱网)最小码率
        public int maxCodeRate; // (弱网)最大码率
        public int minFrameRate; // (弱网)最小帧率
        public int maxFrameRate; // (弱网)最大帧率
        public int codecid; // (弱网)编码类型
        public String encoderParam; // (弱网)编码器配置
    }

    // 双流开播小流配置参数
    public class VideoPublishLowStreamConfig {
        public int type;// 小流id, 区分不同的小流配置
        public int resolutionWidth; // 编码分辨宽
        public int resolutionHeight; // 编码分辨高
        public int encodeCodeRate;// 编码码率
        public int maxCodeRate; // 编码最大码率
        public int minCodeRate;  // 编码最小码率
        public int encodeFrameRate; // 编码帧率
        public int codecid; // 编码器ID
        public int transcoding;// 是否需要转码
        public String encoderParam; // 编码器配置
    }

    public String getVideoEncodeBaseStatics(int sendSeq) {
        return YMFLiveStatisticManager.getInstance().getBaseUploadVideoStatistics(sendSeq);
    }

    public String getVideoEncodeStatics(boolean bKeyStat, int publishId) {
        String anchorStat = "";
        if (mVideoPubHiidoUtil != null) {
            anchorStat = mVideoPubHiidoUtil.getAnchorStatInfo();
        }
        return YMFLiveStatisticManager.getInstance().getUploadVideoStatistics(bKeyStat, publishId) + anchorStat;
    }

    // native
    private native void onVideoEncodedFrame(long callback,
                                            byte[] data,
                                            int len,
                                            long pts,
                                            long dts,
                                            int frameType,
                                            int encodeType,
                                            boolean bHardware,
                                            int svcTid,
                                            int svcSid,
                                            int typeId,
                                            byte[] extraData,
                                            int extraDataLen);

    private native void onVideoCaptureStatus(long callback, int state);

    private native void onUpdateVideoSizeChanged(long callback, long streamId, int w, int h);

    private native void onVideoAnchorStatus(long callback, int status);

    private native void onVideoCaptureEncodeEvent(long callback, int event);

    private native void onVideoCaptureFocusChanged(long callback, float posX, float posY, float w, float h);

    private native void onVideoCaptureExposureChanged(long callback, float posX, float posY, float w, float h);

    private native void onVideoFrameProcessTime(long callback, float avgTime, float maxTime);
}
