package tv.athena.live.component.business.wath

import android.os.Handler
import android.os.Looper
import com.yy.liveplatform.proto.nano.LpfLiveinterconnect
import com.yy.liveplatform.proto.nano.LpfLiveroomtemplateV2
import com.yy.liveplatform.proto.nano.LpfMedia
import tv.athena.live.api.wath.LiveStatus
import tv.athena.live.api.wath.MediaProtocol
import tv.athena.live.api.wath.MediaType
import tv.athena.live.api.wath.MicInfoListener
import tv.athena.live.api.wath.MicInfoV2
import tv.athena.live.api.wath.MicStatusV2
import tv.athena.live.api.wath.bean.LineStreamInfo

import tv.athena.live.utils.ALog

/**
 *     create by chenhaofeng 2020/1/2
 *     麦位处理类
 *
 **/
class MicPositionHandle(val watchComponent: WatchComponent) {
    companion object {
        const val TAG = "MicPositionHandle"
    }

    private val mCacheStreamInfos = HashMap<Int, LpfLiveroomtemplateV2.LiveInterconnectStreamInfo>()
    private var mMicInfoListener: ListenerBuilder? = null
    private var mCacheRoomInfo: LpfLiveroomtemplateV2.LiveRoomInfoV2? = null

    init {
        ALog.i(TAG, "init")
    }

    fun onDestroy() {
        mMicInfoListener = null
    }

    fun setMicInfoListener(micInfoListener: MicInfoListener.() -> Unit) {
        mMicInfoListener = ListenerBuilder().also(micInfoListener)
        ALog.i(TAG, "setMicInfoListener $micInfoListener")
        Handler(Looper.getMainLooper()).post {
            mCacheRoomInfo?.let {
                handleRoomInfoV2(it, true)
            }
        }
    }

    fun handleRoomInfoV2(roomInfoV2: LpfLiveroomtemplateV2.LiveRoomInfoV2, hasForceUpdate: Boolean = false) {
        mCacheRoomInfo = roomInfoV2
        /*if (mMicInfoListener == null) {
            ALog.i(TAG, "handleRoomInfoV2 mMicInfoListener == null")
            return
        }*/
        val newStreamInfos = roomInfoV2.liveInterconnectStreamInfos
        //List 为空，证明该房间已经没有人在直播了
        if (newStreamInfos.isEmpty()) {
            ALog.i(TAG, "************Room is empty************")
            //如果列表为空，则直接使用缓存数据，通知外面麦位当前处于关闭状态
            mCacheStreamInfos.values.forEach { item ->
                //如果以前的缓存数据 item 麦位是打开那么这时候就需要把当前麦位变化为 CLOSE 通知出去
                if (item.liveStatus == LpfMedia.LS_LIVING) {
                    item.liveStatus = LpfMedia.LS_LIVE_END
                    notifyStreamInfoChange(item)
                }
            }
            mCacheStreamInfos.clear()
            return
        }

        if (hasForceUpdate) {
            ALog.i(TAG, "handleRoomInfoV2 [hasForceUpdate :$hasForceUpdate]")
            mCacheStreamInfos.clear()
        }

        for (newStreamInfo in newStreamInfos) {
            updateStreamInfo(newStreamInfo)
        }
    }

    private fun updateStreamInfo(newStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo) {
        // todo 这里有可能拿到 空的 positionInfo
        val position = newStreamInfo.positionInfo.position
        // 本地缓存的 streamInfo
        val cacheStreamInfo = mCacheStreamInfos[position]
        //处理位置的 流管理
        handlePositionStreamStatus(cacheStreamInfo, newStreamInfo)
        //处理位置的 状态例如位置是否是空，位置是否处于静音，是否锁定
        handlePositionStatus(cacheStreamInfo, newStreamInfo)
        //处理位置使用者，当前处于前后台操作
        handlePositionClientStreamStatus(cacheStreamInfo, newStreamInfo)
        //处理位置用户信息拓展字段变更通知
        handlePositionUserInfoChange(cacheStreamInfo, newStreamInfo)
        mCacheStreamInfos[position] = newStreamInfo
    }

    /**
     *  处理麦位的流信息管理
     */
    private fun handlePositionStreamStatus(
        cacheStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo?,
        newStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo
    ) {
        if (cacheStreamInfo == null) {
            notifyStreamInfoChange(newStreamInfo)
            return
        }
        //判断是否是同一个用户
        if (isSamePerson(cacheStreamInfo, newStreamInfo)) {
            //判断各个状态是否一致如果一致 就直接 return 不需要处理
            var statusChange = false
            if (cacheStreamInfo.liveStatus != newStreamInfo.liveStatus) {
                ALog.i(TAG, "handlePositionStreamStatus [cache uid : ${cacheStreamInfo.user.uid} " +
                    "liveStatus : ${cacheStreamInfo.liveStatus}] " +
                    "[newS uid: ${newStreamInfo.user.uid} " +
                    "liveStatus : ${newStreamInfo.liveStatus}]")
                statusChange = true
            }
            if (cacheStreamInfo.mediaType != newStreamInfo.mediaType) {
                ALog.i(TAG, "handlePositionStreamStatus [cache uid : ${cacheStreamInfo.user.uid} " +
                    "mediaType : ${cacheStreamInfo.mediaType}] " +
                    "[newS uid: ${newStreamInfo.user.uid} " +
                    "mediaType : ${newStreamInfo.mediaType}]")
                statusChange = true
            }
            if (statusChange) {
                notifyStreamInfoChange(newStreamInfo)
            }
            return
        }
        //如果不是同一个麦上用户，先把原来的用户通知出去
        if (cacheStreamInfo != null) {
            //需要把原来的状态设置成功 设置为结束
            cacheStreamInfo.liveStatus = LpfMedia.LS_LIVE_END
            notifyStreamInfoChange(cacheStreamInfo)
        }
        notifyStreamInfoChange(newStreamInfo)
    }

    /**
     * THUNDER_URL = "thunder://${streamRoomId}/${uid}?${params}"
     */

    private fun notifyStreamInfoChange(streamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo) {
        if (streamInfo.user != null) {
            val micInfo = MicInfoV2(streamInfo, MicStatusV2.SEAT_NORMAL)
            val position = streamInfo.positionInfo.position
            val liveStatus = when (streamInfo.liveStatus) {
                LpfMedia.LS_LIVE_END -> LiveStatus.END

                LpfMedia.LS_LIVING -> LiveStatus.LIVING
                else -> LiveStatus.END
            }
            val mediaType = when (streamInfo.mediaType) {
                LpfMedia.MT_AUD -> MediaType.MT_AUDIO
                LpfMedia.MT_AV -> MediaType.MT_AV
                LpfMedia.MT_VIDEO -> MediaType.MT_VIDEO
                LpfMedia.MT_NONE -> MediaType.NONE
                else -> MediaType.NONE
            }
            val streamRoomId = streamInfo.streamRoomId
            val uids = ArrayList<Long>().apply {
                add(streamInfo.user.uid)
            }
            val thunderUrl = "thunder://$streamRoomId/${streamInfo.user.uid}/"
            val streamUrls = ArrayList<String>().apply {
                add(thunderUrl)
            }
            val lineStreamInfo = LineStreamInfo(
                LpfLiveroomtemplateV2.SSP_THUNDER_BOLOT,
                streamRoomId,
                uids,
                mediaType,
                MediaProtocol.THUNDER_BOLOT,
                streamUrls,
                "source",
                0,
                0,
                0,
                0)
            ALog.i(TAG, "notifyStreamInfoChange " +
                "[position: $position, " +
                "uid : ${streamInfo.user.uid}" +
                "sid : ${streamInfo.streamRoomId}" +
                "liveStatus: $liveStatus, " +
                "mediaType: $mediaType]")
            mMicInfoListener?.mOnStreamChangeAction?.invoke(position, liveStatus, mediaType, lineStreamInfo)
        }
    }

    private fun handlePositionStatus(
        cacheStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo?,
        newStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo
    ) {
        val currentPositionStatus = newStreamInfo.positionInfo.positionStatus
        val cachePositionStatus = cacheStreamInfo?.positionInfo?.positionStatus
        if (cachePositionStatus == currentPositionStatus) {
            return
        }
        val micStatus = when (currentPositionStatus) {
            LpfLiveinterconnect.LivePositionInfo.EMPTY -> MicStatusV2.SEAT_EMPTY
            LpfLiveinterconnect.LivePositionInfo.AUDIO_CLOSE -> MicStatusV2.SEAT_AUDIO_CLOSE
            LpfLiveinterconnect.LivePositionInfo.LOCKED -> MicStatusV2.SEAT_LOACKED
            LpfLiveinterconnect.LivePositionInfo.NORMAL -> MicStatusV2.SEAT_NORMAL
            else -> MicStatusV2.SEAT_EMPTY
        }
        val micInfoV2 = MicInfoV2(newStreamInfo, micStatus)
        val position = newStreamInfo.positionInfo.position
        ALog.i(TAG, "handlePositionStatus $position, $micStatus")
        mMicInfoListener?.mOnMicChangeAction?.invoke(position, micInfoV2)
    }

    private fun handlePositionClientStreamStatus(
        cacheStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo?,
        newStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo
    ) {

        if (cacheStreamInfo?.user?.uid == newStreamInfo.user?.uid &&
            cacheStreamInfo?.clientStreamStatus == newStreamInfo.clientStreamStatus) {
            return
        }

        val micStatusV2 = when (newStreamInfo.clientStreamStatus) {
            LpfMedia.CSS_LIVE_COMMON -> MicStatusV2.SEAT_FOREGROUND
            LpfMedia.CSS_RESTEAT_BG -> MicStatusV2.SEAT_BACKGROUND
            else -> MicStatusV2.SEAT_FOREGROUND
        }
        val position = newStreamInfo.positionInfo.position
        val micInfoV2 = MicInfoV2(newStreamInfo, micStatusV2)
        ALog.i(TAG, "handlePositionClientStreamStatus $position , $micStatusV2")
        mMicInfoListener?.mOnMicChangeAction?.invoke(position, micInfoV2)
    }

    private fun handlePositionUserInfoChange(
        cacheStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo?,
        newStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo
    ) {
        if (cacheStreamInfo?.user.toString() != newStreamInfo.user?.toString()) {
            val micStatusV2 = MicStatusV2.SEAT_USER_INFO_CHANGE
            val micInfoV2 = MicInfoV2(newStreamInfo, micStatusV2)
            val position = newStreamInfo.positionInfo.position
            ALog.i(TAG, "handlePositionUserInfoChange $position, $micStatusV2")
            mMicInfoListener?.mOnMicChangeAction?.invoke(position, micInfoV2)
        }
    }

    private fun isSamePerson(
        cacheStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo,
        newStreamInfo: LpfLiveroomtemplateV2.LiveInterconnectStreamInfo
    ): Boolean {
        return (cacheStreamInfo.sid != 0L
            && newStreamInfo.sid != 0L
            && cacheStreamInfo.sid == newStreamInfo.sid
            && cacheStreamInfo.user != null
            && newStreamInfo.user != null
            && cacheStreamInfo.user.uid != 0L
            && newStreamInfo.user.uid != 0L
            && cacheStreamInfo.user.uid == newStreamInfo.user.uid)
    }

    fun getMicInfos(): HashMap<Int, LpfLiveroomtemplateV2.LiveInterconnectStreamInfo>? {
        ALog.i(TAG, "getMicInfos size = ${mCacheStreamInfos.size}")
        return mCacheStreamInfos
    }

    inner class ListenerBuilder : MicInfoListener {
        internal var mOnMicChangeAction: ((position: Int, micInfoV2: MicInfoV2) -> Unit)? = null
        internal var mOnStreamChangeAction: ((
            position: Int,
            liveStatus: LiveStatus,
            mediaType: MediaType,
            lineStreamInfo: LineStreamInfo
        ) -> Unit)? = null

        override fun onMicInfoChange(action: (position: Int, micInfoV2: MicInfoV2) -> Unit) {
            this.mOnMicChangeAction = action
        }

        override fun onStreamChange(
            action: (
                position: Int,
                liveStatus: LiveStatus,
                mediaType: MediaType,
                lineStreamInfo: LineStreamInfo
            ) -> Unit
        ) {
            this.mOnStreamChangeAction = action
        }
    }
}