package tv.athena.live.component.business.roomInfov2

import android.arch.lifecycle.Observer
import android.text.TextUtils
import com.thunder.livesdk.ThunderRtcConstant
import com.yy.liveplatform.proto.nano.LpfLiveroomtemplateV2
import tv.athena.annotation.MessageBinding
import tv.athena.live.api.roominfov2.RoomInfoV2Api
import tv.athena.live.base.LiveRoomBzMode
import tv.athena.live.base.arch.IComponentViewModel
import tv.athena.live.basesdk.liveroom.ApplicationStatus
import tv.athena.live.basesdk.thunderblotwrapper.AbscThunderEventListener
import tv.athena.live.basesdk.thunderblotwrapper.ThunderHandleManager
import tv.athena.live.component.business.activitybar.webview.util.JsonParser
import tv.athena.live.utils.ALog
import tv.athena.live.utils.RoomInfoV2Repository
import tv.athena.live.utils.ServiceUtils
import tv.athena.service.api.MessageResponse
import tv.athena.service.api.ServiceFailResult
import tv.athena.service.api.event.ServiceBroadcastEvent

/**
 * @author : chf
 * @e-mail : chenhaofeng@joyy.sg
 * @date : 2019/11/19 20:02
 * @desc :
 */
class RoomInfoV2ViewModel() :
    IComponentViewModel,
    AbscThunderEventListener() {

    companion object {
        const val TAG = "RoomInfoV2ViewModel"
        const val FUNC_UPDATE_LIVE_ROOM_INFO_BROADCAST_V2 = "updateLiveRoomInfoBroadcastV2"
        const val ROOM_INFO_SERVICE_NAME = "lpfLiveRoomTemplateV2"
    }

    //记录上一次RoomInfoV2 的时间戳
    private var lastTimeStamp = -1L
    private val mRoomInfoV2Listeners = ArrayList<RoomInfoV2Api.RoomInfoV2Listener>()
    private var mRoomInfoV2Component: RoomInfoV2Component? = null

    private val observerStreamRoomId = Observer<String> {
        it?.let {
            ALog.i(TAG, "observerStreamRoomId $it ")
            val liveRoomTemplateV2 = RoomInfoV2Repository.mCacheRepository[it]
            ALog.i(TAG, "observerStreamRoomId [mCacheTimeStamp :${RoomInfoV2Repository.mCacheTimeStamp}] ")
            handleLiveRoomInfoV2("onCreate", liveRoomTemplateV2, RoomInfoV2Repository.mCacheTimeStamp)
            //这里还需要主动地更新一次
            getLiveRoomInfoRespV2(it)
        }
    }

    /**
     *  这里需要有这个配置才行，不然不太清楚什么时候主动获取与请求
     */
    private val observerHasRegisterBroadcastFlag = Observer<Boolean> {
        ALog.i(TAG, "observerHasRegisterBroadcastFlag $it ")
        if (it == true) {
            val streamRoomId = mRoomInfoV2Component
                ?.componentContext
                ?.commonViewModel
                ?.streamRoomId
            streamRoomId?.let { it ->
                ALog.i(TAG, "observerHasRegisterBroadcastFlag [sid: $it]")
                val liveRoomTemplateV2 = RoomInfoV2Repository.mCacheRepository[it]
                ALog.i(TAG, "observerHasRegisterBroadcastFlag" +
                    " [mCacheTimeStamp :${RoomInfoV2Repository.mCacheTimeStamp}]" +
                    " [cacheContainer: ${RoomInfoV2Repository.mCacheRepository.containsKey(it)}")
                handleLiveRoomInfoV2("observerHasRegisterBroadcastFlag", liveRoomTemplateV2,
                    RoomInfoV2Repository.mCacheTimeStamp)
                //这里还需要主动地更新一次
                getLiveRoomInfoRespV2(it)
            }
        }
    }

    fun onLeave() {
        ThunderHandleManager.mAthLiveThunderEventCallback.unRegisterRtcEventListener(this)
    }

    fun destroy() {
        ALog.i(TAG, "destroy")
        ThunderHandleManager.mAthLiveThunderEventCallback.unRegisterRtcEventListener(this)
        RoomInfoV2Repository.clearCache()
        mRoomInfoV2Component?.apply {
            componentContext.commonViewModel
                ?.removeObserveStreamRoomIdForever(observerStreamRoomId)
            componentContext.commonViewModel
                ?.removeHasRegisterBroadcastFlag(observerHasRegisterBroadcastFlag)
        }
    }

    fun onCreate(component: RoomInfoV2Component) {
        ALog.i(TAG, "onCreate ($component))")
        ThunderHandleManager.mAthLiveThunderEventCallback.registerThunderEventListener(this)
        mRoomInfoV2Component = component
        mRoomInfoV2Component?.apply {
            componentContext.commonViewModel
                ?.observeStreamRoomIdForever(observerStreamRoomId)
            componentContext.commonViewModel
                ?.observeHasRegisterBroadcastFlag(observerHasRegisterBroadcastFlag)
        }
    }

    private var mLastLiveRoomInfoV2: LpfLiveroomtemplateV2.LiveRoomInfoV2? = null
    private fun handleLiveRoomInfoV2(from: String, roomInfoV2: LpfLiveroomtemplateV2.LiveRoomInfoV2?, timeStamp: Long) {
        if (roomInfoV2 == null) {
            ALog.i(TAG, "handleLiveRoomInfoV2 from:$from,roomInfoV2 = null")
            return
        }

        val currentStreamRoomId = mRoomInfoV2Component?.componentContext?.commonViewModel?.streamRoomId
        val broadcastStreamRoomId = roomInfoV2.channelInfo?.streamRoomId
        ALog.i(TAG, "onBroadcast RoomInfoV2Event from:$from " +
            "[currentStreamSid : $currentStreamRoomId ] ; " +
            "[broadcastStreamRoomId : $broadcastStreamRoomId]")
        if (currentStreamRoomId != broadcastStreamRoomId) {
            return
        }

        if (lastTimeStamp > timeStamp) {
            ALog.i(TAG, "handleLiveRoomInfoV2 from:$from,lastTimeStamp > timeStamp:$lastTimeStamp,$timeStamp")
            return
        }
        lastTimeStamp = timeStamp
        val roomInfoJson = JsonParser.toJson(roomInfoV2)
        val isSame = JsonParser.toJson(mLastLiveRoomInfoV2) == roomInfoJson   //耗时在2ms
        if (isSame) {
            ALog.i(TAG, "handleLiveRoomInfoV2 from:$from,$lastTimeStamp,$timeStamp,mCacheLiveRoomInfoV2 == roomInfoV2")
        } else {
            mLastLiveRoomInfoV2 = roomInfoV2
            ALog.i(TAG, "handleLiveRoomInfoV2 from:$from,$lastTimeStamp,$timeStamp,$roomInfoV2")
        }
        handleRoomInfoV2(roomInfoV2)
        roomInfoV2.channelInfo?.let {
            RoomInfoV2Repository.mCacheRepository[it.streamRoomId] = roomInfoV2
        }
    }

    /**
     *  这里处理流信息
     */
    fun handleRoomInfoV2(roomInfoV2: LpfLiveroomtemplateV2.LiveRoomInfoV2) {
        mRoomInfoV2Listeners.forEach { listener ->
            listener.onRoomInfoV2Change(roomInfoV2)
        }
    }

    fun getLiveRoomInfoRespV2(streamRoomId: String) {
        val req = LpfLiveroomtemplateV2.GetLiveRoomInfoReqV2()
        val liveRoomBzMode = mRoomInfoV2Component?.componentContext?.mLiveRoomBzMode
        val hasRegisterBroadcastByStreamRoomId =
            mRoomInfoV2Component?.componentContext?.commonViewModel?.hasRegisterBroadcastByStreamRoomId

        if (liveRoomBzMode == LiveRoomBzMode.PURESDK &&
            hasRegisterBroadcastByStreamRoomId != true) {
            ALog.i(TAG, "getLiveRoomInfoRespV2 当前 puresdk 模式，但是并没有注册中台sid无法获取")
            return
        }
        if (TextUtils.isEmpty(streamRoomId)) {
            ALog.i(TAG, "getLiveRoomInfoRespV2 当前 puresdk 模式，streamRoomId == null or empty")
            return
        }
        req.streamRoomId = streamRoomId
        ALog.i(TAG, "getLiveRoomInfoRespV2 req: $req")
        RoomInfoV2Repository.getLiveRoomInfoRespV2(req,
            object : ServiceUtils.SvcResultCusRetryCallBack<LpfLiveroomtemplateV2.GetLiveRoomInfoRespV2>() {
                override fun get(): LpfLiveroomtemplateV2.GetLiveRoomInfoRespV2 {
                    return LpfLiveroomtemplateV2.GetLiveRoomInfoRespV2()
                }

                override fun onMessageFail(errorCode: ServiceFailResult, ex: Exception?) {
                    ALog.i(TAG, "getLiveRoomInfoRespV2 onMessageFail($errorCode)")
                }

                override fun onMessageSuccess(
                    response: MessageResponse<LpfLiveroomtemplateV2.GetLiveRoomInfoRespV2>
                ) {
                    ALog.i(TAG, "getLiveRoomInfoRespV2 onMessageSuccess ${response.message.code}")
                    when (response.message.code) {
                        0 -> {
                            val timeStamp = response.message.timestamp
                            val roomInfoV2 = response.message.liveRoomInfo
                            handleLiveRoomInfoV2("onRequest", roomInfoV2, timeStamp)
                        }
                    }
                }
            })
    }

    @MessageBinding
    fun onBroadcastGroupEvent(event: ServiceBroadcastEvent?) {
        event?.let {
            if (event.serverName != ROOM_INFO_SERVICE_NAME) {
                return
            }
            if (event.funcName == FUNC_UPDATE_LIVE_ROOM_INFO_BROADCAST_V2) {
                val broadcastV2 = LpfLiveroomtemplateV2.UpdateLiveRoomInfoBroadcastV2
                    .parseFrom(event.message)
                val roomInfoV2 = broadcastV2.liveRoomInfo
                val timeStamp = broadcastV2.timestamp
                ALog.i(TAG, "onBroadcastGroupEvent $this")
                handleLiveRoomInfoV2("onBroadcast", roomInfoV2, timeStamp)
            }
        }
    }

    @MessageBinding
    fun appStatus(applicationStatus: ApplicationStatus) {
        ALog.i(TAG, "appStatus [status : ${applicationStatus.status}]")
        when (applicationStatus.status) {
            ApplicationStatus.Status.ONBACKGROUND -> {
            }
            ApplicationStatus.Status.ONFOREGROUND -> {
                mRoomInfoV2Component?.apply {
                    componentContext.commonViewModel?.streamRoomId?.let {
                        ALog.i(TAG, "appOnForeground getLiveRoomInfoRespV2")
                        getLiveRoomInfoRespV2(it)
                    }
                }
            }
        }
    }

    fun addRoomInfoV2Listener(roomInfoV2Listener: RoomInfoV2Api.RoomInfoV2Listener) {
        if (!mRoomInfoV2Listeners.contains(roomInfoV2Listener)) {
            ALog.i(TAG, "addStreamInfoListener : $roomInfoV2Listener")
            mRoomInfoV2Listeners.add(roomInfoV2Listener)
        }
    }

    fun removeRoomInfoV2Listener(roomInfoV2Listener: RoomInfoV2Api.RoomInfoV2Listener) {
        if (!mRoomInfoV2Listeners.contains(roomInfoV2Listener)) {
            ALog.i(TAG, "removeRoomInfoV2Listener : $roomInfoV2Listener")
            mRoomInfoV2Listeners.remove(roomInfoV2Listener)
        }
    }

    override fun onNetworkTypeChanged(type: Int) {
        super.onNetworkTypeChanged(type)
        if (type != ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_DISCONNECTED) {
            mRoomInfoV2Component?.apply {
                componentContext.commonViewModel?.streamRoomId?.let {
                    ALog.i(TAG, "appOnForeground getLiveRoomInfoRespV2")
                    getLiveRoomInfoRespV2(it)
                }
            }
        }
    }
}
