package com.yy.platform.loginlite;

public interface IAuthCore {

    /**
     * HiidoSDK由APP层引入，设置接收统计数据的回调
     *
     * @param listener
     */
    void setHiidoMetricsApi(ILoginliteListener.ILoginliteHiidoMetricsStatisApi listener);

    /**
     * 请求短信验证码
     *
     * @param userPhoneNumber 手机号码，加上区码前缀：中国 0086 印尼 0062 美国 001
     * @param smsType         短信类型，0：文本短信，1：语音短信。默认0
     * @param smsLength       短信的长度，"4"：4位长度，"6"：6位长度，填""：6位长度
     * @param dynCode         *   二次验证的验证码，首次请求或者没有启用二次验证策略的则给""
     * @param callback        *   请求的回执，成功则会收到手机短信或提示需要二次验证
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int getSms(String userPhoneNumber, String smsType, String smsLength,
               String dynCode, IGetSmsCallback callback);

    /**
     * 短信验证码登录
     *
     * @param userPhoneNumber 手机号码，加上区码前缀：中国 0086 印尼 0062 美国 001
     * @param smsCode         短信验证码
     * @param dynCode         *   二次验证的验证码，首次请求或者没有启用二次验证策略的则给""
     * @param callback        登录结果回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int smsLogin(String userPhoneNumber, String smsCode, String dynCode, ISmsLoginCallback callback);

    /**
     * 账号密码登录
     *
     * @param account  帐号
     * @param password 密码
     * @param dynCode  *   二次验证的验证码，首次请求或者没有启用二次验证策略的则给""
     * @param callback 登录结果回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int passwordLogin(String account, String password, String dynCode, IPasswordLoginCallback callback);

    /**
     * 凭证登录，前一次登录成功后app可以保留凭证用作下一次调用本接口自动登录
     *
     * @param yyuid    帐号UID
     * @param callback 登录回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int creditLogin(long yyuid, ICreditLoginCallback callback);

    /**
     * 凭证登录
     *
     * @param yyuid    帐号UID
     * @param credit   凭证
     * @param callback 登录回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int creditLogin(long yyuid, String credit, ICreditLoginCallback callback);

    /**
     * 第三方登录，常见失败的错误码请参考ThirdLoginError.java
     *
     * @param channel   渠道，取如下值之一:
     *                  {facebook,google,twitter,instagram,weibo,qq,wechat}
     * @param token     从第三方得到的code 或者 accesstoken
     * @param tokenType token为第三方的code时给0，为accesstoken时给1
     * @param openid    第三方的UserId等，非必填，如果有就填上，没有就给""
     * @param authUrl   在第三方注册app时填的鉴权回调url，非必填，如果有就填上，没有就给""
     * @param extInfo   (预留参数)扩展信息参数，json格式，非必填，没有就给""
     * @param callback  本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     * @return
     */
    int thirdLogin(String channel, String token, int tokenType, String openid, String authUrl,
                   String extInfo, IThirdLoginCallback callback);

    /**
     * 第三方登录，如果未绑定手机号，会弹出h5页面去绑定手机号，常见失败的错误码请参考ThirdLoginError.java
     *
     * @param channel   渠道，取如下值之一:
     *                  {facebook,google,twitter,instagram,weibo,qq,wechat}
     * @param token     从第三方得到的code 或者 accesstoken
     * @param tokenType token为第三方的code时给0，为accesstoken时给1
     * @param openid    第三方的UserId等，非必填，如果有就填上，没有就给""
     * @param authUrl   在第三方注册app时填的鉴权回调url，非必填，如果有就填上，没有就给""
     * @param lang      语言码
     * @param callback  本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     * @return
     */
    int thirdLoginWithBindPhone(String channel, String token, int tokenType, String openid,
                                String authUrl, String lang, IThirdLoginCallback callback);

    /**
     * 获取目标appId的OTP用于登录等
     *
     * @param destAppId 目标业务AppId，取本App的UDB登录token 给空即可
     * @return
     */
    byte[] getOtp(String destAppId, long uid);

    /**
     * 获取bind Service 的Token
     *
     * @return
     */
    byte[] getServiceToken(long uid);

    /**
     * 退出当前登录的UID，退出后所有OTP失效，如有需要则重新登录
     * 注意调用此接口后也需要调用ServiceSDK的unBind，保证登录相关的逻辑正确
     *
     * @param enableAutoLoginNext 是否允许下次可以进行creditLogin 自动登录，
     *                            此参数给false，则下次登录必须要用手机号或者密码重登了，
     *                            所以大多数情况此参数要给true
     */
    void logout(boolean enableAutoLoginNext);

    /***
     * 扫码授权
     * @param yyuid
     *          帐号,uid
     * @param qrId
     *          二维码id，扫码后从二维码图片中获取qrcode_id
     * @param callback
     *          回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int qrScan(long yyuid, String qrId, IQrScanCallback callback);

    /**
     * 点击确认，授权显码方登录
     *
     * @param yyuid    帐号,uid
     * @param qrId     二维码id，扫码后从二维码图片中获取qrcode_id
     * @param callback 回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int qrAuthConfirm(long yyuid, String qrId, IQrConfirmCallback callback);

    /**
     * 点击取消，取消授权登录
     *
     * @param yyuid    帐号,uid
     * @param qrId     二维码id，扫码后从二维码图片中获取qrcode_id
     * @param callback 回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int qrAuthCancel(long yyuid, String qrId, IQrCancelCallback callback);

    /**
     * 第三方登录后绑定手机号
     *
     * @param phoNum   手机号码
     * @param uid      UID
     * @param smsCode  短信验证码
     * @param callback 手机绑定回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int bindMobilePhone(String phoNum, long uid, String smsCode, IBindMobilePhoneCallback callback);

    /**
     * 第三方登录
     *
     * @param channel   渠道，取如下值之一:
     *                  {facebook,google,twitter,instagram,weibo,qq,wechat}
     * @param token     从第三方得到的code 或者 accesstoken
     * @param tokenType token为第三方的code时给0，为accesstoken时给1
     * @param openid    第三方的UserId等，非必填，如果有就填上，没有就给""
     * @param uid       用户id
     * @param optToken  token
     * @param authUrl   在第三方注册app时填的鉴权回调url，非必填，如果有就填上，没有就给""
     * @param extInfo   (预留参数)扩展信息参数，json格式，非必填，没有就给""
     * @param callback  本次请求的taskID
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int bindThirdToken(String channel, String token, int tokenType, String openid, String uid, String optToken,
                       String authUrl, String extInfo, IThirdLoginCallback callback);

    /**
     * 获取短信验证码，带场景的情况
     *
     * @param phoneNum  手机号
     * @param smsType   短信类型，0：文本短信，1：语音短信,2:发两条短信。默认0
     * @param smsLength 短信长度，"4"4位短信码，"6"6位短信码；填nil，默认为6位短信码
     * @param useType   场景， 1：登录，2：注册，3：改密，4：找回密码，5：绑定手机，6：修改手机
     * @param otp       若是用于改密、绑定手机场景，需要传otp
     * @param dynCode   二次验证的验证码，首次请求或者没有启用二次验证策略的则给""
     * @param callback  获取短信验证码的回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int getSmsWithBusiType(String phoneNum, String smsType, String smsLength, int useType,
                           String otp, String dynCode, IGetSmsCallback callback);

    /**
     * 验证短信验证码
     * @param phoneNum    手机号
     * @param verifyCode  验证码
     * @param useType     验证码用途，1：登录，2：注册，3：改密 4：找回密码
     * @param otp         若是用于改密、绑定手机场景，需要传otp
     * @param dynCode     二次验证的验证码，首次请求或者没有启用二次验证策略的则给""
     * @param callback    验证短信验证码的回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int verifySmsCode(String phoneNum, String verifyCode, int useType, String otp,
    String dynCode, IBaseCallBack callback);

    /**
     * 获取邮箱验证码
     *
     * @param email    邮箱
     * @param codeLen  验证码长度，默认是6
     * @param useType  验证码用途，2：注册，3：改密，4：找回密码，7：绑定邮箱，8：修改邮箱
     * @param otp      若是用于改密、绑定邮箱场景，需要传otp
     * @param dynCode  二次验证的验证码，首次请求或者没有启用二次验证策略的则给""
     * @param callback 获取邮箱验证码的回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int getEmailVerifyCode(String email, String codeLen, int useType, String otp, String dynCode,
                           IBaseCallBack callback);

    /**
     * 验证邮箱验证码
     *
     * @param email      邮箱
     * @param verifyCode 验证码
     * @param useType    验证码用途，2：注册，3：改密，4：找回密码
     * @param otp        若是用于改密、绑定邮箱场景，需要传otp
     * @param dynCode    二次验证的验证码，首次请求或者没有启用二次验证策略的则给""
     * @param callback   验证邮箱验证码的回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int verifyEmailCode(String email, String verifyCode, int useType, String otp, String dynCode,
                        IBaseCallBack callback);

    /**
     * 注册账号
     *
     * @param account    账号(邮箱、手机号)
     * @param verifyCode 验证码
     * @param password   密码
     * @param dynCode    二次验证的验证码，首次请求或者没有启用二次验证策略的则给""
     * @param callback   注册账号的回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int registerByVerifyCode(String account, String verifyCode,
                             String password, String dynCode, IAuthInfoCallBack callback);

    /**
     * 修改密码
     *
     * @param account    账号(邮箱、手机号)
     * @param verifyCode 验证码
     * @param password   密码
     * @param otp        找回密码不用otp，改密需要登录态要otp
     * @param dynCode    二次验证的验证码，首次请求或者没有启用二次验证策略的则给""
     * @param callback   修改密码的回调
     * @return 本次请求的结果，如果等于-1表示参数有错误，无法发送本次请求
     */
    int modifyPwdByVerifyCode(String account, String verifyCode,
                              String password, String otp, String dynCode, IBaseCallBack callback);
}
