package com.yy.platform.loginlite;

import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;

import com.yy.platform.loginlite.utils.HttpClient;
import com.yy.platform.loginlite.utils.NetworkUtil;
import com.yy.platform.loginlite.utils.NetworkUtil.NetErrorType;
import com.yy.platform.loginlite.utils.ServiceUrls;
import com.yy.platform.loginlite.utils.StringUtils;

import java.io.IOException;
import java.util.Map;
import java.util.UUID;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Request;
import okhttp3.Response;

public abstract class BaseOkHttp {

    protected Call mCall;
    private final static int count = 3;
    private volatile int index = 0;
    private final static int DURATION = 47 * 1000;
    private Handler mHandler = new Handler(Looper.getMainLooper());
    protected String traceId;
    protected String tag;
    private volatile boolean isUseIp = false;
    private volatile boolean isCancel = false;
    private static final int TYPEIP = 1;
    private static final int TYPEURL = 2;
    private volatile int urlType = TYPEURL;
    private volatile String currentUrl = "";
    private static volatile String extra = "";

    private int retryCount = -1;
    private int seconds = -1;

    public BaseOkHttp() {
        index = 0;
        isUseIp = false;
    }

    public static String getExtra() {
        return extra;
    }

    public static void clearExtra() {
        extra = "";
    }

    public void cancel() {
        extra = "";
        isCancel = true;
        mHandler.removeCallbacksAndMessages(null);
        if (mCall != null) {
            mCall.cancel();
        }
    }

    protected String getCurrentUrl() {
        return currentUrl;
    }

    protected void setRetryCount(int retryCount) {
        if (retryCount <= 0) {
            return;
        }
        this.retryCount = retryCount;
    }

    protected void setRequestTimeoutSeconds(int seconds) {
        if (seconds <= 0) {
            return;
        }
        this.seconds = seconds;
    }

    public boolean isRequestCancel() {
        return isCancel;
    }

    protected void executeInner(
            final String path, final Map<String, String> headers,
            final byte[] datas, final int timeout
    ) {
        extra = "";
        synchronized (this) {
            if (isCancel) {
                ALog.e(tag + " stop");
                mHandler.removeCallbacksAndMessages(null);
                return;
            }
        }
        int cal = count * timeout + 2 * 1000;
        final int finishTime;
        if (cal < DURATION) {
            finishTime = DURATION;
        } else {
            finishTime = cal;
        }
        ALog.e("timeout: " + timeout + ", finishTime: " + finishTime);
        traceId = UUID.randomUUID().toString();

        String url;
        // 使用http的ip直连
        if (isUseIp) {
            String ipUrl = ServiceUrls.getIpUrl();
            if (TextUtils.isEmpty(ipUrl)) {
                urlType = TYPEURL;
                url = ServiceUrls.getHttpUrl();
            } else {
                urlType = TYPEIP;
                url = ipUrl;
            }
        } else {
            urlType = TYPEURL;
            url = ServiceUrls.getHttpUrl();
        }

        if (TextUtils.isEmpty(url)) {
            extra = "";
            onHttpFail(NetErrorType.NO_URL, "lack of url", 0);
            return;
        }
        currentUrl = url;

        url = url + path;

        ALog.i(tag + ", traceId: " + traceId + ", url: " + url);
        headers.put("TraceId", traceId);
        final long startTime = System.currentTimeMillis();
        Request request = HttpClient.makePostRequest(url, headers, datas, timeout);
        mCall = HttpClient.getInstance().newCall(request);
        mCall.enqueue(new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                if (call.isCanceled()) {
                    ALog.e("onFailure call is canceled");
                    return;
                }
                if (urlType == TYPEIP) {
                    ServiceUrls.changeIpUrl();
                }
                canUseIp(e);

                long duration = System.currentTimeMillis() - startTime;
                index++;
                String info = "(" + index + "/" + count + ")";
                int code = NetworkUtil.getErrorCodeByExc(e);
                ALog.e("onFailure " + info + ", use time " + duration + ", code: " + code);
                if (index < count && duration < finishTime) {
                    mHandler.postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            executeInner(path, headers, datas, timeout);
                        }
                    }, 1000);
                } else {
                    String message = e != null ? e.getMessage() : "network error";
                    extra = " , timeout: " + timeout + "ms, realtime: " + finishTime +
                            "ms, retry time: " + index + ", url: " + currentUrl;
                    ALog.e("onFailure message: " + message + extra);
                    onHttpFail(code, message, duration);
                }
                if (urlType == TYPEURL) {
                    ServiceUrls.changeUrl();
                }
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                long duration = System.currentTimeMillis() - startTime;
                String info = "(" + index + "/" + count + ")";
                ALog.e("onResponse " + info + ", use time " + duration + ", code " + response.code());

                if (call.isCanceled()) {
                    ALog.e("onResponse call is canceled");
                    return;
                }
                extra = " , timeout: " + timeout + ", realtime: " + finishTime +
                        ", retry time: " + index + ", url: " + currentUrl;
                if (!response.isSuccessful()) {
                    ALog.e("response is not success " + info + ", use time " + duration);
                    index++;
                    if (index < count && duration < finishTime) {
                        mHandler.postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                executeInner(path, headers, datas, timeout);
                            }
                        }, 1000);
                    } else {
                        String message = response.message();
                        ALog.e("onResponse fail message: " + message + extra);
                        onFail(response.code(), message, duration);
                    }
                    if (urlType == TYPEIP) {
                        ServiceUrls.changeIpUrl();
                    }

                    if (urlType == TYPEURL) {
                        ServiceUrls.changeUrl();
                    }
                } else {
                    int code = response.code();
                    ALog.e("response is success");
                    byte[] data = response.body().bytes();
                    onSuccess(call, data, code, duration);
                }
            }
        });
    }

    private void canUseIp(boolean flag) {
        isUseIp = flag;
    }

    private void canUseIp(IOException e) {
        isUseIp = false;

        String errorMsg = e != null ? e.toString() : "";
        if (TextUtils.isEmpty(errorMsg)) {
            return;
        }
        ALog.e("useIp: " + errorMsg);
        if (errorMsg.length() > 200) {
            errorMsg = errorMsg.substring(0, 200);
        }

        if (errorMsg.contains("java.net.ConnectException")) {
            isUseIp = true;
        } else if (errorMsg.contains("java.net.UnknownHostException")) {
            isUseIp = true;
        } else if (errorMsg.contains("java.net.SocketTimeoutException")) {
            isUseIp = true;
        } else if (errorMsg.contains("javax.net.ssl.SSLHandshakeException")) {
            isUseIp = true;
        } else if (errorMsg.contains("java.net.NoRouteToHostException")) {
            isUseIp = true;
        }
    }

    /**
     * eventType拼接http，https或者ip
     *
     * @param funcType
     * @return
     */
    protected String compose(String funcType) {
        if (TextUtils.isEmpty(funcType)) {
            return "https";
        }
        if (urlType == TYPEIP) {
            return funcType + "/ip";
        }
        if (currentUrl.startsWith("https://")) {
            return funcType + "/https";
        }
        if (currentUrl.startsWith("http://")) {
            return funcType + "/http";
        }
        return funcType + "/https";
    }

    public abstract void onSuccess(Call call, byte[] data, int code, long useTime);

    public abstract void onFail(int code, String message, long useTime);

    public abstract void onHttpFail(int code, String message, long useTime);
}
