package com.thunder.livesdk.video;

import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import com.thunder.livesdk.BuildConfig;
import com.thunder.livesdk.ThunderMultiVideoViewParam;
import com.thunder.livesdk.ThunderRtcConstant;
import com.thunder.livesdk.log.ThunderLog;
import com.thunder.livesdk.helper.ThunderNative;
import com.yy.videoplayer.VideoDecodeEventNotify;
import com.yy.videoplayer.stat.IYMFBehaviorEventListener;
import com.yy.videoplayer.VideoPlayer;
import com.yy.videoplayer.decoder.YYVideoLibMgr;
import com.yy.videoplayer.stat.YMFPlayerStatisticManager;
import com.yy.videoplayer.VideoRenderNotify;
import com.yy.videoplayer.stat.YMFPlayerUsrBehaviorStat;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import static java.lang.Thread.MAX_PRIORITY;

public class ThunderVideoPlayEngineImp implements IThunderVideoPlay, IYMFBehaviorEventListener {

    private static final String TAG = "ThunderVideoPlayEngineImp";
    private static long mCallBackPtr = 0;
    public final int INVALID_VALUE = -1;

    private final static int VIDEO_STAT_FPS = 0;
    private final static int VIDEO_STAT_BITRATE = 1;
    private final static int VIDEO_STATE_RESOLUTION = 2;
    private final static int VIDEO_STATE_DECODER_TYPE = 3;

    private ThunderVideoHiidoUtil mVideoPlayHiidoUtil;
    private static ReentrantReadWriteLock s_observerLock = null;

    private HandlerThread mThunderVideoHandlerThread = null;
    private Handler mThunderVideoHandler = null;
    private ConcurrentHashMap<String, VideoStreamInfo> mVideoStreamInfoMap;
    private ConcurrentHashMap<Long, StrUidVideoDecodeObserver> mVideoDecodeObserverMap;
    private ConcurrentHashMap<String, Integer> mMultiVideoScaleModeMap;
    private ConcurrentHashMap<String, Integer> mMultiVideoMirrorModeMap;
    private ConcurrentHashMap<String, Long> mMultiStreamKeyIdMap;
    private ConcurrentHashMap<String, String> mStreamKeyStrUidMap = new ConcurrentHashMap<String, String>();
    // uid32:streamid 保存大小流切换过程需要回调解码数据的streamid
    private ConcurrentHashMap<Long, Long> mUidToVideoIdCallBackInDualStreamMap;

    //C++层的uint64_t 到java层long会不能完全对应，这里做一层隐射全用C++层的uid
    private Map<Long, Long> mC2JavaUidMap = new HashMap<Long, Long>();

    private AtomicBoolean mThunderPlayerMultiViewMode = new AtomicBoolean(false);
    private ThunderVideoPlayListener mThunderVideoPlayListener = null;

    //多个多人view需要用到的数据结构
    //多个多人view，位置viewId与proxy映射，业务静态绑定
    private HashMap<Integer, ThunderPlayerMultiViewProxy> mThunderPlayerMultiViewProxyMap =
            new HashMap<>(10);
    //多个多人view，streamKey与proxy映射，动态绑定
    private HashMap<String, ThunderPlayerMultiViewProxy> mVideoMultiViewMap =
            new HashMap<String, ThunderPlayerMultiViewProxy>();
    //多个多人view，变更位置前streamKey与proxy映射
    private HashMap<String, ThunderPlayerMultiViewProxy> mVideoMultiViewRecoverMap =
            new HashMap<String, ThunderPlayerMultiViewProxy>();

    @Override
    public void onViewStateNotify(long streamId, int viewState) {
        if (mCallBackPtr != 0) {
            onViewStateNotify(mCallBackPtr, streamId, viewState);
        }
    }

    @Override
    public void onVideoRenderNotify(ArrayList<VideoRenderNotify> notifys) {
        if (mCallBackPtr != 0) {
            onVideoRenderedFrames(mCallBackPtr, notifys);
        }
    }

    @Override
    public void onVideoDecodeNotify(VideoDecodeEventNotify notifys) {
        if (mCallBackPtr != 0) {
            // ThunderLog.info(TAG, "onVideoDecodeNotify streamid:"+notifys.mStreamId+" pts:"+notifys.mPts);
            onVideoDecodedFrames(mCallBackPtr, notifys);
        }
    }

    @Override
    public void onFirstFrameRenderNotify(long streamId, int width, int height, long pts) {
        if (mCallBackPtr != 0) {
            onFirstFrameRender(mCallBackPtr, streamId, width, height, pts);
        }
    }

    public void onMultiViewSeatInfoChangedNotify(long streamId, int seatId) {
/** 这里先干掉：这里涉及到native调用会转thunder线程场景，因为在多人连麦视图场景，更换视图位置时，该通知会导致一下同步问题；
 * 例如：假如曾经多人视图有4个用户：A-1、B-2、C-3、D-4，后面跨频道退订这些用户或则这些用户停播， 后来再次更改成 A-3、B-1、C-4、D-2，
 * 假如先来了A的流，但是没有来C的流通知，此时更新 A-3 给视频库，由于视频库保留了之前设置座位3上时用户C，所以会通过该接口通知LiveEngine C的座位号需要重置为-1，
 * 这样就导致 C 的座位号变成了 -1，当C来流的时候，就会出现座位号为-1更新给视频库，从而看不到C的视频。
        if (mCallBackPtr != 0) {
            Long uid = null;
            try {
                s_observerLock.readLock().lock();
                if (mC2JavaUidMap != null) {
                    uid = mC2JavaUidMap.get(streamId >> 32);
                }
            } catch (Exception e) {
                ThunderLog.error(TAG, "onMultiViewSeatInfoChangedNotify err=%s", e.toString());
            } finally {
                s_observerLock.readLock().unlock();
            }

            if (uid != null) {
                onMultiViewSeatInfoNotify(mCallBackPtr, uid, seatId);
            }
        }
 **/
    }

    @Override
    public void onHardwareDecodeErrorNotify(long userGroupId, long streamId, int errorType) {
        if (mCallBackPtr != 0) {
            onHardwareDecodeError(mCallBackPtr, userGroupId, streamId, errorType);
        }
    }

    @Override
    public void onDecodedFrameData(long streamId, int w, int h, byte[] data, int dataLen, long renderTimeMs) {
        try {
            s_observerLock.readLock().lock();
            do {
                long javaUid = streamId >> 32;
                long uid = mC2JavaUidMap.get(javaUid) == null ? (int) javaUid : mC2JavaUidMap.get(javaUid);
                if (mVideoDecodeObserverMap == null || mVideoDecodeObserverMap.isEmpty()) {
if (ThunderLog.isInfoValid()) {
                    ThunderLog.info(TAG, "onDecodedFrameData empty observer map");
}
                    break;
                }
                StrUidVideoDecodeObserver strUidObserver = mVideoDecodeObserverMap.get(uid);
                if (strUidObserver == null) {
if (ThunderLog.isInfoValid()) {
                    ThunderLog.info(TAG, "onVideoDecodedFrame not found decoder uid %d", uid);
}
                    break;
                }
                IVideoDecodeObserver observer = strUidObserver.getObserver();
                if (observer == null) {
if (ThunderLog.isInfoValid()) {
                    ThunderLog.info(TAG, "onVideoDecodedFrame not found observer uid %d:", uid);
}
                    break;
                }
                long dualStreamId = 0;
                boolean bCalllBack = true;
                if (null != mUidToVideoIdCallBackInDualStreamMap.get(uid)) {
                    dualStreamId = mUidToVideoIdCallBackInDualStreamMap.get(uid);
                    if (dualStreamId != streamId) {
                        bCalllBack = false;
                    }
                }
                if (bCalllBack) {
                    observer.onVideoDecodeFrame(strUidObserver.getStrUid(), w, h, data, dataLen, renderTimeMs);
                }
            } while (false);
        } catch (Exception e) {
            ThunderLog.error(TAG, "onDecodedFrameData err=%s", e.toString());
        } finally {
            s_observerLock.readLock().unlock();
        }
    }

    @Override
    public void onFirstFrameDecodeNotify(final long streamId, final long pts, final long happenTime) {
        if (mCallBackPtr != 0) {
            onFirstFrameDecodeNotify(mCallBackPtr, streamId, pts, happenTime);

            // 重新link streamId，防止主播断网重连的时候，观众产生黑屏
            if (mThunderVideoHandler != null) {
                final long f_streamId = streamId;
                if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
                    if (mThunderPlayerMultiViewMode.get()) {
                        relinkStreamId(streamId);
                    }
                } else {
                    mThunderVideoHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (mThunderPlayerMultiViewMode.get()) {
                                relinkStreamId(f_streamId);
                            }
                        }
                    });
                }
            }
        }
    }

    @Override
    public void onBehaviorEvent(String name, String val, String oval, int level) {
        ThunderNative.makeBehaviorEvent(name, val, oval, level);
    }

    public void playInit() {
        mThunderVideoPlayListener = new ThunderVideoPlayListener(YYVideoLibMgr.instance().getAppContext(), this);
        YYVideoLibMgr.instance().setVideoInfoListener("Thunder", mThunderVideoPlayListener);
        mVideoPlayHiidoUtil = new ThunderVideoHiidoUtil(YYVideoLibMgr.instance().getAppContext());
        mVideoPlayHiidoUtil.register();
        if (s_observerLock == null) {
            s_observerLock = new ReentrantReadWriteLock();
        }
        mVideoDecodeObserverMap = new ConcurrentHashMap<>();
        mThunderVideoHandlerThread = new HandlerThread("ThunderPlayEngine");
        mThunderVideoHandlerThread.setPriority(MAX_PRIORITY);
        mThunderVideoHandlerThread.start();
        mThunderVideoHandler = new android.os.Handler(mThunderVideoHandlerThread.getLooper());
        mVideoStreamInfoMap = new ConcurrentHashMap<>();
        mMultiVideoScaleModeMap = new ConcurrentHashMap<>();
        mMultiVideoMirrorModeMap = new ConcurrentHashMap<>();
        mMultiStreamKeyIdMap = new ConcurrentHashMap<>();
        mUidToVideoIdCallBackInDualStreamMap = new ConcurrentHashMap<>();
        YMFPlayerUsrBehaviorStat.getInstance().setYMFBehaviorEventListener(this);
        setRemotePlayType(ThunderRtcConstant.ThunderRtcRemotePlayType.THUNDER_REMOTE_PLAY_MULTI);
        ThunderVideoConfig config = new ThunderVideoConfig();
        config.AsyncLoad();
    }

    // 更新视图、收到 subscribe 信息时调用下来

    /**
     * 开始播放流，准备一些视图
     *
     * @param streamKey
     * @param toView
     * @param scaleMode
     * @param encodeType
     * @param bSupportHdwDecode
     * @return
     * @parma mirrorMode
     */
    public int startPlayVideoStream(String streamKey,
                                    Object toView,
                                    int scaleMode,
                                    int mirrorMode,
                                    int encodeType,
                                    boolean bSupportHdwDecode) {
        int result = 0;
        if (streamKey == null || streamKey.isEmpty()) {
            ThunderLog.error(TAG,
                    "ERROR startPlayVideoStream without right key:%s, scaleMode:%d, mirrorMode:%d, bSupportHdwDecode:%b",
                    streamKey,
                    scaleMode,
                    mirrorMode,
                    bSupportHdwDecode);
            return -1;
        }
        ThunderLog.release(TAG,
                "startPlayVideoStream streamKey:" + streamKey
                        + " toView:" + toView
                        + " scaleMode:" + scaleMode
                        + " mirrorMode:" + mirrorMode
                        + " encodeType:" + encodeType
                        + " bSupportHdwDecode:" + bSupportHdwDecode);

        if (mThunderVideoHandler != null) {
            final String f_streamKey = streamKey;
            final Object f_toView = toView;
            final int f_scaleMode = scaleMode;
            final int f_mirrorMode = mirrorMode;
            final int f_encodeType = encodeType;
            final boolean f_bSupportHdwDecode = bSupportHdwDecode;
            if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
                if (!mThunderPlayerMultiViewMode.get()) {
                    startPlayNormalVideoStream(f_streamKey, f_toView, f_scaleMode, f_mirrorMode);
                } else {
                    startPlayMultiVideoStream(f_streamKey,
                            f_toView,
                            f_scaleMode,
                            f_mirrorMode,
                            f_encodeType,
                            f_bSupportHdwDecode);
                }
            } else {
                mThunderVideoHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (!mThunderPlayerMultiViewMode.get()) {
                            startPlayNormalVideoStream(f_streamKey, f_toView, f_scaleMode, f_mirrorMode);
                        } else {
                            startPlayMultiVideoStream(f_streamKey,
                                    f_toView,
                                    f_scaleMode,
                                    f_mirrorMode,
                                    f_encodeType,
                                    f_bSupportHdwDecode);
                        }
                    }
                });
            }
        }
        return 0;
    }

    /**
     * 进行双流流切换
     * 两条流使用同一个view
     *
     * @param dstStreamKey 源流streamKey
     * @param srcStreamKey 目标流streamKey
     * @return
     */
    public boolean switchDualVideoView(String dstStreamKey, String srcStreamKey, int dstEncodeType) {
        if (dstStreamKey == null || dstStreamKey.isEmpty()
                || srcStreamKey == null || srcStreamKey.isEmpty()) {
            ThunderLog.error(TAG,
                    "ERROR switchDualVideoView without right dstkey:%s, srcKey:%s",
                    dstStreamKey,
                    srcStreamKey);
            return false;
        }

if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG,
                "switchDualVideoView dstStreamKey:%s srcStreamKey:%s dstEncodeType:%d",
                dstStreamKey,
                srcStreamKey,
                dstEncodeType);
}

        if (mThunderVideoHandler != null) {
            final String f_srcStreamKey = srcStreamKey;
            final String f_dstStreamKey = dstStreamKey;
            final int f_dstEncodeType = dstEncodeType;
            if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
                switchDualVideoViewUI(f_dstStreamKey, f_srcStreamKey, f_dstEncodeType);
            } else {
                mThunderVideoHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        switchDualVideoViewUI(f_dstStreamKey, f_srcStreamKey, f_dstEncodeType);
                    }
                });
            }
        }
        return true;
    }

    private void stopPlayMultiVideoStream(String streamKey) {
        ThunderPlayerMultiViewProxy proxy = mVideoMultiViewMap.get(streamKey);
        if (proxy != null) {
            proxy.unlink(streamKey);
            mVideoMultiViewMap.remove(streamKey);
            mMultiStreamKeyIdMap.remove(streamKey);
        }
    }

    /**
     * 停止播发流，unlink视图和streamid
     *
     * @param streamKey
     * @return
     */
    public int stopPlayVideoStream(String streamKey) {
        ThunderLog.release(TAG, "stopPlayVideoStream streamKey:" + streamKey);
        final String f_streamKey = streamKey;
        if (mThunderVideoHandler != null) {
            if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
                if (!mThunderPlayerMultiViewMode.get()) {
                    stopPlayNormalVideoStream(f_streamKey);
                } else {
                    stopPlayMultiVideoStream(f_streamKey);
                }
            } else{
                mThunderVideoHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (!mThunderPlayerMultiViewMode.get()) {
                            stopPlayNormalVideoStream(f_streamKey);
                        } else {
                            stopPlayMultiVideoStream(f_streamKey);
                        }
                    }
                });
            }
        }
        return 0;
    }

    /**
     * 获取截图
     * *@param streamKey
     *
     * @return Bitmap
     */
    public Object getPlayVideoScreenshot(String streamKey) {
        if (mThunderVideoHandler != null) {
            if (!mThunderPlayerMultiViewMode.get()) {
                if (mVideoStreamInfoMap.get(streamKey) != null) {
                    return mVideoStreamInfoMap.get(streamKey).getThunderPlayerView().getVideoScreenshot(-1);
                }
            } else {
                if (mVideoMultiViewMap.get(streamKey) != null) {
                    return mVideoMultiViewMap.get(streamKey).getVideoScreenshot(streamKey);
                }
            }
            ThunderLog.error(TAG, "stream %s not exist, cannot get video screenshot", streamKey);
        }
        return null;
    }

    /**
     * 添加解码回调 observer
     *
     * @param uid
     * @param strUid
     * @param observer
     */
    public void addVideoFrameObserver(long uid, String strUid, Object observer) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "addVideoFrameObserver uid: %d, strUid: %s, observer %s", uid, strUid, observer.toString());
}
        try {
            s_observerLock.writeLock().lock();
            if (mVideoDecodeObserverMap.get(uid) == null) {
                mVideoDecodeObserverMap.put(uid, new StrUidVideoDecodeObserver());
            }
            mVideoDecodeObserverMap.get(uid).setStrUid(strUid);
            mVideoDecodeObserverMap.get(uid).setObserver((IVideoDecodeObserver) observer);
        } catch (Exception e) {
            ThunderLog.error(TAG, "addVideoFrameObserver err=%s", e.toString());
        } finally {
            s_observerLock.writeLock().unlock();
        }
    }

    /**
     * 删除解码回调 observer
     *
     * @param uid
     * @param strUid
     */
    public void removeVideoFrameObserver(long uid, String strUid) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "removeVideoFrameObserver uid: %d, strUid %s", uid, strUid);
}
        try {
            s_observerLock.writeLock().lock();
            if (mVideoDecodeObserverMap.get(uid) != null) {
                mVideoDecodeObserverMap.get(uid).removeObserver();
            }
            if (mC2JavaUidMap.containsValue(uid)) {
                Iterator<Long> iter = mC2JavaUidMap.keySet().iterator();
                Long key;
                while (iter.hasNext()) {
                    key = iter.next();
                    if (mC2JavaUidMap.get(key) == uid) {
                        mC2JavaUidMap.remove(key);
                        break;
                    }
                }
            }
            if (null != mUidToVideoIdCallBackInDualStreamMap.get(uid)) {
                mUidToVideoIdCallBackInDualStreamMap.remove(uid);
            }
        } catch (Exception e) {
            ThunderLog.error(TAG, "removeVideoFrameObserver err=%s", e.toString());
        } finally {
            s_observerLock.writeLock().unlock();
        }

    }

    /* 处理场景:用户设置自渲染接口, 之后有进行大小流切换
     * 该接口仅仅处理用户有自渲染接口的情况下,在进行大小流切换过程中由于有两条同一个uid的视频解码
     * 会导致自渲染接口将大小流小数据同时回调给业务，为了避免这个问题，大小流切换的时候从thunder通知
     * 到这里切换大小流过程中只回调源流数据，切过去之后回调目标流数据
     * @param uid
     * @param streamId
     * @param observer pc端用到
     */
    public void addVideoFrameObserverInDualStream(long uid32, long streamId, Object observer) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "addVideoFrameObserverInDualStream uid: %d, streamId: %d", uid32, streamId);
}
        try {
            s_observerLock.writeLock().lock();
            mUidToVideoIdCallBackInDualStreamMap.put(uid32, streamId);
        } catch (Exception e) {
            ThunderLog.error(TAG, "addVideoFrameObserverInDualStream err=%s", e.toString());
        } finally {
            s_observerLock.writeLock().unlock();
        }
    }

    /**
     * 更新视图
     *
     * @param streamKey
     * @param toView
     * @param scaleMode
     * @param mirrorMode
     * @return
     */
    public boolean updatePlayVideoView(String streamKey, Object toView, int scaleMode, int mirrorMode) {
        if (streamKey == null || streamKey.isEmpty()) {
            return false;
        }
        final String f_streamKey = streamKey;
        final Object f_toView = toView;
        final int f_scaleMode = scaleMode;
        final int f_mirrorMode = mirrorMode;
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG,
                "updatePlayVideoView streamKey:" + streamKey
                        + " toView:" + toView
                        + " scaleMode:" + scaleMode
                        + " mirrorMode:" + mirrorMode);
}

        if (mThunderVideoHandler != null) {
            if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
                if (!mThunderPlayerMultiViewMode.get()) {
                    updateNormalPlayVideoView(f_streamKey, f_toView, f_scaleMode, f_mirrorMode);
                } else {
                    if (f_toView == null) {
                        return false;
                    }
                    if (mVideoMultiViewMap.get(f_streamKey) == null) {
                        return false;
                    }
                    if (mVideoMultiViewMap.get(f_streamKey).getVideoPlayerView() != null && !mVideoMultiViewMap.get(f_streamKey).getVideoPlayerView().equals(f_toView)) {
                        ThunderLog.warn(TAG, "may toView is change!");
                        return false;
                    }
                    mMultiVideoMirrorModeMap.put(f_streamKey, f_mirrorMode);
                    mMultiVideoScaleModeMap.put(f_streamKey, f_scaleMode);
                    mVideoMultiViewMap.get(f_streamKey).setMirrorMode(f_streamKey, f_mirrorMode);
                    mVideoMultiViewMap.get(f_streamKey).setScaleMode(f_streamKey, f_scaleMode);
                }
            } else {
                mThunderVideoHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (!mThunderPlayerMultiViewMode.get()) {
                            updateNormalPlayVideoView(f_streamKey, f_toView, f_scaleMode, f_mirrorMode);
                        } else {
                            if (f_toView == null) {
                                return;
                            }
                            if (mVideoMultiViewMap.get(f_streamKey) == null) {
                                return;
                            }
                            if (mVideoMultiViewMap.get(f_streamKey).getVideoPlayerView() != null && !mVideoMultiViewMap.get(f_streamKey).getVideoPlayerView().equals(f_toView)) {
                                ThunderLog.warn(TAG, "may toView is change!");
                                return;
                            }
                            mMultiVideoMirrorModeMap.put(f_streamKey, f_mirrorMode);
                            mMultiVideoScaleModeMap.put(f_streamKey, f_scaleMode);
                            mVideoMultiViewMap.get(f_streamKey).setMirrorMode(f_streamKey, f_mirrorMode);
                            mVideoMultiViewMap.get(f_streamKey).setScaleMode(f_streamKey, f_scaleMode);
                        }
                    }
                });
            }
        }
        return true;
    }

    // 流通知时调用下来

    /**
     * 来流通知，将之前的 streamKey 和 streamId绑定在一起
     *
     * @param streamKey
     * @param streamId
     * @return
     */
    public int onVideoStreamArrive(String streamKey, long streamId, String strUid) {
        VideoDecodeRuntimeInfo.instance().addVideoStream(streamId, strUid);
        ThunderLog.release(TAG, "onVideoStreamArrive:%s-%d strUid:%s", streamKey, streamId, strUid);
        if (mThunderVideoHandler != null) {
            final String f_streamKey = streamKey;
            final long f_streamId = streamId;
            final String f_strUid = strUid;
            if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
                if (!mThunderPlayerMultiViewMode.get()) {
                    mStreamKeyStrUidMap.put(f_streamKey, f_strUid);
                    updatePlayNormalVideoStream(f_streamKey, f_streamId);
                } else {
                    // 记录下streamKey对应的streamId信息
                    mMultiStreamKeyIdMap.put(f_streamKey, f_streamId);
                    mStreamKeyStrUidMap.put(f_streamKey, f_strUid);
                    startPlayVideoStreamArrive(f_streamKey, f_streamId);
                }
            } else {
                mThunderVideoHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (!mThunderPlayerMultiViewMode.get()) {
                            mStreamKeyStrUidMap.put(f_streamKey, f_strUid);
                            updatePlayNormalVideoStream(f_streamKey, f_streamId);
                        } else {
                            // 记录下streamKey对应的streamId信息
                            mMultiStreamKeyIdMap.put(f_streamKey, f_streamId);
                            mStreamKeyStrUidMap.put(f_streamKey, f_strUid);
                            startPlayVideoStreamArrive(f_streamKey, f_streamId);
                        }
                    }
                });
            }
        }
        try {
            s_observerLock.writeLock().lock();
            if (mC2JavaUidMap.get(streamId >> 32) != null) {
                long uid32 = mC2JavaUidMap.get(streamId >> 32);
                if (mVideoDecodeObserverMap.get(uid32) == null) {
                    mVideoDecodeObserverMap.put(uid32, new StrUidVideoDecodeObserver());
                }
                mVideoDecodeObserverMap.get(uid32).setbObserverEnable(true);
            }
        } catch (Exception e) {
            ThunderLog.error(TAG, "onVideoStreamArrive err=%s", e.toString());
        } finally {
            s_observerLock.writeLock().unlock();
        }
        return 0;
    }

    /**
     * 停流通知
     *
     * @param streamKey
     * @param streamId
     * @return
     */
    public int onVideoStreamStop(String streamKey, long streamId) {
        ThunderLog.release(TAG, "onVideoStreamStop streamKey:" + streamKey + " streamId:" + streamId);
        try {
            s_observerLock.readLock().lock();
            if (mC2JavaUidMap.get(streamId >> 32) != null) {
                long uid32 = mC2JavaUidMap.get(streamId >> 32);
                StrUidVideoDecodeObserver observer = mVideoDecodeObserverMap.get(uid32);
                if (observer != null) {
                    observer.setbObserverEnable(false);
                }
                if (null != mUidToVideoIdCallBackInDualStreamMap.get(uid32)) {
                    mUidToVideoIdCallBackInDualStreamMap.remove(uid32);
                }
            }
        } catch (Exception e) {
            ThunderLog.error(TAG, "onVideoStreamStop err=%s", e.toString());
        } finally {
            s_observerLock.readLock().unlock();
        }
        VideoDecodeRuntimeInfo.instance().removeVideoStream(streamId);
        return 0;
    }

    /**
     * 设置ILiveEngineVideoEngine 的 callback, 通过该callback回调数据给LiveEngine层
     *
     * @param ptr
     */
    public void setVideoPlayEngineCallBack(long ptr) {
        mCallBackPtr = ptr;
    }

    /**
     * 获取播放库的版本号
     *
     * @return
     */
    public static String getVideoPlayLibVersion() {
        return BuildConfig.YY_VIDEOPLAYER_VERSION;
    }

    /**
     * 获取hiido数据
     *
     * @param streamId
     * @return
     */
    public String getAudienceHiidoStatInfo(long streamId) {
        String audienceStat = "";
        if (mVideoPlayHiidoUtil != null) {
            audienceStat = mVideoPlayHiidoUtil.getAudienceStatInfo();
        }
        return audienceStat;
    }

    /**
     * 获取播放实时数据
     *
     * @param streamId
     * @param type
     * @return
     */
    public long getPlayRuntimeInfo(long streamId, int type) {
        switch (type) {
            case VIDEO_STAT_FPS:
                return VideoPlayer.getInstance().getPlayerInfo(streamId, VideoPlayer.VideoPlayerInfoEnum.FRAME);
            case VIDEO_STAT_BITRATE:
                return VideoPlayer.getInstance().getPlayerInfo(streamId, VideoPlayer.VideoPlayerInfoEnum.BITRATE);
            case VIDEO_STATE_RESOLUTION:
                return VideoPlayer.getInstance().getPlayerInfo(streamId, VideoPlayer.VideoPlayerInfoEnum.RESOLUTION);
            case VIDEO_STATE_DECODER_TYPE:
                return VideoPlayer.getInstance().getPlayerInfo(streamId, VideoPlayer.VideoPlayerInfoEnum.DECODERTYPE);
        }
        return 0;
    }

    /**
     * @param uid
     * @return
     */
    public boolean queryOnlyDecoded(long streamId, long uid) {
        boolean result = false;
        try {
            s_observerLock.readLock().lock();
            mC2JavaUidMap.put((streamId >> 32), uid);
            StrUidVideoDecodeObserver strUidObserver = null;
            IVideoDecodeObserver observer = null;
            if (mVideoDecodeObserverMap != null && !mVideoDecodeObserverMap.isEmpty()) {
                strUidObserver = mVideoDecodeObserverMap.get(uid);
                if (strUidObserver != null) {
                    observer = strUidObserver.getObserver();
                }
            }
            if (strUidObserver == null || observer == null) {
                result = false;
            } else {
                result = true;
            }
        } catch (Exception e) {
            ThunderLog.error(TAG, "queryOnlyDecoded err=%s", e.toString());
        } finally {
            s_observerLock.readLock().unlock();
        }
        return result;
    }

    /**
     * 初始化多人连麦视图
     *
     * @param params
     */
    public void initMultiPlayerViewLayout(Object params) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "initMultiPlayerViewLayout params:" + params);
}
        if (mThunderVideoHandler != null && params != null) {
            final ThunderMultiVideoViewParam f_params = (ThunderMultiVideoViewParam) params;
            final ThunderVideoPlayEngineImp engine = this;
            if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
                initMultiLayout(f_params, engine);
            } else {
                mThunderVideoHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        initMultiLayout(f_params, engine);
                    }
                });
            }
        }
    }

    private void initMultiLayout(ThunderMultiVideoViewParam layoutParam, ThunderVideoPlayEngineImp engine) {
        if (layoutParam == null) {
            ThunderLog.error(TAG, "initMultiPlayerViewLayout params null");
            return;
        }
        if (mThunderPlayerMultiViewMode.get()) {
            int viewId = layoutParam.mViewId;
            Object multiView = layoutParam.mView;
            ThunderLog.error(TAG, "initMultiPlayerViewLayout params:" + layoutParam + " " +
                    " mView:" + multiView + " viewId:" + viewId);
            if (viewId >= 0) {
                ThunderPlayerMultiViewProxy proxy = null;
                if (multiView != null &&
                        checkMultiViewExist(multiView, viewId)) {
                    ThunderLog.error(TAG, "repeated view, already exist，" + multiView);

                    int oldViewId = getMultiViewId(multiView);
                    if (oldViewId < 0 && oldViewId != ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR) {
                        mThunderPlayerMultiViewProxyMap.put(viewId, mThunderPlayerMultiViewProxyMap.remove(oldViewId));
                    }

                    if (mThunderPlayerMultiViewProxyMap.containsKey(viewId)) {
                        mThunderPlayerMultiViewProxyMap.get(viewId).initMultiPlayerViewLayout(layoutParam);
                    }
                    return;
                }
                if (mThunderPlayerMultiViewProxyMap.get(viewId) == null) {
                    proxy = new ThunderPlayerMultiViewProxy(YYVideoLibMgr.instance().getAppContext(), engine);
                    mThunderPlayerMultiViewProxyMap.put(viewId, proxy);
                }

                // init layout param and layout
                // init layout param and layout
                mThunderPlayerMultiViewProxyMap.get(viewId).initMultiPlayerViewLayout(layoutParam);
                mThunderPlayerMultiViewProxyMap.get(viewId).updateMultiPlayerView(multiView);
            }
            if (viewId < 0) {
                ThunderPlayerMultiViewProxy proxy = null;
                if (mThunderPlayerMultiViewProxyMap.get(viewId) == null) {
                    proxy = new ThunderPlayerMultiViewProxy(YYVideoLibMgr.instance().getAppContext(), engine);
                    mThunderPlayerMultiViewProxyMap.put(viewId, proxy);
                }
                // init layout param and layout
                mThunderPlayerMultiViewProxyMap.get(viewId).initMultiPlayerViewLayout(layoutParam);
            }
            if(multiView == null && viewId >= 0 && mThunderPlayerMultiViewProxyMap.containsKey(viewId)){
                mThunderPlayerMultiViewProxyMap.remove(viewId);
            }
        }
    }

    /**
     * 退频道
     */
    public void onPlayEngineLeaveRoom() {
        // TODO: 2020/10/26
        if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
            for (Map.Entry<Integer, ThunderPlayerMultiViewProxy> entry : mThunderPlayerMultiViewProxyMap.entrySet()) {
                ThunderPlayerMultiViewProxy proxy = entry.getValue();
                if (proxy != null) {
                    proxy.quitRoom();
                }
            }
            // mThunderPlayerMultiViewProxyMap.clear();
            mVideoMultiViewMap.clear();
            mVideoMultiViewRecoverMap.clear();
            mStreamKeyStrUidMap.clear();
        } else {
            mThunderVideoHandler.post(new Runnable() {
                @Override
                public void run() {
                    for (Map.Entry<Integer, ThunderPlayerMultiViewProxy> entry : mThunderPlayerMultiViewProxyMap.entrySet()) {
                        ThunderPlayerMultiViewProxy proxy = entry.getValue();
                        if (proxy != null) {
                            proxy.quitRoom();
                        }
                    }
                    // mThunderPlayerMultiViewProxyMap.clear();
                    mVideoMultiViewMap.clear();
                    mVideoMultiViewRecoverMap.clear();
                    mStreamKeyStrUidMap.clear();
                }
            });
        }

    }

    /**
     * 设置当前播放模式
     *
     * @param playType 多人连麦 or 普通观看
     */
    public void setVideoPlayType(int playType) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "setVideoPlayType playType:" + playType);
}
        final int f_playType = playType;
        if (mThunderVideoHandler != null) {
            if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
                setRemotePlayType(f_playType);
            } else {
                mThunderVideoHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        setRemotePlayType(f_playType);
                    }
                });
            }
        }
    }

    private void setRemotePlayType(int playType) {
        if (playType == ThunderRtcConstant.ThunderRtcRemotePlayType.THUNDER_REMOTE_PLAY_MULTI) {
if (ThunderLog.isInfoValid()) {
            ThunderLog.info(TAG, "multiLianMai mode " + playType);
}
            mThunderPlayerMultiViewMode.set(true);
        } else {
if (ThunderLog.isInfoValid()) {
            ThunderLog.info(TAG, "non-multiLianMai mode " + playType);
}
            clearMultiView();
            mThunderPlayerMultiViewProxyMap.clear();
            mVideoMultiViewMap.clear();
            mVideoMultiViewRecoverMap.clear();
            mStreamKeyStrUidMap.clear();
            mThunderPlayerMultiViewMode.set(false);
        }
    }

    private void initDefaultMultiViewParam(int viewId) {
        ThunderMultiVideoViewParam param = new ThunderMultiVideoViewParam();
        ThunderMultiVideoViewCoordinate coordinate = new ThunderMultiVideoViewCoordinate();
        coordinate.mIndex = 0;
        coordinate.mHeight = -2;
        coordinate.mWidth = -2;
        coordinate.mX = 0;
        coordinate.mY = 0;
        param.mVideoViewPositions = new ArrayList<ThunderMultiVideoViewCoordinate>(1);
        param.mVideoViewPositions.add(coordinate);
        param.mViewId = viewId;
        initMultiPlayerViewLayout(param);
    }

    private boolean updateSeat(String streamKey, Object toView, int seat) {
        boolean result = false;
        Object view = toView;
        if (view == null) {
            ThunderLog.error(TAG, "updateUserSeat toView null");
            return false;
        }
        if (mVideoMultiViewMap.get(streamKey) != null && findMultiViewProxy(view) != null) {
            result = (mVideoMultiViewMap.get(streamKey).updateSeat(streamKey, seat) >= 0 ? true : false);
            if (!result) {
                return false;
            }
            // 设置裁剪模式，从setRemoteVideoCanvas带进来的scaleMode
            int scaleMode = mMultiVideoScaleModeMap.get(streamKey) == null ? -1 : mMultiVideoScaleModeMap.get(streamKey);
            if (scaleMode != -1) {
                mVideoMultiViewMap.get(streamKey).setScaleMode(streamKey, scaleMode);
            }
            // 设置镜像模式
            int mirrorMode = mMultiVideoMirrorModeMap.get(streamKey) == null ? -1 : mMultiVideoMirrorModeMap.get(streamKey);
            if (mirrorMode != -1) {
                mVideoMultiViewMap.get(streamKey).setMirrorMode(streamKey, mirrorMode);
            }
        }
        return true;
    }

    /**
     * 多人连麦：更新用户坐席
     *
     * @param streamKey
     * @param seat
     * @return
     */
    public boolean updateUserSeat(String streamKey, Object toView, int seat) {
        ThunderLog.release(TAG,
                "updateUserSeat streamKey:" + streamKey + " toView:" + toView + " seat:" + seat);
        if (mThunderVideoHandler != null) {
            final String f_streamKey = streamKey;
            final Object f_toView = toView;
            final int f_seat = seat;
            if (Thread.currentThread().getId() == mThunderVideoHandler.getLooper().getThread().getId()) {
                boolean result = false;
                if (mThunderPlayerMultiViewMode.get()) {
                    updateSeat(f_streamKey, f_toView, f_seat);
                }
            } else {
                mThunderVideoHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        updateSeat(f_streamKey, f_toView, f_seat);
                    }
                });
            }
        }
        return true;
    }

    /**
     * 多人连麦：查找frameLayout对应生成的proxy
     *
     * @param view 业务设下了的frameLayout
     * @return
     */
    public ThunderPlayerMultiViewProxy findMultiViewProxy(Object view) {
        for (Map.Entry<Integer, ThunderPlayerMultiViewProxy> entry : mThunderPlayerMultiViewProxyMap.entrySet()) {
            ThunderPlayerMultiViewProxy proxy = entry.getValue();
            if (proxy.getVideoPlayerView() == view) {
                return proxy;
            }
        }
        return null;
    }

    /**
     * 多人连麦：查找是否已经存在相同的frameLayout，不允许不同viewId同时存在相同的的frameLayout
     *
     * @param view   业务设下了的frameLayout
     * @param viewId 业务设下了的frameLayout绑定的viewId
     * @return
     */
    public boolean checkMultiViewExist(Object view, int viewId) {
        if (view == null || viewId < 0) {
            return false;
        }
        for (Map.Entry<Integer, ThunderPlayerMultiViewProxy> entry : mThunderPlayerMultiViewProxyMap.entrySet()) {
            ThunderPlayerMultiViewProxy proxy = entry.getValue();
            if (proxy.getVideoPlayerView() == view) {
                if (entry.getKey() != viewId) {
                    ThunderLog.error(TAG, "view " + view + "already locate in viewId:" + entry.getKey());
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 多人连麦： 获取已存在的view的viewId
     *
     * @param view 业务设下了的frameLayout
     * @return viewId
     */
    private int getMultiViewId(Object view) {
        if (view == null) {
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }

        for (Map.Entry<Integer, ThunderPlayerMultiViewProxy> entry : mThunderPlayerMultiViewProxyMap.entrySet()) {
            ThunderPlayerMultiViewProxy proxy = entry.getValue();
            if (proxy.getVideoPlayerView() == view) {
                return entry.getKey();
            }
        }

        return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
    }

    /**
     * 多人连麦：清理
     *
     * @param
     * @return
     */
    public void clearMultiView() {
        for (Map.Entry<Integer, ThunderPlayerMultiViewProxy> entry : mThunderPlayerMultiViewProxyMap.entrySet()) {
            ThunderPlayerMultiViewProxy proxy = entry.getValue();
            if (proxy != null) {
                proxy.destroyVideoPlayerView();
            }
        }
    }

    public void destroyPlayEngine() {
        YMFPlayerUsrBehaviorStat.getInstance().setYMFBehaviorEventListener(null);
        if (mVideoPlayHiidoUtil != null) {
            mVideoPlayHiidoUtil.unRegister();
        }
        try {
            if (mThunderVideoHandler != null) {
                if (Build.VERSION.SDK_INT >= 18) {
                    mThunderVideoHandlerThread.quitSafely();
                } else {
                    mThunderVideoHandlerThread.quit();
                }
                mThunderVideoHandler = null;
                mThunderVideoHandlerThread = null;
            }

            s_observerLock.writeLock().lock();
            mVideoDecodeObserverMap.clear();
            mUidToVideoIdCallBackInDualStreamMap.clear();
            clearMultiView();
            mThunderPlayerMultiViewProxyMap.clear();
        } catch (Exception e) {
            ThunderLog.error(TAG, "destroyPlayEngine err=%s", e.toString());
        } finally {
            s_observerLock.writeLock().unlock();
        }
    }

    /**
     * 更新普通模式播放视图
     *
     * @param streamKey
     * @param toView
     * @param scaleMode
     * @param mirrorMode
     * @return
     */
    private boolean updateNormalPlayVideoView(String streamKey, Object toView, int scaleMode, int mirrorMode) {
        VideoStreamInfo info = mVideoStreamInfoMap.get(streamKey);
        if (info == null) {
            ThunderLog.error(TAG,
                    "updatePlayVideoView unexist info key:" + streamKey);
            return false;
        }
        if (toView != null && !(toView instanceof ThunderPlayerView)) {
            ThunderLog.error(TAG,
                    "updateNormalPlayVideoView toView error:" + toView);
            return false;
        }

        long streamId = info.getStreamId();
        ThunderPlayerView newView = (ThunderPlayerView) toView;
        ThunderPlayerView oldView = info.getThunderPlayerView();

        ThunderLog.release(TAG,
                "updateNormalPlayVideoView key " + streamKey
                        + " toView " + toView
                        + " oldView " + oldView
                        + " scaleMode " + scaleMode
                        + " mirrorMode " + mirrorMode);

        if (oldView == null && null == newView) {
            return false;
        } else if (oldView == null && null != newView) {
            info.setThunderPlayerView(newView);
            newView.unLinkFromStream();
            if (INVALID_VALUE != streamId) {
                newView.linkToStream(streamId);
                newView.setScaleMode(scaleMode);
                newView.setMirrorMode(-1, mirrorMode);
                info.setScaleMode(scaleMode);
                info.setMirrorMode(mirrorMode);
            }
            return true;
        } else if (oldView != null && null == newView) {
            if (INVALID_VALUE != streamId) {
                oldView.unLinkFromStream();
                oldView.drawBlackScreen(true);
            }
            info.setThunderPlayerView(null);
            return true;
        }

        if ((toView instanceof ThunderPlayerView) && !oldView.equals(toView)) {
            ThunderLog.warn(TAG, "may toView is change!");
            return false;
        }

        info.setScaleMode(scaleMode);
        info.setMirrorMode(mirrorMode);
        oldView.setScaleMode(scaleMode);
        oldView.setMirrorMode(-1, mirrorMode);

        return true;
    }

    /**
     * 普通开播订阅
     *
     * @param streamKey
     * @param toView
     * @param scaleMode
     * @param mirrorMode
     * @return
     */
    private int startPlayNormalVideoStream(String streamKey,
                                           Object toView,
                                           int scaleMode,
                                           int mirrorMode) {
        ThunderPlayerView existedView = null;
        ThunderLog.release(TAG,
                "startPlayNormalVideoStream key:%s, scaleMode:%d, mirrorMode:%d", streamKey, scaleMode, mirrorMode);
        checkStartPlayDualStream(streamKey);
        VideoStreamInfo streamInfo = getStreamInfo(streamKey);
        existedView = streamInfo.getThunderPlayerView();
        if (toView != null && (toView instanceof ThunderPlayerView)) {
            ThunderPlayerView view = (ThunderPlayerView) toView;

            // 之前streamKey存在视图的话，可能需要切换视图
            if (existedView != null) {
                if (!view.equals(existedView)) {
                    // 需要更换视图，要重新prepare视图
                    existedView.unLinkFromStream();
                    existedView.drawBlackScreen(true);
                    ThunderLog.release(TAG,
                            "startPlayNormalVideoStream changed view"
                                    + " old:" + existedView
                                    + " new:" + view);
                } else {
if (ThunderLog.isInfoValid()) {
                    ThunderLog.info(TAG,
                            "startPlayNormalVideoStream the same view"
                                    + " old:" + existedView
                                    + " new:" + view);
}
                }
            }
            view.setScaleMode(scaleMode);
            view.setMirrorMode(-1, mirrorMode);
            streamInfo.setThunderPlayerView(view);
        }

        existedView = streamInfo.getThunderPlayerView();
        if (existedView != null) {
            // 判断是否视图是否已经准备好，如果没有则开始准备视图
            if (!streamInfo.isViewPrepared()) {
                existedView.unLinkFromStream();
                if (streamInfo.getStreamId() != INVALID_VALUE) {
                    existedView.linkToStream(streamInfo.getStreamId());
                }
            }

            // 准备好视图后，尝试link
            if (!streamInfo.isLinkedToStream()) {
                if (streamInfo.getStreamId() != INVALID_VALUE) {
                    existedView.linkToStream(streamInfo.getStreamId());
                    ThunderLog.release(TAG,
                            "startPlayNormalVideoStream linkToStream %d",
                            streamInfo.getStreamId());
                }
            } else {
                ThunderLog.warn(TAG,
                        "startPlayNormalVideoStream has linkedToStream %d",
                        streamInfo.getStreamId());
            }
        } else {
            ThunderLog.warn(TAG,
                    "ERROR startPlayNormalVideoStream view is null");
        }

        streamInfo.setScaleMode(scaleMode);
        streamInfo.setMirrorMode(mirrorMode);

        return 0;
    }

    /**
     * 普通模式停止订阅视频流
     *
     * @param streamKey
     * @return
     */
    private int stopPlayNormalVideoStream(String streamKey) {
        VideoStreamInfo info = mVideoStreamInfoMap.get(streamKey);
        ThunderPlayerView view = null;
        if (info == null) {
            ThunderLog.release(TAG,
                    "ERROR stopPlayNormalVideoStream unexsited key:%s"
                    , streamKey);
            return -1;
        }
        ThunderLog.release(TAG, "stopPlayNormalVideoStream %s", streamKey);

        view = info.getThunderPlayerView();
        if (view != null) {
            view.unLinkFromStream();
            view.drawBlackScreen(true);
        } else {
            ThunderLog.release(TAG,
                    "ERROR stopPlayNormalVideoStream unexsited view");
        }
        mVideoStreamInfoMap.remove(streamKey);
        return 0;
    }

    private int onNormalVideoStreamStop(String streamKey, long streamId) {
        VideoStreamInfo info = mVideoStreamInfoMap.get(streamKey);
        if (info == null) {
            ThunderLog.release(TAG, "onVideoStreamStop key:%s unexisted",
                    streamKey);
            return -1;
        }
        ThunderPlayerView view = info.getThunderPlayerView();
        if (view == null) {
            ThunderLog.warn(TAG,
                    "onVideoStreamStop: cann't find video view for stream:%s",
                    streamKey);
            return -1;
        }
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "unlink stream:%s-%d", streamKey, streamId);
}

        view.unLinkFromStream(streamId);
        //view.drawBlackScreen(true);
        return 0;
    }

    /**
     * 多人连麦
     *
     * @param streamKey
     * @param toView
     * @param scaleMode
     * @param mirrorMode
     * @param encodeType
     * @param bSupportHdwDecode
     * @return
     */
    private int startPlayMultiVideoStream(String streamKey,
                                          Object toView,
                                          int scaleMode,
                                          int mirrorMode,
                                          int encodeType,
                                          boolean bSupportHdwDecode) {
        Object view = toView;
        if (view != null) {
            ThunderPlayerMultiViewProxy proxy = findMultiViewProxy(view);
            if (proxy != null) {
                if (mVideoMultiViewMap.get(streamKey) != null) {
                    // 触发跨view操作，要取消老view的对应流的绑定操作
                    ThunderPlayerMultiViewProxy old = mVideoMultiViewMap.get(streamKey);
                    if (old != null && old != proxy) {
                        old.unlink(streamKey);
                    }
                    // 备份上一次
                    mVideoMultiViewRecoverMap.put(streamKey, old);
                }
                // 更新新的映射关系
                mVideoMultiViewMap.put(streamKey, proxy);
                checkStartPlayDualStream(streamKey);
            } else {
                ThunderLog.warn(TAG, "initMultiPlayerViewLayout not called " + toView);
                ThunderPlayerMultiViewProxy old = mVideoMultiViewMap.get(streamKey);
                if (old != null) {
                    old.unlink(streamKey);
                }
                int viewId = getDefaultViewId();
                initDefaultMultiViewParam(viewId);
                // 业务没有标明view的情况，默认创建一个ThunderPlayerMultiViewProxy，用于向上兼容
                ThunderPlayerMultiViewProxy defaultProxy =
                        mThunderPlayerMultiViewProxyMap.get(viewId);
                if (defaultProxy != null) {
                    defaultProxy.updateMultiPlayerView(view);
                }
                mVideoMultiViewMap.put(streamKey, defaultProxy);
                checkStartPlayDualStream(streamKey);
            }
        }

        if (mVideoMultiViewMap.get(streamKey) != null &&
                mVideoMultiViewMap.get(streamKey).getVideoPlayerView() != null &&
                view == null) {
            mVideoMultiViewMap.get(streamKey).unlink(streamKey);
            ThunderLog.release(TAG, "startPlayMultiVideoStream unlink key" + streamKey + " view " + view);
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_SUCCESS;
        }

        // 更新新的映射关系
        mMultiVideoScaleModeMap.put(streamKey, scaleMode);
        mMultiVideoMirrorModeMap.put(streamKey, mirrorMode);
        long streamId = mMultiStreamKeyIdMap.get(streamKey) == null ? -1 : mMultiStreamKeyIdMap.get(streamKey);
        updatePlayMultiVideoStream(streamKey, streamId);
        ThunderLog.release(TAG,
                "startPlayMultiVideoStream key:%s, streamType:%d, scaleMode:%d, mirrorMode:%d, bSupportHdwDecode:%b",
                streamKey,
                encodeType,
                scaleMode,
                mirrorMode,
                bSupportHdwDecode);
        return 0;
    }

    /**
     * 获取默认的视图id，从-1递减
     */
    private int getDefaultViewId() {
        int viewId = -1;

        while (viewId > Integer.MIN_VALUE) {
            if (mThunderPlayerMultiViewProxyMap.containsKey(viewId)) {
                viewId--;
            } else {
                break;
            }
        }

        return viewId;
    }

    /**
     * 获取 streamKey 对应的 VideoStreamInfo 对象
     *
     * @param streamKey
     * @return
     */
    private VideoStreamInfo getStreamInfo(String streamKey) {
        VideoStreamInfo streamInfo = null;
        boolean isExistedKey = mVideoStreamInfoMap.containsKey(streamKey);
        if (!isExistedKey) {
            streamInfo = new VideoStreamInfo();
            mVideoStreamInfoMap.put(streamKey, streamInfo);
        } else {
            streamInfo = mVideoStreamInfoMap.get(streamKey);
        }
        return streamInfo;
    }

    private int updatePlayMultiVideoStream(String streamKey, long streamId) {
        ThunderLog.release(TAG,
                "updatePlayVideoStream streamKey:" + streamKey + " streamId " + streamId);

        if (streamKey == null) {
            ThunderLog.error(TAG, "updatePlayMultiVideoStream, streamKey null");
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }

        // bindStreamId 建立proxy内部streamKey和streamId的关系
        if (mVideoMultiViewMap.get(streamKey) != null) {
            ThunderLog.release(TAG, "updatePlayVideoStream, bindStreamToSeat streamKey "
                    + streamKey + " streamId " + streamId);
            mVideoMultiViewMap.get(streamKey).prepareStreamSeat(streamKey, streamId);
        }
        return 0;
    }

    private int startPlayVideoStreamArrive(String streamKey, long streamId) {
        ThunderLog.release(TAG,
                "startPlayVideoStreamArrive streamKey:" + streamKey + " streamId " + streamId);

        if (streamKey == null) {
            ThunderLog.error(TAG, "updatePlayMultiVideoStream, streamKey null");
            return ThunderRtcConstant.ThunderRet.THUNDER_RET_VIDEO_ENGINE_ERROR;
        }

        // streamKey和不同的多人view的proxy关系还没映射好，需要在startPlay的时候映射
        // bindStreamId 建立proxy内部streamKey和streamId的关系
        if (mVideoMultiViewMap.get(streamKey) != null) {
            ThunderLog.release(TAG, "startPlayVideoStreamArrive, bindStreamToSeat streamKey "
                    + streamKey + " streamId " + streamId);
            mVideoMultiViewMap.get(streamKey).bindStreamToSeat(streamKey, streamId, false);
        }
        return 0;
    }

    private int relinkStreamId(long streamId) {
        ThunderLog.release(TAG, "relinkStreamId streamId " + streamId);
        Iterator<Map.Entry<String, Long>> entryIterator = mMultiStreamKeyIdMap.entrySet().iterator();
        while (entryIterator.hasNext()) {
            Map.Entry<String, Long> next = entryIterator.next();
            if (0 == next.getValue().compareTo(streamId)) {
                String streamKey = next.getKey();
                ThunderLog.release(TAG, "relinkStreamId found streamKey" + streamKey + " streamId: " + streamId);
                if (mVideoMultiViewMap.get(streamKey) != null) {
                    ThunderLog.release(TAG, "relinkStreamId, bindStreamToSeat streamKey " + streamKey + " streamId " + streamId);
                    mVideoMultiViewMap.get(streamKey).bindStreamToSeat(streamKey, streamId, true);
                }
                break;
            }
        }
        return 0;
    }

    private int updatePlayNormalVideoStream(String streamKey, long streamId) {
        VideoStreamInfo streamInfo = getStreamInfo(streamKey);
        ThunderPlayerView view = streamInfo.getThunderPlayerView();

        if (view != null) {
            // 更新 streamid
            long oldStreamId = streamInfo.getStreamId();
            if (oldStreamId != INVALID_VALUE && oldStreamId != streamId) {
                if (streamInfo.isLinkedToStream()) {
                    view.unLinkFromStream(oldStreamId);
                }
            }

            // 视图不为空时且未prepare时，准备视图
            if (!streamInfo.isViewPrepared()) {
                view.unLinkFromStream();
            }

            // link streamId to view
            if (!streamInfo.isLinkedToStream()) {
                view.linkToStream(streamId);
            }

            // 设置scaleMode，这里直接设置，不用判断，否则会导致 view、streamId 对应的 mode 设置不进去
            view.setScaleMode(streamInfo.getScaleMode());
            view.setMirrorMode(-1, streamInfo.getMirrorMode());
            view.drawBlackScreen(false);
        } else {
            ThunderLog.warn(TAG,
                    "updatePlayNormalVideoStream: cannot find video view for stream:%s",
                    streamKey);
        }

        // 保存streamid
        streamInfo.setStreamId(streamId);

if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG,
                "updatePlayNormalVideoStream key:%s,id:%d,isPrepared:%b,isLinked:%b",
                streamKey,
                streamId,
                streamInfo.isViewPrepared(),
                streamInfo.isLinkedToStream());
}
        return 0;
    }

    private boolean switchDualVideoViewUI(String dstStreamKey, String srcStreamKey, int dstEncodeType) {
        if (!mThunderPlayerMultiViewMode.get()) {
            VideoStreamInfo srcStreamInfo = getStreamInfo(srcStreamKey);
            VideoStreamInfo dstStreamInfo = getStreamInfo(dstStreamKey);
            dstStreamInfo.setEncodeType(dstEncodeType);
            dstStreamInfo.setScaleMode(srcStreamInfo.getScaleMode());
            dstStreamInfo.setMirrorMode(srcStreamInfo.getMirrorMode());

            ThunderPlayerView srcStreamView = srcStreamInfo.getThunderPlayerView();
            if (srcStreamView != null) {
                srcStreamView.unLinkFromStream();
                // 大小流切换使用的是同样的view
                dstStreamInfo.setThunderPlayerView(srcStreamView);
            }

            ThunderPlayerView dstStreamView = dstStreamInfo.getThunderPlayerView();
            if (dstStreamView != null) {
                // 如果视图没有准备好则需要准备视图
                if (!dstStreamView.isViewPrepared()) {
                    dstStreamView.unLinkFromStream();
                    if (dstStreamInfo.getStreamId() != INVALID_VALUE) {
                        dstStreamView.linkToStream(dstStreamInfo.getStreamId());
                    }
                    ThunderLog.release(TAG,
                            "switchDualVideoViewUI dstStreamView unPrepared, dstkey:%s, streamId %d",
                            dstStreamKey, dstStreamInfo.getStreamId());
                }

                if (!dstStreamInfo.isLinkedToStream()) {
                    if (dstStreamInfo.getStreamId() != INVALID_VALUE) {
                        dstStreamView.linkToStream(dstStreamInfo.getStreamId());
                        ThunderLog.release(TAG,
                                "switchDualVideoViewUI linkToStream dstkey:%s, streamId %d",
                                dstStreamKey, dstStreamInfo.getStreamId());
                    }
                } else {
                    ThunderLog.release(TAG,
                            "switchDualVideoViewUI has linkToStream dstkey:%s, streamId %d",
                            dstStreamKey, dstStreamInfo.getStreamId());
                }
                mVideoStreamInfoMap.remove(srcStreamKey);
            } else {
                ThunderLog.release(TAG,
                        "switchDualVideoViewUI normal view is null dstKey:%s, srcKey:%s",
                        dstStreamKey,
                        srcStreamKey);
            }
        } else {
            //srcProxy 是同一个因为view是一样的
            ThunderPlayerMultiViewProxy srcProxy = mVideoMultiViewMap.get(srcStreamKey);
            long dstStreamId = (mMultiStreamKeyIdMap.get(dstStreamKey) == null) ? -1 : mMultiStreamKeyIdMap.get(dstStreamKey);

            if (srcProxy != null){
                if (srcProxy.switchDualVideoView(srcStreamKey, dstStreamKey, dstStreamId) >= 0) {
                    mVideoMultiViewMap.remove(srcStreamKey);
                    mVideoMultiViewMap.put(dstStreamKey, srcProxy);
                }
            }

            ThunderLog.release(TAG,
                    "switchDualVideoViewUI multiView linkToStream dstkey:%s, srcKey:%s dstStreamId %d",
                    dstStreamKey, srcStreamKey, dstStreamId);

        }
        return true;
    }

    /**
     * 通过主播停流，设置默认订阅类型进行大小流切换，然后再推流方式
     * 而不是直接修改订阅订阅大小流进行切换
     * 需要移除切换前的流和proxy的动态绑定关系
     * @param
     * @return
     */
    private void checkStartPlayDualStream(String curStreamKey) {
        if (curStreamKey == null) {
            return;
        }
        if (!mStreamKeyStrUidMap.containsKey(curStreamKey)) {
            return;
        }
        String tmpStreamKey = null;
        String curUid = mStreamKeyStrUidMap.get(curStreamKey);
        if (curUid != null && mStreamKeyStrUidMap != null && mStreamKeyStrUidMap.size() > 0) {
            for (Map.Entry<String, String> entry : mStreamKeyStrUidMap.entrySet()) {
                String strUid = entry.getValue();
                if (curUid.equals(strUid)) {
                    if (!curStreamKey.equals(entry.getKey())) {
                        tmpStreamKey = entry.getKey();
                        ThunderLog.warn(TAG, "find dualStream duplicate streamKey:" + tmpStreamKey + " cur:" + curStreamKey
                                + " uid:" + strUid);
                    }
                }
            }
        }
        if (tmpStreamKey != null && !tmpStreamKey.isEmpty()) {
            // 多人情况
            if (mThunderPlayerMultiViewMode.get() && mVideoMultiViewMap.containsKey(tmpStreamKey)) {
                mVideoMultiViewMap.get(tmpStreamKey).unlink(tmpStreamKey);
                mVideoMultiViewMap.remove(tmpStreamKey);
                ThunderLog.warn(TAG, "remove dup streamKey:" + tmpStreamKey);
            }
            // 单人情况
            if (!mThunderPlayerMultiViewMode.get() && mVideoStreamInfoMap.containsKey(tmpStreamKey)) {
                if (mVideoStreamInfoMap.get(tmpStreamKey).getThunderPlayerView() != null &&
                        mVideoStreamInfoMap.get(tmpStreamKey).isLinkedToStream()) {
                    mVideoStreamInfoMap.get(tmpStreamKey).getThunderPlayerView().unLinkFromStream();
                }
                mVideoStreamInfoMap.remove(tmpStreamKey);
                ThunderLog.warn(TAG, "remove dup streamKey:" + tmpStreamKey);
            }
        }
    }

    /**
     * strUid 对应的 IVideoDecodeObserver ，用于通知给用户当前 decode
     */
    public class StrUidVideoDecodeObserver {
        private String strUid;
        private IVideoDecodeObserver observer;
        private boolean bObserverEnable = false;

        public StrUidVideoDecodeObserver() {
        }

        public void setStrUid(String strUid) {
            this.strUid = strUid;
        }

        public String getStrUid() {
            return strUid;
        }

        public void setObserver(IVideoDecodeObserver observer) {
            this.observer = observer;
        }

        public IVideoDecodeObserver getObserver() {
            return observer;
        }

        public void removeObserver() {
            this.observer = null;
        }

        public void setbObserverEnable(boolean bObserverEnable) {
            this.bObserverEnable = bObserverEnable;
        }

        public boolean isbObserverEnable() {
            return bObserverEnable;
        }
    }

    // 定义一个结构体存储streamKey对应的属性，仅用于普通view
    private class VideoStreamInfo {
        private long mStreamId;
        private ThunderPlayerView mThunderPlayerView;
        private int mIsSoftDecodeFlag;
        private int mEncodeType;
        private int mScaleMode;
        private int mMirrorMode;

        public VideoStreamInfo() {
            this.mStreamId = INVALID_VALUE;
            this.mThunderPlayerView = null;
            this.mIsSoftDecodeFlag = INVALID_VALUE;
            this.mEncodeType = ThunderRtcConstant.ThunderVideoEncodeType.THUNDERVIDEO_ENCODE_TYPE_H264;
            this.mScaleMode = INVALID_VALUE;
            ThunderLog.release(TAG, "VideoStreamInfo: " + this + " construct streamId:%d",
                    this.mStreamId);
        }

        public long getStreamId() {
            return mStreamId;
        }

        public void setStreamId(long streamId) {
            mStreamId = streamId;
            ThunderLog.release(TAG, "VideoStreamInfo setStreamId " + streamId);
        }

        public ThunderPlayerView getThunderPlayerView() {
            return mThunderPlayerView;
        }

        public void setThunderPlayerView(ThunderPlayerView thunderPlayerView) {
            mThunderPlayerView = thunderPlayerView;
            if (thunderPlayerView != null) {
                ThunderLog.release(TAG,
                        "VideoStreamInfo setThunderPlayerView streamId:" + this.mStreamId
                                + " view:" + thunderPlayerView);
            } else {
                ThunderLog.error(TAG, "VideoStreamInfo setThunderPlayerView null");
            }
        }

        public int getIsSoftDecodeFlag() {
            return mIsSoftDecodeFlag;
        }

        public void setIsSoftDecodeFlag(int isSoftDecodeFlag) {
            mIsSoftDecodeFlag = isSoftDecodeFlag;
            ThunderLog.release(TAG,
                    "VideoStreamInfo streamId:" + this.mStreamId
                            + " setIsSoftDecodeFlag " + isSoftDecodeFlag);
        }

        public int getEncodeType() {
            return mEncodeType;
        }

        public void setEncodeType(int encodeType) {
            mEncodeType = encodeType;
            ThunderLog.release(TAG,
                    "VideoStreamInfo streamId:" + this.mStreamId
                            + " setEncodeType " + mEncodeType);
        }

        public int getScaleMode() {
            return mScaleMode;
        }

        public void setScaleMode(int scaleMode) {
            mScaleMode = scaleMode;
            ThunderLog.release(TAG,
                    "VideoStreamInfo streamId:" + this.mStreamId
                            + " setScaleMode " + scaleMode);
        }

        public void setMirrorMode(int mirrorMode) {
            mMirrorMode = mirrorMode;
            ThunderLog.release(TAG,
                    "VideoStreamInfo streamId:" + this.mStreamId
                            + " setMirrorMode " + mirrorMode);
        }

        public int getMirrorMode() {
            return mMirrorMode;
        }

        public boolean isLinkedToStream() {
            boolean result = false;
            if (mThunderPlayerView != null) {
                result = mThunderPlayerView.isViewLinkedToStream();
            }

            ThunderLog.release(TAG,
                    "VideoStreamInfo streamId:" + this.mStreamId
                            + " isLinkedToStream " + result);
            return result;
        }

        public boolean isViewPrepared() {
            boolean result = false;
            if (mThunderPlayerView != null) {
                result = mThunderPlayerView.isViewPrepared();
            }

            ThunderLog.release(TAG,
                    "VideoStreamInfo: " + this + "  streamId:" + this.mStreamId
                            + " isViewPrepared " + result);
            return result;
        }
    }

    public String getVideoDecodeBaseStatics(int sendSeq) {
        return YMFPlayerStatisticManager.getInstance().getVideoDecodeBaseStatistics(sendSeq);
    }

    public String getVideoDecodeStatics(boolean bKeyStat, long streamId) {
        String audienceStat = "";
        if (mVideoPlayHiidoUtil != null) {
            audienceStat = mVideoPlayHiidoUtil.getAudienceStatInfo();
        }
        return YMFPlayerStatisticManager.getInstance().getVideoDecodeStatistics(bKeyStat, streamId) + audienceStat;
    }

    // native
    private native void onViewStateNotify(long callback, long streamId, int viewState);

    private native void onHardwareDecodeError(long callback, long userGroupId, long streamId, int errorType);

    private native void onFirstFrameDecodeNotify(long callback, long streamId, long pts, long happenTime);

    private native void onVideoRenderedFrames(long callback, ArrayList<VideoRenderNotify> notify);

    private native void onVideoDecodedFrames(long callback, VideoDecodeEventNotify notify);

    private native void onFirstFrameRender(long callback,
                                           long streamId,
                                           int width,
                                           int height,
                                           long pts);

    private native void onMultiViewSeatInfoNotify(long callback, long uid, int seatId);
}

