package com.thunder.livesdk.video;

import android.content.Context;
import android.graphics.Bitmap;
import android.util.SparseArray;

import com.thunder.livesdk.ThunderMultiVideoViewParam;
import com.thunder.livesdk.log.ThunderLog;
import com.yy.mediaframework.Constant;
import com.yy.videoplayer.videoview.VideoPosition;
import com.yy.videoplayer.view.YMFLayoutParams;
import com.yy.videoplayer.view.YMFVideoPosition;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

//因为统一View之后,可以直接在创建ThunderPlayerView的时候创建实际渲染view，
// 所以不再需要ui线程来创建，所有的渲染操作仅仅只是link和unlink
public class ThunderPlayerMultiViewProxy {

    private static final String TAG = "ThunderPlayerView";

    private YMFVideoPosition[] mVideoPositions;
    private YMFVideoPosition mBgVideoPosition;
    private int mVideoCount;
    private Context mContext;
    private ConcurrentHashMap<String, Seat> mStreamKeyToSeatMap;
    private Constant.MultiLianmaiMode lianMaiMode;
    private YMFLayoutParams mYMFLayoutParams;

    private WeakReference<ThunderVideoPlayEngineImp> engineImp;
    private Object syncLock = new Object();

//    private ThunderPlayerView mVideoPlayerUnityView;
//    private ThunderPlayerMultiView mVideoPlayerView;

    private VideoPlayerView mPlayerView;

    public ThunderPlayerMultiViewProxy(Context context) {
        mContext = context;
        mStreamKeyToSeatMap = new ConcurrentHashMap<String, Seat>();
    }

    public ThunderPlayerMultiViewProxy(Context context, ThunderVideoPlayEngineImp engine) {
        mContext = context;
        mStreamKeyToSeatMap = new ConcurrentHashMap<String, Seat>();
        engineImp = new WeakReference<>(engine);
    }

    public void initMultiPlayerViewLayout(ThunderMultiVideoViewParam layoutParam) {

        if (layoutParam == null) {
            return;
        }
        if (layoutParam.mVideoPositions == null && layoutParam.mVideoViewPositions == null) {
            return;
        }
        int videoCount = 0;
        if (layoutParam.mVideoViewPositions != null) {
            int positionSize = layoutParam.mVideoViewPositions.size();
            mVideoPositions = new YMFVideoPosition[positionSize];
            for (int i = 0; i < positionSize; i++) {
                if (layoutParam.mVideoViewPositions.get(i).mIndex < positionSize) {
                    mVideoPositions[layoutParam.mVideoViewPositions.get(i).mIndex] =
                            convertToVideoPosition(layoutParam.mVideoViewPositions.get(i));
                }
            }
            mBgVideoPosition = convertToVideoPosition(layoutParam.mBgViewPosition);
        } else if (layoutParam.mVideoPositions != null) {
            mVideoPositions = new YMFVideoPosition[layoutParam.mVideoPositions.size()];
            for (int i = 0; i < layoutParam.mVideoPositions.size(); i++) {
                if (layoutParam.mVideoPositions.get(i).mIndex < layoutParam.mVideoPositions.size()) {
                    mVideoPositions[layoutParam.mVideoPositions.get(i).mIndex] = convertToVideoPosition(layoutParam.mVideoPositions.get(i));
                }
            }
            //layoutParam.mVideoPositions.toArray(mVideoPositions);
            mBgVideoPosition = convertToVideoPosition(layoutParam.mBgPosition);
        }
        videoCount = arraySize(mVideoPositions);

        mYMFLayoutParams = new YMFLayoutParams(videoCount);
        mYMFLayoutParams.mDrawPosition = mVideoPositions;
        mYMFLayoutParams.backgroudPosition = mBgVideoPosition;
        mYMFLayoutParams.background = layoutParam.mBgBitmap;

        lianMaiMode = new SparseArray<Constant.MultiLianmaiMode>(10) {{
            put(0, Constant.MultiLianmaiMode.NormalMode);
            put(1, Constant.MultiLianmaiMode.NormalMode);
            put(2, Constant.MultiLianmaiMode.TwoPersonMode);
            put(3, Constant.MultiLianmaiMode.ThreePersonMode);
            put(4, Constant.MultiLianmaiMode.FourPersonMode);
            put(5, Constant.MultiLianmaiMode.FivePersonMode);
            put(6, Constant.MultiLianmaiMode.SixPersonMode);
            put(7, Constant.MultiLianmaiMode.SevenPersonMode);
            put(8, Constant.MultiLianmaiMode.EightPersonMode);
            put(9, Constant.MultiLianmaiMode.NinePersonMode);
        }}.get(arraySize(mVideoPositions));


        updateLayout(videoCount);
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "initMultiPlayerViewLayout:" + mVideoCount + " " + layoutParam.toString());
}
    }

    private void updateLayout(int videoCount) {
        // 布局数量发生变化时，unlink调以前的绑定关系，看视频需要重新link；
        // 布局数量不变，仅仅更新位置等信息时，内部仅仅更新参数
        if (mPlayerView != null) {
            if (mVideoCount != 0 && mVideoCount != videoCount) {
                resetAll();
                mPlayerView.updateMultiViewLayout(mYMFLayoutParams);
                ThunderLog.release(TAG, "need redraw, update MultiPlayerViewLayout:" + mVideoCount);
            } else if (mVideoCount != 0 && mVideoCount == videoCount ||
                    mVideoCount == 0 && mVideoCount < videoCount) {
                mPlayerView.updateMultiViewLayout(mYMFLayoutParams);
                ThunderLog.release(TAG, "just update MultiPlayerViewLayout:" + mVideoCount);
            }
            mPlayerView.enterMultiVideoViewMode(mYMFLayoutParams, lianMaiMode);
        }
        mVideoCount = videoCount;
    }

    private YMFVideoPosition convertToVideoPosition(VideoPosition viewPosition) {
        if (viewPosition == null) {
            return null;
        }
        YMFVideoPosition result = new YMFVideoPosition();
        result.mX = viewPosition.mX;
        result.mY = viewPosition.mY;
        result.mHeight = viewPosition.mHeight;
        result.mWidth = viewPosition.mWidth;
        result.mIndex = viewPosition.mIndex;
        return result;
    }

    private YMFVideoPosition convertToVideoPosition(ThunderMultiVideoViewCoordinate viewPosition) {
        if (viewPosition == null) {
            return null;
        }
        YMFVideoPosition result = new YMFVideoPosition();
        result.mX = viewPosition.mX;
        result.mY = viewPosition.mY;
        result.mHeight = viewPosition.mHeight;
        result.mWidth = viewPosition.mWidth;
        result.mIndex = viewPosition.mIndex;
        return result;
    }

    public VideoPlayerView getPlayerView() {
        return mPlayerView;
    }

    public void updateMultiPlayerView(Object view) {
        ThunderLog.release(TAG, "updateMultiPlayerView enter, mPlayerView: " + mPlayerView +
                " targetView " + view);
        if (view == null) {
            unlink();
            synchronized (syncLock) {
                mPlayerView = null;
            }
        }

        if (mPlayerView == view) {
if (ThunderLog.isInfoValid()) {
            ThunderLog.info(TAG, "updateMultiPlayerView already exist");
}
            return;
        }
        if (mPlayerView != null && mPlayerView != view) {
            unlink();
            mPlayerView = null;
        }

        if (view instanceof ThunderPlayerMultiView) {
            mPlayerView = (ThunderPlayerMultiView) view;
        }

        if (view instanceof ThunderPlayerView) {
            mPlayerView = (ThunderPlayerView) view;
        }

        if (view instanceof ThunderPlayerTextureView) {
            mPlayerView = (ThunderPlayerTextureView) view;
        }

        if (view instanceof ThunderPlayerSurfaceView) {
            mPlayerView = (ThunderPlayerSurfaceView) view;
        }

        if (mPlayerView != null) {
            mPlayerView.enterMultiVideoViewMode(mYMFLayoutParams, lianMaiMode);
        }

        ThunderLog.release(TAG, "updateMultiPlayerView exit");
    }

    /**
     * 卸载SDK 或者设置
     */
    public void destroyVideoPlayerView() {
        ThunderLog.release(TAG, "destroyVideoPlayerView");
        if (mStreamKeyToSeatMap != null) {
            unlink();
            mStreamKeyToSeatMap.clear();
        }

        if (mPlayerView != null) {
            if (mYMFLayoutParams != null) {
                mYMFLayoutParams.background = null;
                mPlayerView.updateMultiViewLayout(mYMFLayoutParams);
            }
            mPlayerView.leaveMultiVideoViewMode();
        }
        mPlayerView = null;

        mYMFLayoutParams = null;
    }

    public VideoPlayerView getVideoPlayerView() {
        return mPlayerView;
    }

    //去掉old的link关系，之后绑定新的link关系，一次完成
    public int switchDualVideoView(String oldStreamKey, String newStreamKey, long newStreamId) {
        int ret = 0;
        if (oldStreamKey == null || oldStreamKey == null || newStreamId == 0L || newStreamId == -1L) {
            ThunderLog.release(TAG,
                    "switchDualVideoView  old " + oldStreamKey + " new:" + newStreamKey +
                            "streamId:" + newStreamId);
            return -1;
        }

        synchronized (mStreamKeyToSeatMap) {
            //将旧的oldSeat里面的streamId替换 并且将mStreamKeyToSeatMap的key替换为newStreamKey
            Seat oldSeat = removeSeatByStreamKey(oldStreamKey);
            if (oldSeat != null) {
                if (mPlayerView != null) {
                    ret = mPlayerView.switchDualVideoView(oldSeat.streamId, newStreamId,
                            oldSeat.seatIdx);
                }
                oldSeat.streamId = newStreamId;
                mStreamKeyToSeatMap.put(newStreamKey, oldSeat);
            }
        }
        return ret;

    }

    /**
     * 更新用户位置信息
     *
     * @param streamKey
     * @param seatId
     */
    public int updateSeat(String streamKey, int seatId) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "innerUpdateSeat() called with: uid = [" + streamKey + "], seat = [" + seatId + "]");
}
        if (seatId >= mVideoCount || streamKey == null) {
            ThunderLog.error(TAG, "updateSeat:" + seatId + "maxSeats:" + mVideoCount + "streamKey:" + streamKey);
            return -1;
        }

        // 默认为单人布局，可以设置seatId为-1
        if (mVideoPositions != null && mVideoPositions.length == 1 && mVideoPositions[0].mHeight == -2 && mVideoPositions[0].mWidth == -2) {
            if (seatId == -1) {
                seatId = 0;
            }
        }
        return bindSeat(streamKey, seatId);
    }

    private int bindSeat(final String streamKey, final int seatId) {
        ThunderLog.release(TAG, "bindSeat called with: uid = [" + streamKey + "], seatNo = [" + seatId + "]");
        synchronized (mStreamKeyToSeatMap) {

            Seat seatItem = findSeatByStreamKey(streamKey);

            // 已经存在的某一路流的座位信息，因为对外接口没有unlink的形式
            // 此处可以通过seatId小于0的方式，unlink某一路流
            if ((seatItem != null) && (seatItem.seatIdx >= 0) &&
                    (seatItem.streamId != -1) &&
                    (seatId < 0)) {
                ThunderLog.release(TAG, "bindSeat unlink:" + streamKey + " seatId: "
                        + seatId + " streamId:" + seatItem.streamId);
                unlink(seatItem);
                seatItem.seatIdx = -1;
                return 0;
            }

            if (seatId < 0) {
                ThunderLog.error(TAG, "updateSeat invalid, seat " + seatId + " invalid!");
                return -1;
            }

            // 下面是正常的处理逻辑，针对某一路流
            // 没有找到该路流对应的位置信息，创建位置，并填充座位号
            if (seatItem == null) {
                seatItem = new Seat() {{
                    this.seatIdx = seatId;
                }};
            } else {
                // 找到该路流对应的位置信息，已经link过，需要unlink之前的，然后更新新的位置
                if (seatItem.seatIdx != seatId
                        && seatItem.status == Seat.STATUS_LINK) {
if (ThunderLog.isInfoValid()) {
                    ThunderLog.info(TAG,
                            "bindSeat updateseat had linked:" + streamKey + ", will unlink!!");
}

                    // 取消老的位置
                    unlink(seatItem);
                }

                seatItem.seatIdx = seatId;

                // 查找该座位是否绑定过，如果说明座位忙，此处选择直接return，或者也可以强制更新
                if (processBusySeatIfExist(streamKey, seatId) > 0) {
                    // 该位置忙
                    ThunderLog.error(TAG, "bindSeat busy!!");
                }
            }
            // 更新位置后，满足条件的可以绑定渲染了
            if (seatItem.seatIdx >= 0 && seatItem.status == Seat.STATUS_IDEL && seatItem.streamId != -1) {
                link(seatItem);
            }
            mStreamKeyToSeatMap.put(streamKey, seatItem);
            ThunderLog.release(TAG, "bindSeat updateseat streamKey:" + streamKey + ",seat:" + seatId + " >> "
                    + seatItem + " " + " streamId " + seatItem.streamId);

            return 0;
        }
    }

    /**
     * 更新位置信息
     *
     * @param streamKey 跟据streamName appId uid生成，早于streamId
     * @param streamId  传输生成
     */
    public void prepareStreamSeat(String streamKey, long streamId) {

        if (streamKey == null) {
            ThunderLog.error(TAG, "prepareStreamSeat streamKey null");
            return;
        }
        if (mStreamKeyToSeatMap == null) {
            ThunderLog.error(TAG, "prepareStreamSeat mStreamKeyToSeatMap null");
            return;
        }

        if (streamId == 0L || streamId == -1L) {
            ThunderLog.error(TAG, "prepareStreamSeat streamId," + streamId);
            return;
        }
        synchronized (mStreamKeyToSeatMap) {
            Seat seatItem = findSeatByStreamKey(streamKey);


            if (seatItem == null) {
                seatItem = new Seat();
                seatItem.streamId = streamId;
                mStreamKeyToSeatMap.put(streamKey, seatItem);
                return;
            }

            if (seatItem != null &&
                    seatItem.seatIdx >= 0 &&
                    seatItem.streamId != streamId &&
                    seatItem.status == Seat.STATUS_LINK) {
                ThunderLog.release(TAG, "prepareStreamSeat replace and link: seatItem.streamId" + seatItem.streamId +
                        " streamId:" + streamId);
                unlink(seatItem);
                seatItem.streamId = streamId;
                link(seatItem);
            } else if (seatItem != null &&
                    seatItem.status == Seat.STATUS_IDEL) {

                seatItem.streamId = streamId;
                ThunderLog.release(TAG, "prepareStreamSeat update: seatItem.streamId:" + streamId +
                        " seatId:" + seatItem.seatIdx);
            }
        }
    }

    /**
     * 更新位置信息
     *
     * @param streamKey 跟据streamName appId uid生成，早于streamId
     * @param streamId  传输生成
     */
    public void bindStreamToSeat(String streamKey, long streamId, boolean isNeedReLink) {

        if (streamKey == null) {
            ThunderLog.error(TAG, "bindStreamToSeat streamKey null");
            return;
        }
        if (mStreamKeyToSeatMap == null) {
            ThunderLog.error(TAG, "bindStreamToSeat mStreamKeyToSeatMap null");
            return;
        }

        if (streamId == 0L || streamId == -1L) {
            ThunderLog.error(TAG, "bindStreamToSeat streamId," + streamId);
            return;
        }
        synchronized (mStreamKeyToSeatMap) {
            Seat seatItem = findSeatByStreamKey(streamKey);


            if (seatItem == null) {
                seatItem = new Seat();
                seatItem.streamId = streamId;
                mStreamKeyToSeatMap.put(streamKey, seatItem);
                return;
            }

            if (seatItem != null &&
                    seatItem.seatIdx >= 0 &&
                    seatItem.status == Seat.STATUS_LINK) {
if (ThunderLog.isInfoValid()) {
                ThunderLog.info(TAG, "bindStreamToSeat replace and link: seatItem.streamId" + seatItem.streamId +
                        " streamId:" + streamId + " isNeedRelink:" + isNeedReLink);
}
                if (isNeedReLink && seatItem.streamId != streamId) {
                    unlink(seatItem);
                    seatItem.streamId = streamId;
                    link(seatItem);
                }
            } else if (seatItem != null &&
                    seatItem.status == Seat.STATUS_IDEL) {

                seatItem.streamId = streamId;
                link(seatItem);
                ThunderLog.release(TAG, "bindStreamToSeat update: seatItem.streamId:" + streamId +
                        " seatId:" + seatItem.seatIdx);
            }
        }
    }

    private int processBusySeatIfExist(String streamKey, int seatId) {
        if (seatId >= 0) {
            Seat curSeat = findSeatByStreamKey(streamKey);
            ArrayList<Seat> seatItemList = findSeatBySeatId(seatId);
            if (seatItemList != null && seatItemList.size() > 0) {
                for (Seat seatItem : seatItemList) {
                    if (seatItem != null && seatItem != curSeat &&
                            seatItem.streamId != 0 && curSeat.streamId != 0 &&
                            (seatItem.streamId >> 32 != curSeat.streamId >> 32)) {
                        ThunderLog.error(TAG, "processBusySeat, process streamKey" + streamKey + " busyStreamId "
                                + seatItem.streamId + " seat " + seatId + " busy!");
                        unlink(seatItem);
                        seatItem.seatIdx = -1;
                        if (engineImp != null && engineImp.get() != null) {
                            engineImp.get().onMultiViewSeatInfoChangedNotify(seatItem.streamId, seatItem.seatIdx);
                        }
                        return 1;
                    }
                }
            }
        }
        return 0;
    }

    public Seat findSeatByStreamKey(String streamKey) {
        if (mStreamKeyToSeatMap == null) {
            ThunderLog.error(TAG, "mStreamKeyToSeatMap null");
            return null;
        }
        Seat seatItem = mStreamKeyToSeatMap.get(streamKey);
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "findSeatByStreamKey:" + streamKey + ",caches:" + mStreamKeyToSeatMap);
}
        return seatItem;
    }

    public Seat removeSeatByStreamKey(String streamKey) {
        return mStreamKeyToSeatMap.remove(streamKey);
    }

    public ArrayList<Seat> findSeatBySeatId(int seatIdx) {
        ArrayList<Seat> seatArrayList = new ArrayList<>(0);
        if (mStreamKeyToSeatMap != null && seatIdx >= 0) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem.seatIdx == seatIdx) {
                    seatArrayList.add(seatItem);
                }
            }
        }
        return seatArrayList;
    }


    public ArrayList<String> findStreamKeyBySeatId(int seatIdx) {
        ArrayList<String> streamKeyList = new ArrayList<>(0);
        if (mStreamKeyToSeatMap != null && seatIdx >= 0) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem.seatIdx == seatIdx) {
                    streamKeyList.add(entry.getKey());
                }
            }
        }
        return streamKeyList;
    }

    public Seat findSeatByStreamId(long streamId) {
        if (mStreamKeyToSeatMap != null) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem.streamId == streamId) {
                    return seatItem;
                }
            }
        }
        return null;
    }

    private int link(Seat seatItem) {
        int result = -1;
        if (seatItem != null) {

            if (mPlayerView != null && seatItem.seatIdx >= 0 && seatItem.streamId != 0 &&
                    seatItem.status == Seat.STATUS_IDEL) {
                result = mPlayerView.linkToStream(seatItem.streamId, seatItem.seatIdx);
            }

            if (result >= 0) {
                seatItem.status = Seat.STATUS_LINK;
            }
            ThunderLog.release(TAG, "link- Seat: streamId " +
                    seatItem.streamId + " seatIdx " + seatItem.seatIdx +
                    " mPlayerView " + mPlayerView + ",ret:" + result);
        } else {
            ThunderLog.error(TAG, "link- Seat:" + seatItem + ",ret:" + result);
        }
        return result;
    }

    private int unlink(Seat seatItem) {
        int result = -1;
        if (seatItem != null) {

            if (mPlayerView != null && seatItem.seatIdx >= 0 && seatItem.streamId != 0 &&
                    seatItem.status == Seat.STATUS_LINK) {
                result = mPlayerView.unLinkFromStream(seatItem.streamId, seatItem.seatIdx);
            }

            if (result >= 0) {
                seatItem.status = Seat.STATUS_IDEL;
            }
            ThunderLog.release(TAG, "unlink- Seat: streamId " +
                    seatItem.streamId + " seatIdx " + seatItem.seatIdx +
                    " mPlayerView " + mPlayerView + ",ret:" + result);
        }
        return result;
    }

    public int link(String streamKey) {
        Seat seatItem = findSeatByStreamKey(streamKey);
        return link(seatItem);
    }

    public int unlink(String streamKey) {
        Seat seatItem = findSeatByStreamKey(streamKey);
        int ret = unlink(seatItem);
        if (seatItem != null) {
            seatItem.streamId = -1; // 停流时需要将streamId置为-1，否则主播重开播会闪屏
        }
        return ret;
    }

    public void releaseSeatByStreamKey(String streamKey) {
        synchronized (mStreamKeyToSeatMap) {
            mStreamKeyToSeatMap.remove(streamKey);
if (ThunderLog.isInfoValid()) {
            ThunderLog.info(TAG, "releaseSeatByStreamKey- Seat:" + streamKey);
}
        }
    }

    public int unlink() {
        if (mStreamKeyToSeatMap != null) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem != null) {
                    unlink(seatItem);
                }
            }
        }
        return 0;
    }


    public int link() {
        if (mStreamKeyToSeatMap != null) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem != null) {
                    link(seatItem);
                }
            }
        }
        return 0;
    }

    public void resetAll() {
        if (mStreamKeyToSeatMap != null) {
            unlinkAndReset();
            //mStreamKeyToSeatMap.clear();
            ThunderLog.release(TAG, "resetAll");
        }
    }

    public void quitRoom() {
        ThunderLog.release(TAG, "quitRoom");
        resetAll();
        mStreamKeyToSeatMap.clear();
    }

    public int unlinkAndReset() {
        if (mStreamKeyToSeatMap != null) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem != null) {
                    unlink(seatItem);
                    seatItem.seatIdx = -1;
                    if (engineImp != null && engineImp.get() != null) {
                        engineImp.get().onMultiViewSeatInfoChangedNotify(seatItem.streamId, seatItem.seatIdx);
                    }
                }
            }
        }
        return 0;
    }

    public boolean setScaleMode(int scaleMode) {
        ThunderLog.release(TAG, "setScaleMode:" + scaleMode);
        boolean result = true;

        if (mStreamKeyToSeatMap != null && mPlayerView != null) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem != null && seatItem.seatIdx >= 0) {
                    result &= mPlayerView.setScaleMode(seatItem.seatIdx, scaleMode);
                }
            }
        }

        return result;
    }

    public boolean setScaleMode(String streamKey, int scaleMode) {
        ThunderLog.release(TAG, "setScaleMode: " + streamKey + " " + scaleMode);
        boolean result = true;

        if (mStreamKeyToSeatMap != null && mPlayerView != null && scaleMode >= 0) {
            Seat seat = findSeatByStreamKey(streamKey);
            if (seat != null) {
                result = mPlayerView.setScaleMode(seat.seatIdx, scaleMode);
            }
        }

        return result;
    }

    public void setMirrorMode(String streamKey, int mirrorMode) {
        ThunderLog.release(TAG, "setMirrorMode: " + streamKey + " " + mirrorMode);
        if (mStreamKeyToSeatMap != null && mPlayerView != null) {
            Seat seat = findSeatByStreamKey(streamKey);
            if (seat != null) {
                mPlayerView.setMirrorMode(seat.seatIdx, mirrorMode);
            }
        }

    }

    public Bitmap getVideoScreenshot(String streamKey) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "getVideoScreenshot: " + streamKey);
}
        if (mStreamKeyToSeatMap != null && mPlayerView != null) {
            Seat seat = findSeatByStreamKey(streamKey);
            if (seat != null) {
                return mPlayerView.getVideoScreenshot(seat.seatIdx);
            }
        }

        return null;
    }

    public static <T> int arraySize(T[] xs) {
        return xs == null ? 0 : xs.length;
    }

    public static class Seat {
        private static final int STATUS_IDEL = 0;
        private static final int STATUS_LINK = 1;

        public int seatIdx = -1;
        public long streamId = 0;
        public int status = STATUS_IDEL;

        @Override
        public String toString() {
            return "Seat{" +
                    ", seatIdx=" + seatIdx +
                    ", status=" + status +
                    '}';
        }
    }
}
