﻿#pragma once

#include <cstdint>

#ifdef __cplusplus
#define DECL_START extern "C" {
#define DECL_END }
#else
#define DECL_START
#define DECL_END
#endif // __cplusplus

#define kFlagNone 0
#define kFlagFile (1 << 0)
#define kFlagStdout (1 << 1)
#define kFlagLogcat (1 << 2)
#define kFlagWindbg (1 << 3)
#define kFlagShowPid (1 << 10)
#define kFlagShowTid (1 << 11)
#define kFlagShowFile (1 << 12)
#define kFlagShowFunction (1 << 13)
#define kFlagShowLine (1 << 14)
#define kFlagShowFileShort (1 << 15)
#define kFlagShowAll (kFlagShowPid | kFlagShowTid | kFlagShowFile | kFlagShowFunction | kFlagShowLine | kFlagShowFileShort)
#define kFlagAll 0xffffffff

#define XL_ERR_FILE_NOT_FOUND  -100
#define XL_ERR_INVALID_PATH    -101
#define XL_ERR_ALREADY_SETUP   -102
#define XL_ERR_INVALID_MODNAME -103
#define XL_ERR_MKDIR_FAILED    -104 

namespace jolog
{
  //日志等级
  enum LogLevel
  {
    kLogTrace = 0,
    kLogDebug = 1,
    kLogInfo = 2,
    kLogWarning = 3,
    kLogError = 4,
    kLogAssert = 5,
    kLogRelease = 10
  };

#ifdef _WIN32
#define MAX_SIZE_PER_FILE (1024 * 1024 * 50)
#else
#define MAX_SIZE_PER_FILE (1024 * 1024 * 5)
#endif

  /**
   * @brief 写日志文件配置
   * 
   */
  struct LogFileConfig
  {
    uint32_t maxFileSize;         //每个文件最大的大小（字节）
    uint32_t maxFileNum;          //写文件的目录下，由jolog写下的日志文件个数上限
    uint32_t maxLogItemStored;    //缓存在内存中没写入文件的日志条数的上限
    uint32_t flushFileInterval;   //flush写文件缓存的间隔(毫秒)
    bool bAutoDeleteZipFile;      //是否自动删除上传后的zip文件，默认删除
    bool bModNameWithDatePid;     //默认为true，控制文件名格式

    LogFileConfig() : maxFileSize(MAX_SIZE_PER_FILE), maxFileNum(8), maxLogItemStored(2048), flushFileInterval(5000), bAutoDeleteZipFile(true), bModNameWithDatePid(true)
    {

    }

    LogFileConfig(uint32_t _maxFileSize, uint32_t _maxFileNum, uint32_t _maxLogItemStored, bool _bAutoDeleteZipFile, bool _bModNameWithDatePid) :
      maxFileSize(_maxFileSize), maxFileNum(_maxFileNum),
      maxLogItemStored(_maxLogItemStored), bAutoDeleteZipFile(_bAutoDeleteZipFile),
      bModNameWithDatePid(_bModNameWithDatePid)
    {
    }
  };

  /**
   * @brief 日志回调函数原型
   * 
   */
  typedef void(*XLogCallback)(int level, int modId, const char* text);

  class IXLogger
  {
  public:

    /**
     * @brief 启动内部hydra相关业务的线程
     * 
     * @param appId 非0值;  若=0, 则同stopXLogger
     */
    virtual void startXLogger(uint32_t appId, const char * logModuleId) = 0;

    /**
     * @brief 停止内部hydra相关业务的线程 需要在hydra deinit之前调用
     * 
     */
    virtual void stopXLogger() = 0;

    /**
     * @brief 添加用户，服务治理平台拉取日志时，指定的用户id
     * 
     * @param uid 用户id
     */
    virtual void addUser(const char* uid) = 0;

    /**
     * @brief 移除用户， 移除后不能用这个uid拉取日志了
     * 
     * @param uid 用户uid
     */
    virtual void removeUser(const char* uid) = 0;

  public:

    /**
     * @brief 设置写日志文件的目录路径、日志文件的限制、模块名称（体现在日志文件名的开头）
     * 
     * @param filesDir 写日志的目录的绝对路径
     * @param fileCfg 日志文件限制
     * @param prefix 日志文件名的前缀
     * @param uid 如果非null, 等同于调用了addUser
     * @return int 返回0，则没有错误。 返回负值，XL_ERR_开头的 宏定义错误码，说明错误原因
     */
    virtual int setup(const char *filesDir, const LogFileConfig &fileCfg, const char *prefix, const char *uid = nullptr) = 0;

    /**
     * @brief 设置日志等级
     * 
     * @param level 低于该等级的日志不输出
     */
    virtual void setLogLevel(LogLevel level) = 0;   

    /**
     * @brief kFlagLogcat android是否输出到logcat, kFlagWindbg windows是否通过outputdebugstring输出， kFlagStdout， ios/mac 是否输出到stdout
     * 其他flag暂时未实现。 
     * 
     * @param flags flags
     */

    virtual void setLogFlags(uint32_t flags) = 0;

    /**
     * @brief 设置日志回调函数
     * 
     * @param callback 回调函数
     */
    virtual void setLogCallback(XLogCallback callback) = 0;

    /**
     * @brief 触发日志上传
     * 
     */
    virtual void triggerLogFileUpload() = 0;

    /**
     * @brief 输出日志
     * 
     * @param level 日志等级
     * @param modId 输出日志的模块的ID，只在回调setLogCallback设置的回调时使用
     * @param modName 模块名称，每条日志都会带上，在日志等级之后
     * @param tag 每条日志带上在[]内部
     * @param text 要输出的日志
     * @param bAssembled 日志输出是否自动加上 线程id和 tag。 true， 不加； false， 加
     */
    virtual void putLog(LogLevel level, int modId, const char* modName, const char* tag, const char* text, bool bAssembled) = 0;

    /**
     * @brief 输出带有format的日志
     * 
     * @param level 日志等级
     * @param tag 每条日志带上在[]内部
     * @param file 文件路径，目前未用到！
     * @param function 函数名，目前未用到！
     * @param line 输出日志的行数，目前未用到！
     * @param format 日志格式
     * @param ... 日志参数
     */
    virtual void putLog(LogLevel level, const char* tag, const char* file, const char* function, unsigned int line, const char* format, ...) = 0;

    /**
     * @brief 设置缓存再内存的日志条数上限
     * 
     * @param maxQsize maxq
     */
    virtual void setMaxQueueSize(int maxQsize) = 0;

    /**
     * @brief 仅在android下使用， 防止再回调日志到java代码后崩溃
     * 
     * @param jvm JavaVM 指针
     */
    virtual void setJvm(void *jvm) = 0;

    /**
     * @brief 清空日志缓存队列中的日志，然后flush日志文件
     * 
     * @return int 0 成功； 非0 参考fflush返回值
     */
    virtual int flushLogs() = 0;

  protected:
    IXLogger() {}
    virtual ~IXLogger() {}
  };
}


DECL_START

/**
 * @brief 创建日志实例
 * 
 * @return jolog::IXLogger* 
 */
jolog::IXLogger* JologCreate();

/**
 * @brief 销毁日志实例
 * 
 * @param logger logger
 */
void JologRelease(jolog::IXLogger* logger);

DECL_END
