/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.text.TextUtils;
import android.util.Pair;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAPushConfigs;
import com.hyphenate.chat.adapter.EMAPushManager;
import com.hyphenate.chat.adapter.EMASilentModeItem;
import com.hyphenate.chat.core.EMPreferenceUtils;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.DeviceUuidFactory;
import com.hyphenate.util.EMLog;

import org.json.JSONException;
import org.json.JSONObject;

import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * \~chinese
 * 离线消息推送管理类，针对 GCM、小米、华为等离线推送以及 APNS。
 *
 * \~english
 * The offline push manager.
 */
public class EMPushManager {
    private static final String TAG = EMPushManager.class.getSimpleName();

    /**
     * \~chinese
     * 推送消息展示样式：SimpleBanner 表示展示简单消息，MessageSummary 表示展示消息内容。
     *
     * \~english
     * The display style of push notifications: 
     * SimpleBanner: Only "You have a new message" is displayed.
     * MessageSummary: The message content is displayed.
     */
    public enum DisplayStyle {
        SimpleBanner, MessageSummary
    }

    EMClient mClient;
    EMAPushManager emaObject;
    EMPushManager(EMClient client, EMAPushManager pushManager) {
        emaObject = pushManager;
        mClient = client;
    }


    /**
     * \~chinese
     * 开启离线消息推送。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @throws HyphenateException
     * @deprecated 使用 {@link EMPushManager#setSilentModeForAll(EMSilentModeParam, EMValueCallBack)} 替代。
     *
     * \~english
     * Turns on the offline push notification.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @throws HyphenateException
     * @deprecated Deprecated. Use {@link EMPushManager#setSilentModeForAll(EMSilentModeParam, EMValueCallBack)} instead.
     */
    @Deprecated
    public void enableOfflinePush() throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.enableOfflineNotification(error);
        handleError(error);
    }

    /**
     * \~chinese
     * 在指定的时间段(24 小时制)内，不推送离线消息。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param start 开始时间。
     * @param end 结束时间。
     * @throws HyphenateException 如果有异常会在这里抛出，包含异常原因。
     * @deprecated 使用 {@link EMPushManager#setSilentModeForAll(EMSilentModeParam, EMValueCallBack)} 替代。
     *
     * \~english
     * Disables the offline push within the specified time period (24-hour clock).

     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param start The start hour.
     * @param end The end hour.
     * @throws HyphenateException A description of the cause of the exception.
     * @deprecated Use {@link EMPushManager#setSilentModeForAll(EMSilentModeParam, EMValueCallBack)} instead.
     */
    @Deprecated
    public void disableOfflinePush(int start, int end) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.disableOfflineNotification(start, end, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 从缓存获取推送配置信息。
     * @return 推送配置信息。
     *
     * \~english
     * Gets the push configurations from the cache.
     * @return The push configurations.
     */
    public EMPushConfigs getPushConfigs(){
        EMAPushConfigs pushConfigs = emaObject.getPushConfigs();
        if(pushConfigs == null){
            return null;
        }
        return new EMPushConfigs(pushConfigs);
    }

    /**
     * \~chinese
     * 从服务器获取推送配置信息。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @return 推送配置信息。
     * @throws HyphenateException 如果有异常会在这里抛出，包含异常原因。
     *
     * \~english
     * Gets the push configurations from the server.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @return The push configurations.
     * @throws HyphenateException A description of the cause of the exception.
     */
    public EMPushConfigs getPushConfigsFromServer() throws HyphenateException{
        EMAError error = new EMAError();
        EMAPushConfigs pushConfigs = emaObject.getPushConfigsFromServer(error);
        handleError(error);

        return new EMPushConfigs(pushConfigs);
    }


    /**
     * \~chinese
     * 设置指定的群组是否接受离线消息推送。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @param groupIds 要设置的群组列表。
     * @param noPush - `true`：不接收离线消息推送；
     *               - `false`：接收推送。
     * @throws HyphenateException 如果有异常会在这里抛出，包含异常原因。
     * @deprecated 使用 {@link EMPushManager#setSilentModeForConversation(String, EMConversation.EMConversationType, EMSilentModeParam, EMValueCallBack)} 替代，设置每个会话的免打扰设置。
     *
     * \~english
     * Sets whether to turn off the offline push notification for the the specified groups.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param groupIds The list of groups.
     * @param noPush - `true`: Turns off the notification;
     *               - `false`: Turns on the notification.
     * @throws HyphenateException A description of the cause of the exception.
     * @deprecated Use {@link EMPushManager#setSilentModeForConversation(String, EMConversation.EMConversationType, EMSilentModeParam, EMValueCallBack)} instead,set DND Settings for each session.
     */
    @Deprecated
    public void updatePushServiceForGroup(List<String> groupIds, boolean noPush) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.updatePushServiceForGroup(groupIds, noPush, error);
        handleError(error);
    }
    /**
     * \~chinese
     * 设置指定的用户是否接收离线消息推送。
     * @param userIds 要设置的用户列表。
     * @param noPush - `true`：不接收离线消息推送；
     *               - `false`：接收推送。
     * @throws HyphenateException 如果有异常会在这里抛出，包含异常原因。
     * @deprecated 使用 {@link EMPushManager#setSilentModeForConversation(String, EMConversation.EMConversationType, EMSilentModeParam, EMValueCallBack)} 替代，设置每个会话的免打扰设置。
     *
     * \~english
     * Sets whether to turn off the offline push for specified users.
     * @param userIds The list of users.
     * @param noPush - `true`：turns off the notification;
     *               - `false`：turns on the notification.
     * @throws HyphenateException A description of the cause of the exception.
     * @deprecated Use {@link EMPushManager#setSilentModeForConversation(String, EMConversation.EMConversationType, EMSilentModeParam, EMValueCallBack)} instead,set DND Settings for each session
     */
    @Deprecated
    public void updatePushServiceForUsers(List<String> userIds, boolean noPush) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.updatePushServiceForUsers(userIds, noPush, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 获取关闭了离线消息推送的群组。
     *
     * 同步方法，会阻塞当前线程。
     *
     * @return 群组列表。
     * @deprecated 使用 {@link EMPushManager#getSilentModeForConversation(String, EMConversation.EMConversationType, EMValueCallBack)} 替代，获取每个会话的免打扰设置来判断。
     *
     * \~english
     * Gets the list of groups for which the offline push is turned off.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @return The list of groups for which the offline push is turned off.
     * @deprecated Use {@link EMPushManager#getSilentModeForConversation(String, EMConversation.EMConversationType, EMValueCallBack)} instead,get the DND Settings for each conversation to determine
     */
    @Deprecated
    public List<String> getNoPushGroups(){
        return emaObject.getNoPushGroups();
    }
    /**
     * \~chinese
     * 从缓存中获取关闭了离线消息推送的用户。
     * 
     * 注意：
     * 如果需要获取最新的数据可先调用 {@link EMPushManager#getPushConfigsFromServer()} 后，再调用本方法。
     * @return 关闭了离线消息推送的用户列表。
     * @deprecated 使用 {@link EMPushManager#getSilentModeForConversation(String, EMConversation.EMConversationType, EMValueCallBack)} 替代，获取每个会话的免打扰设置来判断。
     *
     * \~english
     * Gets the list of user IDs with the offline push notification turned off from the cache.
     * 
     * Note:
     * If you need to get the latest data, call {@Link EmpushManager#getPushConfigsFromServer()} before calling this method.
     * @return  The list of users for which the offline push notification is turned off.
     * @deprecated Use {@link EMPushManager#getSilentModeForConversation(String, EMConversation.EMConversationType, EMValueCallBack)} instead. Get the do-not-disturb settings for each conversation.
     */
    @Deprecated
    public List<String> getNoPushUsers(){
        return emaObject.getNoPushUsers();
    }

    /**
	 * \~chinese
	 * 更新当前用户的推送昵称。
     * 离线消息推送的时候可以显示推送昵称而不是用户 ID。
     * 当用户更改昵称（可通过 {@link EMUserInfoManager#updateOwnInfo(EMUserInfo, EMValueCallBack)} 或者
     * {@link EMUserInfoManager#updateOwnInfoByAttribute(EMUserInfo.EMUserInfoType, String, EMValueCallBack)} 修改）时，
     * 务必也调用此方法更新到 Chat 服务器，防止显示差异。
     *
     * 参考：
     * 异步方法见 {@link #asyncUpdatePushNickname(String, EMCallBack)}
	 *
     * 同步方法，会阻塞当前线程。
     *
	 * @param nickname 推送昵称，需要与用户属性中的昵称区分开。
	 *
	 * \~english
     * Updates the push display nickname of the current user.
	 * The nickname that is displayed in the push notification bar of the recipient's client when a message from the user is pushed. 
     * If no nickname is set, the user ID of the message sender, instead of the nickname, is indicated in the notification bar.
     * The nickname can be different from the nickname in user attributes. However, Agora recommends that you use the same nickname for both.
     * Therefore, if either nickname is updated, the other should be changed at the same time. 
     * For example, if you change the nickname in the user profile ({@link EMUserInfoManager#updateOwnInfo(EMUserInfo, EMValueCallBack)}
     * or {@link EMUserInfoManager#updateOwnInfoByAttribute(EMUserInfo.EMUserInfoType, String, EMValueCallBack)}),
     * remember to call this method to update the push nickname to ensure consistency.
     *
     * For the asynchronous method, see {@link #asyncUpdatePushNickname(String, EMCallBack)}.
	 *
     * This is a synchronous method and blocks the current thread.
     *
	 * @param nickname The push display nickname, which is different from the nickname in the user profile.
	 */
    public boolean updatePushNickname(String nickname) throws IllegalArgumentException, HyphenateException {
        if (TextUtils.isEmpty(nickname)) {
            throw new IllegalArgumentException("nick name is null or empty");
        }
        String currentUser = EMClient.getInstance().getCurrentUser();
        if (TextUtils.isEmpty(currentUser)) {
            throw new IllegalArgumentException("currentUser is null or empty");
        }
        String accessToken = EMClient.getInstance().getAccessToken();
        if (TextUtils.isEmpty(accessToken)) {
            throw new IllegalArgumentException("token is null or empty");
        }
        EMAError error = new EMAError();
        emaObject.updatePushNickname(nickname, error);
        handleError(error);
        return true;
    }

    /**
	 * \~chinese
	 * 更新当前用户的推送昵称。
     * 
     * 异步方法。
     * 
     * 离线消息推送的时候可以显示推送昵称而不是用户 ID。
     * 当用户更改昵称（可通过 {@link EMUserInfoManager#updateOwnInfo(EMUserInfo, EMValueCallBack)} 或者
     * {@link EMUserInfoManager#updateOwnInfoByAttribute(EMUserInfo.EMUserInfoType, String, EMValueCallBack)} 修改）时，

     * 务必也调用此方法更新到环信服务器，防止显示差异。
     * 
          * 同步方法见 {@link #updatePushNickname(String)}。
	 *
	 * @param nickname 推送昵称，需要与用户属性中的昵称区分开。
	 *
	 * \~english
	 * Updates the push display nickname of the current user.
     * 
     * This is an asynchronous method.
     * 
     * If no nickname is set, the user ID of the message sender, instead of the nickname, is indicated in the notification bar.
     * The nickname can be different from the nickname in user attributes. However, Agora recommends that you use the same nickname for both.
     * Therefore, if either nickname is updated, the other should be changed at the same time. 
     * For example, if you change the nickname in the user profile ({@link EMUserInfoManager#updateOwnInfo(EMUserInfo, EMValueCallBack)}
     * or {@link EMUserInfoManager#updateOwnInfoByAttribute(EMUserInfo.EMUserInfoType, String, EMValueCallBack)}),
     * remember to call this method to update the push nickname to ensure consistency.
     *
     * Reference:
     * The synchronous method see {@link #updatePushNickname(String)}.
     *
	 * @param nickname  The push nickname, which is different from the nickname in user profiles.
	 */
    public void asyncUpdatePushNickname(String nickname, EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    updatePushNickname(nickname);
                    callback.onSuccess();
                } catch (IllegalArgumentException e) {
                    callback.onError(EMError.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
	 * \~chinese
	 * 更新推送消息样式，默认是 {@link DisplayStyle#SimpleBanner}。
     * 
     * 参考：
     * 异步方法见 {@link #asyncUpdatePushDisplayStyle(DisplayStyle, EMCallBack)}。
	 *
     * 同步方法，会阻塞当前线程。
     *
	 * @param style 推送消息样式。
	 *
	 * \~english
	 * Updates the display style of push notifications. The default value is {@link DisplayStyle#SimpleBanner}.
     * 
     * Reference:
     * For the asynchronous method, see {@link #asyncUpdatePushDisplayStyle(DisplayStyle, EMCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
	 *
	 * @param style The display style of push notifications.
	 */
    public void updatePushDisplayStyle(DisplayStyle style) throws IllegalArgumentException, HyphenateException
    {
        String currentUser = EMClient.getInstance().getCurrentUser();
        if (TextUtils.isEmpty(currentUser)) {
            throw new IllegalArgumentException("currentUser is null or empty");
        }
        String accessToken = EMClient.getInstance().getAccessToken();
        if (TextUtils.isEmpty(accessToken)) {
            throw new IllegalArgumentException("token is null or empty");
        }
        EMAError error = new EMAError();
        emaObject.updatePushDisplayStyle(style.ordinal(), error);
        handleError(error);
    }

    /**
	 * \~chinese
	 * 更新推送消息样式，默认是 {@link DisplayStyle#SimpleBanner}。
     * 
     * 异步方法。
     * 
     * 参考：
     * 同步方法见 {@link #updatePushDisplayStyle(DisplayStyle)}。
	 *
	 * @param style 推送消息样式。
	 *
	 * \~english
	 * Updates the display style of push notifications. The default value is {@link DisplayStyle#SimpleBanner}.
     *   
     * This is an asynchronous method.
     * 
     * Reference:
     * For the asynchronous method, see {@link #updatePushDisplayStyle(DisplayStyle)}.
	 *
	 * @param style The display style of push notifications.
	 */
    public void asyncUpdatePushDisplayStyle(DisplayStyle style, EMCallBack callback)
    {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    updatePushDisplayStyle(style);
                    callback.onSuccess();
                } catch (IllegalArgumentException e) {
                    callback.onError(EMError.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    private void reportPushAction(String parameters) throws IllegalArgumentException, HyphenateException
    {
        EMAError error = new EMAError();
        emaObject.reportPushAction(parameters, error);
        handleError(error);
    }

    private void handleError(EMAError error) throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

    /**
     * \~chinese
     * 推送事件枚举类。
     *
     * \~english
     * The push events.
     */
    public enum EMPushAction{
        /**
        * \~chinese
        * 送达事件。 
        *
        * \~english
        * The push notification arrival event.
        */
        ARRIVE("arrive"),
        /**
        * \~chinese
        * 点击事件。
        *
        * \~english
        * The push notification click event.
        */
        CLICK("click");

        private String name;

        EMPushAction(String name){
            this.name = name;
        }
    }

    /**
     * \~chinese
     * 会话提醒类型枚举类。
     *
     * \~english
     * Conversation's notification types.
     */
    public enum EMPushRemindType {
        /** 
         * \~chinese 
         * 收取全部通知。
         * 
         * \~english  
         * Receives all  notifications.
         */
        ALL, 
        /** 
         * \~chinese 
         * 只收取 @ 我的通知。
         * 
         * \~english  
         * Receives only  notifications that mention me.
         */
        MENTION_ONLY, 
        /** 
         * \~chinese 
         * 不收取通知。
         * 
         * \~english  
         * Receives no  notifications.
         */
        NONE
    }

    /**
     * \~chinese
     * 设置会话的免打扰。
     * @param conversationId 会话 ID。
     * @param type 会话类型。
     * @param param 离线推送免打扰参数。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Sets the do-not-disturb mode of the conversation.
     * 
     * @param conversationId The conversation ID.
     * @param type The conversation type.
     * @param param The do-not-disturb parameter.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void setSilentModeForConversation(String conversationId, EMConversation.EMConversationType type, EMSilentModeParam param, EMValueCallBack<EMSilentModeResult> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMASilentModeItem item = emaObject.setSilentModeForConversation(conversationId, type.ordinal(), param.emaObject, error);
                    handleError(error);
                    callBack.onSuccess(new EMSilentModeResult(item));
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 清除会话的离线推送提醒类型设置。
     * 清除之后会话跟随当前登录用户的设置 {@link EMPushManager#setSilentModeForAll(EMSilentModeParam, EMValueCallBack)}。
     * @param conversationId 会话 ID。
     * @param type 会话类型。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Clears the setting of offline push notification type for the conversation.
     * After clearing, the conversation follows the settings of the current logged-in user {@link EMPushManager#setSilentModeForAll(EMSilentModeParam, EMValueCallBack)}.
     * @param conversationId The conversation ID.
     * @param type The conversation type.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void clearRemindTypeForConversation(String conversationId, EMConversation.EMConversationType type, EMCallBack callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    emaObject.clearRemindTypeForConversation(conversationId, type.ordinal(), error);
                    handleError(error);
                    callBack.onSuccess();
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 获取会话的免打扰设置。
     * @param conversationId 会话 ID。
     * @param type 会话类型。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Gets the do-not-disturb settings of the conversation.
     * @param conversationId The conversation ID.
     * @param type The conversation type.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void getSilentModeForConversation(String conversationId, EMConversation.EMConversationType type, EMValueCallBack<EMSilentModeResult> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMASilentModeItem item = emaObject.getSilentModeForConversation(conversationId, type.ordinal(), error);
                    handleError(error);
                    callBack.onSuccess(new EMSilentModeResult(item));
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 设置当前登录用户的免打扰设置。
     * @param param 离线推送免打扰参数。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Sets the do-not-disturb mode for the current login user.
     * @param param The do-not-disturb parameter.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void setSilentModeForAll(EMSilentModeParam param, EMValueCallBack<EMSilentModeResult> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMASilentModeItem item = emaObject.setSilentModeForAll(param.emaObject, error);
                    handleError(error);
                    callBack.onSuccess(new EMSilentModeResult(item));
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 获取当前登录用户的免打扰设置。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Gets the do-not-disturb settings of the current login user.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void getSilentModeForAll(EMValueCallBack<EMSilentModeResult> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    EMASilentModeItem item = emaObject.getSilentModeForAll(error);
                    handleError(error);
                    callBack.onSuccess(new EMSilentModeResult(item));
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 批量获取指定会话的免打扰设置。
     * @param conversationList 会话列表。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Gets the do-not-disturb settings of specified conversations in batches.
     * @param conversationList The conversation list.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void getSilentModeForConversations(List<EMConversation> conversationList, EMValueCallBack<Map<String, EMSilentModeResult>> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    StringBuilder userIdStr = new StringBuilder();
                    StringBuilder groupIdStr = new StringBuilder();
                    for(Iterator<EMConversation> iterator = conversationList.iterator(); iterator.hasNext();){
                        EMConversation conversation = iterator.next();
                        if(conversation.getType() == EMConversation.EMConversationType.Chat){
                            if(!userIdStr.toString().isEmpty()){
                                userIdStr.append(",");
                            }
                            userIdStr.append(conversation.conversationId());
                        } else {
                            if(!groupIdStr.toString().isEmpty()){
                                groupIdStr.append(",");
                            }
                            groupIdStr.append(conversation.conversationId());
                        }
                    }
                    Map<String, String> map = new HashMap<>();
                    map.put("user", userIdStr.toString());
                    map.put("group", groupIdStr.toString());
                    EMAError error = new EMAError();
                    List<EMASilentModeItem> items = emaObject.getSilentModeForConversations(map,  error);
                    handleError(error);
                    Map<String, EMSilentModeResult> itemMap = new HashMap<>();
                    for(EMASilentModeItem item : items){
                        itemMap.put(item.getConversationId(), new EMSilentModeResult(item));
                    }
                    callBack.onSuccess(itemMap);
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });

    }

    /**
     * \~chinese
     * 和服务端同步会话免打扰状态，异步方法
     * @param callBack 处理结果回调，失败会返回失败原因。详见 {@link EMCallBack}。
     *
     * \~english
     * Synchronizes the do-not-disturb status of the conversation with the server. Asynchronous method.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void syncSilentModeConversationsFromServer(EMCallBack callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error=new EMAError();
                    emaObject.syncSilentModeConversationsFromServer(error);
                    handleError(error);
                    callBack.onSuccess();
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 设置用户推送翻译语言。
     * @param languageCode 语言 code。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Sets the preferred language for push notifications.
     * @param languageCode The language code.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void setPreferredNotificationLanguage(String languageCode, EMCallBack callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    emaObject.setPushPerformLanguage(languageCode, error);
                    handleError(error);
                    callBack.onSuccess();
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 获取用户设置的推送翻译语言。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Gets the preferred notification language set by the user.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void getPreferredNotificationLanguage(EMValueCallBack<String> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    String languageCode = emaObject.getPushPerformLanguage(error);
                    handleError(error);
                    callBack.onSuccess(languageCode);
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 绑定设备 token 到服务器。
     * 当 deviceToken 为空时，表示从服务器解绑当前设备的 deviceToken。
     * @param notifierName  表示当前设备的ID，具体如下：
     *                      FCM - Sender ID;
     *                      华为 - App ID;
     *                      小米 - App ID;
     *                      魅族 - App ID;
     *                      OPPO - App Key;
     *                      Vivo - App ID + "#" + App Key;
     * @param deviceToken   当前设备返回的设备 Token .
     * @param callBack
     *
     * \~english
     * Binds the device token to the chat server.
     * If device token is null or "", the device token is unbound from the chat server.
     * @param notifierName The current device ID:
     *                     FCM - Sender ID;
     *                     Hawei - App ID;
     *                     Xiaomi - App ID;
     *                     Meizu - App ID;
     *                     OPPO - App Key;
     *                     Vivo - App ID + "#" + App Key;
     * @param deviceToken  Device token generated by the device manufacturer.
     * @param callBack
     */
    public void bindDeviceToken(String notifierName, String deviceToken, EMCallBack callBack) {
        if(mClient == null || !mClient.isSdkInited()) {
            if(callBack != null) {
                callBack.onError(EMError.GENERAL_ERROR, "SDK should init first!");
            }
            return;
        }
        mClient.execute(()-> {
            try {
                bindDeviceToken(notifierName, deviceToken);
                if(callBack != null) {
                    callBack.onSuccess();
                }
            } catch (HyphenateException e) {
                if(callBack != null) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    void unBindDeviceToken() throws HyphenateException {
        bindDeviceToken(EMPreferenceUtils.getInstance().getPushNotifierName(), "");
    }

    synchronized void bindDeviceToken(String notifierName, String deviceToken) throws HyphenateException {
        if(mClient == null || !mClient.isSdkInited()) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "SDK should init first!");
        }
        if(!mClient.isLoggedInBefore()) {
            throw new HyphenateException(EMError.GENERAL_ERROR, "You need to log in first!");
        }
        if(TextUtils.isEmpty(notifierName)) {
            throw new HyphenateException(EMError.INVALID_PARAM, "Notifier name should not be empty!");
        }
        EMPreferenceUtils preferenceUtils = EMPreferenceUtils.getInstance();
        String savedToken = preferenceUtils.getPushToken();
        if(!TextUtils.isEmpty(savedToken) && TextUtils.equals(savedToken, deviceToken)) {
            if (!EMClient.getInstance().getChatConfigPrivate().isNewLoginOnDevice()) {
                EMLog.e(TAG, TAG + " not first login, ignore token upload action.");
                return;
            }
            EMLog.d(TAG, "push token not change, but last login is not on this device, upload to server");
        }
        String remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl(true, false) + "/users/"
                + EMClient.getInstance().getCurrentUser() + "/push/binding";
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(EMClient.getInstance().getContext());
        JSONObject json = new JSONObject();
        try {
            json.put("device_token", TextUtils.isEmpty(notifierName)? "" : deviceToken);
            json.put("notifier_name", notifierName);
            json.put("device_id", deviceFactory.getDeviceUuid().toString());
        } catch (Exception e) {
            EMLog.e(TAG, "uploadTokenInternal put json exception: " + e.toString());
            throw new HyphenateException(EMError.GENERAL_ERROR, "uploadTokenInternal put json exception: " + e.getMessage());
        }
        int retry_times = 2;
        int statusCode = EMError.GENERAL_ERROR;
        String content = "";
        do{
            try {
                EMLog.e(TAG, "uploadTokenInternal, token=" + deviceToken + ", url=" + remoteUrl
                        + ", notifier name=" + notifierName);
                Pair<Integer, String> response = EMHttpClient.getInstance().sendRequestWithToken(remoteUrl,
                        json.toString(), EMHttpClient.PUT);
                statusCode = response.first;
                content = response.second;
                if (statusCode == HttpURLConnection.HTTP_OK) {
                    // Save token info after requested successfully
                    EMPreferenceUtils.getInstance().setPushToken(deviceToken);
                    EMPreferenceUtils.getInstance().setPushNotifierName(notifierName);
                    EMLog.e(TAG, "uploadTokenInternal success.");
                    return;
                }
                EMLog.e(TAG, "uploadTokenInternal failed: " + content);
                remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl(true, true) + "/users/"
                        + EMClient.getInstance().getCurrentUser();
            } catch (HyphenateException e) {
                EMLog.e(TAG, "uploadTokenInternal failed: " + e.getDescription());
                remoteUrl = EMClient.getInstance().getChatConfigPrivate().getBaseUrl(true, true) + "/users/"
                        + EMClient.getInstance().getCurrentUser();
            }
        }while (--retry_times > 0);

        throw new HyphenateException(statusCode, content);
    }

    /**
     * \~chinese
     * 设置离线推送的推送模板。
     * @param templateName 模板名称。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Sets the push template for offline push notifications.
     * @param templateName The name of the push template.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void setPushTemplate(String templateName, EMCallBack callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    emaObject.setPushTemplate(templateName, error);
                    handleError(error);
                    callBack.onSuccess();
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 获取设置的离线推送模板。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Gets the push template for offline push notifications.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void getPushTemplate(EMValueCallBack<String> callBack){
        mClient.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    EMAError error = new EMAError();
                    String templateName = emaObject.getPushTemplate(error);
                    handleError(error);
                    callBack.onSuccess(templateName);
                }catch (HyphenateException e){
                    callBack.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 上报推送事件。
     *
     * 异步方法。
     *
     * @param json 推送数据里携带的 EPush 字段对应的 value。
     * @param action 推送事件。
     * @param callBack 该方法完成调用的回调。如果该方法调用失败，会包含调用失败的原因。
     *
     * \~english
     * Reports the push events.
     *
     * This is an asynchronous method.
     *
     * @param json Value The value of the EPush field included in the push data.
     * @param action Push event.
     * @param callBack The completion callback, which contains the description of the cause to the failure.
     */
    public void reportPushAction(JSONObject json, EMPushAction action, EMCallBack callBack)
    {
        EMClient.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    if(json != null){
                        json.put("action", action.name);
                        reportPushAction(json.toString());
                    }
                    callBack.onSuccess();
                } catch (IllegalArgumentException e) {
                    callBack.onError(EMError.USER_ILLEGAL_ARGUMENT, e.getMessage());
                } catch (HyphenateException e) {
                    callBack.onError(e.getErrorCode(), e.getDescription());
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        });
    }

}