/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatRoom;

import java.util.List;
import java.util.Map;

/**
 * \~chinese
 * 聊天室类型。
 *
 * \~english
 * Chat room types.
 */
public class EMChatRoom extends EMBase<EMAChatRoom> {

    enum EMChatRoomStyle{
        /**
         *\~chinese
         * 私有聊天室类型，只允许聊天室所有者邀请用户加入。
         *
         * \~english
         * Private chat rooms. Only the owner can invite users to join.
         */
        EMChatRoomStylePrivateOnlyOwnerInvite,

        /**
         * \~chinese
         * 私有聊天室类型，只允许聊天室所有者和群成员邀请用户加入。
         *
         * \~english
         * Private chat rooms. Both the chat room owner and members can invite users to join.
         */
        EMChatRoomStylePrivateMemberCanInvite,

        /**
         * \~chinese
         * 公开聊天室类型，只允许聊天室所有者邀请用户加入; 非聊天室成员用户需发送入群申请，所有者同意后才能入聊天室。
         *
         * \~english
         * Public chat rooms. Only the chat room owner can invite users to join; a user can join a chat room after the owner approves the user's chat room request.
         */
        EMChatRoomStylePublicJoinNeedApproval,

        /**
         * \~chinese
         * 公开聊天室类型，允许非群组成员加入，不需要聊天室所有者同意。
         *
         * \~english
         *  Public chat rooms. A user can join a chat room without the owner approving the user's chat room request.
         */
        EMChatRoomStylePublicOpenJoin
    }

    /**
     * \~chinese
     *  聊天室成员枚举类型。
     *
     *  \~english
     *  Chat room role types.
     */
    public enum EMChatRoomPermissionType {
        /**
         * \~chinese
         * 普通成员类型。
         *
         * \~english
         * Regular member.
         */
        member(0),

        /**
         * \~chinese
         * 聊天室管理员类型。
         *
         * \~english
         * Chat room admin.
         */
        admin(1),

        /**
         * \~chinese
         * 聊天室所有者类型。
         *
         * \~english
         * Chat room owner.
         */
        owner(2),

        /**
         * \~chinese
         * 未知类型。
         *
         * \~english
         * Unknown type.
         */
        none(-1);
        private int permissionType;
        private EMChatRoomPermissionType(int permissionType) {
            this.permissionType = permissionType;
        }
    }

	public EMChatRoom(){
	    emaObject = new EMAChatRoom();
	}
	
	public EMChatRoom(EMAChatRoom room) {
	    emaObject = new EMAChatRoom(room);
	}
	
    public EMChatRoom(String roomId){
        emaObject = new EMAChatRoom(roomId);
    }
    
    public EMChatRoom(String roomId, String roomName){
        emaObject = new EMAChatRoom(roomId);
//        this.nick = roomName;
    }

    /**
     * \~chinese
     * 获取聊天室 ID。
     *
     * @return 聊天室 ID。
     *
     * \~english
     * Gets the chat room ID.
     *
     * @return The chat room ID.
     */
    public String getId() {
        return emaObject.getId();
     }

    /**
     * \~chinese
     * 获取聊天室名称。
     * @return  聊天室名称。
     *
     * \~english
     * Gets the chat room name.
     * @return The chat room name.
     */
     public String getName() {
        return emaObject.getName();
     }

    /**
     * \~chinese
     * 获取聊天室描述。
     * @return 聊天室描述。
     *
     * \~english
     * Gets the chat room description.
     * @return The chat room description.
     */
     public String getDescription() {
        return emaObject.getDescription();
    }

    /**
     * \~chinese
     * 获取聊天室所有者。
     * 获取聊天室详情见 {@link EMChatRoomManager#fetchChatRoomFromServer(String)}。
     * @return 聊天室所有者 ID。如果返回的值为空，表示未获取到聊天室所有者。
     *
     * \~english
     * Gets the chat room owner.
     * For how to get chat room details, see {@link EMChatRoomManager#fetchChatRoomFromServer(String)}.
     * @return The chat room owner ID. If this method returns an empty string, the SDK fails to get chat room details.
     */
    public String getOwner() {
        return emaObject.getOwner();
    }

    /**
     * \~chinese
     * 获取聊天室的管理员列表。
     * 获取聊天室详情见 {@link EMChatRoomManager#fetchChatRoomFromServer(String)}。
     * @return 管理员 ID 列表。如果返回的值未空，表示未获取到聊天室管理员列表。
     *
     * \~english
     * Gets the chat room admin list.
     * For how to get chat room, see {@link EMChatRoomManager#fetchChatRoomFromServer(String)}.
     * @return The chat room admin ID list. If this method returns an empty list, the SDK fails to get the admin list of the chat room.
     */
    public List<String> getAdminList() { return emaObject.getAdministratorList();}

    /**
     * \~chinese
     * 返回在线成员数。
     *
     * 包括聊天室所有者、管理员与普通成员。
     *
     * @return 在线成员数。
     *
     * \~english
     * Returns the number of online members.
     *
     * This includes the chat room owner, administrators, and regular members.
     *
     * @return The number of online members.
     */
    public int getMemberCount()
    {
        return emaObject.getAffiliationsCount();
    }

    /**
     * \~chinese
     * 获取聊天室最大成员数，在创建时确定。
     * 需要获取聊天室详情才能得到正确结果。
     * @return  聊天室最大成员数。如果未获取聊天室最大成员数，则返回 0。
     *
     * \~english
     * Gets the maximum number of members in the chat room which is determined during chat room creation.
     * This method can return a correct value only after you call EMChatRoomManager#fetchChatRoomFromServer(String) to get chat room details. 
     * @return The maximum number of members in a chat room. Returns 0 if the SDK fails to get the maximum number.
     */
    public int getMaxUsers() {
        return emaObject.getMaxUserCount();
    }

    /**
     * \~chinese
     * 返回成员列表。
     * 可通过如下方式获取成员列表：
     * - 成员数少于 200 时，可通过 {@link EMChatRoomManager#fetchChatRoomFromServer(String, boolean)},
     * 第二个参数传入 true，最多可获取 200 个成员。
     * - 通过 {@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)} 分页获取，也可通过其异步方法
     * {@link EMChatRoomManager#asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)} 获取.
     * @return  成员 ID 列表。如果返回的列表为空，表示未获取成员列表。
     *
     * \~english
     * Returns the member list.
     * You can get the member list in the following ways:
     * - When there are less than 200 members, use {@link EMChatRoomManager#fetchChatRoomFromServer(String, boolean)} to get them.
     * If true is passed to the second parameter, you can get up to 200 members.
     * - Use {@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)} to get the member list with pagination
     * or use its asynchronous method {@link EMChatRoomManager#asyncFetchChatRoomMembers(String, String, int, EMValueCallBack)}.
     * 
     * @return The member ID list. If this method returns an empty list, the SDK fails to get the member list.
     */
    public List<String> getMemberList(){
        return emaObject.getMemberList();
    }

    /**
     * \~chinese
     * 返回聊天室黑名单。
     * 
     * 获取黑名单可调用 {@link EMChatRoomManager#fetchChatRoomBlackList(String, int, int)},
     * 调用该方法需要聊天室所有者或者管理员身份，也可调用其异步方法实现。
     * {@link EMChatRoomManager#asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}
     * @return 返回黑名单列表。如果返回的列表为空，表示未获取到黑名单列表。
     * @deprecated 已废弃，请用 {@link EMChatRoom#getBlacklist()} 代替。
     *
     * \~english
     * Returns the chat room blocklist.
     * To get the blocklist, you can call {@link EMChatRoomManager#fetchChatRoomBlackList(String, int, int)}.
     * You can call this method as the chat room owner or admin or call its asynchronous method
     * {@link EMChatRoomManager#asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}.
     * @return The blocklist. If this method returns an empty list, the SDK fails to get the blocklist.
     * @deprecated Deprecated. Please use {@link EMChatRoom#getBlacklist()} instead.
     */
    @Deprecated
    public List<String> getBlackList() { return getBlacklist(); }

    /**
     * \~chinese
     * 返回聊天室黑名单。
     * 获取黑名单可调用 {@link EMChatRoomManager#fetchChatRoomBlackList(String, int, int)}。
     * 调用这个方法需要聊天室所有者或者管理员身份，也可调用其异步方法实现。
     * {@link EMChatRoomManager#asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}
     * @return 返回黑聊天室名单列表。如果返回的列表为空，表示未获取到黑名单列表。
     *
     * \~english
     * Returns the chat room blocklist.
     * To get the blocklist, you can call {@link EMChatRoomManager#fetchChatRoomBlackList(String, int, int)}.
     * You can call this method as the chat room owner or admin or call its asynchronous method.
     * {@link EMChatRoomManager#asyncFetchChatRoomBlackList(String, int, int, EMValueCallBack)}.
     * @return The chat room blocklist. If this method returns an empty list, the SDK fails to get the blocklist.
     */
    public List<String> getBlacklist() { return emaObject.getBlockList(); }

    /**
     * \~chinese
     * 返回禁言列表。
     * 
     * 获取禁言列表可调用 {@link EMChatRoomManager#fetchChatRoomMuteList(String, int, int)}。
     * 调用这个方法需要聊天室所有者或者管理员身份，也可调用其异步方法实现
     * {@link EMChatRoomManager#asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}。
     *
     * @return  返回禁言列表，Map.entry.key 是禁言的成员 ID，Map.entry.value 是禁言动作存在的时间，单位是毫秒。
     *          如果返回的 map 为空，表示未获取到禁言列表。
     *
     * \~english
     * Returns the mute list of the chat room.
     * To get the mute list, you can call {@link EMChatRoomManager#fetchChatRoomMuteList(String, int, int)}.
     * You can call this method as the chat room owner or admin or call its asynchronous method
     * {@link EMChatRoomManager#asyncFetchChatRoomMuteList(String, int, int, EMValueCallBack)}.
     *
     * @return The mute list, where Map.entry.key is the ID of the muted member and Map.entry.value is the mute duration in milliseconds.
     *         If this method returns an empty map, the SDK fails to get the mute list.
     */
    public Map<String, Long> getMuteList() { return emaObject.getMuteList(); }

    /**
     * \~chinese
     * 返回白名单列表。
     * 获取白名单可调用 {@link EMChatRoomManager#fetchChatRoomWhiteList(String, EMValueCallBack)}。
     * 调用这个方法需要聊天室所有者或者管理员身份。
     *
     * @return  白名单列表。如果返回的列表为空，表示未获取到白名单列表。
     * @deprecated 已废弃，请用 {@link EMChatRoom#getWhitelist()} 代替。
     *
     * \~english
     * Returns the chat room allowlist.
     * 
     * To get the allowlist, you can call {@link EMChatRoomManager#fetchChatRoomWhiteList(String, EMValueCallBack)}.
     * 
     * You can call this method as the chat room owner or admin.
     * 
     * @return The allowlist. If this method returns an empty list, the SDK fails to get the allowlist.
     * @deprecated Deprecated. Please use {@link EMChatRoom#getWhitelist()} instead.
     */
    @Deprecated
    public List<String> getWhiteList() { return getWhitelist(); }

    /**
     * \~chinese
     * 返回白名单列表。
     * 
     * 获取白名单可调用 {@link EMChatRoomManager#fetchChatRoomWhiteList(String, EMValueCallBack)}。
     * 
     * 调用这个方法需要聊天室所有者或者管理员身份。
     * 
     * @return  白名单列表。如果返回的列表为空，表示未获取到白名单列表。
     *
     * \~english
     * Returns the chat room allowlist.
     * 
     * To get the allowlist, you can call {@link EMChatRoomManager#fetchChatRoomWhiteList(String, EMValueCallBack)}.
     * 
     * You can call this method as the chat room owner or admin.
     * 
     * @return The allowlist. If this method returns an empty list, the SDK fails to get the allowlist.
     */
    public List<String> getWhitelist() { return emaObject.getWhiteList(); }

    /**
     * \~chinese
     * 检查是否全员禁言。
     * 
     * 此方法存在使用局限性，建议谨慎使用。
     * 
     * 使用此方法需注意以下两点：
     * 
     * - 加入聊天室后，接收到一键禁言/取消禁言的回调后，该状态会被更新，此时使用该方法获取的状态是正确的。
     * - 从聊天室退出后再进入聊天室，利用该方法获取的状态不可信。
     * 
     * @return 返回是否全员禁言。
     *         - `true`：开启全员禁言。
     *         - `false`：关闭全员禁言。 
     *
     * \~english
     * Checks whether all members are muted.
     * 
     * This method has use limitations and should be used with caution.
     * 
     * Pay attention to the following when using this method：
     * 
     * - Upon your receipt of the callback of one-click mute or unmute after you join the chat room, the status will be updated and the staus obtained using this method is correct.
     * 
     * - After you exit from the chat room before reentering it, the status obtained using this method is not trustworthy.
     * 
     * @return Whether all members are muted. 
     * - `true`: All members are muted. 
     * - `false`: Not all members are muted.
     */
    public boolean isAllMemberMuted() { return emaObject.isAllMemberMuted(); }


    /**
     * \~chinese
     * 获取聊天室公告。
     * 
     * 获取聊天室公告，可调用 {@link EMChatRoomManager#fetchChatRoomAnnouncement(String)},
     * 
     * 也可调用其异步方法 {@link EMChatRoomManager#asyncFetchChatRoomAnnouncement(String, EMValueCallBack)}。
     * 
     * @return 聊天室公告。如果返回的值为空，表示未获取聊天室公告。
     *
     * \~english
     * Gets the chat room announcement.
     * 
     * To get the chat room announcement, you can call {@link EMChatRoomManager#fetchChatRoomAnnouncement(String)}
     * 
     * or its asynchronous method {@link EMChatRoomManager#asyncFetchChatRoomAnnouncement(String, EMValueCallBack)}.
     * 
     * @return The chat room announcement. If this method returns an empty string, the SDK fails to get the chat room announcement.
     */
    public String getAnnouncement() {
        return emaObject.getAnnouncement();
    }

    /**
     * \~chinese
     * 获取当前用户在聊天室中的角色。
     * 
     * @return 当前用户在聊天室中的角色，见 {@link EMChatRoomPermissionType}。
     *
     * \~english
     * Gets the current user's role in the chat room.
     * 
     * @return The current user's role in the chat room, see {@link EMChatRoomPermissionType}.
     */
    public EMChatRoomPermissionType getChatRoomPermissionType() {
        int permissionType = emaObject.permissionType();
        if(permissionType == EMChatRoomPermissionType.member.permissionType) {
            return EMChatRoomPermissionType.member;
        }else if(permissionType == EMChatRoomPermissionType.admin.permissionType) {
            return EMChatRoomPermissionType.admin;
        }else if(permissionType == EMChatRoomPermissionType.owner.permissionType) {
            return EMChatRoomPermissionType.owner;
        }else {
            return EMChatRoomPermissionType.none;
        }
    }

}
