/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *  
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.net.Uri;

import com.hyphenate.chat.adapter.message.EMAFileMessageBody;
import com.hyphenate.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import com.hyphenate.chat.adapter.message.EMAImageMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.adapter.message.EMAVideoMessageBody;
import com.hyphenate.chat.adapter.message.EMAVoiceMessageBody;
import com.hyphenate.util.EMFileHelper;

import java.io.File;

/**
 * \~chinese
 * 文件类消息的基类。
 * 
 * \~english
 * The base class of file type message.
 */
public abstract class EMFileMessageBody extends EMMessageBody {

	/**
	 * \~chinese
	 * 文件下载状态。
	 *
	 * \~english
	 * The download status of the attachment file .
	 */
	public enum EMDownloadStatus {
        DOWNLOADING,/** \~chinese 正在下载文件消息。  \~english The SDK is downloading the file message. */
        SUCCESSED,/** \~chinese 下载文件消息成功。 \~english The SDK successfully downloads the file message. */
        FAILED,/** \~chinese 下载文件消息失败。  \~english The SDK fails to download the file message.*/
        PENDING /** \~chinese 文件消息下载正在等待中。  \~english File message download is pending.*/
    }

	/**
	 * \~chinese
	 * 文件类型消息体构造方法。
	 * @param localPath	文件的绝对路径或者资源标志符的字符串形式。
	 *
	 * \~english
	 * The constructor method of the file message body.
	 * @param localPath	The absolute path to the attachment file or the path to the URI as a string.
	 */
	public EMFileMessageBody(String localPath) {
    	this(EMFileHelper.getInstance().formatInUri(localPath));
	}

	/**
	 * \~chinese
	 * 文件类型消息体构造方法。
	 * @param localPath	文件的资源标志符。
	 *
	 * \~english
	 * The constructor method of the file message body.
	 * @param localPath  The localPath URI of the attachment file.
	 */
	public EMFileMessageBody(Uri localPath) {
		this(localPath, EMAMessageBody.EMAMessageBodyType_FILE);
	}

	/**
	 * @param localPath The localPath URI of the attachment file.
	 * @param type  The file type.
	 */
	EMFileMessageBody(String localPath, int type) {
    	this(Uri.fromFile(new File(localPath)), type);
	}

	EMFileMessageBody(Uri localPath, int type) {
		switch (type) {
		case EMAMessageBody.EMAMessageBodyType_FILE:
			emaObject = new EMAFileMessageBody(localPath, type);
			setFileLength(EMFileHelper.getInstance().getFileLength(localPath));
			break;
		case EMAMessageBody.EMAMessageBodyType_IMAGE:
			emaObject = new EMAImageMessageBody(localPath, null);
			setFileLength(EMFileHelper.getInstance().getFileLength(localPath));
			break;
		case EMAMessageBody.EMAMessageBodyType_VIDEO:
			emaObject = new EMAVideoMessageBody(localPath, null);
			setFileLength(EMFileHelper.getInstance().getFileLength(localPath));
			break;
		case EMAMessageBody.EMAMessageBodyType_VOICE:
		    emaObject = new EMAVoiceMessageBody(localPath, 0);
			setFileLength(EMFileHelper.getInstance().getFileLength(localPath));
		    break;
		default:
			break;
		}
	}

	public EMFileMessageBody(EMAFileMessageBody body) {
		emaObject = body;
	}
	

	/**
	 * \~chinese
	 * 获得文件名。
	 * 
	 * @return	文件名。
	 * 
	 * \~english
	 * Gets the file name.
	 * @return  The file name.
	 */
	public String getFileName() {
		return ((EMAFileMessageBody)emaObject).displayName();
	}

	/**
	 * \~chinese
	 * 设置文件名。
	 * @param fileName 文件名。
	 *
	 * \~english
	 * Sets the file name.
	 * @param fileName  The file name.
	 */
	public void setFileName(String fileName) {
		((EMAFileMessageBody)emaObject).setDisplayName(fileName);
	}

	/**
	 * \~chinese
	 * 本地图片，语音等文件的路径或者资源标志符的字符串形式。
	 * 
	 * @return 本地文件的绝对路径及资源标志符的字符串形式。
	 *
	 * \~english
	 * The local path or the string in the form of URI of the file.
	 *
	 * @return The local file path or the string form of URI.
	 */
	public String getLocalUrl() {
		return EMFileHelper.getInstance().formatOutLocalUrl(((EMAFileMessageBody)emaObject).getLocalUrl());
	}

	/**
	 * \~chinese
	 * 设置本地文件路径。
	 * @param localUrl	文件的绝对路径或者资源标志符的字符串形式。
	 *
	 * \~english
	 * Sets the local path of the file. 
	 * @param localUrl	The absolute path to the file or the string form of URI.
	 */
	public void setLocalUrl(String localUrl) {
		((EMAFileMessageBody)emaObject).setLocalPath(EMFileHelper.getInstance().formatInUriToString(localUrl));
	}

	/**
	 * \~chinese
	 * 本地图片、语音等文件的资源标识符 URI。
	 *
	 * @return 本地图片、语音等文件的 URI。
	 *
	 * \~english
	 * Gets the URI of the local image, voice and so on.
	 * 
	 * @return  The URI of the local image, voice and so on.
	 */
	public Uri getLocalUri() {
		return EMFileHelper.getInstance().formatOutUri(((EMAFileMessageBody)emaObject).getLocalUrl());
	}

	/**
	 * \~chinese
	 * 设置本地文件的资源标识符。
	 * @param localUrl	文件的资源标志符。
	 *
	 * \~english
	 * Sets the URI of the local file.
	 * @param localUrl	The URI of file.
	 */
	public void setLocalUrl(Uri localUrl) {
		((EMAFileMessageBody)emaObject).setLocalPath(EMFileHelper.getInstance().formatInUriToString(localUrl));
	}

	/**
	 * \~chinese
	 * 远程图片，声音等的服务器路径。
	 * @return	文件的服务器路径。
	 * 
	 * \~english
	 * Gets the path of the attachment file in the server.
	 * @return  The path of the attachment file in the server.
	 */
	public String getRemoteUrl() {
		return ((EMAFileMessageBody)emaObject).getRemoteUrl();
	}

	/**
	 * \~chinese
	 * 设置文件消息体文件的服务器地址。
	 * @param remoteUrl 文件的服务器地址。
	 *
	 * \~english
	 * Sets the path of the attachment file in the server.
	 * @param remoteUrl  The path of the attachment file in the server.
	 */
	public void setRemoteUrl(String remoteUrl) {
		((EMAFileMessageBody)emaObject).setRemotePath(remoteUrl);
	}

	/**
	 * \~chinese
	 * 设置文件的密钥，此方法一般由 SDK 调用。
	 * @param secret 文件的密钥。
	 *
	 * \~english
	 * Sets the file's token, this method is typically invoked by the SDK.
	 * @param secret The file's token.
	 */
	public void setSecret(String secret) {
		((EMAFileMessageBody)emaObject).setSecretKey(secret);
	}

	/**
	 * \~chinese
	 * 获取服务器生成的文件的密钥。
	 * @return 服务器生成的文件的密钥。
	 * \~english
	 * Gets the token for downloading the message attachment.
	 * @return  The file token generated by the server。
	 */
	public String getSecret() {
		return ((EMAFileMessageBody)emaObject).getSecret();
	}

	/**
	 * \~chinese
	 * 获取文件名。
	 * @return 文件名。
	 *
	 * \~english
	 * Gets the filename.
	 * @return The filename.
	 */
	public String displayName() {
		return ((EMAFileMessageBody)emaObject).displayName();
	}

	/**
	 * \~chinese
	 * 设置文件大小，以字节为单位。
	 * @param length  文件大小，以字节为单位。
	 *
	 * \~english
	 * Sets the data length (bytes) of the attachment file.
	 * @param length  The data length (bytes) of the attachment file.
	 */
	public void setFileLength(long length) {
		((EMAFileMessageBody)emaObject).setFileLength(length);
	}

	/**
	 * \~chinese
	 * 获取文件的下载状态。
	 * @return 文件的下载状态。
	 *
	 * \~english
	 * Gets the download status of the attachment file.
	 * @return The download status of the attachment file.
	 */
	public EMDownloadStatus downloadStatus() {
	    EMADownloadStatus _status = ((EMAFileMessageBody)emaObject).downloadStatus();
	    switch (_status) {
		/** \~chinese 正在下载文件消息。  \~english The SDK is downloading the file message. */
	    case DOWNLOADING:
	        return EMDownloadStatus.DOWNLOADING;
		/** \~chinese 下载文件消息成功。 \~english The SDK successfully downloads the file message. */
	    case SUCCESSED:
	        return EMDownloadStatus.SUCCESSED;
		/** \~chinese 下载文件消息失败。  \~english The SDK fails to download the file message.*/
	    case FAILED:
	        return EMDownloadStatus.FAILED;
		/** \~chinese 文件消息下载正在等待中。  \~english File message download is pending.*/
	    case PENDING:
	        return EMDownloadStatus.PENDING;
	    }

	    return EMDownloadStatus.SUCCESSED;
	}

	/**
	 * \~chinese
	 * 设置文件的下载状态。
	 * @param status	文件的下载状态。
	 *
	 * \~english
	 * Sets the download status of the attachment file.
	 * @param status	The downloading status of the attachment file.
	 */
	public void setDownloadStatus(EMDownloadStatus status) {
		((EMAFileMessageBody)emaObject).setDownloadStatus(EMADownloadStatus.valueOf(status.name()));
	}

}
