package com.yy.platform.baseservice;

import android.os.Bundle;
import android.os.Handler;

import com.yy.platform.baseservice.task.RPCTask;

import java.util.Map;

public interface IRPCChannel {

    /**
     * 客户端插件对IDL生成的Service RPC调用回调接口
     *
     * @param <T>
     */
    interface RPCCallback<T> {
        /**
         * 请求成功，具体业务成功与否需由业务自行解业务包后判断
         *
         * @param requestId 请求的上下文RequestId
         * @param response  客户端插件对IDL生成的message
         */
        void onSuccess(int requestId, T response);

        /**
         * 请求失败
         *
         * @param requestId  请求的上下文RequestId
         * @param sdkResCode SDK 的返回码，具体请看{@link ConstCode.SdkResCode}
         * @param srvResCode Service接入服务的返回码，
         *                   如果是rpcCall调用请看{@link ConstCode.SrvResCode}
         *                   如果是bind绑定uid请看{@link ConstCode.BindResCode}
         *                   仅当sdkResCode返回RECV_RESPONSE时，参数srvResCode值有效
         * @param e          包装的返回码描述
         */
        void onFail(int requestId, int sdkResCode, int srvResCode, Exception e);
    }

    /**
     * 客户端插件对IDL生成的Service RPC调用回调接口
     *
     * @param <T>
     */
    interface RPCCallbackWithTrace<T> {
        /**
         * 请求成功，具体业务成功与否需由业务自行解业务包后判断
         *
         * @param requestId 请求的上下文RequestId
         * @param response  客户端插件对IDL生成的message
         */
        void onSuccess(int requestId, String traceId, T response);

        /**
         * 请求失败
         *
         * @param requestId  请求的上下文RequestId
         * @param sdkResCode SDK 的返回码，具体请看{@link ConstCode.SdkResCode}
         * @param srvResCode Service接入服务的返回码，
         *                   如果是rpcCall调用请看{@link ConstCode.SrvResCode}
         *                   如果是bind绑定uid请看{@link ConstCode.BindResCode}
         *                   仅当sdkResCode返回RECV_RESPONSE时，参数srvResCode值有效
         * @param e          包装的返回码描述
         */
        void onFail(int requestId, String traceId, int sdkResCode, int srvResCode, Exception e);
    }

    /**
     * 设置默认的全局生效的RouteArgs
     * 目前需要设置key="setselector"，value=Locale.getDefault().getCountry(),即{https://zh.wikipedia.org/wiki/ISO_3166-1}中的二位代码
     *
     * @param routeArgs
     */
    void setDefaultRouteArgs(Map<String, String> routeArgs);

    /**
     * 设置默认的全局生效的Headers
     *
     * @param headers
     */
    void setDefaultHeaders(Map<String, String> headers);

    /**
     * 上下行请求
     *
     * @param requestParam 请求参数，具体请看{@link RPCTask.RequestParam}构造函数，必填
     * @param options      调用选项，可选，具体请看{@link com.yy.platform.baseservice.task.TaskOptions}中的Option定义
     *                     约定：[1-5)秒内，由SDK决定超时返回，在[5, 60]秒范围时由服务器决定返回超时
     * @param callback     本次请求的回调，响应结构请看{@link RPCTask.ResponseParam}中的成员
     * @return 请求发起成功同步返回一个RequestId标识，请求失败同步返回-1
     */
    int rpcCall(RPCTask.RequestParam requestParam,
                Bundle options,
                RPCCallback<RPCTask.ResponseParam> callback);

    int rpcCall(RPCTask.RequestParam requestParam,
                Bundle options, Handler callbackScheduler,
                RPCCallback<RPCTask.ResponseParam> callback);

    int rpcCall(RPCTask.RequestParam requestParam,
                Bundle options, Handler callbackScheduler,
                RPCCallbackWithTrace<RPCTask.ResponseParam> callback);
}
