package com.tencent.imsdk.v2;

import android.text.TextUtils;
import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.community.CommunityListener;
import com.tencent.imsdk.community.CommunityManager;
import com.tencent.imsdk.community.PermissionGroupInfo;
import com.tencent.imsdk.community.PermissionGroupInfoGetResult;
import com.tencent.imsdk.community.PermissionGroupInfoModifyParam;
import com.tencent.imsdk.community.PermissionGroupMemberInfoResult;
import com.tencent.imsdk.community.PermissionGroupMemberOperationResult;
import com.tencent.imsdk.community.PermissionGroupOperationResult;
import com.tencent.imsdk.community.TopicInfo;
import com.tencent.imsdk.community.TopicInfoGetResult;
import com.tencent.imsdk.community.TopicInfoModifyParam;
import com.tencent.imsdk.community.TopicOperationResult;
import com.tencent.imsdk.community.TopicPermissionResult;
import com.tencent.imsdk.group.GroupInfo;
import com.tencent.imsdk.group.GroupManager;
import com.tencent.imsdk.manager.BaseManager;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

public class V2TIMCommunityManagerImpl extends V2TIMCommunityManager {
    private static final String TAG = "V2TIMCommunityManagerImpl";

    private CommunityListener mInternalCommunityListener;
    private final List<V2TIMCommunityListener> mCommunityListenerList = new ArrayList<>();

    private static class V2TIMCommunityManagerImplHolder {
        private static final V2TIMCommunityManagerImpl v2TIMCommunityManagerImpl = new V2TIMCommunityManagerImpl();
    }

    static V2TIMCommunityManagerImpl getInstance() {
        return V2TIMCommunityManagerImplHolder.v2TIMCommunityManagerImpl;
    }

    private V2TIMCommunityManagerImpl() {
        initCommunityListener();
    }

    private void initCommunityListener() {
        mInternalCommunityListener = new CommunityListener() {
            @Override
            public void onCreateTopic(String groupID, String topicID) {
                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onCreateTopic(groupID, topicID);
                }
            }

            @Override
            public void onDeleteTopic(String groupID, List<String> topicIDList) {
                List<String> unmodifiableList = Collections.unmodifiableList(topicIDList);
                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onDeleteTopic(groupID, unmodifiableList);
                }
            }

            @Override
            public void onChangeTopicInfo(String groupID, TopicInfo topicInfo) {
                V2TIMTopicInfo v2TIMTopicInfo = new V2TIMTopicInfo();
                if (topicInfo != null) {
                    v2TIMTopicInfo.setTopicInfo(topicInfo);
                }

                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onChangeTopicInfo(groupID, v2TIMTopicInfo);
                }
            }

            @Override
            public void onReceiveTopicRESTCustomData(String topicID, byte[] customData) {
                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onReceiveTopicRESTCustomData(topicID, customData);
                }
            }

            @Override
            public void onCreatePermissionGroup(String groupID, PermissionGroupInfo permissionGroupInfo) {
                V2TIMPermissionGroupInfo v2TIMPermissionGroupInfo = new V2TIMPermissionGroupInfo();
                if (permissionGroupInfo != null) {
                    v2TIMPermissionGroupInfo.setPermissionGroupInfo(permissionGroupInfo);
                }

                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onCreatePermissionGroup(groupID, v2TIMPermissionGroupInfo);
                }
            }

            @Override
            public void onDeletePermissionGroup(String groupID, List<String> permissionGroupIDList) {
                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onDeletePermissionGroup(groupID, permissionGroupIDList);
                }
            }

            @Override
            public void onChangePermissionGroupInfo(String groupID, PermissionGroupInfo permissionGroupInfo) {
                V2TIMPermissionGroupInfo v2TIMPermissionGroupInfo = new V2TIMPermissionGroupInfo();
                if (permissionGroupInfo != null) {
                    v2TIMPermissionGroupInfo.setPermissionGroupInfo(permissionGroupInfo);
                }

                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onChangePermissionGroupInfo(groupID, v2TIMPermissionGroupInfo);
                }
            }

            @Override
            public void onAddMembersToPermissionGroup(
                String groupID, String permissionGroupID, List<String> memberIDList) {
                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onAddMembersToPermissionGroup(groupID, permissionGroupID, memberIDList);
                }
            }

            @Override
            public void onRemoveMembersFromPermissionGroup(
                String groupID, String permissionGroupID, List<String> memberIDList) {
                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onRemoveMembersFromPermissionGroup(groupID, permissionGroupID, memberIDList);
                }
            }

            @Override
            public void onAddTopicPermission(
                String groupID, String permissionGroupID, HashMap<String, Long> topicPermissionMap) {
                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onAddTopicPermission(groupID, permissionGroupID, topicPermissionMap);
                }
            }

            @Override
            public void onDeleteTopicPermission(String groupID, String permissionGroupID, List<String> topicIDList) {
                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onDeleteTopicPermission(groupID, permissionGroupID, topicIDList);
                }
            }

            @Override
            public void onModifyTopicPermission(
                String groupID, String permissionGroupID, HashMap<String, Long> topicPermissionMap) {
                for (V2TIMCommunityListener listener : mCommunityListenerList) {
                    listener.onModifyTopicPermission(groupID, permissionGroupID, topicPermissionMap);
                }
            }
        };

        CommunityManager.getInstance().setCommunityListener(mInternalCommunityListener);
    }

    @Override
    public void addCommunityListener(final V2TIMCommunityListener listener) {
        if (listener == null) {
            return;
        }

        IMContext.getInstance().runOnMainThread(new Runnable() {
            @Override
            public void run() {
                if (mCommunityListenerList.contains(listener)) {
                    return;
                }
                mCommunityListenerList.add(listener);
            }
        });
    }

    @Override
    public void removeCommunityListener(final V2TIMCommunityListener listener) {
        if (listener == null) {
            return;
        }

        IMContext.getInstance().runOnMainThread(new Runnable() {
            @Override
            public void run() {
                mCommunityListenerList.remove(listener);
            }
        });
    }

    @Override
    public void createCommunity(
        V2TIMGroupInfo info, List<V2TIMCreateGroupMemberInfo> memberList, V2TIMValueCallback<String> callback) {
        if (info != null) {
            info.setSupportTopic(true);
        }

        V2TIMGroupManagerImpl.getInstance().createGroup(info, memberList, callback);
    }

    @Override
    public void getJoinedCommunityList(final V2TIMValueCallback<List<V2TIMGroupInfo>> callback) {
        V2TIMValueCallback<List<GroupInfo>> v2Callback = new V2TIMValueCallback<List<GroupInfo>>() {
            @Override
            public void onSuccess(List<GroupInfo> groupInfoList) {
                if (callback != null) {
                    List<V2TIMGroupInfo> v2GroupInfoList = new ArrayList<>();
                    for (GroupInfo item : groupInfoList) {
                        V2TIMGroupInfo v2GroupInfo = new V2TIMGroupInfo();
                        v2GroupInfo.setGroupInfo(item);
                        v2GroupInfoList.add(v2GroupInfo);
                    }
                    callback.onSuccess(v2GroupInfoList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        GroupManager.getInstance().getJoinedCommunityList(new IMCallback<List<GroupInfo>>(v2Callback) {
            @Override
            public void success(List<GroupInfo> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });

        BaseManager.getInstance().reportTUIComponentUsage(BaseManager.TUI_COMPONENT_COMMUNITY);
    }

    @Override
    public void createTopicInCommunity(String groupID, V2TIMTopicInfo topicInfo, V2TIMValueCallback<String> callback) {
        if (TextUtils.isEmpty(groupID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid groupID");
            }
            return;
        }

        if (topicInfo == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "topicInfo is null");
            }
            return;
        }

        CommunityManager.getInstance().createTopic(groupID, topicInfo.getTopicInfo(), new IMCallback<String>(callback) {
            @Override
            public void success(String data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void deleteTopicFromCommunity(
        String groupID, List<String> topicIDList, final V2TIMValueCallback<List<V2TIMTopicOperationResult>> callback) {
        if (TextUtils.isEmpty(groupID) || topicIDList == null || topicIDList.isEmpty()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid groupID or topicIDList");
            }
            return;
        }

        V2TIMValueCallback<List<TopicOperationResult>> v2Callback =
            new V2TIMValueCallback<List<TopicOperationResult>>() {
                @Override
                public void onSuccess(List<TopicOperationResult> topicResultList) {
                    if (callback != null) {
                        List<V2TIMTopicOperationResult> v2TIMTopicOperationResultList = new ArrayList<>();
                        for (TopicOperationResult topicResult : topicResultList) {
                            V2TIMTopicOperationResult v2TIMTopicOperationResult = new V2TIMTopicOperationResult();
                            v2TIMTopicOperationResult.setTopicOperationResult(topicResult);
                            v2TIMTopicOperationResultList.add(v2TIMTopicOperationResult);
                        }

                        callback.onSuccess(v2TIMTopicOperationResultList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().deleteTopic(groupID, topicIDList, new IMCallback(v2Callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void setTopicInfo(V2TIMTopicInfo v2TIMTopicInfo, final V2TIMCallback callback) {
        if (v2TIMTopicInfo == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid topicInfo");
            }
            return;
        }

        long modifyFlag = v2TIMTopicInfo.getModifyFlag();
        TopicInfo topicInfo = v2TIMTopicInfo.getTopicInfo();

        TopicInfoModifyParam topicInfoModifyParam = new TopicInfoModifyParam();
        topicInfoModifyParam.setTopicInfo(topicInfo);
        topicInfoModifyParam.setModifyFlag(modifyFlag);

        CommunityManager.getInstance().setTopicInfo(topicInfoModifyParam, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getTopicInfoList(
        String groupID, List<String> topicIDList, final V2TIMValueCallback<List<V2TIMTopicInfoResult>> callback) {
        if (TextUtils.isEmpty(groupID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid groupID");
            }
            return;
        }

        V2TIMValueCallback<List<TopicInfoGetResult>> v2Callback = new V2TIMValueCallback<List<TopicInfoGetResult>>() {
            @Override
            public void onSuccess(List<TopicInfoGetResult> topicInfoGetResults) {
                if (callback != null) {
                    List<V2TIMTopicInfoResult> topicInfoResultList = new ArrayList<>();
                    for (TopicInfoGetResult item : topicInfoGetResults) {
                        V2TIMTopicInfoResult result = new V2TIMTopicInfoResult();
                        result.setTopicInfoGetResult(item);
                        topicInfoResultList.add(result);
                    }
                    callback.onSuccess(topicInfoResultList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        CommunityManager.getInstance().getTopicList(
            groupID, topicIDList, new IMCallback<List<TopicInfoGetResult>>(v2Callback) {
                @Override
                public void success(List<TopicInfoGetResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    @Override
    public void createPermissionGroupInCommunity(
        V2TIMPermissionGroupInfo info, final V2TIMValueCallback<String> callback) {
        if (info == null) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid info");
            }

            return;
        }

        if (TextUtils.isEmpty(info.getGroupID())) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid groupID");
            }

            return;
        }

        V2TIMValueCallback<String> v2TIMValueCallback = new V2TIMValueCallback<String>() {
            @Override
            public void onSuccess(String permissionGroupID) {
                if (callback != null) {
                    callback.onSuccess(permissionGroupID);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        CommunityManager.getInstance().createPermissionGroupInCommunity(
            info.getPermissionGroupInfo(), new IMCallback<String>(v2TIMValueCallback) {
                @Override
                public void success(String data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    @Override
    public void deletePermissionGroupFromCommunity(String groupID, List<String> permissionGroupIDList,
        V2TIMValueCallback<List<V2TIMPermissionGroupOperationResult>> callback) {
        if (TextUtils.isEmpty(groupID) || permissionGroupIDList == null || permissionGroupIDList.isEmpty()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
            }

            return;
        }

        V2TIMValueCallback<List<PermissionGroupOperationResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<PermissionGroupOperationResult>>() {
                @Override
                public void onSuccess(List<PermissionGroupOperationResult> operationResultList) {
                    if (callback != null) {
                        List<V2TIMPermissionGroupOperationResult> resultList = new ArrayList<>();
                        for (PermissionGroupOperationResult result : operationResultList) {
                            V2TIMPermissionGroupOperationResult v2TIMPermissionGroupOperationResult =
                                new V2TIMPermissionGroupOperationResult();
                            v2TIMPermissionGroupOperationResult.setPermissionGroupOperationResult(result);
                            resultList.add(v2TIMPermissionGroupOperationResult);
                        }

                        callback.onSuccess(resultList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().deletePermissionGroupFromCommunity(
            groupID, permissionGroupIDList, new IMCallback<List<PermissionGroupOperationResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<PermissionGroupOperationResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    @Override
    public void modifyPermissionGroupInfoInCommunity(V2TIMPermissionGroupInfo info, V2TIMCallback callback) {
        if (info == null || TextUtils.isEmpty(info.getGroupID()) || TextUtils.isEmpty(info.getPermissionGroupID())) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
            }

            return;
        }

        long modifyFlag = info.getModifyFlag();
        PermissionGroupInfoModifyParam param = new PermissionGroupInfoModifyParam();
        param.setModifyFlag(modifyFlag);
        param.setPermissionGroupInfo(info.getPermissionGroupInfo());

        CommunityManager.getInstance().modifyPermissionGroupInfoOfCommunity(param, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    @Override
    public void getJoinedPermissionGroupListInCommunity(
        String groupID, V2TIMValueCallback<List<V2TIMPermissionGroupInfoResult>> callback) {
        if (TextUtils.isEmpty(groupID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupID is empty");
            }

            return;
        }

        V2TIMValueCallback<List<PermissionGroupInfoGetResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<PermissionGroupInfoGetResult>>() {
                @Override
                public void onSuccess(List<PermissionGroupInfoGetResult> getResultList) {
                    if (callback != null) {
                        List<V2TIMPermissionGroupInfoResult> resultList = new ArrayList<>();
                        for (PermissionGroupInfoGetResult result : getResultList) {
                            V2TIMPermissionGroupInfoResult v2TIMPermissionGroupInfoResult =
                                new V2TIMPermissionGroupInfoResult();
                            v2TIMPermissionGroupInfoResult.setPermissionGroupInfoGetResult(result);

                            resultList.add(v2TIMPermissionGroupInfoResult);
                        }

                        callback.onSuccess(resultList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().getJoinedPermissionGroupListOfCommunity(
            groupID, new IMCallback<List<PermissionGroupInfoGetResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<PermissionGroupInfoGetResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    @Override
    public void getPermissionGroupListInCommunity(String groupID, List<String> permissionGroupIDList,
        V2TIMValueCallback<List<V2TIMPermissionGroupInfoResult>> callback) {
        if (TextUtils.isEmpty(groupID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "groupID is empty");
            }

            return;
        }

        V2TIMValueCallback<List<PermissionGroupInfoGetResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<PermissionGroupInfoGetResult>>() {
                @Override
                public void onSuccess(List<PermissionGroupInfoGetResult> getResultList) {
                    if (callback != null) {
                        List<V2TIMPermissionGroupInfoResult> resultList = new ArrayList<>();
                        for (PermissionGroupInfoGetResult result : getResultList) {
                            V2TIMPermissionGroupInfoResult v2TIMPermissionGroupInfoResult =
                                new V2TIMPermissionGroupInfoResult();
                            v2TIMPermissionGroupInfoResult.setPermissionGroupInfoGetResult(result);

                            resultList.add(v2TIMPermissionGroupInfoResult);
                        }

                        callback.onSuccess(resultList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().getPermissionGroupListOfCommunity(
            groupID, permissionGroupIDList, new IMCallback<List<PermissionGroupInfoGetResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<PermissionGroupInfoGetResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    @Override
    public void addCommunityMembersToPermissionGroup(String groupID, String permissionGroupID, List<String> memberList,
        V2TIMValueCallback<List<V2TIMPermissionGroupMemberOperationResult>> callback) {
        if (TextUtils.isEmpty(groupID) || TextUtils.isEmpty(permissionGroupID) || memberList == null
            || memberList.isEmpty()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
            }

            return;
        }

        V2TIMValueCallback<List<PermissionGroupMemberOperationResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<PermissionGroupMemberOperationResult>>() {
                @Override
                public void onSuccess(List<PermissionGroupMemberOperationResult> operationResultList) {
                    if (callback != null) {
                        List<V2TIMPermissionGroupMemberOperationResult> resultList = new ArrayList<>();
                        for (PermissionGroupMemberOperationResult result : operationResultList) {
                            V2TIMPermissionGroupMemberOperationResult v2TIMPermissionGroupMemberOperationResult =
                                new V2TIMPermissionGroupMemberOperationResult();
                            v2TIMPermissionGroupMemberOperationResult.setPermissionGroupMemberOperationResult(result);

                            resultList.add(v2TIMPermissionGroupMemberOperationResult);
                        }

                        callback.onSuccess(resultList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().addCommunityMembersToPermissionGroup(groupID, permissionGroupID, memberList,
            new IMCallback<List<PermissionGroupMemberOperationResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<PermissionGroupMemberOperationResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    @Override
    public void removeCommunityMembersFromPermissionGroup(String groupID, String permissionGroupID,
        List<String> memberList, V2TIMValueCallback<List<V2TIMPermissionGroupMemberOperationResult>> callback) {
        if (TextUtils.isEmpty(groupID) || TextUtils.isEmpty(permissionGroupID) || memberList == null
            || memberList.isEmpty()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
            }

            return;
        }

        V2TIMValueCallback<List<PermissionGroupMemberOperationResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<PermissionGroupMemberOperationResult>>() {
                @Override
                public void onSuccess(List<PermissionGroupMemberOperationResult> operationResultList) {
                    if (callback != null) {
                        List<V2TIMPermissionGroupMemberOperationResult> resultList = new ArrayList<>();
                        for (PermissionGroupMemberOperationResult result : operationResultList) {
                            V2TIMPermissionGroupMemberOperationResult v2TIMPermissionGroupMemberOperationResult =
                                new V2TIMPermissionGroupMemberOperationResult();
                            v2TIMPermissionGroupMemberOperationResult.setPermissionGroupMemberOperationResult(result);

                            resultList.add(v2TIMPermissionGroupMemberOperationResult);
                        }

                        callback.onSuccess(resultList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().removeCommunityMembersFromPermissionGroup(groupID, permissionGroupID, memberList,
            new IMCallback<List<PermissionGroupMemberOperationResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<PermissionGroupMemberOperationResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    @Override
    public void getCommunityMemberListInPermissionGroup(String groupID, String permissionGroupID, String nextCursor,
        V2TIMValueCallback<V2TIMPermissionGroupMemberInfoResult> callback) {
        if (TextUtils.isEmpty(groupID) || TextUtils.isEmpty(permissionGroupID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
            }

            return;
        }

        V2TIMValueCallback<PermissionGroupMemberInfoResult> v2TIMValueCallback =
            new V2TIMValueCallback<PermissionGroupMemberInfoResult>() {
                @Override
                public void onSuccess(PermissionGroupMemberInfoResult permissionGroupMemberInfoResult) {
                    if (callback != null) {
                        V2TIMPermissionGroupMemberInfoResult v2TIMPermissionGroupMemberInfoResult =
                            new V2TIMPermissionGroupMemberInfoResult();
                        v2TIMPermissionGroupMemberInfoResult.setPermissionGroupMemberInfoResult(
                            permissionGroupMemberInfoResult);

                        callback.onSuccess(v2TIMPermissionGroupMemberInfoResult);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().getCommunityMemberListOfPermissionGroup(groupID, permissionGroupID, nextCursor,
            new IMCallback<PermissionGroupMemberInfoResult>(v2TIMValueCallback) {
                @Override
                public void success(PermissionGroupMemberInfoResult data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    public void addTopicPermissionToPermissionGroup(String groupID, String permissionGroupID,
        HashMap<String, Long> topicPermissionMap, V2TIMValueCallback<List<V2TIMTopicOperationResult>> callback) {
        if (TextUtils.isEmpty(groupID) || TextUtils.isEmpty(permissionGroupID) || topicPermissionMap == null
            || topicPermissionMap.isEmpty()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
            }

            return;
        }

        V2TIMValueCallback<List<TopicOperationResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<TopicOperationResult>>() {
                @Override
                public void onSuccess(List<TopicOperationResult> topicOperationResultList) {
                    if (callback != null) {
                        List<V2TIMTopicOperationResult> v2TIMTopicOperationResultList = new ArrayList<>();
                        for (TopicOperationResult topicResult : topicOperationResultList) {
                            V2TIMTopicOperationResult v2TIMTopicOperationResult = new V2TIMTopicOperationResult();
                            v2TIMTopicOperationResult.setTopicOperationResult(topicResult);
                            v2TIMTopicOperationResultList.add(v2TIMTopicOperationResult);
                        }

                        callback.onSuccess(v2TIMTopicOperationResultList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().addTopicPermissionToPermissionGroup(groupID, permissionGroupID,
            topicPermissionMap, new IMCallback<List<TopicOperationResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<TopicOperationResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    public void deleteTopicPermissionFromPermissionGroup(String groupID, String permissionGroupID,
        List<String> topicIDList, V2TIMValueCallback<List<V2TIMTopicOperationResult>> callback) {
        if (TextUtils.isEmpty(groupID) || TextUtils.isEmpty(permissionGroupID) || topicIDList == null
            || topicIDList.isEmpty()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
            }

            return;
        }

        V2TIMValueCallback<List<TopicOperationResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<TopicOperationResult>>() {
                @Override
                public void onSuccess(List<TopicOperationResult> topicOperationResultList) {
                    if (callback != null) {
                        List<V2TIMTopicOperationResult> v2TIMTopicOperationResultList = new ArrayList<>();
                        for (TopicOperationResult topicResult : topicOperationResultList) {
                            V2TIMTopicOperationResult v2TIMTopicOperationResult = new V2TIMTopicOperationResult();
                            v2TIMTopicOperationResult.setTopicOperationResult(topicResult);
                            v2TIMTopicOperationResultList.add(v2TIMTopicOperationResult);
                        }

                        callback.onSuccess(v2TIMTopicOperationResultList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().deleteTopicPermissionToPermissionGroup(
            groupID, permissionGroupID, topicIDList, new IMCallback<List<TopicOperationResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<TopicOperationResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    public void modifyTopicPermissionInPermissionGroup(String groupID, String permissionGroupID,
        HashMap<String, Long> topicPermissionMap, V2TIMValueCallback<List<V2TIMTopicOperationResult>> callback) {
        if (TextUtils.isEmpty(groupID) || TextUtils.isEmpty(permissionGroupID) || topicPermissionMap == null
            || topicPermissionMap.isEmpty()) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
            }

            return;
        }

        V2TIMValueCallback<List<TopicOperationResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<TopicOperationResult>>() {
                @Override
                public void onSuccess(List<TopicOperationResult> topicOperationResultList) {
                    if (callback != null) {
                        List<V2TIMTopicOperationResult> v2TIMTopicOperationResultList = new ArrayList<>();
                        for (TopicOperationResult topicResult : topicOperationResultList) {
                            V2TIMTopicOperationResult v2TIMTopicOperationResult = new V2TIMTopicOperationResult();
                            v2TIMTopicOperationResult.setTopicOperationResult(topicResult);
                            v2TIMTopicOperationResultList.add(v2TIMTopicOperationResult);
                        }

                        callback.onSuccess(v2TIMTopicOperationResultList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().modifyTopicPermissionOfPermissionGroup(groupID, permissionGroupID,
            topicPermissionMap, new IMCallback<List<TopicOperationResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<TopicOperationResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    public void getTopicPermissionInPermissionGroup(String groupID, String permissionGroupID, List<String> topicIDList,
        V2TIMValueCallback<List<V2TIMTopicPermissionResult>> callback) {
        if (TextUtils.isEmpty(groupID) || TextUtils.isEmpty(permissionGroupID)) {
            if (callback != null) {
                callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
            }

            return;
        }

        V2TIMValueCallback<List<TopicPermissionResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<TopicPermissionResult>>() {
                @Override
                public void onSuccess(List<TopicPermissionResult> resultList) {
                    List<V2TIMTopicPermissionResult> v2TIMTopicPermissionResultList = new ArrayList<>();
                    for (TopicPermissionResult result : resultList) {
                        V2TIMTopicPermissionResult v2TIMTopicPermissionResult = new V2TIMTopicPermissionResult();
                        v2TIMTopicPermissionResult.setTopicPermissionResult(result);

                        v2TIMTopicPermissionResultList.add(v2TIMTopicPermissionResult);
                    }

                    callback.onSuccess(v2TIMTopicPermissionResultList);
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        CommunityManager.getInstance().getTopicPermissionOfPermissionGroup(
            groupID, permissionGroupID, topicIDList, new IMCallback<List<TopicPermissionResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<TopicPermissionResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }
}
