package com.thunder.livesdk.video;

import android.content.Context;
import android.graphics.Bitmap;
import android.os.Handler;
import android.os.Looper;
import android.util.SparseArray;

import com.thunder.livesdk.BuildConfig;
import com.thunder.livesdk.ThunderMultiVideoViewParam;
import com.thunder.livesdk.log.ThunderLog;
import com.yy.mediaframework.Constant;
import com.yy.videoplayer.videoview.VideoPosition;
import com.yy.videoplayer.view.YMFLayoutParams;
import com.yy.videoplayer.view.YMFVideoPosition;

import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;

public class ThunderPlayerMultiViewProxy {

    private static final String TAG = "ThunderPlayerView";

    private YMFVideoPosition[] mVideoPositions;
    private YMFVideoPosition mBgVideoPosition;
    private int mVideoCount;
    private Context mContext;
    private ConcurrentHashMap<String, Seat> mStreamKeyToSeatMap;
    private Constant.MultiLianmaiMode lianMaiMode;
    private YMFLayoutParams mYMFLayoutParams;
    private ThunderPlayerView mVideoPlayerUnityView;
    private ThunderPlayerMultiView mVideoPlayerView;
    private Handler mUiHandler = null;
    private ThunderVideoPlayEngineImp engineImp;
    private Object syncLock = new Object();

    // 该标志位主要用来向上兼容新旧版本对于ThunderMultiPlayerView和ThunderPlayerView的区分
    // 内部都通过BuildConfig切换为最新的YMFPlayerView实现多人连麦
    private boolean mUseUniformView = false;


    public ThunderPlayerMultiViewProxy(Context context) {
        mContext = context;
        mUiHandler = new android.os.Handler(Looper.getMainLooper());
        mStreamKeyToSeatMap = new ConcurrentHashMap<String, Seat>();
    }

    public ThunderPlayerMultiViewProxy(Context context, ThunderVideoPlayEngineImp engine) {
        mContext = context;
        mUiHandler = new android.os.Handler(Looper.getMainLooper());
        mStreamKeyToSeatMap = new ConcurrentHashMap<String, Seat>();
        engineImp = engine;
    }

    public void initMultiPlayerViewLayout(ThunderMultiVideoViewParam layoutParam) {

        if (layoutParam == null) {
            return;
        }
        if (layoutParam.mVideoPositions == null && layoutParam.mVideoViewPositions == null) {
            return;
        }
        int videoCount = 0;
        if (layoutParam.mVideoViewPositions != null) {
            int positionSize = layoutParam.mVideoViewPositions.size();
            mVideoPositions = new YMFVideoPosition[positionSize];
            for (int i = 0; i < positionSize; i++) {
                if (layoutParam.mVideoViewPositions.get(i).mIndex < positionSize) {
                    mVideoPositions[layoutParam.mVideoViewPositions.get(i).mIndex] =
                            convertToVideoPosition(layoutParam.mVideoViewPositions.get(i));
                }
            }
            mBgVideoPosition = convertToVideoPosition(layoutParam.mBgViewPosition);
        } else if (layoutParam.mVideoPositions != null) {
            mVideoPositions = new YMFVideoPosition[layoutParam.mVideoPositions.size()];
            for (int i = 0; i < layoutParam.mVideoPositions.size(); i++) {
                if (layoutParam.mVideoPositions.get(i).mIndex < layoutParam.mVideoPositions.size()) {
                    mVideoPositions[layoutParam.mVideoPositions.get(i).mIndex] = convertToVideoPosition(layoutParam.mVideoPositions.get(i));
                }
            }
            //layoutParam.mVideoPositions.toArray(mVideoPositions);
            mBgVideoPosition = convertToVideoPosition(layoutParam.mBgPosition);
        }
        videoCount = arraySize(mVideoPositions);

        mYMFLayoutParams = new YMFLayoutParams(videoCount);
        mYMFLayoutParams.mDrawPosition = mVideoPositions;
        mYMFLayoutParams.backgroudPosition = mBgVideoPosition;
        mYMFLayoutParams.background = layoutParam.mBgBitmap;

        lianMaiMode = new SparseArray<Constant.MultiLianmaiMode>(10) {{
            put(0, Constant.MultiLianmaiMode.NormalMode);
            put(1, Constant.MultiLianmaiMode.NormalMode);
            put(2, Constant.MultiLianmaiMode.TwoPersonMode);
            put(3, Constant.MultiLianmaiMode.ThreePersonMode);
            put(4, Constant.MultiLianmaiMode.FourPersonMode);
            put(5, Constant.MultiLianmaiMode.FivePersonMode);
            put(6, Constant.MultiLianmaiMode.SixPersonMode);
            put(7, Constant.MultiLianmaiMode.SevenPersonMode);
            put(8, Constant.MultiLianmaiMode.EightPersonMode);
            put(9, Constant.MultiLianmaiMode.NinePersonMode);
        }}.get(arraySize(mVideoPositions));


        updateLayout(videoCount);
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "initMultiPlayerViewLayout:" + mVideoCount);
}
    }

    private void updateLayout(int videoCount) {
        // 布局数量发生变化时，unlink调以前的绑定关系，看视频需要重新link；
        // 布局数量不变，仅仅更新位置等信息时，内部仅仅更新参数
        if (mVideoPlayerView != null) {
            if (mVideoCount != 0 && mVideoCount != videoCount) {
                resetAll();
                mVideoPlayerView.updateMultiViewLayout(mYMFLayoutParams);
                ThunderLog.release(TAG, "need redraw, update MultiPlayerViewLayout:" + mVideoCount);
            } else if (mVideoCount != 0 && mVideoCount == videoCount ||
                    mVideoCount == 0 && mVideoCount < videoCount) {
                mVideoPlayerView.updateMultiViewLayout(mYMFLayoutParams);
                ThunderLog.release(TAG, "just update MultiPlayerViewLayout:" + mVideoCount);
            }
            mVideoPlayerView.enterMultiVideoViewMode(mYMFLayoutParams, lianMaiMode);

        } else if (mVideoPlayerUnityView != null) {
            if (mVideoCount != 0 && mVideoCount != videoCount) {
                resetAll();
                mVideoPlayerUnityView.updateMultiViewLayout(mYMFLayoutParams);
                ThunderLog.release(TAG, "need redraw, update MultiPlayerViewLayout:" + mVideoCount);
            } else if (mVideoCount != 0 && mVideoCount == videoCount ||
                    mVideoCount == 0 && mVideoCount < videoCount) {
                mVideoPlayerUnityView.updateMultiViewLayout(mYMFLayoutParams);
                ThunderLog.release(TAG, "just update MultiPlayerViewLayout:" + mVideoCount);
            }
            mVideoPlayerUnityView.enterMultiVideoViewMode(mYMFLayoutParams, lianMaiMode);
        }
         mVideoCount = videoCount;
    }

    private YMFVideoPosition convertToVideoPosition(VideoPosition viewPosition) {
        if (viewPosition == null) {
            return null;
        }
        YMFVideoPosition result = new YMFVideoPosition();
        result.mX = viewPosition.mX;
        result.mY = viewPosition.mY;
        result.mHeight = viewPosition.mHeight;
        result.mWidth = viewPosition.mWidth;
        result.mIndex = viewPosition.mIndex;
        return result;
    }

    private YMFVideoPosition convertToVideoPosition(ThunderMultiVideoViewCoordinate viewPosition) {
        if (viewPosition == null) {
            return null;
        }
        YMFVideoPosition result = new YMFVideoPosition();
        result.mX = viewPosition.mX;
        result.mY = viewPosition.mY;
        result.mHeight = viewPosition.mHeight;
        result.mWidth = viewPosition.mWidth;
        result.mIndex = viewPosition.mIndex;
        return result;
    }

    public VideoPlayerView getPlayerView() {
        return mVideoPlayerView;
    }

    public void updateMultiPlayerView(Object view) {
        ThunderLog.release(TAG, "updateMultiPlayerView enter, mVideoPlayerView：" + mVideoPlayerView +
               " mVideoPlayerUnityView：" + mVideoPlayerUnityView + " view " + view);
        if (view == null) {
            unlink();
            synchronized (syncLock) {
                unPrepareView();
                if (mUseUniformView) {
                    mVideoPlayerUnityView = null;
                } else {
                    mVideoPlayerView = null;
                }
            }
        }
        if (view instanceof ThunderPlayerMultiView) {
            mUseUniformView = false;

            if (mVideoPlayerView == view) {
if (ThunderLog.isInfoValid()) {
                ThunderLog.info(TAG, "updateMultiPlayerView already exist");
}
                return;
            }
            if (mVideoPlayerView != null && mVideoPlayerView != view) {
                unlink();
                unPrepareView();
                mVideoPlayerView = null;
            }
            mVideoPlayerView = (ThunderPlayerMultiView) view;


            if (mVideoPlayerView != null) {
                mVideoPlayerView.enterMultiVideoViewMode(mYMFLayoutParams, lianMaiMode);
            }

        } else if (view instanceof ThunderPlayerView) {
            mUseUniformView = true;

            if (mVideoPlayerUnityView == view) {
if (ThunderLog.isInfoValid()) {
                ThunderLog.info(TAG, "updateMultiPlayerView already exist");
}
                return;
            }
            if (mVideoPlayerUnityView != null && mVideoPlayerUnityView != view) {
                unlink();
                unPrepareView();
                mVideoPlayerUnityView = null;
            }
            mVideoPlayerUnityView = (ThunderPlayerView) view;
            if (mVideoPlayerUnityView != null) {
                mVideoPlayerUnityView.enterMultiVideoViewMode(mYMFLayoutParams, lianMaiMode);
            }
        }
        ThunderLog.release(TAG, "updateMultiPlayerView exit");
    }

    /**
     * 退频道结束view的生命周期
     */
    public void destroyVideoPlayerView() {
        ThunderLog.release(TAG, "destroyVideoPlayerView");
        if (mStreamKeyToSeatMap != null) {
            unlink();
            mStreamKeyToSeatMap.clear();
        }
        if (!mUseUniformView) {
            if (mVideoPlayerView != null) {
                unPrepareView();
                mVideoPlayerView.leaveMultiVideoViewMode();
            }
            mVideoPlayerView = null;
        } else {
            if (mVideoPlayerUnityView != null) {
                unPrepareView();
                mVideoPlayerUnityView.leaveMultiVideoViewMode();
            }
            mVideoPlayerUnityView = null;
        }
        mYMFLayoutParams = null;
    }

    public VideoPlayerView getVideoPlayerView() {
        if (!mUseUniformView) {
            return mVideoPlayerView;
        } else {
            return mVideoPlayerUnityView;
        }
    }

    /**
     * 更新用户位置信息
     *
     * @param streamKey
     * @param seatId
     */
    public int updateSeat(String streamKey, int seatId) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "innerUpdateSeat() called with: uid = [" + streamKey + "], seat = [" + seatId + "]");
}
        if (seatId >= mVideoCount || streamKey == null) {
            ThunderLog.error(TAG, "updateSeat:" + seatId + "maxSeats:" + mVideoCount + "streamKey:" + streamKey);
            return -1;
        }
        return bindSeat(streamKey, seatId);
    }

    private int bindSeat(final String streamKey, final int seatId) {
        ThunderLog.release(TAG, "bindSeat called with: uid = [" + streamKey + "], seatNo = [" + seatId + "]");
        synchronized (mStreamKeyToSeatMap) {

            Seat seatItem = findSeatByStreamKey(streamKey);

            if (seatId < 0) {
                ThunderLog.error(TAG, "updateSeat invalid, seat " + seatId + " invalid!");
                return -1;
            }

            // 下面是正常的处理逻辑，针对某一路流
            // 没有找到该路流对应的位置信息，创建位置，并填充座位号
            if (seatItem == null) {
                seatItem = new Seat() {{
                    this.seatIdx = seatId;
                }};
            } else {
                // 找到该路流对应的位置信息，已经link过，需要unlink之前的，然后更新新的位置
                if (seatItem.seatIdx != seatId
                        && seatItem.status == Seat.STATUS_LINK) {
if (ThunderLog.isInfoValid()) {
                    ThunderLog.info(TAG,
                            "bindSeat updateseat had linked:" + streamKey + ", will unlink!!");
}

                    // 取消老的位置
                    unlink(seatItem);
                }

                seatItem.seatIdx = seatId;

                // 查找该座位是否绑定过，如果说明座位忙，此处选择直接return，或者也可以强制更新
                if (processBusySeatIfExist(streamKey, seatId) > 0) {
                    // 该位置忙
                    ThunderLog.error(TAG, "bindSeat busy!!");
                }
            }
            // 更新位置后，满足条件的可以绑定渲染了
            if (seatItem.seatIdx >= 0 && seatItem.status == Seat.STATUS_IDEL && seatItem.streamId != -1) {
                link(seatItem);
            }
            mStreamKeyToSeatMap.put(streamKey, seatItem);
            ThunderLog.release(TAG, "bindSeat updateseat streamKey:" + streamKey + ",seat:" + seatId + " >> "
                    + seatItem + " " + " streamId " + seatItem.streamId);

            return 0;
        }
    }

    /**
     * 更新位置信息
     *
     * @param streamKey 跟据streamName appId uid生成，早于streamId
     * @param streamId  传输生成
     */
    public void bindStreamToSeat(String streamKey, long streamId) {

        if (streamKey == null) {
            ThunderLog.error(TAG, "bindStreamToSeat streamKey null");
            return;
        }
        if (mStreamKeyToSeatMap == null) {
            ThunderLog.error(TAG, "bindStreamToSeat mStreamKeyToSeatMap null");
            return;
        }

        if (streamId == 0L || streamId == -1L) {
            ThunderLog.error(TAG, "bindStreamToSeat streamId," + streamId);
            return;
        }
        synchronized (mStreamKeyToSeatMap) {
            Seat seatItem = findSeatByStreamKey(streamKey);


            if (seatItem == null) {
                seatItem = new Seat();
                seatItem.streamId = streamId;
                mStreamKeyToSeatMap.put(streamKey, seatItem);
                return;
            }

            if (seatItem != null &&
                    seatItem.seatIdx >= 0 &&
                    seatItem.status == Seat.STATUS_LINK) {
if (ThunderLog.isInfoValid()) {
                ThunderLog.info(TAG, "bindStreamToSeat replace and link: seatItem.streamId" + seatItem.streamId +
                        " streamId:" + streamId);
}
                unlink(seatItem);
                seatItem.streamId = streamId;
                link(seatItem);
            } else if (seatItem != null &&
                    seatItem.status == Seat.STATUS_IDEL) {

                seatItem.streamId = streamId;
                link(seatItem);
                ThunderLog.release(TAG, "bindStreamToSeat update: seatItem.streamId:" + streamId +
                        " seatId:" + seatItem.seatIdx);
            }
        }
    }

    private int processBusySeatIfExist(String streamKey, int seatId) {
        if (seatId >= 0) {
            Seat curSeat = findSeatByStreamKey(streamKey);
            ArrayList<Seat> seatItemList = findSeatBySeatId(seatId);
            if (seatItemList != null && seatItemList.size() > 0) {
                for (Seat seatItem : seatItemList) {
                    if (seatItem != null && seatItem != curSeat  &&
                            seatItem.streamId != 0 &&  curSeat.streamId != 0 &&
                            (seatItem.streamId >> 32 != curSeat.streamId >>32)) {
                        ThunderLog.error(TAG, "processBusySeat, streamKey" + streamKey + " seat " + seatId +
                                " busy!");
                        unlink(seatItem);
                        seatItem.seatIdx = -1;
                        if (engineImp != null) {
                            engineImp.onMultiViewSeatInfoChangedNotify(seatItem.streamId, seatItem.seatIdx);
                        }
                        return 1;
                    }
                }
            }
        }
        return 0;
    }

    public Seat findSeatByStreamKey(String streamKey) {
        if (mStreamKeyToSeatMap == null) {
            ThunderLog.error(TAG, "mStreamKeyToSeatMap null");
            return null;
        }
        Seat seatItem = mStreamKeyToSeatMap.get(streamKey);
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "findSeatByStreamKey:" + streamKey + ",caches:" + mStreamKeyToSeatMap);
}
        return seatItem;
    }

    public Seat removeSeatByStreamKey(String streamKey) {
        return mStreamKeyToSeatMap.remove(streamKey);
    }

    public ArrayList<Seat> findSeatBySeatId(int seatIdx) {
        ArrayList<Seat> seatArrayList = new ArrayList<>(0);
        if (mStreamKeyToSeatMap != null && seatIdx >= 0) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem.seatIdx == seatIdx) {
                    seatArrayList.add(seatItem);
                }
            }
        }
        return seatArrayList;
    }


    public ArrayList<String> findStreamKeyBySeatId(int seatIdx) {
        ArrayList<String> streamKeyList = new ArrayList<>(0);
        if (mStreamKeyToSeatMap != null && seatIdx >= 0) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem.seatIdx == seatIdx) {
                    streamKeyList.add(entry.getKey());
                }
            }
        }
        return streamKeyList;
    }

    public Seat findSeatByStreamId(long streamId) {
        if (mStreamKeyToSeatMap != null) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem.streamId == streamId) {
                    return seatItem;
                }
            }
        }
        return null;
    }

    private int link(Seat seatItem) {
        int result = -1;
        if (seatItem != null) {
            if (!mUseUniformView) {
                if (mVideoPlayerView != null && seatItem.seatIdx >= 0 && seatItem.streamId != 0 &&
                        seatItem.status == Seat.STATUS_IDEL) {
                    result = mVideoPlayerView.linkToStream(seatItem.streamId, seatItem.seatIdx);
                }
            } else {
                if (mVideoPlayerUnityView != null && seatItem.seatIdx >= 0 && seatItem.streamId != 0 &&
                        seatItem.status == Seat.STATUS_IDEL) {
                    result = mVideoPlayerUnityView.linkToStream(seatItem.streamId, seatItem.seatIdx);
                }
            }
            if (result >= 0) {
                seatItem.status = Seat.STATUS_LINK;
            }
            ThunderLog.release(TAG, "link- Seat:" + seatItem + ",ret:" + result);
        } else {
            ThunderLog.error(TAG, "link- Seat:" + seatItem + ",ret:" + result);
        }
        return result;
    }

    private int unlink(Seat seatItem) {
        int result = -1;
        if (seatItem != null) {
            if (!mUseUniformView) {
                if (mVideoPlayerView != null && seatItem.seatIdx >= 0 && seatItem.streamId != 0 &&
                        seatItem.status == Seat.STATUS_LINK) {
                    result = mVideoPlayerView.unLinkFromStream(seatItem.streamId, seatItem.seatIdx);
                }
            } else {
                if (mVideoPlayerUnityView != null && seatItem.seatIdx >= 0 && seatItem.streamId != 0 &&
                        seatItem.status == Seat.STATUS_LINK) {
                    result = mVideoPlayerUnityView.unLinkFromStream(seatItem.streamId, seatItem.seatIdx);
                }
            }
            if (result >= 0) {
                seatItem.status = Seat.STATUS_IDEL;
            }
            ThunderLog.release(TAG, "unlink- Seat:" + seatItem + ",ret:" + result);
        }
        return result;
    }

    public int link(String streamKey) {
        Seat seatItem = findSeatByStreamKey(streamKey);
        return link(seatItem);
    }

    public int unlink(String streamKey) {
        Seat seatItem = findSeatByStreamKey(streamKey);
        return unlink(seatItem);
    }

    public void releaseSeatByStreamKey(String streamKey) {
        synchronized (mStreamKeyToSeatMap) {
            mStreamKeyToSeatMap.remove(streamKey);
if (ThunderLog.isInfoValid()) {
            ThunderLog.info(TAG, "releaseSeatByStreamKey- Seat:" + streamKey);
}
        }
    }

    public int unlink() {
        if (mStreamKeyToSeatMap != null) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem != null) {
                    unlink(seatItem);
                }
            }
        }
        return 0;
    }


    public int link() {
        if (mStreamKeyToSeatMap != null) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem != null) {
                    link(seatItem);
                }
            }
        }
        return 0;
    }

    public void resetAll() {
        if (mStreamKeyToSeatMap != null) {
            unlinkAndReset();
            //mStreamKeyToSeatMap.clear();
            ThunderLog.release(TAG, "resetAll");
        }
    }

    public int unlinkAndReset() {
        if (mStreamKeyToSeatMap != null) {
            for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                Seat seatItem = entry.getValue();
                if (seatItem != null) {
                    unlink(seatItem);
                    seatItem.seatIdx = -1;
                    if (engineImp != null) {
                        engineImp.onMultiViewSeatInfoChangedNotify(seatItem.streamId, seatItem.seatIdx);
                    }
                }
            }
        }
        return 0;
    }

    public void prepareView() {

        if (!mUseUniformView) {
            if (mVideoPlayerView == null) {
                return;
            }
            if (Thread.currentThread().getId() == mUiHandler.getLooper().getThread().getId()) {
                synchronized (syncLock) {
                    if (mVideoPlayerView != null) {
                        mVideoPlayerView.prepareView();
                    }
                }
            } else {
                //新建程流
                final CountDownLatch barrier = new CountDownLatch(1);
                mUiHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (syncLock) {
                            if (mVideoPlayerView != null) {
                                mVideoPlayerView.prepareView();
                            }
                        }
                        barrier.countDown();
                    }
                });
                try {
                    barrier.await();
                } catch (InterruptedException e) {
                    ThunderLog.error("ThunderVideoEngineImp", "updatePlayVideoStream .barrier.await" + e.toString());
                }
            }
        } else {
            if (mVideoPlayerUnityView == null) {
                return;
            }
            if (Thread.currentThread().getId() == mUiHandler.getLooper().getThread().getId()) {
                synchronized (syncLock) {
                    if (mVideoPlayerUnityView != null) {
                        mVideoPlayerUnityView.prepareView();
                    }
                }
            } else {
                //新建程流
                final CountDownLatch barrier = new CountDownLatch(1);
                mUiHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        synchronized (syncLock) {
                            if (mVideoPlayerUnityView != null) {
                                mVideoPlayerUnityView.prepareView();
                            }
                        }
                        barrier.countDown();
                    }
                });
                try {
                    barrier.await();
                } catch (InterruptedException e) {
                    ThunderLog.error("ThunderVideoEngineImp", "updatePlayVideoStream .barrier.await" + e.toString());
                }
            }
        }
    }

    public void unPrepareView() {

        if (!mUseUniformView) {
            if (mVideoPlayerView == null) {
                return;
            }
            if (Thread.currentThread().getId() == mUiHandler.getLooper().getThread().getId()) {
                if (mVideoPlayerView != null) {
                    mVideoPlayerView.unPrepareView();
                }
            } else {
                //新建程流
                final CountDownLatch barrier = new CountDownLatch(1);
                mUiHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (mVideoPlayerView != null) {
                            mVideoPlayerView.unPrepareView();
                        }
                        barrier.countDown();
                    }
                });
                try {
                    barrier.await();
                } catch (InterruptedException e) {
                    ThunderLog.error("ThunderVideoEngineImp", "updatePlayVideoStream .barrier.await" + e.toString());
                }
            }
        } else {
            if (mVideoPlayerUnityView == null) {
                return;
            }
            if (Thread.currentThread().getId() == mUiHandler.getLooper().getThread().getId()) {
                if (mVideoPlayerUnityView != null) {
                    mVideoPlayerUnityView.unPrepareView();
                }
            } else {
                //新建程流
                final CountDownLatch barrier = new CountDownLatch(1);
                mUiHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (mVideoPlayerUnityView != null) {
                            mVideoPlayerUnityView.unPrepareView();
                        }
                        barrier.countDown();
                    }
                });
                try {
                    barrier.await();
                } catch (InterruptedException e) {
                    ThunderLog.error("ThunderVideoEngineImp", "updatePlayVideoStream .barrier.await" + e.toString());
                }
            }
        }
    }

    public boolean setScaleMode(int scaleMode) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "setScaleMode:" + scaleMode);
}
        boolean result = true;
        if (!mUseUniformView) {
            if (mStreamKeyToSeatMap != null && mVideoPlayerView != null) {
                for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                    Seat seatItem = entry.getValue();
                    if (seatItem != null && seatItem.seatIdx >= 0) {
                        result &= mVideoPlayerView.setScaleMode(seatItem.seatIdx, scaleMode);
                    }
                }
            }
        } else {
            if (mStreamKeyToSeatMap != null && mVideoPlayerUnityView != null) {
                for (Map.Entry<String, Seat> entry : mStreamKeyToSeatMap.entrySet()) {
                    Seat seatItem = entry.getValue();
                    if (seatItem != null && seatItem.seatIdx >= 0) {
                        result &= mVideoPlayerUnityView.setScaleMode(seatItem.seatIdx, scaleMode);
                    }
                }
            }
        }
        return result;
    }

    public boolean setScaleMode(String streamKey, int scaleMode) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "setScaleMode: " + streamKey + " " + scaleMode);
}
        boolean result = true;
        if (!mUseUniformView) {
            if (mStreamKeyToSeatMap != null && mVideoPlayerView != null && scaleMode >= 0) {
                Seat seat = findSeatByStreamKey(streamKey);
                if (seat != null) {
                    result = mVideoPlayerView.setScaleMode(seat.seatIdx, scaleMode);
                }
            }
        } else {
            if (mStreamKeyToSeatMap != null && mVideoPlayerUnityView != null && scaleMode >= 0) {
                Seat seat = findSeatByStreamKey(streamKey);
                if (seat != null) {
                    result = mVideoPlayerUnityView.setScaleMode(seat.seatIdx, scaleMode);
                }
            }
        }
        return result;
    }

    public void setMirrorMode(String streamKey, int mirrorMode) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "setMirrorMode: " + streamKey + " " + mirrorMode);
}
        if (!mUseUniformView) {
            if (mStreamKeyToSeatMap != null && mVideoPlayerView != null) {
                Seat seat = findSeatByStreamKey(streamKey);
                if (seat != null) {
                    mVideoPlayerView.setMirrorMode(seat.seatIdx, mirrorMode);
                }
            }
        } else {
            if (mStreamKeyToSeatMap != null && mVideoPlayerUnityView != null) {
                Seat seat = findSeatByStreamKey(streamKey);
                if (seat != null) {
                    mVideoPlayerUnityView.setMirrorMode(seat.seatIdx, mirrorMode);
                }
            }
        }
    }

    public Bitmap getVideoScreenshot(String streamKey) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "getVideoScreenshot: " + streamKey );
}
        if (!mUseUniformView) {
            if (mStreamKeyToSeatMap != null && mVideoPlayerView != null ) {
                Seat seat = findSeatByStreamKey(streamKey);
                if (seat != null) {
                    return mVideoPlayerView.getVideoScreenshot(seat.seatIdx);
                }
            }
        } else {
            if (mStreamKeyToSeatMap != null && mVideoPlayerUnityView != null ) {
                Seat seat = findSeatByStreamKey(streamKey);
                if (seat != null) {
                    return mVideoPlayerUnityView.getVideoScreenshot(seat.seatIdx);
                }
            }
        }
        return null;
    }

    public static <T> int arraySize(T[] xs) {
        return xs == null ? 0 : xs.length;
    }

    public static class Seat {
        private static final int STATUS_IDEL = 0;
        private static final int STATUS_LINK = 1;

        public int seatIdx = -1;
        public long streamId = 0;
        public int status = STATUS_IDEL;

        @Override
        public String toString() {
            return "Seat{" +
                    ", seatIdx=" + seatIdx +
                    ", status=" + status +
                    '}';
        }
    }
}
