package com.thunder.livesdk.video;

/**
 * Created on 2019/11/06.
 * Copyright (c) 2019 YY Inc. All rights reserved.
 */

import android.content.Context;
import android.graphics.Bitmap;
import android.util.AttributeSet;
import android.view.SurfaceView;

import com.thunder.livesdk.ThunderRtcConstant;
import com.yy.mediaframework.Constant;
import com.yy.mediaframework.YYVideoSDK;
import com.yy.videoplayer.YMFPlayerAPI;
import com.yy.videoplayer.decoder.VideoConstant;
import com.yy.videoplayer.view.YMFLayoutParams;
import com.yy.videoplayer.view.YMFVideoPosition;
import com.thunder.livesdk.log.ThunderLog;


/**
 * 用于播放视频的组件<br/>
 * <p>需要播放视频时，在布局文件中加入这个组件</p>
 * <p>
 * 如下，外框是视频父布局，VideoPosition是视频布局区域
 * |---------------------------------------------|
 * | (0,0)                                       |
 * |                                             |
 * |                             (mX,mY)      |
 * |                               |---width---| |
 * |                               |           | |
 * |                        height |           | |
 * |                               | VideoRect | |
 * |                               |           | |
 * |                               |-----------| |
 * |---------------------------------------------|
 */

public class ThunderPlayerMultiView extends VideoPlayerView {

    private static final String TAG = "ThunderPlayerMultiView";

    private YMFLayoutParams mYMFLayoutParams = null;
    private YMFPlayerAPI mYMFPlayerInstance = null;

    public ThunderPlayerMultiView(Context context) {
        super(context);
        addView(new SurfaceView(context));
        init();
    }

    public ThunderPlayerMultiView(Context context, AttributeSet attrs) {
        super(context, attrs);
        addView(new SurfaceView(context));           //防止一开始进入频道的闪烁
        init();
    }

    public ThunderPlayerMultiView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        addView(new SurfaceView(context));           //防止一开始进入频道的闪烁
        init();
    }

    private void init() {
        mYMFLayoutParams = new YMFLayoutParams(1);
        YMFVideoPosition videoPosition = new YMFVideoPosition();
        videoPosition.mIndex = 0;
        videoPosition.mX = 0;
        videoPosition.mY = 0;
        videoPosition.mHeight = -2;     //-2是铺满surface的标志
        videoPosition.mWidth = -2;      //-2是铺满surface的标志
        mYMFLayoutParams.mDrawPosition[0] = videoPosition;
    }

    private boolean createUnityVideoView() {
        if (mYMFPlayerInstance == null) {
            mYMFPlayerInstance = YMFPlayerAPI.newInstance();
        }
        mUniformView = mYMFPlayerInstance.createView(this.getContext());
        if (mUniformView != null) {
            this.addView(mUniformView);
            mYMFPlayerInstance.setViewLayout(mUniformView, mYMFLayoutParams);
            mYMFPlayerInstance.setScaleMode(mUniformView, mScaleMode, -1);
            mViewType = VideoConstant.ViewType.GL_VIDEO_VIEW;
            mPrepared = true;
        }
        ThunderLog.release(ThunderLog.kLogTagVideo,
                "createUnityVideoView:" + mViewType.name() + " size " + this.getWidth() + "x" + this.getHeight());
        return true;
    }

    /**
     * 初始化多人视频播放的View<br/>
     * <br/><b>注意：</b>sdk库内部逻辑使用，请不要调用<br/>
     *
     * @return 成功or失败
     */
    public boolean prepareView() {
        unPrepareView(); // 清掉构造函数添加的 SurfaceView
        return createUnityVideoView();
    }

    /**
     * 反初始化视频播放的View<br/>
     * <br/><b>注意：</b>sdk库内部逻辑使用，请不要调用<br/>
     */
    public void unPrepareView() {
        synchronized (this) {
            this.removeAllViews();
            if (mYMFPlayerInstance != null) {
                mYMFPlayerInstance.destroyView(mUniformView);
//                mYMFPlayerInstance.release();
                mUniformView = null;
                mPrepared = false;
            }
        }
        ThunderLog.release(ThunderLog.kLogTagVideo, "unPrepareView.");
    }

    /**
     * 绑定视频流到某位置<br/>
     * <br/><b>注意：</b>sdk库内部逻辑使用，请不要调用<br/>
     *
     * @param streamId 流ID
     */
    public int linkToStream(long streamId, int videoIndex) {
        if (streamId == 0L) {
            return -1;
        }

        if (videoIndex < 0) {
            return -1;
        }

        if (mUniformView == null) {
            return -1;
        }

        if (mYMFPlayerInstance != null && mUniformView != null) {
            return mYMFPlayerInstance.linkStream(mUniformView, streamId, videoIndex);
        }
        return 0;
    }

    /**
     * 解绑视频流 <br/>
     * <br/><b>注意：</b>sdk库内部逻辑使用，请不要调用<br/>
     *
     * @param streamId 流ID
     */
    public int unLinkFromStream(long streamId, int videoIndex) {

        if (streamId == 0L) {
            return -1;
        }

        if (videoIndex < 0) {
            return -1;
        }

        if (mUniformView == null) {
            return -1;
        }

        if (mYMFPlayerInstance != null) {
            return mYMFPlayerInstance.unLinkStream(mUniformView, streamId, videoIndex);
        }
        return 0;
    }

    private void setUseMultiVideoView(YMFLayoutParams multiVideoViewParams) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "setUseMultiVideoView");
}
        mYMFLayoutParams = multiVideoViewParams;
        if (mYMFPlayerInstance!= null) {
            mYMFPlayerInstance.setViewLayout(mUniformView, mYMFLayoutParams);
        }
    }

    public void enterMultiVideoViewMode(YMFLayoutParams multiVideoViewParams,
                                        Constant.MultiLianmaiMode lianmaiMode) {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG,
                "enterMultiVideoViewMode multiVideoViewParams = [" + multiVideoViewParams + "], lianmaiMode=" +
                        lianmaiMode);
}
        if (multiVideoViewParams == null) {
            multiVideoViewParams = new YMFLayoutParams(1);
            YMFVideoPosition videoPosition = new YMFVideoPosition();
            videoPosition.mIndex = 0;
            videoPosition.mX = 0;
            videoPosition.mY = 0;
            videoPosition.mHeight = -2;     //-2是铺满surface的标志
            videoPosition.mWidth = -2;      //-2是铺满surface的标志
            multiVideoViewParams.mDrawPosition[0] = videoPosition;
            setUseMultiVideoView(multiVideoViewParams);
            return;
        }
        synchronized (this) {
            YYVideoSDK.getInstance().setMultiVideoLianmaiMode(lianmaiMode);
            setUseMultiVideoView(multiVideoViewParams);
        }
    }

    public void leaveMultiVideoViewMode() {
if (ThunderLog.isInfoValid()) {
        ThunderLog.info(TAG, "leaveMultiVideoViewMode ");
}
        synchronized (this) {
            setUseMultiVideoView(null);
            YYVideoSDK.getInstance().setMultiVideoLianmaiMode(Constant.MultiLianmaiMode.NormalMode);
        }
    }

    /**
     * 修改视频缩放模式<br/>
     *
     * @param scaleMode 缩放模式 {@link ThunderRtcConstant.ThunderVideoRenderMode}
     * @return 成功or失败
     */
    public boolean setScaleMode(int seatId, int scaleMode) {
        boolean result = false;
        mScaleMode = scaleModeConvert(scaleMode);
        if (mYMFPlayerInstance != null) {
            mYMFPlayerInstance.setScaleMode(mUniformView, mScaleMode, seatId);
        }
        return result;
    }

    /**
     * 修改视频镜像模式<br/>
     * @param seatId 座位信息
     * @param mirrorMode 镜像模式
     */
    public void setMirrorMode(int seatId, int mirrorMode) {
        mMirrorMode = mirrorModeConvert(mirrorMode);
        if (mYMFPlayerInstance != null) {
            mYMFPlayerInstance.setMirrorMode(mUniformView, mMirrorMode, seatId);
        }
    }

    /**
     * 获取截图
     * @param seatId 座位
     * @return Bitmap
     */
    public Bitmap getVideoScreenshot(int seatId) {
        if (mYMFPlayerInstance != null) {
            return mYMFPlayerInstance.getVideoScreenshot(mUniformView, seatId);
        }
        return null;
    }

    public void updateMultiViewLayout(YMFLayoutParams params) {
        mYMFLayoutParams = params;
        if (mYMFPlayerInstance != null && mUniformView != null) {
            mYMFPlayerInstance.setViewLayout(mUniformView, params);
        }
    }
}

